-- SPDX-FileCopyrightText: Copyright 2022-present Greg Hurrell and contributors.
-- SPDX-License-Identifier: BSD-2-Clause

local prompt = {}

local Window = require('wincent.commandt.private.window').Window
local merge = require('wincent.commandt.private.merge')

local Prompt = {}

local mt = {
  __index = Prompt,
}

function Prompt.new(options)
  options = merge({
    mappings = {},
    margin = 0,
    height = 15,
    name = nil,
    on_change = nil,
    on_leave = nil,
    on_open = nil,
    on_select = nil,
    position = 'bottom',
  }, options or {})
  -- TODO validate options
  local p = {
    _height = options.height,
    _mappings = options.mappings,
    _margin = options.margin,
    _name = options.name,
    _on_change = options.on_change,
    _on_leave = options.on_leave,
    _on_open = options.on_open,
    _on_select = options.on_select,
    _position = options.position,
    _window = nil,
  }
  setmetatable(p, mt)
  return p
end

function Prompt:close()
  if self._window then
    self._window:close()
  end
end

function Prompt:show()
  local bottom = nil
  local top = nil
  if self._position == 'center' then
    local available_height = vim.o.lines - vim.o.cmdheight
    local used_height = self._height -- note we need to know how high the match listing is going to be
      + 2 -- match listing border
      + 3 -- our height
    local remaining_height = math.max(1, available_height - used_height)
    top = math.floor(remaining_height / 2)
  elseif self._position == 'bottom' then
    bottom = 0
  else
    top = 0
  end

  if self._window == nil then
    local title = self._name and ('CommandT [' .. self._name .. ']') or 'CommandT'
    self._window = Window.new({
      bottom = bottom,
      buftype = 'prompt',
      filetype = 'CommandTPrompt',
      margin = self._margin,
      on_change = function(contents)
        if self._on_change then
          self._on_change(contents)
        end
      end,
      on_close = function()
        self._window = nil
      end,
      on_leave = self._on_leave,
      title = title,
      top = top,
    })
  end

  self._window:show()

  local callbacks = {
    close = function()
      if self._window then
        self._window:close()
      end
    end,
    open = function()
      if self._on_open then
        self._on_open('edit')
      end
    end,
    open_split = function()
      if self._on_open then
        self._on_open('split')
      end
    end,
    open_tab = function()
      if self._on_open then
        self._on_open('tabedit')
      end
    end,
    open_vsplit = function()
      if self._on_open then
        self._on_open('vsplit')
      end
    end,
    select_first = function()
      if self._on_select then
        self._on_select({ absolute = 1 })
      end
    end,
    select_last = function()
      if self._on_select then
        self._on_select({ absolute = -1 })
      end
    end,
    select_middle = function()
      if self._on_select then
        self._on_select({ absolute = 0 })
      end
    end,
    select_next = function()
      if self._on_select then
        self._on_select({ relative = 1 })
      end
    end,
    select_previous = function()
      if self._on_select then
        self._on_select({ relative = -1 })
      end
    end,
  }
  for mode, mappings in pairs(self._mappings) do
    for lhs, rhs in pairs(mappings) do
      if rhs then
        self._window:map(mode, lhs, callbacks[rhs] or rhs)
      end
    end
  end
  self._window:focus()
end

prompt.Prompt = Prompt

return prompt
