{-# LANGUAGE StrictData #-}

-- This file is part of the Wire Server implementation.
--
-- Copyright (C) 2022 Wire Swiss GmbH <opensource@wire.com>
--
-- This program is free software: you can redistribute it and/or modify it under
-- the terms of the GNU Affero General Public License as published by the Free
-- Software Foundation, either version 3 of the License, or (at your option) any
-- later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
-- FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
-- details.
--
-- You should have received a copy of the GNU Affero General Public License along
-- with this program. If not, see <https://www.gnu.org/licenses/>.

module Galley.App
  ( -- * Environment
    Env,
    reqId,
    options,
    applog,
    manager,
    federator,
    brig,
    cstate,
    deleteQueue,
    createEnv,
    extEnv,
    aEnv,
    ExtEnv (..),

    -- * Running Galley effects
    GalleyEffects,
    evalGalleyToIO,
    ask,
    DeleteItem (..),
    toServantHandler,
  )
where

import Bilge hiding (Request, header, host, options, port, statusCode, statusMessage)
import Cassandra hiding (Set)
import Cassandra.Util (initCassandraForService)
import Control.Error hiding (err)
import Control.Lens hiding ((.=))
import Data.Id
import Data.Misc
import Data.Qualified
import Data.Range
import Data.Text qualified as Text
import Galley.API.Error
import Galley.Cassandra.Client
import Galley.Cassandra.Code
import Galley.Cassandra.CustomBackend
import Galley.Cassandra.SearchVisibility
import Galley.Cassandra.Team
  ( interpretInternalTeamListToCassandra,
    interpretTeamListToCassandra,
    interpretTeamMemberStoreToCassandra,
    interpretTeamMemberStoreToCassandraWithPaging,
  )
import Galley.Cassandra.TeamFeatures
import Galley.Cassandra.TeamNotifications
import Galley.Effects
import Galley.Env
import Galley.External.LegalHoldService.Internal qualified as LHInternal
import Galley.Keys
import Galley.Monad (runApp)
import Galley.Options hiding (brig, endpoint, federator)
import Galley.Options qualified as O
import Galley.Queue
import Galley.Queue qualified as Q
import Galley.Types.Teams
import HTTP2.Client.Manager (Http2Manager, http2ManagerWithSSLCtx)
import Hasql.Pool qualified as Hasql
import Hasql.Pool.Extended (initPostgresPool)
import Imports hiding (forkIO)
import Network.AMQP.Extended (mkRabbitMqChannelMVar)
import Network.HTTP.Client (responseTimeoutMicro)
import Network.HTTP.Client.OpenSSL
import Network.Wai.Utilities.JSONResponse
import OpenSSL.Session as Ssl
import Polysemy
import Polysemy.Async
import Polysemy.Conc
import Polysemy.Error
import Polysemy.Fail
import Polysemy.Input
import Polysemy.Internal (Append)
import Polysemy.Resource
import Polysemy.TinyLog (TinyLog, logErrors)
import Polysemy.TinyLog qualified as P
import Servant qualified
import Ssl.Util
import System.Logger qualified as Log
import System.Logger.Class (Logger)
import System.Logger.Extended qualified as Logger
import UnliftIO.Exception qualified as UnliftIO
import Wire.API.Conversation.Protocol
import Wire.API.Error
import Wire.API.Federation.Error
import Wire.API.Team.Collaborator
import Wire.API.Team.Feature
import Wire.AWS qualified as Aws
import Wire.BackendNotificationQueueAccess.RabbitMq qualified as BackendNotificationQueueAccess
import Wire.BrigAPIAccess.Rpc
import Wire.ConversationStore.Cassandra
import Wire.ConversationStore.Postgres
import Wire.ConversationSubsystem.Interpreter (ConversationSubsystemConfig (..), interpretConversationSubsystem)
import Wire.Error
import Wire.ExternalAccess.External
import Wire.FederationAPIAccess.Interpreter
import Wire.FireAndForget
import Wire.GundeckAPIAccess (runGundeckAPIAccess)
import Wire.HashPassword.Interpreter
import Wire.LegalHoldStore.Cassandra (interpretLegalHoldStoreToCassandra)
import Wire.LegalHoldStore.Env (LegalHoldEnv (..))
import Wire.NotificationSubsystem.Interpreter (runNotificationSubsystemGundeck)
import Wire.ParseException
import Wire.ProposalStore.Cassandra
import Wire.RateLimit
import Wire.RateLimit.Interpreter
import Wire.Rpc
import Wire.Sem.Concurrency
import Wire.Sem.Concurrency.IO
import Wire.Sem.Delay
import Wire.Sem.Now.IO (nowToIO)
import Wire.Sem.Random.IO
import Wire.ServiceStore.Cassandra (interpretServiceStoreToCassandra)
import Wire.SparAPIAccess.Rpc
import Wire.TeamCollaboratorsStore.Postgres (interpretTeamCollaboratorsStoreToPostgres)
import Wire.TeamCollaboratorsSubsystem.Interpreter
import Wire.TeamJournal.Aws
import Wire.TeamStore.Cassandra (interpretTeamStoreToCassandra)
import Wire.TeamSubsystem.Interpreter
import Wire.UserGroupStore.Postgres (interpretUserGroupStoreToPostgres)

-- Effects needed by the interpretation of other effects
type GalleyEffects0 =
  '[ Input ClientState,
     Input Hasql.Pool,
     Input Env,
     Input ConversationSubsystemConfig,
     Error MigrationError,
     Error InvalidInput,
     Error ParseException,
     Error InternalError,
     -- federation errors can be thrown by almost every endpoint, so we avoid
     -- having to declare it every single time, and simply handle it here
     Error FederationError,
     Error TeamCollaboratorsError,
     Error Hasql.UsageError,
     Error HttpError,
     Race,
     Async,
     Delay,
     Fail,
     TinyLog,
     Embed IO,
     Error JSONResponse,
     Resource,
     Concurrency 'Unsafe,
     Final IO
   ]

type GalleyEffects = Append GalleyEffects1 GalleyEffects0

-- Define some invariants for the options used
validateOptions :: Opts -> IO (Either HttpsUrl (Map Text HttpsUrl))
validateOptions o = do
  let settings' = view settings o
      optFanoutLimit = fromIntegral . fromRange $ currentFanoutLimit o
  when (settings'._maxConvSize > fromIntegral optFanoutLimit) $
    error "setMaxConvSize cannot be > setTruncationLimit"
  when (settings' ^. maxTeamSize < optFanoutLimit) $
    error "setMaxTeamSize cannot be < setTruncationLimit"
  case (o ^. O.federator, o ^. rabbitmq) of
    (Nothing, Just _) -> error "RabbitMQ config is specified and federator is not, please specify both or none"
    (Just _, Nothing) -> error "Federator is specified and RabbitMQ config is not, please specify both or none"
    _ -> pure ()
  let mlsFlag = settings' ^. featureFlags . to (featureDefaults @MLSConfig)
      mlsConfig = mlsFlag.config
      migrationStatus = (.status) $ settings' ^. featureFlags . to (featureDefaults @MlsMigrationConfig)
  when (migrationStatus == FeatureStatusEnabled && ProtocolMLSTag `notElem` mlsSupportedProtocols mlsConfig) $
    error "For starting MLS migration, MLS must be included in the supportedProtocol list"
  unless (mlsDefaultProtocol mlsConfig `elem` mlsSupportedProtocols mlsConfig) $
    error "The list 'settings.featureFlags.mls.supportedProtocols' must include the value in the field 'settings.featureFlags.mls.defaultProtocol'"
  let errMsg = "Either conversationCodeURI or multiIngress needs to be set."
  case (settings' ^. conversationCodeURI, settings' ^. multiIngress) of
    (Nothing, Nothing) -> error errMsg
    (Nothing, Just mi) -> pure (Right mi)
    (Just uri, Nothing) -> pure (Left uri)
    (Just _, Just _) -> error errMsg

createEnv :: Opts -> Logger -> IO Env
createEnv o l = do
  cass <- initCassandra o l
  mgr <- initHttpManager o
  h2mgr <- initHttp2Manager
  codeURIcfg <- validateOptions o
  postgres <- initPostgresPool o._postgresqlPool o._postgresql o._postgresqlPassword
  let disableTlsV1 = True
  Env (RequestId defRequestId) o l mgr h2mgr (o ^. O.federator) (o ^. O.brig) cass postgres
    <$> Q.new 16000
    <*> initExtEnv disableTlsV1
    <*> maybe (pure Nothing) (\jo -> fmap Just (Aws.mkEnv l mgr (jo ^. O.endpoint) (jo ^. O.queueName))) (o ^. journal)
    <*> traverse loadAllMLSKeys (o ^. settings . mlsPrivateKeyPaths)
    <*> traverse (mkRabbitMqChannelMVar l (Just "galley")) (o ^. rabbitmq)
    <*> pure codeURIcfg
    <*> newRateLimitEnv (o ^. settings . passwordHashingRateLimit)

initCassandra :: Opts -> Logger -> IO ClientState
initCassandra o l =
  initCassandraForService
    (o ^. cassandra)
    "galley"
    (o ^. discoUrl)
    Nothing
    l

initHttpManager :: Opts -> IO Manager
initHttpManager o = do
  ctx <- Ssl.context
  Ssl.contextSetVerificationMode ctx $ Ssl.VerifyPeer True True Nothing
  Ssl.contextAddOption ctx SSL_OP_NO_SSLv2
  Ssl.contextAddOption ctx SSL_OP_NO_SSLv3
  Ssl.contextAddOption ctx SSL_OP_NO_TLSv1
  Ssl.contextSetCiphers ctx rsaCiphers
  Ssl.contextSetDefaultVerifyPaths ctx
  newManager
    (opensslManagerSettings (pure ctx))
      { managerResponseTimeout = responseTimeoutMicro 10000000,
        managerConnCount = o ^. settings . httpPoolSize,
        managerIdleConnectionCount = 3 * (o ^. settings . httpPoolSize)
      }

initHttp2Manager :: IO Http2Manager
initHttp2Manager = do
  ctx <- Ssl.context
  Ssl.contextAddOption ctx SSL_OP_NO_SSLv2
  Ssl.contextAddOption ctx SSL_OP_NO_SSLv3
  Ssl.contextAddOption ctx SSL_OP_NO_TLSv1
  Ssl.contextSetCiphers ctx rsaCiphers
  Ssl.contextSetVerificationMode ctx $
    Ssl.VerifyPeer True True Nothing
  Ssl.contextSetDefaultVerifyPaths ctx
  http2ManagerWithSSLCtx ctx

interpretTinyLog ::
  (Member (Embed IO) r) =>
  Env ->
  Sem (P.TinyLog ': r) a ->
  Sem r a
interpretTinyLog e = interpret $ \case
  P.Log l m -> Logger.log (e ^. applog) l (reqIdMsg (e ^. reqId) . m)

evalGalleyToIO :: Env -> Sem GalleyEffects a -> IO a
evalGalleyToIO env action = do
  r <-
    -- log IO exceptions
    runExceptT (evalGalley env action) `UnliftIO.catch` \(e :: SomeException) -> do
      Log.err (env ^. applog) $
        Log.msg ("IO Exception occurred" :: ByteString)
          . Log.field "message" (displayException e)
          . Log.field "request" (unRequestId (env ^. reqId))
      UnliftIO.throwIO e
  case r of
    -- throw any errors as IO exceptions without logging them
    Left e -> UnliftIO.throwIO e
    Right a -> pure a

toServantHandler :: Env -> Sem GalleyEffects a -> Servant.Handler a
toServantHandler env = liftIO . evalGalleyToIO env

logAndMapError :: forall e1 e2 r a. (Member TinyLog r, Member (Error e2) r) => (e1 -> e2) -> (e1 -> Text) -> Text -> Sem (Error e1 : r) a -> Sem r a
logAndMapError fErr fLog logMsg action =
  mapError fErr $ logErrors @_ @e1 fLog logMsg action

evalGalley :: Env -> Sem GalleyEffects a -> ExceptT JSONResponse IO a
evalGalley e =
  let convStoreInterpreter =
        case (e ^. options . postgresMigration).conversation of
          CassandraStorage -> interpretConversationStoreToCassandra (e ^. cstate)
          MigrationToPostgresql -> interpretConversationStoreToCassandraAndPostgres (e ^. cstate)
          PostgresqlStorage -> interpretConversationStoreToPostgres
      localUnit = toLocalUnsafe (e ^. options . settings . federationDomain) ()
      teamSubsystemConfig =
        TeamSubsystemConfig
          { concurrentDeletionEvents = fromMaybe defConcurrentDeletionEvents e._options._settings._concurrentDeletionEvents
          }
      backendNotificationQueueAccessEnv =
        case e._rabbitmqChannel of
          Nothing -> Nothing
          Just chanMVar ->
            Just
              BackendNotificationQueueAccess.Env
                { BackendNotificationQueueAccess.channelMVar = chanMVar,
                  BackendNotificationQueueAccess.logger = e ^. applog,
                  BackendNotificationQueueAccess.local = localUnit,
                  BackendNotificationQueueAccess.requestId = e ^. reqId
                }
      federationAPIAccessConfig =
        FederationAPIAccessConfig
          { ownDomain = e._options._settings._federationDomain,
            federatorEndpoint = e._options._federator,
            http2Manager = e._http2Manager,
            requestId = e._reqId
          }
      conversationSubsystemConfig =
        ConversationSubsystemConfig
          { mlsKeys = e._mlsKeys,
            federationProtocols = e._options._settings._federationProtocols,
            legalholdDefaults = lh,
            maxConvSize = e._options._settings._maxConvSize
          }
   in ExceptT
        . runFinal @IO
        . unsafelyPerformConcurrency
        . resourceToIOFinal
        . runError
        . embedToFinal @IO
        . interpretTinyLog e
        . failToEmbed @IO
        . runDelay
        . asyncToIOFinal
        . interpretRace
        . mapError httpErrorToJSONResponse
        . logAndMapError postgresUsageErrorToHttpError (Text.pack . show) "postgres usage error"
        . mapError teamCollaboratorsSubsystemErrorToHttpError
        . mapError toResponse
        . mapError toResponse
        . mapError toResponse
        . mapError toResponse
        . logAndMapError toResponse (Text.pack . show) "migration error"
        . runInputConst conversationSubsystemConfig
        . runInputConst e
        . runInputConst (e ^. hasqlPool)
        . runInputConst (e ^. cstate)
        . mapError toResponse
        . mapError toResponse
        . mapError rateLimitExceededToHttpError
        . mapError toResponse -- DynError
        . interpretQueue (e ^. deleteQueue)
        . nowToIO
        . runInputConst (e ^. options)
        . runInputConst localUnit
        . interpretTeamFeatureSpecialContext e
        . runInputSem getAllTeamFeaturesForServer
        . runInputConst (currentFanoutLimit (e ^. options))
        . interpretInternalTeamListToCassandra
        . interpretTeamListToCassandra
        . interpretTeamMemberStoreToCassandraWithPaging lh
        . interpretTeamMemberStoreToCassandra lh
        . interpretTeamFeatureStoreToCassandra
        . interpretMLSCommitLockStoreToCassandra (e ^. cstate)
        . convStoreInterpreter
        . interpretTeamNotificationStoreToCassandra
        . interpretServiceStoreToCassandra (e ^. cstate)
        . interpretUserGroupStoreToPostgres
        . runInputConst legalHoldEnv
        . interpretLegalHoldStoreToCassandra lh
        . interpretTeamJournal (e ^. aEnv)
        . interpretTeamStoreToCassandra
        . interpretSearchVisibilityStoreToCassandra
        . interpretCustomBackendStoreToCassandra
        . randomToIO
        . runHashPassword e._options._settings._passwordHashingOptions
        . interpretRateLimit e._passwordHashingRateLimitEnv
        . interpretProposalStoreToCassandra
        . interpretCodeStoreToCassandra
        . interpretClientStoreToCassandra
        . interpretTeamCollaboratorsStoreToPostgres
        . interpretFireAndForget
        . BackendNotificationQueueAccess.interpretBackendNotificationQueueAccess backendNotificationQueueAccessEnv
        . interpretFederationAPIAccess federationAPIAccessConfig
        . runRpcWithHttp (e ^. manager) (e ^. reqId)
        . runGundeckAPIAccess (e ^. options . gundeck)
        . interpretBrigAccess (e ^. brig)
        . interpretExternalAccess (e ^. extEnv)
        . runNotificationSubsystemGundeck (notificationSubsystemConfig e)
        . interpretSparAPIAccessToRpc (e ^. options . spar)
        . interpretTeamSubsystem teamSubsystemConfig
        . interpretConversationSubsystem
        . interpretTeamCollaboratorsSubsystem
  where
    lh = view (options . settings . featureFlags . to npProject) e
    legalHoldEnv =
      let makeReq fpr url rb = runApp e (LHInternal.makeVerifiedRequest fpr url rb)
          makeReqFresh fpr url rb = runApp e (LHInternal.makeVerifiedRequestFreshManager fpr url rb)
       in LegalHoldEnv {makeVerifiedRequest = makeReq, makeVerifiedRequestFreshManager = makeReqFresh}

interpretTeamFeatureSpecialContext :: Env -> Sem (Input (FeatureDefaults LegalholdConfig) ': r) a -> Sem r a
interpretTeamFeatureSpecialContext e =
  runInputConst
    ( e ^. options . settings . featureFlags . to npProject
    )
