{-# LANGUAGE OverloadedLists #-}

-- This file is part of the Wire Server implementation.
--
-- Copyright (C) 2022 Wire Swiss GmbH <opensource@wire.com>
--
-- This program is free software: you can redistribute it and/or modify it under
-- the terms of the GNU Affero General Public License as published by the Free
-- Software Foundation, either version 3 of the License, or (at your option) any
-- later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
-- FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
-- details.
--
-- You should have received a copy of the GNU Affero General Public License along
-- with this program. If not, see <https://www.gnu.org/licenses/>.

module Test.Wire.API.Golden.Generated.UpdateService_provider where

import Data.Id (Id (Id))
import Data.Range (unsafeRange)
import Data.UUID qualified as UUID (fromString)
import GHC.Exts (IsList (fromList))
import Imports (Maybe (Just, Nothing), fromJust)
import Wire.API.Asset
import Wire.API.Provider
  ( ServiceTag
      ( AudioTag,
        BooksTag,
        BusinessTag,
        EducationTag,
        FitnessTag,
        GamesTag,
        GraphicsTag,
        IntegrationTag,
        MediaTag,
        MedicalTag,
        MoviesTag,
        NewsTag,
        PhotographyTag,
        ProductivityTag,
        QuizTag,
        ShoppingTag,
        SocialTag,
        SportsTag,
        TravelTag,
        TutorialTag,
        VideoTag,
        WeatherTag
      ),
  )
import Wire.API.Provider.Service (UpdateService (..))
import Wire.API.User.Profile (Asset (ImageAsset), AssetSize (AssetComplete, AssetPreview), Name (Name, fromName))

testObject_UpdateService_provider_1 :: UpdateService
testObject_UpdateService_provider_1 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "\1063848\1058496N\60570\1012058\STXsuu\1092854\&1\162919YHuo\53977\1086580\&7\1017095\r\a%O\12682J\DELs\1071215\\\a-A\52293E\EMWz(]I\127069\128359t\SOxRk\ETB?pl\1061595\DC4\184777z!Xre\a\996175K\CANG,D&F\50982@P\97916i\DC22~j)g\23602I~"
              }
          ),
      updateServiceSummary = Nothing,
      updateServiceDescr =
        Just
          ( unsafeRange
              "\a+/z\986763hI}/\DLEkB\1059760\1062810\25608\SYNYC}\135783\EM\SIa\r\43515\ETXh\ESC\17873\4008\SOHCUoR[\160369c\78378fbtV4\DC4\60971<\156071\ACKd}\DC3\164303\541\186277@\DC38|\174852\ENQg\SOH#\1058252KO|\174852\1008939\166018\FSm<o_\v\SUB\166096\96231\1096650))\EM\DC34\33209m\US\990990D\120805=qP\vh>\a\1053371\1036429\1056854\20649;~k\GSP\NAK\ACK\175008s\1051918A\150295\ESC\NULpY\1054181\26848\EM\1078098T\1011719\992748!W\EOT\SO\152351\v]\v\ETB\98006N\1097932\143101\9071\f8]J\14943\SI\EMY\29869p\NAKvk\99744\1017040\176615\998969\STX\151238q\1035677\RS\v\1030236\&6\f\SUB\DC4\\\SOHw\DC1w[\DC3\1103346r+\983054/s\10708\995966\CAN\DC3~/\SO\1039052\1022548%F\DC4h\1000751\78726z\EOT\1015388\ETXdt-b\157874,DzM\1008898G\1039612\16538\1074902\DC2\18234\16087\&0-pE42\t:<\66329}[\ETX~lac\42511/\a\151380Z,(\\\1077666\127957GM\190643#\191090\SOH+%\1084834\STX\175168|\1007726\\\28896\EM\51660\1094971\a[<P\a|vd\1113993\1019540!F>\57676\1023212\1053746\f2@cczh3?`\n$\STX\1094726\EM\fN\180929z/D\179845.<Y\1055530(8A\DEL\NUL,$s\1012010\1025418\"j}\\\1035795\aEc-z\1074412\EOT\60344\&6PEz\991906\180000\aH\DLE\180900) \SOHb\SOZ\1089916\48136\187779\1024738\172248v\8122\1098449>(]g9\3442o\STX%{w\1075429\&4m\STXF\ENQ\49942\16059\CANm\DC1\ENQ/\ESC\42264\1028339QM\991027\176346'43\36345~\t\1036026!\v\14557`qY\1088128zm\DC4fRZGvL\ETB(t\1007154\SYNswr\145599\58315%\1043578\NAK%\1082059G\1691l'\ACK\1029069\137530\170139\149719 \8297\NAK\f>@\40665\1029420\CANu\STX\143750Y\GSVj\DC2\t@O\184863\44709\&4\rf\b\1002476\r_F\DC3\NUL\47001\ETXX$#\t\1093906\ESC7\EOT\b\983099\143369\SO\ETB\EOTA\185268\159378\1015274/:N\DC3\1068202\&1D\96979\1042904 V\DLE\SUB\1087165#\20680\1005166\&8\ETB\a\DLE\RS\995866\USP\ETB\SOH7\r|L\145137R j\ESC\SOH2F>dV=\EMr?\1046227\119883\"&\DC1O\11375\SODuQL$\1032099d\US\157568` <1\\O\445\993915/H\f\r\143532Ah\1032005\ETX\162288uu.lf\1057288/1\1106120\1028078/\7411\138984`9\bq\SUB[Z`\118961eLNyTq\1048960k?{\nWg\72112\1100487\120674q\151928u-\DLE\1008080(\DLE\DC1f\127138\ETB,\rP\7088\&4V\40697\159724(7)..\70295$\n\SOH\78896\989166\92348\134295\FS\5319\47941/0\166710:\94593\SI{]$&\1074979m\1114097\&0\144077\&7)\183400b5f\SYNGyYxU):\1015140L\USQd\121515p\ESC?<\DEL7\DC4W\ESCN\45294;\a\987395\NULm\143966K\ETX\146218\51248\ETB\17306\"\987854*S{G\349r\1010831g\DC4>\NUL\SOH\97274i\NUL\NAKk\ENQK\20758r\1027971!rE\t^\78529@|h'0F'\1037224\157621\1023969\&9)\SYN^\ACKm^\STX\1078787M]\181147R\12517+\1015063^p\43086\&2AzeS\DEL`\141901\DC4\985596\182797e\ENQ\CAN\ETB\36060h=0&kp4\ETB\1023228c\999060\ENQ8$\STX\EOTk\t\CAN\173228y]M\bA\64661x(\STXV\fT\vOO=\1086015`D\1031911i*c\1010700g3\RS\998099\FS\fr\7033g\181534MX\15333\136960\43015x\1089585Rz\154544(P${\98672\DC1*~e\n\t0]z\DC3\EMY\173001\1112133g\152066!\182207@\ACKp\162647\1015149=\62520X\1013875r\65890\1025377\&3u\t\STX\SO\139037n\DC1`\42999;,\DC4\161373D.:\SOH"
          ),
      updateServiceAssets = Just [ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete)],
      updateServiceTags = Just (unsafeRange (fromList [WeatherTag]))
    }

testObject_UpdateService_provider_2 :: UpdateService
testObject_UpdateService_provider_2 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "\STXN\DC3\SUB0|`\137114s\bQ`\ETBzT\r\12429w\142571\22755\1099860!qW\t\t#SL\a&\1037091\SO\1069790e\2540p\1103456\147592\\\1050963\1089998\1010692\r%\43704\EOTt:\1008112\ETB\30175\996623\SILX%7\DC2\34673]\DC3\1111107\DLE\1074098C\FS\SUB\36112\1055143\&7+\1082886\SUB{tV\1047873U@\1010551^)e=\SIr\RSs\RSB3^\32573FjKv.\SUB\168227#"
              }
          ),
      updateServiceSummary = Just (unsafeRange "jTPNu\NAK>\190180s\f\SOH\\kh&A)'&\r1z\36112V\CANC\1095467O"),
      updateServiceDescr =
        Just
          ( unsafeRange
              "o\21030\DLE\DC1K\64697Ip\1108150Isa8/g5\1055120\EOT\NAK\SYN_\ni\1000676\1009051^\SOH\DC2\997677\169427r\SYNT?\142212K_f\1053291\ETX\STX*Z\GS~J\29276\EMh\\\STX\ats\DC3K\1052688a'\983809\74116q@k^S\"\"\19966\&2|c+h\aY\36284U\fdl\153592\131311R\ACKd\1014482\191274\149396\1096141`z\120878o\179018\37963\165224\FS\\\a\EM:p\1094161\&18\1096545/(e\1092652_\148559\1043115\1107436aR\133398+T\ETB\69802\STXa\DC47(\28300pk\990841\&9\1111846\&38\1002472\1026110|\1039857\&9\999499\150656\ETXZ_\1030337\99791n\135936c?\58137L\"\NAKF0(0\1083372\1012104b\171128E-t\ACK\\h\1011109\94802\ETB\vAO|\1097854,s\DEL\67328\EOT\1001529N\\\SYN\"\FSJ\r\SUB\EOT\GS&#\1027667w\149351\b?V\EOTY2\138832\DC4\143450\SYN\bc\1052292\181139\120148\SI0\46398Y,CUC@\terx+KV*K\144736re\1073688V\144579\DC1m#\1006772\GS`~\FS=.\1072947\47425\EOT\STX\1029422\&8t\DELF}{ST\n\EMR\EMR\SI$\1014312L^Q\71320\bE\\\165627~\STXjf\1094043Qz5~\NAK&~\995302\FS\1002450\FS\DELq\1004490\45485\997645y`\f)#\194809\ACK8\1052667Q\DC1P?\1085102\RS\22262se8p\ENQ\SI\DLE\1031405\159290!\1102515\SUB,\DEL\r\US!)\190723\33191}J\1105937\DC1t\52309q1\DC2\SO\7093\ENQE\EM\1057212S;\1105115!xW\1102848\51682\&9\1025741,\b/\989647r2\t\1043051\n\SOH\1068524\162395\DC1}-\SYNcb>6kgG_J\"a\DC2\145957L.+Y\EM\996655?uM=B\STX\a<\n\1018349x@K\1104898\n\1098684SI\SUB\ENQ\DC4o\1061910\ENQw\bV1_K\RS\DC1\DC3,&c\EM:\64417\71913\&3\DC15\DEL!gV\1068202\&1\1036070`\DC2DC\999634\\\2268\RS\156958_\54450Fr foM<\tJ2,6V\988676\1051319\1053468)8\1007703\65741[T\STX\SYN\190556\1093656\ENQ\26039,\ENQT\1006284RufnCntu\1061486\119539f\ENQ&+2\STX\9779c\1106722\a\NUL|1/\989311\&1\FSy]Z\15860}\NAK\1033164\\p\US\1016286\&7jl\1094653T\97198\fY\RS\1000399X\119587`\178025\1065029\996921\SI|\95878\n\33618E\19236\STXY\"0\DC4\ETX=\149257\165419D\992027yfQF\b\1047505\NAK\994364}\140947snsB\DC2\tM\1055570\US\1072934\DC3\ENQ\NAK+<(:P\994057EQGZyHn\1039088\GS\ESCu\1012363\1047153q\33854\986964|\1082886\1022504\1089730\150596\52154)\185867]KX\DC2\DC3H78\ACKXPM*z;\STXf\NAK\bFGIY\ACK&?/\SOS}\40218\1004837\59749p\fqA)T\NUL\165146&#A\179199@]\US\DC4Ya,B\41011\t\1020776\27144D0\20707S\SOH+\1018754&PgJ\b\181353\101073Di\ESCU \US \145617\NAK\131760\179896kL\n\NAK.\DC2&\1091162\DC263RX\36291\183270%,\DC1\1044938\SI\54697\tc!?r\1068674\"#=p\reV\22408\GS\1030567U6j^4W\140886\EM\SYN\t\14808[;\ENQ|P)\\\US\997036]+&\2024\1060157\161435>f\DC3PB\1092496T\37640\SOH\EMTn\92579\&0\166723^6\58654.\1064893\1055714~\ACKB\64515\333u8\42685\DC4\1101449\SI\1077460\rA\SYN\1110943\&8@&\CAN\1000775\987986l\NUL\1077027UC\184091\1113351\1042978p\CAN`\ESC\RS:\98861~\NAK\999928X!]\DLEq\1021452.\156066\1036515#A\17161`p@\129595\2748\DEL\CANsOS\f+<s\DC3<`0h\FS\1038433\986169\SI\1019205\1095612\"Mr\DC1)m\111244\SO\171945`B\f\154306{\158605\CAN:i*9\STX\1096241k\b\989092{\1096625`Tpp\1105244\DC2Q\ACK\1029586\1101473\1023732\ETBS\1101331!3\1104403l\41547u)\f.Y\160094\NAKz\120481\170940\&5:I\r6,\SO$\29388\1037152{\1082002(\29228\74048\n(\67722\1006157\99086Qz,)\1040498\1060142o\1079156 \990392\59711\ETX\170155*p\\?7y#\131431\1048634\15861N\SO\998562\\\a"
          ),
      updateServiceAssets =
        Just
          [ ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing,
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete)
          ],
      updateServiceTags = Just (unsafeRange (fromList [AudioTag, EducationTag, WeatherTag]))
    }

testObject_UpdateService_provider_3 :: UpdateService
testObject_UpdateService_provider_3 =
  UpdateService
    { updateServiceName = Nothing,
      updateServiceSummary =
        Just
          ( unsafeRange
              "t\95902j:;\1005683\1019495\n\998095\RS\DC3\EOT<\SYN\1094518xb\18550\&3\74865u@G\997755L\1062992ar\148403\1025099\DEL\11185G L8U\rauh|J\132432\15362"
          ),
      updateServiceDescr = Nothing,
      updateServiceAssets =
        Just
          [ ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview)
          ],
      updateServiceTags = Just (unsafeRange (fromList [GraphicsTag]))
    }

testObject_UpdateService_provider_4 :: UpdateService
testObject_UpdateService_provider_4 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "\96293\154585+\ACK\179670d\1081072\&4\100326N\a-P\1010896\ESC)\r~\NAKO\EOT\SUB.\DC3\n\1055252\r\1104179\&0&\v\STX\ap7\ETX-\EOT~\1087927\RS\NULeB"
              }
          ),
      updateServiceSummary =
        Just
          ( unsafeRange
              "s\189333\182796\1004409\&7(\163009><sm\185694\NAKvK`\f\DLEo@$>n\GSFQ\NAKL\DC1Scj5yq9+\rj\ESC\1064740!_'9Q_}0WG.jJ%BrYQ\1012431T16{B\142983\125016d*V\1024914gYu\162093\&4^b6x2y\148176>[\RS"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "<\1090124#FE\1086106s*!\62593\DC4;\31772^WMr\1060834\&8RB\NAK\128903\1007550$\t,C\ETX0\11070\1023381\58817\27286j\\nF\175225W\1113162\&7\SO@\94549w\ENQ*g>=-m+\128253\997485JpQGB\1044309\&4\1060466\SOH!'w*M;c\ENQ\98836\1003286\&3)R\29851sZVy\DLEV\ETX\144137\US\EMJ08\DC2\\\ENQ\1081494\1001187a\1018101$\SUBt\181563\DC3f=\141465%:!\\6\172907\aES\1016438;|\67631\1046123*\32113@1p*Y;uGE\1069430e\1102664\f5\SOHWA\ENQ|\SOH\ESC\1009746\&4:*}$7]Z{/*\DC3`\STX&\155842P\t\1053171N\SYNRL&\SI\169000\USs\162298c2t\NUL\SOH)\26500\&2/rm\1051265wkD>}\1070334\NUL\DLE\128068\178727\&1%\1005755\ra\35525J\13316\19695,\1056622\nU\NAKY\1011081\1058839-#!\SYN3\190953\83058z\ESCl!`\DC3e\1102400\t}GW[P\ESC\1004676\189533[\1061401\ESCJF\21715\&9RA\1068756\"\t7[\1111740\n5\NAK~mEU<\nL|)&.Cu5T\121142 y>\9286$^\45932"
          ),
      updateServiceAssets = Just [ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing],
      updateServiceTags = Just (unsafeRange (fromList [BooksTag, BusinessTag, SocialTag]))
    }

testObject_UpdateService_provider_5 :: UpdateService
testObject_UpdateService_provider_5 =
  UpdateService
    { updateServiceName = Just (Name {fromName = "\46107\95998^Q\vLMRojx{\DLE\SOH)m\33573o\34179m"}),
      updateServiceSummary =
        Just
          ( unsafeRange
              "\19119J?%\1084843M#pwC\ENQ\1025817\1093783s0`H\16376\1026040P\1078447j\EM)<\DC2;\b\DC4Ei\1099715\GS\a\1086578m\1029214\148889LZ\14040"
          ),
      updateServiceDescr = Just (unsafeRange "\ETBI\\.z\96610\CANQaIC\1065269\32625\36609k\1091140J\SUB8/\110715"),
      updateServiceAssets =
        Just
          [ ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing,
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing,
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview)
          ],
      updateServiceTags = Just (unsafeRange (fromList [MediaTag, ShoppingTag, SportsTag]))
    }

testObject_UpdateService_provider_6 :: UpdateService
testObject_UpdateService_provider_6 =
  UpdateService
    { updateServiceName = Nothing,
      updateServiceSummary =
        Just
          ( unsafeRange
              "_\1094618U(\DEL\f\ACK\ETBmdi\1049181@\1039076`\DC2B\DC3NSo\STX\161763\53727fRY)\1056987\SUB\CAN:G,N.1*\f\SO.\r\CAN\1086860\147284\98968\37059\1097556\17182\&10lK\1007748\&9\163147\n\ACKsm4\DC20\EMGN\DC26rz\27059dO\r\"\1071786{\ETBc"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "f\SI4 \1063170|\995839;T\139513E\NAK(Qp!X<#\ETBA\NULuW\44248cis\f<?\39808\186109\985064>=~C\1732\1027485N\161808S\SOH\988099;\EOT2\fA&\187694@RHN\1011941\137440\NAK42!#qAM1I\tu\120271\b\t\19488Q\ACKDi\127780tX\990666\1103592EI\SI\ESC\bK\GS\NULo\1044109k\DLE\187241\1005849Z\CANI\10594l\1044875\137688jg]\SUB\1100178\1078023 +e'u1\ao\175647e\US1\t\9732\9316\&0-d-UJTP\1092036W~\184365\&7\1098050tly\1087376\46624Ozw\tH\nW\1062958d:E\NAK@\DLE\1086957f#=\97609\&1\61954g!]\1051221\1055847pz\78590OA\1056922,\\xDL\CAN\1073075\SYNeF*s_/\f25 \1088055\EM\1053116\986882Aj5\74938\DLE\12992eDbG\SUB`\66727uW@\6764\DC32q-pq\DC2%j\ESC\EMq\993522\153753v\ESC/\1050068|\DC1,\DELw\ETX-\25497K\1048380\US\n:\98876\1102356\RS\142008\1050738 4\93016MxyOMq9~c\1082301\1028090!\RSQ\30115ql ?>\ETB\149698>(\EOT\t>\20400A\1079649/c O\59065]\ETX>}\NAK\1071442\75027\ETX\1048970%g\ESCWc\153028B\171118\ESCc!Aq\1045328a\7285\180743\155835\96854\167241\175754\46512\DLEas;\13803\1026445Z[Fs\180513*m\SI\n\DC1\t\155458ML\nX\tTD+\SO\1107343]a3\1082869&i\1000299:X\CAN\1001282s|\az-\1098006\NUL\187905\CAN\CAN/\ACK@v\150658\1010455^o\191090$+k\EOT)>\FS"
          ),
      updateServiceAssets = Just [ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview)],
      updateServiceTags = Just (unsafeRange (fromList [MoviesTag, NewsTag, VideoTag]))
    }

testObject_UpdateService_provider_7 :: UpdateService
testObject_UpdateService_provider_7 =
  UpdateService
    { updateServiceName = Just (Name {fromName = "\29862{/\SIW\ESC^ V`uC"}),
      updateServiceSummary =
        Just
          ( unsafeRange
              "-4\159289yV<\99237I\RSO@\DC4\"I](\rb\\\DC1x\11441<\DLE}3\CAN\1020838=\15343\166658cG]\1005086T1,\CAN\ETB\ETX?Hzy'?\57749?\ETX\143191\1008856I\94398\164055\&7\138557\&7A\18036`D\DC1\SYN\188233\SYN\"\DC3\ENQ\STX|'Jn\22835q\ENQa\ACKxP-\ENQ\36051\&4\NUL\59821\NAK2\29219j\DC2jet\FS=\39971$\DC3\ETB\b\SI-\187658\136084\SI1L\t,A\6239\13394\DC2*mG\53510jC\SI{\vu\b\119843"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "\a\26154;\67699u\138410\GS~v\SI\DEL\1109985\vm\1010621\&7Ff\36362\121032/:\SI!K/K\1010585H\"C\CANH\SYNu{\35999,p!;y\1055119\23628P\SI\1062219\FSC\1047702Hi\63662>q\1108471v\GS\161843\&1i8\SI\72854Yq\64555u\68821_\1093939\&9*\SOa.b\DC4t\ENQ\1104041\v6\55076\147462H6p%\NAK:yW9(bG\15482FZ2U';\1106763,\r~OSXi\\YulJl\52863\&7\155864\GSUVU>\1044094\DLE\1075764\1109098\1080043\1095928\&486\125253x;Q\SI\GS\EM\1025934\GS8\"\62770\139325\DC3\SO!{\n\1060654\GSS\170220f%!?\988710\1039100]\DLE\1098328\1105972D\33656p\SIM\1018815\EOT\95053wQ1 \1100293fR\ETX\24310\174800'-\SYN2\157307SY\19751|}\1113537O\161420-CuT/\3796\1049823M\1055672\ETX[\31140\146644H =\RSM_\1087467b\157548\147705:\50119\7843@|\1103637\US\166272\ACK\SOH\97942\1029888<2\45785{v\51495NPJG\19974g\DC1r5{\1078138\1015695\159216o>\b\EOT\1102944H\1076211W\f\EOT<\DLE4W#RP\\4\NAK4b\SOH\EM%~t$\DEL\1073553\1085016TML)\"I\1087534\42520eD^\1075693(\1019426\134232\DC4\ACK\1109118\SO\t\DC4=\1031053m8\fT\1010830B\175007\1028900\ACK\SYN>5\145041\&4)e+\144223(\1043067Fr\1031244Jp(}TN\DELLO4\NUL?cL\STXe_\15541`~\17501s1R]x\STXi\EM\a\CAN\ESCaz[*\GSFd\1029229\&88\1022821]H\NUL\994641\176548$\1065310\992986\EOTVO+\1102607\29854\1050003\&2s\988600+0\154823\RS\US\1032112\68060\54239&en\57424\13192e\172459w-X\NAK)\127300x(g\1008852\SUBGX\63001\tOh2g'CYU&0W{47\149544\SYN\49323qOw\ESC\NUL\1040179=\NAKO\31362_\33987\&35\US"
          ),
      updateServiceAssets = Nothing,
      updateServiceTags = Just (unsafeRange (fromList [WeatherTag]))
    }

testObject_UpdateService_provider_8 :: UpdateService
testObject_UpdateService_provider_8 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "r\1045358x\ESC03\DLE=\1071672#\169286\ETX~2v`\1011504\ACK%<8\1087017<\"\1025763:\t#5c\1005374\44360\&1K\NAK\1014575IT\1108933G\n\SYN*m\NAK\1039026\&7\1000965s}^\64349\&8x\SOL\1041183\SOv\1066153\DEL[$\7792\997332jC\999909>Sk+\168546,U\1097221o:?\DEL(\EOT\1046620I<\156441\45804\1086047\EM:\165474QgrP\SYN>\SO^,@DL\1019368e8\1060494\&3V\1085992i\DC34IjT\1099033T\1027630\ETX\DLE\RS\1030713\DC4\r"
              }
          ),
      updateServiceSummary =
        Just (unsafeRange "$\162083\ETB\ACK+\47676k\12880O>Gk\177476)\1099950iu\1064783U$\1009529\&0T:K"),
      updateServiceDescr = Nothing,
      updateServiceAssets = Nothing,
      updateServiceTags = Just (unsafeRange (fromList [BusinessTag, PhotographyTag, TutorialTag]))
    }

testObject_UpdateService_provider_9 :: UpdateService
testObject_UpdateService_provider_9 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "ur\993568$[9\r88\f\FSPI5\DC2\1037337\&3B\DC1~\1032391H(\SYNZl\74111T9*\n\1045336Y\1051597ua\1046841^qVb\983094\NAK2S=(Zdm<I\127836\1082751\74842a\DELiBT\38777n\ETX\ESCTnq77%\NAK\a~5\f\1045765s"
              }
          ),
      updateServiceSummary =
        Just
          ( unsafeRange
              "\12394k\38793\RS\1009001ww.\FS\1053343V\1078154K?v\v\63865&\148617E]V\139815\171295\21067\EOT]&f\171568/\1085432`\1052948\fE(\DC19\7765\NUL\CANufg\1045430\25893\54445\183344(\152586\\\5194#\EOTQ\183544\21872\\\165101'K1\NUL>6T<0\46721\1023223Kl\NAK}p`\1107647g\1078667\t\tG\180583\985664\1113503b@4};\145570u\GS&E\48732"
          ),
      updateServiceDescr = Nothing,
      updateServiceAssets =
        Just
          [ ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing
          ],
      updateServiceTags = Just (unsafeRange (fromList [AudioTag, ShoppingTag]))
    }

testObject_UpdateService_provider_10 :: UpdateService
testObject_UpdateService_provider_10 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "U|2\1053864E\NUL!FB\67597mX\NUL#=\1102474?\STXP\ACK.\GSJ4fN(}\EM\DC4\1079510\157411\\<\185110\DC3\SUB:\SYN#Q$\US/\v\RS\EM8\41007+\n"
              }
          ),
      updateServiceSummary =
        Just
          ( unsafeRange
              "\SO\1051666-'Y[Pox\173542\&6K'/b9%z\17135/$\1105355DY U\SOHx>;\EM\39523<{6\\\ESC'Xa\149813M(+\STX\146289q_\1030102hrlyS\1008039\1026970C\147877A\DC3b\1017913\1032820;\DC3\135639\129342$?+"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "oop\\ ]\1063455Q\1022631A\40380Fk\119065PP\1030172\1050289\1052815)@$\1042136\r\tc\1031217\b\1109726a\147334.]t\151607L\100966\1032763\a\EOT\132504r$TK~\1068360=\SOH\n.A\52198P:D\50946CM\ETX\60079c\SYN#\1074978\USV0\NAK48\150131\GS\1113019\1036275\&8'\ENQ\1060728N.\1033676B=m7\10763\NUL,\SYNm\1038409S\1045781\&5dd\46494\b\EOT\29227TZ\b\41520j\49693CBti\RS\CAN\ESC\149078T(Zh,\1100756\tO-Cc\19665\f;\142884\EM\163893fvGA$\1041950+zH7>\b\b=(\1056532DT\1112807\US\ETXm=fA=\1034419pN= \139734\NAK@\DC4\DELnJ\NAKz\166051\1106374,Ih\v\n\1028865\1073490\99357\\p,|}\ESC\1004263,,\SI\SIJC\142111(M5v\158392\1062197a\39931\12181\161230\&4g\39661nH.\1102694\64584bNU)Bk\"\t2Gl:\182360\&3Ip\1071234\&2#\46393HX*\a\983649\&8FUM\SYN\b\984489Eg\1094199\GS\165653Uwq\1046727A\23062x\16202\176884Uz~\164985Qe PB?Ogz\SYN\tP\ESC\DLEc \40033\RS\98706!Wj\NUL\v\1020886\&6f~t\f\154636yw$\1017151!\140564\t\STX\rZ\67311\1095252XF\154859\ACKv\DC4\t-1\CAN\136783\ETX\1052809\DC1\DC2g\1105517kn).\1108890AN^f7!\1108290-\1039969\NUL\1005207\ACK\1012876\1036512T\SYN>\23399\27740Q\52604\&5\1042902o\v]lS\NUL\1108911n\18260%.F<-l\CAN\1752\ETBB\DEL\1065762#\bqF\STX BE\SO\GS\STXl#PHJ\bL\SI\1075788\r\168655\29622W\990693\1062382d\SUBJ\v\1048702N\SOHB?\169870{\1053009:oR\1106540\&4wEi|F6Mq\42723\f\26372V>\1002299\1068476~~\ACK0\1085343\989482C\177471\n5\"\184635\EMX!\NULF\17509\188142\&3#\997079{\ETX\97520_\1092563\38436\15776\SYN\ESC\SOH+\156844y\n2\ETB\\\a\CANc? ;J\1069250\n\67711\1098737Nu1\1025808d\162562K\CAN5k'7\DEL\ay_\18811:\1041011\94696_/\1075835\n\987026t|ef\1080192(4tkk\ETX\1056739o\r=\EM\175596\&5\1019418\&5$\DC4\1046254_r\NAK\EM-H\1045481M`N0\1121\r\DC3\143480\SOHa\181965\131291\&2xL:\184619\986801j\SOH\ETXD\SI\aiHTQnp]\1050206\DLEjta\STX\ETBM4u\1057523v\154744)\43858k\1008991'\1070093\120267\1796\1034554\NAKdc\1042432\1061068t\178667\NUL\\}+^\"~\72875>\RS\2437Gb\25804o5\SUB).\DC2uY\154957 \SUB?#\SOH\tMz>\1021260\162067\1080768\1051271(}~\1026519%\DEL\DELdz\39513Cn\STX\1060100;\DC1=\20280ka\EMG%\ETB~Qe pG\1089669\ESCk|\32218*\DC4Y\6957!\DC3\t.F~\DLE\b(-P\136218\RS\1107570\1053070\ACK@\to8sB\"h\DLE\166866\ETX. LK?\83487\24765,\SYN?\147160_+m\EM\12850\"P\NAK\DC4\ENQ%\63594g&\ENQ\1084345g\GS\v\SYN\159843\55170\995590K\DC3\176841\\@\SUB\\\60150\ft\1077492M+D\SUB\1011659C\178007,\DLEn\119947X\v:l\143677\&3se&z\995136\27106\1102641\fp\1085692\141778D.\b\ETB@\1028055[\983628\184294X\993326<C$nt\n\ENQ@\ETB]c$p0|BC+\97389\SOHf6xb-:@m<\179209\50180\1026498\37805)\f[cCM \1016930\DC4\"m\1059479\&2\995115\49529\97179#\SUB\1095028\ENQ\1113387\fEjz6\1003821=wHD\166395\989728\1055007\38802\1100783*\b\33036\160259\&24osq\SUBe\1046100Y:#\n\RS\DC1\DLE,\EM\CAN\1057415@\v\984598\142779\1037699\ETX4\1004311\rU\998066)t\121073CF\183581SnG\132305BA\1047384\131457z\194906"
          ),
      updateServiceAssets = Nothing,
      updateServiceTags = Just (unsafeRange (fromList [IntegrationTag, TravelTag]))
    }

testObject_UpdateService_provider_11 :: UpdateService
testObject_UpdateService_provider_11 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "\DC1\SUB/J>\136694\148449L\NAK70\46895_?\1102823\"\SO\150890\96874\ENQ\92618\1022066-\1056443\36230h\1032092=H\1087203\ACK+;$\1067085x\"!n\DLE\EM\1050445Jx"
              }
          ),
      updateServiceSummary = Nothing,
      updateServiceDescr =
        Just
          ( unsafeRange
              "%\8461@j\1003803d\vI\ESC5\DC4\v[{\1084424 o\37493^[\r\1065361\ENQ\989379'\SO\DELt\1089106\1026048o\987702\fQ\US*x~\175529\ESC+\DC2\1068746B/\52162\SI^iD#\54368\&3 \133638\154629\DC2?4\1039202O\STX\95415.\USK=#\180982\1050605\"u1\SUBVM\NAK\\\1109193PUbp\1062105)\1059153G\1094845\FS]H\988889sj@-\SOH[{\fWgn\NULD\161004CK:N\DC4 \1086998vAU.q[r\SIN\62399\aK\993185V\18062m;\997502\42898\GSFwUnU\DLE,\54066O~?\1059940\22363\1063548\1050258\SIO7\160145e\1107763\&9%<\1091577\1029068\EOT;m\1067612i2KMR\1068900~\1074983y\31693\997627\NAK\34399\16306\&2\1000541\1060948qM\t/\NAKE\94405\GS\60810g\SO\1262\1097092 ]\NAK*\RS\USn\DLEZ\1083924{N\1094983\1083682*\990011b}d\DC4m\DC2\1099334_klW\aM]\ENQ`\69845:\1075148\ACK[\DC1\DC1\38304\SYNddj-5\1039472sy\2930\a\1033736\128597I< Gn,r-\123627/\1018717:\SYN\1016378\1030411\DC3\99359*_\DC14h\7023\65393\&9]\DC1p\DLEjD\1067498;\1025962\24201dBX$\ETB\11053V[\1111493f\173225\134414\54468[A\131781\EOTZ\GS]\1031666\"\t\1082183\vwOh{m\CAN\1100553S4K[ >AT\DC3\1083585`8&=6]kr<\bud\1019072L\1100833\RS\RSYS\47398\&5:fy\187978\140412X?\159264\1093507\1004041\129521\1062584\44159\CANcA\1047698\t$\ENQGCG\nV\45160\1003477:\1049840l\995620m-\DC2j5(*Dx\94528\151704\165806\97674~M\FSk1\35205k4\EMFi\CANB1\31937\141293&\156048\f\ENQ\151296O\DEL\tN\SYN\99140\997723uV\1011107\&3MF\3047\NUL5\1049414\993707xM\178959\152671\39787\190780]}W(!FU\RS7(\134815\"j\127323\167009GLw:\1013260\SYN\EOT\99156oBsEyz\13164\156896\&9}\147880\1011497\ESC{\STX\1053127\RS5O:im\DLE\140843\v\33580\1007646t-\SYNEP<\1021646\rH\SUB!XJ\185675\SYN@-\ESC\DC4#\EOT\DLE\CAN\FS\EMM\SYN)\186273|\1085860\32691l-z\1060957\DC3J\160637=|cS\6108\9044\&1ay\EMq\1006178\vE^.7?\1004813@O>\SI]\NAK=mc\ETX\EOT[F*Pw4\998410\US\1017055\GSS\DC1d\DC4_\1081353_b\1020990_WF\31307\US\182073Ip\td\1052137C\SOH6\ACK\18560XML,;<\FS<~\DC3`\ESC]G#s\97868\70177\1060632\ESC*\fov\10033\1055124D\EM1\ETX\1015517\&2?\983473\SODlvg7\1097982\DC19i\f\SYNJ\51069\183784[\50674\EOT)h\985450$:\4307\GSA\128782}\NAK\1085155%\1002544\97957\ENQ\ESC\b=^eJ\26383\t\fcJk{\6203\STX\US\270\&1{\GS\1113078HO\NUL\110789\ACKrW\1107697|si\NUL6<\29868\SIa\b=(77\NAKFX?];t\1043849W\ACK\989698\28033\1112140;y3@Cg\1086722\&9\EOT^oN\ETB\4988X\f-V%E\1095817zjmP\1070033\160592\a#k\186757*\1013075!\DC2+\r\n3}M\ETB*m\125016\STX\134937&7$\ESC\145046\176546AC\1038770\1069314a\CAN6Ij\1015212ZV/\1093312\&8rg\1113367gEw\ETB~l\STX|_\131114`'B7)F\1072841A\151187\131364)L}\CAN\SOA(\1078052\DC2\139573?\ENQ\DLE\155611/U\160640K\"_\144691\73889$Z\\1P\1089769u\1068775e\1056643]z\NUL2{\26237Q'\54114Xp\USS\SIGg\128293!!\1045161j\ETXHFMBzD\14002b6\DC4\EOT\a\NUL+_\1076443+\NUL`k\1078887\127520\1112558"
          ),
      updateServiceAssets =
        Just
          [ ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing,
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview)
          ],
      updateServiceTags = Just (unsafeRange (fromList [QuizTag, TutorialTag]))
    }

testObject_UpdateService_provider_12 :: UpdateService
testObject_UpdateService_provider_12 =
  UpdateService
    { updateServiceName = Nothing,
      updateServiceSummary =
        Just
          ( unsafeRange
              "\CAN\1091971e\\c`_[\166499@\1085327,z.!\1026741\&6\40084\161546\173651\&1B\989437\SYN\DLEEo\EM\38562\&3aC\45204\48839m\f\EMK((\25666\141354\1083557\1076964-\EM\ENQx\1036832[\FSbLpC\ESC\7845\"DNR$\v\DLEt\13348f>\48105\184051\DEL\US&\1057184S\v-M\CAN\RS\ESC5W\1074909;sH\ENQN\ACK$7W\994944>\DC43v\1055995m?\SUB?P\ACK\n&<)L\"G\1030670]\160038A\NAK\149977\GSgAfJ\132021\DLE"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "\US\FSX;,\DC3\149563=VNF\NAK%;i\EOT\996832$k\ETBc7\SOH\143354|:d\SO\GS\RSN\10748/\"V\1021294o\DC14\1047613\54437\ESCj\SUB,\1095459<r\1027135\fL\1077296#\SUB{\1107619\ACK'\n\1012295 \DC4\1013830\64669\1101903 /\1094959Skn\FS\SO\FS\61757n4<\60861;{\t@\1062696n\ENQ-Jaq<:\CAN4?q\185499\162701\&5j73#`R\SI!\STX<p!HNe\DC3\28597\&4&\38762\&8n)/G\7910\DC4Z)\47530\992868\1019675_dM7\CANv\159913-\GS\bzD.O\162016\161169N:dDh7Q\1081381!\GS-\RSKq\SOLi\1036129[\1510\47338)Y\39200W\1062114\rAH9=p\1105268a?\t\ETX\1098572\&1N\STX$\DLEn\1083587iN0\74859\DC4R\DC3ZbO\17285\n\NUL\v\1014394u1PC\43095\US\RS\b}\EOT\24434\1113524''\rz\r\1035654(\1039979\1091031?[BW\62371|!^5\174959\1103959\186063l\63430b\n_2\SOc\176642\&0\"5\DC4l]\DC1>}i0m\CAN\31240x_ \1049571\175311Q\1022107JiC1p/[1\\A[o\51780\FS\CAN\NUL\STX+\127172\120462w\EM=\121430dH\1004989Il(#\GSvd+\69876d\anEh\1002617\nQD\\:@{\"\ETXZ\1014379i\1053082J`&;t}zQ\DC3.\1020713Co6\NUL^vvsh\51873\\a\1051720R<\SI{\NAK;%f\144785{\"\22777\&2\140005kp\ENQ\t\ETB\1112840o\97260|@.\RSX\1052971\a>\ETXek\DLE\FS>"
          ),
      updateServiceAssets = Just [ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview)],
      updateServiceTags = Just (unsafeRange (fromList [IntegrationTag, ProductivityTag]))
    }

testObject_UpdateService_provider_13 :: UpdateService
testObject_UpdateService_provider_13 =
  UpdateService
    { updateServiceName = Just (Name {fromName = "\991905\DC32\\\US"}),
      updateServiceSummary = Nothing,
      updateServiceDescr =
        Just
          ( unsafeRange
              "2aX*\1031613\142371o/WfZ(\60157\1004546**@w\1064949\16747@<o!\EM\aa\164171\FSx%\983760!34T38\1084340b\GS\983613g,\1015915\148280^5\1103562sH\150575R;E'Eg\27049u\NAK5=U\ENQa/v\ESCT\53134qMj\NAK8wz\160581\EM`\1095961?h8|#\150325\v\1022887k\GSK5\993603\FS\13579\EMQ\RS@6\93046W\RSw\1101689p9\EMy$\179817\&0\EOT\49919KIUH\DC22\51806#8\1030974)Iz\25850i\GS\"\1101653}\v\152866f;GE\1020767J\STXT\996642Z+\1026532\141808\37445k+\NAK%J\\Myv\DELv\53056Nt,\SO}\DC1`\1051200jT\DEL\24562\SOS\11443|\SIDU\DLE\FS\"\DLE\EMd?\1092513\SOH\1995\n\1110114\121480\b\1109225OE:U+o\DC4xNV\1057911F',\159716&\RS\bo/\1009870>B~I\t\CAN\SIlA?R\19515\\\187765\1044489c\ESC6\145770\a1\b`:#\SOH\DLE\SYN\14043\&0&\CANn9N\NULu\1036534\&3}Pr|\ad\135922LVno\45454de2\SI\178047E2u\DC2\\)t\1056070S\DC3\170473\35154\a\GS\SUBg\SOH\v!HGMt\155629N]U\DLEhG2\SYNE\FS\1072702?-?\1112045\RS0\8659\vA'q%-(\n\24735!dX\DC4N:^<KL\133248_9%\98422\142491\155893<l\1101719\1000503#\985525iG\1089626\46702IGF+\\`\177156\1046373\1055251\35821p\70113\24130WL\1075336\SOH\DC4imn\EOTJ=T;0Lh\1028269E8*g@(D2\1085329'\a\1041486rm\1016536+Dd>\EMt\96855[\1061678[\65756N*\1010769\SI\17596\57669\US\DC31@\t\1065299\995575izL\rH\1051262\1112099\ETX-v\31688\1019357!?\ACK87\1027898\DC1/P^tQ?\1063151?H?>W}\ETBloc\65833\167071\SI\US\186020\142679\23210\DC2 \1042537-\170181V'wY0GD8\SI\v\24585)\1006377\a%\140106\ENQ\69989\\\172578\1088583O\FS1\17363\1042040\998122\DC2{\SOHg\1075128dQUE\991139\EOTX}-\EOT:<n|Wv\45736\SI~I\99138\171908u\1016971\SUB\1075764\150993`m\SI\SOH\1005806\38042Xdr\EM\SOz\167430_\FS\993095%>\DLE\b\\\18491\f*fH\ACK}@\182594)\11785\US~\70427Yx\4262>x\144462\1067300c\1091263:f\154378\994975\DC2\DLEz\1023998N\147661\"8\DLE\1000388P%rOG\FSR\153730`F2\1023094+\"eQ`Gh\39613DvPi5b\SUB!m\DELm\1086370\145735^\1089383\&9\1094246\33130\1005542\NULD\f\FS1\FS\"#q\988117\1039126\ETX\NAK\995510_QQ@\1064192f4\1041779v\1065776i\77942}\172799\aSqA\RS2p-\141701V\\\129429\f\1108556\999033~CB\1059707\EOT\1079272+c\1071047\50772g@\\_u\ESC)*1B\999997\147598:\SUB8\1078279\1094608s\987558\ACKD\ESC_\DLE;\GS_\DC2j\FS\160781Z|0\153888\\\45416M}\fp\EM>\GS\176231^\\*\SYNk\ACKgP\SUBi\\V\SUBni\1077760\1101494\177366\GS\rP\DEL\DC4,4Q:G\63943`qH \ESC\153533\74553eq\SUB<!Ao\ENQ]\r\1085236\1027754\1022253\FS\145360\1065036rTH:;ps<\r\93804\1098527F\FS\1099659^)\15679(\189323DTFbd\1009384D{_!7Lhk\GSD\161373\&1\v\1000857\DLE\997988\GS\DC2M\41989_{&\1001325\\\DC1q\1113700%7:\171205\DC1\1030175\&3]\ETX]\US\52517]\13805i\ACK\1056245\f8Em\f\1103213a| \1096200 )X:%<\DC3\1016956\4988/-\153552V\SOH:*\181333\1065342 HgZY\150106\DC4\1018927^\174332\24987\986476^DTXo\DEL\GS\r@\SOH\FSh'\128212UY{r3\61869[\n\f\f\12614G_z4\EOTh\138480\147153\SI\49225\1048349iUr\\\1082234+\159178|\179700<bQ*3\991071kS\RS\8929\&62\DC3"
          ),
      updateServiceAssets = Just [],
      updateServiceTags = Just (unsafeRange (fromList [BooksTag]))
    }

testObject_UpdateService_provider_14 :: UpdateService
testObject_UpdateService_provider_14 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "r9@h\176936\1093770\&2u\1068502 \ETBr\DC46k9\1093393\NAK\43965\9513\&2\1077631uP'3\DC4:\ACK4~\189709\44125\&4nJ)K\25550\SUBQV\NUL\1021874\64799\STX\NAK|\1088831#UHH\NULb\989619'\1075725J\ETXi\1066664\62882)w\42896\1098782QF&5Z\FS0I8\1099770A\ETX1\984578%.^FvFt\1021897V\1059784t6?j\DC4\SOH\DC2b0Wv\27142\&9\160967"
              }
          ),
      updateServiceSummary = Nothing,
      updateServiceDescr =
        Just
          ( unsafeRange
              "\145244\42890\1012972l?F\147134>JT\44596c\113730\ETB\t\164931\SIIx;\1019324q\EM\77947\165887\EOT\SOHhL\156504^\70812Y<)h\ESC23\38347K\1039668\1110104tSWM\DC3^\DC4\USZr\DC3p\3361\ESC\1041339\DEL\DLEJT\92214c\146429m6>S]`\998393\SUBo'y,\1028028\71087gQ\SOH\vT\fA\SOH%\EOTCs5rTy\166907\1005688R w\DC4+er\ESCF\4275\DC3R\48294\ETBsN\58138\NUL1\1019395 \ENQz4o\NUL\991565\b\SYN\151737\US-\r\ETX\SOH\GS\DC1\1071758\1051154\1041798FU\983498r\1057471\150946\142299+|\173334\179415\19518\"f-5 p}6;k\tA:\165473\DC4=\166096\135668zB\178240\146897ThDyd\38411,\1104929+\n\EM:\9448\DC4$_QbSxz\175790c\998601\1031322q)O5qT\1055963M\73086J\149822g\136535Y\SI|\1041297\GSp\45172Sy\1031231=\158126\1015975TI|f}\136437\vEGNG0`\US+\v\159152`\1097365\SI;$v\1025381\\c\SI\185085H\52570\SI\1075078Yl\185690\NAK\1092632xCl\SI\1067823{Q\ESCv\141647I(\992094\DEL\1013477<$1j\1086220I\RSbG5j\60105_\STX\1008523q+\a\1000446Gj^\GSE\fml#V\27463\CANI)\1041639/+\SOH\983677 #\1111976\31082v\FSV\GS\1096105\1097866erQm\FS#\DC1\ETX\58006\138528/E\SO\b+/)Q\31822\41198B`u%f;k\1111874\1069650\44219h!V\161302\995869=:\92653\54267p=>^\40039\156059\12454N\10388^;{su?bR{}\ETX\ETBw\29089tN%\1069052o*b3\33095\"\993161j\vsrY;_p`\1085887\n]\1013506\&6V\DC4Q\100613/Fl\1085922|k$C\DC2s\1049944V\22306\DC4'\141829\1058264>X\DC1\146384\&5j\12336~\1101445\19559\1007148\1008183\986545\CAN\1082905:v\8201\30912\58796\1035349&\178754\1035946\7490i\b\1053453\STX\96201o\1031898yQq,h(umvD.Z\991654K[@3'$\1035414\9146\147243,ffj\NAK3\ETXUd_\FS\190639]\1044811r\ESC\1075556\DC3\1087129T7o\137795\&6_t\NAKT*q\118800\1108248>ff\a\119121c\1062827\ETX\1057666%dyZ\\\USs\1112881_\1034510\&1$\US0+\DC4Z\NAKb\139923dT\DLE}\ETXD:\a\1044006,\DC1{9x\5782\34862hg\1000845F6:\1101415{sWo\1013265\NAK-V\132335\SOH\CANq) 8@s\EM\1095665\ACK!>I\t\1046159kc\1027792\189480\133655g7\1034481\ESC7\1014752s[L\70028\a\DEL\SO.pJ\180668bBJ\SOH\DC1\1073445\FS\ACKe\SIRY\USy\DC4+.\v\137783\GSng\v\131541RxQ\n_y\95745\164989\SI\171699f`\23443\40019. N\\\v\NAK\v\EMv}3.\987217\146601:y\33754\1095812g{\988687\vJ\ACK\7896\DC2\DLE\185363\ACKY\DEL\1080154e\DC3\n\DC1\DEL\1051752l4\EOT^\1104164\RS\1005794\SIk5Q\US\"6\27210\181495pk\GSG'@Bd.\39058\186204\1058994\1002179\r\CAN\DC2\SUB\DC1uXczm7?kb\SIIBYtIGJw+\1065173cZ\1024620\154639\185984\&4P{h\172748\\1\EOTh\181037\ETB#\145412{*\1070281\991124A\t6^\EOT\ETX~|\1014795\DEL\SIr\NUL\rdCLs#o\21115\999535D#1(\n\1086666'\ENQ\1104183\1105686\SOH\141456{\1013071;\DC1a}Y\ENQ\125205p\ACKy!\140008}9\145831=QiT&T\t\1010343\992010_\1057838M\1062624\187138\"0:7/\1063812k\SO\127349:\DC3=Rc\ETB\1081186\1015039\NAKMw;;+\GS\USBJy\1020750/S@7\1103574\b\1105976^RJ*\DC2Q\EOT4j\1026316w\a\144050c`\141133\ENQtDR"
          ),
      updateServiceAssets = Nothing,
      updateServiceTags = Nothing
    }

testObject_UpdateService_provider_15 :: UpdateService
testObject_UpdateService_provider_15 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "A-\1055359\1001974\NUL\1083969w\EOT?E\1028275\ESC\1053251\EMs\4949\180712>\ESC\a5[\999882\179863=![C\95845\&2^\1047975bZ0\49557\nU\1066068+b1~\SOH\a\133446Hs\23365:-\DC1\EOT\27147\97812\40997\CAN\SO\29634u_m\78696\ACK_r#\156424I]\1048670\27983\ru\ESC\141083(B\ACK\60250\988606\r>\NAK*\t6<\a\SI\US\"\a]\1007461r<D$\SYN(\1112708\&8\STX\t\SO\"\180547n\DC1m"
              }
          ),
      updateServiceSummary =
        Just
          ( unsafeRange
              "f\f;\vNHu=6A;a\13528\7234\RS8\1091114\FS}\1088672k\DC4\43608\nd\SOq\9573\&3\1086782B\r\NUL\UST\43031>^\NAKNdXh\139665k\59684\147290o\1111280:t\136686]W\SIW\vVE%\SYN\ti{\1045016\165649\GS\b\1097984"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "x\a\136203\SUB^\ESC<4\n\17873\SO>v\157431|\1020922(\185983{\US\30184A\SYN/\1034793\FS&\24692w594<f,\157583#\DLE\127120\SUB[b\95571\16229\SYNE\1077161\GSw\73971\1103971\EM\1062787+iMC\113742#\RSo\SUBZ\SIB{)\1103759\159790vX<\SUB\156054\1092784\1105257~\ACK:r\1000703\65058\1088202M\985685!C{\140112\ri\EOT`9\v\1050059ua\ACK`C\SOH|sya&\989821\&8jKR\NULIg\32866c4d\1069546\ESCu\1035617n\SYN[3\21566\NAK`\1054879W\SI\SYNV\1029718\EMli\n\994655\r(~z8BR\1074007\149826B}\NULW9\ETB\EOT\46399;\1101863\1045023\ETBu(3G&3\128930@\1103892'c\8510\STX\NUL\137301V\SI|\51594s%\36716\EM\1114066<V\ETX\136032L\147926\CAN\1025624LK$~a*F:E\1002566\USV\59157\1101707\SO\179826Rl\CAN\STXK\47644\54839tlb\RSF\998476,@\ETByR\CAN>5i\n4\DC1+nk\118834ZSV\1011086R\996947\GS\a\CAN\ESC;D_g7T\61134NN.\1080365,\1035561\SOdPu\SUBF\"e\1071157V\1072899o\1019597\SOH\ETX\RS\1090025J\brXcs<\41273eYl)'\DC3F{wjCio\10430\EOT\DEL\66434=F\EOT\1011500\FSAe\99614\29782j\987688\RS\93035_^7\FSzLW\DEL\v9Oy&\1019281\158269=j:\161191\EOTxXF\v!\SI\DEL{\182824\CAN(q#A\f#Y\GSm\1029668\SYN\33488\1091890Q\21517\DC4N\13674bj\21932H;\55134\26121fz\183843\135947.p\147443X\SI+\22973\29225\14419\b\n\35820\1092258\ACK8\1003849\99533dUVae|'3g\STX\SOH\177742xA\190959T\1088684M\167371\&7\60761:\NUL\100886\DC3\GSs\SIyw\1063851Q_u}\SOH>\1069485\134333?\US\SUB\1106685\6158]5Z\1034719%\57389\183657_\DC4\41432^\28540qa\329\1097112/-\ACK\EOT\45370\1089284~H$\FS\9526\b\SOEVy2obJ\138789FK(\995061H[^\1088420\25954n\160665/\FS\US#\1066635db\1006679\&5?\nM\SO\44147Xs\150820\1112961\f]XR,\GS8{A0.T\ESC4\SIL\SYN\EOT\1028786\GSkX\ESCa=6\"qA7\RS\ETBG\ETXD\DEL\1100961d;\185997\EM\NAK5\DEL\1076613Qj\f'D#\v\1087346gY\110778\CAN\8773\&4P2\ETX_\1048072P+V.F9\SOH\156486-oK&\EOTo*\SYN@\174461&w\1082933\n{\b/\39070<'>\148084GFoF\25642\SOH\t]vwT{+\987769\b(mO\35465\47334xR\1099279\SOHk\120988#\DLEJ\n\1111066/R|^\SYNXj\177224(Dc\RS\64631$jM\1058526\n|_\1023319s\181256\1081025U\1077048'\144694\f\NUL\GS\179974puJ\14349 1PH\986811\147997\DC3p0%!\1096149\&8Q3Hc\DLEb3\1063888\DEL^o~\1054122&u\a1,mgg\1046750\141023'J4\r[6\45643o\FS9\SYN\1020964<\RS\31175\fa\DC2\v\1046951b^2\DC3*\DC2Y\8803&p\ETB\27260#*\DEL\41812\SO~mcH#qFe\1015266\DEL\DC4Aq\DC4(\GS[\CAN%%h3U\1013273U\1099555\131387\1019990\&4\166361Tt\43506d7Z\1059964~\984571"
          ),
      updateServiceAssets = Just [ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing],
      updateServiceTags = Just (unsafeRange (fromList [FitnessTag, TutorialTag]))
    }

testObject_UpdateService_provider_16 :: UpdateService
testObject_UpdateService_provider_16 =
  UpdateService
    { updateServiceName = Just (Name {fromName = "%U#\1023610ov"}),
      updateServiceSummary =
        Just
          ( unsafeRange
              "\28746=\RS\1085059\&9y\\\150352O1^\29094\\%m\1018265\35584\39280\1020414O+aN.\1047053\ESC>'i\ENQ\1020143f:[_Ux\tX`\ETX\DC3\1001861Tp\ACK\1028775f\DC2^\SO:\SI\1010449\DC2%D\135253\1025118mz\1049860:<\DC4\NAKMC\n,\ACK\RS\US*\139802\995836X4\DC1\41788\1096942\a\1045909.Fm\182117\999764\37262\133641\19156\165280u\1093831\32730hhQ\1068949\a\DC3\a@(\NAK]6 \155238\1107506\60061\1034162[oF>\DC3Z"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "/\ACKfQ\1031903s\13506\DC3 \vl@:i\12164\1074774\139757S=\989441\ESC:UX\163277}\83122&)m)\1080365\a\ENQ\f\1058190-B#$\"\983961\69725n\US\\\t\179625Ma\1101379\US\155328\&5xg\10626\15907F\DC3=D\19436rb<saS\DELu@\NAK\EOT!;4\185452a\185176\r\r\SUB\1012914\159108\178943h:HZ\1043622\1087161\165211\1088512\bHh8j\34432y\1000022\13499\ENQ\174782|+@\183270_\v\1052248X\23813\n\987071\1001761CU+d_~G\RS#n@\72790\&0Q\FSz\182789\11752\GS>\DC2:\1041291@WB\145836\n^\1108818\&8\145641YC<#T%a\NAKyX\51879\92218\&8\1112272\SO0\15876\CAN\161412\44252\&8\1043643M\47819}\1011210;\68326\150784\1016962+\DC2\SOX\b\171587\1082608\ETX\SUB\NULK\137124\1056688\&75\US\1080008\ETB\GS\SOHSjP\176968`q\SOH$+#c\157075=G\GS\EMP=\GS\ETXG\72767g\"\187550\167547B\1028906\160084yO\161954\ENQS\187\62446nw\1030875~q\DC1yZ)\138864\51888=}\CANj\n\15967j\22497\SYNlPp\17621\173023\&2\f\ACK\62937\&5?6h\SO\RS\134742\&3,[pF\1013781\DC36]\a4\1040109\1068644\DLE\185640\1020205\&5at\CAN8\ESC\SUB\DC3]\154302h@\1017215\42286N\170346\STXaS\bG\US2DF=\1003391\SOH\4761U \ETBn\CANOI\STX:\DEL\SOH%q\149603\SUBP\nyI\11485\ETB\ACK\ESCu\1101308\DLE.\6382h\171375cM#+\SYN\183868mc&\1105096\174881L\1019909xN\a\96065\1102404`*o]Xl\DC1\1069323z?b)\128572yI\GS>(G\ENQ\NUL\af\994948\183208\EOT=2.~@$.I\FS\1035586\USm\DEL\150438y=\64681N\161422X\64060\&65x\52325q\SYN\1042083y\SYN\28886\1068156\8946\4036\1060792TV\136669R\179446\SOHx:\GS\a7d=\SO\SYN?<H\997677?\164543-e|\53999*e*\1069625\997609\SOHe#\98720\\\8142i\1038024j\983524\"\149967A\1109215U*y\157070[at/q\169759D\a#\NUL\DEL\r|-C\63299=_\1069887n\ETB\n\t\147917V\159773t\1067280qV\GS\a\155562}Qq\SUB\1024685\1019858\t_\ENQUD\985241Z.\RSi\SOH{\1013817:@\156103Q\\\179416E\3058CU|m}L\CAN:@%l|\NUL\RS\1111129)\azU\136224"
          ),
      updateServiceAssets = Nothing,
      updateServiceTags = Nothing
    }

testObject_UpdateService_provider_17 :: UpdateService
testObject_UpdateService_provider_17 =
  UpdateService
    { updateServiceName =
        Just
          ( Name
              { fromName =
                  "\"UgHe\DLE\FS\US\DC2\1044016\SO\\GS\DC2?p\1074506\ESC\157535>E=\EOT(BT\EM\1111083hw/\157019<^ S\ETX\18862U\186544Ie\1113880\13776\&4\51147f;\154081Z\126119\175700p\SO\FS[M\1017447N\ETB\13157F\131460\DLEI~\14413\r\DC2\SYNm4\DEL\142240\15378\18996\1047381\1082499=\RSd\31924$H\96480M{0W_\32136\8221\30211\b\1023125\&4(t;\47306\33675\&9e\1034291\1074178oL\ETX\65029+^#TR,@\RS"
              }
          ),
      updateServiceSummary =
        Just
          ( unsafeRange
              "Y\37457\171247\NUL\1102605\19452;\40109l\1091643\1038961\164211\&3\1060552/\NUL[\STX\ETB\r\1050187\&9\SO9\SUB\NAK?yC&\1087572K\19408X\1008435Z\1043931A\FS\ETB\a\FS\1068870\&2(\FS\1081735Wh\1105128;\30117\SYN\177561\121419F'\ACK,\1008576t\b\148040\178770]Ea.Sr\STX\1021147/\1091479 O&\167108P\1051535\12083 P\fvL\1072069xTw\171454R\CAN"
          ),
      updateServiceDescr = Nothing,
      updateServiceAssets = Just [ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview)],
      updateServiceTags = Nothing
    }

testObject_UpdateService_provider_18 :: UpdateService
testObject_UpdateService_provider_18 =
  UpdateService
    { updateServiceName = Nothing,
      updateServiceSummary =
        Just
          ( unsafeRange
              "x:\1478%c`mAV5E\21174t\v\DC2@\1075181#\RS^\v2\163900H>J\GS\f\\\1024822f b}\99481B\SYN\121002i}\1053609\60688v\132535C\16339_RW\1013824\ESC\72844@\FS\1049907`\DC1\1011649\b\20485\&3Y[\b\DEL\GS\tQ\ETXiHR\t\1035085\22179R\r]gI-\1098499\29250\17941Ui\SI\a\ETBY\58945>59M\984750{m\DC1\DLE\GSbw\NAK\DLE:0Uz"
          ),
      updateServiceDescr =
        Just
          ( unsafeRange
              "\ETX62P\SOH\DC4\1109991=\NUL8}\1103539R\1014278Y\187048\CANz-\50831t\NAK\30991:\1108518\\q5!\CANsz\986662.]\1091331}\EOT\SOHk<\1076580jo\ACK*\1006270<\1068043\v\162015'\\Ky\\d\67224Ea\186085\42476\&7\145875@3.`[\83186%\1013254\1103673\2547^o'\NAK3\DEL\f\32802\&7\155976\US\178005\182126\11804\13566\ETX<2\37455\\\EM7u\1101747\996895\1030597`\aF\DC2\1002903\1065461G\SIUMj??\1082038\163609[q\53362\STX|\STX\f\39680?\60538\US\ETB8\STX\EM\1113089\1024191\DLEZ\n#[ \1010523\RSh(\1031090\&3\142124\&1\bC?2rx7\NULjE\nU\1056190\n)4\EOT*\18936r\NAK\EM\vA\DC42TSw,\SI0\1061258\176021\&6RX\1104923KEU\99028as\DC3/\SYN5`,d\"\60033\DC3\180441y\ACKe&|\SO\USE\991388\NUL\34162\3233\SO;\DLEh,|z0\GSZPK#WSN<qk\SUB\190620O!'c\72145wILr\nnQ\STXi\CAN\12693\994837\1113003k\156304@\996112\1101483\1008462\EOT\1077126s\NUL\1003566\1102059\EOT\1008743a\rNB\t\aR\STX{\1011900\NUL~qW!Q?\a\v-]Uo-\1054200\1094060mL\1038490\ETB\189230L\62266U\7540\184079b$\1013872@\1068202P\EOTXUzp\NAK\SOFn\EM,7mQ1YcVpHS v\STXp\v\SO}\SYN\1017645a\39045e\1019586=\123207\986686\1012662\&2\1027179u\139336\1064815R\1041456\137471&\SYN\f"
          ),
      updateServiceAssets = Just [ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing],
      updateServiceTags = Just (unsafeRange (fromList [BooksTag, GamesTag]))
    }

testObject_UpdateService_provider_19 :: UpdateService
testObject_UpdateService_provider_19 =
  UpdateService
    { updateServiceName = Nothing,
      updateServiceSummary =
        Just
          ( unsafeRange
              "k\54460vy\43155/P\1107392\ENQ\17146\18104\29580v\1015492\ACK\FS'\GST@'&]\119195bGq\183910\ETX\26871\t\DC3jd\SYN\1036848\24330\ENQj\ENQ&1*\b\1048448`4mEa\SOH\7056A\ETX\983940 B\RS;\EOTy\1035997B\DC1g1\"\30253nAX\US%\127094"
          ),
      updateServiceDescr = Nothing,
      updateServiceAssets =
        Just
          [ ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetPreview),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing,
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete)
          ],
      updateServiceTags = Just (unsafeRange (fromList [MedicalTag]))
    }

testObject_UpdateService_provider_20 :: UpdateService
testObject_UpdateService_provider_20 =
  UpdateService
    { updateServiceName = Nothing,
      updateServiceSummary = Nothing,
      updateServiceDescr =
        Just
          ( unsafeRange
              "\r\173641f\145891BG\1000009\65858\ENQ\43536\DC4V\25617\1103211c\SOH\1049555\1062140Q\51583\&1\US\ENQ\STXF\ESC\US\158231\150930\fp$\50673s\1106003,@3 \134343I37hoN*\SUB\n\165469bL\nr\ETB\ETB\1109513<\DEL3%\US\1029609-\ESCAPG\988109\&3\ACKG\178065\FS\1047908w\GS`A\1003677\v\1059871\1108548I\141269\\\131976\1067739u\136657kI\1095695\fx\FSBF\DC4XZ[#/\10934cHD\DEL\ESCN$\156565]\71456\FSi\CAN\RSlr\1041630K|\1053868\SOHni\1036071\SUB\1090967\FS\1090238u\1087884fF\31005$c'\STX\US\1039171V|\CAN\138612\SOl\DC2B\1061057K\16801d\1113491.\f\156836+Q\NAKX,\DC2\1024953Y*-\1023384_#A6\987855>W|2\59920\1034071k\38859\1080991\ACK\26667GOp\1106550\26147o\68058\21445g\120366=\ETX4PJ\DEL\187447\GSjim\SYN\US&,`@\SYN:v ]\NAKO:\ACKN\1105621\EOTu\175621\179993 OI<\NAKy\v\ETX\1098458^\SYNhm~vV^t\9987\SOH\36155\DEL Uv\1086361|\v5O:v\64775j\"@\1090093N\1068364\NUL*WIRz-\t'(G\984249u\113745\GS4`\\#O\98523\&0/>\168702\SOH7\98326d\1082241S\DEL1y\1044551 \136286\&40a\984500\&01\118807{w\170720J\992552dT\1012893o8\998212'\1008071a\n;\SYN2B\ETB\CAN\188685EH\SOHo\54275P\1038172\1061525\49851\&8p^tX\62754DF0\DEL\SOHH\141376E\CANY0\1035536~A\118995\8122\rr>\"\ENQ{\DC4%\1054555\1042977i\a39i\t\b\DC3+\ESCn?W\1034984o\SOH"
          ),
      updateServiceAssets =
        Just
          [ ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) Nothing,
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete),
            ImageAsset (AssetKeyV3 (Id (fromJust (UUID.fromString "5cd81cc4-c643-4e9c-849c-c596a88c27fd"))) AssetExpiring) (Just AssetComplete)
          ],
      updateServiceTags = Just (unsafeRange (fromList [EducationTag, GraphicsTag]))
    }
