{-# LANGUAGE OverloadedLists #-}

-- This file is part of the Wire Server implementation.
--
-- Copyright (C) 2022 Wire Swiss GmbH <opensource@wire.com>
--
-- This program is free software: you can redistribute it and/or modify it under
-- the terms of the GNU Affero General Public License as published by the Free
-- Software Foundation, either version 3 of the License, or (at your option) any
-- later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
-- FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
-- details.
--
-- You should have received a copy of the GNU Affero General Public License along
-- with this program. If not, see <https://www.gnu.org/licenses/>.

module Test.Wire.API.Golden.Generated.NewProvider_provider where

import Data.Coerce (coerce)
import Data.Misc (HttpsUrl (HttpsUrl), plainTextPassword6Unsafe)
import Data.Range (unsafeRange)
import Imports (Maybe (Just, Nothing))
import URI.ByteString
  ( Authority
      ( Authority,
        authorityHost,
        authorityPort,
        authorityUserInfo
      ),
    Host (Host, hostBS),
    Query (Query, queryPairs),
    Scheme (Scheme, schemeBS),
    URIRef
      ( URI,
        uriAuthority,
        uriFragment,
        uriPath,
        uriQuery,
        uriScheme
      ),
  )
import Wire.API.Provider (NewProvider (..))
import Wire.API.User.Identity
import Wire.API.User.Profile (Name (Name, fromName))

testObject_NewProvider_provider_1 :: NewProvider
testObject_NewProvider_provider_1 =
  NewProvider
    { newProviderName = Name {fromName = "\1017845\&8\1098296\58272:(\DLE\1000399f\1005296\a\13015\1080816\&4K\42948L"},
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "eX\US\b\995223Ok?\129176\ACKY\"\144658\DLEW\44441\SI\CAN!u\\\97087\EM.J\992919Z#A2dEH\n/)\994489P8\SOHE\1063260\1072166\&3q\34380_\128663e\FS\158188\\|7H\1056055\46088\1058968\1043255eY\1106811&b\35965\&5\ETB\68661~Z\DLE\1089240Rkli\128950b\SOH\7147E\fNh\138726\&3fFi\t \GSw\DELd\42692HuDSl-\1000121e4:S,p-\ETX4L.$1\DC3\ETB\1017098r\1061346hEr\49284yJ#\DLEW\CANZY\NUL\STX$b\135460\FS\15599Y?\172311\1102259\GS\1020106\134118\96722\SYNl\1096699i&k\53037CB\176916\43991\n-l\23023\170774Q/\SOHtt\164573\1101690A\1098123\RS\1039858j\NAKe\1068011\83452\1106397\NAKEN\26963\RS\511\65094\ETB_f\154470\1062072\66387{.w\991394K\178858<\bV\SIpR/\1066633\50900\60856\EOT\3148\1082502A7I\1109396$\\\ETBk\FS\82981U\1012822W@8F\267n\DC4R\a\FS\163953\FS>\FS\n\STX\1080936nI \1054614V\1026699\RS\67087O'`@H\46961 m3\148748+^s\USm\1007141\33896M\152042\33066\STX\FScP\986670\DLE\137136>\1051639F-\DEL@V\985094\1089834\ESC1\ENQw-\SUB,\1100026\ACK\1088545\78144\1004530Jrl\STXVX3-\1065111\182019X4a\1113560V\SUB?=\1053376|\173660loc\SOHV\STXn?\1060746)U\1002872\1082612>\53460\59632v\1000605\&8v?\11036l)\SI\a3\1012928O\174317\158398\146688U\1068621Dm\STX.V\185462\137591\455\&0\35026\ESC\1043419\aF_\1091056\1093536\1043719\&0\1054413\156291\131934\CAN:OK  !jPv\b_$s\USSC;\ACK\NULi\53285\61206\FS\1066412Ze\ETX\985175LI^1#\69683\DC2s\61897\SYN.|a\"\1092800{BD\NAK|\1036119l_\1103748\1024281\1019820\CAN\".\1020906D0\\\SYN.ZR\SOH\31433\EOT\1100127+\133413'B\1066652&7LM\129170\146670\ESC\1008529\SO6`>\FS}C~UE\147745~A8\EOTB\1083151\DC4b\FS7=\n'\159715m^z\67715\&8[\1038028\&20\EM \987450\1017409'\RSu[\\DI\tz\992390z\ETBFz\1021033>)\63924\NULO\SOH{\SO\1006356hh\1055488\29196b\DC1\48178\1065099\166710\&68\1074840&u\30251\1038941\f&W\145237P\158967Z^%{;\\$jh\1061320\r\rll\DC4jI)\FS\SUB\\\996923t\44820\GS\74802\&0\US0f\DC30Nmzs\CAN\f\ESC`\DLE\162810\161070^t\DC2r\183989.f':\1109934l\17508w\171346\1022383\&9\14042\&5\1110132\ESC\1041237@\95112aU.\1062393\v]\37115s\165263Q\DC4\1075995\DEL\5225\134431\15103\FS6\CAN\1064420Q\n\137666\1015259\1108266if\EOT\FST\1013036\40256k\ACK\34918\SUB\SUBb5<D\10399*y\"\f\t\172380\97004\DC1y\1018978;Mq!\64732\1027635\2320\38903x\1076429\STX>H4~\1035553V\171666]-\1046754\DC4\DC2`zh\1091598\DC2~\1100400L\DC2h\NAK\CAN\160701fJ~\999801^\RS\141113{]ms\96252de{\STX\31083N\1080942H\179600OP]\1023361\149175\ETBz\985800\a\163992\&4\132045\986873\ETB\GS\"P\ENQ\156813\48597L~Oj\"R50G\52610\1077153lhcU\1091924\GS9\174773\t{as\143998\"\67306v.\ETB\SOHSCD/\FS\DC3\169000l\1086905\78356^\183726\1094002\RSo\152541\135639$\37530T^\EM\GS&\ENQTsl_un\1012503\1018353\GSMy:pX\5892n<LQ!}:\167382\NULV\SYN\1104895\1011379u3o=4(KnN5\1001895*9\EOT\994122\46656y'[\ETBI\DC2\1086658SAik&~c\27520\SUB\fO\NAK\1056961tf\DC3",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "u\v6\ACK\STX\f\RS\1064115\SYN\SOH\DC2|2q\10550\1025986f\1063170\59721J\ETXMo\182177 D\1086229|-\993530e|yfd\EOT\DC3\ACK\1026988t\SO\35633\SUB.SftU\985592\NUL\DC1<\1074020\STX'\b\998189\SOU\21330\SUB\6184|blDM0K8\161208w\DEL\RS=z\NAK[<\58467\ETX\DC1\EM\172678\1073880\1093406\"1x*\EM\71124\120771qo\993250\NULo\1074002\DC2On69@\986203\NAKZ\42805^y\RSn\1031130\1032996`U\996238o\SOE3f\SOM\1094226C*Q=O\184769\DC4\133825/o|\r=z\61163(\1051738\DLE\DC1^6\49518?C778D\1052152<~+9\100341hm\161642f\\\SOH>B@`*J(\r\ry\139052\DC1\10228i\CAN-S}\1102675\\\74911Bh\1104717lY\ENQk>`)\EM\173078gZ4\1064140\RSIAJ\1054448WgoA>\53622\987489\1042604Sw\1113231\7766\&8if\\X=\US?d(\SIt\44993\194758\DC4KR\1089533\984650G\NAKC5\NUL\1077670c[\59836\&7-kp'\159389\SO\1037212\41495F\1033061e'=\EOT\a\DC3Z:q\bw7\1044001\131787{]\1031412Ah|!$\62490\auzh\DLEo\1072953\SO\1021796\153148G\1023308\DLE\21074\&1p6\SOUX\SO\4214kY5(\FSV\188078\DC3\NAK\1071166\1007087.JJQg,%c'aq\t5f\1012607\39545\154628\f.\ESC{\r\1046768>'O\47202\DC1\ETX$R\ETX)\997568$\173402Hj7}F\FS%\1110017\1083305\998552\&4\95158\SO\a\179995Av\988645\\\STXWq9\SYNW\ETX\991965C%\b\v\SYN\37742AzG'4Dj\161358|+S\v\STX9|0$tLe>ol\DC4\EM\1088003\r\DELmBK\27258\1077075bX\188456Pw+\SUB|e\30882B\1041022\&6v\RS\1074170\\"
          )
    }

testObject_NewProvider_provider_2 :: NewProvider
testObject_NewProvider_provider_2 =
  NewProvider
    { newProviderName =
        Name {fromName = "Tx\47851q\1065280@\SUB\1025401\aW\FSG\DC4\DLE$\DEL\63098\DC4$`\vO?\"Z\1046679\1083355\&5K"},
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "\1099787Ygo%\944I\1005116`F\ETBYT\US-3\992556\27798\DC1G'\14811<\33364\1054185&\43295\NAK\94529\73798\ESC]\DC2\NAK)!\n2K3\4250[+{\vC\957\1009692\DC4L\111084\t\1050716E\ESC]AF\CAN\35119\1083267\1063239\f\NUL\EM}w[F\18536\NUL\188964\SI2\174320^z\120621>\t\SOY\131199$4f\ENQe\r\1067591\60319E{\1108553\n[>\1107778\991883\1112693xa\SOEG\1034253\1012045usz\CAN\1080970Im\FS1[$\992090\&9/\r\51589S1U\155047qK\ENQ,\DC2\25888\1032623\&2\994470ElX&92\185686T\t\ETB\tkM5\1027665,\998362#\1067165~'vmCROC\1005145`q(\SYNq\1083849]<\136352h\CAN;\ETB9\49328\153749\29441X\131818eH\ENQ\1007071e\1098892\DLEXKDs\SUB\SUB \SUB.\4007GZ\124942@\1006684QDOJ\183338\1065192e\110786\1049183\ESC7\992658\29915L\1029762\DC3h=[F\1062586\afH\137173f\111228*AQb]\4441H\178088\159233o\DC3&\174611\&4\t)yBn25pNa\181971Ex>\ACK%x\52036B\US&MFo.S\1069125=m&us\134137j\ETX{\DC2:\131911\RS\1063145\159338\1017553\176300!u_\ESC?\27584\&1n\DEL1\ETBzg\1107110N\f\176519?\15997p\134128\1031849\EM\DC4X7vS\156481\1048345P\SI'\1034676\1045079Nu=.4M\191448Wu \DC4}N]a,\129145\1085383\NAK)|\12357\&9k\1049331\v\1066664\186640\NUL\GS\169180\RS9\CANT\58203l\1049292\997037\f\51053O%\1059997+D\NAK\DC4\ESCY\1000337\138902M\USc\NULY\RSx\EOTc):\1007927\SOJ\1044503p\SObS?[KU\190321Ap_\164005\ETX\986483\DC3C\a/\38086\&1mEmb\\\669\CAN\ETXB\1042722J\39668&\SYNZ\28973\SUBbniY\DC2B6\DC3\CANvm\100846lM*\99266\147320M\DC1\36270\54145\1027786\ETX\143852Ee\158098\992386\DELw\54653\78178\SOH7#[ck\100330u0P\nPpN\DC1`*\EOT:=-hn#\SOH1u:|<iYy\57405\DC1\\\62340f\EM\ESCkb<\NAK\SO\GS\151613aX+J4\27539\DLE\DLE\1084009o9\1034973'\1111917\1109449=Qn\984734L\127819l\52574\tp\ETB[\ETB\NAKO16\19776:\1066529|\154457|\1058887s\987323\1015545w1jKs]S\ETBa-\SO\23623\rt+$h)dkWz}7q!`2w?\SI\ETB=\RS2EEd\153020\157308\1007648=*\171211\995018*\SYN=\1056543\SIu8$3\\\120108\1007746\1054951|\NULH\1073878Yk_\SYN\1074877M\169317G\EM\v\19638yY\EOT\ETB-\ENQ@\1070058;<\ETBq\DC1!Z)\990670,\163978Z;e29S1\49184\EML\99949 \110678\ACK\1095954\FS\158948\&9w\RS-9\1025332T1\1035847\23464\&9bL\SYNT5g_\151156^3\a\1014679\1005821\1059705\DC4o\17164\SOHnVnyq9r%}\SOH}|\67635\FSai\SI\n13^\ESC\170391A\ACK.2\aw\187172\1028223k\\z&\SUB\1101423{?m\988344\178940\1048681\35111\&9\1094099)\1055056m\1105237-]\r\999433p\f&\ETB/\"P\23846\156747M\DC4\r\SUB\DC20\SI\RS\995363D(\46129\61235`\1073690g\t\22925\ESC\1017562'\145111\&5h0{;\186591h&BY_\156499\DC4\NAKtq\1077142\21631\150266\NAK%`.h<&DA\ETB \SO\NUL\SI\DLE34\29069\1024814c\DC2\NUL\156623\SOHh\ACKucTB&\1066086\1092938\25548I\FSj\ETBW\SOH\155396\ETXM\163697\1064170A[S\STX\1111397\DC2:\EM\RS\990066A\ACKd\ENQ\1063074\96227r \50072\173759\NULM\SOH\1111562\DLE\NAK\1035419\DEL\138399l]\ESC6\FS/e\11039KG\988602>\51970/\NAK\1059948>\DC2\"\29644i^1ER\RS\996078\&8\1125\150806\127030)N\71298wm\44426\&7\SUB\f\132307\CAN\DELW\1113578\&1jf\STX\n\1011631\100631\36624D\ETBx\RSc#\1056581B~\983966T*\SI\125218\1022061\&0.r\EOT9F{u\DC2",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "yx?@/\1064686I\DLE\1095959]L\1029155\DC26C\SOH\37291\RSF0A\1010733\DC1\a\51798v\156301=ul<\SOH\SO\1111486\14157=}\vZ\DC25\vZ\37208yy]\49915\ETB\989906\1023059\ETB\ag!K\EOT~kg\FS3<\GS\155169\NAK\1043661eT\996822j\1101705j+*`\177804\9227wprI&\999931H\SOH\989782Sb\DC4.LaL\1047000\1049757\ETX!x#-\1058415\SUB \1097186\t\SOHo\1060278xc\1056850y\ETX\ESC\101028PH\98408.K\US\1026324b\DC4%6\NUL\27763\SUB}Q\STX\r\b\SOH\1024758\3806\US\1075163\STX!xvB4-g)\42920\ACK!GN<\988790z\1099591\1105946\"\165524n\143776wr2lY\173686\&9k\129536hL\31908/\SUB\RS\1112666}c\1038214\&10R\US\CANisO\t\14749jP#zi\5193n\DELaR\DLE\CANa\SYNd\8141\1082961OrW\1055686\82996\125215\"\137888\51122\DC1+'\bqS\ACK\997894\STX\984520*:\1013827!\168627J\CANs{\DLE\78747#\1109522\993984u\164248\rk\EOTceB\SI\1038689\"\987017\13792q\175366ed\1070052\1066652\95285~\194990~]\10444'\"vi\ACK9{[zG\1095584C^\1100926\26462\&6\SUB\188229-\RSm\v/\RS\185849Y:\1106298<a\69938-[Q\EOTQ\"\DC1a4\126596>\1098425r\SOHx\1111404\1022242OR+\1072403\194678\97452>\181135:q0'\ENQa\1003104np{\"(\1079308\30071\48211#a\1004180\155075QN$\DELT\990246?Dcj\38870:\157406#nW}\97686z\SUB\143116n$\1096601UP`\RS?1Z\99546mx\\\23793!*\1112791\1099074\SI\1109995Qo\994497\1047096\STX\a|/\\6\DC1\163049\DC47e\1063537y\ACKT\CANf&_\36459^sy\1102254\70173;^\DC4\ACK#l:i Y\a|f\ETB.$\52342\t\DC1\ACK\1088723p\a\1099755\\l\DLE?di\NAK\182060yr(\1092195O\1061441\ETX\882\US\166014h\1054820\"j\\\SUB}\ETB\180746\t\DC4\194760\1041315|']u\986944\1047896=8\NUL\n\119249=\DC3Bkuj]<\36839\52668\SI-\144952t\13080'\1009064D\53913&\US\DLET??\1107069\ETB\SOH\169261\US\DEL\15144t\183810\EOT|zUk\DC4\993404\ACKJ_4L\v\r2Egy{\1004502D\1000475\US\51325B\68354%\1043269\t\SO\1094630\95611X\NAK#\"\43540\"\1029360\DELW;ar\144134\162892!B-[\71854\1091829\1032824\17826#!|k-a?O\1081169\66637Y\EM*l\133471c(\SYN]\17396\ETXc#\1037124m;\1098050\186652\&2X9\21563V\EOTb\f5&\RS\162130h\1060067O,gG\40497\30570\fJ\EM\1039903yex\1066851\r\ETX|\NUL>\GS\1061848g\SO\EOTEr\ACKi=O&\32935\95156\66870\"\DLEV\SI\ESC\95998]\vA2\SYNl\121147~pn\1060695H*eK\1007734%K4K~\ETB_O\1077247fV%\994020\DLE\1013505b\ETX~7@'u\1052404\145314\15969FfsljK~\DC4Igb\185366\149793\1018105\&8Nz\1066984\STXDy\120868\1029213\1087480\&3pd3]^NB\4837G\nG}7\ESC\13917\1075937b]\1087002WM!Hf\72865e-\183494\DC1I\1079472\1054949)P\n\bGK-Qh5\fS\"zA\37747\ENQ\ETX\ETB8m|qL`\ETX\138417\167543:\"*\1075522;hq\DLE3b1\72737\ETX{\38227_\1099994\&9\1083603\70853\SUBce\154253\59189\SYN\US\27371F\35445\n\STXdJ\994035c\\4\ENQ\121422\70826v\SYN\FS\ACK\STX\178978iY\1075666\32236\&4n\t\1083298/\83245\DC1t(S[\61140\61270\137272\1056236\&00`\136921~\1013995^\986212\STX\NULe\1060505\GS[\63183\179892TsJ\42513G\1841\n s \RS\CAN ^!b?t\rFe\54373 82\1113441\1093659jf\1009893\rF<P>,\39509UF\1104490\182554\&4\CAN\1056006\60313:R\166058Jm\1089155\137994G`7_\RS\GSX\158978\&9\171246T\1041338E>@\SI\a@\34297\CANf2'Y\1053735V\1073397U\1011704R\NUL\DC3\ACKu\ETX\r\100983g\984942\&1d\994548\&7<r\n\ESC\bUiX\1054327x\1068164\1017362\20284J\54373zZ\146883\DEL.\44079"
          )
    }

testObject_NewProvider_provider_3 :: NewProvider
testObject_NewProvider_provider_3 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "!\STXOF\1038347\1063280\1099707\1025890\162780\97458+|\RS\DC2NYwPo\\%\181456vWF\48577\DC4\FSk\1059188\8605\RSCQv\1054931\1007187b\162434\152999(\139131_\48010\r\1080015\1097027r\ENQ\1051284\997772\1025219A,y%|\10517\USR\"\1008217w*\1067361B8\1001156k!{\19881M|e"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "\29531\&3\1072901T\1057674\DLEy^PS}c\GSm\bQO5\179258 z\SOA{W\991465\SYN\ACK?z`m&\a\995906\1065279\1063704Gb\vR*p \vJ\986730\EMMjg\aXwt\1108903P\DC11m?\SOHR\a\SIA\1048225\4871\r \1061565e\78358\182465\1007523A2\42742\&0\ETX\1000990\147734x\\t\127384\&08\1019094\t\\\b\154845z]sOjm\60993\\\25349\65403w4DW_\157945\f\r\SI\SYN)\ENQ\EM#n=\98893\GS\DLE\1061649^n\DLEj\1111049\&2.\FS\1077485l\n\DLE+5\DC2y0\n.E#\DEL\n\CAN\FS\ESC\64949! 3\r\r\994087 \b\f(S\NULN\NUL'q{\"Ryw\NUL8\f+\USv2\GSP\1004302\1009748jaM\DC2[\b\r\DELx\SUB$l\165531^\995935[*\44786\93825(\DC4]P|\1045216,i!\SI<\1018545q\154108s\US\1069429\48862`bZ^\1010175\70430%B\f\1065584V\131357\&6<z\100127v+\12864G\NUL\a\1103986H\DC14\1067111\DC49Z\1005609U\SYN6\1005970o\171285-\16076\1053898\CAN\1004059V^\188167\NAK\1110311\66197;H8\21162\SI\27035\&7\129003:H\NUL3\"\18771\&8)\bw6l\FS\986202\140318O\EM\1009051\99551\129539\t\ETB)\150549hmuPMW\1054950N\DC3}%\1109422\36454u\10323i(~\DC1x\126477G81\DC4\GSI\1006806\GS\nnP\1002493\1060135\1011651pqt\EM%\DEL+D\150200\SI?dJQE\ETX\54395\STX\95962\1061906\1092635N\19497Zz7\r0\178939M[Y5i-X\1084550#\1034459Ms\"8r\1034279\120750\DLE\22561ks\"V2\"fmlE\EOT;5u%\1026764\50181s>yhj\1054148\184847\DC1\178638\143460Z(\DC2\1094778\SIL|t-\1021785=%\US}\FS\33497\EM\RS\ENQ\r\157743\&3\156252\EMB\32802a;!\145184'M\1002623k)%\58543\&0+\1021211oM\EOT3K\SOH$\STX-tR>$`\ESCd\US\CANZ\STX\139918RHG\1047715\1083460T-;<{\24537K\v=7.mP\57785#\984455}G74\ETB^\997689N\SIF*$b\GS\v'J\999322\&5j\44525\120532\1094145s[\NAK\170359\ENQ\5017l\998944\SOHxN\138107\&6\ACK\ENQv\189907>9aR\1072783\991540T\ESC\b-QZ\ETX\SO\1015222.\DC49]\1109127E\141111\FSl\47199\fCww\DEL9Cys\SI%\25544\1049438WY\34917R3\DC2*Y\va\SYN\ESC\GS\NAK\r\SO@ U\ESCG&,>]\1043753\RS\70814\f*j{\111286\SOHG\1050064\a[m\33062\93765-\1014570\987944)3qH]\29853\&2 C5Ih$D\1091957[\1003561^@Ff\42948},{\1001921OemlSN\1101664\SO|wF\1059365\SOY\DC2\73980\1045596Kt]\\\ETXBQ\984428\FS\23489\1039581Db\54933\1029118\f\1071214\38076x\1001367\1023030!uj",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "~8\14339\1061876\1676\62256> \SYNq-gd\USrD\ETXb\SYNUx\1012524\145801\8889msB\FSqe\v*,\b\1104122\vs\f\EOT\DLEdh`j\DC1$V^\SOH\SI\180782\FS\v\SO\1014316\1059579\\\"\STX\987401hf\CANs\ENQ{\FS e\CAN8K\1092808L\149813f\SYNsP\1004165oin\1112291\rG\"tXU7S\121241\NUL7C`x\1080124\153415z\n\b\DEL\SOH7\1012528\\\\\n\154325\SUBG\SUB\NAK)d%\993609\EM ]7\22351H5t^+Sg \\\DLE\98151\tbI3\170604E5\1082561\DEL\194652\\@vp#Y\ENQBA\1071377y9z\NUL7\127087\SUBis^&\ESCh\ETB\CAN\RStlM\DEL\n\brT@W\59740;qs\1029789n\EOTf\SYN#\7648\999620))M\b`\175142\154221K\1058957(ftW\DC3\1049977r\ESC\62914_\1043878\1052654*$2,\SYN(U,\EM\CANBFp\vXR\SYN\DC2\NUL=3)\DC3j9\1059863s\1088866-k\152843\53102UG\1001312N}\EOT\63390\1050388"
          )
    }

testObject_NewProvider_provider_4 :: NewProvider
testObject_NewProvider_provider_4 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\22553\ETB\SOH\r\n\1065785:\DLEr\1198N[9T#p\1074919\1000932=Ltw\170949\fQ\65194,&JG8I#!8\16806'Y0Q\1051894^\"'u2\DC4\EM_\n\48528\USD\ESC\FS9 QF\EMbh;O|\1049800'2\1091629\1091090#D?\SYNK\1112422"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "\1056051\1017885\984781e\CAN\CAN'Y8}sJO(q\DELR\70378[\GS\DC3\DELW\1035164\1041367\140020\SOX$},u\44224Kv\166118<\DC2\1005239\f\1051148H&UJ\97244$UJ}\ESC\ab!j\128498T\1029724\SOHeWj\USii\vLc \129430\1067209\&7\\\175545\63226(%\SOH\128009T\SUB\986931KC\179485I\148610#\154740~ZS~+\t\13200H5\ACKu+\185915Z\41528\1014839Yd\"y=G\1026695!\EOT8P\DEL$[\ETB3\178201\175948\863|\ACKI\1111319O\1100613)\DC1\ENQ\47947\990833\US\63986G\RSEG<%Q6\1061516T\RS<eedf6'\983933\149251\&3F>d\SIFE)`\ETB\1071039O2\b\1016106\188492I\ESCc?nw\1084281!\1019342T\179816t\137942\25333\ACK\149109p\78558~lu\a\FS\30978}\SO\5431aa\1049051T+\SYN2\183127XRZ\SIb#\CAN\6026XzW\1105442>\1031193\1081488\v\17921\NUL]\1044228\"%\RS\129665\43830\1016999q\ENQ\1092919\177399\186906]S%0 0T<6X<Rz\172539A:I/\1034055\&4\SUB\NUL6\SI",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              ")i\1056944r\DLE\ESC-\STXg*\"K`\v*\170423uh1_\1040008L\1005143\&7\1031532e,&gWb#o\SUB\SOH\bMA3\994210I\DC4-\n\149425\13988\RSxi\1082866\1042324\ETX\SYNY#m$\SO\n3'\1106171\v~8rte\ESCom\128666\983853~\NAK{|\1014394=XAW\CANN8,\989888w\38860\DLE\1041304\41854B\t&\n,5X3\1096365\5543sVvh(G\21675\1094939\"B\NUL[tE;\16170\FSB\FS[\110878m?\33857\b\97993Ns|\SO#\1086830\SOH)\ETXr,\DEL$y6\SO\n\64726fp\ETBo\5849\DELE\ACK'Ac7\1028812\&8r\1033458g\\#(F(R|-l2\SI?\ENQI&\1041856N\144167\"\CAN:mB\176982\"\1003294\1003827\ETB\FSds-z6\1060226\EOT\2920\DEL\1068685\SI\\\DC1%\61371\&0\DELf\EM\"\24495L\DLE\SOk\1061040?&\NAK_2f{.\128588[\119004-f\1089940\b\1076896(K;\DEL/\1109705\SUBKy\1013206\EMuxcwc\1035156Mx(\1025953\168606>\SUB\"i\1003139yBz\168891\SUB\GS\1007006\985272$\20354\11788*\\}\SYN\aSV\185994\b\\4>GN_K\1060691\US\1015276Z~9N\67275\140457H#$\133494&/di!\136294,ZJ\n\61284=O\ry/\22180\DC1\166277\&7nN\16820l?\187084xp\139245q\1018235\1113697Ao\NULN|G\1046357f\1012394\990651\140074z\r(;P\37544\40482\RSh?Yo#\ETBUd\43998\1021108W\1086070a\1040956e^\1100815T\140443\1066701x\134232)M\b(\44807xJ\66367\1073911}k\1088345\182770x\1068760E\999825\&3Q \25468\NUL^gd\34593\EOT}D\119962j\1010982\166754\171245\f\54736o|\50891:RW.\1016128\NULt\173706\1022531I\ESCy2mm\28797\51357]\\R\RS$\nV\t0&\1017663\DC4\1097998[X^\134396$2\1092149s\NAK\EOT\119576qQ\36946\1050506\155134\b.\40867N\1094546.pSw\bWI\\\DLE\a%r\DC1\f\1006832?J\fJD\rLS@{\21565\&6,6\70167\1016889\59435\US\GS\1102314g\1006074\36666b;\95879U\STXd\ETX\1071251\v<&E)\65449`\64644k\125039\172235\ESCu#\RS\ESC\48613\&8\1093243*@VJ\62397Gm\1046894\EM\998081\DC4k\NAKI \US#_!\1057109\1106140\DEL\rc4@\120445\&5 W\FSGFS\35586\r\tnGb\DC1\1009269\32356\SI>gCr?*#k8P\t\t\STX\1060205\NAKw4u\69434&`F\ESCeNk< 0\1042035\992964P.\137169j\6312|{\1098822k\DEL\1104751\1035825\DEL\905n\bX\1078110\rB\1073963y\r\132270U\1053141\43931@\125077ki\ETB@\69700\FSo\44871ZmB\136818\SYN0\74183\170316\&8\992106\&8\25195\917550\5682\EM\23212\NAK\CAN\v\a|.nCX\1021453\SO|4?\DC2t5h\93800S\RSu,&\57378m3WL*\54375\US3\25258\GS\am/$+\ENQV\61556M8$}\988220\r\NUL\1098816\177561\\}g\1044001\1084867G\1069244U\993218P\993720\187689\r7\"\ETX\35733\&7\STX\1078395\SYNL\SI\FSlf4;\994123#\t\155078\NUL2:g@'K\1092407\120226\64261\44003v\ETX9\178319D*Cjlr\35509\GS)\150194{An3\ETB8\1049329eu\1028429B0eO\DC2\146624\"<rS-L[\161504>\DC4\189274Q\ff\988966\1008054b\179179\EM\DLE\ETXXglf\EOT\1007647D%/dd\DC4\138394l\\\NAK\1093918Q\162404:\152373x!3\53493\1031796\&9\15913\&0?\b\1094421<@\21374RQ\b\ESC\8753W\54203x!h\186330\1000126\&6\SUB=~\190685\170749x\187616\SOH\ESC\182234\SOH\113782\30880\22705\USA\187075\n\to\142657\DC2\aeX\998687\t\1003694\60264F~E\10378q8m\54684\EOT\1001755(^\95224H\1052262\rMlD>\1085731\1022024\1060156rC\1094820pfoc\118891J\GS\140795woa-\1033961A/\ETX\1031114\182383\129557R\1004710oGyj\1053733\v\US\136956p\ESC?vw5]\995093r\182759\1081985#\1098045)DQ\DC4\NAKUh`:){x\148691\1037531/MHDZ\1112514O\48710\&1;tR\190339\1090343\1061274M\1051273\ACKQC\992628\1083272"
          )
    }

testObject_NewProvider_provider_5 :: NewProvider
testObject_NewProvider_provider_5 =
  NewProvider
    { newProviderName = Name {fromName = "\169186\DC24)\1066115\ENQv7\r\ETXp\DLEeI\62252\184208\&4SD^\68867;9*"},
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "\1111449^\SYNA%BD\993875ev}Ia\1033204p\52598+PI~%l\1100225\&6ZO\1010258EZ\145008u\173602X\t\DC4\DC3;\44149\1102500c\1044911Nh`\6362{13\fjQ\EMM']\DEL\1046113e\172961^\1063167\&41\99043\173288\&1;\DEL\ACK\ESC;\40119\SYNI0*`t\b\ACK\1009174Art\f\154422\51261\149654|g\1103749dd<a<}\DC34\ACK\1025537\STX\1010576\tF'Gg\1112166\1059704\1109675_%MC{^\1019959\1049400P~k!\a3 MT\NAK\DELh\1052354tB\30634\1110989\SUBQ\NUL\CAN|5SP1\f\1077042\NAKO\36484\f\1024740^Sx\r\67073h\ESC\t#-l<\ESC\177134+w\37764\&6\1796O\b:vp\1104599\EM\1043850A/\1038337\178739\EM\985335M?MM\1105333\ACK:Kaqg5\1025221\NAKf\a\ETB4\7114g\US\SYN%\134815\EOT\NAK\SOH\1008787\rzI\\\985116\1030772Rz?R\18933L\DC3P\ESCEy\129540a}\n*\156382\1063090S\1054922\41296\1093343o M\156743\1044130r{5!\182331\ESC\f\100714\126102A\1050215\1034201\EMp\163324[\1046168=t\CAN>\SOHP\1112490\t\163330x*\US\DC2=;\153747N\161049\1049506\ESCzH\DC3\\\ACK\77917\161818=M\155629@|\37975\USi=\FS\SI\1087353\1101743",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "\137367\SOH\ETX'\EMQz\SOH%\1097013\158928p;\47961e\ACKiF\1056884\ESC\SOHgL#\1036629\ETB\1106817\ACK\ACK+\SI\1037568T\10290r\181349\DC4_0\GS0\1048032])'\1079448\ENQ'\153670^n\SYNc;\RS\1093718KwzB\RS\b\100995qdQ\30984\1026959\a-\37082h\180782>~Hi,\1282\1094492\135994R4'a\153830\RS}p\1053939\993503\175828P?\ENQ\63540naw\RS\US\1020900@zr\DLE)\171214&\12850\EOT\ETB!|T\NUL '\US)cM\CAN\ENQ4`\165858F\27567=)U6.\ESC\FS7\n__~\1058219Ruo\1064774\993210"
          )
    }

testObject_NewProvider_provider_6 :: NewProvider
testObject_NewProvider_provider_6 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\nLY-\1041348\1053736`\r\STX\36143`\rBa\DC2jv*m\n\DC2\SUBpp\DLE\EOT+pq=P\1006208\185123c~\42806Qp5\38725\&5\1051474X \993515*%\1058138&QNA\SO6\17381e\"K\20379B\ETX=\1095143\179544\&2\59383a\1060828\1036355\1047984\ACK\159657.\SOH\fP\194727f\ETB\134004\&2h\1111875t\73697\1109301i\984751x]cx\nC\53880\ETB9;\65505${\1002225v\1057050#SNKZ@\95712-\DC3E\ACK{\GS"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "5ZK~v{\1069783EG\NUL>\\\18905\ESCb\fR\1001794E\DC4\1017505\ETBB6]<WC\DC1\EOT\19881\&9k5\1031\61301\1016836\v@B\39613*\63775;M\t\SOH\12243\ESC\RS@v_`ce i|g\SUB\DLE\DC4x\51063>(WU\1004045t\NUL3u%%\USy<<\181966\&9\SUB\ENQRcFv\64301;nB?w\1054512\SO\1007228\1094998|\1057826|l]\FS\1033486\STX<\1039613hBlj\EMK\f\NUL*z\171828\33916\1031711\983645*\DELU\1014262~\DC1-\1029248\2299@\162474\1082986q>\186443hw\US\999646 P\1071429G\1024928HS\1028078\USp\1017458\1106202w\30887\&0\1079848\1038142ya-J3\47280\170931\EOTJ)\GSki\100657\NULD|\991333s\8692Ue\1014098u\98183w\SO\CAN|\v.\176149\1083876\DC3\158804\38654\183329r\DC3gH\138849B)\NUL\7577,\"\14164n\DC3\1033534}\17953\179252-V\RSl\NULr\t\SYN\1074389^O\DEL*X\\6t\983423.\1011045F\1097784b\NAK\166381z\EOTr\1068092bG.l\SYN\146626J\1033699\154921 /4qOZP\138964Q\EM\GS\162424\SOH\\\EOTj#ZpBZ#\1021431\n\SOH#_XO\ETB6?gg\984690\tX\1108804g\DLE\1049956b\1024232G\ETB8\ESC\1027355V u\DC2+h\ETXR\1085064d\40186\984376Q\US\NUL<o\23406\SUBT\1053932\70853_\13608\39073\\5\DEL\1062137\118944\&1:\1104230\ENQ\ETXl&`\49154>v\t\EOTtj\USU\f+z\126094\GS\1043582F\\@\1103404=\65898n\ENQ\EMVr\1029362TW\35320\DC2i\1036308\b{0XUg\128654\ETXt\78618yz\127080\987395\aOkJ\29692\1112115\&1\vx\STX~\156743fQ\GS\1058364\ACKH;o\1013385[{\1056277S\DC2K\12676A\t~ho9",
      newProviderPassword = Nothing
    }

testObject_NewProvider_provider_7 :: NewProvider
testObject_NewProvider_provider_7 =
  NewProvider
    { newProviderName = Name {fromName = "^\1061540\ENQMSO\1061583\&0\STX"},
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "ia\168661\SUB$\SYN\24545Km\bFc\990300\DLE]m\\'\ETBd\n\1074149mm-\172683L\166381\49815\99530}\NAK\997914\ne\n\NULWy\46820\168943\SYN\SUB=P=\23523\ACKsbD\EMfY\1007583p\"R\99624g\1013961H\1095839\82981L\n\1058938>\ESCl>Q\DC4BlW\166260\a\v\aT+'?rI)\NAK\1016220H\FS\SUBd\1059263\SOH\1029139.A=>\a{@G<zob\DC4\46916*=\134783?eb\133545;\DC2\97637/`Q\SI\a?^`f&\1091123\1045542h'\vV_?y4\DC2\ENQEk\983607r\96334\US\1098136Rz_Z\1024909xs_3\53137\FS7\48256A\1029257\1060191\1038689\1024150\&3wl6\158975\&9q\DLE\SI&P\ttxEj{~4J\37925\SYN8\EM@\1111833W=\1043765\n.u\1072734\66249\162987*\tV\1073065W|\v\SUBV\11496\983310z9M6{W1\34125\1105761\DC3\1017773\&9XM]\50482u/\989343\1051969\v\994876?5Q\DLELhW\STX\1030833\DEL\NAK\STX\1057813jpF=\US{\54311\&3f*=\EOT\1111272\b=\n\f)=$\GS1La\139716;jzm\153155\34103\SUBS0J\ESC\1058341 \131560y\NAK\SUB^\146239,\998374\100034O\1007421\1110547a\48958\&5 Y,\38755)\DLE\998217\&1+A\NAK}\1103469~\SOH\DELb9\1095416\1035799'3\SOS}.\NAKKLP\1027574\1060207\1069307\&6\DEL'=2\STXC\15138\\H8Uf\SYNLSN\1020627\1018464k\1015885O4\ETB\EOT)1}P4\CAN3\SO[Mc\155065h\ENQ(\139924\24697r^z\DC4#DV\\$I\FS/\DC2Z^i\990842}\180057f\1032320QyE21\v\1009004\DELuL\997965\48228\DC3\145191rds\\zj\154595\FS\1080583\1056692\ETB\DC4d\DLEPM/^\1113029\a\31913G\40326\160650@t\8839\GS\SOna)M$K\1040435\1007689\FSR\1044129k6tv\19522\DC1\1027152/\60615,w\ENQU\SUB\\0\DC4m3`eQ\23975 ]>Bf\92391$G2\ETB\1086260:\1047803\987901v\r\RSL\v^ujRN+b#\42663$CulYu7\SI\SI\43246R\DC4?\1006221XiDD\CAN\EM\7036X\DC3\ESCi\154135r3\1020364\\mZ\1091745\1021202\1109376|eJ\59191\DEL;9\49243\FSb\DLE\1071300\CANp\1067056\&7\148429[j\b\NAK)\1025923?X]c\163450\1074879\GS\33164a\ACK\11702 .yu\1023048\1038766\146038\1085467\202\DEL\171231\DC4M\162885?Q*pzVXTk/\1030152*\NULD\45442\SOH\127893\10454aD!6]\v5x\34604\1079046>n|#\RS\995651\1103063\SOA\ENQ\13278\t%4\1024993\437\ESC+-_\SYN\SYN\1008775\1106234\ENQdH\140340\1091570-:\SIQx\\\1003528\149456a\FS\999457\133210eJ\1099136,\1011024\EM|\DEL\63124\2975\&4\27274\1039989k;\b\1038919\&6\ETBs~\SO\68678\SOzJ\118813Dm/ra\DC1S0 \DC1\21072\EMjw\DC3{\169195\b\1100120C\STX`J7#\SOH-\\]$\EM\154575`Z=|\1053044\US\60192\166722\a\DC2\1006477[\EMkX?\68666%0HOW@Ew\173850'4\168609[B\97112\1016577^m\1101324\1030618tE\1069359Q\ACK\156696\STX\62137Z\SYN\991127M\NAK_R\ETBN\SYN\a6\DLE:\19226S\\\v\1081595-Y7\f\n6\1069300hc\1012380&N\1058076\1100628E\1111712\vCqHrWV\161935\1060915\131275srO\1051706 \1026421/\STX~\136990\34590C\24875\145233A6WvF|\997538MK\veS.+)4|\f\987738m<_{\ENQ\vW]\1058832m\ETX(&\92538e\12977\988576\"\26375kC\35034i\986239\n/&,7\1002363v\EM-s\SO\179191\16503\1039654\47789\993240_\ETB,_Rk,=Fx_\158682\v\1066632#y\8484\RS\37028\NULV\r\1021016\DLE\EMWD\917563BV\1033949\&6\1075019\2264q\ACK\22313\162446\27420\1007484\b\1058123>'\DLE\1001778",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "8\\%\STX\45334s\EMF\EMv5BC\169207NL\EOT\22490\nTASqz)\EOTkQ\1065597UD\ACK0\SO\1001536\1048619\31608\1029373\SYN\1041400\128257\SOH\1074590qM_,\170823+pa8\SOHJ\NUL2r\SI#\1109551-\EM\SOH^\t\21117XuS\158660\a\ESC\SUB_\987790XPC\67429\ETB&C73\1034924\SYN|/M2F\SIS\b\58957\29041I\DEL3{/\FS>\DC2H5rv5\"?!0\145539^SC\bS5\78346XP}0VT\170573\FSD\42106`\ACK\1018911f\1108169\131256Qt`P9~\174262\ACK\ETB\73706\1035701K=\NAKRC\45513\1111656\1049156\GS\rY\1110097e&\DC3d\t\29575\999670\1028927W\DLE\t!\vex@]xh\EMo%\182017\1082183\1078398k\22785\&7/y:\1094694\1012810R\99887A3\15587\1054356*tnwXQk\135931\1071955\1071264\166951\110987\SUB\149345\SO\RS\12606\f\41837'!%p\a\1112098x\SOH\1057708\40264SP\1008184T\158739j\EM$5vA;2Y3\DLE\1078911\65812`{\EMS6\ETB\SYN\139198?\1032088m\FS\34734I\183363E0\19414<.G\DELu\ACK\ACK\DC2\" -C_W\18165j\DC3\NAK\NULC\NAKs\27417\n\FS&G \59660-e\13283\vM+pB|\182560d\r\\,2M\1006641\174549qA\27902\b.\1050452gP{5y0\1008449\SOH\1064365d|\\EUG\48708s\194781\STXF\SYN`\1039218O\DELD\n';\1073324$\57526$D\1062134: r\DC14\100606\ESC\v\DC2j+\STXP\SYNcZF,u\1051699\175940\EM\ESC|y\DELAE'\34865\DC301I\\J\DLE\t\NUL\ACK^mn\173175\&7/\SIs\25751\SO4c!u\1031758P(\r-\t\SI\"\36085\v2Y;\"h\1069038`?;\EOT\ETX&tyLb(P\1093335\&2EC\ACK\983169O+Gt\137064\GS\1052511;wwl8\FSzW\1010249\RS\1023409c\187492kSB%\EOTnC-\EM%\FS\1014861.\GSy\1074065\DEL\1066888\1059368\EOT\1045172\143136\119132\1093794i\1017256&7\168603\DC1\1011326\SYN\1020673.W\189771\&9\49494a(d\1080143,\140528xOX\1103043\1072457\1017273\162724)H\991800\&7K\99033@\USM\19502c\55085K6xk\SYN\NAK\135904CH,$\8540y\FSK\1078320iZ\1103518\a\STX\SOHk\SUBx\66307\1074330h\SI\aYH\1097304\"\33305\1054977$u\27806wk\SI\DC13\DC3\EOT\1029381\1095477>\ETB\1111999D\1071976kHAt%\SOH-AN\20943\a0|\nC\ESCLdigb\SYNO\NAK4\157385>\NAK\DC1\21696\181366\53193\8490E\1091428%\184637huLv;\189960\152600\26674ll(}]8#\1101655\16569\&0(\1039561#\165077cd`-\ENQ\78481P\1035406[ \1089978[$\33084^\1078097,uKr\158491!M\50917#b\146402]i\142972\ETXu\GS\DC3\CANbU.pgb\1060162[N\SI,q\ved\te\FS`<7X'ot\983610\v\158865KK\153342g\184580w\1027317\145275['!\USx/.\DC2\3360JV\1045639uC=C\148379r\1575\"\153627\GSjR*"
          )
    }

testObject_NewProvider_provider_8 :: NewProvider
testObject_NewProvider_provider_8 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\14918\RSh\1071695;Io2:\NAK^e3\NAK\152781Vq=\EM\1090805\ETX\1076024\1111879\DC1h5Zqc\1082985vz\bV\DLE\SYNZh-\vZ4\1008751\&5/KI\1017195\14775\NAK\DC39\24480:q5Oe\1084323\&9\v:xx'U\1106866\SUBt|\DC1O-}\SOH\DC4Zz\1021791z[,F{\1074628\v\2561\SUB4\DLE0-\r\b\USw\SI\DC3a\1074460mv\DC1;\39901W&e|\1009415\1081200;\DELo\128316\140382\SOHds\30510\\\1044298\SI.\54580J.1"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "-\bk8Y|'\51494F~?\1005217c\1062443@\12110\EOT=\2625\12156j\134926>s\SIE\RSZ\1078274VI\984461xZ gz\917991\EM{\46186Z;\97072\149669\1044813nX%h_,G\142406\36906+\141167pgd\1027485\159336V:/\fq~j(+Oa\1100396\DC1f\f\FS\177343/\r\RS=v;\1055293\&9=7\139267\996274\fY-\132435\1080745Z%\1066938\165299PO\42182\1049517\1098441<\150198\f`\ETBz\19818~\143818\164663\NUL\ESC\f\986483K\1113826\156664h\STX\1100664\SIt\SO\EM\DC3\DC2]TM\SUB#j\20431_fzNi;\1031829e3\129428_b\n\1039055\\hn6Y\1045520+g\EOT\EMFJ\DC4?\1034652\149956\43275\&4l\69676\USZ\DEL\1002780'Y\1112729\DC48\167862\996262=PZQ@\1107082\n\998271\50704\989765\184422y\1026191d\DLEe\1052386P\173435L\1001859:\1088420I\48620<\SUB\1045117\f\1079007\DC3|E\1037015\STXI\1066896\&3\n\53108\NUL\DEL\1086453\&7\tn\DC4\nF\986614@\DC2\GS\SOiB\DLE\DC1W6)\167124\&8;\fIl\EOT)\DC3\178947k\1083372_\EOT7<=Wx\54693\"YG#\989937#|=P\1108972\f\EOT\ETB\bok\173839V\1068577\EOT2B]\153805\SOH0e\NAK\162342\38789;@\178942R\RS,NUR\1057281\&7\1043948\DLE\SOH3s,'\1036054\156756\1051100\a\SO\1013540\1098046\&58^\985418\DC4da\166744\SO7\ESC(R\68848x\176364{\13517\110828?l#\ENQq\GSxB\1090583(%j\1019113\96592\f^>q\1065860\&7\40056\1072531XL\EOT.\1107693\a4\179100\1097830\DLEcpr?\37342\&4\1061623\1100388\146350\vS\159853gZ\NUL\SOH\NULh(u\1078620`eRZT=Z\SO\149784\SYNHa\ETX\1082328\180417\DC4x\169037abpmUb\SOH\RS\176077\1005287\1079786RA\1105547\DEL\1038839l),\132201\1045343~0\ENQNa\39711\1018543\157714\35703<u<\ESC\EOTK\28788\SO \988373\100672\1090028=\1023334\96288\1068482W|\DLE9\GS\bk;W2$Z\168331BS\123637\1051859}\191235\1106483\&8\r\t_\a#\\P\1078114Oc>r?}\1063016\SOH\"\13966_9Ge\987967\EMBE\1317\892\1048693\36155xmP\138321pg\RS\US\SUB\v\172821\34377J\172017H\STX\41622\68388<iBj\STX1\DC45\1038048\92276\70054Y\29963\1048707\&1%\127338\1079579\DC2/4:\144815\1022156!\FS\RS\US\40841/\ETB\1044585\&7RnU\ESC\1018467\148903\DC4o%}\1097239\EM+--b5\DC3\23770*a\USNvY\1112117\a\SO\190903L\CAN^Tu\1037697MC[\DC4(\1106190\1000615!H2EO\57654\1092140\CAN \ao0x_\1048887R\1933ct\10212\b~!\1023876\a\1088271\1033676$U\SYNq\SI\160034;\SIL\987720\&9yM\995430p\50736;(f?U\1091949a(e!x\24732n\DC1\1091915\1033919\&8\1104367\53124\SYN\151613\188069\&5x'\DELEb$a.]\27167~\16775T{\DC2\"|\b\FS>\"V\20657\v`( JAC\1010983\US\21086m[\49797\DC4\ESCe\EOT\1027105\69673/\CANSX:s>k\9171q-\ACKC $[M\54581T^46\b\996164{q\1081890x&G\178501\101049\125070\DC1\1108278aM\1054922N\1097695\DC3\CAN\995721G%\1019194^7\EOT\1013772(c\\\ETB\1027044\161679gYb\NAKz+\SI`\SUB\STXhT?@\26098\1108351\NUL!zX\ETBP\1070153my\28463\NUL\48441\&6\161625Y\134507\FSE\38308+:\1048271\36828%\DLE\188288p\GSO\1098236\DLE\GS\CAN\GS9\NAKE\146763\DLEi\164938T~i\1030944V\ETB\143762H\n\DLE\NAK\1064486\NAK;\1106012\ENQ\148194\149505\f\ESCv\RSMq59{\ETB\127967\&2J\DC3.\DC3P^",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "\181673\&2\188239qfb\61095\STX\35333\1094613\178860\STXaC\ESC\73945\181776\65893{\CAN\190011w\1111110D\1054736q\DC1!N.]bQ\ETX*\ETB\US\a\1046887q\158066w\152573)p\ACK\996009\1077236\181330){N5\SI'Eb\15220\SYNx\NUL\1073590\988681\60766YB7\1094372co\EOT,[\1011118kDB\1080594qU\tr\51386s\1036440Jv*\32647)TM1\1006104=\17257\US=g\STX\163970\DC4\EOT-5\fr@f\48920\1018635\&80\US\167132\48845ns8WB\f\1093233#C\t\44212_\96106Fl\152020\bK~\ACK\1000313\FS\160850\&9\SYN.\SUB4R\FS#\165301\16648\&2\NUL\ACK\20280\EOT\r\SI\1084824\\\f),\54307kJ\180664\1058240\&2B27\to?G\1933+r\1086717o\DC3\GS(\r\ACKJ:`\tZ\NULKp\RSxK~,E\1058183\DC338\1000895\CAN\1012342\175298ynza\DLEg\1105136fB\SOX\n\1052810\1113899H\1076167\991616b+\1031624IC$\b\1007856\99469}jc\1018185B\ENQ\DEL+\186466\US\31634\SUB\SUB\SUB\ETX|\NAK\1042051Q\1081506\SUB1\998249C\GSA\1048478X\57384_\145168\ETXT;\1014763\60933\b(\143964\SOH\127965\120553\&9no \1013182v\23196b(\ESC48\a\1024742wEL\139151\1089054\FS6ePOm{\DC1\164267,&zj\ACK9f\FSd%z\1105887\1037745\&6\DLE\1039249i,\DLE\GSy,T\41372>z\ACK\12304\EOT{yUcu\21077\SYN\134310gB\45423\at\NULKa>\165013\990540 \182397\1101375~\t\1018709\1080718yzRZW\1008208\SOHT\1041515,\EOTY~F\190089\DC1\STXZ\986110\\8ei$FViM[e2F\SI\1076629RcZ\1044119\NAKN\29972Kr\bo\b\SO;\STXs=\150865v\188844L&r\45009\&3\\\169632\&6\SI\49045\24830\n\1049549\142815\142409\&4\1068154YX\2107\ESC\1034876\24275\151781cFE\f3\1103509QiPb\t\155982\SUB\a9&\f\1023795\ENQ\NUL\STX\SI,M\DC4\DC3~tx0u\69216c6\n\1092523\1059459WA\59326-\10439S\DC1?\989056\158895^\132652u\DC1|#0\63379I\181557\1112693\GSa\174369\70193r<k\1089077\NAKbN\186743\SUB\US\ETBG\SYNw%\STX8u\98595i\GSO\r\987701\v\DEL\SOHyu\185628\DC3\FSs\1092146\97802\184645\&24\n_W2t\r]Ou&\997242\GSu]\GS >\t%\1083075RkwJ:\1082615^\163156\1070110iI\1112809f\rx\33934eN\176533uE\160118\176741\37761];\27612\CAN\FS\155861\34060\rT\b\162737\136891\SUBS|V\1062665&\1059238\DLE:\rYxL[\4648F,c\51911\DC4n>\1070607\EOT;/+\1075294\SUB\161670\NAKt\148103\46608#<\SOV\29818\DC4$\1040961\&5\1000313\1071500\167217eN\DC1\RS\STX\1103111\NAKu3~\\\166939\92376\172159(\1033270[\152712\SIC<\163961|Um\1096297e\57412]J\\\1034954]\NULAc\1107597#\37549\1043607\996535o\EOTS\1034031\39227\&9\a=\DELT\n[WkRO\SYN/\1055333u$`(\1009830\ESCw\DC2\ACK;\1089329B\1093033I\DEL[gi\992857"
          )
    }

testObject_NewProvider_provider_9 :: NewProvider
testObject_NewProvider_provider_9 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "h{k\7076\162133\99501.t\1011685\DC1K\SYN>O\SI\SOH\49104pJ\SOH\nh(\1058570E\29145\&9\18179\NAK\FS\1819\vD\995637\SOg6F\1054281qUD\53666\1028850hn]f\53775$kZg\984559\t~;\1017416\1007006k\989140\&5\161421r\RS\ACK[\RS\1018346\183709\38751\a33\147504\&6\1069673$#{\GS\DLE\EOTX\FS=N\EM\180147\&6<0%\DC2cdm\174481b\14332\ENQ=9F2:\182353~\EM\ETB\47916k!Q\153214\1110656[\NULeld"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "i\1013107gG\SUB[\7209\4780\66752W\7798\NUL/$\1009041PU+\257\t;/0\1104671\22814U0\DC4\4025M2Q}$W\1107672-\164629+\180552\aM\1012620\&5\NAK\40713_\152762\1034368%\SI\DC1t\a\146026R\1024067\&4o?V-:H\29794!P\GS\ETBN,\1052327\&3\\\16298\145240\SOHGqKb\1095421\1062485i=t|K\1046991G 9\1100130X\1007727o\52560\1086302)mR\DC3f\CAN\STX\SUBNU\EMAf\990360Q9t\137430\134724\v\1082014@\1008310\1098034M\EM8S`[e\95763m6F\SYNG}\1022190K>\5129\1018867^ec9\SIYG\"6mcB4\185768\141653\6327h$\ENQ5+|&L\1019708\SUB\390Av\1033703\63155\vxS9\1085603c`D\"\"b\183025\SYN\6304\1037028\DC3z\147911Q`Dr\1081972R\96987h^\NAK>\EOT\FSM>J;\1029303=n\133450z|E\NAK\1113638Vl\1058013P>\1023218\61578V\71864jz\ETX\158568\175094qP-h[\20916R\60258\&9\1025886*\DC2\SUB\157878+\SYN$@\168650<\177563K_\1085929T\155686?_\12310Iz\146267\1001962\43166p8^>\ACKC\50825\4215?\r$\163701f\1067495#\63211\DC4I\27626'\DC3\SUB\156488l\US5\1041013\DC16o\USlk\1019862\168278\160285y\1107376\ESC\n\STX5\DC1\988051Ohq/\ESC\1072369S\1004141ewtk\1076045\1093601H\30464tnSE5\EM\US@a\180029d\172017\ETX@\ENQ\92249\DLEG`\998659\&8\100314\134376\1112753\151770?jN\131485w)4,J\DC3\68042(\FS|*\ETBZNdJ\31537fcx\DC2rjH$\15593\1109064R\1032162\DEL\1008788\bA#=\1112182\159285T\1043479ty!\NULl\27190H\21766OK\ETB[R)\119851\&1`\179288\GS\n\ESC\23117rAnl\136640\&3n\1012218\r+\173920\&5\1073514%R5tIG/,E\1052116_C\DC2\ETX\988921r\47342\b\97968\DC4\DEL\99649Xy-\1043588h|\154858\EOT\1107330s\SYNv4\ETX.Q\fb,\1095420\DC2C\DC4'\35747JFS!I*p\27316&\\E\SOS8\41799\&9\74788\&4t\166059\14517\1014647\&9v\USV\SI\1071712\EM\CAN\t\50514p\ETBV\b\ACKbY\163860o\b\123144j)Y\n\1049136LE[{\SYN'E: \190545\1022915D\182540~\169128\ESC\DC1p\1024609J\SYN\188438[\22763\1078304\ACKMg\989065\SOHxK\1028362C\rK\US\SYN-\40709<\1061655\DELP\ETXE<[4\1067476KIXm\SOV\DC2]Q*r\SYN*\SOH\1077834\NAK\NUL~gA\96442XB{\ENQN2\1091419\46794p\ETB\USn.}OB1\tU\EMJHB\1060731\SI\168105DP\59373\DC2\1026902$\995828\SYN}\50983\GS\138024e72\159270)&ol\CAN9I\aDOW\ENQi\49440\61899\&4\RSzbB\DC4\DEL\32479L\61910\ENQX\1098102\1041035\DC3d\1039172\&2\1014302w8\1049561\1104622I\NUL2\190813\RS\166456\DC3x\45533(yLL\997567\GSiF\58316SE8\DC4#\SOHT\1019446\DC3\FS\175099\NAK\1055536kf\157091\1994\ETB\ACKmh\25928'Lqo,\GS(\991143\ETX1@\NAKb\a\SUB:\CAN/\vA\t\174987\195100gv\1068869\158920i\1007209\SO\64067\52401V4BEy\1006730\SYN\SYN&\1005681\28151ak\1024289\162597$\1078331BV\GS\ESCnV?$/F\19478\38333\&7.\DEL?/K\ACKX\ACKIZ/\RS\136502\162465\153694x^\175873;\SI\RS\181299\CANrwFL\ESC5T\DLEv\175390\USG\167431x\t ;\STXh@\1007112~\1108807\NUL;\1091011*Mu\ETBIy`FI`X\27731\&6#,Hb\NAK~@\61679\185356\ETBm>\45576`H)E\168943p@~f\994145\\\a\SYNH\10470\158984\SI\1038704oJ7\1003717)$\t8pxM\1080786\ETX\SO\64336]\fn\1084682-?^\CANMda/\1065976Ei(\1113536=\1021758\156775&62.\NUL",
      newProviderPassword = Nothing
    }

testObject_NewProvider_provider_10 :: NewProvider
testObject_NewProvider_provider_10 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "?\990628\58645\50907Ykp*V~z8xa\ETX`6^\SOT\1104197\US\1104107\175563z\995556P\an/\1021466{/O\40639f\993031-OG{\DC3\20273\1025488\DLEz\DLE\US\1043327c\RS\147510j\120360A\1109443\1059885\&2Wc\DEL\120607\1018480P\50798m{\FS'9Hi\1063626hMB\1075646\1050859Nl&\46118\&8}\141343\9870'\ETXg\ENQ\1034698\1006401%Ps)C\178746\US\"G.\1034816I "
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "-uD|3n\1019145+\DEL4+@\1092025Jw~\178453\143260Q\998370\97155r\156084\47609\&2,\150337=i\bN#y\170060/8QhH[\172814\1067147yl\SOs\b\SIW\111157\1029945\100804s?`rr\DLE\1113178\STXj\995436\228q~d}*\DC17\DC1\b+^XL\128422\&0\bG\1065568/*\SYN\137635-!\1012385w@3\CAN\STX\153522z8N \CAN6`\nr$B\rgmp\39211\95604G\SI\1048817\195085w\1072534\"\168231&\1076883\1055456\1094455\64720kZMV\1095414\RS\136637\SI\1091382\DC1\SOH\1104991\152647Q\DC2\NUL}\1008804\1113\182260\168413\1111363\b\DELz;\1083533[\SUB-<\ACK'\990099\997933\1061399\163012\US{3\SI~:X\b\37713!.V6rNo#^M\ACKa`W^@/\1072309\STXKP\bAAT_\EM\GS\amd{Pjw\190755k\1061923\ETX\EM\46993\17352e\43085z5tp\168311\"#\154328 \CAN\127520\CAN\1032856z[ \131518\SOAz\FS:;\7096\tc\1057832\CAN\RSw\10839\1054897\1006182)K\1012194\DC3&\1043568r:eE~\FS\EM\5565;\CANT\1062041\119103TH\STX\1015249b8t\1038814^?24Hb=;G\1023817\ACK\FS\1078\CANXGI:|hBb\25556:kj\DC3\DEL\95488:0\t`E3t\15514\EOT\SYNP\1072656-iw1z\ENQA\DLEu\16461\987644\180849\r3\SOH\1102173O\1070807F\f\1063727\36622\1085302\DC1\145047\byb\5063]t\185443i\1000194-JD\STX\155741I\92474\DLE\v\RS3\SO \1076385\1089483\nh\ENQ)n$1F&1\a\t\EM`\990854\1102491\95178a#\18847T\1029486\169304v*i6\RS-q*VAcXM\61930\155529q\37851N\1036348<\ESCr\ENQ&+\DC2$\FS\DLEhT\51163\1113630i3\31527\1011010\1084149\ETBpl\1054720\SI\SUBR\GS",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "po!l\169867\"\EOT\174362\128174z\"mpt\SIf/,\RS^z\92320\1065420Mb>\1038416s\SI\ACK\142342\1036714\1110223\STX\182903\146931\DC2m\178832\b>V?\1091503x\STX\1051440F\996199-ZT_5\1040575\991138'uO\US\48044\1070465\ENQJ\1079452\fM\CANH\1096172YyF/d-)\95139jS\a\CANF\SYN*}\1020007f\1107059\DC37{X\GSaD\55035\NULn\DC2H\DLE\57851\ETBuu/e"
          )
    }

testObject_NewProvider_provider_11 :: NewProvider
testObject_NewProvider_provider_11 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\983133\999447-\a%~\19798\DC4v4@yim\12033\US37\17431\STX\133604#@\1103392a}mE\DC3\172859Es3C\40960TZaTUy\182080\983082u\SO\159200n\EOT[\ETX\164033\51273_\SOHd\SUB \67980\\#\160768{@\DC1%8s\19295\992711,!D<\92708KM\DC15T\n\CAN@sl\NUL*\a\SYNxZ\ACK\6422 ei:V+y\NAK\SOA\17849dX\991225\991020\r\1057765\EOT\1102945"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "VsX|n\180341Ue;\1011698\1030617,\CAN]\SUBV%*\ESC6x\1063888\aD \992925\1069848)\20475\ENQ3\GS<sw\US\vUbvdvQ\DC3\41088%\131086\&5x\DC1\1024860\1047609\NAKjqv\167992\143956\146485^%\47713vctYK-\1014899XxS\184184z\1034494\&5\DC3\1063392\r\180569d\STX\39134r\vJI(\NUL\STX\1097106nk\GS\EOTH\tF\1079790Pb\EM\ETB\1090881\SI\1069005su\DC1e\DC3xe\177907\1024017V1\NUL_h\92268\&1\188487\1030476\&4,K7\SO\1011692\&9\190249\DC1\178086\31473P\1051917kQ\36997\DC4m`\71701\1009114\NUL\SOH(\1062732!\28794\t\100140\169728e\1032809\DC1\1029076\13020Ah\ETB\f/RQ@\1071812RsN\DC4&:E!Y\20510\994151\na/k\te\ETB}#\RSA\83038\RSk\SI\39955D\37808,n.y\141310\1071743\&9\DC4`#O\EOTlazU@??\190615\RSW\1022054\SI\991970\SI! \11246\183803:_\ETX\DLE\149490o@E\GS5\n\1085344\&6rkd>;I\66316\GS\1071298i Y\140052c\1069816\155750\18272r\1109732V\179972*\DLE/|\DC1:\1054078\111317B\tk\v\EOT\1021165\148266.\SOHGA D^9\1076710\1034449\995766\96564\51227}}6L\1028894FHWcu>({4I]\DC2\r\ETX&|5(<L\1053033S \1062762Qq$%c&U\ACK\1054020\177685\ESC\EM\182111\ACK\ETB\1102513\SUB\\`\173374b\ACKE\STX#\1002675\STXLJ\bd`\ESC\1015371\1039361\GS3\"V\136274\RS\GSb/n``S5\NUL-\1110789\RSr\187903sx W\DEL\1034526 o\NUL&\\E\NUL\4527v\1015590@\SILF\1048392>KWw\45350\1104356\1017312R~\1045720\&9+F\162706qG@\160759X\137010gn^?$\985175E\22030\168132\19804\EM\DC2 \DC1D\1000355\1109024\DC14O\194936#\1105183jC|\ETX^n\r\ESCr>Mw\1004802s\GS\31040:\1027519q\STXe@INe\b\1035911\1109700W\SOH\151249ZU\1045563+\54001\ETB\SYN\\\SUB\1019363\FS\137180\&1\EM\17076\1080538}/7\SI)`\44725:\60221)%k.u`BJ\DC2\SOH$\165777\CAN\1003274'9\983462na\1207o\63953\USsC\1007250|\1029870A\1020136\143203o\1063476#\aa\1109911\&2\1082097\1001921\1100346\22841(\1055876|?[\SOH\17330WzQ\26490I'\185161'?\131103*\"{\123590\DC4^\1076887Gy\14977qs2z\1033620a<\62247-r\DC3n.<\152626\NAKF\164516\f\DC4\v\49294\DC1TR\149493\r\35736",
      newProviderPassword = Nothing
    }

testObject_NewProvider_provider_12 :: NewProvider
testObject_NewProvider_provider_12 =
  NewProvider
    { newProviderName = Name {fromName = "E\DEL\DC3MNO\33205\EOTZ\1042578\NULI\9109\&5e3"},
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "Gt'`-h\EOT\\\FSi\1005777k\43748El\1032469?\EM#\999979\v\\\149309\b/O\22853w\SUB0\134995\1079711h\fOs\nP\1051914U0d\STX\1077814q9\nF\7070\DEL\1082435+@\t\SI\t(\1014239\US\1045410\ENQ-3I\STXV\EOT\2546&~\nk kv\1046337\&9\174462\156161x\1063602sC(xNZT\v9+u\\\10661\RSL\NAKL\1089097=\SI\121048q^\DLE\SYN7%/\SOHj\43172Ke\139593\1030368\1053933\&5}\b\EM\1027375tDB\43310\DC1\1057375#q1j(Cg\30798RG\r\a~8\v8\1016115\163960\1096404vOg\DC2;\SYNp\44408Ip\1103176+j|\57697\1097174TO&W\128745\EM=\DC3T\30511\10099\138775\63984@Dg\46535\37832]m C!\131745vF\SOH\1107976N&(\181736\33533x(j\DEL{\SOES\1088931\ETX-\aP\t\47896+ \ESC-\r\14611r\1058352\1088703\SYN99\aZG\US\1025642L$\137939{\39396\126121j\167315bY~i\120764\1352\987043\1077027Ev|\16556\1042106\SYN\1016063\1111938\SOH`\988243Nv\1007345B'?\1053948\SO\SI\1036376\1027486\b'\ACKs\a\191423\4551\&7'\EM\1068168\ETXg\145992\1025032\&5\FSgDuX\EMg}\1095152\1030396\bRN\1038521/D\DC1O0?\DC2RoG;=\"Q1m\ENQ\1101911\SI\a%<\vf\33383\1092267\SI\DC4\52427\162600N%P\1097224g)=QA]t.\1053940F\SYN;\SOH3.wfEL\DC2Ru!\ENQqN\US\1027949\1075883\DC2\NAKP@I(#\SOH<}*2\187682\128555r\v%E\1036369WlL2x)+\1036631=2;\DC2[\DC2H2)\1080261\83171th@\163467\v\29227\nk\DLEe\DELWF-",
      newProviderPassword = Nothing
    }

testObject_NewProvider_provider_13 :: NewProvider
testObject_NewProvider_provider_13 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\146808\25949\&7\DC2\1076199Y<\1069016+Babu\r\189492\132668\EOTGo\163930:\1052561z\1061137\167675\6623\1091167tIid\156982CR+\1076183\1035056\DC3h+Wju\SI'\ESCc|\1043334?y4L\147691Z\FS\nQ*\r\24905\ACKan\1096641Sw\53656\996906\SI\SOHZ\17777I\26208Nm\SI>\SO\ENQ\b\SYN\STX\1104514\173928\RS,@+Fm,;(cl"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "W\NAKx;LV\131199t+#\2103.~\1032247Y+\41648\\^(CK@\\\STX|p\NAK{\\&j^\1039047$P^\1011763\1094679\ACK\DLE\1020408t\1021953=\1054407\57413\161407dx\134284\7330\304O\ESCR\52788\DC4\EOT1GM\aS\1006329x$d\"\\jm;\SID\NAKh\1026266qd\SYN\1059481G\t\188836w\1025987\STX#`6\994888\1064038$`B\72206h?5?5!~9\b\1039803A\1095391\&6\173990\\9(>\SIq\21843\CAN\1105449Kd\SYN\DEL\185005\97400\173875)\NULt0F\GSF/\DC4*\1100775v\n\135438%\1098360q\985701Z\13600V'\47204\ETB\EOTFi\RS\1097738\33587\1037250~\134709KR\DLE\v8\ACKA\1048461b\GS\1069935E\1093436\1073421<bS+,\12916\1105587\1043249T\SYNP\tQ&\tp\67711\1082522V\83272\1086745o\1106521%P?BB*V\DC1\1045570f(V6\13469_G$AC\143635pL\t\STX\136290\995361\7117\&2\DLE\SI\154519\USf#Z+}\137413~\1023498g\1059746c\NAK|NPy_\1029573'N\1084581\\q\1073885X\22121~\ETX\8345G\986983S\1056196u\1038983\&4N\EOTY\nZ\1015004Df\1102332Fr\RSj\DC1T\1073344\&8\1095382mkK;\988234\1004517$ R\32394\46929\139327\"O.\"e\1030353q#\165792Yu\ESC\73962\STXd\179412\DC2;\1028528N\SUBx\ACK&G\EOTL\171293\184731\ESC\STX+;pi\f*{\1020086\154105!-\SO-m\f\SUB, =\997865i5\1028632l\133598\1033705$\1025725^Qr\1093333j\987815V\DEL\f\DC2\149824s\ESCZ\fC/\DC2O\USu1V\73440\172644\NULI\EMS \132830\1085329ci$2\119097\ETX fS.9\1101010\31813\SUB5d!{\DLE)\181272\SOH=w\996093SC\1002398h<29Z\FSv\ACKi~!\DC4\\\1097982&5\999947 \1036249\163744h~\EM`[xtX\SOH\1088766h\1045272\NULo6~=\183737\GS[=V7\985643\DC3Q\78233\DC1\98063\DC1\143345\DLE\992015f \1072086Lg\DLEi3\60370\&7\DC1\166280\&5\1062150i\1014529qB\DEL\43943\ACK\1087440\5060]\STX\1040000.\ts\167495d\162136F\"r\1017588j!\RS<\EOTf\53159\161381\45408\&9\172695\178465%`\SYN>\SO;\1041631I\1059359%~mH\SOH)\DC2,\CAN/9MZ\186665h\RS\ETX,{\ENQ\45529\&4\48843Mi7T]B\1014292\1107980]\vU&<\1062837=k^\140201\ba?)\97208\&8\GS|\a'\13766}@?hL\44203QV/?p\FSd\1072833iv\1062235z\40134\&77Qj \SInhW\tu\SYN%K\171249]\173157q\SYNft\1055800\DC2jS\NUL5q\173146\171547\ENQh\1056381\11907\ACK\195098\&1n{A#XQff\11340\r\1031758\&3\ESC\SYN@z\NUL\ETXP\161082\2121d>|[U\1080516I\4150\1108384\118925\984455KyRA\14537+\FSk\22960\164720\\nQ!\1048917\156304r4\1031804?\bkO\DLEi?JR#V\157527Zp5\NAK\"\t\t\EOT\1081846\vC@\999902\ff\1080966\ETX\f\13177\35834:\EOTf6\DC3\9643\54350\1030754\1007518\&7WnH*\24639z\187301S\96495\EME\ETB\t\DC3TEf\DC3CP\US*\95293`\1036014/\FSa\1030792\"\DC2\1037136\1104871`\173563\FS\SI\\\1067413$\r\DLEB\83072\1020617\168925\f\EOT\94696\&6~\r\1061388\18245@%\1049452\ETX,I\DC2r@\1094492\&1\45509_\46695\1056680\1003378\82953\ESC\f%54pb\DLE\67690\tQ\49471)\1069238~\1043376.\1095857`\41061\1092049G0\995302\1101979uj\DLEw\SO\49009^\DLE,;\\\DC4q\991711m\NAK!j3t\t\1065021\r\ETBn\GS)\1007482mT\EOT\ACKT\EOT\GSh,\24104\EOT\5452\"\DC3[\26881lI\125058\997657\58945\&5\STXy7z",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "\1012148\996993\983127L\1091820\986234T\100338\32163F_\1074194\NULV\45203\tHSx)\EOTx\1018765i[\13618*\1036707\SYNt]g\1029101\42675;\DC2$U\GSY)\137649\DC2a\bey*_\22406\EOT\986586\EOT\63521_\FS\19309(\DC3\1100285\1018311)`\CAN\tc{*\120605\STX\1035782\ETB(\SYN\f\DLE\"\ACK\177597\DC3\987765{A\DC23\b\"k^\ETB02\169370\1014188<\DEL\RS\SUBd\SYNKh\41799L\140775\SOH\1062973\1017755cG\GS\35290\17766\&2l\1090993\170040>\6743\&7g\1107547\1043309HP8t\NAKA[r\GS$mat-\1044321\DC2\ETX?\171930M\190941!\1008129\1039999\&7\ESC\CAN9HB\70804xL\1106268+\EOTN\9759\58857H=\SYN^\18372O\131331\STX`n,\SUB~+\SYN\CAN\1029939<\EM\DC2\SI>\1017104\&9\SO()J[=\f?MBFQ^}\1106639,\1083228\1091741\a\5309\DC1\134199\ESC \1069328fL+;\176749\USp\ENQs#\4959i\1063701[,\ESC*\n\ETXd\987841#v6HW *T4h\r\DLE\SUB\EM2\1101029\1036281\14006\ENQlJ~j\GSd\EOT\n+n\140173\ACK4\DC1(\ENQ\1001490\1010284\&7y+\DC2q\r0\CANeF\SYN\1037799\136556Y\GS]v\167666\v\7565\1035894cY\SYN\STX\45265R\139637b#-\1068999[P\SUB1\ESC\36885<\999558\ACK\r\CAN<\STX\1073640\v1\SUBkw}\n(\1102782x8N\SYNS\FS!zw\EM\1092262/\b\12914\848o\176997\1014722\1037098\1094468\vwN\FS?\1082514\&7Ml\991879DSr\1083083w\DEL\ENQ\fM\1018937\1034640$I\143124\179763:N\CAN\FS\1067436{ \NAK\DC3Q\DELspHP4F\b\NAK\7459TI\DC2\SO\SIp\160267\984188\&2&w\152534\&6\GS8\16256-\t\rA\17296c\re5\US\SOHD\153968k\150966&\62659O\59796\1075499\bL4J)NSkhAY,v[\1046465wPL\1108537a\EOT\SUB\ETX\RS}\aO6\FS\ACK\161611TkjQ[2\RSY#X4j\SYN+\1048030\34961O\119132S%\"g\ETB-\1094762\98031\1095039\16595\SOH!&\1080355\&3$\SOH\1060654s0g\ETX\DC32\DC3\CANkv\59338\&5\163268fI\189340'\RS5\NUL8\1016814]\22506\DC4\1078481~Qs}IGki8r\EOT\144634\22886\60044\GS6C\150162\43488\1040501\159569\8012DP\SUB+~\147614|m+Wp\131588\ESC;N\1085812@\EM\71307\46826\50528\&1\62468\&1\1006213q\994239'l\DC3\1003448Q\994205\1088573sK.\nx\ETX\bV\1099913\DC3\181734q{\EOTW*l\ACKOx@<\142079._~\SUByJ\DLE#Fw_\1038494(i0\EOTo\31194 \nHy+\ETBpF&}\DC2O4\SO@ZDt\r"
          )
    }

testObject_NewProvider_provider_14 :: NewProvider
testObject_NewProvider_provider_14 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\a3\1030411\1108909\145052W\DLE\ETXf\SUBW\1041360\DC26\1032094\174641^5]b\SUBS\986449\f\1100960|=\1079519\fYu\"\f\44824V\11068\SUB4S\ETB\983122\1069918\&9DD\DLE\1076534\1055217.M0p)\1036040>\140974.\ACK$z,\127809s\1044091P\1053904<M\995309\EMs<"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "<*n\SOH@\30958-\STX\r\DC4Wr\SI\1007828\DC3\RS\1067604|\US\34835\SOHE&\153263\1036536\92409!L\ENQJ\ACK\14244\1031101/>\SIF}5u\US\US9\DC4\174526\&1\n\1030274\1066194I1V\1016260m\ENQ\97064g\ENQx\DEL\998092\"\GS\1108808c\62271\1029396\r-/\SO\21439Tf9\99795\ETXa\1038738\1012785q\58411\GS\1042921\SI1\185051wL\NUL\1005866` o^\139188D\r[!Z\EM]\175566\SOH1\SI`_xlwX\62841~\983117<KL\1022537}\EOT(\1034856}@\DC4\ENQ\EOTz\f\60543\181266\78677t\146084\1004648{rUk\ESC\SOC.8d\SI1\DEL6)\989056&}\CAN%\1021520]#ME93XKFZ+`9'R(\\\1077965%\37869j\\O\1091275c7\1065689\&3\1092626\173993d\ETX\988604\1007993\39893\59277\1040615\62155ce[ni\1025597I\25200%\ENQ\DELm\RS\NAK\SYNR\EM[\150258\&4-y\48105I#\1013675=Re\SI\a?bj'Q*xY\rz\26816Ki\DEL\US\47331jB\\\163045@\1045394l\136300b\12279\&2H\DC1\35255E>\DLE9\1031057l\NUL^V6`\73045\1075428\EOTD oo5\EOTu\DLEMA\1022305\ENQ\EMi\13900\ESCuc\987332z\63131\&7t\1078035W\134810e\v)9\1047085f\96604Fn\164370\DC1\EOTe\68756\ACK\EM(\t\67712\153632\70287\ETB\128139Eb\1072526\&1\CAND\NAK>mqH\SI\998818\94648:\DLE1_l,\r@O?\1030904\GS\ETB\1005437\f\74188A\1087068)6Y\31566\1086717N\n\SUB(^Z/~o\NULqNdf5\NAK8`D[t_s\176045\nb<SY\162552qZ\59611]\1023995\ESCe\EOTKQ\1046546\SYN\SYN\US\GS\DEL\996375+$\ESC?#\STX\US\GS\ETB\996090C:\1067905/\173771]'\48271I;\138834\EMz\181747\DEL&Z\SIO'#\n)\1076303\DC2_\1001406\n\ETBS:7\1032743z>=*\rQlb\EMh\1029213O\EMRK\61627s\GS\DC2\SO\26774q\18070=J@+\nz\1066073\47758U,\SUBq\1086977\59546\&7>M\vY\1109677h\41719T]\1110540[-+\34138Z'qW7%\1099155j6\\u\DC3\EOT`k\180585!o\1091363\FS\f\RS\149577\&1\182629\&5MwmP<tY\DLE\DEL\17530`\1035509\1054274V-\150578\n\1015328\157950\&8\ACK\41404C:Ou\34376\f\175237b\nQ\1097823\1051915Q\54809\ACK\SOH\SYN\"'\1033701N\ACK\rJ*T3w0\151190\184682S\b\DC1\a\DEL\1035463+B\985094=\t~\SO\1001870\73128)?D]\ETX\1006978}\\bo\SYNf\rG\1088909\121516\1017754l\CANG@\62485r\1008696\1068383\157283ECH,\1058997!}\145048\128699\ENQJ3\DC2+K\32663x\GSos\1108205\r\b\150194P<28\10628w^\989952\191154\186034K\RS\184075\CAN\v[#\DLE&b\f\STXz-VV\1076463\aB\EM<ST*r4\171667\1103664*\SUB\NUL~y$\26044A.\r\1038027\DC1HX>;%O\1321]Wp\38640\997816\97169)\DC4\50555\1023202MR\1041904P\ENQ'\SOH\SUB-\DC3d\1089710{\185850'<\SOH\CANRGw\154963\18264s,PwtQ~\v\FSN\EM\DC1@\6151R\157258@\ESC(J\r<\6101\a\EOTf\1026350\44174~t\SI\1076269|D()\987977\&5j\998682\181526\CAN\71219\1075942*\1076536#[0\59233\1009694\FS\132084BCWP\998330g=t\119272\NAK\983872\992436\170639V{q}#((\97889\&6\1017885?:'7F\ETB\13285b\1098369D+j\ENQ\RSB'=\1012088\DC4\1062545\1058232k\1098846\1049476SB\1049144\DC3:b\996836\46172\62008\NUL\60444\ESC\1077185n?\1011254i\78840\&8I\36251[%[\RSv /\21654S\137362\176091?\1110438\CAN\1002877\58915=w\1001772V\1000514\53656\b]\CAN\NAKR%\1099968\1095832}\1063203$\188197\NAK\1064833\&4\\\EOT|\140908Ti\1044678\"\1005661\1046781\ETBEX g\127932 \10082X#q2\1054376<\DLE\34304\1063738H\174496\SUB3W\1059788A\185024Rv\1070771#\154633B2\tF*\141620wibh}+y4c\SYN\155689`4\1047660o0}\1059893\40124EC$-\DC4b\DC2?CF\nK\1076070\1053238\99748Y\STXn7H\68523p\1096385\DC17\1031359N%\DC3O\EM{7\1101262h\95725Pn\162776\a\38956_u\162795\27940\SO\71300;<",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "\NUL@*\988888\60665O3]]\DC3R5\1100473\fi\n\52295CM\DC1r\GSM\SOH(\7837\1009970''\1009430\CANd\131985\DLE\44680\134922\CAN\52334\DC1\EM\n\120079\ETXz\ENQy.R/SY\184388\1099078q\f\DC3\1084664D\65907yrQ{\STX\"\988717\1101264[\1090754\28097\13928\\\DC486_5#\1112391RVq[\995776,1y!\46391\&6\1098357kPX\td'z4\t?/3\1080232\1027883 #a\990167\"\vyn3\1045904\DC1\39168,\54979~nr]\\\STX\ETX0$\39765\&0nw\b\63655\52345{&2J\DC3\SI\FSY\23022tR\"\aj3W\49447\b\1075793e\995006_&6\35718\1072621\v\DEL\ETB@\999269l:)\1023775\EM\ENQPyp=wb\178898\131275\&6TB\DEL\USe0\"\CANyb\1007062u5G|\1078610\RSrexL7C\DC1\21904\95639\ETBxa30Qr&.\54962\179605\US)S4\STX\SYNL\142486\1086193\ESC\1099834H\994133Bt\1043215op\1010932e'\CAN\174666v\171038\&3/\1003061\1006550P\128908[o\158079\165762_\53184F^\2557\&80K\RSy\CAN;\ENQv\DC2jp\172951,E9x\26530\1004321m\r.\DLE-@\DC4\NAK;P\ESCW(c_\ESC\rl\1079440K\163685%\1008965p|\FShtf\1001856\NUL\DC1=\ETXE\DLE\545fgj\1011740c}*D \r2w\160434\989466\156503\f<\73827he\GS\148162\1057494LWn\1108533#B\1109649\t\1035629;\178184q\\pr2\EM\37512\t+fS/\CAN$\1034463y \EM\RS\146673\NAK\1080618b9\68919\1004054\164153\DLE\185153EJJlgg\GS\170643\995180<op\21126\135264]!V\1073697\DC25h\f,3e\1044016X\DC2E\ACKS4\1112697\1072223d\1035227O\162426\614HYm\4191/\52071\ENQ\989632.\DC4\DC3N\ETB_V\r\n\ESCH182\53025H\SYN\"P0A\"\168424\1078056\46761X\141586E\ETX\1013617JQd\n=tTU\1102802\1081222#\178969INg\NUL\14227\183160\DLE\1000207\RS\1083276q\1065736@u\1020367\&5\SI:\SYN\GS\a\142657\SOh\DC3\1043711wi\185698ok\SYN7[\985746\STX\GS^\DC4\SUB\\\187622\1054226d{(\SI\12221\&8\a\986915\RST \ESC?Jn\1050555+4\STX\\Z\1010345C>gO_p\1024807\SOH\1028326\29454\158229\&7I\SUBh\188999qCk7\994783\182522\158185N\1028031\60002\997229c\47190t]|\1061071as\1067993\12397\1104481\&5!w%\183013*\CANX\r\1011619R\DC1\137554\a@#\1042651\183219~\SI\SUBP\RS\1078893\\\36828\&9s>\SO##\RS\NAK\1031948\1084602U\10637g\f{\SOi\985716\v\41790\17576\995331\ETX\GSeo@E\RSn\DLE\SUB\22688\DC3PZc(\95975\1079854\992997V2\1010919z\1112130\SO\24848.\1072164\ACKY_\57589#`&\a\SYN\1113381\&5^\994232\EM]Na\SI\1081598\NUL%\62408\&3$o=\131232\ACK\1030071\1059534\53221%\128097\SUB\CAN\SUB\1101375L\94279Q/S\FS\998041<@a\FS&!6\177851~\1076359\63636\17500\v\144004\177924\b\DC4\EMb\1096546\&1\187997\DEL#d\RS&\r}\99877\129542%t\154521U|D\DC4m\183102\NAK#yC\DC3amKV ?&pR8&4\60835=\1041477\SO3\1037010\43983*t<e\997032\158074\NAK\140593s`Y*4iy\99223C\ESC\RS\1054527/'?\1064386\1008501{\SUBO^Z\DC3YgjN\1076147[\ESCl'\NULz\EM\1066666\78610\&0\n:\DC1\1077656\1040290\DC1\1008843\SO\FSj\1093799@\vB\149818z\RS\132112\SOH^%\SOH\1065824v\ETX(_\b\191118\EM2\1078760+qp\n\1060012.\v\\%\190255\1074295\176885\t\2606\FS\t\74373GthH\DLE\bY\160080b!\139027`?\161131mrK\SUB%z\ENQ\136154\1079880\61618\&3\DC3;\38167Ca\\t_hW3\32952~-\67214\1024930\145076\1097939\SYNzL\f\ACK\167792\1015255\1070734\f28i-/\RS\46849Q\163118` b\vNF.UKPl\v/IJUZ \DC2Koh.\1097107\"h.\DC1P\171245\35427\&9Sg]\ESCk\NULZQ}\EM\1083335UB\ESCVeu\SOHU\1048882.g*N\1089949@\1092264\22439[2\b\47694\59263*#o\36226Zjp\SI\ETX\a&]qLD117:/&$\SO\SYNb+\GS^\1106957\ESC\CAN\1100881~;\992666#;K\994926"
          )
    }

testObject_NewProvider_provider_15 :: NewProvider
testObject_NewProvider_provider_15 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\NULA]\185634Yw\"TOP[f\EOTw\992458W\45093C=m\194863@s\995207Z\1101799\52271\&1 \161939(\DLE\26641[\ETXV\b*\185267\ENQ\DEL\997636"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "O1 ~Jc#c\v^V\SOAW_ \1014602`HrZ\EOTlA/<af\74552\t<\1012334p&3WaI\1027722vX\1097048y@\US&\ACK\DC4*n\a)\1010160\SI\EM_t\146575\US0\DEL)?\DELX&d.Gd\SYNZ\ESC\1078938\1022742`H\nv\150543ul7e\ESC\1102662N<\4290J\1108918U<\1107320\48419\184276_@m\142767\DC4~\\\1101205I\SOH\STX\SO8\53090Dno57-K\1085092$\CAN\1078684\13266\1007071\SYN\1091976\1052959\128906g\FS\1087578\FS\1023649Pf~B\ACK\NULr\\\35897\&9{}!m\16423#KV\157440\ETBx\1052323\174057\9047nO5\170488|\1003085\10104\189241Jj\ENQ\SYN\EOT&RT\1041763t/N\61873u\185882zWl\DC3>2g\\0\1095041\t\DC3b\DC1\68383{z\DC2%Hk\1104920sqUA=&A5Q+\\&\96485\1080698A\1069723{^\US)^5\FS<i\8071\f\FS\EOTx]\1084527yl`_\\a0\"%\6563\NUL\NAK\v\SOHB\172335\1004522\59385:\SO6\65206~x\15522\DLE^'\1111078ZL\991863\SI\EM\1108146`E\140828l\35536\133609WnkPPr\1081400\141336@0y\160064Lg\1002537\RSX:Y\1004793\"\1399c\1045487\STXZ\1020378Z\STX\36815\138068\DC3\SYN\DC3\DLEH\95115}\DEL\DC2\1071249\165518W\ETB\1071184/8nh0H\1013891{\EOT\EM\55187\147758\49624.^\NUL\b\175573\1102759&mgC#$\151906\NAK\95058Y!\140525M\f f\rB\DC2\33967YX U&\1034264\a\npd\995173\25493\153515|\1031451\ENQ\40046\GS=s\NUL\1054079!a\ACKi\1028450\ETB;U\47736h\983608#x\120697`\993465\ETB\NUL\ETBWmF\184672'\1047072T\CAN\1102559\14661\120591\136171h2\175789^M~\f\39871z\EOT\v:T\EOT\990417\DLEYq._M\NUL\15904-\ACK\137668A\176455\&7)\DC2\51606+\EMmdw\SUB\169113]iEA\1044929\ENQ[L\35734\988756p\1112376\v]\26062Q\b\SOH\998697\53020\1113027){\151312\1111227\1082576\DC3<\127951\ETX\"H\ACK\1063000GC\NAK\f\\\n\r\SUBA\1079642%N>\EOTW\SUB\DLE\1004660\989420$T>*\DEL\DLE\f\13059c\83481\SOH\DELG\vV\28484\"<f\1001328\1036606_x\SYNKAc#/#\f=\1084641\SUBjwa\20039K\v#\992029\990385\SOFL\DLE\1087631U:\1110501\"<\CAN\160380\GS%+m\DC2XEhJ\r\186306\46637\&02ELVIuo0!\STX\GSn/g\ESC|v\1013352\"kM34Zu/\146244r\DC2\1090472x\SO\ETBV?P\a\180147\2998\60403%T\41363yW\1000654\RS\3361\51734\&7.LdU-\STX\ETBMHZ-{\SOHFD<@|w\SOR\"\ETXade06\SOH2LO0\988665a)M1\"UqSCY\\O\1097342g\ENQG\ENQ@\EOTg4\DC3\58977G\SO\1100809*`a\1024688\&4i\nQE\\_\4331\FSu\ENQe5Zk\USG\NAK4z\ACK\1026663\145265</\1089858'\NAK[\1016491\an\r\DELV!e\SUB\1081164\1073554\1024557\NUL\42154\SO\ETXe\FS>LB9\b\128051+\997889\&7O+\26560\n`q\DEL-8 iu\159483/\b\1022698\DC4\14906E\NUL?c\26013\rK\127979~\v\175845w{\181255\"M)GU\1017157Y_'\DC2A\1018905\1113059\1073846z}\61520UukH:8J{\\M9}\SUB_O\1010643\1002699\1002442A\"\tKA\131151hHA\ETXt\183766\DC3\1109451\a\998436\1111661\DC4\984796\SI,0\ETBPs\162110P\1100762\26332<2Mf\181314\&2\100692MQB\n\FS\SI@\CAN\SOH\DC3/\SO\SUB7z\1058036\1096521\&9(1Dh\984301\61296\41737g\58322\SYN\RSb0-*U\1009736\v\990248i\1085978\170328D\50254Q\1279<)\138421T1F\DC1\1095280$\EOT#\5075\&8\SUB(Y\1096693\DLE>\1083615\50264\68373\96452AC\1016258\&2\180593\&0\r\DLE\1047223{\182103+\NULy45\1016793~\70805Poa\20510T\nj\41413\tf\1035917O\1098693\f]\1069241\ETX\46803&\r\78306yF_\US\DLEASe~[=/\US \EOT\b$\39764\1112829g\r6x\DLEP\RS\DC3\30137OO\1040211r-ARf",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "U\DELL\1031997\3965\1044182[\182989ki\1105227\43452\1060867dMzz\139329\DC1\"!Ut!\1028584\131087\1022923u\149780!\156581\1024037K+'!x*e$^4\b\1071937\1069725&\"\1073836\18493P\SIK\1075979\147223\a$\n\DC4'%\151627\SI\14127\1043974\USq?\163384N \994666NAsu_f\16443\DC1S\ETBJ\ag(\142579cee~\1053954W\7128\SYN`-B\1108390\96984Zc\SYN\EOT\174552.\ESCt\16487g\DEL\CAN\94849\nt\vKD\SOH\1034271\98659w\t(\SUB\fd\SUB\1025124J2\ENQ6;\DELe\1044132c\SYN\1001450&+gU`hp\DC1W\991261<\vq\1005716{\1109069vM\1018971\12099'</\2921A\1063241!rY7x\NUL7\153871PNJ~^A+\CAN\59381_n9\131982\&8\1027974d>c\150736)\SOH\48889'\bXw\DC2y-eil~\1016478s[\66660\1059951\1007726\42327\DEL\GS\DC1\DC1\NAKCWK\53431tJu7\1083032\f[\41026\FS\991253&\DC4\177728\170128\187772C\EOTdb\DC1T\STXt\bvs\GSc1=`\SUBKmu:3\STX\1056258\39200RMW)\1054845c$Qk\1049257\&2\r+\95965\26879EC\128237R\FSm\SIPG\ENQX\GSN\NAK[Z3aVL?\1094684|\64372\1755\SUB{\SO\177529\1006397\26736p\GS'sp\DC3<c8\SI<\SI'\125134\GS@\1029964yC\NAK\77942'Zb\1066329\1002846\ETX.\ACK P\t\997390z\1053655~\132309>\134636\1041903Y\1094846\164802KD+X\18730\FS\147280vT\vO\156837\ENQ6KENY\SUB\63088T\996443\&5\1030054\SYN\32468K`g \151686\163874\DC1\GS\DELh]*(w0_\179610=\ETB8\12974b\1031125\&6\50363\142908P\1034807\1091246\41373cI\DLE\13509\DC3\181871\24514\US\158574\"P\DEL\ETX3\993828c\1014804 \a\ETX\2313z\SYN.^e(Wiy\EOT\SO$\n\27897>\1086272\7381r\1043910)^AE;\61438\DC49\184834\SOH\120628\r\140592$\ETB\1071703\ACK\61909\33962\&0\148931\&6\83441\SO\US\34292\n_\ENQ\1041026\ETX\DC4\1100537u\GS\26703Q\154632H\SOH\ACK\a\1106457jiyh=<\ETB)<\US$\1009945i\DEL\173200? I\59667\SUBp\EMf4V\160676xP\64680M7G2\DLE\1063464\1076234\1000247\987230JF^\1039383\1059239\DC43\EM\164684\ACK\a/VMhe\18744Uw\STX6r\1035269\181211P\SYN>]Z1\USe\23971Os*3\1110545\1082930\83509\148700\1112586\65687=\1043119\GStDk?\153927\&9t\DLE8\v\FS\DLE(\10191\146750NCw\DC4<\1103983C+(|\1092605\1062435&).\SOH\a+b\bn'`M\58133k\1052890\DLE\29087\t\98313h?\58108\ENQ{/\f\1017948\&9\NAK\DC2@\50474\984618g\100141\&4vKVT\ETX\SI[O\EOT\SIay\b\987314\\s\f\1094655\ETB\b\1079118<|P\181328\ETB=a\1003480\NULD$\1022014b~$\1019186\154912r\20036\"\59634R\1032464\ETX\DC1\31209M*sC3\74243k\RSo\EM,\NAK\n\174795\&5\ETBg1!\rd5-\994199\189635C\1009049&%zz3\DEL\128716\1048947\194977:`M5B\1003865\2713\bg\ETBQg\1094900Y\184417kZ\1053719\ESC\111238\"\f\n\SUBFJ\STX\b\DC1b=\rE6N\3398\74574Ns\1012069!J\1067627\SOH\vVT\STX\1001063\1108342`2/\21824\SUB\1001304za]\GS)\1069987l\1015781\72979q\1056952\1060194dtj\1029226:\DELg\38261\8352\138598*U\DC2\SOHu\RSI\1048699i\38798'.=\ENQW"
          )
    }

testObject_NewProvider_provider_16 :: NewProvider
testObject_NewProvider_provider_16 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\SYN\t\ETB\DELS,gN\ACK+mK\994302>Yz\SYN^jl\988007\1032810/@Up`\EOT\19406\ENQ!\59584\1044424Q\GSH~\60784?@=b\STX_x\3136\"\175451cfF\1049641]\13325XP\63382\b\DLE\"\SO\1055893P'Z\rC]\1033580\68368\DEL\1066018#\1004304Cq\1012973s\EOT\1088815=\SYN\SO\1022969(%\157404rU_~\999971\SYNo|v<\r\185761\179093"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "X\ESC&%\SIr\1013622tGpJDG\96662D\ETX\149539p<'a\EOTkQus[=\1064920\EM\DC44b\SOHf\41601xx\SYN\1105345c&Y`\8190&\161851\SUB@<0C\b&op\DC2).q\fR?L\1038536&\1077065\CANQ\"k\t\162224\1011923\172199bg\ESC\1006816\GS\DELAK\DC2Y\b\27439j\20485.*WG\1001813K\1051809\FS\USZ#\CAN\SUB\143399\149513_=\171023I\1111437\b\17184r^\ESC\USA~\DC1\r\48470\SUB\"\SO}\148083\v\v\190769\v\1025287@\n0\v\"{e\ENQf\159560D\98342\1081596G\20826\1010345\1034684\29478RB\16962\DC4%upl\\0\npL V@\1017155 #ZI\60205\72996pv\DC2z.+@l;\46388\989303~\DELQQ\DELa\29366\SUB\172927\nk\40337\STX\1011577Bw\39995\DC3J\58482\97623w$\CAN\CANhJ\DC1\1070950\DC1\SOH\10490*\DLEj\SO\ACKT\1031969o\92518\1016845\&1$rQ?oKn\996456t&\1019633~\GS3 R8\172847\96523[$(\988171Z4\1038490a Bp1\141901t\1095317^/M\60283\145389H\DC1\DC1lW\995784\EOT(\1046615)@4\1096597dM\1033217\DC4 0m\STXIkmV\"\\\154503M3_\SOH\99718\1075024\1051272\EOTk^TU\ESC1D\DC1E$\DC3\1066144\EOT2''Y\1062130g#$e\41156<*YU\ESC]\DC2 ],\1047094\ENQ~+@\rZ@Uw&IMz\995436\US-\RS\1007199\EOTA<sSz1S\1020087\NULV]\135771Q\66700gf1_\5957%$)\DC1\1058102R<\46518\1010008\157032\1052560\194777\168414\SO\1076251\134171cG\SIt-|\991870y\1018640d\186595G\DC4\1050844\DLEfL+x`#\185443\DC2\163714\&5\179843\1048288\&7\b8p\ETXt]\1012308\SI;\ENQ`nbr\1069613\f\\h|\184466\DC4k\1059290\1048102|\131740rWl_\1007383\&2}\998950M;^|e\CANVVUV<\37199\r\1059746G\42588z7aL\1059204j\n\155251}\5190\FS\EM\1041847ai\CAN\"\1003687\1077824\138180\STX$Nn!\SYNHT-\1058454\EM\994865lj\1085724\SOH\170775\987454fV(\t{0lq\136968\&7RRaBGYsq\SYNM\t=a\66209\&69zd\STX\a?1zF)U\DEL\DC3\"v\ENQM\vU*\9741{\EOTcSXr\aR\4010\f2M\SO\178967aW>rWv-[?\1082462\US\148853\DELw\ESC\FS\DC1\176133\1040362\1103573\184875\ETB5\1064090\1082414lr<D\148710a\1057799\1051043>\NULka&Y\1078886;r)U\STXx\ESC?Uw\165234\DC1\FS\1076798\1095261'\69660\157987\1058027\1001344\SYN\n-\44807]\160004\NAKs\184731<\"[>g\"\SOH\STX\60522\&9\996191dY\1083275\CAN\45431\CAN\1092264$L\SUBIO}\n}0(5?k\132369\993416\1030870\ETX4\43016n\n\49404\146810;\1079380\r\vP^u\1113625\DC1#\14950\GS8\1012920\SUBU\1047742O`cT$o\DC3\1025443\150075\12964psO.|[#\SYNX[\n\ESCM, \1101174&y,\NAK,\61693`d,\CANgMNr\NUL\SUB/\3183E{\11337\&9R-\SO\DC2\1080652E.\n$",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "ScV9\b\28227\DC1/\SO\1082245\DEL~\b@\SIN\EOTp,\f\65205\1018180O3\38816cu\vK\95234\180332\999755Sfi\1093574&\1101969`\FSG\nx\189084\&0\CAN\r\9456>\STXxC\29225\SI\1090313\&9\1006724\bH\"\CANIA\DC2R\65943&\133691%\RS\995833Y\12698]*\EMf!g\ENQ;4\1064020M\63310%\1053835\1009127\165604h\131187w\1029436\1060222\32086\GS\ENQ~t&y\66036i\b\1037992`\55290l\DEL6\1076125\1052539+\9788\tj\120921;X\EM)@m\1070590\74001\1028515\&2`\155271\1025681mw=wbEA\SI&\ESC4wE\1063321SN\17555ZQ\1049768)V\NAKQX\FSe\t-I\RSeY{F`h\NULx\1073233\29275\v%\ap)tXiOV;+'6JKh\"\ENQ\DEL\\h0\ETB0\1035905\1071561K;\1034331\&3]\63716H[\1010772O2f\176755\1078605\&9iOa\ETX0]!\1073255\ETB\FSs>&\1018144[\178366O\147558/k\"h\1054719~#'G\69438\39761vmN\143441<p2\SYN\DC1\f\1032692d\160131q_y\US\993295\1017615c\38536/\32617z)FK\1098072\&4\1084724\1016396\bv\ACK&\CAN\US\ETB\991797M/`\\\EM!'\179610\173585~LN^\27247\1081646\ETX\48752\46443\1067521\157143$\996484aV\1070600\v2jE\1069312XNs\ETXMW\180776\STXO\1939Fpz<\1040375(u\ESCB]Sh3U\190929UI?l0\"<\DLE\1043322If\996227\&2S \SYN\ACKDq~e\184936.j\180875\1058283\ESC)'/8:\SOn\54762]b\34653\1080044\ESC\EM\136094\99562\GSpt\120782\1170#\23092\174851<l\1025903\26576\1026879f*z\188560H\138889`f!\r\188953V\1058442\39853I\37984M`\ETBx.7(k7\SYN#b9<\190703\50041\RS\11834<w\135732\NULO\DC2nC\1058245\SI@W3\SOj\1049991}\STX\EOT\SOH\169045$\vK}\144479\DEL\ETB47\tZ\aq\125105\145517Cmhs[/uo`2\166113r2\11108)^\DC1s_\996021\&9\14396\18362T\DC4%h\1003060\DC1\SOH_9O\30969Z4\49255"
          )
    }

testObject_NewProvider_provider_17 :: NewProvider
testObject_NewProvider_provider_17 =
  NewProvider
    { newProviderName = Name {fromName = "\DC4*43\1113144Xg\NAK\SO;\NAKJ}\1043632wZJ\1028947.=c_\f\DLEI\83112\bmU\SInT"},
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "\131409x54c{\1034\1071731\nT`\46796\DLE\37559\DELP+4/\1098552\ETX\1084429R\998943~\32072y+k\146356/\1066974,\USl>\ETX[T&i$Vj\15083\NAK\SYN$\1066742\135160^\FShM\EM\nr{4\2673d\1110258\1039720IG\STX\94258",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "\b\n6Pq=\1051265(y\1020283y\SOH\SYN\1070877PO4\129454LJ~\179735\GSXn Kc'Fn;;S\EM]\175054t\1021763Ib\1110434\GS\NAKR\1113306FhbLz]\NUL\SOH\129084\NAK\ENQ\999527\157712\SIg^,\v\RS\1020880\177520#F\ACKp\68451g\ENQi\SO\SUB\DC3W\DC2\n\f\CAN\NAK\36197wm;iKl\CAN$\1024275\\](*\17336M\158746\100466\NUL\SUBw\131295t\174884\1076019\1059272/\45343$USI-\DC4\v%\1025183[\44439\996977*dh&'I{b\1027055)Y\992834\ESC\1045253A\CANL\3633\RS.\168314i\1108999\1002085Tie=} \\(S'T\DC4m\t\SO\DLET\SI\EOT\1055839m\111062[1\168091H\68893\30193}&\GS\26186^\1026293\&3H\1010481}G\989043\SYN\34883\987835\176810U\1043435\&6\1090808\ETB\1001041*b\8898(N\1060692x\RSS\1000209\1108699]\998597xL\40420}\165417\b\187543}\148080\78597\f\ETB\NAK5K\51455:\158657\18992\SI[\1112826\FS>v\GS&\SYN\62401\ETB\"jK\1015043#\NUL\15819?\1026355emU\t3a~\1018961\187512\155643*u\998993f_8\1074191\1610\988932e\1081085\153783\&2\6584\1022400+c0\182420R\EOT\STX\38370CW_3\187930]\1017105&\173029qU\1010649?7\SOH\50543\DC2gc55u\20745\38427\51756Xt\DC2H\69244;\ESC\1043616\1004496pP\1038981\983553\134052\1079869\ENQN-\SYN)06\1046451\989374`\36070YnY#(\25216t\DC3&\ACK\63160iu^o5y1\194968#\69945=\99282g)\42686\158228\1081922?\1065914\146177\NAK\28022\22688 \1010793\DEL!K5\1062581fk\t;\1041714g\DLEl1\EM\DC4Vgr$"
          )
    }

testObject_NewProvider_provider_18 :: NewProvider
testObject_NewProvider_provider_18 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "\USZ\986527,`g\1091894E\nY\186037 h\985676r\185228\NULxh{\69688\r2L\b\11555\11451\&3\1028095Q;\rzJ*J\173085L\RSu%\983397\EOT\r\1078571\DC3!\DC2l\t\DC3\48324\SOx\1058847B1o\1015110\SUBD+\n`\122902)\1036058[\1090272"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "L\STX\SI\r{\182574kn\54948\1112345VXQ\ESC\DC2\168966\1030248\ETX\29056'\1051323\54658\159251\SUB\EOT\984766\ENQ\f\ENQC{@\189125<[a\1062371gIW\158726!R\1085578\73018\1095069]\1032243z\18237\74513\SI\118992N\36162\DC4\SUB%1,T&\1073768*; +Lu\SO\984660nM\1000448\CAN\ESC\SYNX\STXa$\DLEacT@P\1040057&\132164t]k6\50846;7H\3525\132215\DEL\132411MY.\a\NUL\a\1046590B\SOH\61091Gdb=\NUL\DLE\a\nw\DEL\SI\NAK\171550\136285Bci*\ACKA}y\DC4\b\992897\1055635Z=\1010352S\1104802Rs(\EM\DLE\DC4\1006612\"zA;jEcJ\97062\\\1086318_0v\SI#P\b\1054348x\15378sx\1034676\187841\146049\186211$W\1050036S\1006908\1108979|6-d\ETB\a.\ETX\1109068\1088252)\FSu\1101581.\ACK\1009497^l\20011\&0\1061172q\95615\RSY\1045270ag\38508H\7662\1024914\1060305\&4e\25038aX[#*\150544=:z\45265\1008116\1099580\&7\ESC\ESCH\985321dl\48337\1087757\EMO\156474\DEL'\1084047\1014854\&7:]\22790x9JLR\177154\SOH\1111189eGhco\168958\997980\998816\191085\1063191/]\SOHR\v=B\176884K\SOH@Q%i\SO\DLE\ENQ\USz<r\n\DC1X:\f\EM\DC3VY\42211t\NUL\ACKe2\RS\98823zjmY%b3P\CAN\n lf\agu'\SOH\DC1A4\1026077\1059659HwuhV;\RS\1079409f\1023144\&7\180415Vh\1046490Hw4\1102520jD\SYN\3910)dMI\bD\150484(\1086721\30232r\147154\149345\24071-=\1028104\120526\ETXK1p,\n\1015644\n\FS\US\1111810\1056917\726\ENQ\1059152Q]\984157\ESCS\DC1\141550s/bBZ]\aWF\SOHj\1050110\148012MT\1078559R6\1105535\22766h\"\NUL9|\179805\181371X0W(I+\152480;\1107019\\\"=1i;8<yp\tg\DC3\"j)hQ\ESC\177760Gv\ESC\"\1077229\SYN\1001587EObO\54379`oCG\SYN\1102615zM\1018774\n\1028245\175280%t\1016596# \174468\&0xzv!e[\EMvPq\1034970E\US\1014494\ACKi\"{PIJ\NUL`T@_\GSr\32234pa!l\12815z9P\ACKph\67399\DLEV1\65550\132732\26917d\vlR#\v\US1/\99245\NUL#~=&d\17142E\SO9\25157cm\SItuw<\DC3\SOHY\38513\1050350\r\b\164666\ENQOJ)ZZB\1054179\5227\"\NUL\128209sNO*6\22297c\110839)Yo\1054695\SUB\v{\SO\SUB8\EMD\147359\993424\1067293bxPK5{\1088019\1006639_s\SOH\v\63607\CAN\SIx\SIme:/\145259\174860\1110334=b%3\187032Yx\1028813=\1019511\1094648\19186E\1053999\vQ\1083050\SUB\986511fN\1023602\189243aBQ\11100.\DC14\95686\EMQ&p\147046\SYNcIJ<F:\1004015\DLE\ETB\EM\SO\22142\EOT\f\58176%\\\67259\35362\178801HUd\1021335*59`3+2o\995654x\1004488",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "w\NAK\1026483\ENQ\1075445O5\12919\DC3}Sq0'\188553\SUB?qk;\165043P7_3\b\154843 r,\NAK'\158759\64158\RSV\12407[\161615\995528\1046620\CAN4S\133020\72193D~zyZ\SYNh\1452^\166390\186623\SOH\rX\1015049\990212l$+U\1112452\1016932\f\1029002&\DC46\ETX\1094239\38544>fh\1021329c<(x:_\1025949$\RS|?_\f/\95240bRvD\NAK\1021075\&66\177037jr{ \FS\1042048h?\984080spq%&Rxm!i\DC1@\1042700\145738\&2d\1098910xn\1004958oU \987251\127283f\143815\vZg\FS2\1087485\14445%T\986010\DC1y\35930\NULB[\DC4`\DC10\STX\133655Q\1088929\DC1\1106239_!G*\70084\71862\ACK\1104224R\SOH\SO1RV\1041740\&9<2&34b\DC3D2\1098671Ow.l\RS\SO\rAr\1007207\136610\ra)\129354]!a\CAN\\9Vp6\1035773\1033675M\DLE\"}`8V\1093475\54144ne\ETB&\1061443q\66316\v\21669\EOTDc\42096\&5a4U8\DEL\996381fcm~\2441C\1072107\984736\&4(\1058291Pk\t''M[\33579\148987\&1\162166CBSt)k\NUL\133601\1067835;s\28250~8ml\DC4\580\984312\992382ik&1}L\63264;D]K\184410\SYN\1101524m\1086014\r$\1006862\159405$|\134896M\1089297\1073323}+=\CAN&\1064854\1105965\&1\155415\140958w\ETB\177782\54963\v\164388\1111379\1102914\185569\vt\1077972\&7d\158949e>38R\1095677\1052668\t%i]\CAN\ACK\f\70075^\SYN\ETXt6\8189\145336\1113611\NAKfgC7(\DEL-tv{mb\1096092=\SUB~}L]\NULPpH\FSqW]\nW\132420\"\992416\rgr)n'\SYN\EM\EOTTNk\40031N\1008484Z\1045093\1086526\EM\1025350\162013\13065=F@!`\ETX;\DC4F\SYN0k|8\RSnvY \9231S\n=\53051!l\ESCnUC:[HV\48689\v\132253\DC2N\162194\FSzV\1034180W\GS\1010999\178061\1065310/G\NAKX\155696\CAN bA\182702n\EMtN\135302%%a\1085149\&8`\1021915{_\1053343\98622\NAK\SOHBm\fk'Uz\vIi\EMy\1102591x\CAN\34225\DC2RmF}Ej\CANS;s\US\1104917-\ENQGi.p\1022704Y2\120189\ETX^VUx\18478uA:,T\SI[y\10483\187556/\a\996661\61864\fz\1031813tq=,S\13269\36567-0\1013620i3ld`*\994221\"\FS\FS!\EMLt'W=-MW\1078223\156247\63941\1101759&x!Q\141701\v\1096038\&0IY\1019948\SOHq\1093064\&0G\170326h\ESCH \DC2\70353\1068905\&62\1080445e\DEL\1026847\1027522/_@\155198\SUB\DLE\DC3~(_\RS4H\ETXK\73112}\SO\1079756Qb\1113993Js\f`\FS\1097049\162022(\ff\FS"
          )
    }

testObject_NewProvider_provider_19 :: NewProvider
testObject_NewProvider_provider_19 =
  NewProvider
    { newProviderName =
        Name {fromName = "\DC3-\CAN\ETX\64721\t/\13387T\1058411\r\EM\1080490XNiu\\LB.+\1084178\132410\DC2\1002277"},
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr = unsafeRange "0",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "\rAc\1042188\1090391n\179482N\f^&\RSHl\1007350\GS(V\SYN\26587\DC4!\DC4P\917878\131600j\156578L}D\SYN#u\DEL\1043116[\ACK6y\1106909d\1024893Qr<\1005754l\21273\&89\1086500.\986065JDWC4\b\95048\RSW~t\DLEx\SUB\1058027H\SUBY\133847ej\1014785\f=\\\"z\63346\DC3NFooo\1017922\1010464\RS;n:<=fq=\161672\STX\992600ZV,<\139419\\\1033183aL\1004843`\ETB.8\1077260\132383\&0\992239w\NUL*\1076012![\135952"
          )
    }

testObject_NewProvider_provider_20 :: NewProvider
testObject_NewProvider_provider_20 =
  NewProvider
    { newProviderName =
        Name
          { fromName =
              "s\141535\FS4,v2\DLEE\STXF\SYNf\SYN\v$\127079\DEL\1037853C\1058568\1098731\134544*2K[\1084006\&2?\GS^1` \162115\&6?!\1056733!\\\STX\12449W/\19460\1066195\RS=\USM\ETX~.W"
          },
      newProviderEmail = unsafeEmailAddress "some" "example",
      newProviderUrl =
        coerce
          URI
            { uriScheme = Scheme {schemeBS = "https"},
              uriAuthority =
                Just
                  ( Authority
                      { authorityUserInfo = Nothing,
                        authorityHost = Host {hostBS = "example.com"},
                        authorityPort = Nothing
                      }
                  ),
              uriPath = "",
              uriQuery = Query {queryPairs = []},
              uriFragment = Nothing
            },
      newProviderDescr =
        unsafeRange
          "*S#b\74541\r\DC4\1072925V-\34350;}\1053334\1057549*h\"&\19378mM\36413PWq%v\ENQ5\40637fk[a/\1013322\100662\"\US\n\1089474\DEL\119911S2/O:TRP\DEL[\10083\1027814)\RSs?\r\ETXP0u)\ACK\SYN\145993S_\152944!I+\tj\54703g\187008\58274\28027Hr\SIAd?\DELJ\SUB\ba\121392\1016486Be~e\60888\994938j\DC4-nu\121370M?\69911.NVdMeOTq\bp-\1070087\1083306!@\"\DC2z6JK\162658yM\DC3r\1057085\r2/\DC4Y\"\SYN\bO\1091905t\1085544\DC4rLS.\STX0?x\45330\1022438[Q\1086255\DEL?\RS7\48888=\DC4@4\9674F\1013333uH\1079345\166828\36384e\996621\&9\ETX|\rL\70661\ACK\18517\SUB\ESC\168536$\142878r\DC19\\_/v\72770U/\FSWF\990960u\30070\\+\DLE\32139\&6L\ACK'bc\58714\1036754\989962\aC2\132227\&6\14310<\DC14\1066102M+\EOTP\fG./9z`]\1059616i(\35512\121389\&2\1095506}RBwMyBE\EOTM\144042E\51844O$Le\DC4A\1036114xj\16172\1018320\atW\1017261xm\36950v~\ESC\t5\NAK\39959?\170917g\DEL\n Y\180069\1041733.dG\STX\DC103\1058041[eN:JO\n<[\164646\1074508c\\hV6@\NULM\139064\US\68657]x\1077972uJk<O\987122Ap8+)Jn\ENQk&@6\1010908\1060256f\ETB\f\SYNV\DC2\r!\176506\1024247\39789 \170314\1094216#_[\152155\78040\ETX\136687\1089362\DELy\1113785\DLE:\166267\1023009JG\157787\ETB\137191\37696\1001918o)\RS\119997;\128714\99765R\991165;w9\US\ESC\STXv\183491\FS17@\fE+*\1074908\&0\988277\SOH\f\NUL\DC3\13101\19828\DC4\SI)f\DLEf\1022296\SI\1107861XA\f!\SOHz+\1086845y-\ACK\ACK\1061806]w\1070674|R\GS4s\1028515\1102449\GSE\1113901ewg\99966\176461\1037580\156701\a\"\RS\DLE]\niAz\58307\162574+Y_\1027771E'\EM)\f\10138\EOT\ETX\FS\154835\1080489\f\1000722\SOg\26099$+\DC1{\a\991723\129074yV\149312\62595\EM/\ACKM\134967\194813\1029972;>3\58112)\STXu\f\SUB\58392\&69=\a\\!D3\155981\1098230[U\DC1l#A\132142@\DC4_\ETBb\128395\136223ed\\\1007712h%\FS=U(I#\NAK\97056\NAK\SUB\ENQ%J\a\ESC39}j_\49034\NUL\45579|h&i{\DC4(\RS\t\1047106/h\50694\93978}\r=eRH\1107604\&6\a\DC3\1046755T\b\141122jD\1109067&\DLEk\161435\1046844)\165661\1052461E\188599\186366&K%]X\190285^R|+\SIUc\NAK@\STX\fLI\1052555eH",
      newProviderPassword =
        Just
          ( plainTextPassword6Unsafe
              "Wx\SO\"\DC2\"5v\v\151001\DLE}\RS\1039252w\1100059:^\43558\NAK\\Z\1050333J&\f<\"\99267\1093524&daGH\1105983\RS\1002907\1097720~\DLE\ACKo\986821\59705\49549L\1086772\13384b\8051/\44192\179295-#@_\7758\n\129644jdIs@r~k\157114\1021838X\DC2>\184946\1065626\1071906\"\nN5\183352-d#67\ACKF^G;5\185762f\1034214\RS-g\140366eg\1095591J(0<&\\\162255z\b\993604\RS?Y(\SI\180344ORXUf=\USxW5=\181334\&3\1036314N\STX!YK\1001668,C\1033503#p\1013354\50859r\179560,<\154639\&0\1036571\996797E\133725\rh\1042728\SI(\DC214?pJMN\DC4\SOd\DC1\7595\EM\1005908\128235\b\94673/S\134129\44978Pd \DC2fj\SUBLm'\DC3i7\139357)2B\985360\39935r7tLJ<W!!\fr\99213<\1075847\v\46647\NAKS\65754GV\160574WmGV/\SYNOZa\43407\v\STX\USV]G\1021259=OaCQgi"
          )
    }
