-- This file is part of the Wire Server implementation.
--
-- Copyright (C) 2022 Wire Swiss GmbH <opensource@wire.com>
--
-- This program is free software: you can redistribute it and/or modify it under
-- the terms of the GNU Affero General Public License as published by the Free
-- Software Foundation, either version 3 of the License, or (at your option) any
-- later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
-- FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
-- details.
--
-- You should have received a copy of the GNU Affero General Public License along
-- with this program. If not, see <https://www.gnu.org/licenses/>.

module Test.Wire.API.Golden.Generated.CompletePasswordReset_user where

import Data.Misc
import Data.Text.Ascii
import Imports
import Wire.API.User
import Wire.API.User.Password

testObject_CompletePasswordReset_user_1 :: CompletePasswordReset
testObject_CompletePasswordReset_user_1 =
  CompletePasswordReset
    { cpwrIdent = PasswordResetEmailIdentity (unsafeEmailAddress "some" "example"),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "uLtYG9FEhpfNHht=ndxYbhEsOaZJJTPjEsBsnJt0UngpmW5OvqpW2F9E5VuFikdraC8s1xMQs9yOzKlgdV4rf371UTMjWzc59HdqfqFDx8=ARnQtIJ8VyAnd784fYJv2A=IiSQJhfc=tc7UKa4n_TN9Hq7BvAQ0YLBFuCRsH1cBsr35-I0aEKev_48AFCC2r2LceURv7tsXpODU=pjneuYFQSD2u8GmiQ3NxRqJEiIfvj3IE_S2gFTqb3Qod=rvVoT7yAejNg=F89T6bacNnzM-sdRhB7ZoQrQYQRc7j7d_1hDOzKsmkBVqpZ3466SwlHld09GyIAYBOo7TipyvgBENFlXnor2sPS2TwCtzmMdyMxhEt780DAdUgiasCsS08_rFrx3j8_wNCBzYsWRTYi7LSaY_IxpcH-mOkH86L=8SAMcCs_pJpKsoWa1EY4Ep0h8jTspHT-6tKd2s0gT_v5GvTPEg8BZyz04gt6I5JgdSrOJ1A0=w_zy4O-KSS-ba73v2v4p3x-N19X88brW3VCwbqgS_G3DAMDEr76Ekn7q0UMAd2MR13SgKWjM35lFtS6vN6b5a4QVqIxOqAvA2EPHV2UY4zGhJsgl7KpgtCzUMKIl-mTyjXP_a_c9y0uu9u6I"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "Tu\1075381\&8\DC3\"9\fB]\178630\DLE<\1082059.C<6W sgS8_\1082572\n\nD$A\DLE\39966I\ETX\996087\36300\GS>\74838S\74072EI_\1003352 a[\148144\995126\119062\ETB0\64445\t\1024616~\51451$`\DC1\132612y.Q)\58664\SYNa\1005229\SI^\169417\984185+\bE\142814P\SI\SUB\SOg}>\129335>\NAK&HS\179057\988796M\DLE\"~?\1025845gd\21005\DC2\\)\1061144F\USI\1107931V\127768\ACK\190997[T5'wP\SI\DC4!y+d\US\181414\1008428\1003322\ah\DC22r\SYN~\1036760I\SYNYO\EOT\1096694sT\140136\&6q\DLEXjp(\7195\SOc\FS\DEL\a\1073646fc>8?Sy\FS\SI\47755\nN\v&O\20470\1050569\ACK/\ETB\DEL8\190265\1097196\168739\98170k\31543\95657b\ETB]\\\CAN\DC1;3 j3F+\1086683.0\64117(p\1069437H\ETBj\12185d\SO\143221W] Hz@Eb\1065092zW}\US\52827!\SYN;\1011087T\DC3![*BS\1069863w]\50081\&3\DEL\ACK\1051925\177178G.A\EM}%\DC4\vk\172371\DC3Tf\DC1\ETX\1008924\EOT&\996766o.\65255E?jK|6z\1080190BC\DC1\4815\189727!>\SOH\987832\t;Y\64130\SOH#\121348\SYN`w%\STX\CAN\140041zT\1113224\&2\1112857\CAN>\220\94946\161064&7 8U\RS4\31842\1086714\1048811--\"\ETB"
    }

testObject_CompletePasswordReset_user_2 :: CompletePasswordReset
testObject_CompletePasswordReset_user_2 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "2I7=q77Q8QCiyaoKVP8TXpdDJIFfol2dXkYJJpNtOp6ne39Ktkfsdhiw-Hs=TCndYZBOHhGwXI7xi-0VYGH=B=6n5dUmNlG8IttqQyD17FRIiylJktAq7ZYH884=9TpOE-woQD=XQiwuAp=hXEIMayRAkoT2K=SHUR2n0P-o9tz8oJ=6DOBtQOxhn3yFtvJe6ZbgcJ0dJGHzG3UDxtAR1SBIjsKNNOVZSFckzxzH-K0IZX32h=oTWcNuqllszjvRpA4Q5JdO-cWpH4j71ZDLdrEbjAdj0l7=DgH187pgSvLedE-xi_gNmUKOKHRAHVd-j1"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "a3d\44313\29830ps\a\136373TjCIX\1071131~\127114[BRsm\38256X\3615!<]{\SYN\131207;jYj8\b\1032200\fNT\167561\ACKHGp\136999\&7\DEL\RS\1004158\83348>7\SIJq<\v @\152552;\RS\STX\92420\SYN&}\1062674-\3615\139147\191067Ef\1082924\EM*\2415/[x<*;0\134125>l7\9860k\EOT\\\nhl\99518q\1059994\1025015\&1\1072462\1056328\b\1036337 %R-\t\1019852Sn\1041872\SI1\985289\NUL\DC2\41090\US\37274[6X\98569nRMHw#tB3Gf\r4S%\ETB\rC=*rQ\v~}\DELH\1038274\&8\NULMyd9 p\DC3\SO{@\ACK\ACK\"rs{Uo9\DC1\t\DLE\ENQ)0\74243m\1039889\RS\1091756\&9o\51528\1108221\r\439d\ESC\137380M\ACK\b\1106682:\GS\SOB0\58417\1105367ss5G\183138nmPIoPE\SO2\DC3\ENQH\ETX\18579p\1049009v\vc\1002107\1011754o^?\153218#H\39389@\38468@\100660V\119100+\FSk\1109219)/\181730\&8NO\4502\44741\&95\180373WT\992017\5700i=e\STX\184895\RS!>\166049`\1100744gtG 3\ACK\1056868\a=x8\DC1x@R\ENQ$|\DC4;ZL\SO\EOTt!\SYN\94514\&6^'\14966r+\ACK}\147730\STX\1099738i\ESC\1100952\&4k1};C$4\v\ETX\ETXH\1087817Z#=\15767\n1%\1069094 \1102739\17301$fz\DLEcA\27111Vy2#\1009541\rF9L\1057880\&2Eh\FS\40121_\DELjm\33979\&8\29452\r+/.{K\DC4\\\71364[\1078116\1084406\41552+yU&F\ETBj\SYNN;r~=\t\158066\988192\145386\&1w8ro\1079235\988454\STXd\1090982mN\1078789\39626D\DC1Ne,b\52130\SIB^\v\NAK|E-$n]\150608\1001296d\f3\SUB\NAK\EOTw\1017423\13701\5962'3\94618\RSO\EM\177775U\142455\&9\1083144:tS,?\STX*\a\EM/\rN\43879u\1041719\&4:t\986292~wGh\127829\995525%mVc\CAN#rd\a\69239_T2m\NAKrDfO,\ETX@ \r+Ab\1112886e\ETXH&#\23182\20467[\18409l1\57990\1019597\SUBe\153649'\16083\NAK4\176\v\13290*E\ETX\CAN\DC4\SIJ\987987jo\95783\&3lo--,\136332@\vHTw\EM\27585\994816XY4\95101u}qFJ\1047741\SUB\1004402<\\\1030039,*~\t\bed\1013495<p3\177125<\194743U\bo4*\160664\SOH7R\13398P|\DEL\34911\ACK\a\ACK\n\DC2\t\CAN\62365\DLE'1[L?\DC1\1082033h3&\SI\1083496e\NAKz^o\DLE\21459\r\26919}<Z\b3T\1012839\NULX\997686\997110\110930R\NUL9\1039737\"\150704l\DC2d\1102787\ESC;=\1047685B\146651\&30\19359\98298,>\a\n\1087537u\30465?c\22040\\\188453\CAN!\1099013\GS2s\1070087\1019176Y;\194995\ETB\1069790=]Qs('<hU\45145L\3268\17380\155296\DC2\SIG&2\CAN\NAKni$\\\148274\DC2\FS\1045693\1110997\&9\12083\176418\63380|[\EOT\990230\1092462_//~V\162313\ETB\v#E\ENQ\ETB0\DLE\a\138601bG\NAK\fTT(\DC4\132894$i\SUB\STXe\SOH{\as\1076325\20697b\1101759^\2039fR\1081173e6AEe\1024419\vq\SUB4^:.w\EM\ETB-/\1042024\SUB,GelX\1063701J"
    }

testObject_CompletePasswordReset_user_3 :: CompletePasswordReset
testObject_CompletePasswordReset_user_3 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey
          (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "cuBaj2u5")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "3Hscxcggm_aoPZM=w-hJ5Z-xXSkr-X4MID-WGI42TXbR8rnMfnj4m7Kvv7PsVwkZ-DVxkR4Fitcfs=G4h0B-R5zmaJ67fShX-KWP0pqGoEsn=9jqUrfzJfHlVbEMgvxITTgmbaLF-yDREL5_fUzeWi1X3uM4aoqRQkU8tnJx6_Rl57T-nBM5v_cQPtP_8KrqWPyXjnA0wxZ72OXbSJiGyVZ8S4tM6ibL1SI2zj7xEAinA2wr4xDFsQRnBLNi0kI9wfsU4Ba5uxiMZ_KJOSFWeH_V=C7LaTX4u-ApwjieRe_O9xp6RRU5d-Dw41ZMYRH0QZqTPzaggqDBt0H9pv0k2E3FO4-foGKNreFIhhsvETA3EGetN-dpvb=v6WvBdV3Oke6KXPAGibFuTXZlygqkiG29afP_IwbBTi81tSsThZ_Ellw7Nn=N5Kqpo8qfTqEVr0ou72hQqcFw-d9F7jG30qhcHMoidQzru1qmF01F9mNmJmmFOlhhUImpVxeLroHheb6UaJj5kUFnlzZDjqtMCjKGaOL3che0NUlompYVZezm2KyIOMhCOqWAYzmsKNZZS1aliDqMrj_2B=txX3oLManHxFJljcKAX00CbObuwOw8IX98TmrZbSWUgpdLGL"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "o\132717h#\v\189810\&6\136468~c$\1049271\\l\1090557\FS\r!\176540\1104154HPdHISOx7\72326\11936\DC3&`abl\1038245}\73889\"\1075782\&7XK\DC4\1105310\1084110\1086598d0\1104193I\GS\1032919\20272\158746F\EOT\182097x#+y\139932P\1056262zjK\168943\136597?\3612\DC2Q/\170239j!\185114\NUL,\1026147n9d\166894$\ESC5F\1090228\1112670\1067210\1066055\189755#\1037604\1035903\1050563=\\\US\DEL\1099678>p)\1026694o\95259\120749\1006590\1087681\1075995\1087348\1055836Q\DC1\286\985351\4312\SOHW\1011595\EOT\SO\RS(|3\28370\25459|\vtfO\1092020D\ETB\SI\SOtV\nFJ\EM%\171766b\UScc\EOT\14709\1091572\152547]\r\38490\ENQ\23972q;\SUB 9\DLEL/\1016110\ETBT\7738\EOT\DC4\21916\153808m\1087717\1090815hwib\152516\92752\141123\SI\1107794I\170689}k\SYN:]4\1112947^h\1041650\1044941U\1016696Q@<\t1jU\EMH6iA\1025178YG\DC4\1009215]7b\CAN\152006\&5\58155\&5\1042194\66650\SOHs-\71063\SO\173468;\NAK\DC4hb\152591(\1090066kKv7\US,\f`\64204\DC2\139140t\175577af\"\1005915w[{\993617V\1001027\&0 ]\DC2\161723DbWCZ0\NUL\98681iY\61269vW\EM\181126i\b\147218J\ETX\RSnuC\190207\&6\1081950A5!\DC4F|G\1074732}\1017103\ENQ'l\98651PP\51035\1048386\DLE9\147758\ENQ[\1000903\160498\EM\51632\SIOO\158007\44043\1081671=\139448\&52gc)-)da\STX\1111103\DC3\1112911\1059428]\ETX\1081735\1092992FBo\"\136534h\DELTU\ACK\ESC_z \1015887d\ETB\52927d\ESC\SUB^@JcAF\fy\CAN\1051386\1015818\SI)\GS\41728W3Z\\J\1077392(\1091646t\1099630\f\FS\139631\1103829s\1045024{<9\SUBOBEj<\STX6\185426[\157863\b\1020557\SUB%:\1056059?\1923\DELx\171156x\ESC\65232\&0iv+\988571eF=\1100724\190781\1013223EP\RS4\1014784\&0\FS$?\NUL\72256\1101734c\1026017\CAN\rcb\96496\158485l\166631\1073857J\1072152'\36926w\SOH\163827,>\1004871\b`\179789\179824\1002436p)\132407(ol\145584t\16650\"\NAKo\160744\27930qsiAV\RSHaK/\DC4s\ENQ}\990099\ESCG\1102699\FSd(\"\157273m\NUL+\SUB\1044674b\a+[Z)_@=r\189255}~\49380\&8bE.\EOT3\133006N\b8\FSGn\133491\DC41\165251\FS4\DC4\f;\f\1067211J\t\1064422\1083725O%\6166D\NAKD%6S\ETB,#>\44276c\139025q\1009286\50801=TfJ]\1049633U\1090987\1090406\tB*xn~\140204myl\188730\DC3I.<d\1093001n\SUBq_\v\1048010q\1012509LZ\1077129<`\GS9\25482%h"
    }

testObject_CompletePasswordReset_user_4 :: CompletePasswordReset
testObject_CompletePasswordReset_user_4 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "pg==")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "FKnQwRcbB-bCQwdDH6OTQsc5g-I9IDiHheM6TakPeV4ZwP3x3i5rlkV7mHpp07ciotHX_jsPVzD4xjtCgFHnVTP1jIflndRa558zctQSaIml6BKUej6fC6WmS_Y1=28ti0mhMAlxsU3FX84O6Dt83kW5aeRuVPK-yrzlEg7k_oWB36DY2F_wwbJMJ3294j4owo1Y-p3VAi7nB9FKZZd2WuJGk3jryStw3BNQIeOY66Y4n3RLeFOA3rfTd=TwjuWlD4M42UfTQkT_-H_h-oJiQG-dltrDXiT0bFtuu6u=Cotl8PPUY1YR5wp_fQZgzeNe5C94hXBctqRszGbHZVzdXbwq0-M3J4XL2tyAXqLhMfmyWB3EwCpo5DGA17=PfMkNBMIA0dIAnF5s=Ir7hOlc=Z98t2SxKcLgD8cxMnsVnbYRd3D3FsL0BSya92yYkqWSF=gE2i61-BiwlN=RReUKoHXWHXMGsjpiud5YykaWtUge78A=V2yPkzYCIbp8_3FWi3-Yp6g5eLEPtW=Dv6fcPbkCrQebdHgW4nQp3qRDCvE-qTH92"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "\SUB\DLE\FS|\GS4\1095966+,3RmA\EOT@m\32970\150633t!\1012663NX,|\DC21!+bnXFS$\SYN~\1063408l\GS\1001960l7q,z\ESC\ETX({hu\1085347|\47604\18689Hu\59396\ETBG\DEL\a\1045753kK\98281\1017784\83140m\b\1022143\97254_\98873\f\5490\EM)+N)\f\132110\31698fmyEb\186967tb\ETB>8G-\1110126B]g\b\USS;W\1041127\43602/\177836\STX+\DC3f\1006938J\SYN\a \NUL\1109096\DC2\b\1051438\1023012!\f\38441e\1046765\&6[U\140861\a\1100802;|O\110738\988796#B\118993t>D'wo\t#\42524o\DEL\188551sq\SIY\GS@-<:sY9P|\42677V\ESC]KQ\SO\rJ\SO\nD\1049001\DLE]PX\83306\1064468\178174\38183\STX\70365\DC1\SIO$/\1113950\SI\997632\RSK\50210AzE\175382\111345JZ\143113\ESC#*\STX\SUBUR]\ACK)\1038781\73070\183578\DC1oW\ESC`\1079062\144075\b1u88\DC4\71456\f\182516\1009566\1073420u\173741C\ETBLj\31808Nn>wD\STXvwN\1075045p\13712\1068437\DC1l;N~\1090032\1005469?\1013829Jy?W\184183\61781\996704T%\1048704SwB\131985g }O_yZ}G\997717Khr\1046472#\1087950au.T8\100298s\STX\1010463,\ENQEF\127517QF \42382U\28642\ACK\1076227\1074730*\ACK\t\1006836-BH3\26795\b\ACK\998490\DELjs6_\98507\&1|\DC3\1047406\1009924\DLED8R\CANG\1078545\DC1\1050566\983288v<\1104164\1093995\SUB\US\SOH_\178145\1055476}\1017836q~\GS?\b\53655P\1048664SV\983485Jd O\68042C\1111952)\54328\121513&\1070370\&9\1044268a\2131\43288\DC1.\19786O\49345\996208\1035987\2817\DC1v\SI\190140\100584\SUB\1037477%\1818z\ESC\t\1017722\119588B[\144627Ei\1094065\1060809~<"
    }

testObject_CompletePasswordReset_user_5 :: CompletePasswordReset
testObject_CompletePasswordReset_user_5 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey
          (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "2cpMpg9R9Lk=")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "uflt8-lJr8O5DUtHfpzwZQ5-iv_WTBeV-pAWB1PHemDlUwPAE89lcppmSr43jwfaSLGrRWovF-APHJjreuOTvF9=HVLO63tQ-lE1=wmlKGIZx_guJr_mDF3Xa5aYjUH9"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "\156237\8492\EOT:c\1078897$\RS{6o|ap<V\SUB\133690~P\35036;\DLE\ETB#\1079086\998770%!\DC4/\119650\a\"7\155595\US@Xu\61141u\1106032\135897\1091223G]5#\1093526Z\141547\1055693w\a\vP\999642Nd9L\159253\73002\1095915\1044734\at|\NAK\1052994\983441Ar\EOT%\25187\ENQ\63780/\ENQv\987313%wxZ\FS\1031444\1047622~>\RS"
    }

testObject_CompletePasswordReset_user_6 :: CompletePasswordReset
testObject_CompletePasswordReset_user_6 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey
          (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "nA48KL8=")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "f2375AvFIRnzj4uO7vdU4GrYBhp_Ddfpv-W_0YOyiS6XyDlnNANkEl6ba9exKJfBhxedtQuAeie7L9OnscS8lG6TpFN=nsuSg3BW4a4hYCa-izXNj7"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "lm\SIx\64700mY\US\157\1097646\an<p\ENQ\119950@`)Gq\DC4\137997\&4\b\1026402\181610\ENQ\65228\&7\1003156(l\EOT<x\37845bDw\f\EOT\134122\25110NV\rH4]\1070740gB\DC2\STX\42806L*\DC3]K\EOTx\DC1Im(pY\151669\1072128\78766\ACK\1108436=o\153201\EOT;]D\SI\164866i !\137461\DC2?F\SUB\STX\1020301\992971\1096206\983823\vt<l(3\ACK\SOH#>G\\hm2\990707\94759@\US\1011493\1073683\129291\74338\70738k\1042928\&5\DC4\997060h\60626\DEL\"\NUL4\DC4\vk\3147yC\ENQA\1023742\ETB70\38101\51770>{W\FS\177211r? =\ENQA_&7q\f\ESC\DLEf\68299\&0\SYN_WP\DLE\1021503\EOT\EOT\1065458-4\51411Z\147742\DC1h\ENQX\1079338\&1hG1\175018uD\1012774-v04m\1059849l!+6\1040942\61614\DLE\RS`\50867~Y\"\1102072[hAR\95402\&0R\1006512\999868*o\SI\EOT:-J\1109450\&8\GS8PF^$\992707)H\169642ix\\\rE\989760(]!l_\1004903\26399\DLE\EMp\vEWR)#\13333\&1j~'\1055195\162244\171208\1095755\183826\SUB\172982H\ETX\SOHg \ETX\99545\ACKP\SUB\"\36838\985803u\1053240mZ\61836eROa\te\SO\vrm\1073174\DC1$2q)N\SOH'\SYN\DC4\1016164\1018115Q,P\58405i\"\64433]\GS-&%\fgKib\185920\ENQ\f\1031003\184381\150165\&5JV\148891.\149578\&1\154582\&5\CAN\987132\b\ESCfz\DEL\SUB\RS]1}\ACK\ETBUXlpT`\152367\&4%\EOTH\"\988810F\"\45177MAHa54\DC4QM\1109997\SO[\1113252/9aN\162376\1012532;oI\96047QR\53331+\1004614\1056061!y~\1091556\n\51298I\1077939|m\67254\EOT\DEL%0yMLir7E;\USq\ACKxm\DC4\1105802^/Ui\RS\190838\&32\991950!t\tZ\1058106\b~}o]\vO\14353\1061463'l$\RS}Uy1\f\FS\159057\NUL\FS\"W\SYNK7z\DC4\US#\1060362\DLE\ETX:\182927O=Wx\127159zR\FSP\1025104\1084512nNm\141492\54516kw\1096168+\28681n\CAN\NULW\1006153/L\1022307D\194734\94549m\SYNjOU5\EOT\DC3\GSe{P-\1016146\&95r\13539x_\1089200\SOojT6Y\161102\v\1006119V/Csbbwf+V\\mu6\1027479D\SI\1027110\v\RS\1007185M\159915o\1050522\26252\NUL\ETBys\29086CN\DC2|gNc\DLE=#c\38985\132906\&9m26\992990Gb[$^\DC1J\ETX\1110084SR\f\RS\DC2;\ETX?\b\43794\59646[\1048757\aO\25052\60145\1007943\1028907K\tp\ESC\1025903J\1024238,L{\DC4iA\6474\vE\SYN6U=\1107989\1095661\ESCC-\DEL\62245\1110781\1112386*oh\1065925J\ESC\992849\DC3\USf,#<R7D~\1023825z\GSr\1025122l\11003L\DC4\127103\171973\SOHW:(vf*\ETX\rx\8305\DC4<a|\NUL_"
    }

testObject_CompletePasswordReset_user_7 :: CompletePasswordReset
testObject_CompletePasswordReset_user_7 =
  CompletePasswordReset
    { cpwrIdent = PasswordResetPhoneIdentity (Phone {fromPhone = "+742946612"}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "bjOVKa11ziNp9wvnuPVs=p83OrCk9-7W-B6VqDTfOqVdJOsPguq08yUjmSp7IdMcCm5=ENBWRmN4cNsd2V8PMi9809xW2HncAxUiLCtGehB6y3DGDfi4l3VoajT3Q7E1EmgOtxwMVog5=ppAFPpJdI0vMKSCknZajOA_pOA4p1yAj-m=iH53-NcxRBCMTpVh2t34th03Lyh_sxnYInBzgsw7r1IgO3Uyh0KruIoK6nF=qzVpuy7g_J0Wpq-OemMG9PApaHZpMfGHKkWIevQrmP6DBcWHfmeJJPmzqIPN_cig-8grl=CLKQJM-tuNC5sEj82vFIa0ewAZ_VG-z56DL9v8J9v2Q-XQ6gK48y9VSFwYt=BFAOLRuwNUkbCP=BeM10OVgE6vd5zsz61M5rAGlOIgEXG4MsrDt5_iUmHsocsUGdRfk0_=SO84vkjxZl_D0HX-PhkYBzN71IIXY4Ycg5iswNSMV=4XFn9Pxy1bYGBnWs36Ayn6kDYPBRKPfZgPrX7ogYKWtFE"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "\fR\179777\CAN\t\998259a\ENQ,\DC4\190155\n-\1063725\"u?\95495M\1046417'\18469e\1019714F\ACK^\1000771\&9\1064373cn\ESCoH\144103I\"\1052687\1038263\35241`PG\174695% u'ad|\92322-d\134905M y-\SYN\SYN\GSq[c\vS\EMr,g\1076111!'Ct'p\SOHq\1049084\&1 \aHn\DC2=p\53403\5748K<\DC2r%\1007160 \1090022]UQ@\189355')\ENQ\986345\GS\1105867\vDxu\52881\991976\FS\331IR,X\163542:M\161860\t\r\1074470Yb\b\10169\ENQ\137320\RS#H\143170G1TnY<!`\98564\1067334\1071270mV\tq\SI\1029222d\1075291\fL\1010288d3\DC2\DC3tT.\DC2\46723b6\1062216d$\1052653nm\SUB\DEL\ESCt\SO\DLEg~\a\r\f'\97314+[\v^\49072n\n\vwD\132532\119850\GS\1006364\SUB\NULP\SUB\992339O\a\v>=\188596o\US\ACK\1020404\SOE\USy\5531\129349sJe\b\ENQ'o\"\1026175\f|\1091357.\1059323:\1106347\ACK\134543\1057161m#4\b \tgl\83000\n\US=\t\FS?/\1034660\CAN;\1085085XOW.\DC47\1087092+^\f\GS\11244\1051834\DC1A\DC1Q\1081204`6X\ESCJ~\985505M\NUL\1092477\US.\USU\173976D+_\1100752F\1060217)\72327D\CAN\SYN\DC3h\1039040q7\FSV\40427\NAKA\n\26138jiDW+d\154046\b\1016645B\98880s\134021y_"
    }

testObject_CompletePasswordReset_user_8 :: CompletePasswordReset
testObject_CompletePasswordReset_user_8 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "HT4YR3Ac3K6GCYNAgnzgbhNlP8wQLztp3kTgMzgcfZNgMDkrRlHRubahDXvPmSCGZxbF6wEPajDPybsW3KXV_vc1-ZSOHI4YxOCsOKiKGyE4LNtwZbG9bKAy9QikyM7tHi3he65-0l7heybNjr3z0IW9Ju4oe8CeQyxdCMuL=Qn3bZTqE97t71neQvy_DXv27gjQPTze2KuCHfQskIZI5LJ=iaTkm7V=273xb-8p=tIzAfNGFcBkq136zeeBPUvuiFa5y7YS25ajFbWF3SxFlbBnxclB_Burg2IFF=S5ueWWB0CrhkL0nZTE5c0blnJ9UbieK8L7LnuzEezronadngQcwL0iXW9sRq6MkpV4KISqdjvaTXDzR61ETnLpzM9zLeCPhnJSI9TLlz7BtgrtIlKvr6OCQiX9UF9YH46KdegQbGog3Hgejpeusa9gQLeKoRRstGqTrw1UERzJv=FJ2h4gUz6YuUKsktYUu-vu9C_Cbmaz1COaqsTGLd5Q8fUJUNYGRrl_6oRdoDcu_0YxfNJXRd_vk=7o4I6dtXhpUdvEH7q0X41bs5rli7CN9hoY_6tgeceOeISVI3amefP4fO75ZJAyHPkDJl1W-P2YpKuu8kylD9LvpanoMBCKT0PGStidrTxW0srUYbQi8O7QJ7OH4Tlwh7ndJ9qZYrxLNakB3SvHy62kRgSWSKAf8cZatgQq69easXNalOvs9J6_yNgUv_QbOuOdDJgXaYxNyetKSN2tN8tMAI7nkjJb5htohSPoIxKCnomb971LqSmHt5u-qvPQL9BCIkF=27CjGBr0MK7KdLOoVE9k-T=06uLB7Ah8vNuH6-p0npqBRKcznIVMWCUrqOaMJLvdpbHlPpanfaW1JPH8_HQsUZzW4WKvlQWW0QNb1c-zIYpNw-LXA1NDnYMOu3Rgdg0nDsa8jkvf_NCGrow7ncDH5DdNQc-HnIvzv0NEK5V7y1iLBsW"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "z\179971%]MV\141181\986464phO\3659X\143435v\b\CAN\1068207\\\DEL\51152Ps\1075457\1010611\GSY\t\983543lQ\np4h\FSi\1021645\146896\DC2\1061677Wv\176539\ESCF\SYN\SO\1086781BL\DEL*\STX)M\143511\1025207R?\167872\1096957\&9h,\CANZd\1100074M)\998933\&6g\RS\n\GS\51036l\1096374nhX\1009693\1031085\128645),\ftEZ\1076505X7\DC2X\1073715\62654%\997854#\177266\DC4f?\1055981\DLEa\US\DC3\1003011\42823\STXp\1053817\&7D\f!Z&]\SUBb8>\\Y4\155833\143579\&1-MG\DEL\n\178111\GSw\137384NZ\US|=\RS\155356J\166255s\984393\1020876\98188q\a*\183120\SYNemSmVk\189895mU\39988\r\DC3uz\NUL)\DLEA\152453$\1094006?&\1016471|\SOH\CAN|x\988322\NUL42\t\1089812lm\NULz\1022172_\ESC\bk\999120/\DEL\54527!l\1028113w\DC3#'p\GSK\EM#,3t\1068160K\135067\bI?\7149\SI\f\FSQ..\FS@\SOH\ACKB3pyR\f\1047730\1112024\ESC>\bx\178913\r\1064564o\DC3J\ACK\1015652\1010510\RS15~V\999893\ESCw\DLE\ENQy]C\992636\1060824m?\SUB5$\ESCD\461\1094742\1085880\&8\93047_V~N0\142058\46275\DC1\US\1049132\SOH\78616\DEL\65422\NAKIEX%\v\1074332~\991362\156757p6\NAK\184175Z\1012970\&6\NAK\US\1071418*\1047229x\FS\aHQ\EM\DLE\9533\1049878\1079830>dXY\ETX4l\987867.*,Jt)Q[7\US\rP\SIgy\154044GB\993355\ftX\DC3\DC1OB\1008239\tV\NULKicto\27617\1013290\b\a\FSPqgE\143570\1101916\SUBk\CAN:q2\\\1060253\NULe\ENQw\1045638Rqk+\100693)\FS\988176\&9`\6293Xj\SOH\t\186270\984047\ACK\"\n$3\1008823\141341(\NULt\vb/%'\185387\NAK\DC49@Hb=\ENQ\119536n}o\SYN\1032201'\1025326\19310\to\t(\1070036\DC3\v\983672\185675\ENQ)\ETB\1091867\&9\53850\EOT\1092243\GSEv)$\9575\b^\1002235\1032326v6/\8549\1058328%C.\RSP\a6\DC3pI\11955gpH BAc\1084789Jb;MFI\SUB\41110H\SIVI\153792S\158205I\\\tZq0&\GS?/DE'\1009961\SO21Kq\ENQ\35084\1024466\\ml\DC4F^%I\SOH\174592,\984239\168715x\1035028\43951\\SVb+6\EOTv\b\SOHzA_\1046012\DLE\998932*\1085737+\7522\1049016@>oP\1041902\ACK\DC3\1090159DK\142427z-eh\SO\181642*\SO]Cb\994205\1012718EQ\123624\34155dGy\f\1032174Z\1019635M\74969M\1038424W\DC3\FS\1014867\7823!r\SOH\FS\169164\26472V\1053481?K\82977\"/N=|Jdr0\\\77990\171276$U1\121430rE\1073118\SYNYG\1034547\35447\DC4\SIs\1076916\57864NR9lmd\157485\SOHg\119340\DELFK\a%aq96\1504_\20911\STX\33475\194700g\b.K(\1117\NAK-\51093/dc},\986196x?\18309V\t\8497UC"
    }

testObject_CompletePasswordReset_user_9 :: CompletePasswordReset
testObject_CompletePasswordReset_user_9 =
  CompletePasswordReset
    { cpwrIdent = PasswordResetEmailIdentity (unsafeEmailAddress "some" "example"),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "uyqP0_aQl3yI0f7i0fpyL6quXIf6WSJRbPrU6Z0j2gElHzfIXLenrK4ZwQl42i99XCnAjLGA2=sQczG10h7DBcYH4TmbO-li6YDpcduZ3XkbGQ=EalL5L2xZbwUpVFGp5J5e=yea3gDvfUwq0sdTrRFCFbTJBG5cU9K_5zQMB=DTFJoHAh=L_0uTZCRF_bj36cGxLegs42ji4GGO3kG4kcvpSCMpJV20a47V7GbqfEdQ3HV2gdN5CXpWXxRu71Y2XvAMijj8O-ciqslgJCveAgm6JlkZJf8-Cbj3tmBD1xYveBLOBVOW1=vaD23ST6FDLpzbRJslhJzwInpu5AaIxndPmLzeXH3I5mfrMBFyGO6e9Pro51aJPGV5COmIinyjxcM-vEmWYYkLy7owuVyswR89m--SRwgOWL5UtF-QbkS5bpltl6BmnrTEeaZNMQRPrcpPL4RT=0GFy=ka7Oq1Ixi5OR5EDYgIa_Rl3I9jq034w6wCQjW=33Z5wFRWcdX4lfqvA-66Huc--Xk3hAKScqNeL3Xre5eN1pwOrEFsMhncwuGoFZoXaHSMrQZEqVhVJcFA8afI_vpIk0Ft6NMcS3AtYLQgdqrvaBe42_s"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "t%\1106482'\1085275G\NULG\DC1\1065237V\92721H\54755\1048145[3\150108\ENQ\142934Y\47443\nm\v\1001941d5\1090795O:\DC1\1099456\USb`\129486\nW\EM8\NUL\STX\146413W!\DELmX\132012z-\1074871\65424 \1017374M\axT(\v-\"3Xx\a\f\990702b7*P\1021085.\n\ACKfz\EOTN\121274\EM\24372Z\83127[7\1042390luXV\ETBRh\EMU5}y\996537&\1053537UO\155409L\SOz\1004758\DC3G\22883d\37015\&5(iM\SUB\NUL\"\1022433QKY]_\v\DEL-\1006612\145121\&1a\SOH}5]'L%s7\1097491\&5u\v9|F\ENQ\179220j\US\1034740L\174323\DC4o\48579\1034153\1099174lqDs\1052315 ]\DLE\ETXf\SOHbO.\RS\GSj\1002207E\"\r\1035749\162742Ig;~\US\38681\ETBWQ\DEL\1038511\100506?\1099376\\e:@\4348\986455Cq\DLE\9822>\CANzT]\FS\DC1\36942\ENQ)\ETX\1028763\&7\172341\SOH\ETB\EOT\135000r6;D%FM\":\1053022\SO-S\NAKc`]\50260\CANO\v$\1105473\ENQ\\i\1035531\tk\133720nME\GS\CANmi\1025051~K[K\1008508\&4\DC2jl\DC32\163406w.\EMO\1036572x\1022876\27050\SOH\1072386\34498:JY\b\27018\ETXDPH5\1071848Hh\STXj!y\994661\1064566\FS\SUB0\NAKg\t\1065452q\v(c\bR=\"\1008081\DC4\DC2\NAK\SUBi\998703M\1064362\177599(\24166l\179007\DC2\136782*\32877Y\1041066\453W\t\NUL_pk#=r\1032661{HN\1111623\1021137W3 <%*0\140910?\CAN|Q\151232oE*\SIx,\ETX\15845\74304L{NC\1082549t\181800I\DLE{\NAKM\DC1<\1010573efpSGC\988853Cl\SOH-+\t\f\1090761\DC1'uWmR\1088797\133403\&9\1045264>\142848\&7\DC2s{\1000097my|\1113699\a\NUL+\1035609t'N\DC4\SI\31012\&7\1084643\1027431o\1082014\1096666Yx\133375\&5{;\146931\&0XM#\1014368Q\30231\1081694\tx\1068519M\163412^-[a|\991629\32871\EM\\=\tH\1065484\&2\DLEW\DEL\1084570#h'K\1000324\vpX\EM\SO\t\98778_~\176195\1025397G\USwX\53603\1082556\50458@/E\SI\1006180\DLEQ\157258p#pR\1007809\DC2\1043179$\n`l94.\171953gun\EOTRG\ESCF\989700Hbf\13205e^\1007389\&9R.\b\ETX;\NULN$4\fVy\160381,h\36122\168831\991129\"^\33713J\NUL#\184356\8099\78814z{|\ACK)g\1085443[DK\SOH:(jNn\r:D\1112747dM\1054404G,\1088216\148672s9\SOoy\1107184\990596\&5Y&\164376\v\153046\SYN\"Ob50\137070Q_W+'\"!\74569\t5\155405\1061053\16324>\SUB\148951\&8\96940y:U\1051061\DLE\SOH\190737\170011\"\US,q\SOHa\1113047\SUB_(j \136574B\1107369\62624\DELB!\1090388RL^\142739\b\97613\51579\af\CAN\1019397\niK\133374\EOT_\15082\52701\SYN\992724\a\50769\f\FS;\US|V\996997\DC3\DC4\1008815Km\RS\ENQ\134593\1054688\ESCnz\DC2wy\141383\1079323\156366\&5Q\FS\71249J\SOH\1058926/9Ew\NULSbe\a\35263\149348\ETB\rd1\1006994\20380R\7068\SI\128999\SOHv \\\182039v`i\61613F\183123\&93\ACKdSl^I2.W\64005\1012650V%*Nv\1094255\1047508]\28666xi\1011412\917789H\27528\USJ>M\1058131\186155_&\190081\990889\1012542\46367.unqC\EOTy\1047316\DC1i#}\1061110\147400`^\1043640Q\ESC.\RS)f\990553\146957\NUL^\NULi!\US\49231~\\\1039256t\1096218\9343\DC3\n\STX;E\v\ENQz3\1022880\&2\STXv\ETB5k\1023115\178850\ESCs\NUL\v\43185\&9Y\92218\SYNQ\1087318\ESC\\_7\1106053\1092199J\EM7\1014160v\ETX\45857\SOHG\1048548{\CANPE\1073228\150502\DC4\13687:N/\1099530\1083947\1057108\1032054V\DC3}\174207m>{\147256\&4\NULI\ENQy^$\n=)S; R`\139984@xtE\SI3"
    }

testObject_CompletePasswordReset_user_10 :: CompletePasswordReset
testObject_CompletePasswordReset_user_10 =
  CompletePasswordReset
    { cpwrIdent = PasswordResetPhoneIdentity (Phone {fromPhone = "+9868360466"}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "XN46KP0FmYS7lNGLjjqWYNLUWhPsIIWw_uCi7=2ykPbWDB1UMR97y3rlxkhy7rUQPMyoVm5bwoVGATyN1Tps5dGy8ZojWlX5ESixwgzTJ-4JiOGMwZAkzA8ebLHE7w0D2APu77pur3=X2pU3=paqSBe7U6qSgYP0vmogT_XVJEHvEqGP9qI7wCeF4mBu8WyazncZ8wvb_Ag0AW1vfq=U9coksPKZLLqMByf_XXCgtuNWIikeLYx0qhNMy4cLjYyKXQOfnfFUvYFj01x2pOJqVOnz="
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "Pc=\70672\149108~ix\1049331\&4Bv=j\1032226Rj\1098224\ESC\RShm\STX\164276F\163967\161659\ENQ\169704\ETX\DLE/$,\vm\1044889\&0zS\35123\v\DC1\29224\STX\n\ETB\EOT\30015N`??\22180\FS\nh\1096330/m\RSe\SOhP!\f+Hv\NUL.k\"\9280\ETBQ\1106065\132462%\1073477\1025415D\SI8X_pI\DC3\1034791${c\166694\182232\&7\999085qZ\GS-\ENQvze@\183653\SOK\FS3\1039436\ni\ACKi\146750/!\DC3\1005298\bTV}\ESC\CANRlY8\SI\135485xo\53151\\\189401\f\1043586\135041H\NUL\53392qZ\a?G4\150393\1070479\147240m\988567WX\999319L3'\151394S\167031\25890[SlhR\153001'6\1024896g\1064214;t\NUL\v\ENQ\1016655'Wx\NUL|\141526\156733\DC1g\4560\1031693\v<\1079492\1047033n\DLE?Nn\ACKe\SO\37093F;\n\12097wG\24904\SI9\1051218\SYNpCq8\EOTO4| !\US\DC3\38278\EM\NAK,`'{\DEL4\182683x\1023061\NUL&~w4YtF]e\100748#\EM]\179800\DELK8X\STX\ACK\14007\141199J\FS\1082822\150384?\1066331\1093677\DEL%\184568N\t\STX\DC1\162868\1009718\170538\"\1074945\DLE\48612\ACK=\14180\1024850)\989131\SO\b\ENQ,\147667xI$\95985\139748\SOms\162643K\13701R\CAN\37575\&9\CAN\64928\"\rW2\1044418ql$\154464uw\1667x9u\166803\4595\ACK\50454\134067yT:7*\164509\1009873v7z>Y|\\\SI\\ _k\1091148\993199:\n\94939Y\SUB\ACKs4Y&Y]+5g\191416\1107829!\EOT\983816\f\1043066z\26151M\SI\1031208!;`8-\170696\1104902&nh&\46018\997768{`>o\992346dt-\NAK\994271\&6\1111475C\183919\ENQl\992654\185547WKb6\ACKfz\188460\DC2\US*t\t\"Uz{\1061948\ESC\4918\EOTSp\ESC+5~\EOTz\1057942\ACKu/!\r\24940\4677c\1060220\5898I\a\DELU3M1)l\998068P\ESC\17318:<{\ETXuVsp3X\1052216\SO;:\\`i8SL's\983995d\98727UnI"
    }

testObject_CompletePasswordReset_user_11 :: CompletePasswordReset
testObject_CompletePasswordReset_user_11 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey
          (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "DDWmxyZvXA==")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "AATe5J7gYuNUJw2BWIgyUJ7ld=oZlXDJ90izhMBkm1YvY5P0veHAIZznC7vIb6Z9kqILaWeeMdo10U9FLaWB78Fm1976InX8VLss4e0MP0zB=6LE7monmdCppp_0U88bw2_=6ouW-DzfjrRws4xG-hqwE8kYTf5poAMn26xQbqSRcfwNdD5xfTer1-OsVP-toLd2DwulQ0cHZh9RaosS4Qug6c-K9Fi64WteCMZqeapifL9KIHggmJGBgwN1SSb2iOOfNHRLkHzoXHte0ULW3cXwHTEnl=5a0n4XO59UeTMrjNeVaEYXakxsuagxAce3bZoLCnMeCMgBjJRwCeBIBQjY4WjQ5Q-igN8u4wAAReQ0NtVZpXhGnKYA5g-HQos4nWGfYhNYw5xH1hbS=zAGCksBP9Fd4lqx_oL_fCtK5rjBmAnEtdZD6wETNupejUgC1gLSLKJxa6cTSGvk21-3f-WgMbYDv0HPCzp-w8ZKyhXsD_ndJhm60sBvOc43HCFM8Ruz2WldUSX3GhoyqHBfKABXDWsRZHrs3ssUrYhsconyesE1E6aDxTXC1N6bG-_MEYvPhKzZbmOxiGWTsMYETe7lb=l2OCD1l=EJUyCX8T2_DLH2LlJ87BNkp156_UAmRyaPAaRSS9icTdM4bsHmFeAqoe=stSY5UDB=C3JbXKXbW-1bZMk-HRl3WtYxDO74CSarkSEesdrJwwgbbTvgXfBj7Mth8zBdSCndU9U5GQYBm5i2nPKY=fel2V=YgAF5_8XLi95A"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "B!\1044316}up\NAK\25454p\1069230\1055497 \39886\1023178]\1113089\1061672\DC2M\131499.\FS\167721\1068712LEKN\1037828@0\SI_j!\v\986396<\ENQ\bpJ2\ETB\993035v}\DC2a\190249\GS\DLE\49896\&8W\1103712\1044961\DC2\1106639E%\1112338\STX>\DC1]f\SUBs\133341\168122\1013776\SOH\1061693t.>\14333\&4>\DC49\b]f@k\1034383\EM:\1009024D\1109992\62676\STX\SI\1009774\SUB\a\999523\STX\1038777ua\r;I\129579\1063770\159019)>sa1-\991900R}\1094986^\4306\1095705\f\DC4t\SUBJ\1077394W^\32631a wajy\59881\1077144%Scn\162336\&9\992548\137492b\ESCG}y\1109761xg\NUL$\1088333w)\n9"
    }

testObject_CompletePasswordReset_user_12 :: CompletePasswordReset
testObject_CompletePasswordReset_user_12 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetEmailIdentity
          (unsafeEmailAddress "some" "example"),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "GkNBvt5WkpZiqOtxpVKuBy8dXcbWuV8x4ejoV3EHdIAU=fZo3d_PjWQ36EzyO9eGbt5F8oQ=7vBzrTr9dpeETyJQWi9Vu38Efi7Dz-zsBvBp9p=AszTX69gzjPQ-xgcPvCw2Kvv6EStPojy"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "\DC18\33713\DLE\DC1\43126~KSy0\1098569\US\152372lR\1105208+!\CAN\CAN[kv\SYN<\CAN\1050129kW\1001115\163706+\21051i[\30317e2[u#\CAN "
    }

testObject_CompletePasswordReset_user_13 :: CompletePasswordReset
testObject_CompletePasswordReset_user_13 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey
          (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "Pvdg7A==")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "hidTvA__HaXG0Xa3Ko28nN8AQpYNQf0bdzzlgcsBJ76LL6AqUqJRE6D4G_OF0GF-Xxpbgz1OdRLxVXST7QbLsMVyMkIDamWXJa2TkKzcdZZb4hhVDZnIaWYjSfIHLMZa-ywj6C=a-nd2=E62_Lde3qlQ8544-iTn9TzY-CAqGOyrlwysJJNeib_F8Q1u-VW_blqtP3Oo0=18FlDa9y5U3ARIUQ4IAb5OG4XpvaspBuW_-PSrC79vNmABk0vuwM0DLFQAJNuwmRCOF2mvoQF6rMrYsSdJUYB=Pt5Mv5pGRbUwsdUYT-L3HXk77Ebd=c=nfxW34JvGdCZaqbE1_KnqZ1SA=WFGJ6SZfkmPEGwWkwDT=AKsDeekP_L-7zvQHesSE1xdHYs1LWXGYDYCRWDFzLpy3PTtNoIdKj0HvGubtuJK-3DfFUhG4IeA8qmTPHK8TbLD1KlH-eIfsgrPkVX=ik5Jww4AdPDBl-Ad=si9bWREe1Mzn42pP__pu8h1XL8ue5-z3JLbkDLiuFowzzZZj60a-gtf2hlluf9AVqcM_-6herg58y9GRr8xU3Y5Yuno1Fc_eGGshq5RNh5mbE4VUU7BfRweH6su=q=mxLyy62CZ68FUArY=CL_5SscZ5=e6zCf1=Al96BiNilAu0trDDqg0VCagXf_r4-bwSc159X=WNAI6NpQASkXyxg73ouQwsoX-enuFdb7oJWlnfDABZR4FE8sEO=VCc3A4iYUL1LsqaZ"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "NQi\1045721\&6\DC4(\DEL\14931-I\190282\DLE\DC1sA'\24027\t\USE6\\\DC2W\988244\DC2On\1101116m\92280g\999706k\SO\f'5\DC4%5\NUL}_K8\1022950\&1\14979B\171603\&5\1110874=/)RYG\\\ESC\1007353xd\983066\NUL1R\1036419C!B|1t\96635r\1025569\26267\EOTt\187049Xz\FSI\176746f\1055892\&7N2^ d:\24875\ESClo\189539OPgZ\a\DC2\1041937.x\SYN/\1007898y(\182083\EMBo3\SUB\v\1090249\t\1009316\1033672K\1073036{\157343Q(\74073M\99865\1061851;`rO\f\160403=q%Y\1059816\99237\SUB\141333 UM\f\167405\SOHD7,\vGnW\FS3>\v+\EOTzuQ{JtF\FS|\59814\51903+\1104795{\74101\&3\996441XfBgA\ESC\RS[\DC3#\1106531\1050615i\1088818\DC3*\n<3!\159426\SO(\US\DC1BVHm\175909\1111420\bg\1015312\DC4H!X\141513q\FS\USW\ar_#\183360\994422\SUB\US\\\1046753\95486\1039031\&7\DC4\15528\1100926H\1054195\DC4\180039\n7oEg\US\STX\ETX\"\63699|J\2046\1102234n\1031809&d\992846\165220CK\DEL\CAN?\96973\149920\DLE\DLE\ACK\NULH\EMw\f?\EM\tv^\ETXX\ESC\RS\95863\GS\SO\USB7lFo\STX\1003936\17648 \SYN\1100822L\42069+J\38082Ov\1019920\&6#9@\RS\b'dAKT'tzk\1026514\n\DC23\1091053YI\31165\"\1076177xY\DC3\1058287\41939\SUB}9nyZv\1110098\a\DC26r\1058863\100994:\DC2'\1042394\156114 \148989\&9&\NAKo-Y,[\997856_\1063117\ETBl\1088585\GS\20748\f \EMv}\168751.\1006090\ETBhgE\DC4c\NUL]H\f1\bW\1054531\SOH\1043667QnV\1015191\RS5\RS\137333\&7I\1087977a\NULSF,W\35909\&7~rO\1026587]\1086656?c\96846WlpOk\nok\\\1109331DC7\ETX\f@\NAKJ8\1045622\ETBe(A\1048081\154989.\b\1085843oJ\1092588c\ETX0p\1089200\127373&Hp_w\ACK\t\NAK&T9`\1035767s\33105\DLE_\158024v\DEL\ACKh\ACKF\DC3]\a\NAKH;lMLOqz\999079\1110425t@\996364aK]*\ETB1\EOT\53313\&7b\SUB\172040\40553E\22644pHt\1097939\132992\62032A\40631\986330\SI\EM\DC2X\983548P%o?\1102418\STX\RSx)r\1011553t\169304A\"?8\20267\SUB\186353^\NAK\170078\ETX10kV\1027480\78329\&55\120818\29728\GSC&S\1091150g1\ENQku2ytc\STX\SYN&;\rU\1070548'\996879+OX9_\EOT5fp<YkQG\1027523\138537\1069852o\1057693\ESC\1013509\74066\65394\t\1071456\180316\78200z\"\NAK\137223!%\GS`\1093027[\SO\ETBr 25\1050000+\1090371\1015804\1043162&*5\1010020\133958\127837k~@8k\11341G;]\988509\DLE2e.\143292+7\1061112\1088859K00\DC4^WZ\DC1u\185557\SOH#>\NAK\26556\183502!9s's\DC4\asP\t|\v\DLET\986832V\1102330,\987178\&6\140342P\RS\SYN\US\ESC\1042147\ESC\SOH+8<;/+\ACKBq\140372\1050808Y\1009276\48761n\4903\&8\SIF\a\US(}'\"/\28995Y?\163563}n\45558=4\SOC-\1043773\SOH\\/eI'\1050268u.5\170099Y\ETBY('\20830\DLEz7b\990753\63064\&0\172778\1093593sNkD\1114055\SO\1026850|-\nA\"Em&Sj^\178515}\ENQ\128693\SOEq\1087449\149931ES65.E\a\1053988\EOT{\SYN(\2032\EM\ENQ3\"qZ}2\ETBi'(\990398Dur c-\\c\12880M9mY\ETXL*eAuQ\26703\&5en\985099u\183413\GS7\99705w\1100252+\f\SOYd\\\137213\94577}<g`\67682\149456M\45308\1074632\"%\SI\SOI;y\1104400\1072192\GS\NAK\41866\1051277M\995288O\1064203Ks\ETX\FS3\12297UW~`\CAN\DLE6\1022139[\185065\7045\57580]W+aUq9q7\995965\EM{\CAN\EM2\DC4u\1093171\18885\159423\992755^\NUL\172158\tq\92394vLMm\1098879\b&: u\bLv\DELA\CANp4.\1059654\SO\STX\119311O$y\153136ggU-\16697(\n3\EOTM6*h\SOE=+\SOH}I`\18157d\1046189+n%\163409\FS\986614ud>\EM$W\49026\1009188>7"
    }

testObject_CompletePasswordReset_user_14 :: CompletePasswordReset
testObject_CompletePasswordReset_user_14 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetEmailIdentity
          (unsafeEmailAddress "some" "example"),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "FSTMhXuS1rYF_f_3aJfy8sn7CaY7BMCg6onJCAqtnt54fEvCkS40ml06ufrX9wvy192yCErw5Xei33_FoSQmC0RAjRN9eLFSBq15MclWbPrIsrwluYCiLmIB72IaR7ig8xGPv3-H8v=J_5xfvvpYRYSFZMZvTwTHKqaRL_uF8r=JULb6AQnLUG6__-nBrCq=91TRJ26VknMDuFrk-0Tfu72OJ73LrGfJqmWCR7gcFeyACyR17n3FI4GQquQ5Bb5qbfl5KZc7W_E3H=5sScZCa9r2Hj9ot5noSq-9nq2NlptoDc4mYTaWklhfbNCT8Wn2=3T8GfAx9nYW__2ZyAPlW9NPmbRSj5FYqqJAprLVa4GrT=PELXTFIba3inReJYtM4thgQ2LAgZYew4L0YGpIMOgr=uFKs3I3u4Bgd_77uNR-wayH3ENL0A97aV7p9DLLC6A2FeVugc2jMn1wViS06PkxJoM5ZtGZkibUTuycstG3VmGtC8ZMR3q2lAVNsfsiugBUZLg=MtzPz2Pqe=QaxCNq5N04ekL"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "A\1081942\&9YM4fO\1090110s\1094466\ENQ\DC2\b1\179725\NUL#\1087859\183740\1019632f\1041374\6025,z$h#\f\ETX\2283\US5\996429\35782.\171631 \1068265\tiW\SUB\187596V\SUB5\ESC\SUB\SOH\1080402\DC3\GSf-J\138208}0\DC2(&R*}\157122\&6\EM\DC2$\DC2e\ETB\172943@ \986769M5N\29421\1073651O\13445\SI:]\18243\ETX\1031884\46115*\1016709qG\64812<&\1112925\32634\ACK\1066179\ESC\DEL{\r\1007409l\992191\\\a\EM\t.\166498\1048072\ACK{4g\ENQ\ETB\ENQ\120078\CAN)a{=J#XSJ\t\25101/\fu\1013342v2YZY-k$ #4\155005^\\\1106614\DC3>b/5-Z}\142114\r>\1032648\ETX\b\FSRS9\vE\DEL&\CAN\SIz#m5z\ACK\DC2pe\162939\&0\170848\54557Q\144858f,sH\NAK\DC3@\990148\998664&l\1028462\NAK\ENQ.f$s\f\a\NUL\36236F\"\GS\DC1\ENQH\r\143281j\986076?\r\118847\95137lf\1111505\1083975s'\1022814\1092252\187157t-\ash\US\1048347w\1010070\ESCj\136733L\1070015~H\ETB-\128398\EMvml\1013389Eug\SO\vDTR\ENQTO\SYN\1013663|\ENQ\1101450\SOH\\\EOT0Q*\165251Ly4\1005992It\24685w"
    }

testObject_CompletePasswordReset_user_15 :: CompletePasswordReset
testObject_CompletePasswordReset_user_15 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetEmailIdentity
          (unsafeEmailAddress "some" "example"),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "hmgODqiry_V_t87ih4Ezo7GS8C38DYKENIE2t5nRiJMdagPBW-lTEhID3_8_ApDfxAfSNxAF03y2L8MCLqWWsX_wxkaLYtAI39FLtZZAwxHkSRRazNp7LAc_3QzGXR4O_iFiCqo0f3ZbmODskuoeNVUGBBPJhQ=uw1yVKyMVHojWD16khERjcHww2=hSmqUdh3W-46WPWaZe7IRN0_gk_UaBGwdMb4aDcTHJ6jIaTfQ58djcLVGrKpuO1xO=eQ2BjLJiK6Ik30JgICpvS5ZuumMjgkNFKtHwCu0C-E-oUDUmi3sWKkFQPCxpIy0Ol0SAyN2llCWAADjTR6SW-zRT4qDQNbtDe8nKWpJxZYjFj=IvyBHaK1q6NjPsrXQBEUfajtkh7OwbQwqOOBk5nt8RPP7xwUewzHEtkQUJUjbgGh80nuOdC7sMa2zOSEOy33oC1bjncA23BsaJoisQbFfju_UWiCSyDD-oUXsWkKR1cMGmwyVpf1IpZRnQq_8dwpgMKL4j4ehPxPrVBefQPmzdoK4nncLDB_zDKBBn4M5nbqDsLmO9OqSKeDH6tg=uKTaftrDK2w6Mhfo_fSZOsJAEouS02TJwr6vE_VlJbiOCPysMdVmCdn6Ai2n-p_WlwFoBIHLPkVnx7yYyskHuUMhYQfaq8=CHCwa8CDyOGu=cZVxOd6mTHRD=mXc2_cgkYJ94pdZOL0"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "\SO\16795@?\1034319\DC3\tMt\GS\FSD!Y\159204\1084245\&9\DC3\78687\1098287\1086842 _P\1008697/3\177267bZ7\38004.u\SOj\13467[\74634\DC4\147112%7Y\993096\994803)\64704h\159809\fYt\166228\1036104#%y\ETXO\NUL\50715Qo%E]\183018\&6\52933)\CAN\SYN,\1059969\1017948\1084728\1030365\1107776>\30767\ACKdSfz\FSX\1011690TU\8707jM\161765S\127326\DC4qE\FSJ\SYN\NUL\61212\1055205\SUBFg9\984629x\1038351\1111966\SI\SOH9\DC4\1099666\1049910\21409\1039222aSIN1s\t\128815\FS y\b\STX+\\G\180523\10901\&9\55148 M#|=Bhd\SI\RSt{a\STXAz.\179095\14700\"\1112100\r\65533I[\aT\ETB`\12556~3\1087071\SIn\994851}c;\1050618\&9\SI\985187/n\1048500\&5-\ACKZ=\1030734\&35\146351]B\b\FS7\ENQ\DLEGl'y\NAKb|4\989606=\DLEg*y\1056993\983575\&3Ej\RS\1112431\1088239:\167240\1104231p\ENQV\SYN\92656\SUB\aDBA\CAN\1056225\166252\182312i_x\138250#\vh5\1085011d\1105586\190225`\190602\DC1h\EMK\nW\a\41610\60481\10408Y\SO#3`\US\CANl\1060958\SUB[pr\f\USX)\CAN\SUB\1102505\CAN3NrV3\r}\r\164454\n\DC2h\\\1049410\1034744=\49232%*^|vp\21661ix\DC1\ESCi\168078\&78\a\78281 g\1022135`\t?\1019502'W=\1084341\RSj0\SYNk`Y\ACK\SOH\SI4!\187300\177532\&90\1017023\188333\18019FD!\1075920&\158315)F<j\DELd\NAK6\51903\SI\131898wh&\EM\ACK\bX;\b\135515Rv\1060082i,Y\40929%Hvr\DLE\990401\fQ\164437q\180013_\6680\1063810\169150\48788X\ETX\r\1076018\97009\&2G\EMe1\SYN\EOTR\STX7,\1067501c\SOH?!H.>\ETB&\144361\a\58759\51196\aK].^9~\DC2J6\1013077\&7\154880}\n]W\177439\&0e\ENQL\157053a\"\17874\41525;\EM8\41213]J,m>\b\v^\1020018j60G\997360\&1fMGY;[<2q\1017963\1103307\1016618O\ACK\DC4. +Q\146747L\"\NULQQ\1081550pfK\156025.i\1110373\ACK\SYN\144039g4Z\ESC\1093918yh\ETB \1105457\44791\DC2,\v}\SI\NAKoSV\184902lUZ?Fi^H\99751\RS\1107891oR\\\SI'b\DC3a\STX\SOT\US\184371Xm\EOT\1021048{Y\SIQDO1OZ@\1096649\1098472)\1059966V\\F\DLE\1005888\1064602{0D\bj\DC2l-\ETX\1046901\1028919\1090408\31154O\EMa\1102121J\7867^\CAN?:\1041195\996983\1032028gz\146360\DC4\DEL\30263\STX\127304\1053834]N`\1113890\138696\SYNt2nqJ!p\vI\92251H\"5\DC1y7l\1059822EZ|kA\t\15546l\FS7t\n\SOH?t\1112409\EM.\191318zJ\989580\1044505WA{\1054069\1091947m\r\v4:\15073{\1077651\ENQ\44428\31089;i\38608\60141\148116\1084236\EM~\1030663\95306\n,y7D\EOT\9133\DLE\1050807\61817Zo\1113458MUD\147204\1073606\DC1|'Q\45980\n2\1000549\139927Nk\1023849@\SODia-0\tD\DC1\1035971\1057269\a*L\1105895\FS\EM\1030689\n\ENQ9\1041588\RSF\GSK\134105\NUL\rA_\1107772{\\\"\83191\989807T@\94793\ESCV\64140\1013588l\DC2Ea\127285\1086507\ETBz7\1006547P`\17986\SI\83490S`u\1044811\n\1086255t\\wf\12624m\EM \1008362\1069874R\1065254\">[N\n\"\487IQ\RS2s\ETXJ6\FSU%\37676HnDm\STX@\EOTT \27323imm\SOk\NULJ?*\STX\DC2+hhIj$2\83196\169764\1106792\r.\1106472\144305\1099117+o\SUBg\1090938o\\\a,\60138\STX\\\162017\SUB\1018764$[\1070254Q\CAN\1034607#\DLE\SI\999162\1090043\149519\\\35872B\SYNs_\42449\&9\vZ\100569avv:\1069361\EM\66820rv\986074\1041564&\98108,'/)\1069739'6\1112666\&5)\CAN\44956\&1XG\1077492>&\996088\&0\t,N\SOH4H\1091985\SUBR\tNT^;\SIs;\27408  \986736#\165922|uKo\1038168\136622\SYN.\1057757af\1035602vu%\1036358adVJ\1035793r\159529\14971#\CAN\1033919z\1061723\GS_d\NAK\ETXQl+&fe&2\1009634hbj\1099212[b\SOHkXF\1027185QZ\a*(\fOqN3f\ESC"
    }

testObject_CompletePasswordReset_user_16 :: CompletePasswordReset
testObject_CompletePasswordReset_user_16 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "4rP0eWj_NPxojYzkU_pdkW9LqEqBCpwqu_CzQ2HRWUDWNC-B2gKZnICvQwM-6ynjWXae_jlxWt0A9cQPRCddj-GurZImbB7fuiSpIQH_zsc817_M47P0NZDzEvL7jXOf8RdpUH9n_X9UK0uICUvuwY9voyAwvlyxKnfFmWM9g4VhzyK7-Z5c6M4eMksFBdADNZcjJavPD1hFxfMiTK5wzDrTIXIGetf-jHFYdnbru60wCJh3iCiYLRtOTCBltY8MM5LsoZi4jfAh5qDEZ94NZR2b64MXjVa5QY3FEer0hj5s29zMrUd=pN8YNnfpagV1cN=v5Xy2BCGDPt-vurTbZFWan=KC44G28LYu7fPzAvEgaECUX2OrxZb5B8A4gB59weiDl=HHO88JU5Fp3cfDonTTP97AI_JIvBY0KjnMhvs2JupxCpyPbsgrHe6a=0WSXppIRPCqSAYWCWY_Jldr=g3b-hytjaCcL9iMVXo-L8Xj9ET2k9xUNf_aj1aYVpT4LHBlX3rY91JqLI7QRYccS489z0ydALwWRMO7spGOcUBwcfuxcGhtvwyc4IYlEm=hDWQ7=8lswjoFvGLJXE-P33ChzPTUO2gMZwgKydl6d8t-sQ7g7zxd9XFFDk0G4AJtevGCxL1=79X4ob5yW7bV7D=TBfIUc=L_-B1n3b7STRbN1s88--LK3jBg9P36L32EObOa7T9ExK3DZPlRVyN5J=OWu8mzcTaEXRF5x6Z0oHMyEjBSTTC2a3_GF4YBcuaaKNKIU9WkdYnmkDLYpnJDk"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "U\1013927\38932zjM\ACK\989369\SYN\988971\44411\26179\1099606[\EM\6469\DC3L\1007944C-Q\987858[%D3\SIjP\159117\ACK\1104449\STX.P/\EOTj\99688U\"#X\63255\144060a\FSp\1056477\CANZ\\F}Vy\3567\7303\1017941)#\EOTNJ+\SI\36214\EOT\ENQK\171246fge\65191\SIu^s\rsa\SOHG\EOT\1022250>4\1053106\ETBD\SI\1037295\1082556#\DEL!\1065845\170137qw\27053Je\DC1n1Vu\1042145\&4~$y\146120:Y\SOH\a_K\a\ESC\DC4G3\1069872}a\1051446\ENQ+\150221\t#v\CANP\DC1\ETB\1027017m\8155Ni\995066\ETX\CANC0\1025017\ENQ\1078805\SOo^\156275@'\1001329\1051836\DC3/\984914\&4\1112505\1033864\161819r\1033625u\1086130\1004026nUW\147017\14745f\984915\147167\&6L\1056254J\EOTp:\DLE1\62628\24434\1036737\&8\987062\1025412\ESCf\1058077SB\NUL7\5692|\b@`\164062CF\tOP\n\ESC!5\SUBQZ3\ETB&}\1074005\FSl\1060790\28300\1051838\1057041\92189\170949h\EM.!<\1105788\121176\8485#,^\FS\97089\r\US\1021202E<\SO\DC1\STXe\1047025\157530\ESC\1108461\127768\DEL\1033097\&7Z\RSg\988877^}o\1092235\SOH\b\ETXxi(#\SI\1048177]\983467\DC4\1012517`\173149i(\DEL4@ C\SOH?sT[\59535[o\30407C3\DC4*Vd\1009623\1080174\t\153905\987527)\11605T\1104034\&0\SOH\1032423\1098555\DC3{\29330\45613\buk\997238\59248\1074873\NAK\36212/Q\1034971\\s\n\ENQ\SUB\ETX(\1038812q\n\1065055\\M*,-\69410k\1061499\EOT$\n\1043848\DC3\"\rPf,e\1074584\NAKM^]\DLENtE)\EOT0\1039707\&9}>\SUB\1031821>hr#\fnYe \DC33\US4\142595\"Df\19173O\1069162/+F/Ps\RS-\SOHGxR\163425\DC1\USJS\SOHP\1037710\24155\1070416a0^>\99334q2\14262\1022764\148161.\DC1C\173470?L\157517dPU\1071711\138047UV\ETX\1095932S\1083194\153099\156263g@'/\180511Mp\NULU\1042080\&6\153840\129602Q\DC4q\1010576\&8\ACK1O)~\GS\10101^\1035814\"'\f\FS\SUB\ACKM4\1061452,QR\41977\50386\1101120\\,\"\68775\1010234\GS3\1041457q\188067X\15821\EM$RS[W\SO\53767%Z\1102182\985051P\NUL\27978\aR_T\985127D\rf\987775\SUB<\bB\1009783\1111321\137944\43362\1104166\CAN\1109603[\ENQG\1093769\SYN\DC4CH\34381\t\v0d\1087322/V\1084112\17394r\DLE\147854\r$\150273L\187452\DLET\1012169\991037\47347\&5bw-}/\168300kHB\25586n\147531#z\1046737\1010676T\1096702\SYN_\"\190994Q\182486@4r\1093724y\169135Z#\15462\EM\1089051\158494\1030245k\n\172022(\ESCL&N\32338\1058995H\1102900\EOT0'\153127\EOT@-\157405\SOHt&=e&\GS1\1043880L\r\1039332\10350\DLE\150593%\t\1067637\&4\986137#Po%R\160955"
    }

testObject_CompletePasswordReset_user_17 :: CompletePasswordReset
testObject_CompletePasswordReset_user_17 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetEmailIdentity (unsafeEmailAddress "some" "example"),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "3gq6=cswHZ9ri64_HJPb0GHqnIvQsgakJ=HkufysG_pLk8piT7CmIFMoO0lif83sPks6mv-UWRbQCOyTECbFMlPIR57uJSHFmxolrFw"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "\NAK\ETX\NAK\RSVB-^K[QI!h\154998\71117\1084382Z\35906,\r\73897>\FSMI<Y\SO\168581f\1113328*PG\ETX\ETBR\94733d-\RSf\167649\DLE\GSQHlG\131571\b\133878\EMZ\SUB@\1069651I\DELA\DC3|\1017555\120368)\1059547:dtD@QG\153043|kV\GS\171548\1076342WO\DLE`z\rS\163051/A)\SYN\100730j\b\1082560q\SUB]Nx\1032927\GS\1113623\DC41\1036652!M?\EOT\1008968P\183050\1104009\"P/5\DC1\fK}$O\USq@\NULhi>\121373\DEL\1041332\13065\&44\SIiw\NAK7(\vu\191279oy\DC2\NAK,GLefv}q2Zn@jp\SUBRucnj\50153\&1\180385BC`\EM\1024436\DC1dA\aw\36359\\)yrkSp\1052631\19089\1083696d\ACKb\1104238V'\b{\ESC\v5\186684\SOHLp\1061917\STXb\98718\&69\ao\v\1038378\19929\&7\14137\DC1\186147\ETB9qT\NUL\NAK\USN{=\63014w\987855P\1069375>\1082182\1058366U\DC1L\78567\SIqV\ENQ<\EOT\GS?\1071802X\1081478rtX$F!R\DLE\STX\142620\9784H\65618*\DC2\24357Z!Sq%3\34833\"r\1107644W\DELCc\8262P]\168433i\1047019e\SYN\1048476L\49468\1075404\&27V\30820\94221L\1066619P\44283m\992353\167937S\1062257s!\GS\1012749Sa\164423(\1109764\US\1057958\999861<nR\1096622\r\162061-\152556\159469Pb\164878U\166572f\nm\1031706xo T\1033262\74047mY>\ACK1\v\ACK\985311\1098842\1029412\r5N53\SYNk\60720\187189=<1\SYNH\94843M\5786\35376E\148966QSy*fwg]Q]Y\"s\1023439\NAK\1112462!\145645\NUL\ETBn\1103776\US\STX!g\46993[\32717\3350\1109732_\1095512\ENQ\FS\v$\nth\177004\FS\"y\SOH\1070850\ESC:6ZfF.Fs\CAN\178064jnSm\tO9\ETBP@;\rO\RSM\DLE\1029277\NAK\1011137\120416\49923\1038900&5\97187\1060750\181666\ESC\1052105\DC4k0\160161\DC4OcMAZy\1065334\&5\FS}t8rUzI\34999\SIIL\n|q|\\W\DC4c\1102652=\141891S(\121160K\1021413O\41185\"7\13500Axp\aF\CAN$H6E\DC2{\178358\DC3\145547\NAK\25538\1029182'\2102\DC2j\140870W?&\DEL\DC4nF^s\988659\1069309\141794\SUB\SOsV\12922\NAK7\187903{\RS\41369\ESC\41141W\74457\SO\EOTg\NULn\SOH}\RS\ETXT\1101298\a\f\157153ne\111107NP"
    }

testObject_CompletePasswordReset_user_18 :: CompletePasswordReset
testObject_CompletePasswordReset_user_18 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey
          (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "gKIq1jpiCDg=")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode = fromRight undefined (validate "gBlCfS7vL5ZlXMN2EWV5eSisvsqKezrNgWoI05VsTNJTtsB")
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "\US6f\1111789/l\\L\ETX\57836\ESC\SO\96423\995453HW?-v\nqn\163962n\ACK@\ESCS5z\ETXi\EM\156843P\1043153\SOHEn\1108441}\166673\95024z\1011478Fx\SYNt`\vt\1036893p\SYN@\1047041'\ESC\4911cIWs}0Bgj\1090394\DC1\162989\181264\SI\1097419\ETX\1110190mK]\160292&Md\1061797\CAN\1008874\r\173984y\CAN\1035416\39608|^\144442\157286\1084325}V\ACK\989501[w\SOH[C"
    }

testObject_CompletePasswordReset_user_19 :: CompletePasswordReset
testObject_CompletePasswordReset_user_19 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "jNA=")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "jiXROuSghHnJYlGM3vQmMTKoAHxSW_pRvQy1uP=aZ3FYAv1Cx_gTn7H2bQpQ1eIVFykU3yTMU5oUoCb-4ZjTXblJXM7vK1YEcZt9j8gp5Oh_lUUvAA2g8Z5zDkvuror_Rkmu2hqb1haFUrbF_fsIpsUWe5yaJyb2v3sIFRRSYtDCxWyUUPi--4mXwUd7kyB8NeOsn8nzdjp5YbgDAil=Iz2zxnry_lhjTYSdGs_MZGg=sekX0llThwNr5P_eD8xOl2vkMSxFyZd0DL_3YkosLLupwjCrngbmvGurqHbJ0-=11LsFDwqdvVesj15wSXW-XaUTkYfBbOBfmHRRp4GSYZ-yU5aL8_pNXU2FG7OMEZFmrok=y6V3L=AzVQpC8RFCpDat3E6uartY31DXFb8NjSxlYmVn4KUj4el1l2cvkv5hdLhsqxX--80fn7KP=_Y7ToGMX4E7i7Jzp5gBI9h2nNFyG76l6dfMvhXGcUyYjm1R74VUsRD5lCeG0mFgvGct2gTmDhd_3Lkb4vFsZTOKP0qE4w4wEuqKJFJrgpQcqCmX_E2WOQLT=bagdpS7tob-Rf1CT3cKNEiVqNno-hiNcHtdYzAe=QnBAQ1OG6O-eJzkdVKmeVeXy6tycFCg_fusv023l4TB53=sBHCXly_pAdWqOGXJkI2RY9ZcI0S6z8=A3APUVDGZzd3jn7-SPsv9q4XrvS-78VnChlw8KRl-l5HjxykqxAu=BW9XtCPVysQCbvD-mNnBUu4FBs"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "Nf%\26857`C$1\ETB\1016123\1013601#:\1093135\1993\DC4_D> q\EM<7\1107146{\1061674\97572~Z\STX\a #\aW\53248'&\SYN\CANs\DC4=\RSa\1085316\989974gA\1079584\ETX!C\a\990713(4PN?@\1081857^#\n\STXbrdyr\1106974\993303\v4\1097791!#\1052766\SOHdK\NUL\72402^Si\SOH\146114'nw]M\ETB\1012025i#*W<lQ{)\SO\1021626\tb\166398M\1014955\1003098\ESCdi\43605;\1055936a\ESC#\DC2\USy)\DC1\rD\183265~GmjH\158521\1055712\138778v\1043633\1107125S\ESC\1031422\178196g+_\"Lz\1057665\SO\v\1100202tZDq;?k\n\1012551<Gp5\ETB\ETB>\f\DC4\1097711\1091349f\1034270\EOT\364\SO\ENQ.\"\SYN+\DEL\1087866\144785\ACKCT\DELe\146451\r9\184888ifa\1113864\17863k^\DEL-^|I4f\169485\985795E\DC3\987554t0h\STX0\1015621\a\ETBp\63117(}\NAKtB\1060605+\SO\135099\1044767\&9\1000776\&5S\57916\37617qQV\147525?\54219Zrt\DC21\EM\b\GS\ACK?\DC3\156312\1092629\69667Q\31050y\179088y\139970S]I}\NAKi\991482\NULh\SYNR\28464[(\SOHHxq\1036471\&7\SUB\1094145\1022472V/.(]\991630R\fa\USJ\998815\ACK7~\DEL7TT\SYNRo\DC2\30121\RSrm\1027017\151151ry\GS\GS\51765e\DC4/`N\NAK\DEL$k\nx\DC3U\ESC\166495Q=\1019375\r\ENQ\1031360\1109306\188394Z\53745gp\42504\t\bI:'k\983534|Nu\1075753\&6s$#%p\17281cB\1003026M9S\173530\4950\100374\tv)\69736M\66455Xwug\96822\&7e\"Nj\DC4\1023797\f)\995080\STX\DC1>Xi%\1019327\168371b!\SYNXq\RS6\SYN)\DC1\SOH\SUBq\132317,8\RS[\74835R[l\1103025\CANE\46590-Ccw /\r\133918=\996631W\DEL\1106772\1096813\98986\t\1038602m\1087137\1111197g\172456.\CAN?$%Y5$\b\DEL\10731W\1032611\30462H|\173272\US\SIUq\1071566\ACK\1076198\987790\1045529\SI\STXSs%8Yo\FS\27056Uw%\143377R/'\NAK\1017747\31651~A@\NAK]M\t\1043105k\rvR%N< %hWk<a\60368\185283Z\SO@`;5\vc\1059011\1053370\rj\CAN\f|\GSo.]}1sLm\133306\SOHd\SUBW\DC2UNuKW\1036178N{QlZ\142412B\NULR8\DC3:Ctw\DC1s8\1004581#s\18312\1053146[\1113018\ESC0HC\142129\1063212K\DC2pCJ\493B@mw6\f\125097\DC3a'\SO)\1004663\&8`\1051123=BL)EWbi\1082959\167654\1016491\160609Q5\98665\33931\r\1060621\1017687\1050147\1106117[\ESCNBd9\"\166650;\120354\30477\&0f\998277LD\NAKx\\M.58R\SUBjL%{\1068540\123195Q8y\1012120\1040621>7\SO\t\1022082\1007073\&2HDZ5\SYND\v\"3\1083081!L\1028467u\1044115,\STX%\1105946$iN(\f\187862\147976\988324dAA\b\EM<\t%F\41597S\989085\\\1098403\110714ax]\35131\998139\54320\988848@*\1036145?\1040527S$\DC4\146906\96397\b:a<0\SYN\SOH\rd/z-\DC2\128549\fG\152978O?H\140755\\\4798i^\127894\12415\1041013\36494;\52340\DC4\rI\163748\1098064\1074828\a(DwA\ryq@\1029109\1077261\DC1\DC3\DLE\67107sn4r#\120414E\1113581;\SUB\NULDq856\1014066\171865\ETX\1065435%TP\1086344*;S\SI{>\a#\177106\EOTZc\SYNF7\1009077a\20098Y?\STXp\135345!p\26071.\DLE\ENQ\t\\\1036223a\1040873+\163305\n65\ESCNb8\1020528sc\rQ\DC1\NAK\1107289B%=vP\DC19\190607\&8$WC!\190590\b\1075330i.V[\47896W\186257I\1091816\92618D{\1104672OBm~xx.aVlU\CAN\"Ou\USoA\r|OD\SUB^pB\166696Q,H\ENQ\63043\DC4\1003911\42643J\DC3\NUL'bVi\SUBb&WS\1106725\rqm(\187473\&4;$N\1001886\STXA\62480U\DC3t\CAN\SIJ\\~_\DC1+hq;\180132C\DC1\US&\1075833s\ETB\GS\145589\ENQ5v\169060\&1\59025\&2s+8\DC4\996138\167208\at\50528j\STX\143681\189398U\1009286"
    }

testObject_CompletePasswordReset_user_20 :: CompletePasswordReset
testObject_CompletePasswordReset_user_20 =
  CompletePasswordReset
    { cpwrIdent =
        PasswordResetIdentityKey (PasswordResetKey {fromPasswordResetKey = fromRight undefined (validate "wGwL")}),
      cpwrCode =
        PasswordResetCode
          { fromPasswordResetCode =
              fromRight
                undefined
                ( validate
                    "S4EnT8ajkuHyeuozGd_HX3VmHqhmNMJn3LuxAiPku8F9hwk8fWvQlmoZkhreAOGYE1o5dWORSFivpNp1RRCOP2-SvkAxCX5TFlx8Pv=ZD1O5tusMN2jraJpT060KRHe9tpQeEzOItpavn_M=L8JfXdu_KPimxKGMvqedw4QSqpRAtbWSPyn0YIWwnBzGM1=UNzlueBptrYkNbxLN4jmTBvw4dys8pUEUW71uKeHM0HcGHVfkKn4LDGtJnA=4UX6duOsGee_GVLePjlAQP8gzeV68siIbbVJp4BmUIwh0FyZ2tcaN5=nYxs9rg78V8ukl1lH7srQFh1TtvuHTnR6e9bUkF2IP2MbJKPCBL4DnLNfZE7yoW5X"
                )
          },
      cpwrPassword =
        plainTextPassword8Unsafe
          "%:F]T\ENQ\vt`c/$x\f\r\110962<\167682\DC2%3gF1!&DM[Alw\8366M :\tg\SI +3\\\35195r\186137A'\1042721\aU8\1016516\78238c\NUL\DC1[&w\158228\1097827\&6\ETBU\SO\1035337_\\\58202\DC4?\1112500\1082086q3\1107715Z\1025233\&20\ENQ\158606fkc\STX,FT\37265F0m_8o~!\SOH\1095300`A\169824!\180135\21290Kh4;f;.\1061263*q\1103264\1071770\1079751\176712\993711_Ay]\1038017I\RS\183781b4\160789Vt\63109\a;!\f\1078906\1021229Z^,r\SUBC\DC4\73459\&1\161616\1017807 aS=ak\SYN1\SUB\1019433\1071476\bH\25304]\176465]%\1098929\23086;\1000067\&01\EOT\DEL\1070921X\148186\71226\RS\1045965\1095376\1107687.+e\DC1\1092989MeAY\1031197YvK\DLEi\r\1067660\RS@f[>>\\m;T_\DC3u\157897\1007461\30666T\1060491\FS+\1019481\1111804\1017079\&9ta6k\1014740\&7&\1019716RD81g(L\52575\1105513^k\US\1087923C%\71252\13479T-(\EOT\r:V\182162\99976\&1\173091\167405\rJ\1058506T7NM6D\1093179=+\DLEU\144210\SYNv\NAKf\SYNz\64962\917993[64=b\EM\DC2\110592\1037416>C\EOT\135323S*\39272p\61918/\DC2\CANm#\DEL\1019756\n_~*P\46796D~\185934\1061284\RS\EM\SYNh\1058927<\57662^\1076167V:yL\DC1\983936\54304\95217Me\DC1\173637\28857}\23233SEyGN~eC\41589\1008496\DC1{)\48719^NC\1041531\1010402\152999\ESC}\1050714\1044315<yp\28398o\990904\135267[Ks\SYNFTF\156689\CAN\SI\30470\a\SI\DC3`i\"$h*!Tu\SI\"\1043498\1032505\SUBv\DC4}\1015062\SYN\1031536`\SI/jH)\20456\167926n\40120*\1004805\ETB\1102940\70378Q\USp\SUB\25456'{=\b\"\CANDqBTN\NAKR ?\1046246\NUL\SI;'\993569\141389j@\144845\1105353\178426\US.Gi\1744G!s\183744\GScHK\a`Tkr\1024687BZN\1033868\177060+\DC4\15423\24791\98330ah2\14238{\187474Q?`\1045065\&5\SYN\1073710\ACKD\EM7\1061749#p\994345h^m\1001222\"\152717\189487lV\1075847P\US-)\1044056\145421jr#,X\1077320\6145\f\983662\CAN>50h*\DC1\SI{\SI\SUB]\1071376>|\ESC_$[\181480\EOTbN3j\RS\171025\ETB\145187gP\1065781\173942\1009175\99521\1027637\1025619\59790.z\185081\DC2\t\157620fB\\\CAN\DC3|\DC4G\39064\&6\ACK\b\DLEh\DEL\124937\NAK\74416Gw,+5\v+k\99425O\1004788F&T\100343\ETX>GTX#\120481\1109419?BS\29595lQ\169334\35479\1049632=\RS\SO\996776\55138^\r0n\NAK\146675\128353\1038461\STXEq\177483\993226]h\DC3B\SUB!\990150\29868N\987137\1026589\&2.\179310\6355\&2u\DC1$\1057066H\1077906ua)\1043473v\RSy\EOT7KC\ENQ\1024303\&8kuY\991388\1075309:F\169126E2\ACKZY\186798\1061629J\149940s\US\1101984\&7G6\ESC\vf\DC1\100667\1062487\99642\ETB\bE\1074206\47318\52993wl'\92404\168932\993434gl\989551!U);v\ETXI\aK\USa\NUL\987059F\DLE\1096453Z\DC2%\994420u\1107348{Ah\ACK\1113953\b\SO\DLEz\DC3\ENQ]\SUB\164854Ft`\SUB&\1037715bw\330VSO\ACKz\EOT@\\`\147544\4111r%\"%;\1056968\&7\rs?\1098051g\1098269P\SOHt\NUL$\186214\1071744\1022584\1013974c\ENQo,[\1055940\SOH\172608rc.\147909\SYNg\NUL<\NUL1hO\ETXhhN\1007263P\DC1\SI\SYNY\98489mBIk\SUB\n\1080085\135468\DC4U\fu(La\DELkUt\3227\128054t\b'\1111124{=8\DC4\9850jzt\ESC\134303\STXQ\DC1 7\RS\175534;\1108063\NAKe\1014967\36268\SOH2.F\33192:\1035967\1453\152125\DC3\ENQ8EC|\173778xy\RS/~P\140732[\1087310\RS\1018753+0j7z\184026\USfr\f\CANSm\SO\1067099\1014662U\DLE\NAK\164724^+*\\k\v\US'<2\1022414\ESC\14889\1065\158191\188417)\GS~\46128Nr\fu\145379\168360<R\995080FUs\999288#\1047540Q>\GSl\ACK\DELcUjm X\EOTM\\3j\na-Y"
    }
