-- This file is part of the Wire Server implementation.
--
-- Copyright (C) 2022 Wire Swiss GmbH <opensource@wire.com>
--
-- This program is free software: you can redistribute it and/or modify it under
-- the terms of the GNU Affero General Public License as published by the Free
-- Software Foundation, either version 3 of the License, or (at your option) any
-- later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
-- FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
-- details.
--
-- You should have received a copy of the GNU Affero General Public License along
-- with this program. If not, see <https://www.gnu.org/licenses/>.

module Test.Wire.API.Golden.Generated.CompletePasswordReset_provider where

import Data.Code (Key (Key, asciiKey), Value (Value, asciiValue))
import Data.Misc (plainTextPassword6Unsafe)
import Data.Range (unsafeRange)
import Data.Text.Ascii (AsciiChars (validate))
import Imports (fromRight, undefined)
import Wire.API.Provider (CompletePasswordReset (..))

testObject_CompletePasswordReset_provider_1 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_1 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "Cd9b4n7KaooqOhOciMIf"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "W0CLFxLOL"))},
      password =
        plainTextPassword6Unsafe
          "\1012683\1112273\39028\&5\168679\169133rs\93986\&4wo~\1002561l=\1032023\13042\SI1nt\35892\1050889N\46503>?\"\aT\69782\USgg\\f\SYN\165120#tS\NAK8\DC1C\36700q\r!2d\DC4\189369m\SUB\a\\V'W\\\110825,\r\143398?\ACKx\agVQy9\SI3'h]\78709n0ue\b\1032695?@\ETB1zJ6\NULI\a;DL\ENQ\37006c\92669\US\ETBz\1097017?0\NUL\184657\"A&&\36577E\157691\US7fG\1081322Vpx\DELI'\1102879\DLE\1008567g,\NULH\DC2@+\1085033\1064315\DC4\1091186\STXJ\1103240dPQ\STX|\EOT9^9_\1033902\SO]\a\1022683Of'd\SYN\"^\EOTw\1073515_\1113440\DLE}\95632\DC1s5\161851N1\1078798RkTZ&\150149X\1065364~''v{4MDK\153974\US\SOH|oB\143604'q,HU\1025306\SUB\NUL\1060487+%~v\DEL\97853V|5\127943|\999498\1059223HTFhF\FSdelLB\CAN\SUBbiC\1027783\n\110976u}g!\38540M\141506\1037727Pt$2(W%\149078\&0i-H\SUB@ii\1037533\NAK2\2636hg\50874\28429#{\23697\SO\NUL\146715\f\f\1039241A\GS:\EOT]\99785qf\SOH'\DELx\139534\SYN\f\DLE\nT\149322sK5O\EOT\SYN^&3\SOf!\150976\GS\SYN\f\1112187wy\1052535\1091937\1045148\SYN\ACKijjq\58477&\RS\"\DC2\1063939e\129001\ETX-\\\DC2E\ETX\40256\39310Z\DC3\22084iD7Xv\137008m\SUB>~\CANW\139109\33037YYZE\1022090J|\5247\CAN.\137437p\1011705\ETXS:Y<.YBcP\31609\1107733v4U\f\987772\1070124W!9Z\1035690;\1106506\DLE\132101\SOH(kH\SUB\"\vdX\136713\10837x\154948\&6/b$A\"jH\133538\48869\&9\DC3,\144088\1091851{\DC2\12495&>\1040461"
    }

testObject_CompletePasswordReset_provider_2 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_2 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "8XosCtq4Dzhyo=UoMRg_"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "EoNo4PH=cFSyQ-yuHhP"))},
      password =
        plainTextPassword6Unsafe
          "\DC3~d{ \988098\1008471\&7\DLE\NULd\1065586\SOH?NT.\186651\1106270JJ\64065^rd\146603N[\43292\SOHt#dn\142707}u\SO\1022368<\1094323\18349\51616\GS\CANn\n05\983885\&4Z\vIJXz1ia\20698&\SYN'<\162555\v\19677B\ENQ\SI\1049058\DLE1dt\1038032)$\135798\&1b\97041Fvi\36729J\a_T(-`S\NAK\fU\20849dBbTgi\167678\rfp\171973ED=\STX\1086228\SUBXa<*#\1037916<\1106037\191075^%Xx\ESCOM\DEL\994881\1059244X _3\DC4K\GS\a(&6\59167\&8[\1045759\1111435M\681>f]o\ENQ`m\DEL\1112157\1102641\11945\f\161652)Q1\1018093q\1005011\&9\1102348UD]$\41477\f6j\190919\&3jAG\1007534!ys\NAKs?\17249Z\160153cfpz\fGC_\SIf%xb\99796\&1\ESCj\94762\&4K\rQ7\150803:\55009%:\r\"-Zq\DELU|\DLENa>\131324K\131830G\ACK3#\"V\NAK-w\ACK\1081085(\23629\1091792\\H\21182\ENQ\1049732\1036941~M;FHW$X\988437Wy|x5N\CANTrX\US,\n!\51726U==I}\ACK\1067103\1041045\1085401\EOT\983701{ }1\144729yu8_\DC2p\1053610l)S\128946fZ7\ETB>hnRX\458M{U~Hw;\69816\1035492v=J\8990:\1000731\1096086\70367o\ESCs=\NAK\1017016\SOH\NULb\1111472\152433H%f\1040890\EOT"
    }

testObject_CompletePasswordReset_provider_3 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_3 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "=aYXtgLJZX77qMIx0Oah"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "RMQ-RtgFDI-b"))},
      password =
        plainTextPassword6Unsafe
          "\1073786\1022541\1030619|@\DLE\1050256\58722\5028\SOH\25945J\EMkH\986937=\11472\"SP\\\FSw\95016lR[.29\137466\&1W_\64827\96388M\RSU\a!\GS\43687NKv\993525\1097611X\50069;?\157751\&47.\CAN\1103688\137799\186574}\v8\STX3fj\DC1\SI\181630=-3ZmNn10\DC1\997119\1059249\161874CT\NUL:N\"\SYN\\@|q\128174\FSv_u\95666\1080533J-*\1034203;\1068818hC (_u\161608g\43952\33809\NAK\US',m}\a\30792\DC2Dt\171459\152195Him\395|\125271q\161223r\110828\&27A\NAK\EOT\FSgP\1090390\US\993009\62450\1042020O9\EOTEB]\DLE<\156612\127142\133358\1015398rJu\t\1027420\1050082F\bfxm/f\a\rC\152680t~D\ESCO]_i\US\39307\SOH\35670>\SYN\1086602\NAK\STXDz\DC3\1048748ZC\DC1x0bLFjXI\148199\EMZ\GSR2!\ENQ\DC3\\Mffm\986388\1043076\94041F\1096421u\7179*\DLEM.q\33878\a\1106357GdxHmu\DELSTrb`cn\NAK+(@KZ\ENQ]\1034430QEf?fw\ETX\177531.W\STX~k\ENQ\993340\1112261\US\tB\SO-\STX4b\185882o,\CAN}P\SOKD\v\1100259O*\b\1061589\RS\1106367\ACK\NAK=\1048333eh\DLE\EMY\12994\986285\185764\GS\DC1#)v>a\1050729L\DEL\16992&gh1\SO\24688\&18\DC1\1091353(\167196\1031220lc\ACK#\1096547Poe\178761~\ETX[%e\133630{\1020978\&31\99380\45215\SOHI1z\1093633s#y\1048198\FS\8988g\USPE5P\SO/\n\1089996 *Z\DC3\2954\33162p}sh;[Sr\STX<Y\1063314R\ENQ\142587;m\SO!j\1064742Y\26339\&0\1009114f!\DC1\1058452:Fm\DC2<h\1062377\1075391L;q\133290\v\37381!\b\ETB#B\DC2\1035646L\98706\a~K\SOH\1083020J?ix\1103716\26074\1041219\1023735\&9pq\66414\1032397u\NUL)\1105066\&7a\1001390\SI~/b\asu&\v.\1000260)\1048216Vo\172707\"8U%\1086813{=\53356\NULj\b)u\1062111e4\59669a\f\1017418o\983279jRi\DC2\SUBM\ACKg\156836\154465O\60473\DC2Q\1040352C\ENQ\1048864y~\NUL\CAN\157850R\b2\ETX\1108470aZ\td\45558P6\175121\&8#\137251\1044187\DC1_%?\ESC\172320\&1\157792\t$\ENQt%\NUL\986756u/g*\1105690'\ESClY\34211Pv\SUB2\53037\1071677\1102754\&1B\1041189g[\DEL\1006478\SYN\99181\ACK\n}\1037516\13897\1097935\1096922\1541\n\173191\ACK\99814\SYN\23263wx\159959\a\ENQ?;\13131\ETXU{^\ESC\13969\38608\1019277\DC3p\RSH\USX\SYN8\65281\1105157e\SYN\1091876m"
    }

testObject_CompletePasswordReset_provider_4 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_4 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "jZVyHFe3sRymJkt0NFpJ"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "WrNRVUakdiR3TfEn"))},
      password =
        plainTextPassword6Unsafe
          "\1523\ETB53G\1063255\DC3\1041445<&eK\1103902\US\RS\ETXe\r\31072|F\NAK\DC2U`f\na\EOT\SOHvz\NAK=\994055,\NUL\64119U_SJ\ENQ\f\NAK\DC4\b\EMN\67105\SYN6Qd\174256(O\1112460O\ESC\1082698\amQwe\f\1028955\&9\1039328\GS\SID@2Q,L\ETX\DC1\175142\SOH-\1099006\DC1\n\DLE\SYN6*rW@x\EMPjd\ETBk'uAe\SOHt[-$\SUB\1028095:\RS\DC4\DC42\nsz8\STX\1098365\1067932l\1091276{%|pN\2061o\DC2b\1035531\1045061Bjhn57\CAN\bKY\1052158\DC3\177379Gs)\ESC\1051605\1100444/4\a!\1031098z,\999798%9\ETB\1041178!m\1076001?4\1012500\FS\RS\148537)s\987738>\1015744\ESC\tO\152390\STX/_Q^a\157142\1101351\985165y"
    }

testObject_CompletePasswordReset_provider_5 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_5 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "OU56F44t-0ybJj7eKUaS"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "rr3lleg-Tu4eJ"))},
      password =
        plainTextPassword6Unsafe
          "k\1044075Pnu'6Z\NAK\1017783\149108\ENQ\129297l\18438[\1054432TMgddIb\186517mt.TCQW\1025717O\1111819M\ETX\27672\ETX\ETB\1083603\1091383F\RS^\182596C\SOH<\rs\f#\STX?A\n\170555\68821\t 88|;\SUB\1015442&\n\1042330'\1003626\151074 <\63465\v\EOT\1043258w\1012648\DC3l\62396\FS2)\SYN\1003311o4G\161486\&1;0IVKt6t$Y\":\13086\156982\1055032\"\GS\6275$y\ESC\15469)#\1011445H\SUB \SYNLk|\DLE$\GSh;\19798G(?ft*V%|\9608\bC\b,\131877\SYN\7628eI?:T1\ENQ2\1042416B+\STX\\\GS>4\1042921\1015196\DEL\1050654\ENQ\RSdH\NAK\SI\vK\NUL\1020294\a\b:9\163015\&3\53363%^[X\r:\1044970c\n\1035333kk'RA\78616\1054694\24158\1051573c\RS!\167908\28730\ENQ\SI\1068557\r/\SUB\1106472\&1ott&\SOK"
    }

testObject_CompletePasswordReset_provider_6 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_6 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "54Yh4fa_ClTVqjEEubnW"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "RcplMOQiGa-JY"))},
      password =
        plainTextPassword6Unsafe
          "\1068965Mz\1112587\\b\988910\33388\1081682\FSSi8:\"\r3\GSc\989625I=8L>uA'\SI&I\94104!W\995368\&7z;r\ENQnj_+3u/8\31470{\32573\170260\EM$vy\rB)\125105l\58284\1022117'iN8\SO}vd\1025869\132023uw\996610\&17\ETBF#\154217:s\1019264\EOT\CAN\12331\127284p$\53580\&2\14658\DLE\13233\SUB\59635Hl\25906\SOHw\1054216\&4[\171724\DC1\RS\SO!lS\EM\1073106\66443\\(\47504\61628N\1029483M\NUL\"\SOHd\1088943 \58859U?\31664d\138217(o\RS'\47111\v\1097785{A\ETBb=\1039402\1096760?o\n\164402*\12095P\SO84,Qf\1065714D\EMZ\SOHux\1096460<\v)\1109779\185595\25160\69876\&8t\136448Ya\GS\ENQ\9575\NUL`\US7\1022950p\1032880\&42\32304h\68036\EOT+W\a\1022685aH+XE\1016645p\SUB\8531\n\DLE\136210\1080841\1069380\119885\t\31849k\1020979\159730\RS\99244\1100479\14782G\nh\168920\SUB\DC4{\1107942\&5,\US\DC2L\DC1(\137496<|\bZ\172359\SIK\EM7\t2V|K\ETX,\SYN)F\50452\20991\100678\1098846\1109927\tJ\SYN)\133930"
    }

testObject_CompletePasswordReset_provider_7 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_7 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "muTkNflRkN4ZV2Tsx=ZS"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "X-ySKT"))},
      password =
        plainTextPassword6Unsafe
          ")jtk/z\184222F!N~\ETX\990448\1055900{8\73979\153166!D\1043025%\135850\168364u7WynrV\ETB\148520p\1077327Lt\842e^}?\1093891l`.`Y\vZ\STX\1112581P}[~\30935=}L\1095875\a\v!\1028719\ETBH)>5\ETX{\NAKD\ETXUEh^ ~\EOTCC\ETX\SO\16392p\38296z3jt\NAK\984409\bB7 P\CANSu_\183789o\17912\DC2\178168I\v`,\1022887N8\\\DC1^\10311m\CAN\1030400\FSZ_\"$\ETBB/\NUL!\SI[\DC3\vy\f\ENQ\ESC\137923OC\SIt\12293:\EOTl\\\b\EOTrG@\US\45550J\95310\166637-\10023\&8tTT#MD\FS\DC4lJQ9s\64189\25142\DC1jlVF\96794P{\5228\25037\NAKKEC\1098620[kg2*C\991918\NUL[\35874&\74062\188051?\182094\&8\145055\rSYlf\95342q\30892\94613\NULM.\b\t\1102963\1018631;\DC3_\1029835|@\SYNd\1082087)\n$an\SI\RSp\n=\1013045D+\97624\f\1106118\988197\1113\GSb\181818\SI\1091492YQx]\1063062c\18044\993702\148181\1072483\1042478J:\ESC\RS\1052622\186566<>\EOT\DC1\FS,\1076029i4@\ENQu^\178972\1082722Dd\63135\1006290\EOT\66041>Tx\1091471#u\\`\STX\1093786,Kt-\1035926D\1024804\154425,I.\190722:\15722&3n\v!\40042Pm\41694$\n\SOH\183103\75035\1093394\3121>ihpLGl@L\DEL\ENQ\ETB\182031\SOH \21434\SI)D` wC\STX\v\ENQ`\54406}$\39750\DLE[\"\1087944'q\1043619tP\EOT%\ENQeG\r\1058468\1110447C\DC3g\1038268#\FSYrht\164459@\1085349tMo\ACKWM\SUB\v\40317o&}~45\160190\&4K\1104579\CANl!x\167229k\ESC\\h\ENQ/4,\177887Yp\995759d\98258N\1108317vw\ESCK\1098528\FS\ETBRSf0\DLE\148633\93011*Wukxd3>\ACK'gN\1044418\DC38;2FN\747 '\1005699Yt<\1105770\21737\1045228\DC3]\13220\ETX@\f\1101655\42506f9i.\1005751\&5\n\131677\&2%$\1047618N\169552Y~47\986154\SO\1007292\1001379\31676\&3\1056996le\1059155\&4\DLE1Q\FS\986744#5?\73770\1092436\1011458\171368\167096\&4l<\1069261H7]=\DC1a\62925od\1064417A\GS:l\SI4q^b\1057856D\173253\1059916$b _oH'\DC1Kv\\n<-\t\US\1083436\163231\ESC\1098850F\1329\STX-\ENQ,\CANG$\NUL\38340.\1107219;\125009\169728\167O\ENQH\1018301%\ACK\1025545\1011306j\RS\994143\1094533mEB\120644\1031761A\20411\180256YN\STXFRm\US\ETXQ\1072397V`+\95270m\SYN1\1013314\b\1024313\&1}O\1108229\1002097\49175\f\1007287j$t\47188\&4!8%#v\f=\t<\49120\61960\ACKM\1056844\SUB3\"\r\989243\SUBX%~n+:\NULM\134421X\DEL-v\72197\f\ETB\996041\EOT\DLE07\1009115\CANU},,}\141362\bHy\fLa\\\n\64444\983949;jo0\157407\1061450\1041761\EOTMlW\DLE7\45112\1113654\984581B\1087787Z \1067937/\1027501R5F]X\ENQF|`\162826E\128973\r\v\984688c\1100696\1074387T\1041206\SO*(\RS\ACKbNs\1056623ST\139333\170914K\1032627?\SOH\1095798\1006647\13962\"S[TY};\SOH*r55\aT\1006364\SYN\SOH\1111555\1082650\RSZ\a\1020940s\162901t\1055866}\1055756deI\153662\46739\rR\\]'\1084483\1056412\\y\135616\FS)@o\30437Ci\1081016\1042881|[Q}}\1025142\SOH^\1085438|S\EOTWa\nE\DEL,\1014498S\DLEq\DC3s\"h\36770)\1084960\RSB:"
    }

testObject_CompletePasswordReset_provider_8 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_8 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "4h1kCFffI4sHePSIIfS1"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "jgfbzV60"))},
      password =
        plainTextPassword6Unsafe
          "[_.VrDh\1015708\1032560\&3\DC2=M\163597rhfOlZuP\65504\DC1\SUB\f\rx\FSJ5\f\DEL\181294si\166877{P\CAN\GSG3%'O\a\f\RSa\1092468x:\1053642\61514\1073484+\39638fMP\1054011B\\nu\\\SI:a4\15010qI\n{\1029779\NAK\1041484(r\44941EI\13466G\141832>\FS\1022348\EM@*y\US4{,\ETB\151574p\ACK\1107549R\1055583H,\DC1\v\US\1009911C!\SYN\1027699i}2\1006393\1013086pu\t?4\ETB\35803\44095\NUL\t7&\f\94064\993295\1068521\1077762\t&\ab\160257'\NULM:\29880oI\DC3\ENQtG\DC3`/0\RS\166279v\b_c}m\UST7;he\155120#\99948\1018238\1062963S4K\EMR;\ETB\US\ENQ\1021792\STX\1003450:\24440\DEL\EOT|p<e\69991\10140\52305}\99428\38218*\RS9oZ'\1099723\a?\166949+$\171381[\168054H\1066561\STX\SUB2hN\"e\EOT\140767\1048611g\40479G(G?\159396\1020949;\1107378cF\110620\1056506\28257\EM\ETBk^\SI6\96751ph{1!3\168419\&6bCPZ\1106130E\62582\DLE\tCJ\98062\ngl\42421d\157568m\155755\&3\1005498V\"\1081037\SUB\1060409\&0\45591\1073579%BR\ETB\1031186~]\ETX-^\GS\1088103\DC2_Q6mk\78071N\40310\766Ib\38909\b!hdBz\DLE\EM\ESC\RS\r\FSG2{{\DLEVw\b\1070378a9\1098941\50611\DC2\1098281f\ENQ\69435\ETBw\SYN^\CANrm\52930\59445\DC1\183627r1**\"\DC1\1033821\t\12999H\1109725}.\154488%\DC4\54554\1086324}\73026(\ETBCS\1054338w9!a\RSKSL5\DC3\26831]\EOTEU\EM\1051053s\78320B\DC4SOW\DLE\155904\&46\991484O<\82981\&6\1103639\v\33215%S\149010\DLE\991078\&4\65069=38!\a\1085782w \DC1\149711f<\SO\EOT,m<\EM\1065573\&2BX\\\RSK\22613#\ETX^\1015295\7744}S\1061702`\DC3\139963M\ACK\1080001F&K\164165\&6\153768L\RS\CANe\ETBxp\NAKI\1043610.@>^ZN\30349&WtKz(S&M\SO`\SO\181996#\1011887C:^\ETB\147530f\EM\a3jp/\1058108|p\SYN/9?Wn\13780\RSH\ENQ*\168131\1075215\119182gh\2225\1089941T1\133460\77864\1037953=\986510\1004229&1Z[\1043805\1002639\&4U\DC4\998270K%\DEL2\USp&q\1055724o3QhHE:}\ENQhil\1096277fc\f\SYN1U\ACKTK\DC2\173882!4Ch>f\DEL\SYNV\49106QcXO3\t\SYN1\185658\147541ii5;?\ACK\1023746\994599W\63325\DC2\45506yDu\132949\140075\1007168\"\EOTVsg\1088989`\1042945:\38432'\STXE\992832\SYNJ\ETX\64654\DEL\RS\rV)6K\1001241u\n\1061707\ESCWq4k'xZ\CAN\1004671Pp`\78706\DC3s\vb'\1026286\DLE\51253\49630.v\1078713W2u*\1026823\f\rc;=l2.\135778\1067475\66363'AT\1038064\20692mc\ESC\DC3?Y\EM\1043502erF?lU\177756\SYN2\137736ZW\SYNe}\110678i\r8\1045526%\DLE\1060820Wu\ESCwr\SYNZ\984526\DC1\DC4*F\1025876j\4244\NAK\69844\SI&\24155t?\SYN:\996677\EOT\1096939\\d\ESC\rV\1048902\DLEY\SOH\DELHDi#'#\SO3\DLE\1033528\1066728hP'\SI>,#;B-\DEL\ETX\FS\b\1080220\\O\173118\155899\33548\161628r\DC3v\1036063\NAKwY>@P{&\126581muC\30489\DLE\RSW\DC3bzp#\SINO\ng.f\SOH8\1044888\USM3\STX9M#\31452A,S\144295\DLEiK\ACKi5\DC2\1106504\163392\&9\DEL3~\SUB;z\37537H\SOn\74309\1097966\22046h\SOHH\SO\1014941rSW!\1076838\1019303\ENQ,Texo\1103981\\U\60688\1107601ef~\NAKA\CAN\1095090\b1\FSiW\EM:i\1063110\100555\1028434\f@\45876^20\EMn!\1110881\ETB'\t`\"^"
    }

testObject_CompletePasswordReset_provider_9 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_9 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "8QW8mjnVnIisvrtQDzWV"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "qXaaBJ"))},
      password =
        plainTextPassword6Unsafe
          "7\CAN\995057\1082858>#\149981T\1113543e\thUr\189434/\186737o%\DC4\SI\8198o6n8\20176c\1043600[C\1057789\&72@t;6t\169068\11814\120655\DC1\EOT\1079958\v\aS\SOH\EMey\ACK:\aii>\1079059u-<\1112894\1083324\SYN[#b$<\r\1056477\1033082\1105819\ETB!e</\ETX\ESC\158384\GS\SOF\988758Z8#\RS\45584$7^\DC2P#!\555\140892gY\nUw\1042750\1054944\1043236<wT%4J\96044XWK\1027244F\133675a`\CANLJDRa|Bl\1024726\DELS[]\19068\54318F\SOHp\GSLZ2~>Wg\991833d\DLE\CAN\ETX @\SI\185824\&7\b(\40642\&3\NUL\1110157!X\FSe,t\ETX\1095428\&3\128629\1025661p\1000552\184281\184297l\25688V]\1068327v\152194MF\v*\1050101\1065061 \ESCT\SUB-\21105\&0>`{|bal\1060553\ESC\US\GS|\ACK\1028192\DEL\DELV\143705dq'\DEL6mCCjv&\1015677\DC4n9\1022140My[ K0p\\`6r\182750\1080218\DC1$|#\137636H\DC2?0{%`\STX\1005371k!\RSIg\SOH\SYNJ\FS{9b\1059876/4@\1060707ldKAH\ETX'8\180338\178999\1013270O\1075685Dko\23121\&04%/N9B\1003052aW*\1070751\1043722w8\SYN\RSr=\EMnX\1071326]\NUL\GS\1082718\139251\1079728\DC2EfW\t\SYN&G\196\&2\1008326b\1023329\1102771\1047159\&5[f\NAK\100090J/7\26364\t4\SOHS\CAN;7\185137R<;`L\1112382\1022626\&1?yCIiS\153111\GS\FS\EM\ESC\156314LH\140232\\:K\1002577MP}q\139293J\ETX\151699\1052232\1108510\NUL+X\1029314\181545D}-!EF<wm:\f\7753m[w\RS3m\1041473\f\SI\141166bs\29841C\ENQ\RS\NULr\t\189684\EM\1049892%\60586#\179226\1063969\1101785\51178\EMv\1027818\r\1023300\t\NULj2|\22154,\152682~6G\SOy\64751K(\NAK\66192\1071406{Er\SUB\DC3I\EM\ETX\1052448\ENQ\160359,qo\45615>\SOHE|\131183\&6\39841\1062330\21504\SOH<*x\179748\1015132k\DC1\DC3\98575\ETB\EOT|\SI~gr\DEL\2694YyEY)Z\155604&\DC4\997375\1004619\36183\151489\143359\29364\DC3P0R(|\1044843(%Y4\1044821?3\ENQ\v<X~\167494a\128927ZcA\1100393\133917D\DC4\ESC\1006656:t\ENQR\1039789hmq#x\1014426TC7l\96606^\1017734\136048\ESC\NUL\tfysN\141140+`f\33317\US(\157198\STX\1063516Rv[n^b.K\43200\1046214R\DC2}Tf\ESC\51490#\1028129\995802\167950\&1y\176382+\14251\&7\nt-\SUB\DC2g\SOH4\NUL3\v\ESCe1\1011441\993290\SO\149023:\19272\1091388Z ,\CAN{e\173546\EOTH\163432$V\STX\1088204*\DC4\32071T\1067699\SI \EM\EM\EOT\CAN\SOHY\1098823A\140995NJ\138272\98089\180181h\1070176\1101222bKFr'\DLEn@\EM\141727EK\150774>\ACKU\988376\30638Y\f0L\b\986153\STX\997297,\ru['"
    }

testObject_CompletePasswordReset_provider_10 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_10 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "Myzdj2g7NTl0ppCPXiN1"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "BBwqW3"))},
      password = plainTextPassword6Unsafe "\ESC.\63992\SYN\128619\1086386\&0EI\50894\1058818A\ny\65231\1092012~\CAN;p"
    }

testObject_CompletePasswordReset_provider_11 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_11 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "nQSYG43lVn8kYS-MPtOO"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "5BuwQHalK"))},
      password =
        plainTextPassword6Unsafe
          "\1045282k\1026750)t\NUL\15552jgI\ETBP\SO\188738\147525\1066604G\39626jb\f`\nTq+Ut\92361\27743UBpVU \992919f9\21139\1020059\&1hYp9Ja\147132EBN\DC3t@\1079146i;P\1042445\180008\&8a\1091375T\158952V\138448\SI\18953fx\11087d:\SO\\\1054972?b\NAKKtz\GS\1104407\39067\n\1074206\&3\SOH\1025715\r87\ENQ9@\5471Y\ESC\62699\11493O\1045551\ETB\10550\1037708$Fph\US9\ETBe\fC\20273%>\USP@\STXo\34112h*\1042645\1104430\987562E\43000\11020\32229Ft{A=\38646#k\SYN\185887Fi\99911>&oy\98658\f_\1099272YIL\65827\&2\184583\1063350v3\RS\DC4\27853T\141265S\1048343\NAKK\150089\&1Y\1059308\NULk\DLEy\1067797\162645\92680B\78890 of ,"
    }

testObject_CompletePasswordReset_provider_12 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_12 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "NeuDtdyLCvq11nGkkEal"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "sj64oWB"))},
      password =
        plainTextPassword6Unsafe
          "an-<\r\ESCa$\n\SOH\150355@, daTW\1040876\1086641\1008932O\a\173984\1089573\187195E=\1033471\v\142301l\ESC\ACK\ACK\67616g>h(;\983436\ENQN:d\6803R\SYN$<;\18099:@\fWrO\119365\\\SOHE\NULX\SIL&Qc\143803\ACKKx+K\FSF\US\1018415\n:\\\SOH~a\147255\78868\RS\DC3Jr0c\DEL5\1004711\&88\1048447(m\1018537e-cWRQ`N\1091454\127355+i\DEL?\DC1\172339(\1079229\1021542\1023479\1095290\NULzNO\"h\61187q\NAK%\148825|\8495A\171333>\1023153\NUL*\144781\b\1096599\&6\ty\1084884\1106372\&9\991761\&4\54666\993909\&69\188610\78768/\EM\3120\SYNX\160680\1093419e\1101140!(|\EOT\180765C\15108\GSj\73803\t@\rsQ{ZZ-\22170\ETB_\EM&6#\bsD\v\n\td\74406D\37637\&9\72882\1015558\&3\NAKN\1028309Fnk\ENQ&\EOT3Q&\28043Ys\97711H\181981\999099\46018\&5VB\1044294I+\1104448\61690\&7f\12643\133501]\r '\163623\SYNmbE\1015369\n2?HK@\DLE\DC3\DC3\1023424Z(\DC4\SO\DLEk{r6%*\1034286\EOTM=/\STX\1035914&4\1098394\b)TI}\998716[+2\EMV\SOH0h\v\18412@\bQZG\GS_\DEL\999345Cim\DC4)m\1021546\RSP\23785zv\50314\1005770mi\DC1\100847\1042938\USC-zT\SIQY^;f\NUL\ESC\30038N\93068\SOH\DLET\1038908I\DC1\187625n\DC3\CAN\\\r&<\SOH5\71118\1027153X\1092148mF#h/{\DLE!(\16202\&3\ESC\32283\185971[h}I\1071533:\183293R/\1004445\140257#\"\1028937\v\177329\61790_m\138219(\SOH%^\1105873[\1035020RF\CAN\1054790\24076\DC1\FS\NUL\72138\STX\ACKd\ETX>#qn\SYNw=}\1001530\177147&^\NUL$BP%5\3450\179283\DLE\ETB\SYN{y\34999\1114051\NAK:\17208na\133899\1014430c\1106626NDB\160028\15282u\43902Xpr*#\172705\a\SUB\188880\1026535F$\ENQ2\ETBB?Q'asS\ESC\96583\DLE\DLE\1012383e?\f\STXT\1096814Q{\DC3R\CAN\1065288$\1074134j\SI\135241\&3\DEL\1035586\1073529\43493\&9ecd2\ETX\139431@Pvv\123147\157284q\r\1091419<H-;\b\GS\DEL\NUL\1052987ma\40272Yg\1057809\1053440@;W\US-\129526\&8P\ESC*@g\168183\n\78217\7045E/\DC4\986274jD&[@+AIx\24621-S\t\1049509\US"
    }

testObject_CompletePasswordReset_provider_13 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_13 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "4b48rrAxwAV84s13Xtv5"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "eBfsdKsok0a"))},
      password =
        plainTextPassword6Unsafe
          "f\1004775#MW\v\29303\1091159PO\SO\1020956T\DC1&/fR,#a\1109331\DEL\1005467\&6\t\RS\1068688\"\ETB\NUL\983938\6053\SI\EMs\r\2877\182118`M;W~%\19565&XKO\ETXH\1072156\181265\SOH>\1052105\23426\185829\1098874,[a%\1087411\"RLOU\31476V\1060394K\NAK\EOT\180111S\"Wes\ACKH415\78735-S\SUB\DC4h:d\1036393NZ\t\1043380m\167051i& \1107753`dP4/\DC4Q\ETX\54045B%\186624\&0;Nb1\DC1\EOT9\SO.\1014579\187014q\ESC\1078099f\ETX\64604H\1060225\vY\RS\1045658\DLEC\179470\a\NAKWw\ENQ\1035817[3^3B\154130\"_\nPK\1076894{\ACK\ETXO\DLEr\SIvc=+af\SO\ACK\1101910\167540\STX@\GSQ\1011496s\ETB^c\CANwJY$\1107843s\DC1Gs\1049240\DC4\NAK\171080k\US\ETB|\1065322\EM\1035477tJ(\1075051\1687xc\b\1056830q.\34099\&7\NAKF\1023165\DC3C\a\172318S][\DC1:\ACK\26422qL\1039209\&2\EM\44805\ETX?NG|x\1065136>/Iz\1061649ms\US\SI\1005398\131153\159667\&3\NAK\1048772\997425nd\tv4\DC2\1080172\1101786\v1Iw\1050069\v}?1m^\STX5#V\147028\1063172w\EOT#\1030144\145884\f\DC2\131840\6065\FS8O\NULS\ESC\1033971X8N\142482\1041006\59926.\ETX\163181\ACK\DC2\RS2\GSr\EMV\nK\NUL\DC1\1019014\30036_W\61065\9477\SOH\1094473/\392\20690\159848\181387\EM\vGDR\188046\SOH2G0{\FS\1084240JX)\188982SE\176663B\1089777\US\132402&\SYNg\"\DEL\1902UCP\1054969\1106547\1106033YU\EOTi+,?\147075\1044086\1028895\1110977\1016778\1106548\DC21\186874\1095378L\1030254\997653\998721\SYNc'\ESCp[\STX\EOTN\ETB8^\1021121Bk\b\t["
    }

testObject_CompletePasswordReset_provider_14 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_14 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "4hqO6D9=V3BKXLXcLie2"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "emsaYZVuPvQ1U"))},
      password =
        plainTextPassword6Unsafe
          "\1028432Q~\7949F+9Dc7\1026106jl\SIC?xdB\ENQx\33993\62067\ETX\DEL\GSj/^#NS}fiO\119558At>\GSh0U\62526`\r\aV\US\1112085_v\33980w\ENQ\184054\&8\11831\1032958}e\ETXi\NULRC-- \37583Xd\ENQ\an/,?\SUB\SUB\1066224\42328gQ`\70388\41959\1012806Q\US\ETB\184603&LR\149821>\1012033tR\DELg"
    }

testObject_CompletePasswordReset_provider_15 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_15 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "bbFHebGp6h_3F4QpSrud"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "K_xVBcX5bLpjvL"))},
      password =
        plainTextPassword6Unsafe
          "\ETB\26894'\fd3E\120233\1029573\1064918Y\r\1104541H~4aF\16111&&\US\1085044\1086081Kt\2880XoS\r\DLE\NUL\1044509\v\983386q\182823\1075779\148618N=\1062701\1004214R.\\\t6!E\164333\GS_$F\STX\DC4l#!\ETB\DC3+\1078110P_\1037691\GS\1003847~HhsY\1008817\CAN'\996168wy\ESCA\1096877\&2\170187\1060412\SOHO\SUB\ETBc\t\1090646ud\1037884\CAN~\173115\1096337\rB7\1049690o\DLE\1095190fL\996695\ETXi_=\a\SYNLE}<\1106966\aEl\49881\v.6H\CANw\995916ZH\176178\49327\19051o1?\61005\1065006!W\DLEY@!\1058199S`mq\15087\161424\167582\a8\127764~rL\41008\171779\DC3[\989714K\SO\ETB\152791$jRxH\SYNm\1076533\DLE\169669'\EM/xd\52526\95412\&8\ESC\38505\&6bYZ%\139602\20809\35764~\72852SG\1075777`0pL.\185639\&4f`7\DC3\1113337*\v\60813/T\180136C\1111167Z\SUBZ\44799\DC1@\\\1060472\&2\EOT\11121K\1039363||\DC1h]3&\DLEY]\GSk \EOT\SOHU\161853.\DLEk_\133547O\1041592h\1083420{\64532\aw\ETX3K\41041l\1069560\a=\EOT\152591\"\fyH\78163\SOH/L09\149680\DLEvw:\ETBO\1066598%#(Js\169845'9s_&9\32941m\149591$\1021728N\984156WE\DC4=y$=3\1083024\21817<\t\th\1087258\NAKx/\999799V)\STX\183098\1073874BI)\\gk2I]#l\NAKPn$\172450\ETX&\DC1;\GSY\EMO\180851\ETB\1095722d\ESC6\151131}$\175277\NAKajf<Y\1439\DC3\173480o\1113489\1074137p\52021S`v\"f\1024186\5161\12017\32446)\8536\988618[R\USw\ajR5\1096978\1063623\1056330\ETB\154034mU9-\1076969QYX?\EOT)\a\52453\DC1/}\SOsG5\STX\174021\1002136M&\1042057\\\1003878\1077033\b\1066513\1021338E\1016849d\1111672\154208\NUL\97430S"
    }

testObject_CompletePasswordReset_provider_16 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_16 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "kMY9Yz6qKv_aDoeJTELh"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "JRb0=pLR6I3a9YN8jH"))},
      password =
        plainTextPassword6Unsafe
          "\1029358~IpB\100771@\1028017x}\99441\1097880\43331`\v\SOHsg\995399\1027105 /oO(\SO\44667k\GS&H^\1035595 \CAN\1012475Ztl\bhR\SI\EOT\129332v\tF\EM\EOT\ENQ\48831 \DC2y\1090874T\ESC\77920^\SUB#\42123\1042165t\1625\1018909\tNR\DC1 .e\ACK\35002\FS\RSn#j\CAN*\SYN\FS1\32805\&0 #\173905E\SYN\CANHj'\DC4D\1098423kFo\40437#IK\vr\1090168$\fw\1091640\&3\1083417Eu\185272\23334\ETB\168631@e-@\GS2%/\1094533\a\SYN\11825m\t\EMN\57880j\179569X\166587\189365Y<\1026596Jm\aJ\1113327D$v\146947@jX*\1050039\ACK\150143\52596[XE\1008339\NUL\1047339\DLE\EMV\ENQ%*4\1087152\SI\US,Hr\ta\"\996187\CAN\US\52660\97204\154345{CS$\GS4\ENQ2\ENQ\40066\138546\f\1101420_o\1104818\ENQ\50027#\1006840\145908?&\EOTFOk\NAK\35077\110647\1055893\188120\a\nc\1100202\1094615\1030592\ACK7d\ETX\STXV\1096474?/6Yz\16366\&7\1026994\CAN\195032\184756`\f,$\158970,AN\"V&i\SUBH\ACK\1040850\1104568\&6\GSrPS\1017189\1074704\SO=\EOT6paA\US%\n\1020194\62455\1026102\NUL\ETBk\v&\1000767\RS:_\SYNT\FSRM!\DLE\RSRU1B\38740\&4*\br\33104s<2\DC1\ETB \1102563_H\99806\151207s\b\140222b\1088220\n+np~l\1052561'<\74550\150683ru`\GS\984875\162220\US\CAN\\\49060f\t(\DEL\NAK\SOH\15646\167739]T<\1033263w<\EOTd>\1093922v\184717 \ACKa!\v\166519\EOTS\1012345'\153953j\1098235\EOT\FSE\1061729\1052832#5Zg=\172012\4883\66029<W\ENQae\1024917\ESC\1073985y?'hi\29678\&9\DEL\FS\1109976\24649.Pg\62379\1073003P2\48187El\EMc6Q\39602\32228\1111843bpD\176915Cc\1104988\1034926\1091185\GS\165569\DEL\USB\STX\DC3:\1017972 \73671)r\1096518*+2\31487\CAN\DC4\DC2\1089916\27970&\72219\153103\&8\1016432D\1107986*\1003877\&7J[:Cyd\EOT\DC2-\26397\&5a\176695O"
    }

testObject_CompletePasswordReset_provider_17 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_17 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "ivJjwO0avz22igK80GCq"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "YuQRgAUJdiXJ"))},
      password =
        plainTextPassword6Unsafe
          "\13425\EMu\DC4mS\162784\b,\SUB\EM\ESCE\NUL\1019426\&29\SOttH~w!hU\63472|\r\998151\FSUeJw\149442\998666\&3\169812\f\GS={w\SUBP\DC3\1084492d\1113524\&0\1001092T\ACKn;uZC\NAK\ENQW7Ip5\tqU8+\GS2\29751z\38528\20474\DC4\1012585\ETB\DC44\172112[g\1049211s\22039\b\1090096x\DELF\1009027\1052417\\\ENQw`PE\3359\&0^N,gi\1063690I\STX\1085502\1388K;\7397z\31399Z;X;1y+h\1069040\1084020\1021100@\1018143>ZU\36791\22747&C\STXZh,\1088719\7021\1087041\ENQxC\987916\39597=l,\fu\998370\ETX\60675 }&\183212\989435\165094\1040277\135097\"\v\SId\US\EMJ\59927'6WK\13266\ACK>\70807\995567y\r\"\98652,\DC4\SUBd\NULne+\64011,&!9Y\15584T\127281<\1077668d\31074e.#w|?\1034255G\1027753S1\24647\\\1090505\bz3\DC4,\988313<\\\1073727\DC3\1032879\997224%-\64532\EOTC\ESC!2\156292\145116DT \f\SIXja\\R\1014521\SYN`\CANJ\n\45882\1023562\SO\13921ab\NAK d0\SYNX{:\51467\CAN?\187194\&1txQ]\1005159?\176303[)\78300\&1O[Xl#\DLE\38014\50691~g\1043081{\132217_g\ESC!\SOH't\1101558I\1003044\1063761?\137915`Nd\182690`*1rD25c\169907owK\20714(\1055173\&6i&j(U6p\1104351zK\73918mE\11375\vjr:\43447`\1094897w\SOH\SYN^3\DC3<**`j\f "
    }

testObject_CompletePasswordReset_provider_18 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_18 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "m3ruXwhym9ERHyTAJo1y"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "88xU9QOF1FPXdL6e4"))},
      password =
        plainTextPassword6Unsafe
          "cW\ac,#\12631\n}\188543\EOT\NUL`v\STX\48639\SYNO^\1061062\1045730\1096180\tUs\EM\SOH[b[uz\DC1\1106362\DC3-\995083\1054859+]\GS90\DC3X\GS\v>8\154400\78507\t!\SYN\24475\&2 \RS\fW?\SO\EM\17276>\\S\ACK@Qt~8`T\1073440\97220a,#\1054560]X\1019169-\1112078]\1005234\SYN}\1112649T\999008\1062688\rH\171818d@9/pab%\52983**pK4q$\984140+e\NAK\USZ\DC3\29392\&3\176130\1073376(\ENQy\\(5$'|\2800!_\154876_\1073420\&2\SYN\996743\187317I\DC4Y\"B\1049376=\1103936\&6\66599v\153333\NUL\137084\119859\147584'\16885GJe\FS\SIPk\NAK\ENQ#\29575\23580\SIa>m.\161669to\SO\1049661_\v\165212\FS\ETB\ESC\trn\1029796\1078206\61317\rM\149956\&2S8\a35\\\ETBo\191128\DC4\58762~?\178576\STX/\nNZSjbWH\r\1098678X\993718\4120/\1046632y}#\63730\ACK.Voi9\50993\f*Y\STX\1001056\43180\DC1\rS\1021396\144641qSj\17576X\149262\1081745\1076445M\1063531\22347\CAN\45875\40887B\NAK)U\"~V\1036888\1007909/S\61542y\SOH<\b\ENQ^\SOY\1013585;\SYN-RKy\ESCO\1033537[;\b\1094937EDjW\997383\182740bKx\128165lZ\DC3J(\1032322\&1mK\DEL\69897z\131148\144121i/RxiQ\1085090}E\23345pA\1065790q\v.\v\ng\20319Gb\46475Kt#\USP@8s0\vg[c\169328\DC3\US}{/\1002448D8\170376\159999\987435\67200\1053165M\1079934\1073683\DC10i\159626\1111106\ESC\"\1019962\SYNg\1025072M\1022474\1059584IsD\b\1086244\70682Z\1015255g\DC2;\SOH\1009422cQ0f\STX]0p<\1065421.j\DC3\r[W^rsM\fU\65479=h\1059093L94\993336oNs\1016719Z;8\30468lw\t;S\GS`V\\f\993287\1001923\49875\1018016\1032042X\ESC\NAK\132703sb\SI\1110714C\ETXi_7\138308\NAK\35645}\12913\100683go\FSVj\vtr\SYN\181280\166083;\137762\161816\EM6\1068253\1058678/\n\71064\fp\1036795O\SUB\45835>S1=\DC4>m=Li]y\1014422\r\US\5961+D\230\54691UWo@\1104594n/\EOT\FSDR\1084131\&4\CAN`-}/\v=<\DC1\1011393\DLE\SYN\1000229oB8\1073774\fT\185994?\DLE5lJ\917988\1051232\993358\&1\\\SYNGx\160450\993275HF\988493\1096467N&\DC3A\1078985\ETB\1085595\71193@\a\SON\ETB7\RS8kT\13512\SOH\128792}!`].7C\ACK\EMa\991996\SOH\ESCR\\Iw/y\1052927\162141;*!\SUB;)\1034215\DC3\GSjQ)\98905\1083130 \aQJf\143466\3112\1088669q\183516D\47434Z\r\1051585\1066298\1011799\DEL\31175\1077158\19157\f}\1074960\CAN{\1026108\ACK\165269\989993\1021383\4839\993646_9\FSYzI=JL0]\45720/W\NAKD\ENQ\143508WJ\CAN\DLE8\EOT2JsPn\1025590\20415\bhB\DC1\1537Xj\ESC\GS%:p\64920@gL\ETB\1087542\145056\1111605Q"
    }

testObject_CompletePasswordReset_provider_19 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_19 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "nBmwchpz6q_fDPCPZYQe"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "AHGkmRBXJr="))},
      password =
        plainTextPassword6Unsafe
          "o\58515ZHN\83385\a%S\1039235I\1072272\&0\DC4!\1105051\&0\DC4\FSc|\SYNo\r\141433m\DC4\1017077&\ETBX\GS\101008sXzlOS\1086195nZ\"\DC29\30572S\n\1028791\1110650\SUB1>\155164*K\63527\917991\34537\"\152219\174017GW\165251\917540PPO\1099839\983424\&9c\1001124U{fLy\SOH\FS\STX|\35808\&52\SOHP\n \126255/\42693\188778s^_\1012451MW7M\EOTs\DC2M\RS9\ENQg\1084863\61924@%o\DC2\DC16m&B\95458\190903wL\1066100o\r\1082662y\ETX\tk;>\1108088\1053265A\US9\4469\STX%\44556\SUBghb\1046982\DEL\b%0\1011473\16374'kz\61155\126123\NAK\1040333\&3:<\1028617\12709\1085958X[g77\59354_\ETX\1018780\1031200\&7\STX\GS\163106\60867\n\f\129490\993680iG\181984\58073\168758\44094`i\49314r\65104\GSu\1030407b\1002850\1053366O)\1042687YQ\190781\DEL=fZa)T0j\1070016K/\1104693%v\100085qsY\1017025R\33451\997088\&3a\45926\r\DEL\ESC\1031881\NAK\35199\183615'a\11657B\111310\STX=\RS>n\1055557)d\US\FS.F\1111038M\133759\1021129\&3x\ETX\51747\1102182\11790Z\35206\&3\173723P\RSV3GeGN\v3\28136a^k`\29343\22637\rK\SYN\NAKe(\GSQ%b\1080735u\DLE\ACK\NUL\99272\1095099y{\61538\&9IbP\SI% \CAN^\1064103+f\144228\59518r\18266G\DC2?y\DC3\1073829\RS8\34346}r\45176|y,\b\1026988\145851/\DC1R\1017813\&0W\988979N\NUL\35979q\bc\1091121\NUL\1087940\GS,d\CAN{J8[\1031817\CAN\r\138592\EM|\nz\28160G-{\SI(1\47823\GSl1\18854iL\77903j^\DLE4\ETX\159954\1105693\83316\FSoj\ESC2z\STX\1021083t\17703;K\STX\DEL4Yhr\STX\987287fO6h\158330t\1076871\&3Tpef\SI\ETB\1109588jC\150352eh\10328nf"
    }

testObject_CompletePasswordReset_provider_20 :: CompletePasswordReset
testObject_CompletePasswordReset_provider_20 =
  CompletePasswordReset
    { key = Key {asciiKey = unsafeRange (fromRight undefined (validate "n7oUiCMAvjokyCwCwIZx"))},
      code = Value {asciiValue = unsafeRange (fromRight undefined (validate "padtz-lbyFICM8PEzCj"))},
      password =
        plainTextPassword6Unsafe
          "n.\30576`Ab}1x?dyrWEI\SO\b|\r n\\\174375\163710yzk\CAN\1032873\13076q\1104973h\1078766\1065080\1073163\1024180\SYNzt\1041454oY\SIv\1109814\999708!\DC4b\r\ETB\139978\1037986\&97|\68831Is\78227\48210\984397\1108736R\1076978y*\ESC \1080452\6350;\1002645*F\ETB|l0\\-\49792/d,76W\SOH\1091954a\52507\ESCkY6\EOTe*6\1068076\1010489\STX\1109890%B\"\DC1X\145857Z\1107907\988601\SOH\r\983601\ETB\1027493's\\g\SI_=\187494\52638\139634oOcZ\EM ZWd\EM\STXe\25610Zz\1055806\1023881Me\25012\DC2N\1061919o\154179BmCD\ESC\146744\165530Wsw<\ETXs\SO\992482\11825,\NAK\97960\ACK\1112588je1m\1080113\&5\CAN@\SI)(\39581y%~d\1022649\&0z1||L{1\EOT\1083342Ja\74536\EOThHi\NAK\1033200\SOH\CAN\ACK\175120\183861\DC34&q\GS>\SOH\184139\SOH\ESCQs\41951a\59763\1069217bv[u\bX\1078841\1048633^\1015710;[\STX\DC3\1001312jYw\1003565\1077047\te3\148232+\7427\\\SUBq\1108026r$(zD)\\7p\"\168984\STX\59311?\8657<\NUL\1035836cP\194909[>\USm2Y\1010432\1106430\21518P\DC1.\1074512\35480\ACK\EM\SOH1npVW\SUB2+\ESC\1059649\33997\GSk\v \993759A)*uk\1030453L0\1078688\1044139\DC2\1029875\DLEn\EM$\1054292?\NUL\vji4Y\DC1\EM\1027716=/S\1024040`P.\ENQ\SI\GS\1090161\50097mww\61962\59664e\994460\1030466\f\83226f\"\CAN{X)\v\4796\SYN\STX\119946\DEL\992301+\39597jv^\169149\SUB%C\"]v5?\185720/M\991044\1010224\1027231\984290+\SUB+\186874VG\SOH2\1003544VM\SI\f2'\1009297\1059762?Lx\986666<,Q\1009359t?\1067784\18910\GS\CAN-\1090445C\2603\1004458\10478XZ\STXo\1019324\by\985769,\46054'\a\21265\DLE\SIH\1003281$J\US\1051584\ETB\r6\ACK\FS_1\45810\1013879\998189\167043\DC2>\1082944<0\47209G,T\1055523\12871\1057078:>4\1005909\1060368\GS\FSED\DC2:\rzSMQw)P\50826s\1051230^-~\95981,v\DEL,\SOH1=/GNsO\129350\&6\GS\16013_4\62900\1097318!\SOH'M\139907+$\29092\154621<~E\96994, U8\ETX\986557#\1092210[\1042274.H\DLE\1098681\ACK\1062248\"\133455?I\1005507e\167230\t\28751\1016604\159825\GS'\160639\&9k\EOTZPj\1084498\1039215O\131535L@\171949\r%2>M<\n\120995\1031232\&9/\985482C\SOHav\142062\ENQ-|\ESC)\b$\"\DC26\16379\CANCT|Ut\131524\149842\96725X\64829\v\28384\DEL'yR\1022028\1056329r\25908o\165079\1077144.\185928\&8\NUL;\NAK5\ENQ\ETB8Y\DC4g\1101865Q\1085552\150701\&7!HO;\br\1026135C\24186\37827\SO\ACK\165967E\r\DC2\DC4l3\1090105\127078\DC4\SYN\bUF\15427\DC3.wO\EOThm\164680L]y&\1024985\&0\308;Nwyw\61385#\r8Om@x\1007233\b\ACKo823U\146708C\SIMN6t\DC28\1047608\SOF\ETXRna\a~\r6\fE\US#\tn\1006471wr'B\rnlolj\1017148\144338\1087477tT\119355\1044444\SYN|4G}\SYNEn\1000211\&0D\DLE\SOjn}`0\994578+\1019070\184767"
    }
