import fs from 'node:fs';
import { PuppeteerAgent } from '@/puppeteer';
import { sleep } from '@midscene/core/utils';
import { afterEach, describe, expect, it, vi } from 'vitest';
import { launchPage } from './utils';

vi.setConfig({
  testTimeout: 60 * 1000,
});

describe('Cache Configuration Tests', () => {
  let resetFn: () => Promise<void>;
  afterEach(async () => {
    if (resetFn) {
      await resetFn();
    }
  });

  it('should work with explicit cache ID (read-write mode)', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const agent = new PuppeteerAgent(originPage, {
      cache: {
        id: 'cache-functionality-test',
        strategy: 'read-write',
      },
      testId: 'explicit-cache-test-001',
    });

    // Verify cache is configured correctly
    expect(agent.taskCache).toBeDefined();
    // Cache ID should be explicitly provided
    expect(agent.taskCache?.cacheId).toBe('cache-functionality-test');
    expect(agent.taskCache?.isCacheResultUsed).toBe(true);
    expect(agent.taskCache?.readOnlyMode).toBe(false);

    // Perform an action that should be cached
    await agent.aiAction('click the title');

    // Verify cache file path is set correctly
    const cacheFilePath = agent.taskCache?.cacheFilePath;
    expect(cacheFilePath).toBeDefined();
    if (cacheFilePath) {
      expect(cacheFilePath).toContain('cache-functionality-test'); // Explicit ID
      expect(cacheFilePath).toContain('.cache.yaml');
    }
  });

  it('should work with cache: false (disabled mode)', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const agent = new PuppeteerAgent(originPage, {
      cache: false,
    });

    // Verify cache is configured correctly
    expect(agent.taskCache).toBeUndefined();

    // Perform an action that should be cached
    await agent.aiAction('click the title');

    // Verify cache file path is set correctly
    const cacheFilePath = agent.taskCache?.cacheFilePath;
    expect(cacheFilePath).toBeUndefined();
  });

  it('should work with cache: { strategy: "read-only" } mode', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const agent = new PuppeteerAgent(originPage, {
      cache: {
        id: 'cache-functionality-test',
        strategy: 'read-only',
      },
      testId: 'explicit-cache-test-001',
    });

    // Verify cache is configured correctly
    expect(agent.taskCache).toBeDefined();
    // Cache ID should be explicitly provided
    expect(agent.taskCache?.cacheId).toBe('cache-functionality-test');
    expect(agent.taskCache?.isCacheResultUsed).toBe(true);
    expect(agent.taskCache?.readOnlyMode).toBe(true);

    // Perform an action that should be cached
    await agent.aiAction('click the title');

    // Verify cache file path is set correctly
    const cacheFilePath = agent.taskCache?.cacheFilePath;
    expect(cacheFilePath).toBeDefined();
    if (cacheFilePath) {
      expect(cacheFilePath).toContain('cache-functionality-test'); // Explicit ID
      expect(cacheFilePath).toContain('.cache.yaml');
    }
  });

  it('should work with cache: { strategy: "write-only" } mode', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const agent = new PuppeteerAgent(originPage, {
      cache: {
        id: 'cache-functionality-test',
        strategy: 'write-only',
      },
      testId: 'explicit-cache-test-001',
    });

    expect(agent.taskCache).toBeDefined();
    expect(agent.taskCache?.cacheId).toBe('cache-functionality-test');
    expect(agent.taskCache?.isCacheResultUsed).toBe(false);
    expect(agent.taskCache?.readOnlyMode).toBe(false);
    expect(agent.taskCache?.writeOnlyMode).toBe(true);

    await agent.aiAction('click the title');

    const cacheFilePath = agent.taskCache?.cacheFilePath;
    expect(cacheFilePath).toBeDefined();
    if (cacheFilePath) {
      expect(fs.existsSync(cacheFilePath)).toBe(true);
    }
  });

  it('should prioritize new cache config over legacy cacheId', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const agent = new PuppeteerAgent(originPage, {
      cacheId: 'legacy-cache-id', // This should be ignored
      cache: { id: 'new-cache-id-001' }, // This should take priority
    });

    // Perform an action that should be cached
    await agent.aiAction('click the title');

    // Verify new cache config takes priority
    expect(agent.taskCache).toBeDefined();
    expect(agent.taskCache?.cacheId).toBe('new-cache-id-001');
    expect(agent.taskCache?.readOnlyMode).toBe(false);
  });

  it('should support backward compatibility with legacy cacheId when MIDSCENE_CACHE is enabled', async () => {
    // Mock environment variable
    const originalEnv = process.env.MIDSCENE_CACHE;
    process.env.MIDSCENE_CACHE = 'true';

    try {
      const { originPage, reset } = await launchPage('https://example.com/');
      resetFn = reset;

      const agent = new PuppeteerAgent(originPage, {
        cacheId: 'legacy-cache-test-001',
        // No new cache config, should fall back to legacy
      });

      // Perform an action that should be cached
      await agent.aiAction('click the title');

      // Verify legacy cache works
      expect(agent.taskCache).toBeDefined();
      expect(agent.taskCache?.cacheId).toBe('legacy-cache-test-001');
      expect(agent.taskCache?.readOnlyMode).toBe(false);

      await agent.aiAssert('this is the example.com page');
    } finally {
      // Restore original environment
      if (originalEnv !== undefined) {
        process.env.MIDSCENE_CACHE = originalEnv;
      } else {
        process.env.MIDSCENE_CACHE = undefined;
      }
    }
  });

  it('should not create cache with legacy cacheId when MIDSCENE_CACHE is disabled', async () => {
    // Mock environment variable
    const originalEnv = process.env.MIDSCENE_CACHE;
    process.env.MIDSCENE_CACHE = 'false';

    try {
      const { originPage, reset } = await launchPage('https://example.com/');
      resetFn = reset;

      const agent = new PuppeteerAgent(originPage, {
        cacheId: 'legacy-cache-disabled',
        // No new cache config and env var is false
      });

      // Also perform an aiAction to generate planning cache
      try {
        // Perform an action that should be cached
        await agent.aiAction('click the title');
      } catch (error) {
        // If aiAction fails due to AI parsing, that's ok for this test
        console.log(
          'aiAction failed, but cache configuration test is still valid',
        );
      }
    } finally {
      // Restore original environment
      if (originalEnv !== undefined) {
        process.env.MIDSCENE_CACHE = originalEnv;
      } else {
        process.env.MIDSCENE_CACHE = undefined;
      }
    }
  });
});

describe('Cache Operation Tests', () => {
  let resetFn: () => Promise<void>;
  afterEach(async () => {
    if (resetFn) {
      await resetFn();
    }
  });

  it('should cache and reuse planning results', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    // First agent - should create cache
    const agent1 = new PuppeteerAgent(originPage, {
      cache: { id: 'planning-cache-test-001' },
    });

    // Perform an action that would be planned
    await agent1.aiAction('check if this is the example.com website');

    // Wait for cache to be written
    await sleep(1000);

    // Second agent with same cache ID - should reuse cache
    const agent2 = new PuppeteerAgent(originPage, {
      cache: { id: 'planning-cache-test-001' },
    });

    // Same action should use cached plan
    await agent2.aiAction('check if this is the example.com website');

    // Both should have same cache ID
    expect(agent1.taskCache?.cacheId).toBe(agent2.taskCache?.cacheId);
  });

  it('should handle cache operations correctly', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    // Test flushCache with no cache configured
    const agentNoCache = new PuppeteerAgent(originPage, {
      cache: false,
    });

    await expect(agentNoCache.flushCache()).rejects.toThrow(
      'Cache is not configured',
    );

    // Test flushCache in read-write mode (should fail)
    const agentReadWrite = new PuppeteerAgent(originPage, {
      cache: { id: 'readwrite-flush-test' },
    });

    await expect(agentReadWrite.flushCache()).rejects.toThrow(
      'flushCache() can only be called in read-only mode',
    );

    // Test with normal cache mode (temporarily removing read-only)
    const agentReadOnly = new PuppeteerAgent(originPage, {
      cache: { id: 'readonly-flush-test' }, // Temporarily remove read-only to generate cache
    });

    // Perform some actions to generate cache content
    await agentReadOnly.aiAssert('this is the example.com page');
    await agentReadOnly.aiQuery('What is the page title?');
  });

  it('should handle cache file operations correctly', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const cacheId = 'file-ops-test-001';
    const agent = new PuppeteerAgent(originPage, {
      cache: { id: cacheId },
    });

    // Perform multiple operations to build cache
    await agent.aiAssert('this is example.com');
    await agent.aiQuery('What is the main heading on this page?');

    await sleep(1000);

    // Verify cache configuration is correct
    const cacheFilePath = agent.taskCache?.cacheFilePath;
    expect(cacheFilePath).toBeDefined();
    if (cacheFilePath) {
      expect(cacheFilePath).toContain(cacheId);
      expect(cacheFilePath).toContain('.cache.yaml');

      // Check if cache file might exist (it may or may not be written yet)
      if (fs.existsSync(cacheFilePath)) {
        const cacheContent = fs.readFileSync(cacheFilePath, 'utf-8');
        expect(cacheContent).toContain('midsceneVersion');
        expect(cacheContent).toContain(cacheId);
        expect(cacheContent).toContain('caches:');
      }
    }
  });

  it('should handle cache with cacheable: false option', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const agent = new PuppeteerAgent(originPage, {
      cache: { id: 'non-cacheable-test-001' },
    });

    // Perform action with cacheable: false (use assert instead of action to avoid AI parsing issues)
    await agent.aiAssert('this is the example.com page');

    await sleep(1000);

    // Cache should exist
    expect(agent.taskCache).toBeDefined();

    // Perform another action (use assert which is more reliable)
    await agent.aiAssert('the page title contains text');

    await sleep(1000);

    // Verify cache configuration
    const cacheFilePath = agent.taskCache?.cacheFilePath;
    expect(cacheFilePath).toBeDefined();
    if (cacheFilePath) {
      expect(cacheFilePath).toContain('.cache.yaml');
    }
  });
});

describe('Cache Edge Cases', () => {
  let resetFn: () => Promise<void>;
  afterEach(async () => {
    if (resetFn) {
      await resetFn();
    }
  });

  it('should handle very long cache IDs by truncating', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const longCacheId = 'a'.repeat(300); // Very long ID
    const agent = new PuppeteerAgent(originPage, {
      cache: { id: longCacheId },
    });

    // Should truncate and add hash
    expect(agent.taskCache).toBeDefined();
    expect(agent.taskCache?.cacheId).not.toBe(longCacheId);
    expect(agent.taskCache?.cacheId.length).toBeLessThan(longCacheId.length);

    // Should still work
    await agent.aiAssert('this is example.com');
  });

  it('should handle special characters in cache ID', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const specialCacheId = 'test/cache\\id:with*special?chars<>|"';
    const agent = new PuppeteerAgent(originPage, {
      cache: { id: specialCacheId },
    });

    // Should sanitize illegal filename characters (but preserve / and \ as they might be path separators)
    expect(agent.taskCache).toBeDefined();
    const sanitizedId = agent.taskCache?.cacheId || '';
    // These characters should be replaced with dashes
    expect(sanitizedId).not.toContain(':');
    expect(sanitizedId).not.toContain('*');
    expect(sanitizedId).not.toContain('?');
    expect(sanitizedId).not.toContain('<');
    expect(sanitizedId).not.toContain('>');
    expect(sanitizedId).not.toContain('|');
    expect(sanitizedId).not.toContain('"');
    expect(sanitizedId).not.toContain(' '); // spaces should be replaced
    // Note: / and \ might be preserved as path separators

    // Should still work
    await agent.aiAssert('this is example.com');
  });

  it('should handle multiple agents with same cache ID correctly', async () => {
    const { originPage, reset } = await launchPage('https://example.com/');
    resetFn = reset;

    const sharedCacheId = 'shared-cache-test-001';

    // Create first agent
    const agent1 = new PuppeteerAgent(originPage, {
      cache: { id: sharedCacheId },
    });

    await agent1.aiAssert('this is example.com');
    await sleep(500);

    // Create second agent with same cache ID
    const agent2 = new PuppeteerAgent(originPage, {
      cache: { id: sharedCacheId },
    });

    await agent2.aiAssert('this is still example.com');

    // Both should share the same cache
    expect(agent1.taskCache?.cacheId).toBe(agent2.taskCache?.cacheId);
    expect(agent1.taskCache?.cacheFilePath).toBe(
      agent2.taskCache?.cacheFilePath,
    );
  });
});
