import type { GlobalConfig, TestContext, TestInfo } from '../-types';
import type { DiagnosticReport, InteractionEvent, Reporter, TestReport } from '../-types/report';
import equiv from '../legacy/equiv';

class InternalCompat<TC extends TestContext> {
  declare _diagnostic: Diagnostic<TC>;

  constructor(diagnostic: Diagnostic<TC>) {
    this._diagnostic = diagnostic;
  }

  get testId(): string {
    return this._diagnostic.__currentTest.id;
  }

  get expected(): number | null {
    return this._diagnostic.expected;
  }
  set expected(value) {
    this._diagnostic.expected = value;
  }
}

export class Diagnostic<TC extends TestContext> {
  declare __currentTest: TestInfo<TC>;
  declare __report: TestReport;
  declare __config: GlobalConfig;
  declare __reporter: Reporter;
  declare expected: number | null;
  declare _steps: string[];

  // QUnit private API compat
  declare test: InternalCompat<TC>;

  constructor(reporter: Reporter, config: GlobalConfig, test: TestInfo<TC>, report: TestReport) {
    this.__currentTest = test;
    this.__report = report;
    this.__config = config;
    this.__reporter = reporter;
    this.expected = null;
    this._steps = [];
    this.test = new InternalCompat(this);
  }

  pushInteraction(interaction: InteractionEvent): void {
    if (this.__config.params.timeline.value) {
      const timestamp = this.__config.params.instrument ? performance.now() : null;
      this.__report.timeline.push({ event: interaction, timestamp });
      this.__reporter.updateTimeline(this.__report);
    }
  }

  pushResult(
    result: Pick<DiagnosticReport, 'actual' | 'expected' | 'message' | 'passed' | 'stack'> & { result?: boolean }
  ): void {
    const diagnostic = Object.assign({ passed: result.passed ?? result.result }, result, {
      testId: this.__currentTest.id,
    });
    this.__report.result.diagnostics.push(diagnostic);

    if (this.__config.params.timeline.value) {
      const timestamp = this.__config.params.instrument ? performance.now() : null;
      this.__report.timeline.push({ event: diagnostic, timestamp });
      this.__reporter.updateTimeline(this.__report);
    }

    if (!diagnostic.passed) {
      this.__report.result.passed = false;
      this.__report.result.failed = true;
    }

    this.__reporter.onDiagnostic(diagnostic);
  }

  equal<T>(actual: T, expected: T, message?: string): void {
    if (actual !== expected) {
      if (!this.__config.params.noTryCatch.value) {
        try {
          throw new Error(message || `Expected ${String(actual)} to equal ${String(expected)}`);
        } catch (err) {
          this.pushResult({
            message: message || 'equal',
            stack: (err as Error).stack!,
            passed: false,
            actual,
            expected,
          });
        }
      } else {
        this.pushResult({
          message: message || 'equal',
          stack: '',
          passed: false,
          actual,
          expected,
        });
      }
    } else {
      this.pushResult({
        message: message || 'equal',
        stack: '',
        passed: true,
        actual: true,
        expected: true,
      });
    }
  }

  notEqual<T>(actual: T, expected: T, message?: string): void {
    if (actual === expected) {
      if (!this.__config.params.noTryCatch.value) {
        try {
          throw new Error(message || `Expected ${String(actual)} to not equal ${String(expected)}`);
        } catch (err) {
          this.pushResult({
            message: message || 'notEqual',
            stack: (err as Error).stack!,
            passed: false,
            actual,
            expected,
          });
        }
      } else {
        this.pushResult({
          message: message || 'notEqual',
          stack: '',
          passed: false,
          actual,
          expected,
        });
      }
    } else {
      this.pushResult({
        message: message || 'notEqual',
        stack: '',
        passed: true,
        actual: true,
        expected: true,
      });
    }
  }

  deepEqual<T>(actual: T, expected: T, message?: string): void {
    const isEqual = equiv(actual, expected, true);
    if (!isEqual) {
      if (!this.__config.params.noTryCatch.value) {
        try {
          throw new Error(message || `Expected items to be equivalent`);
        } catch (err) {
          this.pushResult({
            message: message || 'deepEqual',
            stack: (err as Error).stack!,
            passed: false,
            actual,
            expected,
          });
        }
      } else {
        this.pushResult({
          message: message || 'deepEqual',
          stack: '',
          passed: false,
          actual,
          expected,
        });
      }
    } else {
      this.pushResult({
        message: message || 'deepEqual',
        stack: '',
        passed: true,
        actual: true,
        expected: true,
      });
    }
  }

  /**
   * Checks if the actual object satisfies the expected object.
   *
   * This is a deep comparison that will check if all the properties
   * of the expected object are present in the actual object with the
   * same values.
   *
   * This differs from deepEqual in that extra properties on the actual
   * object are allowed.
   *
   * This is great for contract testing APIs that may accept a broader
   * object from which a subset of properties are used, or for testing
   * higher priority or more stable properties of an object in a dynamic
   * environment.
   *
   */
  satisfies<T extends object, J extends T>(actual: J | undefined | null, expected: T, message?: string): void {
    const isEqual = equiv(actual, expected, false);
    if (!isEqual) {
      if (!this.__config.params.noTryCatch.value) {
        try {
          throw new Error(message || `Expected items to be equivalent`);
        } catch (err) {
          this.pushResult({
            message: message || 'satisfies',
            stack: (err as Error).stack!,
            passed: false,
            actual,
            expected,
          });
        }
      } else {
        this.pushResult({
          message: message || 'satisfies',
          stack: '',
          passed: false,
          actual,
          expected,
        });
      }
    } else {
      this.pushResult({
        message: message || 'satisfies',
        stack: '',
        passed: true,
        actual: true,
        expected: true,
      });
    }
  }

  arrayEquals<T>(actual: T[], expected: T[], message: string): void {
    if (!Array.isArray(actual)) {
      this.pushResult({
        message: 'Expected the value for "actual" to be an array | ' + message,
        stack: '',
        passed: false,
        actual: false,
        expected: true,
      });
      return;
    }
    if (!Array.isArray(expected)) {
      this.pushResult({
        message: 'Expected the value for "expected" to be an array',
        stack: '',
        passed: false,
        actual: false,
        expected: true,
      });
      return;
    }
    let passed = actual.length === expected.length;

    const actualRefs = new Map<T, string>();
    const actualSerialized: string[] = actual.map((item, index) => {
      const ref = refFromIndex(index, '');
      actualRefs.set(item, ref);
      return ref;
    });
    const expectedSerialized: string[] = expected.map((item, index) => {
      return getRefForItem(actualRefs, item, index);
    });

    if (passed) {
      for (let i = 0; i < actual.length; i++) {
        if (actual[i] !== expected[i]) {
          passed = false;
          break;
        }
      }
    }

    this.pushResult({
      message,
      stack: '',
      passed,
      actual: actualSerialized,
      expected: expectedSerialized,
    });
  }

  notDeepEqual<T>(actual: T, expected: T, message?: string): void {
    const isEqual = equiv(actual, expected, true);
    if (isEqual) {
      if (!this.__config.params.noTryCatch.value) {
        try {
          throw new Error(message || `Expected items to not be equivalent`);
        } catch (err) {
          this.pushResult({
            message: message || 'notDeepEqual',
            stack: (err as Error).stack!,
            passed: false,
            actual,
            expected,
          });
        }
      } else {
        this.pushResult({
          message: message || 'notDeepEqual',
          stack: '',
          passed: false,
          actual,
          expected,
        });
      }
    } else {
      this.pushResult({
        message: message || 'notDeepEqual',
        stack: '',
        passed: true,
        actual: true,
        expected: true,
      });
    }
  }

  true(actual: boolean, message?: string): void {
    this.equal(actual, true, message);
  }

  false(actual: boolean, message?: string): void {
    this.equal(actual, false, message);
  }

  ok(actual: unknown, message?: string): void {
    this.equal(!!actual, true, message);
  }

  notOk(actual: unknown, message?: string): void {
    this.equal(!!actual, false, message);
  }

  expect(count: number): void {
    this.expected = count;
  }

  step(name: string): void {
    this._steps.push(name);
  }

  verifySteps(steps: string[], message?: string): void {
    this.deepEqual(this._steps, steps, message);
    this._steps = [];
  }

  _finalize(): void {
    if (this.expected !== null && this.expected !== this.__report.result.diagnostics.length) {
      this.pushResult({
        message: `Expected ${this.expected} assertions, but ${this.__report.result.diagnostics.length} were run`,
        stack: '',
        passed: false,
        actual: false,
        expected: true,
      });
    }
    if (this.__report.result.diagnostics.length === 0) {
      this.pushResult({
        message: `Expected at least one assertion, but none were run`,
        stack: '',
        passed: false,
        actual: false,
        expected: true,
      });
    }
    if (this._steps.length) {
      this.pushResult({
        message: `Expected 0 steps remaining to verify, but ${this._steps.length} were run`,
        stack: '',
        passed: false,
        actual: false,
        expected: true,
      });
    }
  }

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  throws(fn: () => Promise<any>, expected?: string | RegExp, message?: string): Promise<void>;
  throws(fn: () => void, expected?: string | RegExp, message?: string): void;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  throws(fn: () => void | Promise<any>, expected?: string | RegExp, message?: string): Promise<void> | void {
    try {
      const result = fn();
      const resolved = Promise.resolve(result);
      const isPromise = resolved === result;

      if (!isPromise) {
        throw new Error(`Expected function to throw ${expected}`);
      }

      return resolved.then(
        () => {
          throw new Error(`Expected function to throw ${expected}`);
        },
        (err: Error | string) => {
          if (expected) {
            if (typeof expected === 'string') {
              this.equal(typeof err === 'string' ? err : err.message, expected, message);
            } else {
              this.equal(typeof err === 'string' ? err : expected.test(err.message), true, message);
            }
          }
        }
      );
    } catch (err) {
      if (expected) {
        if (typeof expected === 'string') {
          const result = err instanceof Error ? err.message === expected || String(err) === expected : err === expected;
          this.pushResult({
            message: message || 'expected an error to be thrown',
            stack:
              err instanceof Error || (typeof err === 'object' && err !== null)
                ? ((err as { stack?: string }).stack ?? '')
                : '',
            passed: result,
            actual: err instanceof Error ? err.message : (err as string),
            expected,
          });
        } else {
          const result =
            err instanceof Error
              ? expected.test(err.message) || expected.test(String(err))
              : expected.test(err as string);
          this.pushResult({
            message: message || 'expected an error to be thrown',
            stack:
              err instanceof Error || (typeof err === 'object' && err !== null)
                ? ((err as { stack?: string }).stack ?? '')
                : '',
            passed: result,
            actual: err instanceof Error ? err.message : (err as string),
            expected,
          });
        }
      }
    }
  }

  doesNotThrow(fn: () => Promise<void>, expected?: string | RegExp, message?: string): Promise<void>;
  doesNotThrow(fn: () => void, expected?: string | RegExp, message?: string): void;
  doesNotThrow(fn: () => void | Promise<void>, expected?: string | RegExp, message?: string): Promise<void> | void {
    try {
      const result = fn();
      const resolved = Promise.resolve(result);
      const isPromise = resolved === result;

      if (!isPromise) {
        return;
      }

      return resolved.then(
        () => {
          return;
        },
        (err: Error | string) => {
          if (expected) {
            if (typeof expected === 'string') {
              this.equal(typeof err === 'string' ? err : err.message, expected, message);
            } else {
              this.equal(expected.test(typeof err === 'string' ? err : err.message), true, message);
            }
          }
        }
      );
    } catch (err) {
      if (expected) {
        if (typeof expected === 'string') {
          this.equal(err instanceof Error ? err.message : err, expected, message);
        } else {
          this.equal(expected.test(err instanceof Error ? err.message : (err as string)), true, message);
        }
      }
    }
  }
}

function refFromIndex(index: number, suffix: string): string {
  return `<ref:@${index}${suffix}>`;
}
function getRefForItem<T>(map: Map<T, string>, item: T, index: number): string {
  let ref = map.get(item);
  if (ref === undefined) {
    ref = refFromIndex(index, 'b');
  }
  return ref;
}
