use crate::cli::{ApplyRangeMode, StorageSource};
use crate::commands::{maybe_print_db_stats, maybe_save_trie_changes};
use crate::progress_reporter::{timestamp_ms, ProgressReporter};
use near_chain::chain::collect_receipts_from_response;
use near_chain::migrations::check_if_block_is_first_with_chunk_of_version;
use near_chain::types::{
    ApplyChunkBlockContext, ApplyChunkResult, ApplyChunkShardContext, RuntimeAdapter,
};
use near_chain::{ChainStore, ChainStoreAccess, ChainStoreUpdate};
use near_chain_configs::Genesis;
use near_epoch_manager::{EpochManagerAdapter, EpochManagerHandle};
use near_primitives::apply::ApplyChunkReason;
use near_primitives::receipt::DelayedReceiptIndices;
use near_primitives::transaction::{Action, ExecutionOutcomeWithId, ExecutionOutcomeWithProof};
use near_primitives::trie_key::TrieKey;
use near_primitives::types::chunk_extra::ChunkExtra;
use near_primitives::types::{BlockHeight, ShardId};
use near_store::flat::{BlockInfo, FlatStateChanges, FlatStorageStatus};
use near_store::{DBCol, Store};
use nearcore::NightshadeRuntime;
use rayon::iter::{IntoParallelIterator, ParallelIterator};
use std::fs::File;
use std::io::Write;
use std::sync::atomic::{AtomicU64, Ordering};
use std::sync::{Arc, Mutex};

fn old_outcomes(
    store: Store,
    new_outcomes: &[ExecutionOutcomeWithId],
) -> Vec<ExecutionOutcomeWithId> {
    new_outcomes
        .iter()
        .map(|outcome| {
            let old_outcome = store
                .iter_prefix_ser::<ExecutionOutcomeWithProof>(
                    DBCol::TransactionResultForBlock,
                    outcome.id.as_ref(),
                )
                .next()
                .unwrap()
                .unwrap()
                .1
                .outcome;
            ExecutionOutcomeWithId { id: outcome.id, outcome: old_outcome }
        })
        .collect()
}

fn maybe_add_to_csv(csv_file_mutex: &Mutex<Option<&mut File>>, s: &str) {
    let mut csv_file = csv_file_mutex.lock().unwrap();
    if let Some(csv_file) = csv_file.as_mut() {
        writeln!(csv_file, "{}", s).unwrap();
    }
}

fn apply_block_from_range(
    mode: ApplyRangeMode,
    height: BlockHeight,
    shard_id: ShardId,
    read_store: Store,
    write_store: Option<Store>,
    genesis: &Genesis,
    epoch_manager: &EpochManagerHandle,
    runtime_adapter: Arc<dyn RuntimeAdapter>,
    progress_reporter: &ProgressReporter,
    verbose_output: bool,
    csv_file_mutex: &Mutex<Option<&mut File>>,
    only_contracts: bool,
    storage: StorageSource,
) {
    // normally save_trie_changes depends on whether the node is
    // archival, but here we don't care, and can just set it to false
    // since we're not writing anything to the read store anyway
    let mut read_chain_store =
        ChainStore::new(read_store.clone(), genesis.config.genesis_height, false);
    let block_hash = match read_chain_store.get_block_hash_by_height(height) {
        Ok(block_hash) => block_hash,
        Err(_) => {
            // Skipping block because it's not available in ChainStore.
            progress_reporter.inc_and_report_progress(0);
            return;
        }
    };
    let block = read_chain_store.get_block(&block_hash).unwrap();
    let shard_uid = epoch_manager.shard_id_to_uid(shard_id, block.header().epoch_id()).unwrap();
    assert!(block.chunks().len() > 0);
    let mut existing_chunk_extra = None;
    let mut prev_chunk_extra = None;
    let mut num_tx = 0;
    let mut num_receipt = 0;
    let chunk_present: bool;

    let block_author = epoch_manager
        .get_block_producer(block.header().epoch_id(), block.header().height())
        .unwrap();

    let apply_result = if block.header().is_genesis() {
        if verbose_output {
            println!("Skipping the genesis block #{}.", height);
        }
        progress_reporter.inc_and_report_progress(0);
        return;
    } else if block.chunks()[shard_id as usize].height_included() == height {
        chunk_present = true;
        let res_existing_chunk_extra = read_chain_store.get_chunk_extra(&block_hash, &shard_uid);
        assert!(
            res_existing_chunk_extra.is_ok(),
            "Can't get existing chunk extra for block #{}",
            height
        );
        existing_chunk_extra = Some(res_existing_chunk_extra.unwrap());
        let chunk_hash = block.chunks()[shard_id as usize].chunk_hash();
        let chunk = read_chain_store.get_chunk(&chunk_hash).unwrap_or_else(|error| {
            panic!(
                "Can't get chunk on height: {} chunk_hash: {:?} error: {}",
                height, chunk_hash, error
            );
        });

        let prev_block = match read_chain_store.get_block(block.header().prev_hash()) {
            Ok(prev_block) => prev_block,
            Err(_) => {
                if verbose_output {
                    println!("Skipping applying block #{} because the previous block is unavailable and I can't determine the gas_price to use.", height);
                }
                maybe_add_to_csv(
                    csv_file_mutex,
                    &format!(
                        "{},{},{},,,{},,{},,",
                        height,
                        block_hash,
                        block_author,
                        block.header().raw_timestamp(),
                        chunk_present
                    ),
                );
                progress_reporter.inc_and_report_progress(0);
                return;
            }
        };

        let chain_store_update = ChainStoreUpdate::new(&mut read_chain_store);
        let receipt_proof_response = chain_store_update
            .get_incoming_receipts_for_shard(
                epoch_manager,
                shard_id,
                block_hash,
                prev_block.chunks()[shard_id as usize].height_included(),
            )
            .unwrap();
        let receipts = collect_receipts_from_response(&receipt_proof_response);

        let chunk_inner = chunk.cloned_header().take_inner();
        let is_first_block_with_chunk_of_version = check_if_block_is_first_with_chunk_of_version(
            &read_chain_store,
            epoch_manager,
            block.header().prev_hash(),
            shard_id,
        )
        .unwrap();

        num_receipt = receipts.len();
        num_tx = chunk.transactions().len();
        if only_contracts {
            let mut has_contracts = false;
            for tx in chunk.transactions() {
                for action in tx.transaction.actions() {
                    has_contracts = has_contracts
                        || matches!(action, Action::FunctionCall(_) | Action::DeployContract(_));
                }
            }
            if !has_contracts {
                progress_reporter.skipped.fetch_add(1, Ordering::Relaxed);
                return;
            }
        }

        runtime_adapter
            .apply_chunk(
                storage.create_runtime_storage(*chunk_inner.prev_state_root()),
                ApplyChunkReason::UpdateTrackedShard,
                ApplyChunkShardContext {
                    shard_id,
                    last_validator_proposals: chunk_inner.prev_validator_proposals(),
                    gas_limit: chunk_inner.gas_limit(),
                    is_new_chunk: true,
                    is_first_block_with_chunk_of_version,
                },
                ApplyChunkBlockContext::from_header(
                    block.header(),
                    prev_block.header().next_gas_price(),
                    block.block_congestion_info(),
                ),
                &receipts,
                chunk.transactions(),
            )
            .unwrap()
    } else {
        chunk_present = false;
        let chunk_extra =
            read_chain_store.get_chunk_extra(block.header().prev_hash(), &shard_uid).unwrap();
        prev_chunk_extra = Some(chunk_extra.clone());

        runtime_adapter
            .apply_chunk(
                storage.create_runtime_storage(*chunk_extra.state_root()),
                ApplyChunkReason::UpdateTrackedShard,
                ApplyChunkShardContext {
                    shard_id,
                    last_validator_proposals: chunk_extra.validator_proposals(),
                    gas_limit: chunk_extra.gas_limit(),
                    is_new_chunk: false,
                    is_first_block_with_chunk_of_version: false,
                },
                ApplyChunkBlockContext::from_header(
                    block.header(),
                    block.header().next_gas_price(),
                    block.block_congestion_info(),
                ),
                &[],
                &[],
            )
            .unwrap()
    };

    let protocol_version =
        epoch_manager.get_epoch_protocol_version(block.header().epoch_id()).unwrap();
    let (outcome_root, _) = ApplyChunkResult::compute_outcomes_proof(&apply_result.outcomes);
    let chunk_extra = ChunkExtra::new(
        protocol_version,
        &apply_result.new_root,
        outcome_root,
        apply_result.validator_proposals.clone(),
        apply_result.total_gas_burnt,
        genesis.config.gas_limit,
        apply_result.total_balance_burnt,
        apply_result.congestion_info,
    );

    let state_update =
        runtime_adapter.get_tries().new_trie_update(shard_uid, *chunk_extra.state_root());
    let delayed_indices =
        near_store::get::<DelayedReceiptIndices>(&state_update, &TrieKey::DelayedReceiptIndices);

    match existing_chunk_extra {
        Some(existing_chunk_extra) => {
            if verbose_output {
                println!("block_height: {}, block_hash: {}\nchunk_extra: {:#?}\nexisting_chunk_extra: {:#?}\noutcomes: {:#?}", height, block_hash, chunk_extra, existing_chunk_extra, apply_result.outcomes);
            }
            if !smart_equals(&existing_chunk_extra, &chunk_extra) {
                maybe_print_db_stats(write_store);
                panic!("Got a different ChunkExtra:\nblock_height: {}, block_hash: {}\nchunk_extra: {:#?}\nexisting_chunk_extra: {:#?}\nnew outcomes: {:#?}\n\nold outcomes: {:#?}\n", height, block_hash, chunk_extra, existing_chunk_extra, apply_result.outcomes, old_outcomes(read_store, &apply_result.outcomes));
            }
        }
        None => {
            assert!(prev_chunk_extra.is_some());
            assert!(apply_result.outcomes.is_empty());
            if verbose_output {
                println!("block_height: {}, block_hash: {}\nchunk_extra: {:#?}\nprev_chunk_extra: {:#?}\noutcomes: {:#?}", height, block_hash, chunk_extra, prev_chunk_extra, apply_result.outcomes);
            }
        }
    };
    maybe_add_to_csv(
        csv_file_mutex,
        &format!(
            "{},{},{},{},{},{},{},{},{},{},{}",
            height,
            block_hash,
            block_author,
            num_tx,
            num_receipt,
            block.header().raw_timestamp(),
            apply_result.total_gas_burnt,
            chunk_present,
            apply_result.processed_delayed_receipts.len(),
            delayed_indices.unwrap_or(None).map_or(0, |d| d.next_available_index - d.first_index),
            apply_result.trie_changes.state_changes().len(),
        ),
    );
    progress_reporter.inc_and_report_progress(apply_result.total_gas_burnt);

    if mode == ApplyRangeMode::Benchmarking {
        // Compute delta and immediately apply to flat storage.
        let changes =
            FlatStateChanges::from_state_changes(apply_result.trie_changes.state_changes());
        let delta = near_store::flat::FlatStateDelta {
            metadata: near_store::flat::FlatStateDeltaMetadata {
                block: BlockInfo {
                    hash: block_hash,
                    height: block.header().height(),
                    prev_hash: *block.header().prev_hash(),
                },
                prev_block_with_changes: None,
            },
            changes,
        };

        let flat_storage_manager = runtime_adapter.get_flat_storage_manager();
        let flat_storage = flat_storage_manager.get_flat_storage_for_shard(shard_uid).unwrap();
        let store_update = flat_storage.add_delta(delta).unwrap();
        store_update.commit().unwrap();
        flat_storage.update_flat_head(&block_hash).unwrap();

        // Apply trie changes to trie node caches.
        let mut fake_store_update = read_store.store_update();
        apply_result.trie_changes.insertions_into(&mut fake_store_update);
        apply_result.trie_changes.deletions_into(&mut fake_store_update);
    } else {
        if let Err(err) = maybe_save_trie_changes(
            write_store,
            genesis.config.genesis_height,
            apply_result,
            height,
            shard_id,
        ) {
            panic!("Error while saving trie changes at height {height}, shard {shard_id} ({err})");
        }
    }
}

pub fn apply_chain_range(
    mode: ApplyRangeMode,
    read_store: Store,
    write_store: Option<Store>,
    genesis: &Genesis,
    start_height: Option<BlockHeight>,
    end_height: Option<BlockHeight>,
    shard_id: ShardId,
    epoch_manager: &EpochManagerHandle,
    runtime_adapter: Arc<NightshadeRuntime>,
    verbose_output: bool,
    csv_file: Option<&mut File>,
    only_contracts: bool,
    storage: StorageSource,
) {
    let parent_span = tracing::debug_span!(
        target: "state_viewer",
        "apply_chain_range",
        ?mode,
        ?start_height,
        ?end_height,
        %shard_id,
        only_contracts,
        ?storage)
    .entered();
    let chain_store = ChainStore::new(read_store.clone(), genesis.config.genesis_height, false);
    let (start_height, end_height) = match mode {
        ApplyRangeMode::Benchmarking => {
            // Benchmarking mode requires flat storage and retrieves start and
            // end heights from flat storage and chain.
            assert!(matches!(storage, StorageSource::FlatStorage));
            assert!(start_height.is_none());
            assert!(end_height.is_none());

            let chain_store =
                ChainStore::new(read_store.clone(), genesis.config.genesis_height, false);
            let final_head = chain_store.final_head().unwrap();
            let shard_layout = epoch_manager.get_shard_layout(&final_head.epoch_id).unwrap();
            let shard_uid = near_primitives::shard_layout::ShardUId::from_shard_id_and_layout(
                shard_id,
                &shard_layout,
            );
            let flat_head = match near_store::flat::store_helper::get_flat_storage_status(
                &read_store,
                shard_uid,
            ) {
                Ok(FlatStorageStatus::Ready(ready_status)) => ready_status.flat_head,
                status => {
                    panic!("cannot create flat storage for shard {shard_id} with status {status:?}")
                }
            };
            let flat_storage_manager = runtime_adapter.get_flat_storage_manager();
            flat_storage_manager.create_flat_storage_for_shard(shard_uid).unwrap();

            // Note that first height to apply is the first one after flat
            // head.
            (flat_head.height + 1, final_head.height)
        }
        _ => (
            start_height.unwrap_or_else(|| chain_store.tail().unwrap()),
            end_height.unwrap_or_else(|| chain_store.head().unwrap().height),
        ),
    };

    println!(
        "Applying chunks in the range {}..={} for shard_id {}",
        start_height, end_height, shard_id
    );

    println!("Printing results including outcomes of applying receipts");
    let csv_file_mutex = Mutex::new(csv_file);
    maybe_add_to_csv(&csv_file_mutex, "Height,Hash,Author,#Tx,#Receipt,Timestamp,GasUsed,ChunkPresent,#ProcessedDelayedReceipts,#DelayedReceipts,#StateChanges");

    let range = start_height..=end_height;
    let progress_reporter = ProgressReporter {
        cnt: AtomicU64::new(0),
        ts: AtomicU64::new(timestamp_ms()),
        all: (end_height + 1).saturating_sub(start_height),
        skipped: AtomicU64::new(0),
        empty_blocks: AtomicU64::new(0),
        non_empty_blocks: AtomicU64::new(0),
        tgas_burned: AtomicU64::new(0),
    };
    let process_height = |height| {
        apply_block_from_range(
            mode,
            height,
            shard_id,
            read_store.clone(),
            write_store.clone(),
            genesis,
            epoch_manager,
            runtime_adapter.clone(),
            &progress_reporter,
            verbose_output,
            &csv_file_mutex,
            only_contracts,
            storage,
        );
    };

    match mode {
        ApplyRangeMode::Sequential | ApplyRangeMode::Benchmarking => {
            range.into_iter().for_each(|height| {
                let _span = tracing::debug_span!(
                    target: "state_viewer",
                    parent: &parent_span,
                    "process_block_in_order",
                    height)
                .entered();
                process_height(height)
            });
        }
        ApplyRangeMode::Parallel => {
            range.into_par_iter().for_each(|height| {
                let _span = tracing::debug_span!(
                target: "mock_node",
                parent: &parent_span,
                "process_block_in_parallel",
                height)
                .entered();
                process_height(height)
            });
        }
    }

    println!(
        "No differences found after applying chunks in the range {}..={} for shard_id {}",
        start_height, end_height, shard_id
    );
}

/**
 * With the database migration we can get into the situation where there are different
 * ChunkExtra versions in database and produced by `neard` playback. Consider them equal as
 * long as the content is equal.
 */
fn smart_equals(extra1: &ChunkExtra, extra2: &ChunkExtra) -> bool {
    if (extra1.outcome_root() != extra2.outcome_root())
        || (extra1.state_root() != extra2.state_root())
        || (extra1.gas_limit() != extra2.gas_limit())
        || (extra1.gas_used() != extra2.gas_used())
        || (extra1.balance_burnt() != extra2.balance_burnt())
    {
        return false;
    }
    let mut proposals1 = extra1.validator_proposals();
    let mut proposals2 = extra2.validator_proposals();
    if proposals1.len() != proposals2.len() {
        return false;
    }
    for _ in 0..proposals1.len() {
        let p1 = proposals1.next().unwrap();
        let p2 = proposals2.next().unwrap();

        if p1.into_v1() != p2.into_v1() {
            return false;
        }
    }
    true
}

#[cfg(test)]
mod test {
    use std::io::{Read, Seek, SeekFrom};
    use std::path::Path;

    use near_chain::Provenance;
    use near_chain_configs::test_utils::TESTING_INIT_STAKE;
    use near_chain_configs::Genesis;
    use near_client::test_utils::TestEnv;
    use near_client::ProcessTxResponse;
    use near_crypto::{InMemorySigner, KeyType};
    use near_epoch_manager::EpochManager;
    use near_primitives::transaction::SignedTransaction;
    use near_primitives::types::{BlockHeight, BlockHeightDelta, NumBlocks};
    use near_store::genesis::initialize_genesis_state;
    use near_store::test_utils::create_test_store;
    use near_store::Store;
    use nearcore::NightshadeRuntime;

    use crate::apply_chain_range::apply_chain_range;
    use crate::cli::{ApplyRangeMode, StorageSource};

    fn setup(epoch_length: NumBlocks) -> (Store, Genesis, TestEnv) {
        let mut genesis =
            Genesis::test(vec!["test0".parse().unwrap(), "test1".parse().unwrap()], 1);
        genesis.config.num_block_producer_seats = 2;
        genesis.config.num_block_producer_seats_per_shard = vec![2];
        genesis.config.epoch_length = epoch_length;
        let store = create_test_store();
        initialize_genesis_state(store.clone(), &genesis, None);
        let epoch_manager = EpochManager::new_arc_handle(store.clone(), &genesis.config);
        let nightshade_runtime = NightshadeRuntime::test(
            Path::new("."),
            store.clone(),
            &genesis.config,
            epoch_manager.clone(),
        );
        let env = TestEnv::builder(&genesis.config)
            .validator_seats(2)
            .stores(vec![store.clone()])
            .epoch_managers(vec![epoch_manager])
            .runtimes(vec![nightshade_runtime])
            .build();
        (store, genesis, env)
    }

    /// Produces blocks, avoiding the potential failure where the client is not the
    /// block producer for each subsequent height (this can happen when a new validator
    /// is staked since they will also have heights where they should produce the block instead).
    fn safe_produce_blocks(
        env: &mut TestEnv,
        initial_height: BlockHeight,
        num_blocks: BlockHeightDelta,
        block_without_chunks: Option<BlockHeight>,
    ) {
        let mut h = initial_height;
        let mut blocks = vec![];
        for _ in 1..=num_blocks {
            let mut block = None;
            // `env.clients[0]` may not be the block producer at `h`,
            // loop until we find a height env.clients[0] should produce.
            while block.is_none() {
                block = env.clients[0].produce_block(h).unwrap();
                h += 1;
            }
            let mut block = block.unwrap();
            if let Some(block_without_chunks) = block_without_chunks {
                if block_without_chunks == h {
                    assert!(!blocks.is_empty());
                    testlib::process_blocks::set_no_chunk_in_block(
                        &mut block,
                        blocks.last().unwrap(),
                    )
                }
            }
            blocks.push(block.clone());
            env.process_block(0, block, Provenance::PRODUCED);
        }
    }

    #[test]
    fn test_apply_chain_range() {
        let epoch_length = 4;
        let (store, genesis, mut env) = setup(epoch_length);
        let genesis_hash = *env.clients[0].chain.genesis().hash();
        let signer =
            InMemorySigner::from_seed("test1".parse().unwrap(), KeyType::ED25519, "test1").into();
        let tx = SignedTransaction::stake(
            1,
            "test1".parse().unwrap(),
            &signer,
            TESTING_INIT_STAKE,
            signer.public_key(),
            genesis_hash,
        );
        assert_eq!(env.clients[0].process_tx(tx, false, false), ProcessTxResponse::ValidTx);

        safe_produce_blocks(&mut env, 1, epoch_length * 2 + 1, None);

        initialize_genesis_state(store.clone(), &genesis, None);
        let epoch_manager = EpochManager::new_arc_handle(store.clone(), &genesis.config);
        let runtime = NightshadeRuntime::test(
            Path::new("."),
            store.clone(),
            &genesis.config,
            epoch_manager.clone(),
        );
        apply_chain_range(
            ApplyRangeMode::Parallel,
            store,
            None,
            &genesis,
            None,
            None,
            0,
            epoch_manager.as_ref(),
            runtime,
            true,
            None,
            false,
            StorageSource::Trie,
        );
    }

    #[test]
    fn test_apply_chain_range_no_chunks() {
        let epoch_length = 4;
        let (store, genesis, mut env) = setup(epoch_length);
        let genesis_hash = *env.clients[0].chain.genesis().hash();
        let signer =
            InMemorySigner::from_seed("test1".parse().unwrap(), KeyType::ED25519, "test1").into();
        let tx = SignedTransaction::stake(
            1,
            "test1".parse().unwrap(),
            &signer,
            TESTING_INIT_STAKE,
            signer.public_key(),
            genesis_hash,
        );
        assert_eq!(env.clients[0].process_tx(tx, false, false), ProcessTxResponse::ValidTx);

        safe_produce_blocks(&mut env, 1, epoch_length * 2 + 1, Some(5));

        initialize_genesis_state(store.clone(), &genesis, None);
        let epoch_manager = EpochManager::new_arc_handle(store.clone(), &genesis.config);
        let runtime = NightshadeRuntime::test(
            Path::new("."),
            store.clone(),
            &genesis.config,
            epoch_manager.clone(),
        );
        let mut file = tempfile::NamedTempFile::new().unwrap();
        apply_chain_range(
            ApplyRangeMode::Parallel,
            store,
            None,
            &genesis,
            None,
            None,
            0,
            epoch_manager.as_ref(),
            runtime,
            true,
            Some(file.as_file_mut()),
            false,
            StorageSource::Trie,
        );
        let mut csv = String::new();
        file.as_file_mut().seek(SeekFrom::Start(0)).unwrap();
        file.as_file_mut().read_to_string(&mut csv).unwrap();
        let lines: Vec<&str> = csv.split("\n").collect();
        assert!(lines[0].contains("Height"));
        let mut has_tx = 0;
        let mut no_tx = 0;
        for line in &lines {
            if line.contains(",test0,1,0,") {
                has_tx += 1;
            }
            if line.contains(",test0,0,0,") {
                no_tx += 1;
            }
        }
        assert_eq!(has_tx, 1, "{:#?}", lines);
        assert_eq!(no_tx, 8, "{:#?}", lines);
    }
}
