//! Contract that adds keys and creates accounts

use near_sdk::borsh::{self, BorshDeserialize, BorshSerialize};
use near_sdk::{env, near_bindgen, AccountId, Balance, Promise, PublicKey};
use std::str::FromStr;

#[near_bindgen]
#[derive(Default, BorshDeserialize, BorshSerialize)]
pub struct KeyAdder {}

#[near_bindgen]
impl KeyAdder {
    pub fn add_key(&mut self, public_key: String) -> Promise {
        let signer_id = env::signer_account_id();
        if signer_id == env::current_account_id() {
            let public_key = PublicKey::from_str(&public_key).unwrap();
            Promise::new(signer_id).add_full_access_key(public_key)
        } else {
            // here we call the contract again on the signer, assuming that
            // it has been deployed. This lets us test the two cases, one where
            // we have a local receipt generated by a tx with the same signer and receiver,
            // and one where test1 calls test0, and what we want to look for on chain is
            // the receipt generated by this cross-contract call here
            Self::ext(signer_id).add_key(public_key)
        }
    }

    #[payable]
    pub fn create_account(&mut self, account_id: AccountId, public_key: Option<String>) -> Promise {
        let mut p = Promise::new(account_id).create_account();
        if let Some(public_key) = public_key {
            let public_key = PublicKey::from_str(&public_key).unwrap();
            p = p.add_full_access_key(public_key);
        }
        p.transfer(env::attached_deposit())
    }

    pub fn stake(&mut self, amount: Balance, public_key: PublicKey) -> Promise {
        Promise::new(env::current_account_id()).stake(amount, public_key)
    }
}
