/*
 Copyright 2019 Vimeo Inc.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
*/

package grpc

import (
	"context"
	"errors"

	gc "github.com/vimeo/galaxycache"
	pb "github.com/vimeo/galaxycache/galaxycachepb"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"
)

// serviceImpl implements the GalaxyCacheServer
// interface generated by the GalaxyCache pb service
type serviceImpl struct {
	pb.UnimplementedGalaxyCacheServer
	universe *gc.Universe
}

// RegisterGRPCServer registers the given grpc.Server with
// a Universe for GetFromPeer calls over RPC. In order for traces
// to propogate properly, create the grpcServer with an opencensus
// ServerHandler (https://godoc.org/go.opencensus.io/plugin/ocgrpc#example-ServerHandler)
func RegisterGRPCServer(universe *gc.Universe, grpcServer *grpc.Server) {
	pb.RegisterGalaxyCacheServer(grpcServer, &serviceImpl{universe: universe})
}

// GetFromPeer implements the generated GalaxyCacheServer
// interface, making an internal Get() after receiving a
// remote call from a peer
func (gp *serviceImpl) GetFromPeer(ctx context.Context, req *pb.GetRequest) (*pb.GetResponse, error) {
	galaxy := gp.universe.GetGalaxy(req.GetGalaxy())
	if galaxy == nil {
		return nil, status.Errorf(codes.InvalidArgument, "Unable to find galaxy [%s]", req.GetGalaxy())
	}

	galaxy.Stats.ServerRequests.Add(1) // keep track of the num of req

	var value unsafeByteCodec
	getInfo, err := galaxy.GetWithOptions(ctx, gc.GetOptions{FetchMode: gc.FetchModeNoPeerBackend}, string(req.GetKey()), &value)
	if err != nil {
		if nfErr := gc.NotFoundErr(nil); errors.As(err, &nfErr) {
			return nil, status.Errorf(codes.NotFound, "not found: [%s]: %v", req, err)
		}
		return nil, status.Errorf(status.Code(err), "Failed to retrieve [%s]: %v", req, err)
	}
	ret := pb.GetResponse_builder{Value: value}.Build()
	if !getInfo.Expiry.IsZero() {
		ret.SetExpiry(timestamppb.New(getInfo.Expiry))
	}

	return ret, nil
}

// GetFromPeer implements the generated GalaxyCacheServer
// interface, making an internal Get() after receiving a
// remote call from a peer
func (gp *serviceImpl) PeekPeer(ctx context.Context, req *pb.PeekRequest) (*pb.PeekResponse, error) {
	galaxy := gp.universe.GetGalaxy(req.GetGalaxy())
	if galaxy == nil {
		return nil, status.Errorf(codes.InvalidArgument, "Unable to find galaxy [%s]", req.GetGalaxy())
	}

	galaxy.Stats.ServerRequests.Add(1) // keep track of the num of req

	var value unsafeByteCodec
	getInfo, err := galaxy.GetWithOptions(ctx, gc.GetOptions{FetchMode: gc.FetchModePeek}, string(req.GetKey()), &value)
	if err != nil {
		if nfErr := gc.NotFoundErr(nil); errors.As(err, &nfErr) {
			return nil, status.Errorf(codes.NotFound, "not found: [%s]: %v", req, err)
		}
		return nil, status.Errorf(status.Code(err), "Failed to retrieve [%s]: %v", req, err) // not clear what this would be
	}

	ret := pb.PeekResponse_builder{Value: value}.Build()
	if !getInfo.Expiry.IsZero() {
		ret.SetExpiry(timestamppb.New(getInfo.Expiry))
	}
	return ret, nil
}
