import logging

from ddt import data, ddt, unpack
from django.contrib.auth import get_user_model
from django.test import LiveServerTestCase

from dhcp_agent import Cache, DHCPConfigManager
from ralph.data_center.tests.factories import DataCenterFactory
from ralph.dhcp.models import DHCPServer


logger = logging.getLogger(__file__)


default_params = {
    'dhcp_config_entries': None,
    'verbose': False,
    'proto': 'http',
    'log_path': 'STDOUT',
    'restart': None,
    'host': None,
    'dhcp_config_networks': None,
    'sections': ['entries'],
    'key': None,
    'dhcp_service_name': 'isc-dhcp-server',
    'dc': None,
    'net_env': None
}


@ddt
class DHCPAgentTest(LiveServerTestCase):
    def setUp(self):
        self.user = get_user_model().objects.create_superuser(
            'test', 'test@test.test', 'test'
        )
        self.default_params = default_params.copy()
        proto, host = self.live_server_url.split('://')
        self.default_params.update({
            'proto': proto,
            'host': host,
            'key': self.user.auth_token.key
        })

    @unpack
    @data(
        ({'dc': 'fake-404,fake-dc', 'env': None},),
        ({'dc': None, 'env': 'fake-env'},),
    )
    def test_get_configuration_should_return_false_when_dc_or_env_doesnt_exist(
        self, request_params
    ):
        params = self.default_params.copy()
        params.update(**request_params)
        with Cache('/tmp/') as cache:
            manager = DHCPConfigManager(
                logger=logger,
                cache=cache,
                **params
            )
            self.assertEqual(
                manager._get_configuration(mode='entries'),
                False
            )

    @unpack
    @data(
        ('entries',),
        ('networks',),
    )
    def test_get_configuration_should_return_configuration(self, mode):
        dc = DataCenterFactory()
        params = self.default_params.copy()
        params.update({
            'dc': dc.name,
        })
        with Cache('/tmp/') as cache:
            manager = DHCPConfigManager(
                logger=logger,
                cache=cache,
                **params
            )
            data = manager._get_configuration(mode=mode)
        self.assertTrue('config generated by Ralph' in data.decode())

    def test_agent_should_send_confirmation(self):
        dc = DataCenterFactory()
        dhcp_server = DHCPServer.objects.create(
            ip='127.0.0.1', last_synchronized=None
        )
        with Cache('/tmp/') as cache:
            manager = DHCPConfigManager(
                logger=logger,
                cache=cache,
                **self.default_params
            )
            manager._send_sync_confirmation()
        dhcp_server.refresh_from_db()
        self.assertTrue(dhcp_server.last_synchronized)
