export const xdgOpenScript = () =>
  '#!/bin/sh\n#---------------------------------------------\n#   xdg-open\n#\n#   Utility script to open a URL in the registered default application.\n#\n#   Refer to the usage() function below for usage.\n#\n#   Copyright 2009-2010, Fathi Boudra <fabo@freedesktop.org>\n#   Copyright 2009-2010, Rex Dieter <rdieter@fedoraproject.org>\n#   Copyright 2006, Kevin Krammer <kevin.krammer@gmx.at>\n#   Copyright 2006, Jeremy White <jwhite@codeweavers.com>\n#\n#   LICENSE:\n#\n#   Permission is hereby granted, free of charge, to any person obtaining a\n#   copy of this software and associated documentation files (the "Software"),\n#   to deal in the Software without restriction, including without limitation\n#   the rights to use, copy, modify, merge, publish, distribute, sublicense,\n#   and/or sell copies of the Software, and to permit persons to whom the\n#   Software is furnished to do so, subject to the following conditions:\n#\n#   The above copyright notice and this permission notice shall be included\n#   in all copies or substantial portions of the Software.\n#\n#   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS\n#   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\n#   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL\n#   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR\n#   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,\n#   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR\n#   OTHER DEALINGS IN THE SOFTWARE.\n#\n#---------------------------------------------\n\nmanualpage()\n{\ncat << _MANUALPAGE\nName\n\n   xdg-open -- opens a file or URL in the user\'s preferred\n   application\n\nSynopsis\n\n   xdg-open { file | URL }\n\n   xdg-open { --help | --manual | --version }\n\nDescription\n\n   xdg-open opens a file or URL in the user\'s preferred\n   application. If a URL is provided the URL will be opened in the\n   user\'s preferred web browser. If a file is provided the file\n   will be opened in the preferred application for files of that\n   type. xdg-open supports file, ftp, http and https URLs.\n\n   xdg-open is for use inside a desktop session only. It is not\n   recommended to use xdg-open as root.\n\nOptions\n\n   --help\n          Show command synopsis.\n\n   --manual\n          Show this manual page.\n\n   --version\n          Show the xdg-utils version information.\n\nExit Codes\n\n   An exit code of 0 indicates success while a non-zero exit code\n   indicates failure. The following failure codes can be returned:\n\n   1\n          Error in command line syntax.\n\n   2\n          One of the files passed on the command line did not\n          exist.\n\n   3\n          A required tool could not be found.\n\n   4\n          The action failed.\n\nSee Also\n\n   xdg-mime(1), xdg-settings(1), MIME applications associations\n   specification\n\nExamples\n\nxdg-open \'http://www.freedesktop.org/\'\n\n   Opens the freedesktop.org website in the user\'s default\n   browser.\n\nxdg-open /tmp/foobar.png\n\n   Opens the PNG image file /tmp/foobar.png in the user\'s default\n   image viewing application.\n_MANUALPAGE\n}\n\nusage()\n{\ncat << _USAGE\n   xdg-open -- opens a file or URL in the user\'s preferred\n   application\n\nSynopsis\n\n   xdg-open { file | URL }\n\n   xdg-open { --help | --manual | --version }\n\n_USAGE\n}\n\n#@xdg-utils-common@\n\n#----------------------------------------------------------------------------\n#   Common utility functions included in all XDG wrapper scripts\n#----------------------------------------------------------------------------\n\nDEBUG()\n{\n  [ -z "${XDG_UTILS_DEBUG_LEVEL}" ] && return 0;\n  [ ${XDG_UTILS_DEBUG_LEVEL} -lt $1 ] && return 0;\n  shift\n  echo "$@" >&2\n}\n\n# This handles backslashes but not quote marks.\nfirst_word()\n{\n    read first rest\n    echo "$first"\n}\n\n#-------------------------------------------------------------\n# map a binary to a .desktop file\nbinary_to_desktop_file()\n{\n    search="${XDG_DATA_HOME:-$HOME/.local/share}:${XDG_DATA_DIRS:-/usr/local/share:/usr/share}"\n    binary="`which "$1"`"\n    binary="`readlink -f "$binary"`"\n    base="`basename "$binary"`"\n    IFS=:\n    for dir in $search; do\n        unset IFS\n        [ "$dir" ] || continue\n        [ -d "$dir/applications" ] || [ -d "$dir/applnk" ] || continue\n        for file in "$dir"/applications/*.desktop "$dir"/applications/*/*.desktop "$dir"/applnk/*.desktop "$dir"/applnk/*/*.desktop; do\n            [ -r "$file" ] || continue\n            # Check to make sure it\'s worth the processing.\n            grep -q "^Exec.*$base" "$file" || continue\n            # Make sure it\'s a visible desktop file (e.g. not "preferred-web-browser.desktop").\n            grep -Eq "^(NoDisplay|Hidden)=true" "$file" && continue\n            command="`grep -E "^Exec(\\[[^]=]*])?=" "$file" | cut -d= -f 2- | first_word`"\n            command="`which "$command"`"\n            if [ x"`readlink -f "$command"`" = x"$binary" ]; then\n                # Fix any double slashes that got added path composition\n                echo "$file" | sed -e \'s,//*,/,g\'\n                return\n            fi\n        done\n    done\n}\n\n#-------------------------------------------------------------\n# map a .desktop file to a binary\ndesktop_file_to_binary()\n{\n    search="${XDG_DATA_HOME:-$HOME/.local/share}:${XDG_DATA_DIRS:-/usr/local/share:/usr/share}"\n    desktop="`basename "$1"`"\n    IFS=:\n    for dir in $search; do\n        unset IFS\n        [ "$dir" ] && [ -d "$dir/applications" ] || [ -d "$dir/applnk" ] || continue\n        # Check if desktop file contains -\n        if [ "${desktop#*-}" != "$desktop" ]; then\n            vendor=${desktop%-*}\n            app=${desktop#*-}\n            if [ -r $dir/applications/$vendor/$app ]; then\n                file_path=$dir/applications/$vendor/$app\n            elif [ -r $dir/applnk/$vendor/$app ]; then\n                file_path=$dir/applnk/$vendor/$app\n            fi\n        fi\n        if test -z "$file_path" ; then\n            for indir in "$dir"/applications/ "$dir"/applications/*/ "$dir"/applnk/ "$dir"/applnk/*/; do\n                file="$indir/$desktop"\n                if [ -r "$file" ]; then\n                    file_path=$file\n                    break\n                fi\n            done\n        fi\n        if [ -r "$file_path" ]; then\n            # Remove any arguments (%F, %f, %U, %u, etc.).\n            command="`grep -E "^Exec(\\[[^]=]*])?=" "$file_path" | cut -d= -f 2- | first_word`"\n            command="`which "$command"`"\n            readlink -f "$command"\n            return\n        fi\n    done\n}\n\n#-------------------------------------------------------------\n# Exit script on successfully completing the desired operation\n\nexit_success()\n{\n    if [ $# -gt 0 ]; then\n        echo "$@"\n        echo\n    fi\n\n    exit 0\n}\n\n\n#-----------------------------------------\n# Exit script on malformed arguments, not enough arguments\n# or missing required option.\n# prints usage information\n\nexit_failure_syntax()\n{\n    if [ $# -gt 0 ]; then\n        echo "xdg-open: $@" >&2\n        echo "Try \'xdg-open --help\' for more information." >&2\n    else\n        usage\n        echo "Use \'man xdg-open\' or \'xdg-open --manual\' for additional info."\n    fi\n\n    exit 1\n}\n\n#-------------------------------------------------------------\n# Exit script on missing file specified on command line\n\nexit_failure_file_missing()\n{\n    if [ $# -gt 0 ]; then\n        echo "xdg-open: $@" >&2\n    fi\n\n    exit 2\n}\n\n#-------------------------------------------------------------\n# Exit script on failure to locate necessary tool applications\n\nexit_failure_operation_impossible()\n{\n    if [ $# -gt 0 ]; then\n        echo "xdg-open: $@" >&2\n    fi\n\n    exit 3\n}\n\n#-------------------------------------------------------------\n# Exit script on failure returned by a tool application\n\nexit_failure_operation_failed()\n{\n    if [ $# -gt 0 ]; then\n        echo "xdg-open: $@" >&2\n    fi\n\n    exit 4\n}\n\n#------------------------------------------------------------\n# Exit script on insufficient permission to read a specified file\n\nexit_failure_file_permission_read()\n{\n    if [ $# -gt 0 ]; then\n        echo "xdg-open: $@" >&2\n    fi\n\n    exit 5\n}\n\n#------------------------------------------------------------\n# Exit script on insufficient permission to write a specified file\n\nexit_failure_file_permission_write()\n{\n    if [ $# -gt 0 ]; then\n        echo "xdg-open: $@" >&2\n    fi\n\n    exit 6\n}\n\ncheck_input_file()\n{\n    if [ ! -e "$1" ]; then\n        exit_failure_file_missing "file \'$1\' does not exist"\n    fi\n    if [ ! -r "$1" ]; then\n        exit_failure_file_permission_read "no permission to read file \'$1\'"\n    fi\n}\n\ncheck_vendor_prefix()\n{\n    file_label="$2"\n    [ -n "$file_label" ] || file_label="filename"\n    file=`basename "$1"`\n    case "$file" in\n       [[:alpha:]]*-*)\n         return\n         ;;\n    esac\n\n    echo "xdg-open: $file_label \'$file\' does not have a proper vendor prefix" >&2\n    echo \'A vendor prefix consists of alpha characters ([a-zA-Z]) and is terminated\' >&2\n    echo \'with a dash ("-"). An example \'"$file_label"\' is \'"\'example-$file\'" >&2\n    echo "Use --novendor to override or \'xdg-open --manual\' for additional info." >&2\n    exit 1\n}\n\ncheck_output_file()\n{\n    # if the file exists, check if it is writeable\n    # if it does not exists, check if we are allowed to write on the directory\n    if [ -e "$1" ]; then\n        if [ ! -w "$1" ]; then\n            exit_failure_file_permission_write "no permission to write to file \'$1\'"\n        fi\n    else\n        DIR=`dirname "$1"`\n        if [ ! -w "$DIR" ] || [ ! -x "$DIR" ]; then\n            exit_failure_file_permission_write "no permission to create file \'$1\'"\n        fi\n    fi\n}\n\n#----------------------------------------\n# Checks for shared commands, e.g. --help\n\ncheck_common_commands()\n{\n    while [ $# -gt 0 ] ; do\n        parm="$1"\n        shift\n\n        case "$parm" in\n            --help)\n            usage\n            echo "Use \'man xdg-open\' or \'xdg-open --manual\' for additional info."\n            exit_success\n            ;;\n\n            --manual)\n            manualpage\n            exit_success\n            ;;\n\n            --version)\n            echo "xdg-open 1.1.3"\n            exit_success\n            ;;\n        esac\n    done\n}\n\ncheck_common_commands "$@"\n\n[ -z "${XDG_UTILS_DEBUG_LEVEL}" ] && unset XDG_UTILS_DEBUG_LEVEL;\nif [ ${XDG_UTILS_DEBUG_LEVEL-0} -lt 1 ]; then\n    # Be silent\n    xdg_redirect_output=" > /dev/null 2> /dev/null"\nelse\n    # All output to stderr\n    xdg_redirect_output=" >&2"\nfi\n\n#--------------------------------------\n# Checks for known desktop environments\n# set variable DE to the desktop environments name, lowercase\n\ndetectDE()\n{\n    # see https://bugs.freedesktop.org/show_bug.cgi?id=34164\n    unset GREP_OPTIONS\n\n    if [ -n "${XDG_CURRENT_DESKTOP}" ]; then\n      case "${XDG_CURRENT_DESKTOP}" in\n         # only recently added to menu-spec, pre-spec X- still in use\n         Cinnamon|X-Cinnamon)\n           DE=cinnamon;\n           ;;\n         ENLIGHTENMENT)\n           DE=enlightenment;\n           ;;\n         # GNOME, GNOME-Classic:GNOME, or GNOME-Flashback:GNOME\n         GNOME*)\n           DE=gnome;\n           ;;\n         KDE)\n           DE=kde;\n           ;;\n         # Deepin Desktop Environments\n         DEEPIN|Deepin|deepin)\n           DE=dde;\n           ;;\n         LXDE)\n           DE=lxde;\n           ;;\n         LXQt)\n           DE=lxqt;\n           ;;\n         MATE)\n           DE=mate;\n           ;;\n         XFCE)\n           DE=xfce\n           ;;\n         X-Generic)\n           DE=generic\n           ;;\n      esac\n    fi\n\n    if [ x"$DE" = x"" ]; then\n      # classic fallbacks\n      if [ x"$KDE_FULL_SESSION" != x"" ]; then DE=kde;\n      elif [ x"$GNOME_DESKTOP_SESSION_ID" != x"" ]; then DE=gnome;\n      elif [ x"$MATE_DESKTOP_SESSION_ID" != x"" ]; then DE=mate;\n      elif `dbus-send --print-reply --dest=org.freedesktop.DBus /org/freedesktop/DBus org.freedesktop.DBus.GetNameOwner string:org.gnome.SessionManager > /dev/null 2>&1` ; then DE=gnome;\n      elif xprop -root _DT_SAVE_MODE 2> /dev/null | grep \' = \\"xfce4\\"$\' >/dev/null 2>&1; then DE=xfce;\n      elif xprop -root 2> /dev/null | grep -i \'^xfce_desktop_window\' >/dev/null 2>&1; then DE=xfce\n      elif echo $DESKTOP | grep -q \'^Enlightenment\'; then DE=enlightenment;\n      elif [ x"$LXQT_SESSION_CONFIG" != x"" ]; then DE=lxqt;\n      fi\n    fi\n\n    if [ x"$DE" = x"" ]; then\n      # fallback to checking $DESKTOP_SESSION\n      case "$DESKTOP_SESSION" in\n         gnome)\n           DE=gnome;\n           ;;\n         LXDE|Lubuntu)\n           DE=lxde;\n           ;;\n         MATE)\n           DE=mate;\n           ;;\n         xfce|xfce4|\'Xfce Session\')\n           DE=xfce;\n           ;;\n      esac\n    fi\n\n    if [ x"$DE" = x"" ]; then\n      # fallback to uname output for other platforms\n      case "$(uname 2>/dev/null)" in\n        CYGWIN*)\n          DE=cygwin;\n          ;;\n        Darwin)\n          DE=darwin;\n          ;;\n      esac\n    fi\n\n    if [ x"$DE" = x"gnome" ]; then\n      # gnome-default-applications-properties is only available in GNOME 2.x\n      # but not in GNOME 3.x\n      which gnome-default-applications-properties > /dev/null 2>&1  || DE="gnome3"\n    fi\n\n    if [ -f "$XDG_RUNTIME_DIR/flatpak-info" ]; then\n      DE="flatpak"\n    fi\n}\n\n#----------------------------------------------------------------------------\n# kfmclient exec/openURL can give bogus exit value in KDE <= 3.5.4\n# It also always returns 1 in KDE 3.4 and earlier\n# Simply return 0 in such case\n\nkfmclient_fix_exit_code()\n{\n    version=`LC_ALL=C.UTF-8 kde-config --version 2>/dev/null | grep \'^KDE\'`\n    major=`echo $version | sed \'s/KDE.*: \\([0-9]\\).*/\\1/\'`\n    minor=`echo $version | sed \'s/KDE.*: [0-9]*\\.\\([0-9]\\).*/\\1/\'`\n    release=`echo $version | sed \'s/KDE.*: [0-9]*\\.[0-9]*\\.\\([0-9]\\).*/\\1/\'`\n    test "$major" -gt 3 && return $1\n    test "$minor" -gt 5 && return $1\n    test "$release" -gt 4 && return $1\n    return 0\n}\n\n#----------------------------------------------------------------------------\n# Returns true if there is a graphical display attached.\n\nhas_display()\n{\n    if [ -n "$DISPLAY" ] || [ -n "$WAYLAND_DISPLAY" ]; then\n        return 0\n    else\n        return 1\n    fi\n}\n\n# This handles backslashes but not quote marks.\nlast_word()\n{\n    read first rest\n    echo "$rest"\n}\n\n# Get the value of a key in a desktop file\'s Desktop Entry group.\n# Example: Use get_key foo.desktop Exec\n# to get the values of the Exec= key for the Desktop Entry group.\nget_key()\n{\n    local file="${1}"\n    local key="${2}"\n    local desktop_entry=""\n\n    IFS_="${IFS}"\n    IFS=""\n    while read line\n    do\n        case "$line" in\n            "[Desktop Entry]")\n                desktop_entry="y"\n            ;;\n            # Reset match flag for other groups\n            "["*)\n                desktop_entry=""\n            ;;\n            "${key}="*)\n                # Only match Desktop Entry group\n                if [ -n "${desktop_entry}" ]\n                then\n                    echo "${line}" | cut -d= -f 2-\n                fi\n        esac\n    done < "${file}"\n    IFS="${IFS_}"\n}\n\n# Returns true if argument is a file:// URL or path\nis_file_url_or_path()\n{\n    if echo "$1" | grep -q \'^file://\' \\\n            || ! echo "$1" | egrep -q \'^[[:alpha:]+\\.\\-]+:\'; then\n        return 0\n    else\n        return 1\n    fi\n}\n\n# If argument is a file URL, convert it to a (percent-decoded) path.\n# If not, leave it as it is.\nfile_url_to_path()\n{\n    local file="$1"\n    if echo "$file" | grep -q \'^file:///\'; then\n        file=${file#file://}\n        file=${file%%#*}\n        file=$(echo "$file" | sed -r \'s/\\?.*$//\')\n        local printf=printf\n        if [ -x /usr/bin/printf ]; then\n            printf=/usr/bin/printf\n        fi\n        file=$($printf "$(echo "$file" | sed -e \'s@%\\([a-f0-9A-F]\\{2\\}\\)@\\\\x\\1@g\')")\n    fi\n    echo "$file"\n}\n\nopen_cygwin()\n{\n    cygstart "$1"\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_darwin()\n{\n    open "$1"\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_kde()\n{\n    if [ -n "${KDE_SESSION_VERSION}" ]; then\n      case "${KDE_SESSION_VERSION}" in\n        4)\n          kde-open "$1"\n        ;;\n        5)\n          kde-open${KDE_SESSION_VERSION} "$1"\n        ;;\n      esac\n    else\n        kfmclient exec "$1"\n        kfmclient_fix_exit_code $?\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_dde()\n{\n    if dde-open -version >/dev/null 2>&1; then\n        dde-open "$1"\n    else\n        open_generic "$1"\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_gnome3()\n{\n    if gio help open 2>/dev/null 1>&2; then\n        gio open "$1"\n    elif gvfs-open --help 2>/dev/null 1>&2; then\n        gvfs-open "$1"\n    else\n        open_generic "$1"\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_gnome()\n{\n    if gio help open 2>/dev/null 1>&2; then\n        gio open "$1"\n    elif gvfs-open --help 2>/dev/null 1>&2; then\n        gvfs-open "$1"\n    elif gnome-open --help 2>/dev/null 1>&2; then\n        gnome-open "$1"\n    else\n        open_generic "$1"\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_mate()\n{\n    if gio help open 2>/dev/null 1>&2; then\n        gio open "$1"\n    elif gvfs-open --help 2>/dev/null 1>&2; then\n        gvfs-open "$1"\n    elif mate-open --help 2>/dev/null 1>&2; then\n        mate-open "$1"\n    else\n        open_generic "$1"\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_xfce()\n{\n    if exo-open --help 2>/dev/null 1>&2; then\n        exo-open "$1"\n    elif gio help open 2>/dev/null 1>&2; then\n        gio open "$1"\n    elif gvfs-open --help 2>/dev/null 1>&2; then\n        gvfs-open "$1"\n    else\n        open_generic "$1"\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_enlightenment()\n{\n    if enlightenment_open --help 2>/dev/null 1>&2; then\n        enlightenment_open "$1"\n    else\n        open_generic "$1"\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_flatpak()\n{\n    gdbus call --session \\\n        --dest org.freedesktop.portal.Desktop \\\n        --object-path /org/freedesktop/portal/desktop \\\n        --method org.freedesktop.portal.OpenURI.OpenURI \\\n        "" "$1" {}\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\n#-----------------------------------------\n# Recursively search .desktop file\n\nsearch_desktop_file()\n{\n    local default="$1"\n    local dir="$2"\n    local target="$3"\n\n    local file=""\n    # look for both vendor-app.desktop, vendor/app.desktop\n    if [ -r "$dir/$default" ]; then\n      file="$dir/$default"\n    elif [ -r "$dir/`echo $default | sed -e \'s|-|/|\'`" ]; then\n      file="$dir/`echo $default | sed -e \'s|-|/|\'`"\n    fi\n\n    if [ -r "$file" ] ; then\n        command="$(get_key "${file}" "Exec" | first_word)"\n        command_exec=`which $command 2>/dev/null`\n        icon="$(get_key "${file}" "Icon")"\n        # FIXME: Actually LC_MESSAGES should be used as described in\n        # http://standards.freedesktop.org/desktop-entry-spec/latest/ar01s04.html\n        localised_name="$(get_key "${file}" "Name")"\n        set -- $(get_key "${file}" "Exec" | last_word)\n        # We need to replace any occurrence of "%f", "%F" and\n        # the like by the target file. We examine each\n        # argument and append the modified argument to the\n        # end then shift.\n        local args=$#\n        local replaced=0\n        while [ $args -gt 0 ]; do\n            case $1 in\n                %[c])\n                    replaced=1\n                    arg="${localised_name}"\n                    shift\n                    set -- "$@" "$arg"\n                    ;;\n                %[fFuU])\n                    replaced=1\n                    arg="$target"\n                    shift\n                    set -- "$@" "$arg"\n                    ;;\n                %[i])\n                    replaced=1\n                    shift\n                    set -- "$@" "--icon" "$icon"\n                    ;;\n                *)\n                    arg="$1"\n                    shift\n                    set -- "$@" "$arg"\n                    ;;\n            esac\n            args=$(( $args - 1 ))\n        done\n        [ $replaced -eq 1 ] || set -- "$@" "$target"\n        "$command_exec" "$@"\n\n        if [ $? -eq 0 ]; then\n            exit_success\n        fi\n    fi\n\n    for d in $dir/*/; do\n        [ -d "$d" ] && search_desktop_file "$default" "$d" "$target"\n    done\n}\n\n\nopen_generic_xdg_mime()\n{\n    filetype="$2"\n    default=`xdg-mime query default "$filetype"`\n    if [ -n "$default" ] ; then\n        xdg_user_dir="$XDG_DATA_HOME"\n        [ -n "$xdg_user_dir" ] || xdg_user_dir="$HOME/.local/share"\n\n        xdg_system_dirs="$XDG_DATA_DIRS"\n        [ -n "$xdg_system_dirs" ] || xdg_system_dirs=/usr/local/share/:/usr/share/\n\nDEBUG 3 "$xdg_user_dir:$xdg_system_dirs"\n        for x in `echo "$xdg_user_dir:$xdg_system_dirs" | sed \'s/:/ /g\'`; do\n            search_desktop_file "$default" "$x/applications/" "$1"\n        done\n    fi\n}\n\nopen_generic_xdg_file_mime()\n{\n    filetype=`xdg-mime query filetype "$1" | sed "s/;.*//"`\n    open_generic_xdg_mime "$1" "$filetype"\n}\n\nopen_generic_xdg_x_scheme_handler()\n{\n    scheme="`echo $1 | sed -n \'s/\\(^[[:alnum:]+\\.-]*\\):.*$/\\1/p\'`"\n    if [ -n $scheme ]; then\n        filetype="x-scheme-handler/$scheme"\n        open_generic_xdg_mime "$1" "$filetype"\n    fi\n}\n\nhas_single_argument()\n{\n  test $# = 1\n}\n\nopen_envvar()\n{\n    local oldifs="$IFS"\n    local browser browser_with_arg\n\n    IFS=":"\n    for browser in $BROWSER; do\n        IFS="$oldifs"\n\n        if [ -z "$browser" ]; then\n            continue\n        fi\n\n        if echo "$browser" | grep -q %s; then\n            # Avoid argument injection.\n            # See https://bugs.freedesktop.org/show_bug.cgi?id=103807\n            # URIs don\'t have IFS characters spaces anyway.\n            has_single_argument $1 && $(printf "$browser" "$1")\n        else\n            $browser "$1"\n        fi\n\n        if [ $? -eq 0 ]; then\n            exit_success\n        fi\n    done\n}\n\nopen_generic()\n{\n    if is_file_url_or_path "$1"; then\n        local file="$(file_url_to_path "$1")"\n\n        check_input_file "$file"\n\n        if has_display; then\n            filetype=`xdg-mime query filetype "$file" | sed "s/;.*//"`\n            open_generic_xdg_mime "$file" "$filetype"\n        fi\n\n        if which run-mailcap 2>/dev/null 1>&2; then\n            run-mailcap --action=view "$file"\n            if [ $? -eq 0 ]; then\n                exit_success\n            fi\n        fi\n\n        if has_display && mimeopen -v 2>/dev/null 1>&2; then\n            mimeopen -L -n "$file"\n            if [ $? -eq 0 ]; then\n                exit_success\n            fi\n        fi\n    fi\n\n    if has_display; then\n        open_generic_xdg_x_scheme_handler "$1"\n    fi\n\n    if [ -n "$BROWSER" ]; then\n        open_envvar "$1"\n    fi\n\n    # if BROWSER variable is not set, check some well known browsers instead\n    if [ x"$BROWSER" = x"" ]; then\n        BROWSER=www-browser:links2:elinks:links:lynx:w3m\n        if has_display; then\n            BROWSER=x-www-browser:firefox:iceweasel:seamonkey:mozilla:epiphany:konqueror:chromium:chromium-browser:google-chrome:microsoft-edge:$BROWSER\n        fi\n    fi\n\n    open_envvar "$1"\n\n    exit_failure_operation_impossible "no method available for opening \'$1\'"\n}\n\nopen_lxde()\n{\n\n    # pcmanfm only knows how to handle file:// urls and filepaths, it seems.\n    if pcmanfm --help >/dev/null 2>&1 && is_file_url_or_path "$1"; then\n        local file="$(file_url_to_path "$1")"\n\n        # handle relative paths\n        if ! echo "$file" | grep -q ^/; then\n            file="$(pwd)/$file"\n        fi\n\n        pcmanfm "$file"\n    else\n        open_generic "$1"\n    fi\n\n    if [ $? -eq 0 ]; then\n        exit_success\n    else\n        exit_failure_operation_failed\n    fi\n}\n\nopen_lxqt()\n{\n    open_generic "$1"\n}\n\n[ x"$1" != x"" ] || exit_failure_syntax\n\nurl=\nwhile [ $# -gt 0 ] ; do\n    parm="$1"\n    shift\n\n    case "$parm" in\n      -*)\n        exit_failure_syntax "unexpected option \'$parm\'"\n        ;;\n\n      *)\n        if [ -n "$url" ] ; then\n            exit_failure_syntax "unexpected argument \'$parm\'"\n        fi\n        url="$parm"\n        ;;\n    esac\ndone\n\nif [ -z "${url}" ] ; then\n    exit_failure_syntax "file or URL argument missing"\nfi\n\ndetectDE\n\nif [ x"$DE" = x"" ]; then\n    DE=generic\nfi\n\nDEBUG 2 "Selected DE $DE"\n\n# sanitize BROWSER (avoid caling ourselves in particular)\ncase "${BROWSER}" in\n    *:"xdg-open"|"xdg-open":*)\n        BROWSER=$(echo $BROWSER | sed -e \'s|:xdg-open||g\' -e \'s|xdg-open:||g\')\n        ;;\n    "xdg-open")\n        BROWSER=\n        ;;\nesac\n\ncase "$DE" in\n    kde)\n    open_kde "$url"\n    ;;\n\n    dde)\n    open_dde "$url"\n    ;;\n\n    gnome3|cinnamon)\n    open_gnome3 "$url"\n    ;;\n\n    gnome)\n    open_gnome "$url"\n    ;;\n\n    mate)\n    open_mate "$url"\n    ;;\n\n    xfce)\n    open_xfce "$url"\n    ;;\n\n    lxde)\n    open_lxde "$url"\n    ;;\n\n    lxqt)\n    open_lxqt "$url"\n    ;;\n\n    enlightenment)\n    open_enlightenment "$url"\n    ;;\n\n    cygwin)\n    open_cygwin "$url"\n    ;;\n\n    darwin)\n    open_darwin "$url"\n    ;;\n\n    flatpak)\n    open_flatpak "$url"\n    ;;\n\n    generic)\n    open_generic "$url"\n    ;;\n\n    *)\n    exit_failure_operation_impossible "no method available for opening \'$url\'"\n    ;;\nesac\n';
