/*******************************************************************************
* Copyright 2021 Intel Corporation
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#include <initializer_list>

#include "oneapi/dal/algo/subgraph_isomorphism/graph_matching.hpp"
#include "oneapi/dal/graph/undirected_adjacency_vector_graph.hpp"
#include "oneapi/dal/table/common.hpp"
#include "oneapi/dal/exceptions.hpp"
#include "oneapi/dal/table/row_accessor.hpp"
#include "oneapi/dal/test/engine/common.hpp"
#include "oneapi/dal/graph/service_functions.hpp"

namespace oneapi::dal::algo::subgraph_isomorphism::test {

typedef dal::preview::subgraph_isomorphism::kind isomorphism_kind;

class graph_base_data {
public:
    graph_base_data() = default;

    std::int64_t get_vertex_count() const {
        return vertex_count;
    }

    std::int64_t get_edge_count() const {
        return edge_count;
    }

    std::int64_t get_cols_count() const {
        return cols_count;
    }

    std::int64_t get_rows_count() const {
        return rows_count;
    }

protected:
    std::int64_t vertex_count;
    std::int64_t edge_count;
    std::int64_t cols_count;
    std::int64_t rows_count;
};

/*
  O------O---
 /|\    /|\  \
O | O--O-+-O--O
 \|/    \|/  /
  O------O---
*/
class double_triangle_target_type : public graph_base_data {
public:
    double_triangle_target_type() {
        vertex_count = 9;
        edge_count = 16;
        cols_count = 32;
        rows_count = 10;
    }
    std::array<std::int32_t, 9> degrees = { 2, 4, 3, 4, 4, 5, 3, 5, 2 };
    std::array<std::int32_t, 32> cols = { 1, 3, 0, 2, 5, 3, 4, 1, 3, 7, 0, 2, 1, 2, 6, 5,
                                          7, 1, 8, 7, 4, 6, 4, 7, 5, 3, 5, 6, 4, 8, 5, 7 };
    std::array<std::int64_t, 10> rows = { 0, 2, 6, 9, 13, 17, 22, 25, 30, 32 };
    std::array<std::int32_t, 9> labels = { 1, 0, 1, 0, 0, 1, 0, 1, 0 };
};

/*
  O
 / \
O---O
 \ /
  O
*/
class double_triangle_pattern_type : public graph_base_data {
public:
    double_triangle_pattern_type() {
        vertex_count = 4;
        edge_count = 5;
        cols_count = 10;
        rows_count = 5;
    }
    std::array<std::int32_t, 4> degrees = { 2, 3, 2, 3 };
    std::array<std::int32_t, 10> cols = { 1, 3, 0, 2, 3, 1, 3, 0, 2, 1 };
    std::array<std::int64_t, 5> rows = { 0, 2, 5, 7, 10 };
    std::array<std::int32_t, 4> labels = { 0, 1, 0, 1 };
};

class k_6_type : public graph_base_data {
public:
    k_6_type() {
        vertex_count = 6;
        edge_count = 15;
        cols_count = 30;
        rows_count = 7;
    }
    std::array<std::int32_t, 6> degrees = { 5, 5, 5, 5, 5, 5 };
    std::array<std::int32_t, 30> cols = { 1, 4, 3, 2, 5, 0, 2, 5, 4, 3, 4, 1, 3, 0, 5,
                                          4, 0, 2, 1, 5, 2, 0, 3, 1, 5, 1, 4, 0, 2, 3 };
    std::array<std::int64_t, 7> rows = { 0, 5, 10, 15, 20, 25, 30 };
    std::array<std::int32_t, 6> labels = { 0, 0, 0, 0, 0, 0 };
};

class k_5_without_edge_type : public graph_base_data {
public:
    k_5_without_edge_type() {
        vertex_count = 5;
        edge_count = 9;
        cols_count = 18;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 3, 4, 4, 3, 4 };
    std::array<std::int32_t, 18> cols = { 1, 4, 2, 0, 2, 4, 3, 4, 1, 3, 0, 4, 2, 1, 2, 0, 3, 1 };
    std::array<std::int64_t, 6> rows = { 0, 3, 7, 11, 14, 18 };
    std::array<std::int32_t, 5> labels = { 0, 0, 0, 0, 0 };
};

// k_10(0-9) + path_15(10-24) connected by edge 9-10
class lolipop_10_15_type : public graph_base_data {
public:
    lolipop_10_15_type() {
        vertex_count = 25;
        edge_count = 60;
        cols_count = 120;
        rows_count = 26;
    }
    std::array<std::int32_t, 25> degrees = { 9, 9, 9, 9, 9, 9, 9, 9, 9, 10, 2, 2, 2,
                                             2, 2, 2, 2, 2, 2, 2, 2, 2, 2,  2, 1 };
    std::array<std::int32_t, 120> cols = {
        2,  5,  8,  1,  7,  4,  3,  9,  6,  6,  3,  9,  0,  2,  5,  8,  4,  7,  0,  5,
        8,  4,  1,  7,  3,  9,  6,  4,  7,  1,  9,  6,  5,  8,  0,  2,  3,  9,  6,  5,
        8,  2,  0,  7,  1,  7,  0,  2,  4,  6,  9,  1,  3,  8,  4,  1,  8,  5,  3,  7,
        0,  2,  9,  3,  5,  0,  2,  9,  6,  4,  1,  8,  9,  0,  2,  6,  4,  1,  3,  5,
        7,  4,  8,  1,  3,  5,  10, 7,  0,  2,  6,  9,  11, 10, 12, 13, 11, 12, 14, 13,
        15, 14, 16, 17, 15, 16, 18, 17, 19, 18, 20, 19, 21, 22, 20, 21, 23, 22, 24, 23
    };
    std::array<std::int64_t, 26> rows = { 0,   9,   18,  27,  36,  45,  54,  63,  72,
                                          81,  91,  93,  95,  97,  99,  101, 103, 105,
                                          107, 109, 111, 113, 115, 117, 119, 120 };
    std::array<std::int32_t, 25> labels = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0,
                                            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 };
};

class path_16_type : public graph_base_data {
public:
    path_16_type() {
        vertex_count = 16;
        edge_count = 15;
        cols_count = 30;
        rows_count = 17;
    }
    std::array<std::int32_t, 16> degrees = { 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 30> cols = { 1, 0, 2,  1, 3, 4,  2,  3,  5,  4,  6,  7,  5,  6,  8,
                                          9, 7, 10, 8, 9, 11, 10, 12, 13, 11, 14, 12, 13, 15, 14 };
    std::array<std::int64_t, 17> rows = { 0,  1,  3,  5,  7,  9,  11, 13, 15,
                                          17, 19, 21, 23, 25, 27, 29, 30 };
    std::array<std::int32_t, 16> labels = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 };
};

class single_vertex_type : public graph_base_data {
public:
    single_vertex_type() {
        vertex_count = 1;
        edge_count = 0;
        cols_count = 0;
        rows_count = 2;
    }
    std::array<std::int32_t, 1> degrees = { 0 };
    std::array<std::int32_t, 0> cols = {};
    std::array<std::int64_t, 2> rows = { 0, 0 };
    std::array<std::int32_t, 1> labels = { 1 };
};

// cycle 5 with self loops on 1, 3, 4 vertexes (0-index)
class cycle_5_self_loops_type : public graph_base_data {
public:
    cycle_5_self_loops_type() {
        vertex_count = 5;
        edge_count = 8;
        cols_count = 16;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 2, 4, 2, 4, 4 };
    std::array<std::int32_t, 16> cols = { 1, 4, 0, 2, 1, 1, 1, 3, 2, 4, 3, 3, 0, 3, 4, 4 };
    std::array<std::int64_t, 6> rows = { 0, 2, 6, 8, 12, 16 };
    std::array<std::int32_t, 5> labels = { 1, 0, 1, 0, 1 };
};

// path 3 with self loops on 0, 2 vertexes (0-index)
class path_3_self_loops_type : public graph_base_data {
public:
    path_3_self_loops_type() {
        vertex_count = 3;
        edge_count = 4;
        cols_count = 8;
        rows_count = 4;
    }
    std::array<std::int32_t, 3> degrees = { 3, 2, 3 };
    std::array<std::int32_t, 8> cols = { 1, 0, 0, 0, 2, 1, 2, 2 };
    std::array<std::int64_t, 4> rows = { 0, 3, 5, 8 };
    std::array<std::int32_t, 3> labels = { 1, 0, 1 };
};

/*
  O--O---O--O
    / \ / \
O--O---*---O--O
    \ / \ /
  O--O---O--O
*/
class self_matching_type : public graph_base_data {
public:
    self_matching_type() {
        vertex_count = 12;
        edge_count = 15;
        cols_count = 30;
        rows_count = 13;
    }
    std::array<std::int32_t, 12> degrees = { 1, 4, 4, 1, 1, 4, 4, 1, 1, 4, 4, 1 };
    std::array<std::int32_t, 30> cols = { 1, 0, 2, 5,  10, 1, 3,  9, 6, 2, 5, 1,  4, 6, 9,
                                          7, 2, 5, 10, 6,  9, 10, 2, 8, 5, 9, 11, 1, 6, 10 };
    std::array<std::int64_t, 13> rows = { 0, 1, 5, 9, 10, 11, 15, 19, 20, 21, 25, 29, 30 };
    std::array<std::int32_t, 12> labels = { 0, 3, 3, 1, 2, 3, 3, 2, 0, 3, 3, 1 };
};

class wheel_11_type : public graph_base_data {
public:
    wheel_11_type() {
        vertex_count = 11;
        edge_count = 20;
        cols_count = 40;
        rows_count = 12;
    }
    std::array<std::int32_t, 11> degrees = { 10, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3 };
    std::array<std::int32_t, 40> cols = { 2, 5, 8, 1, 7, 4, 10, 3,  9, 6, 0, 2, 10, 0,
                                          1, 3, 4, 0, 2, 3, 5,  0,  0, 4, 6, 5, 7,  0,
                                          0, 6, 8, 9, 0, 7, 8,  10, 0, 9, 0, 1 };
    std::array<std::int64_t, 12> rows = { 0, 10, 13, 16, 19, 22, 25, 28, 31, 34, 37, 40 };
    std::array<std::int32_t, 11> labels = { 0, 1, 2, 3, 0, 0, 0, 0, 0, 0, 0 };
};

class cycle_10_type : public graph_base_data {
public:
    cycle_10_type() {
        vertex_count = 10;
        edge_count = 10;
        cols_count = 20;
        rows_count = 11;
    }
    std::array<std::int32_t, 10> degrees = { 2, 2, 2, 2, 2, 2, 2, 2, 2, 2 };
    std::array<std::int32_t, 20> cols = {
        1, 9, 0, 2, 1, 3, 4, 2, 3, 5, 4, 6, 7, 5, 6, 8, 9, 7, 0, 8
    };
    std::array<std::int64_t, 11> rows = { 0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20 };
    std::array<std::int32_t, 10> labels = { 1, 2, 3, 0, 0, 0, 0, 0, 0, 0 };
};

// k3(0-2), k5(3-7), k4(8-11), k4(12-13). additional edges 0-3, 3-8, 9-12, 11-13
class difficult_graph_type : public graph_base_data {
public:
    difficult_graph_type() {
        vertex_count = 16;
        edge_count = 29;
        cols_count = 58;
        rows_count = 17;
    }
    std::array<std::int32_t, 16> degrees = { 3, 2, 2, 6, 4, 4, 4, 4, 4, 4, 3, 4, 4, 4, 3, 3 };
    std::array<std::int32_t, 58> cols = { 2,  1,  3,  0,  2,  0,  1,  4,  7,  6,  5,  8,
                                          0,  3,  6,  5,  7,  7,  4,  6,  3,  4,  5,  3,
                                          7,  3,  5,  6,  4,  9,  11, 3,  10, 8,  11, 10,
                                          13, 9,  11, 8,  9,  8,  10, 12, 13, 15, 14, 11,
                                          12, 14, 9,  15, 13, 15, 12, 12, 14, 13 };
    std::array<std::int64_t, 17> rows = { 0,  3,  5,  7,  13, 17, 21, 25, 29,
                                          33, 37, 40, 44, 48, 52, 55, 58 };
    std::array<std::int32_t, 16> labels = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
};

// k3(0-2), k3(3-5), additional edge 2-3
class triangles_edge_link_type : public graph_base_data {
public:
    triangles_edge_link_type() {
        vertex_count = 6;
        edge_count = 7;
        cols_count = 14;
        rows_count = 7;
    }
    std::array<std::int32_t, 6> degrees = { 2, 2, 3, 3, 2, 2 };
    std::array<std::int32_t, 14> cols = { 1, 2, 0, 2, 1, 3, 0, 4, 2, 5, 3, 5, 4, 3 };
    std::array<std::int64_t, 7> rows = { 0, 2, 4, 7, 10, 12, 14 };
    std::array<std::int32_t, 6> labels = { 0, 0, 0, 0, 0, 0 };
};

class wheel_5_type : public graph_base_data {
public:
    wheel_5_type() {
        vertex_count = 5;
        edge_count = 8;
        cols_count = 16;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 4, 3, 3, 3, 3 };
    std::array<std::int32_t, 16> cols = { 1, 4, 3, 2, 0, 2, 4, 1, 3, 0, 4, 0, 2, 0, 3, 1 };
    std::array<std::int64_t, 6> rows = { 0, 4, 7, 10, 13, 16 };
    std::array<std::int32_t, 5> labels = { 0, 1, 1, 0, 0 };
};

class triangle_type : public graph_base_data {
public:
    triangle_type() {
        vertex_count = 3;
        edge_count = 3;
        cols_count = 6;
        rows_count = 4;
    }
    std::array<std::int32_t, 3> degrees = { 2, 2, 2 };
    std::array<std::int32_t, 6> cols = { 1, 2, 0, 2, 0, 1 };
    std::array<std::int64_t, 4> rows = { 0, 2, 4, 6 };
    std::array<std::int32_t, 3> labels = { 0, 0, 1 };
};

/*
  O       O
  |       |
O-O-O-O-O-O-O-O
*/
class paths_1_2_3_single_target_type : public graph_base_data {
public:
    paths_1_2_3_single_target_type() {
        vertex_count = 10;
        edge_count = 9;
        cols_count = 18;
        rows_count = 11;
    }
    std::array<std::int32_t, 10> degrees = { 3, 1, 1, 2, 2, 2, 1, 3, 2, 1 };
    std::array<std::int32_t, 18> cols = { 1, 3, 2, 0, 0, 4, 0, 3, 5, 7, 4, 7, 5, 6, 8, 9, 7, 8 };
    std::array<std::int64_t, 11> rows = { 0, 3, 4, 5, 7, 9, 11, 12, 15, 17, 18 };
    std::array<std::int32_t, 10> labels = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
};

/*
    O
    |
O-O-O-O-O-O
*/
class paths_1_2_3_type : public graph_base_data {
public:
    paths_1_2_3_type() {
        vertex_count = 7;
        edge_count = 6;
        cols_count = 12;
        rows_count = 8;
    }
    std::array<std::int32_t, 7> degrees = { 3, 1, 2, 1, 2, 2, 1 };
    std::array<std::int32_t, 12> cols = { 1, 4, 2, 0, 3, 0, 2, 0, 5, 4, 6, 5 };
    std::array<std::int64_t, 8> rows = { 0, 3, 4, 6, 7, 9, 11, 12 };
    std::array<std::int32_t, 7> labels = { 0, 0, 0, 0, 0, 0, 0 };
};

class cycle_100_type : public graph_base_data {
public:
    cycle_100_type() {
        vertex_count = 100;
        edge_count = 100;
        cols_count = 200;
        rows_count = 101;
    }
    std::array<std::int32_t, 100> degrees = { 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2 };
    std::array<std::int32_t, 200> cols = {
        1,  99, 0,  2,  1,  3,  2,  4,  3,  5,  4,  6,  5,  7,  6,  8,  7,  9,  8,  10, 9,  11, 10,
        12, 11, 13, 12, 14, 13, 15, 14, 16, 15, 17, 16, 18, 17, 19, 18, 20, 19, 21, 20, 22, 21, 23,
        22, 24, 23, 25, 24, 26, 25, 27, 26, 28, 27, 29, 28, 30, 29, 31, 30, 32, 31, 33, 32, 34, 33,
        35, 34, 36, 35, 37, 36, 38, 37, 39, 38, 40, 39, 41, 40, 42, 41, 43, 42, 44, 43, 45, 44, 46,
        45, 47, 46, 48, 47, 49, 48, 50, 49, 51, 50, 52, 51, 53, 52, 54, 53, 55, 54, 56, 55, 57, 56,
        58, 57, 59, 58, 60, 59, 61, 60, 62, 61, 63, 62, 64, 63, 65, 64, 66, 65, 67, 66, 68, 67, 69,
        68, 70, 69, 71, 70, 72, 71, 73, 72, 74, 73, 75, 74, 76, 75, 77, 76, 78, 77, 79, 78, 80, 79,
        81, 80, 82, 81, 83, 82, 84, 83, 85, 84, 86, 85, 87, 86, 88, 87, 89, 88, 90, 89, 91, 90, 92,
        91, 93, 92, 94, 93, 95, 94, 96, 95, 97, 96, 98, 97, 99, 98, 0
    };
    std::array<std::int64_t, 101> rows = {
        0,   2,   4,   6,   8,   10,  12,  14,  16,  18,  20,  22,  24,  26,  28,  30,  32,
        34,  36,  38,  40,  42,  44,  46,  48,  50,  52,  54,  56,  58,  60,  62,  64,  66,
        68,  70,  72,  74,  76,  78,  80,  82,  84,  86,  88,  90,  92,  94,  96,  98,  100,
        102, 104, 106, 108, 110, 112, 114, 116, 118, 120, 122, 124, 126, 128, 130, 132, 134,
        136, 138, 140, 142, 144, 146, 148, 150, 152, 154, 156, 158, 160, 162, 164, 166, 168,
        170, 172, 174, 176, 178, 180, 182, 184, 186, 188, 190, 192, 194, 196, 198, 200
    };
    std::array<std::int32_t, 100> labels = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
                                             0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1,
                                             0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
                                             1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
};

class path_10_type : public graph_base_data {
public:
    path_10_type() {
        vertex_count = 10;
        edge_count = 9;
        cols_count = 18;
        rows_count = 11;
    }
    std::array<std::int32_t, 10> degrees = { 1, 2, 2, 2, 2, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 18> cols = { 1, 0, 2, 1, 3, 2, 4, 3, 5, 4, 6, 5, 7, 6, 8, 7, 9, 8 };
    std::array<std::int64_t, 11> rows = { 0, 1, 3, 5, 7, 9, 11, 13, 15, 17, 18 };
    std::array<std::int32_t, 10> labels = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 1 };
};

class star_99_type : public graph_base_data {
public:
    star_99_type() {
        vertex_count = 100;
        edge_count = 99;
        cols_count = 198;
        rows_count = 101;
    }
    std::array<std::int32_t, 100> degrees = { 99, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                              1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                              1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                              1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                              1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                              1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };
    std::array<std::int32_t, 198> cols = {
        1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22,
        23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44,
        45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66,
        67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88,
        89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
        0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
        0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
        0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
        0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
    };
    std::array<std::int64_t, 101> rows = {
        0,   99,  100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114,
        115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129, 130, 131,
        132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148,
        149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165,
        166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182,
        183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198
    };
    std::array<std::int32_t, 100> labels = { 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                             1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                             2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                             2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3,
                                             3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
                                             3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3 };
};

class star_3_type : public graph_base_data {
public:
    star_3_type() {
        vertex_count = 4;
        edge_count = 3;
        cols_count = 6;
        rows_count = 5;
    }
    std::array<std::int32_t, 4> degrees = { 3, 1, 1, 1 };
    std::array<std::int32_t, 6> cols = { 1, 2, 3, 0, 0, 0 };
    std::array<std::int64_t, 5> rows = { 0, 3, 4, 5, 6 };
    std::array<std::int32_t, 4> labels = { 0, 1, 2, 3 };
};

class wheel_201_type : public graph_base_data {
public:
    wheel_201_type() {
        vertex_count = 201;
        edge_count = 400;
        cols_count = 800;
        rows_count = 202;
    }
    std::array<std::int32_t, 201> degrees = {
        200, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3,   3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3,   3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3,   3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3,   3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3,   3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3,   3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
    };
    std::array<std::int32_t, 800> cols = {
        1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,  14,  15,  16,  17,  18,
        19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,
        37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,
        55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,
        73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,
        91,  92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102, 103, 104, 105, 106, 107, 108,
        109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126,
        127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144,
        145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162,
        163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180,
        181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198,
        199, 200, 0,   2,   200, 0,   1,   3,   0,   2,   4,   0,   3,   5,   0,   4,   6,   0,
        5,   7,   0,   6,   8,   0,   7,   9,   0,   8,   10,  0,   9,   11,  0,   10,  12,  0,
        11,  13,  0,   12,  14,  0,   13,  15,  0,   14,  16,  0,   15,  17,  0,   16,  18,  0,
        17,  19,  0,   18,  20,  0,   19,  21,  0,   20,  22,  0,   21,  23,  0,   22,  24,  0,
        23,  25,  0,   24,  26,  0,   25,  27,  0,   26,  28,  0,   27,  29,  0,   28,  30,  0,
        29,  31,  0,   30,  32,  0,   31,  33,  0,   32,  34,  0,   33,  35,  0,   34,  36,  0,
        35,  37,  0,   36,  38,  0,   37,  39,  0,   38,  40,  0,   39,  41,  0,   40,  42,  0,
        41,  43,  0,   42,  44,  0,   43,  45,  0,   44,  46,  0,   45,  47,  0,   46,  48,  0,
        47,  49,  0,   48,  50,  0,   49,  51,  0,   50,  52,  0,   51,  53,  0,   52,  54,  0,
        53,  55,  0,   54,  56,  0,   55,  57,  0,   56,  58,  0,   57,  59,  0,   58,  60,  0,
        59,  61,  0,   60,  62,  0,   61,  63,  0,   62,  64,  0,   63,  65,  0,   64,  66,  0,
        65,  67,  0,   66,  68,  0,   67,  69,  0,   68,  70,  0,   69,  71,  0,   70,  72,  0,
        71,  73,  0,   72,  74,  0,   73,  75,  0,   74,  76,  0,   75,  77,  0,   76,  78,  0,
        77,  79,  0,   78,  80,  0,   79,  81,  0,   80,  82,  0,   81,  83,  0,   82,  84,  0,
        83,  85,  0,   84,  86,  0,   85,  87,  0,   86,  88,  0,   87,  89,  0,   88,  90,  0,
        89,  91,  0,   90,  92,  0,   91,  93,  0,   92,  94,  0,   93,  95,  0,   94,  96,  0,
        95,  97,  0,   96,  98,  0,   97,  99,  0,   98,  100, 0,   99,  101, 0,   100, 102, 0,
        101, 103, 0,   102, 104, 0,   103, 105, 0,   104, 106, 0,   105, 107, 0,   106, 108, 0,
        107, 109, 0,   108, 110, 0,   109, 111, 0,   110, 112, 0,   111, 113, 0,   112, 114, 0,
        113, 115, 0,   114, 116, 0,   115, 117, 0,   116, 118, 0,   117, 119, 0,   118, 120, 0,
        119, 121, 0,   120, 122, 0,   121, 123, 0,   122, 124, 0,   123, 125, 0,   124, 126, 0,
        125, 127, 0,   126, 128, 0,   127, 129, 0,   128, 130, 0,   129, 131, 0,   130, 132, 0,
        131, 133, 0,   132, 134, 0,   133, 135, 0,   134, 136, 0,   135, 137, 0,   136, 138, 0,
        137, 139, 0,   138, 140, 0,   139, 141, 0,   140, 142, 0,   141, 143, 0,   142, 144, 0,
        143, 145, 0,   144, 146, 0,   145, 147, 0,   146, 148, 0,   147, 149, 0,   148, 150, 0,
        149, 151, 0,   150, 152, 0,   151, 153, 0,   152, 154, 0,   153, 155, 0,   154, 156, 0,
        155, 157, 0,   156, 158, 0,   157, 159, 0,   158, 160, 0,   159, 161, 0,   160, 162, 0,
        161, 163, 0,   162, 164, 0,   163, 165, 0,   164, 166, 0,   165, 167, 0,   166, 168, 0,
        167, 169, 0,   168, 170, 0,   169, 171, 0,   170, 172, 0,   171, 173, 0,   172, 174, 0,
        173, 175, 0,   174, 176, 0,   175, 177, 0,   176, 178, 0,   177, 179, 0,   178, 180, 0,
        179, 181, 0,   180, 182, 0,   181, 183, 0,   182, 184, 0,   183, 185, 0,   184, 186, 0,
        185, 187, 0,   186, 188, 0,   187, 189, 0,   188, 190, 0,   189, 191, 0,   190, 192, 0,
        191, 193, 0,   192, 194, 0,   193, 195, 0,   194, 196, 0,   195, 197, 0,   196, 198, 0,
        197, 199, 0,   198, 200, 0,   199, 1
    };
    std::array<std::int64_t, 202> rows = {
        0,   200, 203, 206, 209, 212, 215, 218, 221, 224, 227, 230, 233, 236, 239, 242, 245,
        248, 251, 254, 257, 260, 263, 266, 269, 272, 275, 278, 281, 284, 287, 290, 293, 296,
        299, 302, 305, 308, 311, 314, 317, 320, 323, 326, 329, 332, 335, 338, 341, 344, 347,
        350, 353, 356, 359, 362, 365, 368, 371, 374, 377, 380, 383, 386, 389, 392, 395, 398,
        401, 404, 407, 410, 413, 416, 419, 422, 425, 428, 431, 434, 437, 440, 443, 446, 449,
        452, 455, 458, 461, 464, 467, 470, 473, 476, 479, 482, 485, 488, 491, 494, 497, 500,
        503, 506, 509, 512, 515, 518, 521, 524, 527, 530, 533, 536, 539, 542, 545, 548, 551,
        554, 557, 560, 563, 566, 569, 572, 575, 578, 581, 584, 587, 590, 593, 596, 599, 602,
        605, 608, 611, 614, 617, 620, 623, 626, 629, 632, 635, 638, 641, 644, 647, 650, 653,
        656, 659, 662, 665, 668, 671, 674, 677, 680, 683, 686, 689, 692, 695, 698, 701, 704,
        707, 710, 713, 716, 719, 722, 725, 728, 731, 734, 737, 740, 743, 746, 749, 752, 755,
        758, 761, 764, 767, 770, 773, 776, 779, 782, 785, 788, 791, 794, 797, 800
    };
    std::array<std::int32_t, 201> labels = {
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };
};

class path_100_type : public graph_base_data {
public:
    path_100_type() {
        vertex_count = 100;
        edge_count = 99;
        cols_count = 198;
        rows_count = 101;
    }
    std::array<std::int32_t, 100> degrees = { 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 198> cols = {
        1,  0,  2,  1,  3,  2,  4,  3,  5,  4,  6,  5,  7,  6,  8,  7,  9,  8,  10, 9,  11, 10,
        12, 11, 13, 12, 14, 13, 15, 14, 16, 15, 17, 16, 18, 17, 19, 18, 20, 19, 21, 20, 22, 21,
        23, 22, 24, 23, 25, 24, 26, 25, 27, 26, 28, 27, 29, 28, 30, 29, 31, 30, 32, 31, 33, 32,
        34, 33, 35, 34, 36, 35, 37, 36, 38, 37, 39, 38, 40, 39, 41, 40, 42, 41, 43, 42, 44, 43,
        45, 44, 46, 45, 47, 46, 48, 47, 49, 48, 50, 49, 51, 50, 52, 51, 53, 52, 54, 53, 55, 54,
        56, 55, 57, 56, 58, 57, 59, 58, 60, 59, 61, 60, 62, 61, 63, 62, 64, 63, 65, 64, 66, 65,
        67, 66, 68, 67, 69, 68, 70, 69, 71, 70, 72, 71, 73, 72, 74, 73, 75, 74, 76, 75, 77, 76,
        78, 77, 79, 78, 80, 79, 81, 80, 82, 81, 83, 82, 84, 83, 85, 84, 86, 85, 87, 86, 88, 87,
        89, 88, 90, 89, 91, 90, 92, 91, 93, 92, 94, 93, 95, 94, 96, 95, 97, 96, 98, 97, 99, 98
    };
    std::array<std::int64_t, 101> rows = {
        0,   1,   3,   5,   7,   9,   11,  13,  15,  17,  19,  21,  23,  25,  27,  29,  31,
        33,  35,  37,  39,  41,  43,  45,  47,  49,  51,  53,  55,  57,  59,  61,  63,  65,
        67,  69,  71,  73,  75,  77,  79,  81,  83,  85,  87,  89,  91,  93,  95,  97,  99,
        101, 103, 105, 107, 109, 111, 113, 115, 117, 119, 121, 123, 125, 127, 129, 131, 133,
        135, 137, 139, 141, 143, 145, 147, 149, 151, 153, 155, 157, 159, 161, 163, 165, 167,
        169, 171, 173, 175, 177, 179, 181, 183, 185, 187, 189, 191, 193, 195, 197, 198
    };
    std::array<std::int32_t, 100> labels = { 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1,
                                             0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0,
                                             0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0,
                                             0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0,
                                             1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1,
                                             0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0 };
};

class path_5_type : public graph_base_data {
public:
    path_5_type() {
        vertex_count = 5;
        edge_count = 4;
        cols_count = 8;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 1, 2, 2, 2, 1 };
    std::array<std::int32_t, 8> cols = { 1, 0, 2, 1, 3, 2, 4, 3 };
    std::array<std::int64_t, 6> rows = { 0, 1, 3, 5, 7, 8 };
    std::array<std::int32_t, 5> labels = { 1, 0, 0, 0, 1 };
};

// k_5(0-4) + path_15(5-104) connected by edge 4-5
class lolipop_5_100_type : public graph_base_data {
public:
    lolipop_5_100_type() {
        vertex_count = 105;
        edge_count = 110;
        cols_count = 220;
        rows_count = 106;
    }
    std::array<std::int32_t, 105> degrees = { 4, 4, 4, 4, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 220> cols = {
        1,  2,  3,  4,  0,  2,  3,  4,  0,  1,  3,   4,  0,   1,   2,   4,   0,   1,   2,   3,
        5,  4,  6,  5,  7,  6,  8,  7,  9,  8,  10,  9,  11,  10,  12,  11,  13,  12,  14,  13,
        15, 14, 16, 15, 17, 16, 18, 17, 19, 18, 20,  19, 21,  20,  22,  21,  23,  22,  24,  23,
        25, 24, 26, 25, 27, 26, 28, 27, 29, 28, 30,  29, 31,  30,  32,  31,  33,  32,  34,  33,
        35, 34, 36, 35, 37, 36, 38, 37, 39, 38, 40,  39, 41,  40,  42,  41,  43,  42,  44,  43,
        45, 44, 46, 45, 47, 46, 48, 47, 49, 48, 50,  49, 51,  50,  52,  51,  53,  52,  54,  53,
        55, 54, 56, 55, 57, 56, 58, 57, 59, 58, 60,  59, 61,  60,  62,  61,  63,  62,  64,  63,
        65, 64, 66, 65, 67, 66, 68, 67, 69, 68, 70,  69, 71,  70,  72,  71,  73,  72,  74,  73,
        75, 74, 76, 75, 77, 76, 78, 77, 79, 78, 80,  79, 81,  80,  82,  81,  83,  82,  84,  83,
        85, 84, 86, 85, 87, 86, 88, 87, 89, 88, 90,  89, 91,  90,  92,  91,  93,  92,  94,  93,
        95, 94, 96, 95, 97, 96, 98, 97, 99, 98, 100, 99, 101, 100, 102, 101, 103, 102, 104, 103
    };
    std::array<std::int64_t, 106> rows = {
        0,   4,   8,   12,  16,  21,  23,  25,  27,  29,  31,  33,  35,  37,  39,  41,  43,  45,
        47,  49,  51,  53,  55,  57,  59,  61,  63,  65,  67,  69,  71,  73,  75,  77,  79,  81,
        83,  85,  87,  89,  91,  93,  95,  97,  99,  101, 103, 105, 107, 109, 111, 113, 115, 117,
        119, 121, 123, 125, 127, 129, 131, 133, 135, 137, 139, 141, 143, 145, 147, 149, 151, 153,
        155, 157, 159, 161, 163, 165, 167, 169, 171, 173, 175, 177, 179, 181, 183, 185, 187, 189,
        191, 193, 195, 197, 199, 201, 203, 205, 207, 209, 211, 213, 215, 217, 219, 220
    };
    std::array<std::int32_t, 105> labels = { 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
};

class k_5_labeled_type : public graph_base_data {
public:
    k_5_labeled_type() {
        vertex_count = 5;
        edge_count = 10;
        cols_count = 20;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 4, 4, 4, 4, 4 };
    std::array<std::int32_t, 20> cols = {
        1, 4, 3, 2, 0, 2, 4, 3, 4, 1, 3, 0, 4, 0, 2, 1, 2, 0, 3, 1
    };
    std::array<std::int64_t, 6> rows = { 0, 4, 8, 12, 16, 20 };
    std::array<std::int32_t, 5> labels = { 0, 1, 0, 1, 0 };
};

/*
    O
    |
O-O-O-...100 vertexes...
*/
class paths_1_2_100_type : public graph_base_data {
public:
    paths_1_2_100_type() {
        vertex_count = 104;
        edge_count = 103;
        cols_count = 206;
        rows_count = 105;
    }
    std::array<std::int32_t, 104> degrees = { 3, 1, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 206> cols = {
        1,  2,  4,  0,  3,  0,  2,  5,  0,  4,   6,  5,   7,   6,   8,   7,   9,  8,  10, 9,  11,
        10, 12, 11, 13, 12, 14, 13, 15, 14, 16,  15, 17,  16,  18,  17,  19,  18, 20, 19, 21, 20,
        22, 21, 23, 22, 24, 23, 25, 24, 26, 25,  27, 26,  28,  27,  29,  28,  30, 29, 31, 30, 32,
        31, 33, 32, 34, 33, 35, 34, 36, 35, 37,  36, 38,  37,  39,  38,  40,  39, 41, 40, 42, 41,
        43, 42, 44, 43, 45, 44, 46, 45, 47, 46,  48, 47,  49,  48,  50,  49,  51, 50, 52, 51, 53,
        52, 54, 53, 55, 54, 56, 55, 57, 56, 58,  57, 59,  58,  60,  59,  61,  60, 62, 61, 63, 62,
        64, 63, 65, 64, 66, 65, 67, 66, 68, 67,  69, 68,  70,  69,  71,  70,  72, 71, 73, 72, 74,
        73, 75, 74, 76, 75, 77, 76, 78, 77, 79,  78, 80,  79,  81,  80,  82,  81, 83, 82, 84, 83,
        85, 84, 86, 85, 87, 86, 88, 87, 89, 88,  90, 89,  91,  90,  92,  91,  93, 92, 94, 93, 95,
        94, 96, 95, 97, 96, 98, 97, 99, 98, 100, 99, 101, 100, 102, 101, 103, 102
    };
    std::array<std::int64_t, 105> rows = {
        0,   3,   4,   6,   7,   9,   11,  13,  15,  17,  19,  21,  23,  25,  27,  29,  31,  33,
        35,  37,  39,  41,  43,  45,  47,  49,  51,  53,  55,  57,  59,  61,  63,  65,  67,  69,
        71,  73,  75,  77,  79,  81,  83,  85,  87,  89,  91,  93,  95,  97,  99,  101, 103, 105,
        107, 109, 111, 113, 115, 117, 119, 121, 123, 125, 127, 129, 131, 133, 135, 137, 139, 141,
        143, 145, 147, 149, 151, 153, 155, 157, 159, 161, 163, 165, 167, 169, 171, 173, 175, 177,
        179, 181, 183, 185, 187, 189, 191, 193, 195, 197, 199, 201, 203, 205, 206
    };
    std::array<std::int32_t, 104> labels = { 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
};

class k_5_without_edge_labeled_type : public graph_base_data {
public:
    k_5_without_edge_labeled_type() {
        vertex_count = 5;
        edge_count = 9;
        cols_count = 18;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 3, 4, 4, 3, 4 };
    std::array<std::int32_t, 18> cols = { 1, 4, 2, 0, 2, 4, 3, 4, 1, 3, 0, 4, 2, 1, 2, 0, 3, 1 };
    std::array<std::int64_t, 6> rows = { 0, 3, 7, 11, 14, 18 };
    std::array<std::int32_t, 5> labels = { 0, 1, 0, 1, 0 };
};

class cycle_5_type : public graph_base_data {
public:
    cycle_5_type() {
        vertex_count = 5;
        edge_count = 5;
        cols_count = 10;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 2, 2, 2, 2, 2 };
    std::array<std::int32_t, 10> cols = { 1, 4, 0, 2, 1, 3, 4, 2, 0, 3 };
    std::array<std::int64_t, 6> rows = { 0, 2, 4, 6, 8, 10 };
    std::array<std::int32_t, 5> labels = { 0, 0, 1, 1, 0 };
};

// cycle 5 0-2-4-1-3
class linked_star_5_type : public graph_base_data {
public:
    linked_star_5_type() {
        vertex_count = 5;
        edge_count = 5;
        cols_count = 10;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 2, 2, 2, 2, 2 };
    std::array<std::int32_t, 10> cols = { 3, 2, 4, 3, 4, 0, 0, 1, 2, 1 };
    std::array<std::int64_t, 6> rows = { 0, 2, 4, 6, 8, 10 };
    std::array<std::int32_t, 5> labels = { 0, 1, 0, 0, 1 };
};

class star_5_type : public graph_base_data {
public:
    star_5_type() {
        vertex_count = 6;
        edge_count = 5;
        cols_count = 10;
        rows_count = 7;
    }
    std::array<std::int32_t, 6> degrees = { 5, 1, 1, 1, 1, 1 };
    std::array<std::int32_t, 10> cols = { 1, 4, 3, 2, 5, 0, 0, 0, 0, 0 };
    std::array<std::int64_t, 7> rows = { 0, 5, 6, 7, 8, 9, 10 };
    std::array<std::int32_t, 6> labels = { 0, 0, 0, 0, 0, 0 };
};

class star_4_type : public graph_base_data {
public:
    star_4_type() {
        vertex_count = 5;
        edge_count = 4;
        cols_count = 8;
        rows_count = 6;
    }
    std::array<std::int32_t, 5> degrees = { 4, 1, 1, 1, 1 };
    std::array<std::int32_t, 8> cols = { 1, 2, 3, 4, 0, 0, 0, 0 };
    std::array<std::int64_t, 6> rows = { 0, 4, 5, 6, 7, 8 };
    std::array<std::int32_t, 5> labels = { 1, 0, 0, 0, 0 };
};

/*
3 cycle_80 graphs connected by edges: 0-80, 0-160, 80-160,
                                      20-100, 20-180, 100-180,
                                      40-120, 40-200, 120-200,
                                      60-140, 60-220, 140-220
*/
class connected_3_cycle_80_type : public graph_base_data {
public:
    connected_3_cycle_80_type() {
        vertex_count = 240;
        edge_count = 252;
        cols_count = 504;
        rows_count = 241;
    }
    std::array<std::int32_t, 240> degrees = {
        4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
    };
    std::array<std::int32_t, 504> cols = {
        1,   79,  80,  160, 0,   2,   1,   3,   2,   4,   3,   5,   4,   6,   5,   7,   6,   8,
        7,   9,   8,   10,  9,   11,  10,  12,  11,  13,  12,  14,  13,  15,  14,  16,  15,  17,
        16,  18,  17,  19,  18,  20,  19,  21,  100, 180, 20,  22,  21,  23,  22,  24,  23,  25,
        24,  26,  25,  27,  26,  28,  27,  29,  28,  30,  29,  31,  30,  32,  31,  33,  32,  34,
        33,  35,  34,  36,  35,  37,  36,  38,  37,  39,  38,  40,  39,  41,  120, 200, 40,  42,
        41,  43,  42,  44,  43,  45,  44,  46,  45,  47,  46,  48,  47,  49,  48,  50,  49,  51,
        50,  52,  51,  53,  52,  54,  53,  55,  54,  56,  55,  57,  56,  58,  57,  59,  58,  60,
        59,  61,  140, 220, 60,  62,  61,  63,  62,  64,  63,  65,  64,  66,  65,  67,  66,  68,
        67,  69,  68,  70,  69,  71,  70,  72,  71,  73,  72,  74,  73,  75,  74,  76,  75,  77,
        76,  78,  77,  79,  78,  0,   81,  159, 0,   160, 80,  82,  81,  83,  82,  84,  83,  85,
        84,  86,  85,  87,  86,  88,  87,  89,  88,  90,  89,  91,  90,  92,  91,  93,  92,  94,
        93,  95,  94,  96,  95,  97,  96,  98,  97,  99,  98,  100, 99,  101, 20,  180, 100, 102,
        101, 103, 102, 104, 103, 105, 104, 106, 105, 107, 106, 108, 107, 109, 108, 110, 109, 111,
        110, 112, 111, 113, 112, 114, 113, 115, 114, 116, 115, 117, 116, 118, 117, 119, 118, 120,
        119, 121, 40,  200, 120, 122, 121, 123, 122, 124, 123, 125, 124, 126, 125, 127, 126, 128,
        127, 129, 128, 130, 129, 131, 130, 132, 131, 133, 132, 134, 133, 135, 134, 136, 135, 137,
        136, 138, 137, 139, 138, 140, 139, 141, 60,  220, 140, 142, 141, 143, 142, 144, 143, 145,
        144, 146, 145, 147, 146, 148, 147, 149, 148, 150, 149, 151, 150, 152, 151, 153, 152, 154,
        153, 155, 154, 156, 155, 157, 156, 158, 157, 159, 158, 80,  161, 239, 0,   80,  160, 162,
        161, 163, 162, 164, 163, 165, 164, 166, 165, 167, 166, 168, 167, 169, 168, 170, 169, 171,
        170, 172, 171, 173, 172, 174, 173, 175, 174, 176, 175, 177, 176, 178, 177, 179, 178, 180,
        179, 181, 20,  100, 180, 182, 181, 183, 182, 184, 183, 185, 184, 186, 185, 187, 186, 188,
        187, 189, 188, 190, 189, 191, 190, 192, 191, 193, 192, 194, 193, 195, 194, 196, 195, 197,
        196, 198, 197, 199, 198, 200, 199, 201, 40,  120, 200, 202, 201, 203, 202, 204, 203, 205,
        204, 206, 205, 207, 206, 208, 207, 209, 208, 210, 209, 211, 210, 212, 211, 213, 212, 214,
        213, 215, 214, 216, 215, 217, 216, 218, 217, 219, 218, 220, 219, 221, 60,  140, 220, 222,
        221, 223, 222, 224, 223, 225, 224, 226, 225, 227, 226, 228, 227, 229, 228, 230, 229, 231,
        230, 232, 231, 233, 232, 234, 233, 235, 234, 236, 235, 237, 236, 238, 237, 239, 238, 160
    };
    std::array<std::int64_t, 241> rows = {
        0,   4,   6,   8,   10,  12,  14,  16,  18,  20,  22,  24,  26,  28,  30,  32,  34,  36,
        38,  40,  42,  46,  48,  50,  52,  54,  56,  58,  60,  62,  64,  66,  68,  70,  72,  74,
        76,  78,  80,  82,  84,  88,  90,  92,  94,  96,  98,  100, 102, 104, 106, 108, 110, 112,
        114, 116, 118, 120, 122, 124, 126, 130, 132, 134, 136, 138, 140, 142, 144, 146, 148, 150,
        152, 154, 156, 158, 160, 162, 164, 166, 168, 172, 174, 176, 178, 180, 182, 184, 186, 188,
        190, 192, 194, 196, 198, 200, 202, 204, 206, 208, 210, 214, 216, 218, 220, 222, 224, 226,
        228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 252, 256, 258, 260, 262, 264,
        266, 268, 270, 272, 274, 276, 278, 280, 282, 284, 286, 288, 290, 292, 294, 298, 300, 302,
        304, 306, 308, 310, 312, 314, 316, 318, 320, 322, 324, 326, 328, 330, 332, 334, 336, 340,
        342, 344, 346, 348, 350, 352, 354, 356, 358, 360, 362, 364, 366, 368, 370, 372, 374, 376,
        378, 382, 384, 386, 388, 390, 392, 394, 396, 398, 400, 402, 404, 406, 408, 410, 412, 414,
        416, 418, 420, 424, 426, 428, 430, 432, 434, 436, 438, 440, 442, 444, 446, 448, 450, 452,
        454, 456, 458, 460, 462, 466, 468, 470, 472, 474, 476, 478, 480, 482, 484, 486, 488, 490,
        492, 494, 496, 498, 500, 502, 504
    };
    std::array<std::int32_t, 240> labels = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
    };
};

/*
4 cycle_80 graphs connected by edges: 0-8, 0-16, 0-24, 8-16, 8-24, 16-24,
                                     2-10, 2-18, 2-26, 10-18, 10-26, 18-26,
                                     4-12, 4-20, 4-28, 12-20, 12-28, 20-28,
                                     6-14, 6-22, 6-30, 14-22, 14-30, 22-30
*/
class connected_4_cycle_80_type : public graph_base_data {
public:
    connected_4_cycle_80_type() {
        vertex_count = 320;
        edge_count = 344;
        cols_count = 688;
        rows_count = 321;
    }
    std::array<std::int32_t, 320> degrees = {
        5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
    };
    std::array<std::int32_t, 688> cols = {
        1,   79,  80,  160, 240, 0,   2,   1,   3,   2,   4,   3,   5,   4,   6,   5,   7,   6,
        8,   7,   9,   8,   10,  9,   11,  10,  12,  11,  13,  12,  14,  13,  15,  14,  16,  15,
        17,  16,  18,  17,  19,  18,  20,  19,  21,  100, 180, 260, 20,  22,  21,  23,  22,  24,
        23,  25,  24,  26,  25,  27,  26,  28,  27,  29,  28,  30,  29,  31,  30,  32,  31,  33,
        32,  34,  33,  35,  34,  36,  35,  37,  36,  38,  37,  39,  38,  40,  39,  41,  120, 200,
        280, 40,  42,  41,  43,  42,  44,  43,  45,  44,  46,  45,  47,  46,  48,  47,  49,  48,
        50,  49,  51,  50,  52,  51,  53,  52,  54,  53,  55,  54,  56,  55,  57,  56,  58,  57,
        59,  58,  60,  59,  61,  140, 220, 300, 60,  62,  61,  63,  62,  64,  63,  65,  64,  66,
        65,  67,  66,  68,  67,  69,  68,  70,  69,  71,  70,  72,  71,  73,  72,  74,  73,  75,
        74,  76,  75,  77,  76,  78,  77,  79,  78,  0,   81,  159, 0,   160, 240, 80,  82,  81,
        83,  82,  84,  83,  85,  84,  86,  85,  87,  86,  88,  87,  89,  88,  90,  89,  91,  90,
        92,  91,  93,  92,  94,  93,  95,  94,  96,  95,  97,  96,  98,  97,  99,  98,  100, 99,
        101, 20,  180, 260, 100, 102, 101, 103, 102, 104, 103, 105, 104, 106, 105, 107, 106, 108,
        107, 109, 108, 110, 109, 111, 110, 112, 111, 113, 112, 114, 113, 115, 114, 116, 115, 117,
        116, 118, 117, 119, 118, 120, 119, 121, 40,  200, 280, 120, 122, 121, 123, 122, 124, 123,
        125, 124, 126, 125, 127, 126, 128, 127, 129, 128, 130, 129, 131, 130, 132, 131, 133, 132,
        134, 133, 135, 134, 136, 135, 137, 136, 138, 137, 139, 138, 140, 139, 141, 60,  220, 300,
        140, 142, 141, 143, 142, 144, 143, 145, 144, 146, 145, 147, 146, 148, 147, 149, 148, 150,
        149, 151, 150, 152, 151, 153, 152, 154, 153, 155, 154, 156, 155, 157, 156, 158, 157, 159,
        158, 80,  161, 239, 0,   80,  240, 160, 162, 161, 163, 162, 164, 163, 165, 164, 166, 165,
        167, 166, 168, 167, 169, 168, 170, 169, 171, 170, 172, 171, 173, 172, 174, 173, 175, 174,
        176, 175, 177, 176, 178, 177, 179, 178, 180, 179, 181, 20,  100, 260, 180, 182, 181, 183,
        182, 184, 183, 185, 184, 186, 185, 187, 186, 188, 187, 189, 188, 190, 189, 191, 190, 192,
        191, 193, 192, 194, 193, 195, 194, 196, 195, 197, 196, 198, 197, 199, 198, 200, 199, 201,
        40,  120, 280, 200, 202, 201, 203, 202, 204, 203, 205, 204, 206, 205, 207, 206, 208, 207,
        209, 208, 210, 209, 211, 210, 212, 211, 213, 212, 214, 213, 215, 214, 216, 215, 217, 216,
        218, 217, 219, 218, 220, 219, 221, 60,  140, 300, 220, 222, 221, 223, 222, 224, 223, 225,
        224, 226, 225, 227, 226, 228, 227, 229, 228, 230, 229, 231, 230, 232, 231, 233, 232, 234,
        233, 235, 234, 236, 235, 237, 236, 238, 237, 239, 238, 160, 241, 319, 0,   80,  160, 240,
        242, 241, 243, 242, 244, 243, 245, 244, 246, 245, 247, 246, 248, 247, 249, 248, 250, 249,
        251, 250, 252, 251, 253, 252, 254, 253, 255, 254, 256, 255, 257, 256, 258, 257, 259, 258,
        260, 259, 261, 20,  100, 180, 260, 262, 261, 263, 262, 264, 263, 265, 264, 266, 265, 267,
        266, 268, 267, 269, 268, 270, 269, 271, 270, 272, 271, 273, 272, 274, 273, 275, 274, 276,
        275, 277, 276, 278, 277, 279, 278, 280, 279, 281, 40,  120, 200, 280, 282, 281, 283, 282,
        284, 283, 285, 284, 286, 285, 287, 286, 288, 287, 289, 288, 290, 289, 291, 290, 292, 291,
        293, 292, 294, 293, 295, 294, 296, 295, 297, 296, 298, 297, 299, 298, 300, 299, 301, 60,
        140, 220, 300, 302, 301, 303, 302, 304, 303, 305, 304, 306, 305, 307, 306, 308, 307, 309,
        308, 310, 309, 311, 310, 312, 311, 313, 312, 314, 313, 315, 314, 316, 315, 317, 316, 318,
        317, 319, 318, 240
    };
    std::array<std::int64_t, 321> rows = {
        0,   5,   7,   9,   11,  13,  15,  17,  19,  21,  23,  25,  27,  29,  31,  33,  35,  37,
        39,  41,  43,  48,  50,  52,  54,  56,  58,  60,  62,  64,  66,  68,  70,  72,  74,  76,
        78,  80,  82,  84,  86,  91,  93,  95,  97,  99,  101, 103, 105, 107, 109, 111, 113, 115,
        117, 119, 121, 123, 125, 127, 129, 134, 136, 138, 140, 142, 144, 146, 148, 150, 152, 154,
        156, 158, 160, 162, 164, 166, 168, 170, 172, 177, 179, 181, 183, 185, 187, 189, 191, 193,
        195, 197, 199, 201, 203, 205, 207, 209, 211, 213, 215, 220, 222, 224, 226, 228, 230, 232,
        234, 236, 238, 240, 242, 244, 246, 248, 250, 252, 254, 256, 258, 263, 265, 267, 269, 271,
        273, 275, 277, 279, 281, 283, 285, 287, 289, 291, 293, 295, 297, 299, 301, 306, 308, 310,
        312, 314, 316, 318, 320, 322, 324, 326, 328, 330, 332, 334, 336, 338, 340, 342, 344, 349,
        351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385,
        387, 392, 394, 396, 398, 400, 402, 404, 406, 408, 410, 412, 414, 416, 418, 420, 422, 424,
        426, 428, 430, 435, 437, 439, 441, 443, 445, 447, 449, 451, 453, 455, 457, 459, 461, 463,
        465, 467, 469, 471, 473, 478, 480, 482, 484, 486, 488, 490, 492, 494, 496, 498, 500, 502,
        504, 506, 508, 510, 512, 514, 516, 521, 523, 525, 527, 529, 531, 533, 535, 537, 539, 541,
        543, 545, 547, 549, 551, 553, 555, 557, 559, 564, 566, 568, 570, 572, 574, 576, 578, 580,
        582, 584, 586, 588, 590, 592, 594, 596, 598, 600, 602, 607, 609, 611, 613, 615, 617, 619,
        621, 623, 625, 627, 629, 631, 633, 635, 637, 639, 641, 643, 645, 650, 652, 654, 656, 658,
        660, 662, 664, 666, 668, 670, 672, 674, 676, 678, 680, 682, 684, 686, 688
    };
    std::array<std::int32_t, 320> labels = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
    };
};

class cycle01_100_type : public graph_base_data {
public:
    cycle01_100_type() {
        vertex_count = 100;
        edge_count = 100;
        cols_count = 200;
        rows_count = 101;
    }
    std::array<std::int32_t, 100> degrees = { 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                              2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2 };
    std::array<std::int32_t, 200> cols = {
        1,  99, 0,  2,  1,  3,  2,  4,  3,  5,  4,  6,  5,  7,  6,  8,  7,  9,  8,  10, 9,  11, 10,
        12, 11, 13, 12, 14, 13, 15, 14, 16, 15, 17, 16, 18, 17, 19, 18, 20, 19, 21, 20, 22, 21, 23,
        22, 24, 23, 25, 24, 26, 25, 27, 26, 28, 27, 29, 28, 30, 29, 31, 30, 32, 31, 33, 32, 34, 33,
        35, 34, 36, 35, 37, 36, 38, 37, 39, 38, 40, 39, 41, 40, 42, 41, 43, 42, 44, 43, 45, 44, 46,
        45, 47, 46, 48, 47, 49, 48, 50, 49, 51, 50, 52, 51, 53, 52, 54, 53, 55, 54, 56, 55, 57, 56,
        58, 57, 59, 58, 60, 59, 61, 60, 62, 61, 63, 62, 64, 63, 65, 64, 66, 65, 67, 66, 68, 67, 69,
        68, 70, 69, 71, 70, 72, 71, 73, 72, 74, 73, 75, 74, 76, 75, 77, 76, 78, 77, 79, 78, 80, 79,
        81, 80, 82, 81, 83, 82, 84, 83, 85, 84, 86, 85, 87, 86, 88, 87, 89, 88, 90, 89, 91, 90, 92,
        91, 93, 92, 94, 93, 95, 94, 96, 95, 97, 96, 98, 97, 99, 98, 0
    };
    std::array<std::int64_t, 101> rows = {
        0,   2,   4,   6,   8,   10,  12,  14,  16,  18,  20,  22,  24,  26,  28,  30,  32,
        34,  36,  38,  40,  42,  44,  46,  48,  50,  52,  54,  56,  58,  60,  62,  64,  66,
        68,  70,  72,  74,  76,  78,  80,  82,  84,  86,  88,  90,  92,  94,  96,  98,  100,
        102, 104, 106, 108, 110, 112, 114, 116, 118, 120, 122, 124, 126, 128, 130, 132, 134,
        136, 138, 140, 142, 144, 146, 148, 150, 152, 154, 156, 158, 160, 162, 164, 166, 168,
        170, 172, 174, 176, 178, 180, 182, 184, 186, 188, 190, 192, 194, 196, 198, 200
    };
    std::array<std::int32_t, 100> labels = { 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0,
                                             1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1,
                                             0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0,
                                             1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1,
                                             0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0,
                                             1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1 };
};

class path01_24_type : public graph_base_data {
public:
    path01_24_type() {
        vertex_count = 24;
        edge_count = 23;
        cols_count = 46;
        rows_count = 25;
    }
    std::array<std::int32_t, 24> degrees = { 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                             2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 46> cols = { 1,  0,  2,  1,  3,  2,  4,  3,  5,  4,  6,  5,
                                          7,  6,  8,  7,  9,  8,  10, 9,  11, 10, 12, 11,
                                          13, 12, 14, 13, 15, 14, 16, 15, 17, 16, 18, 17,
                                          19, 18, 20, 19, 21, 20, 22, 21, 23, 22 };
    std::array<std::int64_t, 25> rows = { 0,  1,  3,  5,  7,  9,  11, 13, 15, 17, 19, 21, 23,
                                          25, 27, 29, 31, 33, 35, 37, 39, 41, 43, 45, 46 };
    std::array<std::int32_t, 24> labels = { 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1,
                                            0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1 };
};

/*
vertex (0) and 3 cycle_80 graphs connected by edges: 0-1, 0-81, 0-161, 1-81, 1-161, 81-161,
                                                     0-21, 0-101, 0-181, 21-101, 21-181, 101-181,
                                                     0-41, 0-121, 0-201, 41-121, 41-201, 121-201,
                                                     0-61, 0-141, 0-221, 61-141, 61-221, 141-221
*/
class center_cycles_target_type : public graph_base_data {
public:
    center_cycles_target_type() {
        vertex_count = 241;
        edge_count = 264;
        cols_count = 528;
        rows_count = 242;
    }
    std::array<std::int32_t, 241> degrees = {
        12, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        5,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 5, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
    };
    std::array<std::int32_t, 528> cols = {
        1,   81,  161, 21,  101, 181, 41,  121, 201, 61,  141, 221, 2,   80,  0,   81,  161, 1,
        3,   2,   4,   3,   5,   4,   6,   5,   7,   6,   8,   7,   9,   8,   10,  9,   11,  10,
        12,  11,  13,  12,  14,  13,  15,  14,  16,  15,  17,  16,  18,  17,  19,  18,  20,  19,
        21,  20,  22,  0,   101, 181, 21,  23,  22,  24,  23,  25,  24,  26,  25,  27,  26,  28,
        27,  29,  28,  30,  29,  31,  30,  32,  31,  33,  32,  34,  33,  35,  34,  36,  35,  37,
        36,  38,  37,  39,  38,  40,  39,  41,  40,  42,  0,   121, 201, 41,  43,  42,  44,  43,
        45,  44,  46,  45,  47,  46,  48,  47,  49,  48,  50,  49,  51,  50,  52,  51,  53,  52,
        54,  53,  55,  54,  56,  55,  57,  56,  58,  57,  59,  58,  60,  59,  61,  60,  62,  0,
        141, 221, 61,  63,  62,  64,  63,  65,  64,  66,  65,  67,  66,  68,  67,  69,  68,  70,
        69,  71,  70,  72,  71,  73,  72,  74,  73,  75,  74,  76,  75,  77,  76,  78,  77,  79,
        78,  80,  79,  1,   82,  160, 0,   1,   161, 81,  83,  82,  84,  83,  85,  84,  86,  85,
        87,  86,  88,  87,  89,  88,  90,  89,  91,  90,  92,  91,  93,  92,  94,  93,  95,  94,
        96,  95,  97,  96,  98,  97,  99,  98,  100, 99,  101, 100, 102, 0,   21,  181, 101, 103,
        102, 104, 103, 105, 104, 106, 105, 107, 106, 108, 107, 109, 108, 110, 109, 111, 110, 112,
        111, 113, 112, 114, 113, 115, 114, 116, 115, 117, 116, 118, 117, 119, 118, 120, 119, 121,
        120, 122, 0,   41,  201, 121, 123, 122, 124, 123, 125, 124, 126, 125, 127, 126, 128, 127,
        129, 128, 130, 129, 131, 130, 132, 131, 133, 132, 134, 133, 135, 134, 136, 135, 137, 136,
        138, 137, 139, 138, 140, 139, 141, 140, 142, 0,   61,  221, 141, 143, 142, 144, 143, 145,
        144, 146, 145, 147, 146, 148, 147, 149, 148, 150, 149, 151, 150, 152, 151, 153, 152, 154,
        153, 155, 154, 156, 155, 157, 156, 158, 157, 159, 158, 160, 159, 81,  162, 240, 0,   1,
        81,  161, 163, 162, 164, 163, 165, 164, 166, 165, 167, 166, 168, 167, 169, 168, 170, 169,
        171, 170, 172, 171, 173, 172, 174, 173, 175, 174, 176, 175, 177, 176, 178, 177, 179, 178,
        180, 179, 181, 180, 182, 0,   21,  101, 181, 183, 182, 184, 183, 185, 184, 186, 185, 187,
        186, 188, 187, 189, 188, 190, 189, 191, 190, 192, 191, 193, 192, 194, 193, 195, 194, 196,
        195, 197, 196, 198, 197, 199, 198, 200, 199, 201, 200, 202, 0,   41,  121, 201, 203, 202,
        204, 203, 205, 204, 206, 205, 207, 206, 208, 207, 209, 208, 210, 209, 211, 210, 212, 211,
        213, 212, 214, 213, 215, 214, 216, 215, 217, 216, 218, 217, 219, 218, 220, 219, 221, 220,
        222, 0,   61,  141, 221, 223, 222, 224, 223, 225, 224, 226, 225, 227, 226, 228, 227, 229,
        228, 230, 229, 231, 230, 232, 231, 233, 232, 234, 233, 235, 234, 236, 235, 237, 236, 238,
        237, 239, 238, 240, 239, 161
    };
    std::array<std::int64_t, 242> rows = {
        0,   12,  17,  19,  21,  23,  25,  27,  29,  31,  33,  35,  37,  39,  41,  43,  45,  47,
        49,  51,  53,  55,  60,  62,  64,  66,  68,  70,  72,  74,  76,  78,  80,  82,  84,  86,
        88,  90,  92,  94,  96,  98,  103, 105, 107, 109, 111, 113, 115, 117, 119, 121, 123, 125,
        127, 129, 131, 133, 135, 137, 139, 141, 146, 148, 150, 152, 154, 156, 158, 160, 162, 164,
        166, 168, 170, 172, 174, 176, 178, 180, 182, 184, 189, 191, 193, 195, 197, 199, 201, 203,
        205, 207, 209, 211, 213, 215, 217, 219, 221, 223, 225, 227, 232, 234, 236, 238, 240, 242,
        244, 246, 248, 250, 252, 254, 256, 258, 260, 262, 264, 266, 268, 270, 275, 277, 279, 281,
        283, 285, 287, 289, 291, 293, 295, 297, 299, 301, 303, 305, 307, 309, 311, 313, 318, 320,
        322, 324, 326, 328, 330, 332, 334, 336, 338, 340, 342, 344, 346, 348, 350, 352, 354, 356,
        361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 391, 393, 395,
        397, 399, 404, 406, 408, 410, 412, 414, 416, 418, 420, 422, 424, 426, 428, 430, 432, 434,
        436, 438, 440, 442, 447, 449, 451, 453, 455, 457, 459, 461, 463, 465, 467, 469, 471, 473,
        475, 477, 479, 481, 483, 485, 490, 492, 494, 496, 498, 500, 502, 504, 506, 508, 510, 512,
        514, 516, 518, 520, 522, 524, 526, 528
    };
    std::array<std::int32_t, 241> labels = {
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
    };
};

/*
vertex (0) and 3 cycle_80 graphs connected by edges: 0-1, 0-81, 0-161,
                                                     0-21, 0-101, 0-181,
                                                     0-41, 0-121, 0-201,
                                                     0-61, 0-141, 0-221
*/
class center_cycles_pattern_type : public graph_base_data {
public:
    center_cycles_pattern_type() {
        vertex_count = 241;
        edge_count = 252;
        cols_count = 504;
        rows_count = 242;
    }
    std::array<std::int32_t, 241> degrees = {
        12, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        3,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2,  2, 2, 2, 2, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
    };
    std::array<std::int32_t, 504> cols = {
        1,   81,  161, 21,  101, 181, 41,  121, 201, 61,  141, 221, 2,   80,  0,   1,   3,   2,
        4,   3,   5,   4,   6,   5,   7,   6,   8,   7,   9,   8,   10,  9,   11,  10,  12,  11,
        13,  12,  14,  13,  15,  14,  16,  15,  17,  16,  18,  17,  19,  18,  20,  19,  21,  20,
        22,  0,   21,  23,  22,  24,  23,  25,  24,  26,  25,  27,  26,  28,  27,  29,  28,  30,
        29,  31,  30,  32,  31,  33,  32,  34,  33,  35,  34,  36,  35,  37,  36,  38,  37,  39,
        38,  40,  39,  41,  40,  42,  0,   41,  43,  42,  44,  43,  45,  44,  46,  45,  47,  46,
        48,  47,  49,  48,  50,  49,  51,  50,  52,  51,  53,  52,  54,  53,  55,  54,  56,  55,
        57,  56,  58,  57,  59,  58,  60,  59,  61,  60,  62,  0,   61,  63,  62,  64,  63,  65,
        64,  66,  65,  67,  66,  68,  67,  69,  68,  70,  69,  71,  70,  72,  71,  73,  72,  74,
        73,  75,  74,  76,  75,  77,  76,  78,  77,  79,  78,  80,  79,  1,   82,  160, 0,   81,
        83,  82,  84,  83,  85,  84,  86,  85,  87,  86,  88,  87,  89,  88,  90,  89,  91,  90,
        92,  91,  93,  92,  94,  93,  95,  94,  96,  95,  97,  96,  98,  97,  99,  98,  100, 99,
        101, 100, 102, 0,   101, 103, 102, 104, 103, 105, 104, 106, 105, 107, 106, 108, 107, 109,
        108, 110, 109, 111, 110, 112, 111, 113, 112, 114, 113, 115, 114, 116, 115, 117, 116, 118,
        117, 119, 118, 120, 119, 121, 120, 122, 0,   121, 123, 122, 124, 123, 125, 124, 126, 125,
        127, 126, 128, 127, 129, 128, 130, 129, 131, 130, 132, 131, 133, 132, 134, 133, 135, 134,
        136, 135, 137, 136, 138, 137, 139, 138, 140, 139, 141, 140, 142, 0,   141, 143, 142, 144,
        143, 145, 144, 146, 145, 147, 146, 148, 147, 149, 148, 150, 149, 151, 150, 152, 151, 153,
        152, 154, 153, 155, 154, 156, 155, 157, 156, 158, 157, 159, 158, 160, 159, 81,  162, 240,
        0,   161, 163, 162, 164, 163, 165, 164, 166, 165, 167, 166, 168, 167, 169, 168, 170, 169,
        171, 170, 172, 171, 173, 172, 174, 173, 175, 174, 176, 175, 177, 176, 178, 177, 179, 178,
        180, 179, 181, 180, 182, 0,   181, 183, 182, 184, 183, 185, 184, 186, 185, 187, 186, 188,
        187, 189, 188, 190, 189, 191, 190, 192, 191, 193, 192, 194, 193, 195, 194, 196, 195, 197,
        196, 198, 197, 199, 198, 200, 199, 201, 200, 202, 0,   201, 203, 202, 204, 203, 205, 204,
        206, 205, 207, 206, 208, 207, 209, 208, 210, 209, 211, 210, 212, 211, 213, 212, 214, 213,
        215, 214, 216, 215, 217, 216, 218, 217, 219, 218, 220, 219, 221, 220, 222, 0,   221, 223,
        222, 224, 223, 225, 224, 226, 225, 227, 226, 228, 227, 229, 228, 230, 229, 231, 230, 232,
        231, 233, 232, 234, 233, 235, 234, 236, 235, 237, 236, 238, 237, 239, 238, 240, 239, 161
    };
    std::array<std::int64_t, 242> rows = {
        0,   12,  15,  17,  19,  21,  23,  25,  27,  29,  31,  33,  35,  37,  39,  41,  43,  45,
        47,  49,  51,  53,  56,  58,  60,  62,  64,  66,  68,  70,  72,  74,  76,  78,  80,  82,
        84,  86,  88,  90,  92,  94,  97,  99,  101, 103, 105, 107, 109, 111, 113, 115, 117, 119,
        121, 123, 125, 127, 129, 131, 133, 135, 138, 140, 142, 144, 146, 148, 150, 152, 154, 156,
        158, 160, 162, 164, 166, 168, 170, 172, 174, 176, 179, 181, 183, 185, 187, 189, 191, 193,
        195, 197, 199, 201, 203, 205, 207, 209, 211, 213, 215, 217, 220, 222, 224, 226, 228, 230,
        232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 252, 254, 256, 258, 261, 263, 265, 267,
        269, 271, 273, 275, 277, 279, 281, 283, 285, 287, 289, 291, 293, 295, 297, 299, 302, 304,
        306, 308, 310, 312, 314, 316, 318, 320, 322, 324, 326, 328, 330, 332, 334, 336, 338, 340,
        343, 345, 347, 349, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377,
        379, 381, 384, 386, 388, 390, 392, 394, 396, 398, 400, 402, 404, 406, 408, 410, 412, 414,
        416, 418, 420, 422, 425, 427, 429, 431, 433, 435, 437, 439, 441, 443, 445, 447, 449, 451,
        453, 455, 457, 459, 461, 463, 466, 468, 470, 472, 474, 476, 478, 480, 482, 484, 486, 488,
        490, 492, 494, 496, 498, 500, 502, 504
    };
    std::array<std::int32_t, 241> labels = {
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
    };
};

// k3(0-2), k5(3-7), k4(8-11), k4(12-13). additional edges 0-3, 3-8, 9-12, 11-13
class difficult_graph_labeled_type : public graph_base_data {
public:
    difficult_graph_labeled_type() {
        vertex_count = 16;
        edge_count = 29;
        cols_count = 58;
        rows_count = 17;
    }
    std::array<std::int32_t, 16> degrees = { 3, 2, 2, 6, 4, 4, 4, 4, 4, 4, 3, 4, 4, 4, 3, 3 };
    std::array<std::int32_t, 58> cols = { 2,  1,  3,  0,  2,  0,  1,  4,  7,  6,  5,  8,
                                          0,  3,  6,  5,  7,  7,  4,  6,  3,  4,  5,  3,
                                          7,  3,  5,  6,  4,  9,  11, 3,  10, 8,  11, 10,
                                          13, 9,  11, 8,  9,  8,  10, 12, 13, 15, 14, 11,
                                          12, 14, 9,  15, 13, 15, 12, 12, 14, 13 };
    std::array<std::int64_t, 17> rows = { 0,  3,  5,  7,  13, 17, 21, 25, 29,
                                          33, 37, 40, 44, 48, 52, 55, 58 };
    std::array<std::int32_t, 16> labels = { 0, 1, 2, 0, 1, 2, 1, 2, 2, 0, 1, 0, 0, 0, 1, 2 };
};

// k3(0-2), k3(3-5), additional edge 2-3
class triangles_edge_link_labeled_type : public graph_base_data {
public:
    triangles_edge_link_labeled_type() {
        vertex_count = 6;
        edge_count = 7;
        cols_count = 14;
        rows_count = 7;
    }
    std::array<std::int32_t, 6> degrees = { 2, 2, 3, 3, 2, 2 };
    std::array<std::int32_t, 14> cols = { 1, 2, 0, 2, 1, 3, 0, 4, 2, 5, 3, 5, 4, 3 };
    std::array<std::int64_t, 7> rows = { 0, 2, 4, 7, 10, 12, 14 };
    std::array<std::int32_t, 6> labels = { 1, 2, 0, 0, 2, 1 };
};

class path_4_type : public graph_base_data {
public:
    path_4_type() {
        vertex_count = 4;
        edge_count = 3;
        cols_count = 6;
        rows_count = 5;
    }
    std::array<std::int32_t, 4> degrees = { 1, 2, 2, 1 };
    std::array<std::int32_t, 6> cols = { 1, 0, 2, 1, 3, 2 };
    std::array<std::int64_t, 5> rows = { 0, 1, 3, 5, 6 };
    std::array<std::int32_t, 4> labels = { 0, 3, 3, 3 };
};

class wheel_11_labeled_type : public graph_base_data {
public:
    wheel_11_labeled_type() {
        vertex_count = 11;
        edge_count = 20;
        cols_count = 40;
        rows_count = 12;
    }
    std::array<std::int32_t, 11> degrees = { 10, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3 };
    std::array<std::int32_t, 40> cols = { 2, 5, 8, 1, 7, 4, 10, 3,  9, 6, 0, 2, 10, 0,
                                          1, 3, 4, 0, 2, 3, 5,  0,  0, 4, 6, 5, 7,  0,
                                          0, 6, 8, 9, 0, 7, 8,  10, 0, 9, 0, 1 };
    std::array<std::int64_t, 12> rows = { 0, 10, 13, 16, 19, 22, 25, 28, 31, 34, 37, 40 };
    std::array<std::int32_t, 11> labels = { 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };
};

class cycle_4_type : public graph_base_data {
public:
    cycle_4_type() {
        vertex_count = 4;
        edge_count = 4;
        cols_count = 8;
        rows_count = 5;
    }
    std::array<std::int32_t, 4> degrees = { 2, 2, 2, 2 };
    std::array<std::int32_t, 8> cols = { 1, 3, 0, 2, 1, 3, 2, 0 };
    std::array<std::int64_t, 5> rows = { 0, 2, 4, 6, 8 };
    std::array<std::int32_t, 4> labels = { 0, 1, 1, 1 };
};

class k_6_labeled_type : public graph_base_data {
public:
    k_6_labeled_type() {
        vertex_count = 6;
        edge_count = 15;
        cols_count = 30;
        rows_count = 7;
    }
    std::array<std::int32_t, 6> degrees = { 5, 5, 5, 5, 5, 5 };
    std::array<std::int32_t, 30> cols = { 1, 4, 3, 2, 5, 0, 2, 5, 4, 3, 4, 1, 3, 0, 5,
                                          4, 0, 2, 1, 5, 2, 0, 3, 1, 5, 1, 4, 0, 2, 3 };
    std::array<std::int64_t, 7> rows = { 0, 5, 10, 15, 20, 25, 30 };
    std::array<std::int32_t, 6> labels = { 0, 0, 0, 0, 1, 1 };
};

/*
        O         O
       / \       / \
O--O--O---O--O--O---O--O
*/
class triangle_path_target_type : public graph_base_data {
public:
    triangle_path_target_type() {
        vertex_count = 10;
        edge_count = 11;
        cols_count = 22;
        rows_count = 11;
    }
    std::array<std::int32_t, 10> degrees = { 1, 3, 2, 3, 2, 3, 2, 3, 2, 1 };
    std::array<std::int32_t, 22> cols = { 1, 0, 2, 3, 1, 3, 1, 2, 4, 3, 5,
                                          4, 6, 7, 5, 7, 5, 6, 8, 7, 9, 8 };
    std::array<std::int64_t, 11> rows = { 0, 1, 4, 6, 9, 11, 14, 16, 19, 21, 22 };
    std::array<std::int32_t, 10> labels = { 1, 0, 0, 1, 0, 0, 1, 0, 0, 1 };
};

/*
                     O
                    / \
O--O--O--O---O--O--O---O--O
*/
class triangle_path_pattern_type : public graph_base_data {
public:
    triangle_path_pattern_type() {
        vertex_count = 10;
        edge_count = 10;
        cols_count = 20;
        rows_count = 11;
    }
    std::array<std::int32_t, 10> degrees = { 1, 2, 2, 2, 2, 2, 3, 2, 3, 1 };
    std::array<std::int32_t, 20> cols = {
        1, 0, 2, 1, 3, 2, 4, 3, 5, 4, 6, 5, 7, 8, 6, 8, 6, 7, 9, 8
    };
    std::array<std::int64_t, 11> rows = { 0, 1, 3, 5, 7, 9, 11, 14, 16, 19, 20 };
    std::array<std::int32_t, 10> labels = { 1, 0, 0, 0, 0, 0, 0, 1, 0, 1 };
};

/*
     /--O--\
   /O   |   O\
  O   \ O /   O
 /  \   |   /  \
 O    O-O-O    O
/ \  / \|/ \  / \
O---O---O---O---O
\ /  \ /|\ /  \ /
 O    O-O-O    O
 \  /   |   \  /
  O   / O \   O
   \O   |   O/
     \--O--/
*/
class tree_1_3_25_target_type : public graph_base_data {
public:
    tree_1_3_25_target_type() {
        vertex_count = 25;
        edge_count = 48;
        cols_count = 96;
        rows_count = 26;
    }
    std::array<std::int32_t, 25> degrees = { 8, 6, 3, 3, 3, 4, 3, 6, 3, 3, 3, 4, 3,
                                             6, 3, 3, 3, 4, 3, 6, 3, 3, 3, 4, 3 };
    std::array<std::int32_t, 96> cols = { 1,  5,  7,  11, 13, 17, 19, 23, 0,  2,  3,  4,  5,  23,
                                          1,  3,  24, 1,  2,  4,  1,  3,  6,  0,  6,  1,  7,  5,
                                          4,  8,  0,  8,  9,  10, 5,  11, 7,  6,  9,  7,  8,  10,
                                          7,  9,  12, 0,  12, 7,  13, 11, 10, 14, 0,  14, 15, 16,
                                          11, 17, 13, 12, 15, 13, 14, 16, 13, 15, 18, 0,  18, 13,
                                          19, 17, 16, 20, 0,  20, 21, 22, 17, 23, 19, 18, 21, 19,
                                          20, 22, 19, 21, 24, 0,  24, 19, 1,  23, 22, 2 };
    std::array<std::int64_t, 26> rows = { 0,  8,  14, 17, 20, 23, 27, 30, 36, 39, 42, 45, 49,
                                          52, 58, 61, 64, 67, 71, 74, 80, 83, 86, 89, 93, 96 };
    std::array<std::int32_t, 25> labels = { 0, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2,
                                            3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4 };
};

/*
        O
    O   |   O
  O   \ O /   O
    \   |   /
 O    O O O    O
  \    \|/    /
O---O---O---O---O
  /    /|\    \
 O    O O O    O
    /   |   \
  O   / O \   O
    O   |   O
        O
*/
class tree_1_3_25_pattern_type : public graph_base_data {
public:
    tree_1_3_25_pattern_type() {
        vertex_count = 25;
        edge_count = 24;
        cols_count = 48;
        rows_count = 26;
    }
    std::array<std::int32_t, 25> degrees = { 8, 4, 1, 1, 1, 2, 1, 4, 1, 1, 1, 2, 1,
                                             4, 1, 1, 1, 2, 1, 4, 1, 1, 1, 2, 1 };
    std::array<std::int32_t, 48> cols = { 1,  5,  7,  11, 13, 17, 19, 23, 0,  2,  3,  4,
                                          1,  1,  1,  0,  6,  5,  0,  8,  9,  10, 7,  7,
                                          7,  0,  12, 11, 0,  14, 15, 16, 13, 13, 13, 0,
                                          18, 17, 0,  20, 21, 22, 19, 19, 19, 0,  24, 23 };
    std::array<std::int64_t, 26> rows = { 0,  8,  12, 13, 14, 15, 17, 18, 22, 23, 24, 25, 27,
                                          28, 32, 33, 34, 35, 37, 38, 42, 43, 44, 45, 47, 48 };
    std::array<std::int32_t, 25> labels = { 0, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2,
                                            3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4 };
};

class k_4_type : public graph_base_data {
public:
    k_4_type() {
        vertex_count = 4;
        edge_count = 4;
        cols_count = 8;
        rows_count = 5;
    }
    std::array<std::int32_t, 4> degrees = { 2, 2, 2, 2 };
    std::array<std::int32_t, 8> cols = { 1, 3, 0, 2, 1, 3, 2, 0 };
    std::array<std::int64_t, 5> rows = { 0, 2, 4, 6, 8 };
    std::array<std::int32_t, 4> labels = { 0, 1, 1, 1 };
};

/*
      O
      |
O--O--O--O--O--O--O--O
*/
class paths_1_2_5_type : public graph_base_data {
public:
    paths_1_2_5_type() {
        vertex_count = 9;
        edge_count = 8;
        cols_count = 16;
        rows_count = 10;
    }
    std::array<std::int32_t, 9> degrees = { 3, 1, 2, 1, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 16> cols = { 1, 2, 4, 0, 3, 0, 2, 5, 0, 4, 6, 5, 7, 6, 8, 7 };
    std::array<std::int64_t, 10> rows = { 0, 3, 4, 6, 7, 9, 11, 13, 15, 16 };
    std::array<std::int32_t, 9> labels = { 1, 1, 1, 0, 0, 0, 0, 0, 0 };
};

class path_25_type : public graph_base_data {
public:
    path_25_type() {
        vertex_count = 25;
        edge_count = 24;
        cols_count = 48;
        rows_count = 26;
    }
    std::array<std::int32_t, 25> degrees = { 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                                             2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1 };
    std::array<std::int32_t, 48> cols = { 1,  0,  2,  1,  3,  2,  4,  3,  5,  4,  6,  5,
                                          7,  6,  8,  7,  9,  8,  10, 9,  11, 10, 12, 11,
                                          13, 12, 14, 13, 15, 14, 16, 15, 17, 16, 18, 17,
                                          19, 18, 20, 19, 21, 20, 22, 21, 23, 22, 24, 23 };
    std::array<std::int64_t, 26> rows = { 0,  1,  3,  5,  7,  9,  11, 13, 15, 17, 19, 21, 23,
                                          25, 27, 29, 31, 33, 35, 37, 39, 41, 43, 45, 47, 48 };
    std::array<std::int32_t, 25> labels = { 2, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                                            0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0 };
};

class k_20_labeled_type : public graph_base_data {
public:
    k_20_labeled_type() {
        vertex_count = 20;
        edge_count = 190;
        cols_count = 380;
        rows_count = 21;
    }
    std::array<std::int32_t, 20> degrees = { 19, 19, 19, 19, 19, 19, 19, 19, 19, 19,
                                             19, 19, 19, 19, 19, 19, 19, 19, 19, 19 };
    std::array<std::int32_t, 380> cols = {
        1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  2,  3,  4,
        5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  3,  4,  5,  6,  7,  8,
        9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  4,  5,  6,  7,  8,  9,  10, 11, 12,
        13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16,
        17, 18, 19, 0,  1,  2,  3,  4,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,
        1,  2,  3,  4,  5,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,
        5,  6,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  9,
        10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  10, 11, 12, 13,
        14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  11, 12, 13, 14, 15, 16, 17,
        18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,
        2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,
        6,  7,  8,  9,  10, 11, 12, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,
        10, 11, 12, 13, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13,
        14, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 17, 18,
        19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 18, 19, 0,  1,  2,
        3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 19, 0,  1,  2,  3,  4,  5,  6,
        7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18
    };
    std::array<std::int64_t, 21> rows = { 0,   19,  38,  57,  76,  95,  114, 133, 152, 171, 190,
                                          209, 228, 247, 266, 285, 304, 323, 342, 361, 380 };
    std::array<std::int32_t, 20> labels = { 0, 0, 0, 1, 1,  1,  2,  3,  4,  5,
                                            6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };
};

class k_20_without_edge_labeled_type : public graph_base_data {
public:
    k_20_without_edge_labeled_type() {
        vertex_count = 20;
        edge_count = 189;
        cols_count = 378;
        rows_count = 21;
    }
    std::array<std::int32_t, 20> degrees = { 18, 18, 19, 19, 19, 19, 19, 19, 19, 19,
                                             19, 19, 19, 19, 19, 19, 19, 19, 19, 19 };
    std::array<std::int32_t, 378> cols = {
        2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 2,  3,  4,  5,  6,
        7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  3,  4,  5,  6,  7,  8,  9,  10,
        11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14,
        15, 16, 17, 18, 19, 0,  1,  2,  3,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18,
        19, 0,  1,  2,  3,  4,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,
        3,  4,  5,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,
        8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  9,  10, 11,
        12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  10, 11, 12, 13, 14, 15,
        16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  11, 12, 13, 14, 15, 16, 17, 18, 19,
        0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 12, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,
        4,  5,  6,  7,  8,  9,  10, 11, 13, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,
        8,  9,  10, 11, 12, 14, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11,
        12, 13, 15, 16, 17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 16,
        17, 18, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 17, 18, 19, 0,
        1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 18, 19, 0,  1,  2,  3,  4,
        5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 19, 0,  1,  2,  3,  4,  5,  6,  7,  8,
        9,  10, 11, 12, 13, 14, 15, 16, 17, 18
    };
    std::array<std::int64_t, 21> rows = { 0,   18,  36,  55,  74,  93,  112, 131, 150, 169, 188,
                                          207, 226, 245, 264, 283, 302, 321, 340, 359, 378 };
    std::array<std::int32_t, 20> labels = { 0, 0, 0, 1, 1,  1,  2,  3,  4,  5,
                                            6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };
};

class vertexes_300_edge_1_labeled_type : public graph_base_data {
public:
    vertexes_300_edge_1_labeled_type() {
        vertex_count = 300;
        edge_count = 1;
        cols_count = 2;
        rows_count = 301;
    }
    std::array<std::int32_t, 300> degrees = {
        1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };
    std::array<std::int32_t, 2> cols = { 1, 0 };
    std::array<std::int64_t, 301> rows = {
        0, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
        2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
    };
    std::array<std::int32_t, 300> labels = {
        0,   0,   0,   0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,  14,
        15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,
        33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,
        51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,
        69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,  85,  86,
        87,  88,  89,  90,  91,  92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102, 103, 104,
        105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122,
        123, 124, 125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140,
        141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158,
        159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176,
        177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194,
        195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212,
        213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230,
        231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248,
        249, 250, 251, 252, 253, 254, 255, 256, 257, 258, 259, 260, 261, 262, 263, 264, 265, 266,
        267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284,
        285, 286, 287, 288, 289, 290, 291, 292, 293, 294, 295, 296
    };
};

/*
O-\          /-O
O--O        O--O
O-/ \      / \-O
O-\  \    /  /-O
   O--O--O--O
O-/  /    \  \-O
O-\ /      \ /-O
O--O        O--O
O-/          \-O
*/
class tree_pattern_type : public graph_base_data {
public:
    tree_pattern_type() {
        vertex_count = 24;
        edge_count = 23;
        cols_count = 46;
        rows_count = 25;
    }
    std::array<std::int32_t, 24> degrees = { 4, 4, 3, 4, 1, 1, 1, 1, 1, 1, 1, 1,
                                             4, 4, 3, 4, 1, 1, 1, 1, 1, 1, 1, 1 };
    std::array<std::int32_t, 46> cols = { 1,  2,  3,  12, 0,  4,  5,  6,  0,  7,  8,  0,
                                          9,  10, 11, 1,  1,  1,  2,  2,  3,  3,  3,  13,
                                          14, 15, 0,  12, 16, 17, 18, 12, 19, 20, 12, 21,
                                          22, 23, 13, 13, 13, 14, 14, 15, 15, 15 };
    std::array<std::int64_t, 25> rows = { 0,  4,  8,  11, 15, 16, 17, 18, 19, 20, 21, 22, 23,
                                          27, 31, 34, 38, 39, 40, 41, 42, 43, 44, 45, 46 };
    std::array<std::int32_t, 24> labels = { 0, 0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 1,
                                            1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 1 };
};

template <class T>
struct LimitedAllocator {
    typedef T value_type;
    typedef T *pointer;

    bool is_limited;
    std::size_t max_allocation_size;

    LimitedAllocator(bool is_limited = false, std::size_t max_allocation_size = 0)
            : is_limited(is_limited),
              max_allocation_size(max_allocation_size) {}

    template <class U>
    LimitedAllocator(const LimitedAllocator<U> &other) noexcept {
        is_limited = other.is_limited;
        max_allocation_size = other.max_allocation_size;
    }

    template <class U>
    bool operator==(const LimitedAllocator<U> &) const noexcept {
        return true;
    }

    template <class U>
    bool operator!=(const LimitedAllocator<U> &) const noexcept {
        return false;
    }

    T *allocate(const std::size_t n) const {
        if (n == 0 || (is_limited && max_allocation_size < n)) {
            return nullptr;
        }
        if (n > static_cast<std::size_t>(-1) / sizeof(T)) {
            throw std::bad_array_new_length();
        }
        void *const pv = malloc(n * sizeof(T));
        if (!pv) {
            throw std::bad_alloc();
        }
        return static_cast<T *>(pv);
    }

    void deallocate(T *const p, std::size_t n) const noexcept {
        free(p);
    }
};

class subgraph_isomorphism_test {
public:
    using my_graph_type = dal::preview::undirected_adjacency_vector_graph<std::int32_t>;

    template <typename GraphType>
    auto create_graph() {
        GraphType graph_data;
        my_graph_type my_graph;

        auto &graph_impl = oneapi::dal::detail::get_impl(my_graph);
        auto &vertex_allocator = graph_impl._vertex_allocator;
        auto &edge_allocator = graph_impl._edge_allocator;

        const std::int64_t vertex_count = graph_data.get_vertex_count();
        const std::int64_t edge_count = graph_data.get_edge_count();
        const std::int64_t cols_count = graph_data.get_cols_count();
        const std::int64_t rows_count = graph_data.get_rows_count();

        typedef std::allocator_traits<std::allocator<char>>::rebind_traits<std::int32_t>
            int32_traits_t;
        typedef std::allocator_traits<std::allocator<char>>::rebind_traits<std::int64_t>
            int64_traits_t;
        std::int32_t *degrees = int32_traits_t::allocate(vertex_allocator, vertex_count);
        std::int32_t *cols = int32_traits_t::allocate(vertex_allocator, cols_count);
        std::int64_t *rows = int64_traits_t::allocate(edge_allocator, rows_count);
        std::int32_t *rows_vertex = int32_traits_t::allocate(vertex_allocator, rows_count);

        for (int i = 0; i < vertex_count; i++) {
            degrees[i] = graph_data.degrees[i];
        }

        for (int i = 0; i < cols_count; i++) {
            cols[i] = graph_data.cols[i];
        }
        for (int i = 0; i < rows_count; i++) {
            rows[i] = graph_data.rows[i];
            rows_vertex[i] = graph_data.rows[i];
        }
        graph_impl.set_topology(vertex_count, edge_count, rows, cols, cols_count, degrees);
        graph_impl.get_topology()._rows_vertex =
            oneapi::dal::preview::detail::container<std::int32_t>::wrap(rows_vertex, rows_count);
        return my_graph;
    }

    template <typename GraphType>
    auto create_graph_with_vertex_labels() {
        GraphType graph_data;
        auto graph = create_graph<GraphType>();
        auto &graph_impl = oneapi::dal::detail::get_impl(graph);
        auto &vertex_allocator = graph_impl._vertex_allocator;
        auto &vv_p = graph_impl.get_vertex_values();

        auto vertex_count = graph_data.get_vertex_count();

        std::int32_t *labels_array =
            oneapi::dal::preview::detail::allocate(vertex_allocator, vertex_count);
        vv_p = oneapi::dal::array<std::int32_t>::wrap(labels_array, vertex_count);
        for (int i = 0; i < vertex_count; i++) {
            labels_array[i] = graph_data.labels[i];
        }
        return graph;
    }

    template <typename GraphType>
    std::vector<std::pair<std::int32_t, std::int32_t>> build_edgelist() {
        GraphType graph_data;
        std::vector<std::pair<std::int32_t, std::int32_t>> edgelist;
        edgelist.reserve(graph_data.get_cols_count());
        for (std::int32_t current_vertex = 0; current_vertex < graph_data.get_rows_count() - 1;
             ++current_vertex) {
            for (std::int64_t edge_index = graph_data.rows[current_vertex];
                 edge_index < graph_data.rows[current_vertex + 1];
                 ++edge_index) {
                edgelist.emplace_back(current_vertex, graph_data.cols[edge_index]);
            }
        }
        return edgelist;
    }

    bool check_isomorphism(
        const std::vector<std::int32_t> &permutation,
        const std::vector<std::pair<std::int32_t, std::int32_t>> &target_edgelist,
        const std::vector<std::pair<std::int32_t, std::int32_t>> &pattern_edgelist,
        bool is_induced) {
        std::map<std::int32_t, std::int32_t> reverse_permutation;
        for (std::size_t pattern_index = 0; pattern_index < permutation.size(); ++pattern_index) {
            reverse_permutation[permutation[pattern_index]] = pattern_index;
        }
        std::vector<std::pair<std::int32_t, std::int32_t>> subgraph_edgelist;
        for (const auto &edge : target_edgelist) {
            if (reverse_permutation.find(edge.first) != reverse_permutation.end() &&
                reverse_permutation.find(edge.second) != reverse_permutation.end()) {
                subgraph_edgelist.emplace_back(reverse_permutation[edge.first],
                                               reverse_permutation[edge.second]);
            }
        }
        std::sort(subgraph_edgelist.begin(), subgraph_edgelist.end());
        if (is_induced) {
            return subgraph_edgelist == pattern_edgelist;
        }
        for (const auto &edge : pattern_edgelist) {
            auto iterator =
                std::lower_bound(subgraph_edgelist.begin(), subgraph_edgelist.end(), edge);
            if (iterator == subgraph_edgelist.end() || *iterator != edge) {
                return false;
            }
        }
        return true;
    }

    template <typename TargetGraphType, typename PatternGraphType>
    bool check_isomorphism_result(const oneapi::dal::table &table,
                                  bool is_induced,
                                  bool is_vertex_labeled) {
        if (!table.has_data())
            return true;
        auto table_data = oneapi::dal::row_accessor<const int>(table).pull();
        const auto x = table_data.get_data();

        TargetGraphType target_graph_data;
        PatternGraphType pattern_graph_data;
        std::vector<std::pair<std::int32_t, std::int32_t>> target_edgelist =
            build_edgelist<TargetGraphType>();
        std::vector<std::pair<std::int32_t, std::int32_t>> pattern_edgelist =
            build_edgelist<PatternGraphType>();
        std::sort(pattern_edgelist.begin(), pattern_edgelist.end());

        for (std::int64_t i = 0; i < table.get_row_count(); i++) {
            std::vector<std::int32_t> permutation(table.get_column_count());
            for (std::int64_t j = 0; j < table.get_column_count(); j++) {
                permutation[j] = x[i * table.get_column_count() + j];
            }
            if (is_vertex_labeled) {
                for (std::size_t pattern_vertex = 0; pattern_vertex < permutation.size();
                     ++pattern_vertex) {
                    if (target_graph_data.labels[permutation[pattern_vertex]] !=
                        pattern_graph_data.labels[pattern_vertex]) {
                        return false;
                    }
                }
            }
            if (!check_isomorphism(permutation, target_edgelist, pattern_edgelist, is_induced)) {
                return false;
            }
        }
        return true;
    }

    template <typename TargetGraphType, typename PatternGraphType>
    void check_subgraph_isomorphism(bool semantic_match,
                                    isomorphism_kind kind,
                                    std::int64_t max_match_count,
                                    std::int64_t expected_match_count,
                                    bool is_vertex_labeled = false) {
        check_subgraph_isomorphism<TargetGraphType, PatternGraphType>(semantic_match,
                                                                      kind,
                                                                      max_match_count,
                                                                      expected_match_count,
                                                                      is_vertex_labeled,
                                                                      std::allocator<char>());
    }

    template <typename TargetGraphType, typename PatternGraphType, typename AllocatorType>
    void check_subgraph_isomorphism(bool semantic_match,
                                    isomorphism_kind kind,
                                    std::int64_t max_match_count,
                                    std::int64_t expected_match_count,
                                    bool is_vertex_labeled,
                                    AllocatorType alloc) {
        TargetGraphType target_graph_data;
        PatternGraphType pattern_graph_data;
        const auto target_graph = is_vertex_labeled
                                      ? create_graph_with_vertex_labels<TargetGraphType>()
                                      : create_graph<TargetGraphType>();
        const auto pattern_graph = is_vertex_labeled
                                       ? create_graph_with_vertex_labels<PatternGraphType>()
                                       : create_graph<PatternGraphType>();

        const auto subgraph_isomorphism_desc =
            dal::preview::subgraph_isomorphism::descriptor<
                float,
                dal::preview::subgraph_isomorphism::method::by_default,
                dal::preview::subgraph_isomorphism::task::by_default,
                AllocatorType>(alloc)
                .set_kind(kind)
                .set_max_match_count(max_match_count)
                .set_semantic_match(semantic_match);

        const auto result =
            dal::preview::graph_matching(subgraph_isomorphism_desc, target_graph, pattern_graph);
        REQUIRE(expected_match_count == result.get_match_count());
        REQUIRE(check_isomorphism_result<TargetGraphType, PatternGraphType>(
            result.get_vertex_match(),
            kind == isomorphism_kind::induced,
            is_vertex_labeled));
    }
};

#define SUBGRAPH_ISOMORPHISM_INDUCED_TEST(name) \
    TEST_M(subgraph_isomorphism_test, name, "[subgraph_isomorphism][induced]")

#define SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(name) \
    TEST_M(subgraph_isomorphism_test, name, "[subgraph_isomorphism][non-induced]")

#define SUBGRAPH_ISOMORPHISM_ALLOCATOR_TEST(name) \
    TEST_M(subgraph_isomorphism_test, name, "[subgraph_isomorphism][allocator]")

SUBGRAPH_ISOMORPHISM_INDUCED_TEST("Induced: Bit target representation, all matches check") {
    this->check_subgraph_isomorphism<double_triangle_target_type, double_triangle_pattern_type>(
        false,
        isomorphism_kind::induced,
        0,
        12);
    this->check_subgraph_isomorphism<lolipop_10_15_type, path_16_type>(false,
                                                                       isomorphism_kind::induced,
                                                                       0,
                                                                       20);
}

// SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
//     "Induced: Bit target representation, single vertex pattern check") {
//     this->check_subgraph_isomorphism<k_6_type, single_vertex_type>(false,
//                                                                    isomorphism_kind::induced,
//                                                                    0,
//                                                                    12);
// }

SUBGRAPH_ISOMORPHISM_INDUCED_TEST("Induced: Bit target representation, self loops check") {
    this->check_subgraph_isomorphism<cycle_5_self_loops_type, path_3_self_loops_type>(
        false,
        isomorphism_kind::induced,
        0,
        4);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced: Bit target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<self_matching_type, self_matching_type>(
        false,
        isomorphism_kind::induced,
        100,
        72);
    this->check_subgraph_isomorphism<wheel_11_type, cycle_10_type>(false,
                                                                   isomorphism_kind::induced,
                                                                   21,
                                                                   20);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced: Bit target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<difficult_graph_type, triangles_edge_link_type>(
        false,
        isomorphism_kind::induced,
        50,
        50);
    this->check_subgraph_isomorphism<wheel_5_type, triangle_type>(false,
                                                                  isomorphism_kind::induced,
                                                                  10,
                                                                  10);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST("Induced: Bit target representation, single match") {
    this->check_subgraph_isomorphism<paths_1_2_3_single_target_type, paths_1_2_3_type>(
        false,
        isomorphism_kind::induced,
        1,
        1);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST("Induced: Bit target representation, no matches") {
    this->check_subgraph_isomorphism<k_6_type, k_5_without_edge_type>(false,
                                                                      isomorphism_kind::induced,
                                                                      1,
                                                                      0);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced: Bit target representation, vertex count > number of threads & edge count > number of threads") {
    this->check_subgraph_isomorphism<connected_4_cycle_80_type, connected_3_cycle_80_type>(
        false,
        isomorphism_kind::induced,
        0,
        192);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST("Induced: List target representation, all matches check") {
    this->check_subgraph_isomorphism<cycle_100_type, path_10_type>(false,
                                                                   isomorphism_kind::induced,
                                                                   0,
                                                                   200);
    this->check_subgraph_isomorphism<star_99_type, star_3_type>(false,
                                                                isomorphism_kind::induced,
                                                                0,
                                                                941094);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced: List target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<wheel_201_type, triangle_type>(false,
                                                                    isomorphism_kind::induced,
                                                                    1201,
                                                                    1200);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced: List target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<path_100_type, path_5_type>(false,
                                                                 isomorphism_kind::induced,
                                                                 100,
                                                                 100);
    this->check_subgraph_isomorphism<lolipop_5_100_type, k_5_labeled_type>(
        false,
        isomorphism_kind::induced,
        10,
        10);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST("Induced: List target representation, single match") {
    this->check_subgraph_isomorphism<paths_1_2_100_type, paths_1_2_3_type>(
        false,
        isomorphism_kind::induced,
        2,
        1);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST("Induced: List target representation, no matches") {
    this->check_subgraph_isomorphism<cycle_100_type, triangle_type>(false,
                                                                    isomorphism_kind::induced,
                                                                    1,
                                                                    0);
    this->check_subgraph_isomorphism<lolipop_5_100_type, k_5_without_edge_labeled_type>(
        false,
        isomorphism_kind::induced,
        0,
        0);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced: List target representation, vertex count > number of threads") {
    this->check_subgraph_isomorphism<cycle01_100_type, path01_24_type>(false,
                                                                       isomorphism_kind::induced,
                                                                       0,
                                                                       200);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced: List target representation, vertex count > number of threads & edge count > number of threads") {
    this->check_subgraph_isomorphism<cycle01_100_type, cycle01_100_type>(false,
                                                                         isomorphism_kind::induced,
                                                                         0,
                                                                         200);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, all matches check") {
    this->check_subgraph_isomorphism<cycle_5_type, linked_star_5_type>(false,
                                                                       isomorphism_kind::induced,
                                                                       0,
                                                                       2,
                                                                       true);
    this->check_subgraph_isomorphism<self_matching_type, self_matching_type>(
        false,
        isomorphism_kind::induced,
        0,
        4,
        true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<double_triangle_target_type, double_triangle_pattern_type>(
        false,
        isomorphism_kind::induced,
        10,
        8,
        true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<wheel_5_type, triangle_type>(false,
                                                                  isomorphism_kind::induced,
                                                                  3,
                                                                  3,
                                                                  true);
    this->check_subgraph_isomorphism<lolipop_10_15_type, path_16_type>(false,
                                                                       isomorphism_kind::induced,
                                                                       1,
                                                                       1,
                                                                       true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, single match") {
    this->check_subgraph_isomorphism<wheel_11_type, cycle_10_type>(false,
                                                                   isomorphism_kind::induced,
                                                                   10,
                                                                   1,
                                                                   true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, no matches") {
    this->check_subgraph_isomorphism<star_5_type, star_4_type>(false,
                                                               isomorphism_kind::induced,
                                                               12,
                                                               0,
                                                               true);
    // this->check_subgraph_isomorphism<k_6_type, single_vertex_type>(false,
    //                                                                isomorphism_kind::induced,
    //                                                                1,
    //                                                                0,
    //                                                                true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, edge count > number of threads") {
    this->check_subgraph_isomorphism<k_20_labeled_type, k_20_labeled_type>(
        false,
        isomorphism_kind::induced,
        0,
        36,
        true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, vertex count > number of threads") {
    this->check_subgraph_isomorphism<vertexes_300_edge_1_labeled_type,
                                     vertexes_300_edge_1_labeled_type>(false,
                                                                       isomorphism_kind::induced,
                                                                       0,
                                                                       4,
                                                                       true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: Bit target representation, vertex count > number of threads & edge count > number of threads") {
    this->check_subgraph_isomorphism<connected_4_cycle_80_type, connected_3_cycle_80_type>(
        false,
        isomorphism_kind::induced,
        0,
        16,
        true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: List target representation, all matches check") {
    this->check_subgraph_isomorphism<cycle_100_type, path_10_type>(false,
                                                                   isomorphism_kind::induced,
                                                                   0,
                                                                   8,
                                                                   true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: List target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<path_100_type, path_5_type>(false,
                                                                 isomorphism_kind::induced,
                                                                 50,
                                                                 48,
                                                                 true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: List target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<star_99_type, star_3_type>(false,
                                                                isomorphism_kind::induced,
                                                                128,
                                                                128,
                                                                true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: List target representation, single match") {
    this->check_subgraph_isomorphism<wheel_201_type, triangle_type>(false,
                                                                    isomorphism_kind::induced,
                                                                    1,
                                                                    1,
                                                                    true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: List target representation, no matches") {
    this->check_subgraph_isomorphism<paths_1_2_100_type, paths_1_2_3_type>(
        false,
        isomorphism_kind::induced,
        1,
        0,
        true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: List target representation, vertex count > number of threads") {
    this->check_subgraph_isomorphism<cycle01_100_type, path01_24_type>(false,
                                                                       isomorphism_kind::induced,
                                                                       0,
                                                                       100,
                                                                       true);
}

SUBGRAPH_ISOMORPHISM_INDUCED_TEST(
    "Induced + Labeled vertexes: List target representation, vertex count > number of threads & edge count > number of threads") {
    this->check_subgraph_isomorphism<cycle01_100_type, cycle01_100_type>(false,
                                                                         isomorphism_kind::induced,
                                                                         0,
                                                                         100,
                                                                         true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST("Non-induced: Bit target representation, all matches check") {
    this->check_subgraph_isomorphism<cycle_5_self_loops_type, path_3_self_loops_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        4,
        false);
    this->check_subgraph_isomorphism<difficult_graph_labeled_type,
                                     triangles_edge_link_labeled_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        336,
        false);
    this->check_subgraph_isomorphism<double_triangle_target_type, double_triangle_pattern_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        36,
        false);
    this->check_subgraph_isomorphism<self_matching_type, path_4_type>(false,
                                                                      isomorphism_kind::non_induced,
                                                                      0,
                                                                      162,
                                                                      false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced: Bit target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<wheel_11_labeled_type, cycle_4_type>(
        false,
        isomorphism_kind::non_induced,
        81,
        80,
        false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced: Bit target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<k_6_labeled_type, k_5_without_edge_labeled_type>(
        false,
        isomorphism_kind::non_induced,
        50,
        50,
        false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST("Non-induced: Bit target representation, single match") {
    this->check_subgraph_isomorphism<triangle_path_target_type, triangle_path_pattern_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        1,
        false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST("Non-induced: Bit target representation, no matches") {
    this->check_subgraph_isomorphism<tree_1_3_25_pattern_type, tree_1_3_25_target_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        0,
        false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced: List target representation, vertex count > number of threads & edge count > number of threads") {
    this->check_subgraph_isomorphism<lolipop_5_100_type, lolipop_5_100_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        24,
        false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced: List target representation, all matches check") {
    this->check_subgraph_isomorphism<wheel_201_type, k_4_type>(false,
                                                               isomorphism_kind::non_induced,
                                                               0,
                                                               1600,
                                                               false);
    this->check_subgraph_isomorphism<lolipop_5_100_type, k_4_type>(false,
                                                                   isomorphism_kind::non_induced,
                                                                   0,
                                                                   120,
                                                                   false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced: List target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<lolipop_5_100_type, paths_1_2_5_type>(
        false,
        isomorphism_kind::non_induced,
        50,
        48,
        false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced: List target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<wheel_201_type, star_3_type>(false,
                                                                  isomorphism_kind::non_induced,
                                                                  100,
                                                                  100,
                                                                  false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST("Non-induced: List target representation, no matches") {
    this->check_subgraph_isomorphism<wheel_201_type, tree_pattern_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        0,
        false);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: Bit target representation, vertex count > number of threads") {
    this->check_subgraph_isomorphism<vertexes_300_edge_1_labeled_type,
                                     vertexes_300_edge_1_labeled_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        4,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: Bit target representation, edge count > number of threads") {
    this->check_subgraph_isomorphism<k_20_labeled_type, k_20_without_edge_labeled_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        36,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: Bit target representation, vertex count > number of threads & edge count > number of threads") {
    this->check_subgraph_isomorphism<center_cycles_target_type, center_cycles_pattern_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        512,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: Bit target representation, all matches check") {
    this->check_subgraph_isomorphism<difficult_graph_labeled_type,
                                     triangles_edge_link_labeled_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        12,
        true);
    this->check_subgraph_isomorphism<double_triangle_target_type, double_triangle_pattern_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        12,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: Bit target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<self_matching_type, path_4_type>(false,
                                                                      isomorphism_kind::non_induced,
                                                                      20,
                                                                      12,
                                                                      true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: Bit target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<wheel_11_labeled_type, cycle_4_type>(
        false,
        isomorphism_kind::non_induced,
        19,
        19,
        true);
    this->check_subgraph_isomorphism<k_6_labeled_type, k_5_without_edge_labeled_type>(
        false,
        isomorphism_kind::non_induced,
        30,
        30,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: Bit target representation, no matches") {
    this->check_subgraph_isomorphism<cycle_5_self_loops_type, path_3_self_loops_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        0,
        true);
    this->check_subgraph_isomorphism<triangle_path_target_type, triangle_path_pattern_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        0,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: List target representation, vertex count > number of threads") {
    this->check_subgraph_isomorphism<wheel_201_type, path_25_type>(false,
                                                                   isomorphism_kind::non_induced,
                                                                   0,
                                                                   3,
                                                                   true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: List target representation, vertex count > number of threads & edge count > number of threads") {
    this->check_subgraph_isomorphism<lolipop_5_100_type, lolipop_5_100_type>(
        false,
        isomorphism_kind::non_induced,
        0,
        4,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: List target representation, all matches check") {
    this->check_subgraph_isomorphism<lolipop_5_100_type, k_4_type>(false,
                                                                   isomorphism_kind::non_induced,
                                                                   0,
                                                                   12,
                                                                   true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: List target representation, max_match_count > total number of SI") {
    this->check_subgraph_isomorphism<lolipop_5_100_type, paths_1_2_5_type>(
        false,
        isomorphism_kind::non_induced,
        10,
        4,
        true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: List target representation, max_match_count <= total number of SI") {
    this->check_subgraph_isomorphism<wheel_201_type, k_4_type>(false,
                                                               isomorphism_kind::non_induced,
                                                               1,
                                                               1,
                                                               true);
}

SUBGRAPH_ISOMORPHISM_NON_INDUCED_TEST(
    "Non-induced + labels on vertexes: LList target representation, no matches") {
    this->check_subgraph_isomorphism<wheel_201_type, star_3_type>(false,
                                                                  isomorphism_kind::non_induced,
                                                                  0,
                                                                  0,
                                                                  true);
}

SUBGRAPH_ISOMORPHISM_ALLOCATOR_TEST("Custom allocator, positive case") {
    this->check_subgraph_isomorphism<lolipop_5_100_type, paths_1_2_5_type>(
        false,
        isomorphism_kind::non_induced,
        10,
        4,
        true,
        LimitedAllocator<char>());
}

SUBGRAPH_ISOMORPHISM_ALLOCATOR_TEST("Custom allocator with limit") {
    REQUIRE_THROWS_AS((this->check_subgraph_isomorphism<lolipop_5_100_type, paths_1_2_5_type>(
                          false,
                          isomorphism_kind::non_induced,
                          10,
                          4,
                          true,
                          LimitedAllocator<char>(true, 10))),
                      std::bad_alloc);
}

} // namespace oneapi::dal::algo::subgraph_isomorphism::test
