#!/bin/bash

function install {
  # create service directory
  mkdir -p /srv/ntopng

  # create yml(s)
  {
    echo "services:"
    echo "  ntopng:"
    echo "    image: jonbackhaus/ntopng"
    echo "    ports:"
    echo "      - \"8084:3000\""
    echo "    volumes:"
    echo "      - \"/var/run/docker.sock:/var/run/docker.sock\""
    echo "version: \"2\""
  } > /srv/ntopng/ntopng.yml

  # add autorun
  {
    echo "ntopng_autorun=true"
    echo
    echo "if [ \"\$ntopng_autorun\" = true ]; then"
    echo "  treehouses services ntopng up"
    echo "fi"
    echo
    echo
  } > /srv/ntopng/autorun
}

# environment var
function uses_env {
  echo false
}

# add supported arch(es)
function supported_arches {
  echo "armv7l"
}

# add port(s)
function get_ports {
  echo "8084"
}

# add size (in MB)
function get_size {
  echo "400"
}

# add description
function get_description {
  echo "Ntopng is a network traffic probe that monitors network usage"
}

# add info
function get_info {
  echo "https://github.com/ntop/ntopng"
  echo
  echo "\"ntopng is the next generation version of the original ntop,"
  echo "a network traffic probe that monitors network usage. ntopng is"
  echo "based on libpcap and it has been written in a portable way in order"
  echo "to virtually run on every Unix platform, MacOSX and on Windows as well."
  echo "Educational users can obtain commercial products at no cost please see here:"
  echo "https://www.ntop.org/support/faq/do-you-charge-universities-no-profit-and-research/\""
}

# add svg icon
function get_icon {
  cat <<EOF
<svg id="svg" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="400" height="400.9876543209877" viewBox="0, 0, 400,400.9876543209877">
  <g id="svgg">
    <path id="path0" d="M67.000 0.397 C 60.131 0.638,54.529 1.319,54.000 1.977 C 53.890 2.114,52.810 2.468,51.600 2.763 C 50.390 3.058,48.645 3.637,47.722 4.050 C 46.799 4.462,45.945 4.800,45.824 4.800 C 45.507 4.800,39.513 7.528,38.800 7.997 C 38.470 8.214,37.480 8.778,36.600 9.250 C 35.720 9.721,34.910 10.238,34.800 10.398 C 34.690 10.558,34.015 10.988,33.300 11.354 C 32.585 11.721,32.000 12.152,32.000 12.313 C 32.000 12.473,31.447 12.850,30.770 13.149 C 30.094 13.448,29.059 14.106,28.470 14.611 C 27.882 15.116,26.860 15.991,26.200 16.555 C 24.019 18.422,19.429 23.006,17.600 25.145 C 16.610 26.303,15.523 27.561,15.185 27.940 C 14.847 28.319,14.133 29.343,13.598 30.215 C 13.063 31.087,12.305 32.149,11.913 32.575 C 11.521 33.002,11.200 33.488,11.200 33.655 C 11.200 33.823,10.660 34.760,10.000 35.738 C 9.340 36.716,8.800 37.693,8.800 37.909 C 8.800 38.125,8.560 38.501,8.267 38.744 C 7.974 38.987,7.397 40.179,6.984 41.393 C 6.572 42.607,6.120 43.600,5.980 43.600 C 5.841 43.600,5.392 44.725,4.983 46.100 C 4.573 47.475,4.124 48.690,3.985 48.800 C 3.662 49.055,2.400 54.109,2.400 55.149 C 2.400 55.580,2.231 55.993,2.025 56.067 C -0.806 57.073,-0.869 343.522,1.961 345.600 C 2.110 345.710,2.556 347.195,2.951 348.900 C 3.345 350.605,3.819 352.000,4.002 352.000 C 4.186 352.000,4.462 352.571,4.615 353.268 C 4.768 353.966,5.590 355.901,6.442 357.568 C 7.293 359.236,7.992 360.825,7.995 361.100 C 7.998 361.375,8.168 361.600,8.373 361.600 C 8.578 361.600,8.871 362.005,9.024 362.500 C 9.176 362.995,9.908 364.234,10.651 365.254 C 11.393 366.273,12.000 367.289,12.000 367.511 C 12.000 367.733,12.270 368.018,12.600 368.145 C 12.930 368.271,13.200 368.580,13.200 368.831 C 13.200 369.973,22.623 380.221,26.000 382.751 C 26.770 383.328,27.635 384.115,27.923 384.500 C 28.210 384.885,28.711 385.200,29.035 385.200 C 29.360 385.200,29.720 385.447,29.836 385.750 C 29.952 386.052,30.666 386.633,31.423 387.041 C 32.181 387.448,32.800 387.896,32.800 388.036 C 32.800 388.176,33.700 388.717,34.800 389.239 C 35.900 389.762,36.800 390.299,36.800 390.434 C 36.800 390.568,37.565 391.018,38.500 391.432 C 39.435 391.847,40.438 392.414,40.729 392.693 C 41.019 392.972,41.424 393.200,41.628 393.200 C 41.831 393.200,43.141 393.740,44.538 394.400 C 45.935 395.060,47.213 395.600,47.378 395.600 C 47.544 395.600,48.246 395.862,48.940 396.182 C 50.474 396.891,56.228 398.400,57.395 398.400 C 57.860 398.400,58.411 398.563,58.620 398.762 C 59.833 399.916,69.434 400.000,199.716 400.000 C 324.826 399.999,339.336 399.890,341.600 398.931 C 341.930 398.792,344.162 398.267,346.560 397.766 C 348.957 397.264,351.027 396.698,351.160 396.508 C 351.292 396.317,352.525 395.827,353.900 395.417 C 355.275 395.008,356.400 394.522,356.400 394.337 C 356.400 394.152,356.655 394.000,356.966 394.000 C 357.529 394.000,363.619 390.918,364.000 390.440 C 364.110 390.302,365.055 389.758,366.100 389.232 C 367.145 388.705,368.000 388.135,368.000 387.964 C 368.000 387.793,368.328 387.550,368.729 387.422 C 369.131 387.295,369.808 386.878,370.234 386.495 C 370.660 386.113,371.906 385.083,373.004 384.207 C 377.273 380.801,382.815 375.270,385.625 371.610 C 386.381 370.625,387.229 369.573,387.509 369.272 C 387.789 368.971,388.310 368.156,388.665 367.462 C 389.021 366.768,389.473 366.096,389.670 365.968 C 390.114 365.680,391.346 363.579,392.466 361.200 C 392.932 360.210,393.440 359.310,393.593 359.200 C 393.747 359.090,394.206 358.100,394.613 357.000 C 395.020 355.900,395.566 354.640,395.827 354.200 C 396.088 353.760,396.628 352.140,397.026 350.600 C 397.425 349.060,397.873 347.710,398.023 347.600 C 399.806 346.287,399.778 348.317,399.913 208.600 C 400.047 69.178,399.915 58.611,398.024 57.200 C 397.877 57.090,397.433 55.470,397.039 53.600 C 396.645 51.730,396.092 49.750,395.810 49.200 C 395.528 48.650,394.707 46.760,393.984 45.000 C 392.771 42.044,390.146 36.867,389.623 36.400 C 389.500 36.290,388.988 35.358,388.486 34.329 C 387.984 33.300,387.444 32.400,387.286 32.329 C 387.129 32.258,386.257 31.115,385.349 29.789 C 381.915 24.775,373.155 15.978,369.000 13.372 C 368.340 12.958,367.656 12.390,367.480 12.110 C 367.304 11.829,366.977 11.600,366.752 11.600 C 366.528 11.600,365.592 11.085,364.672 10.455 C 363.752 9.825,362.595 9.183,362.100 9.028 C 361.605 8.873,361.200 8.590,361.200 8.400 C 361.200 8.210,360.795 7.930,360.300 7.778 C 359.805 7.626,358.320 7.003,357.000 6.393 C 355.680 5.784,353.790 5.050,352.800 4.762 C 351.810 4.474,350.910 4.117,350.800 3.967 C 350.690 3.818,349.250 3.388,347.600 3.011 C 345.950 2.635,344.240 2.162,343.800 1.961 C 340.042 0.244,340.273 0.247,203.800 0.187 C 132.740 0.156,71.180 0.251,67.000 0.397 M315.200 10.195 C 331.234 10.278,335.459 10.598,342.600 12.272 C 357.020 15.651,369.466 24.673,379.106 38.732 C 383.983 45.845,388.400 57.711,389.175 65.780 C 390.344 77.961,390.856 120.478,389.845 121.488 C 389.528 121.805,383.765 118.246,369.000 108.612 C 366.580 107.033,363.816 105.244,362.857 104.637 L 361.114 103.532 361.457 100.866 C 366.545 61.311,321.700 35.455,289.700 59.493 C 272.299 72.563,267.142 98.968,278.362 117.538 C 280.320 120.777,280.486 120.015,276.475 126.200 C 275.049 128.400,273.125 131.370,272.200 132.800 C 263.190 146.731,258.134 154.433,248.502 168.900 C 243.558 176.325,239.445 182.400,239.362 182.400 C 239.280 182.400,237.364 181.785,235.106 181.034 C 195.774 167.952,160.279 176.156,131.442 204.993 L 128.055 208.379 127.800 206.090 C 123.727 169.518,86.403 151.135,57.400 171.416 C 41.016 182.874,33.578 205.119,40.584 221.715 C 41.451 223.769,43.855 221.196,18.262 245.607 C 12.960 250.664,10.613 252.781,10.300 252.791 C 9.815 252.806,9.730 75.980,10.212 70.200 C 12.746 39.808,32.408 16.679,60.587 10.944 C 66.054 9.831,179.673 9.497,315.200 10.195 M351.603 137.001 C 355.891 139.807,360.840 143.042,362.600 144.190 C 364.360 145.339,367.870 147.629,370.400 149.280 C 378.567 154.609,384.155 158.212,386.125 159.420 C 387.183 160.069,388.547 160.995,389.155 161.477 L 390.261 162.354 390.180 248.477 C 390.089 344.375,390.264 338.263,387.363 347.200 C 381.119 366.432,362.495 383.490,342.609 388.189 C 335.263 389.924,332.274 389.921,333.970 388.180 C 335.670 386.435,337.998 376.849,337.996 371.600 C 337.993 359.921,334.097 352.033,322.785 340.800 C 317.668 335.718,294.455 313.421,291.026 310.294 L 289.452 308.859 290.114 307.329 C 298.089 288.891,299.455 263.701,293.558 243.800 C 289.293 229.406,281.414 216.066,269.923 203.781 C 267.193 200.863,267.262 201.204,268.890 198.635 C 270.414 196.229,283.847 175.712,296.200 156.921 C 299.280 152.236,303.150 146.333,304.800 143.802 C 306.450 141.272,308.059 139.202,308.377 139.201 C 308.694 139.200,310.966 139.404,313.425 139.653 C 323.375 140.662,332.355 138.492,340.800 133.038 C 343.457 131.322,342.236 130.874,351.603 137.001 M210.600 189.148 C 215.709 189.750,221.172 190.715,222.200 191.197 C 222.640 191.404,224.170 191.865,225.600 192.222 C 227.030 192.579,228.380 193.012,228.600 193.184 C 228.820 193.355,229.720 193.724,230.600 194.002 C 232.514 194.607,239.098 198.011,241.376 199.573 C 242.269 200.185,245.159 202.107,247.798 203.843 C 261.742 213.019,272.790 225.837,277.583 238.400 C 278.044 239.610,278.599 240.960,278.816 241.400 C 279.533 242.855,280.871 247.741,281.665 251.800 C 282.095 254.000,282.611 256.340,282.811 257.000 C 283.981 260.852,284.260 279.785,283.218 284.600 C 283.028 285.480,282.583 287.640,282.230 289.400 C 281.243 294.324,279.227 299.523,275.824 305.919 C 274.610 308.201,275.650 309.733,282.000 315.008 C 285.725 318.102,311.409 342.215,314.427 345.451 C 318.774 350.112,321.565 357.599,321.565 364.600 C 321.565 376.428,313.841 387.154,303.000 390.380 C 302.120 390.641,300.968 391.068,300.439 391.328 C 298.382 392.339,288.166 391.938,284.925 390.718 C 278.654 388.357,278.715 388.409,257.000 366.527 C 234.644 343.997,236.276 345.301,232.563 347.000 C 231.360 347.550,230.247 348.001,230.088 348.002 C 229.930 348.004,229.304 348.264,228.698 348.580 C 219.827 353.207,197.870 354.613,184.200 351.428 C 179.776 350.397,175.002 349.070,174.800 348.814 C 174.690 348.675,173.823 348.334,172.873 348.056 C 158.791 343.941,139.598 327.861,130.791 312.800 C 129.004 309.743,126.400 304.662,126.400 304.230 C 126.400 304.099,126.044 303.228,125.608 302.296 C 120.152 290.615,117.848 269.740,120.561 256.574 C 121.328 252.854,122.795 247.495,123.182 247.000 C 123.354 246.780,123.791 245.700,124.154 244.600 C 124.859 242.460,130.485 231.099,132.968 226.800 C 141.608 211.839,157.818 198.566,173.000 194.023 C 173.880 193.760,174.960 193.372,175.400 193.162 C 177.904 191.967,186.761 190.175,193.945 189.411 C 196.885 189.099,199.402 188.732,199.538 188.595 C 199.863 188.270,205.575 188.556,210.600 189.148 M194.989 214.814 C 193.003 215.002,190.663 215.335,189.789 215.553 C 185.341 216.664,181.757 217.703,181.400 217.986 C 181.180 218.160,180.550 218.433,180.000 218.593 C 178.127 219.139,172.608 222.095,169.624 224.150 C 149.083 238.301,140.003 263.550,146.712 287.866 C 149.619 298.404,157.273 309.613,166.236 316.460 C 168.990 318.565,174.350 321.857,176.386 322.694 C 177.164 323.014,178.340 323.526,179.000 323.831 C 186.351 327.227,201.254 329.409,207.800 328.047 C 208.900 327.818,211.504 327.343,213.588 326.991 C 215.671 326.640,217.561 326.215,217.788 326.046 C 218.014 325.878,219.010 325.510,220.000 325.229 C 223.575 324.214,229.724 321.181,233.351 318.644 C 234.643 317.740,236.083 316.732,236.551 316.406 C 237.879 315.476,245.319 307.958,246.452 306.400 C 254.514 295.318,258.171 284.397,258.171 271.400 C 258.171 249.683,246.489 230.680,226.974 220.652 C 225.420 219.853,224.023 219.200,223.870 219.200 C 223.717 219.200,222.828 218.843,221.896 218.406 C 215.911 215.605,203.762 213.983,194.989 214.814 M108.271 246.300 C 100.639 284.420,108.853 314.489,133.632 339.150 C 159.347 364.741,200.668 374.070,233.104 361.608 C 234.262 361.164,235.546 360.800,235.959 360.800 C 236.764 360.800,264.101 388.368,264.329 389.409 C 264.486 390.127,72.427 390.305,65.675 389.594 C 39.901 386.879,15.900 364.780,10.942 339.200 C 10.049 334.593,9.546 321.294,9.842 310.133 L 10.109 300.066 17.155 293.122 C 21.030 289.303,26.810 283.663,30.001 280.589 C 33.191 277.515,38.685 272.210,42.210 268.800 C 53.948 257.444,63.143 248.800,63.482 248.800 C 63.667 248.800,65.177 249.323,66.838 249.963 C 79.525 254.848,93.206 253.740,105.000 246.872 C 108.352 244.920,108.554 244.885,108.271 246.300 " stroke="none" fill="#fbfbfb" fill-rule="evenodd"></path>
    <path id="path1" d="M195.200 189.236 C 189.011 189.865,178.982 191.818,177.000 192.781 C 176.560 192.994,175.120 193.468,173.800 193.833 C 172.480 194.198,171.220 194.624,171.000 194.779 C 170.780 194.935,169.700 195.390,168.600 195.791 C 154.036 201.102,139.137 214.954,130.949 230.796 C 130.097 232.444,129.305 233.884,129.189 233.996 C 129.073 234.108,128.628 235.010,128.200 236.000 C 127.772 236.990,127.321 237.890,127.197 238.000 C 126.241 238.847,121.807 250.719,121.364 253.613 C 121.214 254.596,120.955 255.580,120.790 255.800 C 119.584 257.403,119.166 274.755,120.169 281.600 C 120.874 286.418,122.267 293.157,122.772 294.200 C 122.985 294.640,123.514 296.170,123.947 297.600 C 125.314 302.112,129.163 310.379,131.570 313.970 C 136.369 321.130,137.962 323.115,143.045 328.266 C 150.931 336.258,161.151 343.423,168.672 346.233 C 169.733 346.629,170.960 347.167,171.400 347.429 C 171.840 347.691,172.740 348.030,173.400 348.183 C 174.060 348.336,175.140 348.713,175.800 349.020 C 177.118 349.632,182.227 350.956,186.600 351.818 C 194.757 353.425,210.454 353.423,217.400 351.813 C 223.615 350.373,227.921 348.975,232.285 346.978 C 236.681 344.967,234.237 342.913,265.138 374.600 C 270.930 380.540,276.329 385.882,277.135 386.471 C 279.994 388.562,282.797 389.995,285.282 390.638 C 286.667 390.996,288.170 391.461,288.623 391.671 C 289.076 391.881,291.416 391.987,293.823 391.906 C 320.144 391.027,330.893 361.840,311.813 343.055 C 305.724 337.059,286.880 319.476,280.713 314.035 C 274.249 308.332,274.329 308.627,277.520 302.191 C 278.827 299.556,280.127 296.500,280.410 295.400 C 280.693 294.300,281.075 293.203,281.259 292.963 C 284.374 288.894,284.489 260.113,281.429 250.400 C 281.117 249.410,280.562 247.430,280.195 246.000 C 279.828 244.570,279.397 243.220,279.236 243.000 C 279.076 242.780,278.701 241.790,278.404 240.800 C 276.035 232.904,268.179 221.226,260.068 213.541 C 251.945 205.844,234.750 194.824,228.600 193.373 C 227.940 193.217,227.208 192.940,226.973 192.756 C 226.738 192.573,225.080 192.073,223.289 191.645 C 221.497 191.217,219.858 190.759,219.645 190.628 C 217.823 189.502,201.647 188.581,195.200 189.236 M206.600 214.809 C 211.371 215.252,215.564 215.947,216.200 216.400 C 216.420 216.556,217.490 216.902,218.577 217.168 C 220.480 217.632,221.606 218.081,226.400 220.286 C 241.156 227.072,254.093 243.297,256.789 258.400 C 257.044 259.830,257.441 261.344,257.670 261.764 C 258.264 262.854,258.553 274.869,258.064 278.200 C 256.520 288.727,252.225 299.393,246.804 306.154 C 244.385 309.172,240.467 313.291,238.580 314.800 C 237.342 315.790,236.119 316.780,235.862 317.000 C 233.471 319.046,222.963 324.797,221.606 324.802 C 221.273 324.804,220.483 325.075,219.851 325.404 C 218.783 325.961,213.984 327.118,208.800 328.069 C 206.105 328.563,192.249 328.293,191.523 327.732 C 191.235 327.510,189.650 327.100,188.000 326.821 C 186.350 326.543,184.460 326.080,183.800 325.792 C 183.140 325.504,181.880 325.046,181.000 324.774 C 163.754 319.440,147.448 299.624,145.195 281.261 C 144.961 279.358,144.576 276.630,144.340 275.200 C 144.103 273.770,144.004 271.700,144.119 270.600 C 144.874 263.413,145.828 257.516,146.409 256.453 C 146.605 256.094,147.061 254.720,147.423 253.400 C 149.727 244.989,158.430 232.427,165.870 226.773 C 167.152 225.799,168.290 224.908,168.400 224.792 C 169.091 224.067,174.359 221.016,177.388 219.587 C 186.414 215.327,195.740 213.801,206.600 214.809 " stroke="none" fill="#ec741c" fill-rule="evenodd"></path>
    <path id="path2" d="M118.200 10.376 C 57.779 10.668,61.214 10.499,52.508 13.605 C 30.493 21.458,15.090 40.273,11.405 63.810 L 10.418 70.110 10.352 132.155 C 10.316 166.280,10.312 207.325,10.343 223.368 L 10.400 252.535 13.302 249.768 C 14.899 248.245,19.984 243.385,24.602 238.967 C 29.221 234.549,34.665 229.347,36.700 227.407 C 40.768 223.529,40.880 223.254,39.649 220.115 C 39.237 219.062,38.664 216.850,38.377 215.200 C 32.160 179.442,70.613 151.062,102.905 167.577 C 116.706 174.634,127.004 189.524,127.879 203.687 L 128.119 207.573 132.390 203.487 C 146.518 189.967,161.178 181.473,178.000 177.059 C 189.728 173.981,211.483 173.839,221.000 176.777 C 221.660 176.981,223.280 177.357,224.600 177.613 C 226.961 178.071,234.188 180.166,237.480 181.348 L 239.160 181.951 240.498 180.075 C 241.777 178.280,259.482 151.626,263.264 145.800 C 264.264 144.260,266.483 140.840,268.195 138.200 C 269.907 135.560,272.423 131.690,273.786 129.600 C 275.149 127.510,277.006 124.645,277.913 123.233 L 279.561 120.666 277.158 116.153 C 265.329 93.939,274.272 65.822,296.717 54.658 C 317.129 44.506,341.596 50.633,353.990 69.000 C 356.475 72.683,356.657 72.990,357.771 75.400 C 361.187 82.793,362.199 88.121,361.909 97.193 L 361.709 103.465 363.555 104.608 C 364.570 105.236,368.190 107.583,371.600 109.823 C 378.919 114.631,388.822 120.800,389.222 120.800 C 389.755 120.800,390.060 90.743,389.642 79.400 C 389.013 62.290,387.739 56.280,382.479 45.600 C 381.666 43.950,380.713 42.150,380.360 41.600 C 377.386 36.962,376.287 35.406,374.080 32.712 C 368.081 25.387,359.737 19.136,350.800 15.270 C 345.954 13.173,339.838 11.872,330.000 10.844 C 325.075 10.329,189.799 10.030,118.200 10.376 M341.289 133.227 C 333.811 138.727,321.798 141.328,310.541 139.885 L 308.082 139.570 306.293 142.285 C 305.310 143.778,302.563 147.970,300.190 151.600 C 297.816 155.230,293.804 161.350,291.274 165.200 C 280.609 181.425,273.842 191.757,269.869 197.879 C 267.570 201.420,267.440 200.693,271.064 204.544 C 275.163 208.900,278.588 213.182,281.398 217.464 C 282.674 219.409,284.042 221.474,284.437 222.054 C 286.955 225.745,291.023 234.710,293.125 241.200 C 295.212 247.642,295.562 248.970,296.193 252.848 C 296.529 254.911,296.986 257.590,297.209 258.800 C 298.544 266.054,297.755 282.144,295.649 290.607 C 295.431 291.483,294.964 293.370,294.611 294.800 C 293.753 298.274,291.296 305.855,290.576 307.245 C 290.259 307.859,290.000 308.436,290.000 308.529 C 290.000 308.622,295.355 313.803,301.900 320.042 C 326.266 343.271,331.408 348.755,334.549 354.861 C 339.461 364.411,339.617 377.995,334.922 387.337 L 333.949 389.275 335.074 389.059 C 335.693 388.940,337.190 388.729,338.400 388.590 C 361.221 385.968,383.770 364.233,388.812 340.000 L 389.685 335.800 389.629 249.000 C 389.580 171.918,389.506 162.155,388.967 161.800 C 388.634 161.580,386.885 160.450,385.080 159.290 C 383.276 158.129,378.958 155.339,375.485 153.090 C 369.640 149.305,366.149 147.027,350.400 136.719 C 342.459 131.522,343.214 131.811,341.289 133.227 M106.600 246.476 C 94.378 254.284,78.505 255.388,64.000 249.440 C 63.385 249.188,62.634 249.816,56.000 256.126 C 51.186 260.705,32.339 278.902,23.803 287.212 C 19.841 291.068,15.176 295.589,13.435 297.259 L 10.270 300.294 10.280 316.847 C 10.294 338.249,11.076 342.864,16.582 354.000 C 25.302 371.636,44.022 385.849,62.200 388.634 C 65.971 389.212,263.600 389.520,263.600 388.948 C 263.600 388.809,262.655 387.748,261.500 386.591 C 260.345 385.433,254.211 379.246,247.868 372.843 C 241.526 366.439,236.208 361.200,236.052 361.200 C 235.896 361.200,234.106 361.739,232.075 362.397 C 212.495 368.743,189.776 368.815,170.965 362.591 C 167.087 361.307,162.739 359.621,162.200 359.191 C 161.980 359.015,160.990 358.548,160.000 358.153 C 154.257 355.860,145.270 350.005,138.200 343.950 C 114.815 323.920,102.253 293.052,105.176 262.800 C 105.641 257.995,106.071 255.270,107.383 248.827 C 108.088 245.360,108.134 245.496,106.600 246.476 " stroke="none" fill="#343434" fill-rule="evenodd"></path>
    <path id="path3" d="M114.130 10.300 C 124.872 10.359,142.332 10.359,152.930 10.300 C 163.529 10.241,154.740 10.192,133.400 10.192 C 112.060 10.192,103.389 10.241,114.130 10.300 M265.927 10.300 C 270.617 10.364,278.177 10.364,282.727 10.299 C 287.277 10.235,283.440 10.183,274.200 10.183 C 264.960 10.183,261.237 10.236,265.927 10.300 M65.712 10.695 C 66.764 10.777,68.384 10.776,69.312 10.692 C 70.240 10.608,69.380 10.541,67.400 10.542 C 65.420 10.543,64.660 10.612,65.712 10.695 M328.100 10.676 C 328.485 10.776,329.115 10.776,329.500 10.676 C 329.885 10.575,329.570 10.493,328.800 10.493 C 328.030 10.493,327.715 10.575,328.100 10.676 M332.305 11.083 C 332.802 11.179,333.522 11.175,333.905 11.075 C 334.287 10.975,333.880 10.896,333.000 10.901 C 332.120 10.905,331.807 10.987,332.305 11.083 M335.300 11.476 C 335.685 11.576,336.315 11.576,336.700 11.476 C 337.085 11.375,336.770 11.293,336.000 11.293 C 335.230 11.293,334.915 11.375,335.300 11.476 M31.375 26.100 L 30.200 27.400 31.500 26.225 C 32.215 25.579,32.800 24.994,32.800 24.925 C 32.800 24.616,32.470 24.888,31.375 26.100 M369.200 27.325 C 369.200 27.394,369.785 27.979,370.500 28.625 L 371.800 29.800 370.625 28.500 C 369.530 27.288,369.200 27.016,369.200 27.325 M26.175 31.300 L 25.000 32.600 26.300 31.425 C 27.015 30.779,27.600 30.194,27.600 30.125 C 27.600 29.816,27.270 30.088,26.175 31.300 M378.000 38.000 C 378.470 38.660,378.944 39.200,379.054 39.200 C 379.164 39.200,378.870 38.660,378.400 38.000 C 377.930 37.340,377.456 36.800,377.346 36.800 C 377.236 36.800,377.530 37.340,378.000 38.000 M315.500 49.895 C 316.655 49.976,318.545 49.976,319.700 49.895 C 320.855 49.814,319.910 49.748,317.600 49.748 C 315.290 49.748,314.345 49.814,315.500 49.895 M349.000 63.200 C 349.859 64.080,350.652 64.800,350.762 64.800 C 350.872 64.800,350.259 64.080,349.400 63.200 C 348.541 62.320,347.748 61.600,347.638 61.600 C 347.528 61.600,348.141 62.320,349.000 63.200 M388.893 66.000 C 388.893 66.770,388.975 67.085,389.076 66.700 C 389.176 66.315,389.176 65.685,389.076 65.300 C 388.975 64.915,388.893 65.230,388.893 66.000 M10.131 76.200 C 10.133 77.740,10.205 78.321,10.292 77.491 C 10.379 76.661,10.377 75.401,10.289 74.691 C 10.200 73.981,10.129 74.660,10.131 76.200 M389.737 82.200 C 389.739 83.960,389.809 84.630,389.893 83.689 C 389.978 82.749,389.977 81.309,389.891 80.489 C 389.805 79.670,389.736 80.440,389.737 82.200 M10.157 92.000 C 10.157 94.970,10.220 96.185,10.297 94.700 C 10.374 93.215,10.374 90.785,10.297 89.300 C 10.220 87.815,10.157 89.030,10.157 92.000 M10.191 136.400 C 10.191 154.990,10.240 162.595,10.300 153.300 C 10.360 144.005,10.360 128.795,10.300 119.500 C 10.240 110.205,10.191 117.810,10.191 136.400 M389.516 113.400 C 389.452 117.691,389.535 121.201,389.700 121.200 C 389.865 121.200,390.000 117.690,390.000 113.400 C 390.000 109.110,389.917 105.600,389.816 105.600 C 389.715 105.600,389.580 109.110,389.516 113.400 M274.428 110.600 C 274.428 110.820,274.686 111.450,275.000 112.000 C 275.314 112.550,275.572 112.820,275.572 112.600 C 275.572 112.380,275.314 111.750,275.000 111.200 C 274.686 110.650,274.428 110.380,274.428 110.600 M307.549 139.900 C 307.072 140.509,307.078 140.540,307.600 140.143 C 307.930 139.892,308.560 139.598,309.000 139.489 C 309.760 139.303,309.757 139.290,308.949 139.246 C 308.481 139.221,307.851 139.515,307.549 139.900 M310.500 139.876 C 310.885 139.976,311.515 139.976,311.900 139.876 C 312.285 139.775,311.970 139.693,311.200 139.693 C 310.430 139.693,310.115 139.775,310.500 139.876 M314.100 140.287 C 314.705 140.378,315.695 140.378,316.300 140.287 C 316.905 140.195,316.410 140.120,315.200 140.120 C 313.990 140.120,313.495 140.195,314.100 140.287 M319.300 140.287 C 319.905 140.378,320.895 140.378,321.500 140.287 C 322.105 140.195,321.610 140.120,320.400 140.120 C 319.190 140.120,318.695 140.195,319.300 140.287 M389.796 238.800 C 389.796 281.150,389.843 298.475,389.900 277.300 C 389.957 256.125,389.957 221.475,389.900 200.300 C 389.843 179.125,389.796 196.450,389.796 238.800 M197.712 174.695 C 198.764 174.777,200.384 174.776,201.312 174.692 C 202.240 174.608,201.380 174.541,199.400 174.542 C 197.420 174.543,196.660 174.612,197.712 174.695 M115.183 176.900 C 116.164 177.945,117.019 178.800,117.083 178.800 C 117.384 178.800,117.066 178.441,115.300 176.783 L 113.400 175.000 115.183 176.900 M227.000 178.400 C 227.440 178.589,228.070 178.744,228.400 178.744 C 228.778 178.744,228.704 178.617,228.200 178.400 C 227.760 178.211,227.130 178.056,226.800 178.056 C 226.422 178.056,226.496 178.183,227.000 178.400 M10.190 223.200 C 10.190 239.150,10.239 245.675,10.300 237.700 C 10.360 229.725,10.360 216.675,10.300 208.700 C 10.239 200.725,10.190 207.250,10.190 223.200 M136.975 199.300 L 135.800 200.600 137.100 199.425 C 138.312 198.330,138.584 198.000,138.275 198.000 C 138.206 198.000,137.621 198.585,136.975 199.300 M131.392 204.500 L 127.800 208.200 131.500 204.608 C 134.936 201.272,135.382 200.800,135.092 200.800 C 135.032 200.800,133.367 202.465,131.392 204.500 M270.400 204.125 C 270.400 204.194,270.985 204.779,271.700 205.425 L 273.000 206.600 271.825 205.300 C 270.730 204.088,270.400 203.816,270.400 204.125 M37.724 207.400 C 37.726 208.720,37.801 209.212,37.890 208.493 C 37.979 207.774,37.977 206.694,37.886 206.093 C 37.795 205.492,37.722 206.080,37.724 207.400 M40.446 222.978 C 40.421 223.406,39.909 224.306,39.309 224.978 L 38.218 226.200 39.549 224.953 C 40.427 224.130,40.813 223.450,40.686 222.953 C 40.502 222.236,40.491 222.237,40.446 222.978 M30.788 233.100 L 28.200 235.800 30.900 233.212 C 33.408 230.807,33.783 230.400,33.488 230.400 C 33.427 230.400,32.212 231.615,30.788 233.100 M20.386 243.100 L 18.200 245.400 20.500 243.214 C 21.765 242.011,22.800 240.976,22.800 240.914 C 22.800 240.617,22.418 240.963,20.386 243.100 M13.586 249.500 L 11.400 251.800 13.700 249.614 C 14.965 248.411,16.000 247.376,16.000 247.314 C 16.000 247.017,15.618 247.363,13.586 249.500 M56.181 255.700 L 54.600 257.400 56.300 255.819 C 57.881 254.348,58.184 254.000,57.881 254.000 C 57.816 254.000,57.051 254.765,56.181 255.700 M297.314 261.800 C 297.317 262.900,297.395 263.303,297.487 262.695 C 297.579 262.086,297.577 261.186,297.482 260.695 C 297.387 260.203,297.311 260.700,297.314 261.800 M46.800 264.600 C 45.161 266.250,43.910 267.600,44.020 267.600 C 44.130 267.600,45.561 266.250,47.200 264.600 C 48.839 262.950,50.090 261.600,49.980 261.600 C 49.870 261.600,48.439 262.950,46.800 264.600 M104.909 266.400 C 104.909 267.390,104.987 267.795,105.082 267.300 C 105.178 266.805,105.178 265.995,105.082 265.500 C 104.987 265.005,104.909 265.410,104.909 266.400 M297.757 270.000 C 297.757 272.970,297.820 274.185,297.897 272.700 C 297.974 271.215,297.974 268.785,297.897 267.300 C 297.820 265.815,297.757 267.030,297.757 270.000 M33.791 277.300 L 30.600 280.600 33.900 277.409 C 35.715 275.655,37.200 274.170,37.200 274.109 C 37.200 273.818,36.754 274.235,33.791 277.300 M297.314 278.200 C 297.317 279.300,297.395 279.703,297.487 279.095 C 297.579 278.486,297.577 277.586,297.482 277.095 C 297.387 276.603,297.311 277.100,297.314 278.200 M20.994 289.700 L 15.800 295.000 21.100 289.806 C 24.015 286.949,26.400 284.564,26.400 284.506 C 26.400 284.219,25.819 284.777,20.994 289.700 M10.159 306.000 C 10.159 309.190,10.221 310.495,10.297 308.900 C 10.373 307.305,10.373 304.695,10.297 303.100 C 10.221 301.505,10.159 302.810,10.159 306.000 M291.800 310.800 C 292.883 311.900,293.860 312.800,293.970 312.800 C 294.080 312.800,293.283 311.900,292.200 310.800 C 291.117 309.700,290.140 308.800,290.030 308.800 C 289.920 308.800,290.717 309.700,291.800 310.800 M299.200 317.800 C 300.617 319.230,301.867 320.400,301.977 320.400 C 302.087 320.400,301.017 319.230,299.600 317.800 C 298.183 316.370,296.933 315.200,296.823 315.200 C 296.713 315.200,297.783 316.370,299.200 317.800 M10.144 326.800 C 10.144 328.890,10.212 329.745,10.294 328.700 C 10.376 327.655,10.376 325.945,10.294 324.900 C 10.212 323.855,10.144 324.710,10.144 326.800 M308.200 326.400 C 309.950 328.160,311.471 329.600,311.581 329.600 C 311.691 329.600,310.350 328.160,308.600 326.400 C 306.850 324.640,305.329 323.200,305.219 323.200 C 305.109 323.200,306.450 324.640,308.200 326.400 M389.714 331.800 C 389.717 332.900,389.795 333.303,389.887 332.695 C 389.979 332.086,389.977 331.186,389.882 330.695 C 389.787 330.203,389.711 330.700,389.714 331.800 M133.000 339.200 C 134.971 341.180,136.673 342.800,136.783 342.800 C 136.893 342.800,135.371 341.180,133.400 339.200 C 131.429 337.220,129.727 335.600,129.617 335.600 C 129.507 335.600,131.029 337.220,133.000 339.200 M320.200 338.000 C 321.506 339.320,322.665 340.400,322.775 340.400 C 322.885 340.400,321.906 339.320,320.600 338.000 C 319.294 336.680,318.135 335.600,318.025 335.600 C 317.915 335.600,318.894 336.680,320.200 338.000 M371.792 370.900 L 367.800 375.000 371.900 371.008 C 374.155 368.812,376.000 366.967,376.000 366.908 C 376.000 366.618,375.501 367.092,371.792 370.900 M252.200 377.200 C 255.494 380.500,258.280 383.200,258.390 383.200 C 258.500 383.200,255.894 380.500,252.600 377.200 C 249.306 373.900,246.520 371.200,246.410 371.200 C 246.300 371.200,248.906 373.900,252.200 377.200 M335.265 385.817 C 334.595 387.288,334.723 387.602,335.447 386.265 C 335.786 385.640,335.988 385.054,335.896 384.963 C 335.805 384.871,335.521 385.256,335.265 385.817 M164.600 389.200 L 65.400 389.402 164.700 389.501 C 230.699 389.567,264.000 389.466,264.000 389.200 C 264.000 388.980,263.955 388.844,263.900 388.899 C 263.845 388.953,219.160 389.089,164.600 389.200 " stroke="none" fill="#777777" fill-rule="evenodd"></path>
    <path id="path4" d="M193.300 9.900 C 205.125 9.959,224.475 9.959,236.300 9.900 C 248.125 9.841,238.450 9.793,214.800 9.793 C 191.150 9.793,181.475 9.841,193.300 9.900 M82.700 10.298 C 84.955 10.370,88.645 10.370,90.900 10.298 C 93.155 10.227,91.310 10.169,86.800 10.169 C 82.290 10.169,80.445 10.227,82.700 10.298 M293.100 10.296 C 294.475 10.374,296.725 10.374,298.100 10.296 C 299.475 10.218,298.350 10.154,295.600 10.154 C 292.850 10.154,291.725 10.218,293.100 10.296 M313.100 10.298 C 315.135 10.371,318.465 10.371,320.500 10.298 C 322.535 10.225,320.870 10.166,316.800 10.166 C 312.730 10.166,311.065 10.225,313.100 10.298 M346.800 13.533 C 347.130 13.736,347.850 14.024,348.400 14.171 C 349.095 14.357,349.217 14.325,348.800 14.067 C 348.470 13.864,347.750 13.576,347.200 13.429 C 346.505 13.243,346.383 13.275,346.800 13.533 M45.900 15.829 C 44.349 16.670,44.525 16.961,46.100 16.159 C 47.608 15.393,47.875 15.194,47.367 15.216 C 47.165 15.225,46.505 15.501,45.900 15.829 M367.400 25.600 C 368.259 26.480,369.052 27.200,369.162 27.200 C 369.272 27.200,368.659 26.480,367.800 25.600 C 366.941 24.720,366.148 24.000,366.038 24.000 C 365.928 24.000,366.541 24.720,367.400 25.600 M30.575 26.500 L 29.400 27.800 30.700 26.625 C 31.415 25.979,32.000 25.394,32.000 25.325 C 32.000 25.016,31.670 25.288,30.575 26.500 M27.200 29.800 C 26.228 30.790,25.523 31.600,25.633 31.600 C 25.743 31.600,26.628 30.790,27.600 29.800 C 28.572 28.810,29.277 28.000,29.167 28.000 C 29.057 28.000,28.172 28.810,27.200 29.800 M372.800 31.000 C 373.546 31.770,374.247 32.400,374.357 32.400 C 374.467 32.400,373.946 31.770,373.200 31.000 C 372.454 30.230,371.753 29.600,371.643 29.600 C 371.533 29.600,372.054 30.230,372.800 31.000 M22.000 36.000 C 21.530 36.660,21.236 37.200,21.346 37.200 C 21.456 37.200,21.930 36.660,22.400 36.000 C 22.870 35.340,23.164 34.800,23.054 34.800 C 22.944 34.800,22.470 35.340,22.000 36.000 M383.628 47.400 C 383.628 47.620,383.886 48.250,384.200 48.800 C 384.514 49.350,384.772 49.620,384.772 49.400 C 384.772 49.180,384.514 48.550,384.200 48.000 C 383.886 47.450,383.628 47.180,383.628 47.400 M313.505 50.283 C 314.002 50.379,314.722 50.375,315.105 50.275 C 315.487 50.175,315.080 50.096,314.200 50.101 C 313.320 50.105,313.007 50.187,313.505 50.283 M320.500 50.276 C 320.885 50.376,321.515 50.376,321.900 50.276 C 322.285 50.175,321.970 50.093,321.200 50.093 C 320.430 50.093,320.115 50.175,320.500 50.276 M305.000 51.600 C 303.981 52.038,305.032 52.038,306.400 51.600 C 307.204 51.343,307.243 51.277,306.600 51.268 C 306.160 51.262,305.440 51.411,305.000 51.600 M347.000 61.600 C 347.859 62.480,348.652 63.200,348.762 63.200 C 348.872 63.200,348.259 62.480,347.400 61.600 C 346.541 60.720,345.748 60.000,345.638 60.000 C 345.528 60.000,346.141 60.720,347.000 61.600 M285.381 62.900 L 283.800 64.600 285.500 63.019 C 287.081 61.548,287.384 61.200,287.081 61.200 C 287.016 61.200,286.251 61.965,285.381 62.900 M349.200 63.725 C 349.200 63.794,349.785 64.379,350.500 65.025 L 351.800 66.200 350.625 64.900 C 349.530 63.688,349.200 63.416,349.200 63.725 M389.293 69.200 C 389.293 69.970,389.375 70.285,389.476 69.900 C 389.576 69.515,389.576 68.885,389.476 68.500 C 389.375 68.115,389.293 68.430,389.293 69.200 M10.109 72.000 C 10.109 72.990,10.187 73.395,10.282 72.900 C 10.378 72.405,10.378 71.595,10.282 71.100 C 10.187 70.605,10.109 71.010,10.109 72.000 M276.581 75.237 C 276.229 75.918,276.012 76.545,276.099 76.632 C 276.186 76.719,276.556 76.162,276.922 75.395 C 277.744 73.670,277.459 73.538,276.581 75.237 M389.724 77.000 C 389.726 78.320,389.801 78.812,389.890 78.093 C 389.979 77.374,389.977 76.294,389.886 75.693 C 389.795 75.092,389.722 75.680,389.724 77.000 M9.748 82.800 C 9.748 85.110,9.814 86.055,9.895 84.900 C 9.976 83.745,9.976 81.855,9.895 80.700 C 9.814 79.545,9.748 80.490,9.748 82.800 M390.181 96.800 C 390.181 105.050,390.234 108.425,390.299 104.300 C 390.364 100.175,390.364 93.425,390.299 89.300 C 390.234 85.175,390.181 88.550,390.181 96.800 M272.501 89.000 C 272.505 89.880,272.587 90.193,272.683 89.695 C 272.779 89.198,272.775 88.478,272.675 88.095 C 272.575 87.713,272.496 88.120,272.501 89.000 M361.693 90.800 C 361.693 91.570,361.775 91.885,361.876 91.500 C 361.976 91.115,361.976 90.485,361.876 90.100 C 361.775 89.715,361.693 90.030,361.693 90.800 M272.153 95.400 C 272.153 98.040,272.218 99.068,272.297 97.684 C 272.375 96.301,272.374 94.141,272.295 92.884 C 272.216 91.628,272.152 92.760,272.153 95.400 M10.134 100.000 C 10.134 101.650,10.205 102.325,10.291 101.500 C 10.378 100.675,10.378 99.325,10.291 98.500 C 10.205 97.675,10.134 98.350,10.134 100.000 M272.493 101.600 C 272.493 102.370,272.575 102.685,272.676 102.300 C 272.776 101.915,272.776 101.285,272.676 100.900 C 272.575 100.515,272.493 100.830,272.493 101.600 M312.900 139.876 C 313.285 139.976,313.915 139.976,314.300 139.876 C 314.685 139.775,314.370 139.693,313.600 139.693 C 312.830 139.693,312.515 139.775,312.900 139.876 M321.300 139.876 C 321.685 139.976,322.315 139.976,322.700 139.876 C 323.085 139.775,322.770 139.693,322.000 139.693 C 321.230 139.693,320.915 139.775,321.300 139.876 M390.189 188.000 C 390.189 202.190,390.239 207.995,390.300 200.900 C 390.361 193.805,390.361 182.195,390.300 175.100 C 390.239 168.005,390.189 173.810,390.189 188.000 M80.915 163.096 C 82.188 163.176,84.168 163.175,85.315 163.094 C 86.462 163.014,85.420 162.949,83.000 162.950 C 80.580 162.951,79.642 163.016,80.915 163.096 M88.100 163.476 C 88.485 163.576,89.115 163.576,89.500 163.476 C 89.885 163.375,89.570 163.293,88.800 163.293 C 88.030 163.293,87.715 163.375,88.100 163.476 M100.833 166.949 C 101.245 167.317,103.600 168.400,103.600 168.221 C 103.600 167.990,101.382 166.800,100.953 166.800 C 100.796 166.800,100.742 166.867,100.833 166.949 M9.779 181.800 C 9.779 189.060,9.833 192.086,9.899 188.525 C 9.965 184.964,9.966 179.024,9.899 175.325 C 9.833 171.626,9.779 174.540,9.779 181.800 M247.353 170.100 C 245.976 172.436,245.886 172.635,246.587 171.800 C 247.400 170.832,248.100 169.600,247.838 169.600 C 247.733 169.600,247.515 169.825,247.353 170.100 M108.000 170.925 C 108.000 171.024,108.661 171.609,109.468 172.225 C 110.276 172.841,110.811 173.141,110.656 172.891 C 110.340 172.379,108.000 170.647,108.000 170.925 M113.000 175.200 C 114.083 176.300,115.060 177.200,115.170 177.200 C 115.280 177.200,114.483 176.300,113.400 175.200 C 112.317 174.100,111.340 173.200,111.230 173.200 C 111.120 173.200,111.917 174.100,113.000 175.200 M50.989 176.500 L 48.200 179.400 51.100 176.611 C 53.794 174.020,54.183 173.600,53.889 173.600 C 53.828 173.600,52.523 174.905,50.989 176.500 M192.305 175.083 C 192.802 175.179,193.522 175.175,193.905 175.075 C 194.287 174.975,193.880 174.896,193.000 174.901 C 192.120 174.905,191.807 174.987,192.305 175.083 M205.105 175.083 C 205.602 175.179,206.322 175.175,206.705 175.075 C 207.087 174.975,206.680 174.896,205.800 174.901 C 204.920 174.905,204.607 174.987,205.105 175.083 M212.900 175.876 C 213.285 175.976,213.915 175.976,214.300 175.876 C 214.685 175.775,214.370 175.693,213.600 175.693 C 212.830 175.693,212.515 175.775,212.900 175.876 M232.000 179.933 C 232.330 180.136,233.050 180.424,233.600 180.571 C 234.295 180.757,234.417 180.725,234.000 180.467 C 233.670 180.264,232.950 179.976,232.400 179.829 C 231.705 179.643,231.583 179.675,232.000 179.933 M118.575 181.332 C 119.191 182.139,119.776 182.800,119.875 182.800 C 120.153 182.800,118.421 180.460,117.909 180.144 C 117.659 179.989,117.959 180.524,118.575 181.332 M238.313 182.123 C 238.867 182.372,239.370 182.390,239.591 182.167 C 239.813 181.945,239.452 181.782,238.678 181.756 C 237.488 181.714,237.463 181.740,238.313 182.123 M155.500 186.229 C 154.005 187.040,154.096 187.378,155.600 186.600 C 156.260 186.259,156.800 185.894,156.800 185.790 C 156.800 185.532,156.757 185.547,155.500 186.229 M201.312 188.695 C 202.364 188.777,203.984 188.776,204.912 188.692 C 205.840 188.608,204.980 188.541,203.000 188.542 C 201.020 188.543,200.260 188.612,201.312 188.695 M270.953 195.300 C 269.576 197.636,269.486 197.835,270.187 197.000 C 271.000 196.032,271.700 194.800,271.438 194.800 C 271.333 194.800,271.115 195.025,270.953 195.300 M127.293 201.600 C 127.293 202.370,127.375 202.685,127.476 202.300 C 127.576 201.915,127.576 201.285,127.476 200.900 C 127.375 200.515,127.293 200.830,127.293 201.600 M134.181 202.100 L 132.600 203.800 134.300 202.219 C 135.881 200.748,136.184 200.400,135.881 200.400 C 135.816 200.400,135.051 201.165,134.181 202.100 M268.800 202.600 C 269.546 203.370,270.247 204.000,270.357 204.000 C 270.467 204.000,269.946 203.370,269.200 202.600 C 268.454 201.830,267.753 201.200,267.643 201.200 C 267.533 201.200,268.054 201.830,268.800 202.600 M127.709 204.800 C 127.709 205.790,127.787 206.195,127.882 205.700 C 127.978 205.205,127.978 204.395,127.882 203.900 C 127.787 203.405,127.709 203.810,127.709 204.800 M33.591 230.700 L 30.200 234.200 33.700 230.809 C 35.625 228.944,37.200 227.369,37.200 227.309 C 37.200 227.018,36.741 227.449,33.591 230.700 M24.791 239.100 L 21.400 242.600 24.900 239.209 C 26.825 237.344,28.400 235.769,28.400 235.709 C 28.400 235.418,27.941 235.849,24.791 239.100 M293.656 244.000 C 293.656 244.330,293.811 244.960,294.000 245.400 C 294.217 245.904,294.344 245.978,294.344 245.600 C 294.344 245.270,294.189 244.640,294.000 244.200 C 293.783 243.696,293.656 243.622,293.656 244.000 M16.400 247.000 C 15.428 247.990,14.723 248.800,14.833 248.800 C 14.943 248.800,15.828 247.990,16.800 247.000 C 17.772 246.010,18.477 245.200,18.367 245.200 C 18.257 245.200,17.372 246.010,16.400 247.000 M107.902 246.200 C 108.007 246.750,108.162 247.200,108.247 247.200 C 108.331 247.200,108.400 246.750,108.400 246.200 C 108.400 245.650,108.245 245.200,108.055 245.200 C 107.866 245.200,107.797 245.650,107.902 246.200 M60.585 251.300 L 58.600 253.400 60.700 251.415 C 61.855 250.323,62.800 249.378,62.800 249.315 C 62.800 249.017,62.429 249.349,60.585 251.300 M295.687 252.000 C 295.683 252.550,295.824 253.450,296.000 254.000 C 296.431 255.344,296.431 253.892,296.000 252.200 C 295.726 251.122,295.694 251.102,295.687 252.000 M90.000 252.400 C 89.208 252.654,89.167 252.720,89.800 252.720 C 90.240 252.720,91.050 252.576,91.600 252.400 C 92.392 252.146,92.433 252.080,91.800 252.080 C 91.360 252.080,90.550 252.224,90.000 252.400 M52.800 258.600 C 51.161 260.250,49.910 261.600,50.020 261.600 C 50.130 261.600,51.561 260.250,53.200 258.600 C 54.839 256.950,56.090 255.600,55.980 255.600 C 55.870 255.600,54.439 256.950,52.800 258.600 M282.901 259.000 C 282.905 259.880,282.987 260.193,283.083 259.695 C 283.179 259.198,283.175 258.478,283.075 258.095 C 282.975 257.713,282.896 258.120,282.901 259.000 M105.693 259.200 C 105.693 259.970,105.775 260.285,105.876 259.900 C 105.976 259.515,105.976 258.885,105.876 258.500 C 105.775 258.115,105.693 258.430,105.693 259.200 M257.701 263.400 C 257.705 264.280,257.787 264.593,257.883 264.095 C 257.979 263.598,257.975 262.878,257.875 262.495 C 257.775 262.113,257.696 262.520,257.701 263.400 M297.363 270.000 C 297.363 273.630,297.423 275.115,297.498 273.300 C 297.572 271.485,297.572 268.515,297.498 266.700 C 297.423 264.885,297.363 266.370,297.363 270.000 M41.193 269.900 L 37.000 274.200 41.300 270.007 C 43.665 267.701,45.600 265.766,45.600 265.707 C 45.600 265.418,45.087 265.906,41.193 269.900 M104.946 271.800 C 104.947 274.000,105.014 274.849,105.095 273.687 C 105.176 272.524,105.176 270.724,105.093 269.687 C 105.011 268.649,104.945 269.600,104.946 271.800 M144.134 271.600 C 144.134 273.250,144.205 273.925,144.291 273.100 C 144.378 272.275,144.378 270.925,144.291 270.100 C 144.205 269.275,144.134 269.950,144.134 271.600 M119.293 271.200 C 119.293 271.970,119.375 272.285,119.476 271.900 C 119.576 271.515,119.576 270.885,119.476 270.500 C 119.375 270.115,119.293 270.430,119.293 271.200 M30.793 279.900 L 26.200 284.600 30.900 280.007 C 33.485 277.480,35.600 275.365,35.600 275.307 C 35.600 275.018,35.060 275.534,30.793 279.900 M257.701 279.400 C 257.705 280.280,257.787 280.593,257.883 280.095 C 257.979 279.598,257.975 278.878,257.875 278.495 C 257.775 278.113,257.696 278.520,257.701 279.400 M105.693 283.600 C 105.693 284.370,105.775 284.685,105.876 284.300 C 105.976 283.915,105.976 283.285,105.876 282.900 C 105.775 282.515,105.693 282.830,105.693 283.600 M17.994 292.300 L 13.000 297.400 18.100 292.406 C 22.837 287.767,23.381 287.200,23.094 287.200 C 23.036 287.200,20.741 289.495,17.994 292.300 M108.056 295.600 C 108.056 295.930,108.211 296.560,108.400 297.000 C 108.617 297.504,108.744 297.578,108.744 297.200 C 108.744 296.870,108.589 296.240,108.400 295.800 C 108.183 295.296,108.056 295.222,108.056 295.600 M390.185 314.000 C 390.185 324.230,390.236 328.415,390.300 323.300 C 390.363 318.185,390.363 309.815,390.300 304.700 C 390.236 299.585,390.185 303.770,390.185 314.000 M109.829 301.600 C 109.976 302.150,110.264 302.870,110.467 303.200 C 110.725 303.617,110.757 303.495,110.571 302.800 C 110.424 302.250,110.136 301.530,109.933 301.200 C 109.675 300.783,109.643 300.905,109.829 301.600 M289.786 308.045 C 289.607 308.610,289.756 309.106,290.237 309.545 C 290.906 310.155,290.921 310.153,290.445 309.519 C 290.164 309.145,290.051 308.470,290.194 308.019 C 290.337 307.569,290.364 307.200,290.254 307.200 C 290.144 307.200,289.933 307.580,289.786 308.045 M241.178 312.300 L 239.800 313.800 241.300 312.422 C 242.696 311.138,242.984 310.800,242.678 310.800 C 242.612 310.800,241.937 311.475,241.178 312.300 M9.762 317.400 C 9.762 320.920,9.824 322.306,9.898 320.481 C 9.972 318.656,9.972 315.776,9.897 314.081 C 9.822 312.386,9.761 313.880,9.762 317.400 M297.600 316.600 C 299.017 318.030,300.267 319.200,300.377 319.200 C 300.487 319.200,299.417 318.030,298.000 316.600 C 296.583 315.170,295.333 314.000,295.223 314.000 C 295.113 314.000,296.183 315.170,297.600 316.600 M305.993 324.500 C 308.299 326.865,310.234 328.800,310.293 328.800 C 310.582 328.800,310.094 328.287,306.100 324.393 L 301.800 320.200 305.993 324.500 M192.305 327.883 C 192.802 327.979,193.522 327.975,193.905 327.875 C 194.287 327.775,193.880 327.696,193.000 327.701 C 192.120 327.705,191.807 327.787,192.305 327.883 M208.900 327.876 C 209.285 327.976,209.915 327.976,210.300 327.876 C 210.685 327.775,210.370 327.693,209.600 327.693 C 208.830 327.693,208.515 327.775,208.900 327.876 M311.600 329.710 C 311.600 329.771,312.995 331.166,314.700 332.810 L 317.800 335.800 314.810 332.700 C 312.033 329.821,311.600 329.417,311.600 329.710 M10.109 332.000 C 10.109 332.990,10.187 333.395,10.282 332.900 C 10.378 332.405,10.378 331.595,10.282 331.100 C 10.187 330.605,10.109 331.010,10.109 332.000 M299.600 331.325 C 299.600 331.394,300.185 331.979,300.900 332.625 L 302.200 333.800 301.025 332.500 C 299.930 331.288,299.600 331.016,299.600 331.325 M389.693 334.800 C 389.693 335.570,389.775 335.885,389.876 335.500 C 389.976 335.115,389.976 334.485,389.876 334.100 C 389.775 333.715,389.693 334.030,389.693 334.800 M130.383 336.100 C 132.041 337.866,132.400 338.184,132.400 337.883 C 132.400 337.819,131.545 336.964,130.500 335.983 L 128.600 334.200 130.383 336.100 M308.000 339.400 C 309.195 340.610,310.263 341.600,310.373 341.600 C 310.483 341.600,309.595 340.610,308.400 339.400 C 307.205 338.190,306.137 337.200,306.027 337.200 C 305.917 337.200,306.805 338.190,308.000 339.400 M135.800 341.600 C 136.883 342.700,137.860 343.600,137.970 343.600 C 138.080 343.600,137.283 342.700,136.200 341.600 C 135.117 340.500,134.140 339.600,134.030 339.600 C 133.920 339.600,134.717 340.500,135.800 341.600 M324.200 342.000 C 325.059 342.880,325.852 343.600,325.962 343.600 C 326.072 343.600,325.459 342.880,324.600 342.000 C 323.741 341.120,322.948 340.400,322.838 340.400 C 322.728 340.400,323.341 341.120,324.200 342.000 M326.800 344.925 C 326.800 344.994,327.385 345.579,328.100 346.225 L 329.400 347.400 328.225 346.100 C 327.130 344.888,326.800 344.616,326.800 344.925 M237.200 346.504 C 237.200 346.561,240.755 350.116,245.100 354.404 L 253.000 362.200 245.204 354.300 C 237.960 346.960,237.200 346.219,237.200 346.504 M144.400 348.546 C 144.400 348.656,144.940 349.130,145.600 349.600 C 146.260 350.070,146.800 350.364,146.800 350.254 C 146.800 350.144,146.260 349.670,145.600 349.200 C 144.940 348.730,144.400 348.436,144.400 348.546 M14.800 351.349 C 14.800 351.781,16.389 354.800,16.616 354.800 C 16.726 354.800,16.430 354.035,15.959 353.100 C 15.032 351.261,14.800 350.911,14.800 351.349 M200.912 353.495 C 201.964 353.577,203.584 353.576,204.512 353.492 C 205.440 353.408,204.580 353.341,202.600 353.342 C 200.620 353.343,199.860 353.412,200.912 353.495 M333.800 354.400 C 334.141 355.060,334.511 355.600,334.621 355.600 C 334.731 355.600,334.541 355.060,334.200 354.400 C 333.859 353.740,333.489 353.200,333.379 353.200 C 333.269 353.200,333.459 353.740,333.800 354.400 M153.600 354.579 C 153.600 354.689,154.140 355.059,154.800 355.400 C 155.460 355.741,156.000 355.931,156.000 355.821 C 156.000 355.711,155.460 355.341,154.800 355.000 C 154.140 354.659,153.600 354.469,153.600 354.579 M242.200 366.800 C 245.494 370.100,248.280 372.800,248.390 372.800 C 248.500 372.800,245.894 370.100,242.600 366.800 C 239.306 363.500,236.520 360.800,236.410 360.800 C 236.300 360.800,238.906 363.500,242.200 366.800 M176.200 364.000 C 176.640 364.189,177.270 364.344,177.600 364.344 C 177.978 364.344,177.904 364.217,177.400 364.000 C 176.960 363.811,176.330 363.656,176.000 363.656 C 175.622 363.656,175.696 363.783,176.200 364.000 M337.693 366.800 C 337.693 367.570,337.775 367.885,337.876 367.500 C 337.976 367.115,337.976 366.485,337.876 366.100 C 337.775 365.715,337.693 366.030,337.693 366.800 M187.300 366.276 C 187.685 366.376,188.315 366.376,188.700 366.276 C 189.085 366.175,188.770 366.093,188.000 366.093 C 187.230 366.093,186.915 366.175,187.300 366.276 M213.300 366.276 C 213.685 366.376,214.315 366.376,214.700 366.276 C 215.085 366.175,214.770 366.093,214.000 366.093 C 213.230 366.093,212.915 366.175,213.300 366.276 M195.705 367.087 C 196.314 367.179,197.214 367.177,197.705 367.082 C 198.197 366.987,197.700 366.911,196.600 366.914 C 195.500 366.917,195.097 366.995,195.705 367.087 M204.700 367.082 C 205.195 367.178,206.005 367.178,206.500 367.082 C 206.995 366.987,206.590 366.909,205.600 366.909 C 204.610 366.909,204.205 366.987,204.700 367.082 M28.600 371.200 C 30.571 373.180,32.273 374.800,32.383 374.800 C 32.493 374.800,30.971 373.180,29.000 371.200 C 27.029 369.220,25.327 367.600,25.217 367.600 C 25.107 367.600,26.629 369.220,28.600 371.200 M372.191 370.900 L 369.000 374.200 372.300 371.009 C 375.365 368.046,375.782 367.600,375.491 367.600 C 375.430 367.600,373.945 369.085,372.191 370.900 M338.134 371.600 C 338.134 373.250,338.205 373.925,338.291 373.100 C 338.378 372.275,338.378 370.925,338.291 370.100 C 338.205 369.275,338.134 369.950,338.134 371.600 M264.200 374.000 C 265.728 375.540,267.069 376.800,267.179 376.800 C 267.289 376.800,266.128 375.540,264.600 374.000 C 263.072 372.460,261.731 371.200,261.621 371.200 C 261.511 371.200,262.672 372.460,264.200 374.000 M33.164 375.335 C 33.749 376.063,36.000 377.777,36.000 377.494 C 36.000 377.406,35.235 376.726,34.300 375.983 C 33.365 375.240,32.854 374.948,33.164 375.335 M337.693 376.400 C 337.693 377.170,337.775 377.485,337.876 377.100 C 337.976 376.715,337.976 376.085,337.876 375.700 C 337.775 375.315,337.693 375.630,337.693 376.400 M261.100 386.009 C 262.695 387.665,264.000 389.149,264.000 389.309 C 264.000 389.468,219.855 389.644,165.900 389.700 L 67.800 389.802 166.100 389.901 C 245.554 389.981,264.400 389.902,264.400 389.490 C 264.400 389.209,263.005 387.634,261.300 385.990 L 258.200 383.000 261.100 386.009 M47.600 384.600 C 48.150 384.914,48.780 385.172,49.000 385.172 C 49.220 385.172,48.950 384.914,48.400 384.600 C 47.850 384.286,47.220 384.028,47.000 384.028 C 46.780 384.028,47.050 384.286,47.600 384.600 M351.600 385.000 C 351.050 385.314,350.780 385.572,351.000 385.572 C 351.220 385.572,351.850 385.314,352.400 385.000 C 352.950 384.686,353.220 384.428,353.000 384.428 C 352.780 384.428,352.150 384.686,351.600 385.000 M333.867 388.267 C 333.117 389.016,333.681 389.597,335.100 389.536 L 336.600 389.472 335.268 389.308 C 334.314 389.191,333.998 388.982,334.155 388.572 C 334.393 387.953,334.289 387.844,333.867 388.267 " stroke="none" fill="#9c9c9c" fill-rule="evenodd"></path>
  </g>
</svg>
EOF
}
