#!/bin/bash

function install {
  # create service directory
  mkdir -p /srv/invoiceninja

  # create yml(s)
  {
    echo "version: '3.6'"
    echo ""
    echo "volumes:"
    echo "  db:"
    echo "  storage:"
    echo "  logo:"
    echo "  public:"
    echo ""
    echo "# uncomment if you want to use external network (example network: \"web\")"
    echo "#networks:"
    echo "#  web:"
    echo "#    external: true"
    echo ""
    echo "services:"
    echo "  db:"
    echo "    image: beercan1989/arm-mysql:5"
    echo "    env_file: .env"
    echo "    restart: always"
    echo "    volumes:"
    echo "      - db:/var/lib/mysql"
    echo "    networks:"
    echo "      - default"
    echo ""
    echo "  app:"
    echo "    image: treehouses/rpi-invoiceninja "
    echo "    env_file: .env"
    echo "    restart: always"
    echo "    depends_on:"
    echo "      - db"
    echo "    volumes:"
    echo "      -  storage:/var/www/app/storage"
    echo "      -  logo:/var/www/app/public/logo"
    echo "      -  public:/var/www/app/public"
    echo "    networks: "
    echo "      - default  "
    echo ""
    echo "  web:"
    echo "    image: nginx:1"
    echo "    volumes:"
    echo "      - ./nginx.conf:/etc/nginx/nginx.conf:ro"
    echo "      -  storage:/var/www/app/storage"
    echo "      -  logo:/var/www/app/public/logo"
    echo "      -  public:/var/www/app/public"
    echo "    expose: "
    echo "      - \"80\""
    echo "    depends_on:"
    echo "      - app"
    echo "    ports: # Delete if you want to use reverse proxy"
    echo "      - 8090:80"
    echo "    networks:"
    echo "#      - web        # uncomment if you want to use  external network (reverse proxy for example)"
    echo "      - default"
    echo ""
    echo "#  cron:"
    echo "#    image: invoiceninja/invoiceninja"
    echo "#    env_file: .env"
    echo "#    volumes:"
    echo "#      -  storage:/var/www/app/storage"
    echo "#      -  logo:/var/www/app/public/logo"
    echo "#      -  public:/var/www/app/public"
    echo "#    entrypoint: |"
    echo "#      bash -c 'bash -s <<EOF"
    echo "#      trap \"break;exit\" SIGHUP SIGINT SIGTERM"
    echo "#      sleep 300s"
    echo "#      while /bin/true; do"
    echo "#        ./artisan ninja:send-invoices"
    echo "#        ./artisan ninja:send-reminders"
    echo "#        sleep 1d"
    echo "#      done"
    echo "#      EOF'"
    echo "#    networks:"
    echo "#      - default"
  } > /srv/invoiceninja/invoiceninja.yml

  # create .env with default values
  {
    echo "MYSQL_DATABASE=ninja"
    echo "MYSQL_ROOT_PASSWORD=pwd"
    echo "APP_DEBUG=0"
    echo "APP_URL=http://localhost:8000"
    echo "APP_KEY=SomeRandomStringSomeRandomString"
    echo "APP_CIPHER=AES-256-CBC"
    echo "DB_USERNAME=root"
    echo "DB_PASSWORD=pwd"
    echo "DB_HOST=db"
    echo "DB_DATABASE=ninja"
    echo "MAIL_HOST=mail.service.host"
    echo "MAIL_USERNAME=username"
    echo "MAIL_PASSWORD=password"
    echo "MAIL_DRIVER=smtp"
    echo "MAIL_FROM_NAME=/"My name/""
    echo "MAIL_FROM_ADDRESS=user@mail.com"
  } > /srv/invoiceninja/.env

  # create nginx.conf
  {
    echo "user www-data;"
    echo ""
    echo "events {"
    echo "  worker_connections 768;"
    echo "}"
    echo ""
    echo "http {"
    echo "    upstream backend {"
    echo "        server app:9000;"
    echo "    }"
    echo "    include /etc/nginx/mime.types;"
    echo "    default_type application/octet-stream;"
    echo "    gzip on;"
    echo "    gzip_disable \"msie6\";"
    echo ""
    echo "    server {"
    echo "        listen      80 default;"
    echo "        server_name your_ininja_site;"
    echo ""
    echo "        root /var/www/app/public;"
    echo ""
    echo "        index index.php;"
    echo ""
    echo "        charset utf-8;"
    echo ""
    echo "        location / {"
    echo "            try_files \$uri \$uri/ /index.php?\$query_string;"
    echo "        }"
    echo ""
    echo "        location = /favicon.ico { access_log off; log_not_found off; }"
    echo "        location = /robots.txt  { access_log off; log_not_found off; }"
    echo ""
    echo "        sendfile off;"
    echo ""
    echo "        location ~ \.php$ {"
    echo "            fastcgi_split_path_info ^(.+\.php)(/.+)$;"
    echo "            fastcgi_pass backend;"
    echo "            fastcgi_index index.php;"
    echo "            include fastcgi_params;"
    echo "            fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;"
    echo "            fastcgi_intercept_errors off;"
    echo "            fastcgi_buffer_size 16k;"
    echo "            fastcgi_buffers 4 16k;"
    echo "        }"
    echo ""
    echo "        location ~ /\.ht {"
    echo "            deny all;"
    echo "        }"
    echo "    }"
    echo "}"
  } > /srv/invoiceninja/nginx.conf

  # add autorun
  {
    echo "invoiceninja_autorun=true"
    echo
    echo "if [ \"\$invoiceninja_autorun\" = true ]; then"
    echo "  treehouses services invoiceninja up"
    echo "fi"
    echo
    echo
  } > /srv/invoiceninja/autorun
}

# environment var
function uses_env {
  echo true
}

# add supported arch(es)
function supported_arches {
  echo "armv7l"
}

# add port(s)
function get_ports {
  echo "8090"
}

# add size (in MB)
function get_size {
  echo "1341"
}

# add description
function get_description {
  echo "Invoiceninja is the leading self-host platform to create invoices"
}

# add info
function get_info {
  echo "https://github.com/ole-vi/invoiceninja"
  echo
  echo "\"Invoiceninja is the leading self-host platform to create invoices," 
  echo "accept payments, track expenses & time tasks."
  echo "Support WePay, Stripe, Braintree, PayPal, Zapier, and more!\""
}

# add svg icon
function get_icon {
  cat <<EOF
<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<svg
   xmlns:dc="http://purl.org/dc/elements/1.1/"
   xmlns:cc="http://creativecommons.org/ns#"
   xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
   xmlns:svg="http://www.w3.org/2000/svg"
   xmlns="http://www.w3.org/2000/svg"
   xmlns:xlink="http://www.w3.org/1999/xlink"
   viewBox="0 0 485.33331 402.66666"
   height="402.66666"
   width="485.33331"
   xml:space="preserve"
   id="svg2"
   version="1.1"><metadata
     id="metadata8"><rdf:RDF><cc:Work
         rdf:about=""><dc:format>image/svg+xml</dc:format><dc:type
           rdf:resource="http://purl.org/dc/dcmitype/StillImage" /></cc:Work></rdf:RDF></metadata><defs
     id="defs6" /><g
     transform="matrix(1.3333333,0,0,-1.3333333,0,402.66666)"
     id="g10"><g
       transform="scale(0.1)"
       id="g12"><g
         transform="matrix(3640,0,0,3020,0,2.81e-6)"
         id="g14"><image
           id="image16"
           xlink:href="data:image/png;base64,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"
           transform="matrix(1,0,0,-1,0,1)"
           preserveAspectRatio="none"
           style="image-rendering:optimizeSpeed"
           height="1"
           width="1" /></g></g></g></svg>
EOF
}
