"""Test the UptimeRobot init."""
from unittest.mock import patch

from pytest import LogCaptureFixture
from pyuptimerobot import UptimeRobotAuthenticationException, UptimeRobotException

from homeassistant import config_entries
from homeassistant.components.uptimerobot.const import (
    COORDINATOR_UPDATE_INTERVAL,
    DOMAIN,
)
from homeassistant.const import STATE_ON, STATE_UNAVAILABLE
from homeassistant.core import HomeAssistant
from homeassistant.helpers.device_registry import (
    async_entries_for_config_entry,
    async_get_registry,
)
from homeassistant.util import dt

from .common import (
    MOCK_UPTIMEROBOT_CONFIG_ENTRY_DATA,
    MOCK_UPTIMEROBOT_MONITOR,
    UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY,
    MockApiResponseKey,
    mock_uptimerobot_api_response,
    setup_uptimerobot_integration,
)

from tests.common import MockConfigEntry, async_fire_time_changed


async def test_reauthentication_trigger_in_setup(
    hass: HomeAssistant, caplog: LogCaptureFixture
):
    """Test reauthentication trigger."""
    mock_config_entry = MockConfigEntry(**MOCK_UPTIMEROBOT_CONFIG_ENTRY_DATA)
    mock_config_entry.add_to_hass(hass)

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        side_effect=UptimeRobotAuthenticationException,
    ):

        await hass.config_entries.async_setup(mock_config_entry.entry_id)
        await hass.async_block_till_done()

    flows = hass.config_entries.flow.async_progress()

    assert mock_config_entry.state == config_entries.ConfigEntryState.SETUP_ERROR
    assert mock_config_entry.reason == "could not authenticate"

    assert len(flows) == 1
    flow = flows[0]
    assert flow["step_id"] == "reauth_confirm"
    assert flow["handler"] == DOMAIN
    assert flow["context"]["source"] == config_entries.SOURCE_REAUTH
    assert flow["context"]["entry_id"] == mock_config_entry.entry_id

    assert (
        "Config entry 'test@test.test' for uptimerobot integration could not authenticate"
        in caplog.text
    )


async def test_reauthentication_trigger_after_setup(
    hass: HomeAssistant, caplog: LogCaptureFixture
):
    """Test reauthentication trigger."""
    mock_config_entry = await setup_uptimerobot_integration(hass)

    binary_sensor = hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY)
    assert mock_config_entry.state == config_entries.ConfigEntryState.LOADED
    assert binary_sensor.state == STATE_ON

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        side_effect=UptimeRobotAuthenticationException,
    ):

        async_fire_time_changed(hass, dt.utcnow() + COORDINATOR_UPDATE_INTERVAL)
        await hass.async_block_till_done()

    flows = hass.config_entries.flow.async_progress()
    assert (
        hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state
        == STATE_UNAVAILABLE
    )

    assert "Authentication failed while fetching uptimerobot data" in caplog.text

    assert len(flows) == 1
    flow = flows[0]
    assert flow["step_id"] == "reauth_confirm"
    assert flow["handler"] == DOMAIN
    assert flow["context"]["source"] == config_entries.SOURCE_REAUTH
    assert flow["context"]["entry_id"] == mock_config_entry.entry_id


async def test_integration_reload(hass: HomeAssistant):
    """Test integration reload."""
    mock_entry = await setup_uptimerobot_integration(hass)

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        return_value=mock_uptimerobot_api_response(),
    ):
        assert await hass.config_entries.async_reload(mock_entry.entry_id)
        async_fire_time_changed(hass, dt.utcnow() + COORDINATOR_UPDATE_INTERVAL)
        await hass.async_block_till_done()

    entry = hass.config_entries.async_get_entry(mock_entry.entry_id)
    assert entry.state == config_entries.ConfigEntryState.LOADED
    assert hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state == STATE_ON


async def test_update_errors(hass: HomeAssistant, caplog: LogCaptureFixture):
    """Test errors during updates."""
    await setup_uptimerobot_integration(hass)

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        side_effect=UptimeRobotException,
    ):
        async_fire_time_changed(hass, dt.utcnow() + COORDINATOR_UPDATE_INTERVAL)
        await hass.async_block_till_done()
        assert (
            hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state
            == STATE_UNAVAILABLE
        )

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        return_value=mock_uptimerobot_api_response(),
    ):
        async_fire_time_changed(hass, dt.utcnow() + COORDINATOR_UPDATE_INTERVAL)
        await hass.async_block_till_done()
        assert hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state == STATE_ON

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        return_value=mock_uptimerobot_api_response(key=MockApiResponseKey.ERROR),
    ):
        async_fire_time_changed(hass, dt.utcnow() + COORDINATOR_UPDATE_INTERVAL)
        await hass.async_block_till_done()
        assert (
            hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state
            == STATE_UNAVAILABLE
        )

    assert "Error fetching uptimerobot data: test error from API" in caplog.text


async def test_device_management(hass: HomeAssistant):
    """Test that we are adding and removing devices for monitors returned from the API."""
    mock_entry = await setup_uptimerobot_integration(hass)
    dev_reg = await async_get_registry(hass)

    devices = async_entries_for_config_entry(dev_reg, mock_entry.entry_id)
    assert len(devices) == 1

    assert devices[0].identifiers == {(DOMAIN, "1234")}
    assert devices[0].name == "Test monitor"

    assert hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state == STATE_ON
    assert hass.states.get(f"{UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY}_2") is None

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        return_value=mock_uptimerobot_api_response(
            data=[MOCK_UPTIMEROBOT_MONITOR, {**MOCK_UPTIMEROBOT_MONITOR, "id": 12345}]
        ),
    ):
        async_fire_time_changed(hass, dt.utcnow() + COORDINATOR_UPDATE_INTERVAL)
        await hass.async_block_till_done()

    devices = async_entries_for_config_entry(dev_reg, mock_entry.entry_id)
    assert len(devices) == 2
    assert devices[0].identifiers == {(DOMAIN, "1234")}
    assert devices[1].identifiers == {(DOMAIN, "12345")}

    assert hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state == STATE_ON
    assert (
        hass.states.get(f"{UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY}_2").state == STATE_ON
    )

    with patch(
        "pyuptimerobot.UptimeRobot.async_get_monitors",
        return_value=mock_uptimerobot_api_response(),
    ):
        async_fire_time_changed(hass, dt.utcnow() + COORDINATOR_UPDATE_INTERVAL)
        await hass.async_block_till_done()

    devices = async_entries_for_config_entry(dev_reg, mock_entry.entry_id)
    assert len(devices) == 1
    assert devices[0].identifiers == {(DOMAIN, "1234")}

    assert hass.states.get(UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY).state == STATE_ON
    assert hass.states.get(f"{UPTIMEROBOT_BINARY_SENSOR_TEST_ENTITY}_2") is None
