"""The tests for deCONZ logbook."""

from unittest.mock import patch

from homeassistant.components import logbook
from homeassistant.components.deconz.const import CONF_GESTURE, DOMAIN as DECONZ_DOMAIN
from homeassistant.components.deconz.deconz_event import (
    CONF_DECONZ_ALARM_EVENT,
    CONF_DECONZ_EVENT,
)
from homeassistant.const import (
    CONF_CODE,
    CONF_DEVICE_ID,
    CONF_EVENT,
    CONF_ID,
    CONF_UNIQUE_ID,
    STATE_ALARM_ARMED_AWAY,
)
from homeassistant.setup import async_setup_component
from homeassistant.util import slugify

from .test_gateway import DECONZ_WEB_REQUEST, setup_deconz_integration

from tests.components.logbook.test_init import MockLazyEventPartialState


async def test_humanifying_deconz_alarm_event(hass, aioclient_mock):
    """Test humanifying deCONZ event."""
    data = {
        "sensors": {
            "1": {
                "config": {
                    "armed": "disarmed",
                    "enrolled": 0,
                    "on": True,
                    "panel": "disarmed",
                    "pending": [],
                    "reachable": True,
                },
                "ep": 1,
                "etag": "3c4008d74035dfaa1f0bb30d24468b12",
                "lastseen": "2021-04-02T13:07Z",
                "manufacturername": "Universal Electronics Inc",
                "modelid": "URC4450BC0-X-R",
                "name": "Keypad",
                "state": {
                    "action": "armed_away,1111,55",
                    "lastupdated": "2021-04-02T13:08:18.937",
                    "lowbattery": False,
                    "tampered": True,
                },
                "type": "ZHAAncillaryControl",
                "uniqueid": "00:0d:6f:00:13:4f:61:39-01-0501",
            }
        }
    }
    with patch.dict(DECONZ_WEB_REQUEST, data):
        await setup_deconz_integration(hass, aioclient_mock)

    device_registry = await hass.helpers.device_registry.async_get_registry()

    keypad_event_id = slugify(data["sensors"]["1"]["name"])
    keypad_serial = data["sensors"]["1"]["uniqueid"].split("-", 1)[0]
    keypad_entry = device_registry.async_get_device(
        identifiers={(DECONZ_DOMAIN, keypad_serial)}
    )

    hass.config.components.add("recorder")
    assert await async_setup_component(hass, "logbook", {})
    entity_attr_cache = logbook.EntityAttributeCache(hass)

    events = list(
        logbook.humanify(
            hass,
            [
                MockLazyEventPartialState(
                    CONF_DECONZ_ALARM_EVENT,
                    {
                        CONF_CODE: 1234,
                        CONF_DEVICE_ID: keypad_entry.id,
                        CONF_EVENT: STATE_ALARM_ARMED_AWAY,
                        CONF_ID: keypad_event_id,
                        CONF_UNIQUE_ID: keypad_serial,
                    },
                ),
            ],
            entity_attr_cache,
            {},
        )
    )

    assert events[0]["name"] == "Keypad"
    assert events[0]["domain"] == "deconz"
    assert events[0]["message"] == "fired event 'armed_away'."


async def test_humanifying_deconz_event(hass, aioclient_mock):
    """Test humanifying deCONZ event."""
    data = {
        "sensors": {
            "1": {
                "name": "Switch 1",
                "type": "ZHASwitch",
                "state": {"buttonevent": 1000},
                "config": {},
                "uniqueid": "00:00:00:00:00:00:00:01-00",
            },
            "2": {
                "name": "Hue remote",
                "type": "ZHASwitch",
                "modelid": "RWL021",
                "state": {"buttonevent": 1000},
                "config": {},
                "uniqueid": "00:00:00:00:00:00:00:02-00",
            },
            "3": {
                "name": "Xiaomi cube",
                "type": "ZHASwitch",
                "modelid": "lumi.sensor_cube",
                "state": {"buttonevent": 1000, "gesture": 1},
                "config": {},
                "uniqueid": "00:00:00:00:00:00:00:03-00",
            },
            "4": {
                "name": "Faulty event",
                "type": "ZHASwitch",
                "state": {},
                "config": {},
                "uniqueid": "00:00:00:00:00:00:00:04-00",
            },
        }
    }
    with patch.dict(DECONZ_WEB_REQUEST, data):
        await setup_deconz_integration(hass, aioclient_mock)

    device_registry = await hass.helpers.device_registry.async_get_registry()

    switch_event_id = slugify(data["sensors"]["1"]["name"])
    switch_serial = data["sensors"]["1"]["uniqueid"].split("-", 1)[0]
    switch_entry = device_registry.async_get_device(
        identifiers={(DECONZ_DOMAIN, switch_serial)}
    )

    hue_remote_event_id = slugify(data["sensors"]["2"]["name"])
    hue_remote_serial = data["sensors"]["2"]["uniqueid"].split("-", 1)[0]
    hue_remote_entry = device_registry.async_get_device(
        identifiers={(DECONZ_DOMAIN, hue_remote_serial)}
    )

    xiaomi_cube_event_id = slugify(data["sensors"]["3"]["name"])
    xiaomi_cube_serial = data["sensors"]["3"]["uniqueid"].split("-", 1)[0]
    xiaomi_cube_entry = device_registry.async_get_device(
        identifiers={(DECONZ_DOMAIN, xiaomi_cube_serial)}
    )

    faulty_event_id = slugify(data["sensors"]["4"]["name"])
    faulty_serial = data["sensors"]["4"]["uniqueid"].split("-", 1)[0]
    faulty_entry = device_registry.async_get_device(
        identifiers={(DECONZ_DOMAIN, faulty_serial)}
    )

    hass.config.components.add("recorder")
    assert await async_setup_component(hass, "logbook", {})
    entity_attr_cache = logbook.EntityAttributeCache(hass)

    events = list(
        logbook.humanify(
            hass,
            [
                # Event without matching device trigger
                MockLazyEventPartialState(
                    CONF_DECONZ_EVENT,
                    {
                        CONF_DEVICE_ID: switch_entry.id,
                        CONF_EVENT: 2000,
                        CONF_ID: switch_event_id,
                        CONF_UNIQUE_ID: switch_serial,
                    },
                ),
                # Event with matching device trigger
                MockLazyEventPartialState(
                    CONF_DECONZ_EVENT,
                    {
                        CONF_DEVICE_ID: hue_remote_entry.id,
                        CONF_EVENT: 2001,
                        CONF_ID: hue_remote_event_id,
                        CONF_UNIQUE_ID: hue_remote_serial,
                    },
                ),
                # Gesture with matching device trigger
                MockLazyEventPartialState(
                    CONF_DECONZ_EVENT,
                    {
                        CONF_DEVICE_ID: xiaomi_cube_entry.id,
                        CONF_GESTURE: 1,
                        CONF_ID: xiaomi_cube_event_id,
                        CONF_UNIQUE_ID: xiaomi_cube_serial,
                    },
                ),
                # Unsupported device trigger
                MockLazyEventPartialState(
                    CONF_DECONZ_EVENT,
                    {
                        CONF_DEVICE_ID: xiaomi_cube_entry.id,
                        CONF_GESTURE: "unsupported_gesture",
                        CONF_ID: xiaomi_cube_event_id,
                        CONF_UNIQUE_ID: xiaomi_cube_serial,
                    },
                ),
                # Unknown event
                MockLazyEventPartialState(
                    CONF_DECONZ_EVENT,
                    {
                        CONF_DEVICE_ID: faulty_entry.id,
                        "unknown_event": None,
                        CONF_ID: faulty_event_id,
                        CONF_UNIQUE_ID: faulty_serial,
                    },
                ),
            ],
            entity_attr_cache,
            {},
        )
    )

    assert events[0]["name"] == "Switch 1"
    assert events[0]["domain"] == "deconz"
    assert events[0]["message"] == "fired event '2000'."

    assert events[1]["name"] == "Hue remote"
    assert events[1]["domain"] == "deconz"
    assert events[1]["message"] == "'Long press' event for 'Dim up' was fired."

    assert events[2]["name"] == "Xiaomi cube"
    assert events[2]["domain"] == "deconz"
    assert events[2]["message"] == "fired event 'Shake'."

    assert events[3]["name"] == "Xiaomi cube"
    assert events[3]["domain"] == "deconz"
    assert events[3]["message"] == "fired event 'unsupported_gesture'."

    assert events[4]["name"] == "Faulty event"
    assert events[4]["domain"] == "deconz"
    assert events[4]["message"] == "fired an unknown event."
