package replaytests

import (
	"context"
	"reflect"
	"testing"

	"github.com/golang/mock/gomock"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
	commonpb "go.temporal.io/api/common/v1"
	"go.temporal.io/api/workflowservicemock/v1"

	"go.temporal.io/sdk/client"
	"go.temporal.io/sdk/converter"
	"go.temporal.io/sdk/internal"
	ilog "go.temporal.io/sdk/internal/log"
	"go.temporal.io/sdk/worker"
)

type replayTestSuite struct {
	suite.Suite
	mockCtrl *gomock.Controller
	service  *workflowservicemock.MockWorkflowServiceClient
}

func TestReplayTestSuite(t *testing.T) {
	s := new(replayTestSuite)
	suite.Run(t, s)
}

func (s *replayTestSuite) SetupTest() {
	s.mockCtrl = gomock.NewController(s.T())
	s.service = workflowservicemock.NewMockWorkflowServiceClient(s.mockCtrl)
}

func (s *replayTestSuite) TearDownTest() {
	s.mockCtrl.Finish() // assert mock’s expectations
}

func (s *replayTestSuite) TestGenerateWorkflowHistory() {
	s.T().Skip("Remove this Skip to regenerate the history.")
	c, _ := client.Dial(client.Options{
		Logger: ilog.NewDefaultLogger(),
	})
	defer c.Close()

	w := worker.New(c, "replay-test", worker.Options{})

	w.RegisterWorkflow(Workflow1)
	w.RegisterWorkflow(Workflow2)
	w.RegisterActivity(helloworldActivity)

	_ = w.Start()
	defer w.Stop()

	workflowOptions1 := client.StartWorkflowOptions{
		ID:        "replay-tests-workflow1",
		TaskQueue: "replay-test",
	}
	we1, _ := c.ExecuteWorkflow(context.Background(), workflowOptions1, Workflow1, "Workflow1")
	var res1 string
	_ = we1.Get(context.Background(), &res1)

	workflowOptions2 := client.StartWorkflowOptions{
		ID:        "replay-tests-workflow2",
		TaskQueue: "replay-test",
	}
	we2, _ := c.ExecuteWorkflow(context.Background(), workflowOptions2, Workflow2, "Workflow2")
	var res2 string
	_ = we2.Get(context.Background(), &res2)

	// Now run:
	// tctl workflow show --workflow_id replay-tests-workflow1 --of workflow1.json
	// tctl workflow show --workflow_id replay-tests-workflow2 --of workflow2.json
}

func (s *replayTestSuite) TestReplayWorkflowHistoryFromFile() {
	testFiles := []string{"workflow1.json", "workflow2.json"}
	var err error

	for _, testFile := range testFiles {
		replayer := worker.NewWorkflowReplayer()
		replayer.RegisterWorkflow(Workflow1)
		replayer.RegisterWorkflow(Workflow2)

		err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), testFile)
		require.NoError(s.T(), err, "file: %s", testFile)
	}
}

func (s *replayTestSuite) TestReplayWorkflowWithBadUnknownEvent() {
	// Test replaying a history with an unknown event that cannot be ignored fails
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(Workflow1)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "workflow_with_bad_unknown_event.json")
	require.ErrorContains(s.T(), err, "unknown history event")
}

func (s *replayTestSuite) TestReplayWorkflowWithUnknownEvent() {
	// Test replaying a history with an unknown event that can be ignored does not fails
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(Workflow1)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "workflow_with_unknown_event.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestReplayBadWorkflowHistoryFromFile() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(Workflow1)

	// Test bad history that calls the no activities
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-history.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "nondeterministic workflow definition")

	// Test bad history that calls the wrong activity
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-history-2.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "nondeterministic workflow: history event is ActivityTaskScheduled: (ActivityId:19, ActivityType:(Name:BAD ACTIVITY)")
}

func (s *replayTestSuite) TestReplayUnhandledTimerFiredWithCancel() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(TimerWf)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "unhandled_cmd_timer_cancel.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestReplayLocalActivity() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(LocalActivityWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "local-activity.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestBadReplayLocalActivity() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(LocalActivityWorkflow)

	// Test bad history that does not call any local activities
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-local-activity.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "replay workflow doesn't return the same result as the last event, resp: *workflowservice.RespondWorkflowTaskCompletedRequest")

	// Test bad history that calls two local activities
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-local-activity-2.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "nondeterministic workflow: missing replay command for MarkerRecorded")
}

func (s *replayTestSuite) TestContinueAsNewWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(ContinueAsNewWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "continue-as-new.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestBadContinueAsNewWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(ContinueAsNewWorkflow)

	// Continue as new as the wrong workflow type
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-continue-as-new.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "replay workflow doesn't return the same result")

	// Do not continue as new
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-continue-as-new-2.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "replay workflow doesn't return the same result")
}

func (s *replayTestSuite) TestUpsertMemoWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(UpsertMemoWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "upsert-memo-workflow.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestBadUpsertMemoWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(UpsertMemoWorkflow)

	// Test bad history that does not upsert a memo after the activity.
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-upsert-memo-workflow.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "nondeterministic workflow: extra replay command for ModifyWorkflowProperties")

	// Test bad history that does not upsert a memo before the activity.
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-upsert-memo-workflow-2.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "lookup failed for scheduledEventID to activityID: scheduleEventID: 5")
}

func (s *replayTestSuite) TestSearchAttributeWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(UpsertSearchAttributesWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "search-attribute.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestBadSearchAttributeWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(UpsertSearchAttributesWorkflow)

	// Test bad history that does not upsert a search attributes after the activity.
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-search-attribute.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "nondeterministic workflow: extra replay command for UpsertWorkflowSearchAttributes")

	// Test bad history that does not upsert a search attributes before the activity.
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-search-attribute-2.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "lookup failed for scheduledEventID to activityID: scheduleEventID: 5")
}

func (s *replayTestSuite) TestSideEffectWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(SideEffectWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "side-effect.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestBadEmptyWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(EmptyWorkflow)

	// Test a long history on a workflow that just immediately returns.
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "bad-empty-workflow.json")
	require.Error(s.T(), err)
	require.Contains(s.T(), err.Error(), "lookup failed for scheduledEventID to activityID: scheduleEventID: 7")
}

// TestMutableSideEffectLegacyWorkflow test that old, nondeterministic, mutable side effect behaviour
// was not changed and is still broken in the same way.
func (s *replayTestSuite) TestMutableSideEffectLegacyWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(MutableSideEffectWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "mutable-side-effect-legacy.json")
	require.NoError(s.T(), err)
	var result []int
	require.NoError(s.T(), replayer.(*internal.WorkflowReplayer).GetWorkflowResult("ReplayId", &result))
	require.Equal(s.T(), []int{2, 2, 2, 2, 2, 2, 4, 4, 4, 5, 5}, result)
}

func (s *replayTestSuite) TestMutableSideEffectWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(MutableSideEffectWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "mutable-side-effect.json")
	require.NoError(s.T(), err)
	var result []int
	require.NoError(s.T(), replayer.(*internal.WorkflowReplayer).GetWorkflowResult("ReplayId", &result))
	require.Equal(s.T(), []int{0, 0, 0, 1, 1, 2, 3, 3, 4, 4, 5}, result)
}

func (s *replayTestSuite) TestDuplciateChildWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(DuplicateChildWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "duplicate-child-workflow.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestVersionLoopWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(VersionLoopWorkflow)
	// Verify we can still replay an old workflow that does not have sdk flags
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "version-loop-workflow-legacy-10.json")
	require.NoError(s.T(), err)

	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "version-loop-workflow-10.json")
	require.NoError(s.T(), err)

	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "version-loop-workflow-256.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestVersionLoopWorkflowTaskWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(VersionLoopWorkflowMultipleTasks)
	// Verify we can replay a workflow with SDK flags and multiple workflow tasks
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "workflow_loop_task.json")
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestUnkownSDKFlag() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(VersionLoopWorkflow)
	// version-loop-workflow-unkown-version-flag.json had a very high sdk flag value set that the sdk does not know about.
	// Verify if the SDK does not understand a flag we fail replay.
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "version-loop-workflow-unkown-version-flag.json")
	require.Error(s.T(), err)
}

func (s *replayTestSuite) TestUpdateWorkflow() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(UpdateWorkflow)
	replayer.RegisterWorkflow(UpdateAndExit)

	for _, name := range [...]string{
		"update_before_protocol_message_command",
		"update_with_protocol_message_command",
		// accept and complete update _and_  complete workflow in same wft
		"update_accept_and_complete_and_exit",
		// accept in wft then complete both update and workflow in next wft
		"update_complete_and_exit",
	} {
		s.Run(name, func() {
			err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), name+".json")
			require.NoError(s.T(), err)
		})
	}
}

func (s *replayTestSuite) TestNonDeterministicUpdate() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(NonDeterministicUpdate)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "non_deterministic_update.json")
	s.Error(err)
}

func TestReplayCustomConverter(t *testing.T) {
	conv := &captureConverter{DataConverter: converter.GetDefaultDataConverter()}
	replayer, err := worker.NewWorkflowReplayerWithOptions(worker.WorkflowReplayerOptions{
		DataConverter: conv,
	})
	require.NoError(t, err)
	replayer.RegisterWorkflow(Workflow1)
	replayer.RegisterWorkflow(Workflow2)

	// Run workflow 1
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "workflow1.json")
	require.NoError(t, err)
	// Confirm 3 activity inputs and outputs
	require.Subset(t, conv.toPayloads, []string{"Workflow1", "Workflow1", "Workflow1"})
	require.Subset(t, conv.fromPayloads, []string{"Hello Workflow1!", "Hello Workflow1!", "Hello Workflow1!"})

	// Run workflow 2
	conv.toPayloads, conv.fromPayloads = nil, nil
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "workflow2.json")
	require.NoError(t, err)
	// Confirm 1 activity input and output
	require.Contains(t, conv.toPayloads, "Workflow2")
	require.Contains(t, conv.fromPayloads, "Hello Workflow2!")
}

func TestReplayDeadlockDetection(t *testing.T) {
	defaultReplayer := worker.NewWorkflowReplayer()
	noDeadlockReplayer, err := worker.NewWorkflowReplayerWithOptions(worker.WorkflowReplayerOptions{
		DisableDeadlockDetection: true,
	})
	require.NoError(t, err)

	defaultReplayer.RegisterWorkflow(DeadlockedWorkflow)
	noDeadlockReplayer.RegisterWorkflow(DeadlockedWorkflow)

	err = defaultReplayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "deadlocked-workflow.json")
	require.Error(t, err)

	err = noDeadlockReplayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "deadlocked-workflow.json")
	require.NoError(t, err)
}

func (s *replayTestSuite) TestVersionAndMutableSideEffect() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(VersionAndMutableSideEffectWorkflow)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "replay-tests-version-and-mutable-side-effect.json")
	s.NoError(err)
}

func (s *replayTestSuite) TestCancelOrder() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(CancelOrderSelectWorkflow)

	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "replay-tests-cancel-order.json")
	s.NoError(err)

	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "replay-tests-cancel-order-timer-resolved.json")
	s.NoError(err)
}

func (s *replayTestSuite) TestChildWorkflowCancelWithUpdate() {
	// Test that update requests are still replayed successfully
	// when commands are generated before the update requests are handled
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(ChildWorkflowCancelWithUpdate)
	replayer.RegisterWorkflow(ChildWorkflowWaitOnSignal)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "child-workflow-cancel-with-update.json")
	s.NoError(err)
}

func (s *replayTestSuite) TestMultipleUpdates() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(MultipleUpdateWorkflow)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "multiple-updates.json")
	s.NoError(err)
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "multiple-updates-canceled.json")
	s.NoError(err)
}

func (s *replayTestSuite) TestResetWithUpdateAccepted() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(CounterWorkflow)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "update-reset-accepted.json")
	s.NoError(err)
}

func (s *replayTestSuite) TestResetWithUpdateRejected() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(CounterWorkflow)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "update-reset-rejected.json")
	s.NoError(err)
}

func (s *replayTestSuite) TestGogoprotoPayloadWorkflow() {
	conv := converter.NewCompositeDataConverter(
		converter.NewNilPayloadConverter(),
		converter.NewByteSlicePayloadConverter(),
		converter.NewProtoJSONPayloadConverterWithOptions(converter.ProtoJSONPayloadConverterOptions{
			LegacyTemporalProtoCompat: true,
		}),
		converter.NewProtoPayloadConverter(),
		converter.NewJSONPayloadConverter(),
	)
	replayer, err := worker.NewWorkflowReplayerWithOptions(worker.WorkflowReplayerOptions{
		DataConverter: conv,
	})
	s.NoError(err)
	replayer.RegisterWorkflow(ListAndDescribeWorkflow)
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "gogoproto-payload-workflow.json")
	s.NoError(err)
}

func (s *replayTestSuite) TestSelectorBlockingDefault() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(SelectorBlockingDefaultWorkflow)
	// Verify we can still replay an old workflow that does
	// not have the SDKFlagBlockedSelectorSignalReceive flag
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "selector-blocking-default.json")
	s.NoError(err)
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestSelectorNonBlocking() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(SelectorBlockingDefaultWorkflow)
	// Verify we can replay the new workflow that has the
	// SDKFlagBlockedSelectorSignalReceive flag
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "selector-non-blocking.json")
	s.NoError(err)
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestPartialReplayNonCommandEvent() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(TripWorkflow)
	// Verify we can replay partial history that has ended on a non-command event
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "partial-replay-non-command-event.json")
	s.NoError(err)
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestResetWorkflowBeforeChildInit() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(ResetWorkflowWithChild)
	// Verify we can replay workflow history containing a reset before StartChildWorkflowExecutionInitiated & ChildWorkflowExecutionCompleted events.
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "reset-workflow-before-child-init.json")
	s.NoError(err)
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestResetWorkflowAfterChildComplete() {
	replayer := worker.NewWorkflowReplayer()
	replayer.RegisterWorkflow(ResetWorkflowWithChild)
	// Verify we can replay workflow history containing a reset event after StartChildWorkflowExecutionInitiated & ChildWorkflowExecutionCompleted events.
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "reset-workflow-after-child-complete.json")
	s.NoError(err)
	require.NoError(s.T(), err)
}

func (s *replayTestSuite) TestCancelNexusOperation() {
	replayer := worker.NewWorkflowReplayer()

	replayer.RegisterWorkflow(CancelNexusOperationBeforeSentWorkflow)
	err := replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "nexus-cancel-before-sent.json")
	s.NoErrorf(err, "Encountered error replaying cancel before schedule Nexus operation command is sent")

	replayer.RegisterWorkflow(CancelNexusOperationBeforeStartWorkflow)
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "nexus-cancel-before-start.json")
	s.NoErrorf(err, "Encountered error replaying cancel before Nexus operation is started")

	replayer.RegisterWorkflow(CancelNexusOperationAfterStartWorkflow)
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "nexus-cancel-after-start.json")
	s.NoErrorf(err, "Encountered error replaying cancel after Nexus operation is started")

	replayer.RegisterWorkflow(CancelNexusOperationAfterCompleteWorkflow)
	err = replayer.ReplayWorkflowHistoryFromJSONFile(ilog.NewDefaultLogger(), "nexus-cancel-after-complete.json")
	s.NoErrorf(err, "Encountered error replaying cancel after Nexus operation is completed")
}

type captureConverter struct {
	converter.DataConverter
	toPayloads   []interface{}
	fromPayloads []interface{}
}

func (c *captureConverter) ToPayloads(value ...interface{}) (*commonpb.Payloads, error) {
	c.toPayloads = append(c.toPayloads, value...)
	return c.DataConverter.ToPayloads(value...)
}

func (c *captureConverter) FromPayloads(payloads *commonpb.Payloads, valuePtrs ...interface{}) error {
	// Call then get pointers
	err := c.DataConverter.FromPayloads(payloads, valuePtrs...)
	for _, v := range valuePtrs {
		c.fromPayloads = append(c.fromPayloads, reflect.ValueOf(v).Elem().Interface())
	}
	return err
}
