import type { LanguageServicePlugin } from '@volar/language-service';
import type * as vscode from 'vscode-languageserver-protocol';
import { getEmbeddedInfo } from './utils.js';

/**
 * Provides symbols for `<template>` tags within .gts/.gjs files.
 * This among other things faciliates code folding of the `<template>` region.
 *
 * Note that this ONLY provides symbols for the `<template>` tag and none of the
 * contents of the template tag. Symbols inside or outside the template tag are actually
 * generated by the syntactic TS LanguageServicePlugin (search `createTypeScriptSyntacticPlugin`).
 */
export function create(): LanguageServicePlugin {
  return {
    name: 'g-template-symbols',
    capabilities: {
      documentSymbolProvider: true,
    },
    create(context) {
      return {
        provideDocumentSymbols(document) {
          const virtualCode = getEmbeddedInfo(context, document, 'gts')?.root;

          if (!virtualCode) {
            return;
          }

          const result: vscode.DocumentSymbol[] = [];
          const { transformedModule } = virtualCode;

          if (transformedModule) {
            const templateSymbols = transformedModule.templateSymbols();
            for (const templateSymbol of templateSymbols) {
              result.push({
                name: 'template',
                kind: 2 satisfies typeof vscode.SymbolKind.Module,
                range: {
                  start: document.positionAt(templateSymbol.start),
                  end: document.positionAt(templateSymbol.end),
                },
                selectionRange: {
                  start: document.positionAt(templateSymbol.start),
                  end: document.positionAt(templateSymbol.startTagEnd),
                },
              });
            }
          }

          return result;
        },
      };
    },
  };
}
