"use strict";

/**
 *	AES Symmetric Encryption Class
 *	Copyright (c) 2013, Jeff Lyon. (http://rubbingalcoholic.com)
 * 
 *	Licensed under The MIT License. (http://www.opensource.org/licenses/mit-license.php)
 *	Redistributions of files must retain the above copyright notice.
 *
 *	@class
 *	@classdesc				Implements the AES symmetric encryption algorithm (specified in FIPS 197 [{@link http://csrc.nist.gov/publications/fips/fips197/fips-197.pdf}])
 *	@extends				BlockCipher
 *	@requires				convert
 *
 *	@author					Jeff Lyon <jeff@rubbingalcoholic.com>
 *	@copyright				Copyright (c) 2013, Jeff Lyon.
 *	@license				{@link http://www.opensource.org/licenses/mit-license.php|The MIT License}
 *
 *	@desc					Creates a new AES instance
 *	@param {Object} data	Initialization options for the class, passed automatically into {@link AES#initialize}
 */
var AES = new Class(
/** @lends AES.prototype */
{
	Extends: BlockCipher,

	/**
	 *	Internal state for the key (not-expanded). Can be 16, 24 or 32 bytes.
	 *	@private
	 */
	key: 			[],				// 16 byte binstring

	/**
	 *	Internal state for the expanded key
	 *	@private
	 */
	key_expanded: 	[],

	/**
	 *	Internal state for the key length in bits. Can be overridden by {@link AES#set_key}
	 *	@private
	 */
	key_length: 	256,			// 128 | 192 | 256 (can be overridden by set key)

	/**
	 *	Internal state for the block size in bits.
	 *	@private
	 */
	block_size: 	128,			// 128 bits
	
	/**
	 *	Controls whether to log debug output to the console.
	 *	@override
	 *	@type {Boolean}
	 *  @default false
	 *	@see {@link BlockCipher#debug_mode}
	 */
	debug_mode: false,

	/**
	 *	Called automatically on class instantiation.
	 *	Invokes {@link BlockCipher#initialize} before handling class-specific functionality.
	 *
	 *	@override
	 *	@param {Object} data See {@link BlockCipher#initialize} for a list of supported properties.
	 *	@return {AES}
	 */
	initialize: function(data)
	{
		this.parent(data);
	
		if (this.key.length)
			this.set_key(this.key);

		return this;
	},

	/**
	 *	Set the key, either from a string or an array of byte values
	 *
	 *	@throws						Will throw an error if the key is not 128, 192, or 256 bits
	 *
	 *	@param {string|Array} key	The symmetric key. Must be ASCII-encoded binary string or Array.
	 *	@return {AES} 				This AES instance (chainable)
	 */
	set_key: function(key)
	{
		if (typeof key == 'string')
			key = convert.to_bytes(key)

		if (key.length != 16 && key.length != 24 && key.length != 32)
			throw new Error('AES key must be 16, 24 or 32 bytes!');

		this.key			= key;
		this.key_expanded	= [];
		this.key_length		= (this.key.length * 8);
		
		return this;
	},

	/**
	 *	Getter function for the key
	 *	@return {Array} The byte array for the key
	 */
	get_key: function()
	{
		return this.key;
	},

	/**
	 *	Getter function for the key length
	 *	@return {Number} The key length (in bits)
	 */
	get_key_length: function()
	{
		return this.key_length;
	},

	/**
	 *	Getter function for the block size
	 *	@return {Number} The block size (in bits)
	 */
	get_block_size: function()
	{
		return this.block_size;
	},

	/**
	 *	Encrypts a block. This is normally called internally by a subclass instance of {@link BlockCipherMode}.
	 *
	 *	@private
	 *	@param {Array} state	An array of 32-bit words
	 *	@return {Array}			An array of encrypted 32-bit words
	 */
	block_encrypt: function(state)
	{
		this.debug_write('Encrypting...');

		var k 		= 	this.get_expanded_key();
		state 		= 	this.add_round_key(state, k.slice(0, 4));

		for (var i = 0; i < (this.key.length / 4) + 5; i++)
			state	= 	this.add_round_key(this.mix_columns(this.shift_rows(this.bytes_sub(state))), k.slice((i+1)*4, ((i+1)*4)+4));

		state	= this.add_round_key(this.shift_rows(this.bytes_sub(state)), k.slice((i+1)*4, ((i+1)*4)+4));

		return state;
	},

	/**
	 *	Decrypts a block. This is normally called internally by a subclass instance of {@link BlockCipherMode}.
	 *
	 *	@private
	 *	@param {Array} state	An array of 32-bit words
	 *	@return {Array}			An array of decrypted 32-bit words
	 */
	block_decrypt: function(state)
	{
		this.debug_write('Decrypting...');

		var k 		= 	this.get_expanded_key();
		var ki		= 	k.length;

		state 		= 	this.add_round_key(state, k.slice(ki-4, ki));

		for (var i = (this.key.length / 4) + 5, ki = ki - 4; i > 0; i--, ki -= 4)
			state 	= 	this.inverse_mix_columns(this.add_round_key(this.inverse_bytes_sub(this.inverse_shift_rows(state)), k.slice(ki-4, ki)));
		
		state = this.add_round_key(this.inverse_bytes_sub(this.inverse_shift_rows(state)), k.slice(0, 4));

		return state;
	},

	/**
	 *	Adds the stupid round key
	 *	@private
	 *	@param {Array} state		Word array for the current state
	 *	@param {Array} round_key	Word array for the round key
	 *	@return {Array}				Word array of XOR'ed result
	 */
	add_round_key: function(state, round_key)
	{
		return [
			state[0] ^ round_key[0],
			state[1] ^ round_key[1],
			state[2] ^ round_key[2],
			state[3] ^ round_key[3]
		];
	},

	/**
	 *	This simulates Galois Field multiplication
	 *	It does matrix multiplication with a lookup table
	 *	@private
	 *	@param {Array} state	Word array for the current state
	 *	@return {Array}			Word array
	 */
	mix_columns: function(state)
	{
		var m2 = this.mult_2, m3 = this.mult_3, out = [];

		for (var i = 0; i < 4; i++)
		{
			var a0 = (state[i] >>> 24) & 255;
			var a1 = (state[i] >>> 16) & 255;
			var a2 = (state[i] >>> 8) & 255;
			var a3 = (state[i] & 255);
			
			out[i]	= (( m2[a0] ^ m3[a1] ^ a2     ^ a3)     << 24)
					| (( a0     ^ m2[a1] ^ m3[a2] ^ a3)     << 16)
					| (( a0     ^ a1     ^ m2[a2] ^ m3[a3]) << 8)
					| (( m3[a0] ^ a1     ^ a2     ^ m2[a3]) << 0)
		}
		return out;
	},

	/**
	 *	This is the inverse of mix_columns
	 *	It uses a lookup table for the inverted matrix used previously
	 *	@private
	 *	@param {Array} state	Word array for the current state
	 *	@return {Array}			Word array
	 */
	inverse_mix_columns: function(state)
	{
		var m9 = this.mult_9, mb = this.mult_11, md = this.mult_13, me = this.mult_14, out = [];

		for (var i = 0; i < 4; i++)
		{
			var a0 = (state[i] >>> 24) & 255;
			var a1 = (state[i] >>> 16) & 255;
			var a2 = (state[i] >>> 8) & 255;
			var a3 = (state[i] & 255);
			
			out[i]	= (( me[a0] ^ mb[a1] ^ md[a2] ^ m9[a3]) << 24)
					| (( m9[a0] ^ me[a1] ^ mb[a2] ^ md[a3]) << 16)
					| (( md[a0] ^ m9[a1] ^ me[a2] ^ mb[a3]) << 8)
					| (( mb[a0] ^ md[a1] ^ m9[a2] ^ me[a3]) << 0)
		}
		return out;
	},

	/**
	 *	Shuffles bytes between the 4 32-bit integers that make up the state
	 *	@private
	 *	@param {Array} state	Word array for the current state
	 *	@return {Array}			Word array
	 */
	shift_rows: function(state)
	{
		return [
			(state[3] & 255) | (((state[2] >>> 8) & 255) << 8) | (((state[1] >>> 16) & 255) << 16) | (((state[0] >>> 24) & 255) << 24),
			(state[0] & 255) | (((state[3] >>> 8) & 255) << 8) | (((state[2] >>> 16) & 255) << 16) | (((state[1] >>> 24) & 255) << 24),
			(state[1] & 255) | (((state[0] >>> 8) & 255) << 8) | (((state[3] >>> 16) & 255) << 16) | (((state[2] >>> 24) & 255) << 24),
			(state[2] & 255) | (((state[1] >>> 8) & 255) << 8) | (((state[0] >>> 16) & 255) << 16) | (((state[3] >>> 24) & 255) << 24)
		];
	},

	/**
	 *	(Inverse of shift_rows)
	 *	Unshuffles bytes between the 4 32-bit integers that make up the state
	 *	@private
	 *	@param {Array} state	Word array for the current state
	 *	@return {Array}			Word array
	 */
	inverse_shift_rows: function(state)
	{
		return [
			(state[1] & 255) | (((state[2] >>> 8) & 255) << 8) | (((state[3] >>> 16) & 255) << 16) | (((state[0] >>> 24) & 255) << 24),
			(state[2] & 255) | (((state[3] >>> 8) & 255) << 8) | (((state[0] >>> 16) & 255) << 16) | (((state[1] >>> 24) & 255) << 24),
			(state[3] & 255) | (((state[0] >>> 8) & 255) << 8) | (((state[1] >>> 16) & 255) << 16) | (((state[2] >>> 24) & 255) << 24),
			(state[0] & 255) | (((state[1] >>> 8) & 255) << 8) | (((state[2] >>> 16) & 255) << 16) | (((state[3] >>> 24) & 255) << 24)
		];
	},

	/**
	 *	Does an s-box lookup replacement on each byte of the state
	 *	@private
	 *	@param {Array} state	Word array for the current state
	 *	@return {Array}			Word array
	 */
	bytes_sub: function(state)
	{
		var s		= this.s_e;

		return [
			(s[state[0] >>> 24] << 24) | (s[(state[0] >>> 16) & 255] << 16) | (s[(state[0] >>> 8) & 255] << 8) | s[state[0] & 255],
			(s[state[1] >>> 24] << 24) | (s[(state[1] >>> 16) & 255] << 16) | (s[(state[1] >>> 8) & 255] << 8) | s[state[1] & 255],
			(s[state[2] >>> 24] << 24) | (s[(state[2] >>> 16) & 255] << 16) | (s[(state[2] >>> 8) & 255] << 8) | s[state[2] & 255],
			(s[state[3] >>> 24] << 24) | (s[(state[3] >>> 16) & 255] << 16) | (s[(state[3] >>> 8) & 255] << 8) | s[state[3] & 255]
		];
	},

	/**
	 *	Does an inverse s-box lookup replacement on each byte of the state for decryption
	 *	@private
	 *	@param {Array} state	Word array for the current state
	 *	@return {Array}			Word array
	 */
	inverse_bytes_sub: function(state)
	{
		var s		= this.s_d;

		return [
			(s[state[0] >>> 24] << 24) | (s[(state[0] >>> 16) & 255] << 16) | (s[(state[0] >>> 8) & 255] << 8) | s[state[0] & 255],
			(s[state[1] >>> 24] << 24) | (s[(state[1] >>> 16) & 255] << 16) | (s[(state[1] >>> 8) & 255] << 8) | s[state[1] & 255],
			(s[state[2] >>> 24] << 24) | (s[(state[2] >>> 16) & 255] << 16) | (s[(state[2] >>> 8) & 255] << 8) | s[state[2] & 255],
			(s[state[3] >>> 24] << 24) | (s[(state[3] >>> 16) & 255] << 16) | (s[(state[3] >>> 8) & 255] << 8) | s[state[3] & 255]
		];
	},

	/**
	 *	Performs the stupid key expansion routine
	 *	@private
	 *	@throws	Throws an error if key is missing or invalid
	 *	@return {Array} The expanded key
	 */
	get_expanded_key: function()
	{
		if (this.key_expanded.length)
			return this.key_expanded;

		if (this.key.length != 32 && this.key.length != 24 && this.key.length != 16)
			throw new Error('Missing or invalid key!');

		this.debug_write('Generating key schedule...');

		var w	= [];
		var n_k	= (this.key.length / 4);
		var n_r = n_k + 6;
		var n_b = 4;

		for (var i = 0; i < n_k; i++)
			w[i] = convert.to_word(this.key[4*i], this.key[4*i+1], this.key[4*i+2], this.key[4*i+3]);

		// It was nice of the FIPS-197 spec to give pseudo code that actually works
		for (i = i; i < n_b * (n_r + 1); i++)
		{
			var temp = w[i-1];
			
			if (i % n_k == 0)
				temp = this.word_bytes_sub(this.rot_word(temp)) ^ this.rcon[(i/n_k)-1];
			else if (n_k > 6 && i % n_k == 4)
				temp = this.word_bytes_sub(temp);
			
			w[i] = w[i-n_k] ^ temp;
		}

		this.key_expanded = w;

		return w;
	},

	/**
	 *	Does an s-box lookup replacement on each byte of a word
	 *	@private
	 *	@param {Number} word	32-bit word
	 *	@return {Number}		S-box'ed word
	 */
	word_bytes_sub: function(word)
	{
		var s = this.s_e;
		return (s[word >>> 24] << 24) | (s[(word >>> 16) & 255] << 16) | (s[(word >>> 8) & 255] << 8) | s[word & 255];
	},

	/**
	 *	rot_word function used in key generation (does a circular left shift on a 32-bit word)
	 *	@private
	 *	@param {Number} word	32-bit word
	 *	@return {Number}		Rotated word
	 */
	rot_word: function(word)
	{
		return (word << 8) | (word >>> 24);
	},

	/**
	 *	debug.writes a pretty formatted table containing the state values
	 *	only outputs anything if this.debug_mode is true.
	 *
	 *	@private
	 *	@see 					{@link BlockCipher#debug_mode}
	 *	@param {Array} state	The state array
	 */
	debug_dump_state: function(state)
	{
		var hex = function(num)
		{
			var str = (num).toString(16)
			if (str.length != 2)
				str = '0' + str;
			return str;
		};
		var pad32 = function(num)
		{
			var str = (num >>> 0).toString(16);
			for (var i=0; str.length % 8 != 0; i++)
				var str = '0' + str;
			return str;
		};
		this.debug_write('----\n'
			+	pad32(state[0]) + pad32(state[1]) + pad32(state[2]) + pad32(state[3]) + '\n'
			+	hex((state[0] >>> 24) & 255) + ' ' + hex((state[1] >>> 24) & 255) + ' ' + hex((state[2] >>> 24) & 255) + ' ' + hex((state[3] >>> 24) & 255) + '\n'
			+	hex((state[0] >>> 16) & 255) + ' ' + hex((state[1] >>> 16) & 255) + ' ' + hex((state[2] >>> 16) & 255) + ' ' + hex((state[3] >>> 16) & 255) + '\n'
			+	hex((state[0] >>> 8) & 255)  + ' ' + hex((state[1] >>> 8) & 255)  + ' ' + hex((state[2] >>> 8) & 255)  + ' ' + hex((state[3] >>> 8) & 255) + '\n'
			+	hex((state[0]) & 255)        + ' ' + hex((state[1]) & 255)        + ' ' + hex((state[2]) & 255)        + ' ' + hex((state[3]) & 255) + '\n----'
			);
	},

	/**
	 *	RCON array used in Key Generation
	 *	@private
	 */
	rcon: [
		0x01000000,
		0x02000000,
		0x04000000,
		0x08000000,
		0x10000000,
		0x20000000,
		0x40000000,
		0x80000000,
		0x1B000000,
		0x36000000,
		0x6C000000,
		0xD8000000,
		0xAB000000,
		0x4D000000,
		0x9A000000
	],

	/**
	 *	s-box for encryption
	 *	@private
	 */
	s_e: [
		0x63, 0x7C, 0x77, 0x7B, 0xF2, 0x6B, 0x6F, 0xC5, 0x30, 0x01, 0x67, 0x2B, 0xFE, 0xD7, 0xAB, 0x76,
		0xCA, 0x82, 0xC9, 0x7D, 0xFA, 0x59, 0x47, 0xF0, 0xAD, 0xD4, 0xA2, 0xAF, 0x9C, 0xA4, 0x72, 0xC0,
		0xB7, 0xFD, 0x93, 0x26, 0x36, 0x3F, 0xF7, 0xCC, 0x34, 0xA5, 0xE5, 0xF1, 0x71, 0xD8, 0x31, 0x15,
		0x04, 0xC7, 0x23, 0xC3, 0x18, 0x96, 0x05, 0x9A, 0x07, 0x12, 0x80, 0xE2, 0xEB, 0x27, 0xB2, 0x75,
		0x09, 0x83, 0x2C, 0x1A, 0x1B, 0x6E, 0x5A, 0xA0, 0x52, 0x3B, 0xD6, 0xB3, 0x29, 0xE3, 0x2F, 0x84,
		0x53, 0xD1, 0x00, 0xED, 0x20, 0xFC, 0xB1, 0x5B, 0x6A, 0xCB, 0xBE, 0x39, 0x4A, 0x4C, 0x58, 0xCF,
		0xD0, 0xEF, 0xAA, 0xFB, 0x43, 0x4D, 0x33, 0x85, 0x45, 0xF9, 0x02, 0x7F, 0x50, 0x3C, 0x9F, 0xA8,
		0x51, 0xA3, 0x40, 0x8F, 0x92, 0x9D, 0x38, 0xF5, 0xBC, 0xB6, 0xDA, 0x21, 0x10, 0xFF, 0xF3, 0xD2,
		0xCD, 0x0C, 0x13, 0xEC, 0x5F, 0x97, 0x44, 0x17, 0xC4, 0xA7, 0x7E, 0x3D, 0x64, 0x5D, 0x19, 0x73,
		0x60, 0x81, 0x4F, 0xDC, 0x22, 0x2A, 0x90, 0x88, 0x46, 0xEE, 0xB8, 0x14, 0xDE, 0x5E, 0x0B, 0xDB,
		0xE0, 0x32, 0x3A, 0x0A, 0x49, 0x06, 0x24, 0x5C, 0xC2, 0xD3, 0xAC, 0x62, 0x91, 0x95, 0xE4, 0x79,
		0xE7, 0xC8, 0x37, 0x6D, 0x8D, 0xD5, 0x4E, 0xA9, 0x6C, 0x56, 0xF4, 0xEA, 0x65, 0x7A, 0xAE, 0x08,
		0xBA, 0x78, 0x25, 0x2E, 0x1C, 0xA6, 0xB4, 0xC6, 0xE8, 0xDD, 0x74, 0x1F, 0x4B, 0xBD, 0x8B, 0x8A,
		0x70, 0x3E, 0xB5, 0x66, 0x48, 0x03, 0xF6, 0x0E, 0x61, 0x35, 0x57, 0xB9, 0x86, 0xC1, 0x1D, 0x9E,
		0xE1, 0xF8, 0x98, 0x11, 0x69, 0xD9, 0x8E, 0x94, 0x9B, 0x1E, 0x87, 0xE9, 0xCE, 0x55, 0x28, 0xDF,
		0x8C, 0xA1, 0x89, 0x0D, 0xBF, 0xE6, 0x42, 0x68, 0x41, 0x99, 0x2D, 0x0F, 0xB0, 0x54, 0xBB, 0x16
	],

	/**
	 *	s-box for decryption
	 *	@private
	 */
	s_d: [
		0x52, 0x09, 0x6A, 0xD5, 0x30, 0x36, 0xA5, 0x38, 0xBF, 0x40, 0xA3, 0x9E, 0x81, 0xF3, 0xD7, 0xFB,
		0x7C, 0xE3, 0x39, 0x82, 0x9B, 0x2F, 0xFF, 0x87, 0x34, 0x8E, 0x43, 0x44, 0xC4, 0xDE, 0xE9, 0xCB,
		0x54, 0x7B, 0x94, 0x32, 0xA6, 0xC2, 0x23, 0x3D, 0xEE, 0x4C, 0x95, 0x0B, 0x42, 0xFA, 0xC3, 0x4E,
		0x08, 0x2E, 0xA1, 0x66, 0x28, 0xD9, 0x24, 0xB2, 0x76, 0x5B, 0xA2, 0x49, 0x6D, 0x8B, 0xD1, 0x25,
		0x72, 0xF8, 0xF6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xD4, 0xA4, 0x5C, 0xCC, 0x5D, 0x65, 0xB6, 0x92,
		0x6C, 0x70, 0x48, 0x50, 0xFD, 0xED, 0xB9, 0xDA, 0x5E, 0x15, 0x46, 0x57, 0xA7, 0x8D, 0x9D, 0x84,
		0x90, 0xD8, 0xAB, 0x00, 0x8C, 0xBC, 0xD3, 0x0A, 0xF7, 0xE4, 0x58, 0x05, 0xB8, 0xB3, 0x45, 0x06,
		0xD0, 0x2C, 0x1E, 0x8F, 0xCA, 0x3F, 0x0F, 0x02, 0xC1, 0xAF, 0xBD, 0x03, 0x01, 0x13, 0x8A, 0x6B,
		0x3A, 0x91, 0x11, 0x41, 0x4F, 0x67, 0xDC, 0xEA, 0x97, 0xF2, 0xCF, 0xCE, 0xF0, 0xB4, 0xE6, 0x73,
		0x96, 0xAC, 0x74, 0x22, 0xE7, 0xAD, 0x35, 0x85, 0xE2, 0xF9, 0x37, 0xE8, 0x1C, 0x75, 0xDF, 0x6E,
		0x47, 0xF1, 0x1A, 0x71, 0x1D, 0x29, 0xC5, 0x89, 0x6F, 0xB7, 0x62, 0x0E, 0xAA, 0x18, 0xBE, 0x1B,
		0xFC, 0x56, 0x3E, 0x4B, 0xC6, 0xD2, 0x79, 0x20, 0x9A, 0xDB, 0xC0, 0xFE, 0x78, 0xCD, 0x5A, 0xF4,
		0x1F, 0xDD, 0xA8, 0x33, 0x88, 0x07, 0xC7, 0x31, 0xB1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xEC, 0x5F,
		0x60, 0x51, 0x7F, 0xA9, 0x19, 0xB5, 0x4A, 0x0D, 0x2D, 0xE5, 0x7A, 0x9F, 0x93, 0xC9, 0x9C, 0xEF,
		0xA0, 0xE0, 0x3B, 0x4D, 0xAE, 0x2A, 0xF5, 0xB0, 0xC8, 0xEB, 0xBB, 0x3C, 0x83, 0x53, 0x99, 0x61,
		0x17, 0x2B, 0x04, 0x7E, 0xBA, 0x77, 0xD6, 0x26, 0xE1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0C, 0x7D
	],

	/**
	 *	precomputed matrix for Galois field multiplication by 2
	 *	@private
	 */
	mult_2: [
		0x00, 0x02, 0x04, 0x06, 0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1a, 0x1c, 0x1e, 
		0x20, 0x22, 0x24, 0x26, 0x28, 0x2a, 0x2c, 0x2e, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3a, 0x3c, 0x3e, 
		0x40, 0x42, 0x44, 0x46, 0x48, 0x4a, 0x4c, 0x4e, 0x50, 0x52, 0x54, 0x56, 0x58, 0x5a, 0x5c, 0x5e, 
		0x60, 0x62, 0x64, 0x66, 0x68, 0x6a, 0x6c, 0x6e, 0x70, 0x72, 0x74, 0x76, 0x78, 0x7a, 0x7c, 0x7e, 
		0x80, 0x82, 0x84, 0x86, 0x88, 0x8a, 0x8c, 0x8e, 0x90, 0x92, 0x94, 0x96, 0x98, 0x9a, 0x9c, 0x9e, 
		0xa0, 0xa2, 0xa4, 0xa6, 0xa8, 0xaa, 0xac, 0xae, 0xb0, 0xb2, 0xb4, 0xb6, 0xb8, 0xba, 0xbc, 0xbe, 
		0xc0, 0xc2, 0xc4, 0xc6, 0xc8, 0xca, 0xcc, 0xce, 0xd0, 0xd2, 0xd4, 0xd6, 0xd8, 0xda, 0xdc, 0xde, 
		0xe0, 0xe2, 0xe4, 0xe6, 0xe8, 0xea, 0xec, 0xee, 0xf0, 0xf2, 0xf4, 0xf6, 0xf8, 0xfa, 0xfc, 0xfe, 
		0x1b, 0x19, 0x1f, 0x1d, 0x13, 0x11, 0x17, 0x15, 0x0b, 0x09, 0x0f, 0x0d, 0x03, 0x01, 0x07, 0x05, 
		0x3b, 0x39, 0x3f, 0x3d, 0x33, 0x31, 0x37, 0x35, 0x2b, 0x29, 0x2f, 0x2d, 0x23, 0x21, 0x27, 0x25, 
		0x5b, 0x59, 0x5f, 0x5d, 0x53, 0x51, 0x57, 0x55, 0x4b, 0x49, 0x4f, 0x4d, 0x43, 0x41, 0x47, 0x45, 
		0x7b, 0x79, 0x7f, 0x7d, 0x73, 0x71, 0x77, 0x75, 0x6b, 0x69, 0x6f, 0x6d, 0x63, 0x61, 0x67, 0x65, 
		0x9b, 0x99, 0x9f, 0x9d, 0x93, 0x91, 0x97, 0x95, 0x8b, 0x89, 0x8f, 0x8d, 0x83, 0x81, 0x87, 0x85, 
		0xbb, 0xb9, 0xbf, 0xbd, 0xb3, 0xb1, 0xb7, 0xb5, 0xab, 0xa9, 0xaf, 0xad, 0xa3, 0xa1, 0xa7, 0xa5, 
		0xdb, 0xd9, 0xdf, 0xdd, 0xd3, 0xd1, 0xd7, 0xd5, 0xcb, 0xc9, 0xcf, 0xcd, 0xc3, 0xc1, 0xc7, 0xc5, 
		0xfb, 0xf9, 0xff, 0xfd, 0xf3, 0xf1, 0xf7, 0xf5, 0xeb, 0xe9, 0xef, 0xed, 0xe3, 0xe1, 0xe7, 0xe5
	],

	/**
	 *	precomputed matrix for Galois field multiplication by 3
	 *	@private
	 */
	mult_3: [
		0x00, 0x03, 0x06, 0x05, 0x0c, 0x0f, 0x0a, 0x09, 0x18, 0x1b, 0x1e, 0x1d, 0x14, 0x17, 0x12, 0x11, 
		0x30, 0x33, 0x36, 0x35, 0x3c, 0x3f, 0x3a, 0x39, 0x28, 0x2b, 0x2e, 0x2d, 0x24, 0x27, 0x22, 0x21, 
		0x60, 0x63, 0x66, 0x65, 0x6c, 0x6f, 0x6a, 0x69, 0x78, 0x7b, 0x7e, 0x7d, 0x74, 0x77, 0x72, 0x71, 
		0x50, 0x53, 0x56, 0x55, 0x5c, 0x5f, 0x5a, 0x59, 0x48, 0x4b, 0x4e, 0x4d, 0x44, 0x47, 0x42, 0x41, 
		0xc0, 0xc3, 0xc6, 0xc5, 0xcc, 0xcf, 0xca, 0xc9, 0xd8, 0xdb, 0xde, 0xdd, 0xd4, 0xd7, 0xd2, 0xd1, 
		0xf0, 0xf3, 0xf6, 0xf5, 0xfc, 0xff, 0xfa, 0xf9, 0xe8, 0xeb, 0xee, 0xed, 0xe4, 0xe7, 0xe2, 0xe1, 
		0xa0, 0xa3, 0xa6, 0xa5, 0xac, 0xaf, 0xaa, 0xa9, 0xb8, 0xbb, 0xbe, 0xbd, 0xb4, 0xb7, 0xb2, 0xb1, 
		0x90, 0x93, 0x96, 0x95, 0x9c, 0x9f, 0x9a, 0x99, 0x88, 0x8b, 0x8e, 0x8d, 0x84, 0x87, 0x82, 0x81, 
		0x9b, 0x98, 0x9d, 0x9e, 0x97, 0x94, 0x91, 0x92, 0x83, 0x80, 0x85, 0x86, 0x8f, 0x8c, 0x89, 0x8a, 
		0xab, 0xa8, 0xad, 0xae, 0xa7, 0xa4, 0xa1, 0xa2, 0xb3, 0xb0, 0xb5, 0xb6, 0xbf, 0xbc, 0xb9, 0xba, 
		0xfb, 0xf8, 0xfd, 0xfe, 0xf7, 0xf4, 0xf1, 0xf2, 0xe3, 0xe0, 0xe5, 0xe6, 0xef, 0xec, 0xe9, 0xea, 
		0xcb, 0xc8, 0xcd, 0xce, 0xc7, 0xc4, 0xc1, 0xc2, 0xd3, 0xd0, 0xd5, 0xd6, 0xdf, 0xdc, 0xd9, 0xda, 
		0x5b, 0x58, 0x5d, 0x5e, 0x57, 0x54, 0x51, 0x52, 0x43, 0x40, 0x45, 0x46, 0x4f, 0x4c, 0x49, 0x4a, 
		0x6b, 0x68, 0x6d, 0x6e, 0x67, 0x64, 0x61, 0x62, 0x73, 0x70, 0x75, 0x76, 0x7f, 0x7c, 0x79, 0x7a, 
		0x3b, 0x38, 0x3d, 0x3e, 0x37, 0x34, 0x31, 0x32, 0x23, 0x20, 0x25, 0x26, 0x2f, 0x2c, 0x29, 0x2a, 
		0x0b, 0x08, 0x0d, 0x0e, 0x07, 0x04, 0x01, 0x02, 0x13, 0x10, 0x15, 0x16, 0x1f, 0x1c, 0x19, 0x1a
	],

	/**
	 *	precomputed matrix for Galois field multiplication by 9
	 *	@private
	 */
	mult_9: [
		0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f, 0x48, 0x41, 0x5a, 0x53, 0x6c, 0x65, 0x7e, 0x77, 
		0x90, 0x99, 0x82, 0x8b, 0xb4, 0xbd, 0xa6, 0xaf, 0xd8, 0xd1, 0xca, 0xc3, 0xfc, 0xf5, 0xee, 0xe7, 
		0x3b, 0x32, 0x29, 0x20, 0x1f, 0x16, 0x0d, 0x04, 0x73, 0x7a, 0x61, 0x68, 0x57, 0x5e, 0x45, 0x4c, 
		0xab, 0xa2, 0xb9, 0xb0, 0x8f, 0x86, 0x9d, 0x94, 0xe3, 0xea, 0xf1, 0xf8, 0xc7, 0xce, 0xd5, 0xdc, 
		0x76, 0x7f, 0x64, 0x6d, 0x52, 0x5b, 0x40, 0x49, 0x3e, 0x37, 0x2c, 0x25, 0x1a, 0x13, 0x08, 0x01, 
		0xe6, 0xef, 0xf4, 0xfd, 0xc2, 0xcb, 0xd0, 0xd9, 0xae, 0xa7, 0xbc, 0xb5, 0x8a, 0x83, 0x98, 0x91, 
		0x4d, 0x44, 0x5f, 0x56, 0x69, 0x60, 0x7b, 0x72, 0x05, 0x0c, 0x17, 0x1e, 0x21, 0x28, 0x33, 0x3a, 
		0xdd, 0xd4, 0xcf, 0xc6, 0xf9, 0xf0, 0xeb, 0xe2, 0x95, 0x9c, 0x87, 0x8e, 0xb1, 0xb8, 0xa3, 0xaa, 
		0xec, 0xe5, 0xfe, 0xf7, 0xc8, 0xc1, 0xda, 0xd3, 0xa4, 0xad, 0xb6, 0xbf, 0x80, 0x89, 0x92, 0x9b, 
		0x7c, 0x75, 0x6e, 0x67, 0x58, 0x51, 0x4a, 0x43, 0x34, 0x3d, 0x26, 0x2f, 0x10, 0x19, 0x02, 0x0b, 
		0xd7, 0xde, 0xc5, 0xcc, 0xf3, 0xfa, 0xe1, 0xe8, 0x9f, 0x96, 0x8d, 0x84, 0xbb, 0xb2, 0xa9, 0xa0, 
		0x47, 0x4e, 0x55, 0x5c, 0x63, 0x6a, 0x71, 0x78, 0x0f, 0x06, 0x1d, 0x14, 0x2b, 0x22, 0x39, 0x30, 
		0x9a, 0x93, 0x88, 0x81, 0xbe, 0xb7, 0xac, 0xa5, 0xd2, 0xdb, 0xc0, 0xc9, 0xf6, 0xff, 0xe4, 0xed, 
		0x0a, 0x03, 0x18, 0x11, 0x2e, 0x27, 0x3c, 0x35, 0x42, 0x4b, 0x50, 0x59, 0x66, 0x6f, 0x74, 0x7d, 
		0xa1, 0xa8, 0xb3, 0xba, 0x85, 0x8c, 0x97, 0x9e, 0xe9, 0xe0, 0xfb, 0xf2, 0xcd, 0xc4, 0xdf, 0xd6, 
		0x31, 0x38, 0x23, 0x2a, 0x15, 0x1c, 0x07, 0x0e, 0x79, 0x70, 0x6b, 0x62, 0x5d, 0x54, 0x4f, 0x46
	],

	/**
	 *	precomputed matrix for Galois field multiplication by 11
	 *	@private
	 */
	mult_11: [
		0x00, 0x0b, 0x16, 0x1d, 0x2c, 0x27, 0x3a, 0x31, 0x58, 0x53, 0x4e, 0x45, 0x74, 0x7f, 0x62, 0x69, 
		0xb0, 0xbb, 0xa6, 0xad, 0x9c, 0x97, 0x8a, 0x81, 0xe8, 0xe3, 0xfe, 0xf5, 0xc4, 0xcf, 0xd2, 0xd9, 
		0x7b, 0x70, 0x6d, 0x66, 0x57, 0x5c, 0x41, 0x4a, 0x23, 0x28, 0x35, 0x3e, 0x0f, 0x04, 0x19, 0x12, 
		0xcb, 0xc0, 0xdd, 0xd6, 0xe7, 0xec, 0xf1, 0xfa, 0x93, 0x98, 0x85, 0x8e, 0xbf, 0xb4, 0xa9, 0xa2, 
		0xf6, 0xfd, 0xe0, 0xeb, 0xda, 0xd1, 0xcc, 0xc7, 0xae, 0xa5, 0xb8, 0xb3, 0x82, 0x89, 0x94, 0x9f, 
		0x46, 0x4d, 0x50, 0x5b, 0x6a, 0x61, 0x7c, 0x77, 0x1e, 0x15, 0x08, 0x03, 0x32, 0x39, 0x24, 0x2f, 
		0x8d, 0x86, 0x9b, 0x90, 0xa1, 0xaa, 0xb7, 0xbc, 0xd5, 0xde, 0xc3, 0xc8, 0xf9, 0xf2, 0xef, 0xe4, 
		0x3d, 0x36, 0x2b, 0x20, 0x11, 0x1a, 0x07, 0x0c, 0x65, 0x6e, 0x73, 0x78, 0x49, 0x42, 0x5f, 0x54, 
		0xf7, 0xfc, 0xe1, 0xea, 0xdb, 0xd0, 0xcd, 0xc6, 0xaf, 0xa4, 0xb9, 0xb2, 0x83, 0x88, 0x95, 0x9e, 
		0x47, 0x4c, 0x51, 0x5a, 0x6b, 0x60, 0x7d, 0x76, 0x1f, 0x14, 0x09, 0x02, 0x33, 0x38, 0x25, 0x2e, 
		0x8c, 0x87, 0x9a, 0x91, 0xa0, 0xab, 0xb6, 0xbd, 0xd4, 0xdf, 0xc2, 0xc9, 0xf8, 0xf3, 0xee, 0xe5, 
		0x3c, 0x37, 0x2a, 0x21, 0x10, 0x1b, 0x06, 0x0d, 0x64, 0x6f, 0x72, 0x79, 0x48, 0x43, 0x5e, 0x55, 
		0x01, 0x0a, 0x17, 0x1c, 0x2d, 0x26, 0x3b, 0x30, 0x59, 0x52, 0x4f, 0x44, 0x75, 0x7e, 0x63, 0x68, 
		0xb1, 0xba, 0xa7, 0xac, 0x9d, 0x96, 0x8b, 0x80, 0xe9, 0xe2, 0xff, 0xf4, 0xc5, 0xce, 0xd3, 0xd8, 
		0x7a, 0x71, 0x6c, 0x67, 0x56, 0x5d, 0x40, 0x4b, 0x22, 0x29, 0x34, 0x3f, 0x0e, 0x05, 0x18, 0x13, 
		0xca, 0xc1, 0xdc, 0xd7, 0xe6, 0xed, 0xf0, 0xfb, 0x92, 0x99, 0x84, 0x8f, 0xbe, 0xb5, 0xa8, 0xa3
	],

	/**
	 *	precomputed matrix for Galois field multiplication by 13
	 *	@private
	 */
	mult_13: [
		0x00, 0x0d, 0x1a, 0x17, 0x34, 0x39, 0x2e, 0x23, 0x68, 0x65, 0x72, 0x7f, 0x5c, 0x51, 0x46, 0x4b, 
		0xd0, 0xdd, 0xca, 0xc7, 0xe4, 0xe9, 0xfe, 0xf3, 0xb8, 0xb5, 0xa2, 0xaf, 0x8c, 0x81, 0x96, 0x9b, 
		0xbb, 0xb6, 0xa1, 0xac, 0x8f, 0x82, 0x95, 0x98, 0xd3, 0xde, 0xc9, 0xc4, 0xe7, 0xea, 0xfd, 0xf0, 
		0x6b, 0x66, 0x71, 0x7c, 0x5f, 0x52, 0x45, 0x48, 0x03, 0x0e, 0x19, 0x14, 0x37, 0x3a, 0x2d, 0x20, 
		0x6d, 0x60, 0x77, 0x7a, 0x59, 0x54, 0x43, 0x4e, 0x05, 0x08, 0x1f, 0x12, 0x31, 0x3c, 0x2b, 0x26, 
		0xbd, 0xb0, 0xa7, 0xaa, 0x89, 0x84, 0x93, 0x9e, 0xd5, 0xd8, 0xcf, 0xc2, 0xe1, 0xec, 0xfb, 0xf6, 
		0xd6, 0xdb, 0xcc, 0xc1, 0xe2, 0xef, 0xf8, 0xf5, 0xbe, 0xb3, 0xa4, 0xa9, 0x8a, 0x87, 0x90, 0x9d, 
		0x06, 0x0b, 0x1c, 0x11, 0x32, 0x3f, 0x28, 0x25, 0x6e, 0x63, 0x74, 0x79, 0x5a, 0x57, 0x40, 0x4d, 
		0xda, 0xd7, 0xc0, 0xcd, 0xee, 0xe3, 0xf4, 0xf9, 0xb2, 0xbf, 0xa8, 0xa5, 0x86, 0x8b, 0x9c, 0x91, 
		0x0a, 0x07, 0x10, 0x1d, 0x3e, 0x33, 0x24, 0x29, 0x62, 0x6f, 0x78, 0x75, 0x56, 0x5b, 0x4c, 0x41, 
		0x61, 0x6c, 0x7b, 0x76, 0x55, 0x58, 0x4f, 0x42, 0x09, 0x04, 0x13, 0x1e, 0x3d, 0x30, 0x27, 0x2a, 
		0xb1, 0xbc, 0xab, 0xa6, 0x85, 0x88, 0x9f, 0x92, 0xd9, 0xd4, 0xc3, 0xce, 0xed, 0xe0, 0xf7, 0xfa, 
		0xb7, 0xba, 0xad, 0xa0, 0x83, 0x8e, 0x99, 0x94, 0xdf, 0xd2, 0xc5, 0xc8, 0xeb, 0xe6, 0xf1, 0xfc, 
		0x67, 0x6a, 0x7d, 0x70, 0x53, 0x5e, 0x49, 0x44, 0x0f, 0x02, 0x15, 0x18, 0x3b, 0x36, 0x21, 0x2c, 
		0x0c, 0x01, 0x16, 0x1b, 0x38, 0x35, 0x22, 0x2f, 0x64, 0x69, 0x7e, 0x73, 0x50, 0x5d, 0x4a, 0x47, 
		0xdc, 0xd1, 0xc6, 0xcb, 0xe8, 0xe5, 0xf2, 0xff, 0xb4, 0xb9, 0xae, 0xa3, 0x80, 0x8d, 0x9a, 0x97
	],

	/**
	 *	precomputed matrix for Galois field multiplication by 14
	 *	@private
	 */
	mult_14: [
		0x00, 0x0e, 0x1c, 0x12, 0x38, 0x36, 0x24, 0x2a, 0x70, 0x7e, 0x6c, 0x62, 0x48, 0x46, 0x54, 0x5a, 
		0xe0, 0xee, 0xfc, 0xf2, 0xd8, 0xd6, 0xc4, 0xca, 0x90, 0x9e, 0x8c, 0x82, 0xa8, 0xa6, 0xb4, 0xba, 
		0xdb, 0xd5, 0xc7, 0xc9, 0xe3, 0xed, 0xff, 0xf1, 0xab, 0xa5, 0xb7, 0xb9, 0x93, 0x9d, 0x8f, 0x81, 
		0x3b, 0x35, 0x27, 0x29, 0x03, 0x0d, 0x1f, 0x11, 0x4b, 0x45, 0x57, 0x59, 0x73, 0x7d, 0x6f, 0x61, 
		0xad, 0xa3, 0xb1, 0xbf, 0x95, 0x9b, 0x89, 0x87, 0xdd, 0xd3, 0xc1, 0xcf, 0xe5, 0xeb, 0xf9, 0xf7, 
		0x4d, 0x43, 0x51, 0x5f, 0x75, 0x7b, 0x69, 0x67, 0x3d, 0x33, 0x21, 0x2f, 0x05, 0x0b, 0x19, 0x17, 
		0x76, 0x78, 0x6a, 0x64, 0x4e, 0x40, 0x52, 0x5c, 0x06, 0x08, 0x1a, 0x14, 0x3e, 0x30, 0x22, 0x2c, 
		0x96, 0x98, 0x8a, 0x84, 0xae, 0xa0, 0xb2, 0xbc, 0xe6, 0xe8, 0xfa, 0xf4, 0xde, 0xd0, 0xc2, 0xcc, 
		0x41, 0x4f, 0x5d, 0x53, 0x79, 0x77, 0x65, 0x6b, 0x31, 0x3f, 0x2d, 0x23, 0x09, 0x07, 0x15, 0x1b, 
		0xa1, 0xaf, 0xbd, 0xb3, 0x99, 0x97, 0x85, 0x8b, 0xd1, 0xdf, 0xcd, 0xc3, 0xe9, 0xe7, 0xf5, 0xfb, 
		0x9a, 0x94, 0x86, 0x88, 0xa2, 0xac, 0xbe, 0xb0, 0xea, 0xe4, 0xf6, 0xf8, 0xd2, 0xdc, 0xce, 0xc0, 
		0x7a, 0x74, 0x66, 0x68, 0x42, 0x4c, 0x5e, 0x50, 0x0a, 0x04, 0x16, 0x18, 0x32, 0x3c, 0x2e, 0x20, 
		0xec, 0xe2, 0xf0, 0xfe, 0xd4, 0xda, 0xc8, 0xc6, 0x9c, 0x92, 0x80, 0x8e, 0xa4, 0xaa, 0xb8, 0xb6, 
		0x0c, 0x02, 0x10, 0x1e, 0x34, 0x3a, 0x28, 0x26, 0x7c, 0x72, 0x60, 0x6e, 0x44, 0x4a, 0x58, 0x56, 
		0x37, 0x39, 0x2b, 0x25, 0x0f, 0x01, 0x13, 0x1d, 0x47, 0x49, 0x5b, 0x55, 0x7f, 0x71, 0x63, 0x6d, 
		0xd7, 0xd9, 0xcb, 0xc5, 0xef, 0xe1, 0xf3, 0xfd, 0xa7, 0xa9, 0xbb, 0xb5, 0x9f, 0x91, 0x83, 0x8d
	],

	// ----------------------------------------------------------------------------------------------------

	/**
	 *	Sanity test for the class.
	 *
	 *	NOTE: You must initialize the instance with data.block_mode set to {@link ECB}
	 *	and data.pad_mode set to {@link ZeroPadding} for this to work!
	 *	See {@link BlockCipher#initialize} for more information on initialization properties.
	 *
	 *	@return {boolean}
	 */
	test: function()
	{
		this.debug_write('AES-256 TEST');
		this._do_test(
			'000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f', 
			'00112233445566778899aabbccddeeff',
			'8ea2b7ca516745bfeafc49904b496089'
		);

		this.debug_write('AES-192 TEST');
		this._do_test(
			'000102030405060708090a0b0c0d0e0f1011121314151617', 
			'00112233445566778899aabbccddeeff',
			'dda97ca4864cdfe06eaf70a0ec0d7191'
		);		
		
		this.debug_write('AES-128 TEST');
		this._do_test(
			'000102030405060708090a0b0c0d0e0f', 
			'00112233445566778899aabbccddeeff',
			'69c4e0d86a7b0430d8cdb78070b4c55a'
		);
		
		return true;
	},

	/**
	 *	Run the test for a key / plaintext / expected value combo
	 *
	 *	@private
	 *	@throws						Throws an error if any test fails.
	 *	@param {string} key			Hexadecimal string for key
	 *	@param {string} plaintext	Hexadecimal string for plaintext
	 *	@param {string} expected	Hexadecimal string for expected value
	 *	@return {boolean}
	 */
	_do_test: function(key, plaintext, expected)
	{
		var key_bin		= convert.hex_to_binstring(key);
		var plain_bin	= convert.hex_to_binstring(plaintext);

		this.set_key(key_bin);

		this.debug_write('------------------------------------------');
		this.debug_write('Test key: [binary] (length: '+key_bin.length+' bytes)');
		this.debug_write('Test key (hex): '+key);
		this.debug_write('Plaintext: [binary] (length: '+plain_bin.length+' bytes)');
		this.debug_write('Plaintext (hex): '+plaintext);
		this.debug_write('Expecting ciphertext (hex): '+expected);

		var ciphertext	= this.encrypt(plain_bin);
		var hex 		= convert.binstring_to_hex(ciphertext);

		this.debug_write('Ciphertext: [binary] (length: '+ciphertext.length+' bytes)');
		this.debug_write('Ciphertext (hex): '+hex);

		if (hex != expected)
			throw new Error('TEST FAILED: Invalid ciphertext! Expected: '+expected+', Got: ' + hex);

		this.debug_write('GOT EXPECTED CIPHERTEXT!');

		var plaintext2	= this.decrypt(ciphertext);
		var hex2 		= convert.binstring_to_hex(plaintext2);

		this.debug_write('Decrypted ciphertext: [binary] (length: '+plaintext2.length+' bytes)');
		this.debug_write('Decrypted ciphertext (hex): '+hex2);

		if (hex2 != plaintext)
			throw new Error('TEST FAILED: Invalid decrypted ciphertext! Expected: '+plaintext+', Got: ' + hex2);

		this.debug_write('SUCCESSFULLY DECRYPTED CIPHERTEXT!');
		this.debug_write('------------------------------------------');

		return true;
	}

});
