"use strict";

/**
 *	Twofish Symmetric Encryption Class
 *	Copyright (c) 2013, Jeff Lyon. (http://rubbingalcoholic.com)
 * 
 *	Licensed under The MIT License. (http://www.opensource.org/licenses/mit-license.php)
 *	Redistributions of files must retain the above copyright notice.
 *
 *	@class
 *	@classdesc				Implements the Twofish symmetric encryption algorithm by Bruce Schneier ({@link http://www.schneier.com/twofish.html})
 *	@extends				BlockCipher
 *	@requires				convert
 *
 *	@author					Jeff Lyon <jeff@rubbingalcoholic.com>
 *	@copyright				Copyright (c) 2013, Jeff Lyon.
 *	@license				{@link http://www.opensource.org/licenses/mit-license.php|The MIT License}
 *
 *	@desc					Creates a new Twofish instance
 *	@param {Object} data	Initialization options for the class, passed automatically into {@link Twofish#initialize}
 */
var Twofish = new Class(
/** @lends Twofish.prototype */
{
	Extends: BlockCipher,

	/**
	 *	Internal state for the key (not-expanded). Can be 16, 24 or 32 bytes.
	 *	@private
	 */
	key: 			[],

	/**
	 *	Internal state for the expanded key
	 *	@private
	 */
	key_expanded: 	[],

	/**
	 *	Internal state for the key length in bits. Can be overridden by {@link Twofish#set_key}
	 *	@private
	 */
	key_length: 	256,

	/**
	 *	Internal state for the block size in bits.
	 *	@private
	 */
	block_size: 	128,			// 128 bits

	/**
	 *	Convenience variable to track information about the key length
	 *	@private
	 */
	k_size: 		0,

	/**
	 *	Key-dependent substitution box (generated by {@link Twofish#expand_key}
	 */
	s_box:			[],
	
	/**
	 *	Controls whether to log debug output to the console.
	 *	@override
	 *	@type {Boolean}
	 *  @default false
	 *	@see {@link BlockCipher#debug_mode}
	 */
	debug_mode: false,

	/**
	 *	Called automatically on class instantiation.
	 *	Invokes {@link BlockCipher#initialize} before handling class-specific functionality.
	 *
	 *	@override
	 *	@param {Object} data See {@link BlockCipher#initialize} for a list of supported properties.
	 *	@return {Twofish}
	 */
	initialize: function(data)
	{
		this.parent(data);
	
		if (this.key.length)
			this.set_key(this.key);
	},

	/**
	 *	Getter function for the key
	 *	@return {Array} The byte array for the key
	 */
	get_key: function()
	{
		return this.key;
	},

	/**
	 *	Set the key, either from a string or an array of byte values
	 *
	 *	@param {string|Array} key	The symmetric key. Must be ASCII-encoded binary string or Array.
	 *	@return {Twofish} 			This Twofish instance (chainable)
	 */
	set_key: function(key)
	{
		if (typeof key == 'string')
			key				= convert.to_bytes(key);

		if (key.length < 16)
			key				= this.pad_key(key, 16);
		else if (key.length < 24)
			key				= this.pad_key(key, 24);
		else if (key.length < 32)
			key				= this.pad_key(key, 32);

		this.key			= key;
		this.key_expanded	= [];
		this.key_length		= (key.length * 8);
		
		return this;
	},

	/**
	 *	Pads key data to the specified length in bytes
	 *	@private
	 *	@param {string} key		The key
	 *	@param {number} length	Length after padding
	 *	@return {string}		Padded key
	 */
	pad_key: function(key, length)
	{
		for (var i=0; key.length != length; i++)
			key.push(0);

		return key;
	},

	/**
	 *	Getter function for the key length
	 *	@return {Number} The key length (in bits)
	 */
	get_key_length: function()
	{
		return this.key_length;
	},

	/**
	 *	Getter function for the block size
	 *	@return {Number} The block size (in bits)
	 */
	get_block_size: function()
	{
		return this.block_size;
	},

	/**
	 *	Encrypts a block. This is normally called internally by a subclass instance of {@link BlockCipherMode}.
	 *
	 *	@private
	 *	@param {Array} state	An array of 32-bit words
	 *	@return {Array}			An array of encrypted 32-bit words
	 */
	block_encrypt: function(state)
	{
		this.debug_write('Encrypting...');

		// Get the expanded key words
		var k	= this.get_expanded_key();

		// Shortcuts
		var rw	= this.reverse_word, rol = this.rol, ror = this.ror, h = this.h.bind(this), s = this.s_box;

		// Input whitening step (XOR the first 4 key words into our input state)
		var r	= Array([state[0] ^ k[0], state[1] ^ k[1], state[2] ^ k[2], state[3] ^ k[3]]);

		// Now loop over and apply the ecryption transformations to alternating halves
		for (var i = 0; i < 16; i += 2)
		{
			var t0 = h(r[i][0], s);
			var t1 = h(rol(r[i][1], 8), s);

			r[i+1] = [
				r[i][0],
				r[i][1],
				ror((t0 + t1 + k[2*i+8] % Math.pow(2, 32)) ^ r[i][2], 1),
				rol(r[i][3], 1) ^ ((t0 + 2*t1 + k[2*i+9]) % Math.pow(2, 32))
			];

			var t0 = h(r[i+1][2], s);
			var t1 = h(rol(r[i+1][3], 8), s);

			r[i+2] = [
				ror((t0 + t1 + k[2*(i+1)+8] % Math.pow(2, 32)) ^ r[i+1][0], 1),
				rol(r[i+1][1], 1) ^ ((t0 + 2*t1 + k[2*(i+1)+9]) % Math.pow(2, 32)),
				r[i+1][2],
				r[i+1][3]
			];
		}

		// Output whiten and byte swap
		return [rw(r[16][2] ^ k[4]), rw(r[16][3] ^ k[5]), rw(r[16][0] ^ k[6]), rw(r[16][1] ^ k[7])];
	},

	/**
	 *	Decrypts a block. This is normally called internally by a subclass instance of {@link BlockCipherMode}.
	 *
	 *	@private
	 *	@param {Array} state	An array of 32-bit words
	 *	@return {Array}			An array of decrypted 32-bit words
	 */
	block_decrypt: function(state)
	{
		this.debug_write('Decrypting...');

		// Get the expanded key words
		var k	= this.get_expanded_key();

		// Shortcuts
		var rw	= this.reverse_word, rol = this.rol, ror = this.ror, h = this.h.bind(this), s = this.s_box;

		// Initialize our round variables
		var r	= Array(16);

		// Input whitening step (XOR the second 4 key words into our input state), reverse endian and swap byte order too
		r[16]	= [rw(state[2]) ^ k[6], rw(state[3]) ^ k[7], rw(state[0]) ^ k[4], rw(state[1]) ^ k[5]];

		// Now loop over and apply the ecryption transformations to alternating halves
		for (var i = 16, key_i = 39; i > 0; i -= 2, key_i -= 4)
		{
			var t0 = h(r[i][2], s);
			var t1 = h(rol(r[i][3], 8), s);

			r[i-1] = [
				rol(r[i][0], 1) ^ ((t0 + t1 + k[key_i-1]) % Math.pow(2, 32)),
				ror(r[i][1] ^ ((t0 + 2*t1 + k[key_i]) % Math.pow(2, 32)), 1),
				r[i][2],
				r[i][3]
			];

			var t0 = h(r[i-1][0], s);
			var t1 = h(rol(r[i-1][1], 8), s);

			r[i-2] = [
				r[i-1][0],
				r[i-1][1],
				rol(r[i-1][2], 1) ^ ((t0 + t1 + k[key_i-3]) % Math.pow(2, 32)),
				ror(r[i-1][3] ^ ((t0 + 2*t1 + k[key_i-2]) % Math.pow(2, 32)), 1),
			];
		}

		// Output whitening step
		return [r[0][0] ^ k[0], r[0][1] ^ k[1], r[0][2] ^ k[2], r[0][3] ^ k[3]];
	},

	/**
	 *	Multiplies vectors by the RS matrix. Used in key generation.
	 *	@private
	 *	@param {Array} v	Array of 8-bit integers. Sorta kind of like a Galois field vector.
	 *	@param {Array}		32-bit integer word.
	 */
	rs_multiply: function(v)
	{
		var mod = 0x14d;	// primitive polynomial x^8 + x^6 + x^3 + x^2 + 1
		var m	= this.galois_multiply;

		/*
		01 A4 55 87 5A 58 DB 9E
		A4 56 82 F3 1E C6 68 E5
		02 A1 FC C1 47 AE 3D 19
		A4 55 87 5A 58 DB 9E 03
		*/
		
		var out = [
			//        01                   A4                   55                   87                   5A                   58                   DB                   9E
			m(v[0], 0x01, mod) ^ m(v[1], 0xa4, mod) ^ m(v[2], 0x55, mod) ^ m(v[3], 0x87, mod) ^ m(v[4], 0x5a, mod) ^ m(v[5], 0x58, mod) ^ m(v[6], 0xdb, mod) ^ m(v[7], 0x9e, mod),
			//        A4                   56                   82                   F3                   1E                   C6                   68                   E5	
			m(v[0], 0xa4, mod) ^ m(v[1], 0x56, mod) ^ m(v[2], 0x82, mod) ^ m(v[3], 0xf3, mod) ^ m(v[4], 0x1e, mod) ^ m(v[5], 0xc6, mod) ^ m(v[6], 0x68, mod) ^ m(v[7], 0xe5, mod),
			//        02                   A1                   FC                   C1                   47                   AE                   3D                   19
			m(v[0], 0x02, mod) ^ m(v[1], 0xa1, mod) ^ m(v[2], 0xfc, mod) ^ m(v[3], 0xc1, mod) ^ m(v[4], 0x47, mod) ^ m(v[5], 0xae, mod) ^ m(v[6], 0x3d, mod) ^ m(v[7], 0x19, mod),
			//        A4                   55                   87                   5A                   58                   DB                   9E                   03
			m(v[0], 0xa4, mod) ^ m(v[1], 0x55, mod) ^ m(v[2], 0x87, mod) ^ m(v[3], 0x5a, mod) ^ m(v[4], 0x58, mod) ^ m(v[5], 0xdb, mod) ^ m(v[6], 0x9e, mod) ^ m(v[7], 0x03, mod)
		];

		// Remember this spec puts the bytes into words in reverse-endian order
		return convert.to_word(out[3], out[2], out[1], out[0]);
	},

	/**
	 *	Multiplies vectors by the MDS matrix. Used in key generation and the symmetric *cryption methods.
	 *	@private
	 *	@param {Array} v	Array of 8-bit integers. Sorta kind of like a Galois field vector.
	 *	@param {Array}		32-bit integer word.
	 */
	mds_multiply: function(v)
	{
		var mod = 0x169;	// primitive polynomial x^8 +x^6 +x^5 +x^3 +1
		var m	= this.galois_multiply;

		/*
		01 EF 5B 5B
		5B EF EF 01
		EF 5B 01 EF
		EF 01 EF 5B
		*/

		var out = [
			m(v[0], 0x01, mod) ^ m(v[1], 0xef, mod) ^ m(v[2], 0x5b, mod) ^ m(v[3], 0x5b, mod),
			m(v[0], 0x5b, mod) ^ m(v[1], 0xef, mod) ^ m(v[2], 0xef, mod) ^ m(v[3], 0x01, mod),
			m(v[0], 0xef, mod) ^ m(v[1], 0x5b, mod) ^ m(v[2], 0x01, mod) ^ m(v[3], 0xef, mod),
			m(v[0], 0xef, mod) ^ m(v[1], 0x01, mod) ^ m(v[2], 0xef, mod) ^ m(v[3], 0x5b, mod)
		];

		return convert.to_word(out[3], out[2], out[1], out[0]);
	},

	/**
	 *	Performs Galois Field Multiplication. Interprets input bytes as polynomials of degree 7
	 *	in the Galois Field 2^8. The polynomials are multiplied together and then reduced
	 *	modulo a primitive polynomial of degree 8. Don't worry you don't need to actually know
	 *	what any of this means.
	 *	@private
	 *	@param {number} a	8-bit integer for polynomial a
	 *	@param {number} b	8-bit integer for polynomial b
	 *	@param {number} mod	9-bit integer for the primitive modulus polynomial
	 *	@return {number}	The result of the multiplication
	 */
	galois_multiply: function(a, b, mod)
	{
		var p = 0;
		var carry;

		for (var counter = 0; counter < 8; counter++)
		{
			if (b & 1) 
				p ^= a;
			carry = (a & 0x80); // 1 if most significant bit of a is 1
			a <<= 1;
			if (carry) 
				a ^= mod;
			b >>>= 1;
		}
		return p;
	},

	/**
	 *	The real "meat" of the transformations happening throughout the algorithm.
	 *	This is used in key expansion, as well as the key-dependent S-box
	 *	substitutions used in encryption / decryption.
	 *
	 *	@private
	 *	@param {number} x	32-bit integer word
	 *	@param {Array} l	Array of 32-bit integer words (of length {@link Twofish#k_size})
	 *	@return {number}	Transformed 32-bit integer word
	 */
	h: function(x, l)
	{
		// Shortcuts
		var q0 = this.q0, q1 = this.q1;
		var b0 = this.b0, b1 = this.b1, b2 = this.b2, b3 = this.b3;

		// Initialize y values
		var y0 = b0(x), y1 = b1(x), y2 = b2(x), y3 = b3(x);

		// 256 bit keys get here
		if (this.k_size == 4)
		{
			y0	= q1[y0] ^ b0(l[3]);
			y1	= q0[y1] ^ b1(l[3]);
			y2	= q0[y2] ^ b2(l[3]);
			y3	= q1[y3] ^ b3(l[3]);
		}
		// 192+ bit keys get here
		if (this.k_size > 2)
		{
			y0	= q1[y0] ^ b0(l[2]);
			y1	= q1[y1] ^ b1(l[2]);
			y2	= q0[y2] ^ b2(l[2]);
			y3	= q0[y3] ^ b3(l[2]);
		}

		// All keys get here
		y0		= q1[q0[q0[y0] ^ b0(l[1])] ^ b0(l[0])];
		y1		= q0[q0[q1[y1] ^ b1(l[1])] ^ b1(l[0])];
		y2		= q1[q1[q0[y2] ^ b2(l[1])] ^ b2(l[0])];
		y3		= q0[q1[q1[y3] ^ b3(l[1])] ^ b3(l[0])];

		return this.mds_multiply([y0, y1, y2, y3]);
	},

	/**
	 *	Gets the expanded key, and calls the expansion routine if it doesn't exist.
	 *	@private
	 *	@return {Array}	The expanded key
	 */
	get_expanded_key: function()
	{
		return this.key_expanded.length ? this.key_expanded : this.expand_key();
	},

	/**
	 *	Expands the key data and creates our key-dependent S-boxes
	 *	@private
	 *	@return {Array} The expanded key
	 */
	expand_key: function()
	{
		if (this.key.length != 16 && this.key.length != 24 && this.key.length != 32)
			throw new Error('Key is missing or invalid.');
		
		this.debug_write('Generating key schedule...');

		// Some variables to help us follow the spec
		var N	= this.key.length * 8;
		var k	= N / 64;
		var M	= convert.to_words(this.key, {reverse_endian: true});

		var Me 	= [];
		var Mo 	= [];

		// Drop even / odd words into Me / Mo
		for (var i = 0; i < k; i++)
		{
			Me.push(M[2*i]);
			Mo.push(M[(2*i)+1]);
		}

		var s 	= [];

		for (var i = 0; i < this.key.length; i += 8)
		{
			var vector 		= this.key.slice(i, i+8);
			var s_vector	= this.rs_multiply(vector);

			s.push(s_vector);
		}

		// In a surprise twist, the S word vector is reversed
		s.reverse();

		// we must set this before we call our h function
		this.k_size = k;
		
		// ρ = 2^24 + 2^16 + 2^8 + 2^0
		var rho	= 16843009;
		var key	= []

		for (var i = 0; i < 20; i++)
		{
			var Ai			= this.h(2*i*rho, Me);
			var Bi			= this.rol(this.h(((2*i) + 1)*rho, Mo), 8);

			// console.log('A (input '+i+'): '+(2*i*rho)+'; A: '+Ai.toString(16) + '; B (input '+i+'): '+(((2*i) + 1)*rho)+'; B: '+Bi.toString(16));

			key[i*2]		= (Ai + Bi) % Math.pow(2, 32);
			key[i*2+1]		= this.rol((Ai + 2*Bi) % Math.pow(2, 32), 9);
		}

		// Set our S box and expanded key
		this.s_box 			= s;
		this.key_expanded	= key;

		// this.debug_write('Key: ', key);

		// for (var i = 0; i < key.length; i++) this.debug_write('key['+i+']: ', (key[i] >>> 0).toString(16));

		return key;
	},

	/**
	 *	circular left shift bitwise operation lol.
	 *	@private
	 */
	rol: function(word, bits) { return (word << bits) | (word >>> (32 - bits));	},

	/**
	 *	circular right shift bitwise operation lol.
	 *	@private
	 */
	ror: function(word, bits) { return (word >>> bits) | word << (32 - bits); },

	/**
	 *	Grab the first byte out of a word
	 *	@private
	 */
	b0: function(word) { return word & 255; },

	/**
	 *	Grab the second byte out of a word
	 *	@private
	 */
	b1: function(word) { return (word >>> 8) & 255; },

	/**
	 *	Grab the third byte out of a word
	 *	@private
	 */
	b2: function(word) { return (word >>> 16) & 255; },

	/**
	 *	Grab the last byte out of a word
	 *	@private
	 */
	b3: function(word) { return (word >>> 24) & 255; },

	/**
	 *	Reverses the endianness of a word's byte order
	 */
	reverse_word: function(word) {
		return convert.to_word(word & 255, (word >>> 8) & 255, (word >>> 16) & 255, (word >>> 24) & 255);
	},

	/**
	 *	Precomputed q0 permutation substitution
	 *	@type {Array}
	 *	@private
	 */
	q0: [
		0xA9, 0x67, 0xB3, 0xE8, 0x04, 0xFD, 0xA3, 0x76, 
		0x9A, 0x92, 0x80, 0x78, 0xE4, 0xDD, 0xD1, 0x38, 
		0x0D, 0xC6, 0x35, 0x98, 0x18, 0xF7, 0xEC, 0x6C, 
		0x43, 0x75, 0x37, 0x26, 0xFA, 0x13, 0x94, 0x48, 
		0xF2, 0xD0, 0x8B, 0x30, 0x84, 0x54, 0xDF, 0x23, 
		0x19, 0x5B, 0x3D, 0x59, 0xF3, 0xAE, 0xA2, 0x82, 
		0x63, 0x01, 0x83, 0x2E, 0xD9, 0x51, 0x9B, 0x7C, 
		0xA6, 0xEB, 0xA5, 0xBE, 0x16, 0x0C, 0xE3, 0x61, 
		0xC0, 0x8C, 0x3A, 0xF5, 0x73, 0x2C, 0x25, 0x0B, 
		0xBB, 0x4E, 0x89, 0x6B, 0x53, 0x6A, 0xB4, 0xF1, 
		0xE1, 0xE6, 0xBD, 0x45, 0xE2, 0xF4, 0xB6, 0x66, 
		0xCC, 0x95, 0x03, 0x56, 0xD4, 0x1C, 0x1E, 0xD7, 
		0xFB, 0xC3, 0x8E, 0xB5, 0xE9, 0xCF, 0xBF, 0xBA, 
		0xEA, 0x77, 0x39, 0xAF, 0x33, 0xC9, 0x62, 0x71, 
		0x81, 0x79, 0x09, 0xAD, 0x24, 0xCD, 0xF9, 0xD8, 
		0xE5, 0xC5, 0xB9, 0x4D, 0x44, 0x08, 0x86, 0xE7, 
		0xA1, 0x1D, 0xAA, 0xED, 0x06, 0x70, 0xB2, 0xD2, 
		0x41, 0x7B, 0xA0, 0x11, 0x31, 0xC2, 0x27, 0x90, 
		0x20, 0xF6, 0x60, 0xFF, 0x96, 0x5C, 0xB1, 0xAB, 
		0x9E, 0x9C, 0x52, 0x1B, 0x5F, 0x93, 0x0A, 0xEF, 
		0x91, 0x85, 0x49, 0xEE, 0x2D, 0x4F, 0x8F, 0x3B, 
		0x47, 0x87, 0x6D, 0x46, 0xD6, 0x3E, 0x69, 0x64, 
		0x2A, 0xCE, 0xCB, 0x2F, 0xFC, 0x97, 0x05, 0x7A, 
		0xAC, 0x7F, 0xD5, 0x1A, 0x4B, 0x0E, 0xA7, 0x5A, 
		0x28, 0x14, 0x3F, 0x29, 0x88, 0x3C, 0x4C, 0x02, 
		0xB8, 0xDA, 0xB0, 0x17, 0x55, 0x1F, 0x8A, 0x7D, 
		0x57, 0xC7, 0x8D, 0x74, 0xB7, 0xC4, 0x9F, 0x72, 
		0x7E, 0x15, 0x22, 0x12, 0x58, 0x07, 0x99, 0x34, 
		0x6E, 0x50, 0xDE, 0x68, 0x65, 0xBC, 0xDB, 0xF8, 
		0xC8, 0xA8, 0x2B, 0x40, 0xDC, 0xFE, 0x32, 0xA4, 
		0xCA, 0x10, 0x21, 0xF0, 0xD3, 0x5D, 0x0F, 0x00, 
		0x6F, 0x9D, 0x36, 0x42, 0x4A, 0x5E, 0xC1, 0xE0
	],

	/**
	 *	Precomputed q1 permutation substitution
	 *	@type {Array}
	 *	@private
	 */
	q1: [
		0x75, 0xF3, 0xC6, 0xF4, 0xDB, 0x7B, 0xFB, 0xC8, 
		0x4A, 0xD3, 0xE6, 0x6B, 0x45, 0x7D, 0xE8, 0x4B, 
		0xD6, 0x32, 0xD8, 0xFD, 0x37, 0x71, 0xF1, 0xE1, 
		0x30, 0x0F, 0xF8, 0x1B, 0x87, 0xFA, 0x06, 0x3F, 
		0x5E, 0xBA, 0xAE, 0x5B, 0x8A, 0x00, 0xBC, 0x9D, 
		0x6D, 0xC1, 0xB1, 0x0E, 0x80, 0x5D, 0xD2, 0xD5, 
		0xA0, 0x84, 0x07, 0x14, 0xB5, 0x90, 0x2C, 0xA3, 
		0xB2, 0x73, 0x4C, 0x54, 0x92, 0x74, 0x36, 0x51, 
		0x38, 0xB0, 0xBD, 0x5A, 0xFC, 0x60, 0x62, 0x96, 
		0x6C, 0x42, 0xF7, 0x10, 0x7C, 0x28, 0x27, 0x8C, 
		0x13, 0x95, 0x9C, 0xC7, 0x24, 0x46, 0x3B, 0x70, 
		0xCA, 0xE3, 0x85, 0xCB, 0x11, 0xD0, 0x93, 0xB8, 
		0xA6, 0x83, 0x20, 0xFF, 0x9F, 0x77, 0xC3, 0xCC, 
		0x03, 0x6F, 0x08, 0xBF, 0x40, 0xE7, 0x2B, 0xE2, 
		0x79, 0x0C, 0xAA, 0x82, 0x41, 0x3A, 0xEA, 0xB9, 
		0xE4, 0x9A, 0xA4, 0x97, 0x7E, 0xDA, 0x7A, 0x17, 
		0x66, 0x94, 0xA1, 0x1D, 0x3D, 0xF0, 0xDE, 0xB3, 
		0x0B, 0x72, 0xA7, 0x1C, 0xEF, 0xD1, 0x53, 0x3E, 
		0x8F, 0x33, 0x26, 0x5F, 0xEC, 0x76, 0x2A, 0x49, 
		0x81, 0x88, 0xEE, 0x21, 0xC4, 0x1A, 0xEB, 0xD9, 
		0xC5, 0x39, 0x99, 0xCD, 0xAD, 0x31, 0x8B, 0x01, 
		0x18, 0x23, 0xDD, 0x1F, 0x4E, 0x2D, 0xF9, 0x48, 
		0x4F, 0xF2, 0x65, 0x8E, 0x78, 0x5C, 0x58, 0x19, 
		0x8D, 0xE5, 0x98, 0x57, 0x67, 0x7F, 0x05, 0x64, 
		0xAF, 0x63, 0xB6, 0xFE, 0xF5, 0xB7, 0x3C, 0xA5, 
		0xCE, 0xE9, 0x68, 0x44, 0xE0, 0x4D, 0x43, 0x69, 
		0x29, 0x2E, 0xAC, 0x15, 0x59, 0xA8, 0x0A, 0x9E, 
		0x6E, 0x47, 0xDF, 0x34, 0x35, 0x6A, 0xCF, 0xDC, 
		0x22, 0xC9, 0xC0, 0x9B, 0x89, 0xD4, 0xED, 0xAB, 
		0x12, 0xA2, 0x0D, 0x52, 0xBB, 0x02, 0x2F, 0xA9, 
		0xD7, 0x61, 0x1E, 0xB4, 0x50, 0x04, 0xF6, 0xC2, 
		0x16, 0x25, 0x86, 0x56, 0x55, 0x09, 0xBE, 0x91
	],

	// ----------------------------------------------------------------------------------------------------

	/**
	 *	Sanity test for the class.
	 *
	 *	NOTE: You must initialize the instance with data.block_mode set to {@link ECB}
	 *	and data.pad_mode set to {@link ZeroPadding} for this to work!
	 *	See {@link BlockCipher#initialize} for more information on initialization properties.
	 *
	 *	@return {boolean}
	 */
	test: function()
	{
		this.debug_write('Twofish (Keysize 256) TEST');
		this._do_test(
			'0123456789ABCDEFFEDCBA987654321000112233445566778899AABBCCDDEEFF', 
			'00000000000000000000000000000000',
			'37527BE0052334B89F0CFCCAE87CFA20'
		);
		
		return true;
	},

	/**
	 *	Run the test for a key / plaintext / expected value combo
	 *
	 *	@private
	 *	@throws						Throws an error if any test fails.
	 *	@param {string} key			Hexadecimal string for key
	 *	@param {string} plaintext	Hexadecimal string for plaintext
	 *	@param {string} expected	Hexadecimal string for expected value
	 *	@return {boolean}
	 */
	_do_test: function(key, plaintext, expected)
	{
		var key_bin		= convert.hex_to_binstring(key);
		var plain_bin	= convert.hex_to_binstring(plaintext);

		this.set_key(key_bin);

		this.debug_write('------------------------------------------');
		this.debug_write('Test key: [binary] (length: '+key_bin.length+' bytes)');
		this.debug_write('Test key (hex): '+key);
		this.debug_write('Plaintext: [binary] (length: '+plain_bin.length+' bytes)');
		this.debug_write('Plaintext (hex): '+plaintext);
		this.debug_write('Expecting ciphertext (hex): '+expected);

		var ciphertext	= this.encrypt(plain_bin);
		var hex 		= convert.binstring_to_hex(ciphertext);

		this.debug_write('Ciphertext: [binary] (length: '+ciphertext.length+' bytes)');
		this.debug_write('Ciphertext (hex): '+hex);

		if (hex.toLowerCase() != expected.toLowerCase())
			throw new Error('TEST FAILED: Invalid ciphertext! Expected: '+expected+', Got: ' + hex);

		this.debug_write('GOT EXPECTED CIPHERTEXT!');

		var plaintext2	= this.decrypt(ciphertext);
		var hex2 		= convert.binstring_to_hex(plaintext2);

		this.debug_write('Decrypted ciphertext: [binary] (length: '+plaintext2.length+' bytes)');
		this.debug_write('Decrypted ciphertext (hex): '+hex2);

		if (hex2.toLowerCase() != plaintext.toLowerCase())
			throw new Error('TEST FAILED: Invalid decrypted ciphertext! Expected: '+plaintext+', Got: ' + hex2);

		this.debug_write('SUCCESSFULLY DECRYPTED CIPHERTEXT!');
		this.debug_write('------------------------------------------');

		return true;
	}

});
