# Prefect Cloud API

Prefect Cloud exposes a powerful GraphQL API for interacting with the platform. There are a variety of ways you can access the API.

# Authentication

In order to interact with Cloud from your local machine, you'll need to generate an API token.

To generate an API token, use the Cloud UI or the following GraphQL call (from an already authenticated client!):

```graphql
mutation {
  createAPIToken(input: { name: "My API token", role: USER }) {
    token
  }
}
```

## API Token Scopes

Prefect Cloud can generate API tokens with three different scopes.

### `USER`

`USER`-scoped API tokens function as personal access tokens. These tokens have very few permissions on their own, but can be used to authenticate with the Cloud API. Once authenticated, `USER` tokens can be used to generate short-lived JWT auth tokens for any tenant the user belongs to. These auth tokens inherit any permissions the user has in that tenant, allowing full API access. The Client manages the process of provisioning and refreshing these tokens.

### `TENANT`

`TENANT`-scoped API tokens are used for long-lived programmatic access to a specific Cloud tenant. Unlike `USER` tokens, which can adopt any tenant membership the user has, `TENANT` tokens are fixed to a specific membership in a specific tenant. They adopt whatever permissions the user has in the tenant.

### `RUNNER`

`RUNNER`-scoped API tokens are used for processes like the Prefect Agent, which require the ability to execute flows on behalf of a tenant. Unlike the other token types, `RUNNER` tokens are not scoped to a particular user. Consequently, they can only be generated by tenant admins.

# Python Client

## About the Client

Prefect Core includes a Python client for Prefect Cloud. The Python client was designed for both interactive and programmatic use, and includes convenience methods for transparently managing authentication when used with `USER`-scoped tokens.

## Getting started

For interactive use, the most common way to use the Cloud Client is to generate a `USER`-scoped token and provide it to the client. After doing so, users can save the token so it persists across all Python sessions:

```python
import prefect
client = prefect.Client(api_token="YOUR_USER_TOKEN")
client.save_api_token()
```

Now, starting a client in another session will automatically reload the token:

```python
client = prefect.Client()
assert client._api_token == "YOUR_USER_TOKEN"  # True
```

Note that a token can be provided by environment variable (`PREFECT__CLOUD__AUTH_TOKEN`) or in your Prefect config (under `cloud.auth_token`).

:::tip Using `USER` tokens
The steps shown here were designed to be used with `USER`-scoped tokens. They will work with `TENANT` scoped tokens as well, but unexpected errors could result.
:::

Once provisioned with a `USER` token, the Cloud Client can query for available tenants and login to those tenants. In order to query for tenants, call:

```python
client.get_available_tenants()
```

This will print the id, name, and slug of all the tenants the user can login to.

```python
client.login_to_tenant(tenant_slug='a-tenant-slug')
# OR
client.login_to_tenant(tenant_id='A_TENANT_ID')
```

Both of these calls persist the active tenant in local storage, so you won't have to login again until you're ready to switch tenants.

Once logged in, you can make any GraphQL query against the Cloud API:

```python
client.graphql(
    {
        'query': {
            'flow': {
                'id'
            }
        }
    }
)
```

(Note that this illustrates how Prefect can parse Python structures to construct GraphQL query strings!)

Finally, you can logout:

```python
client.logout_from_tenant()
```

# GraphQL

The Cloud API also supports direct GraphQL access. While you can still use `USER`-scoped tokens to access and log in to tenants, you will need to manage the short-lived auth and refresh tokens yourself. Therefore, we recommend using the Python client for `USER`-scoped access.

For `TENANT`-scoped tokens, simply include the token as the authorization header of your GraphQL requests:

```json
{ "authorization": "Bearer YOUR_TOKEN" }
```
