import datetime

import pendulum
import pytest

import prefect
from prefect.engine.signals import (
    FAIL,
    PAUSE,
    RETRY,
    SKIP,
    SUCCESS,
    TRIGGERFAIL,
    PrefectStateSignal,
)
from prefect.engine.state import (
    Failed,
    Paused,
    Retrying,
    Skipped,
    State,
    Success,
    TriggerFailed,
)


def test_exceptions_are_displayed_with_messages():
    exc = PrefectStateSignal("you did something incorrectly")
    assert "you did something incorrectly" in repr(exc)
    assert "PrefectStateSignal" in repr(exc)


def test_signals_create_states():
    with pytest.raises(Exception) as exc:
        raise PrefectStateSignal("message")
    assert isinstance(exc.value.state, State)
    assert exc.value.state.result is exc.value
    assert exc.value.state.message == "message"


def test_signals_dont_pass_invalid_arguments_to_states():
    with pytest.raises(TypeError):
        raise SUCCESS(bad_result=100)


def test_retry_signals_can_set_retry_time():
    date = pendulum.datetime(2019, 1, 1)
    with pytest.raises(PrefectStateSignal) as exc:
        raise RETRY(start_time=date)
    assert exc.value.state.start_time == date


def test_retry_signals_accept_run_count():
    with pytest.raises(PrefectStateSignal) as exc:
        raise RETRY(run_count=5)
    assert exc.value.state.run_count == 5


def test_retry_signals_take_run_count_from_context():
    with prefect.context(task_run_count=5):
        with pytest.raises(PrefectStateSignal) as exc:
            raise RETRY()
    assert exc.value.state.run_count == 5


def test_retry_signals_prefer_supplied_run_count_to_context():
    with prefect.context(task_run_count=5):
        with pytest.raises(PrefectStateSignal) as exc:
            raise RETRY(run_count=6)
    assert exc.value.state.run_count == 6


@pytest.mark.parametrize(
    "signal,state",
    [
        (FAIL, Failed),
        (TRIGGERFAIL, TriggerFailed),
        (SUCCESS, Success),
        (PAUSE, Paused),
        (RETRY, Retrying),
        (SKIP, Skipped),
    ],
)
def test_signals_creates_correct_states(signal, state):
    with pytest.raises(Exception) as exc:
        raise signal(state.__name__)
    assert isinstance(exc.value, signal)
    assert type(exc.value.state) is state
    assert exc.value.state.result is exc.value
    assert exc.value.state.message == state.__name__


def test_retry_signals_carry_default_retry_time_on_state():
    with pytest.raises(Exception) as exc:
        raise RETRY()
    assert exc.value.state.start_time is not None
    now = pendulum.now("utc")
    assert now - exc.value.state.start_time < datetime.timedelta(seconds=0.1)
