/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

use std::fmt;

use dom_struct::dom_struct;

use crate::conversions::Convert;
use crate::dom::bindings::codegen::Bindings::TrustedHTMLBinding::TrustedHTMLMethods;
use crate::dom::bindings::codegen::UnionTypes::{
    TrustedHTMLOrNullIsEmptyString, TrustedHTMLOrString,
};
use crate::dom::bindings::error::Fallible;
use crate::dom::bindings::reflector::{Reflector, reflect_dom_object};
use crate::dom::bindings::root::DomRoot;
use crate::dom::bindings::str::DOMString;
use crate::dom::globalscope::GlobalScope;
use crate::dom::trustedtypepolicy::TrustedType;
use crate::dom::trustedtypepolicyfactory::{DEFAULT_SCRIPT_SINK_GROUP, TrustedTypePolicyFactory};
use crate::script_runtime::CanGc;

#[dom_struct]
pub struct TrustedHTML {
    reflector_: Reflector,

    data: DOMString,
}

impl TrustedHTML {
    fn new_inherited(data: DOMString) -> Self {
        Self {
            reflector_: Reflector::new(),
            data,
        }
    }

    #[cfg_attr(crown, allow(crown::unrooted_must_root))]
    pub(crate) fn new(data: DOMString, global: &GlobalScope, can_gc: CanGc) -> DomRoot<Self> {
        reflect_dom_object(Box::new(Self::new_inherited(data)), global, can_gc)
    }

    pub(crate) fn get_trusted_script_compliant_string(
        global: &GlobalScope,
        value: TrustedHTMLOrString,
        sink: &str,
        can_gc: CanGc,
    ) -> Fallible<DOMString> {
        match value {
            TrustedHTMLOrString::String(value) => {
                TrustedTypePolicyFactory::get_trusted_type_compliant_string(
                    TrustedType::TrustedHTML,
                    global,
                    value,
                    sink,
                    DEFAULT_SCRIPT_SINK_GROUP,
                    can_gc,
                )
            },

            TrustedHTMLOrString::TrustedHTML(trusted_html) => Ok(trusted_html.data.clone()),
        }
    }

    pub(crate) fn data(&self) -> &DOMString {
        &self.data
    }
}

impl fmt::Display for TrustedHTML {
    #[inline]
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.write_str(&self.data.str())
    }
}

impl TrustedHTMLMethods<crate::DomTypeHolder> for TrustedHTML {
    /// <https://www.w3.org/TR/trusted-types/#trustedhtml-stringification-behavior>
    fn Stringifier(&self) -> DOMString {
        self.data.clone()
    }

    /// <https://www.w3.org/TR/trusted-types/#dom-trustedhtml-tojson>
    fn ToJSON(&self) -> DOMString {
        self.data.clone()
    }
}

impl Convert<TrustedHTMLOrString> for TrustedHTMLOrNullIsEmptyString {
    fn convert(self) -> TrustedHTMLOrString {
        match self {
            TrustedHTMLOrNullIsEmptyString::TrustedHTML(trusted_html) => {
                TrustedHTMLOrString::TrustedHTML(trusted_html)
            },
            TrustedHTMLOrNullIsEmptyString::NullIsEmptyString(str) => {
                TrustedHTMLOrString::String(str)
            },
        }
    }
}
