/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

use std::cell::Cell;

use dom_struct::dom_struct;
use euclid::{Scale, Size2D};
use script_bindings::reflector::Reflector;
use servo_url::ServoUrl;
use style_traits::CSSPixel;
use webrender_api::ImageKey;
use webrender_api::units::DevicePixel;

use super::canvas_state::CanvasState;
use crate::dom::bindings::codegen::Bindings::CanvasRenderingContext2DBinding::{
    CanvasFillRule, CanvasImageSource, CanvasLineCap, CanvasLineJoin,
};
use crate::dom::bindings::codegen::Bindings::DOMMatrixBinding::DOMMatrix2DInit;
use crate::dom::bindings::codegen::Bindings::PaintRenderingContext2DBinding::PaintRenderingContext2DMethods;
use crate::dom::bindings::codegen::UnionTypes::StringOrCanvasGradientOrCanvasPattern;
use crate::dom::bindings::error::{ErrorResult, Fallible};
use crate::dom::bindings::inheritance::Castable;
use crate::dom::bindings::num::Finite;
use crate::dom::bindings::reflector::{DomGlobal as _, reflect_dom_object};
use crate::dom::bindings::root::DomRoot;
use crate::dom::bindings::str::DOMString;
use crate::dom::canvasgradient::CanvasGradient;
use crate::dom::canvaspattern::CanvasPattern;
use crate::dom::dommatrix::DOMMatrix;
use crate::dom::paintworkletglobalscope::PaintWorkletGlobalScope;
use crate::dom::path2d::Path2D;
use crate::script_runtime::CanGc;

#[dom_struct]
pub(crate) struct PaintRenderingContext2D {
    reflector_: Reflector,
    canvas_state: CanvasState,
    #[no_trace]
    device_pixel_ratio: Cell<Scale<f32, CSSPixel, DevicePixel>>,
    /// An [`ImageKey`] used to store the results of this [`PaintWorkletGlobalScope`].
    #[no_trace]
    image_key: ImageKey,
}

impl PaintRenderingContext2D {
    #[cfg_attr(crown, allow(crown::unrooted_must_root))]
    fn new_inherited(global: &PaintWorkletGlobalScope) -> Option<PaintRenderingContext2D> {
        let canvas_state = CanvasState::new(global.upcast(), Size2D::zero())?;
        let image_cache = global.image_cache();
        let image_key = image_cache.get_image_key()?;
        canvas_state.set_image_key(image_key);

        Some(PaintRenderingContext2D {
            reflector_: Reflector::new(),
            canvas_state,
            device_pixel_ratio: Cell::new(Scale::new(1.0)),
            image_key,
        })
    }

    #[cfg_attr(crown, allow(crown::unrooted_must_root))]
    pub(crate) fn new(
        global: &PaintWorkletGlobalScope,
        can_gc: CanGc,
    ) -> Option<DomRoot<PaintRenderingContext2D>> {
        Some(reflect_dom_object(
            Box::new(PaintRenderingContext2D::new_inherited(global)?),
            global,
            can_gc,
        ))
    }

    pub(crate) fn update_rendering(&self) -> bool {
        self.canvas_state.update_rendering(None)
    }

    pub(crate) fn image_key(&self) -> ImageKey {
        self.image_key
    }

    pub(crate) fn take_missing_image_urls(&self) -> Vec<ServoUrl> {
        std::mem::take(&mut self.canvas_state.get_missing_image_urls().borrow_mut())
    }

    pub(crate) fn set_bitmap_dimensions(
        &self,
        size: Size2D<f32, CSSPixel>,
        device_pixel_ratio: Scale<f32, CSSPixel, DevicePixel>,
    ) {
        let size = size * device_pixel_ratio;
        self.device_pixel_ratio.set(device_pixel_ratio);
        self.canvas_state
            .set_bitmap_dimensions(size.to_untyped().to_u64());
        self.scale_by_device_pixel_ratio();
    }

    fn scale_by_device_pixel_ratio(&self) {
        let device_pixel_ratio = self.device_pixel_ratio.get().get() as f64;
        if device_pixel_ratio != 1.0 {
            self.Scale(device_pixel_ratio, device_pixel_ratio);
        }
    }
}

impl PaintRenderingContext2DMethods<crate::DomTypeHolder> for PaintRenderingContext2D {
    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-save>
    fn Save(&self) {
        self.canvas_state.save()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-restore>
    fn Restore(&self) {
        self.canvas_state.restore()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-reset>
    fn Reset(&self) {
        self.canvas_state.reset()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-scale>
    fn Scale(&self, x: f64, y: f64) {
        self.canvas_state.scale(x, y)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-rotate>
    fn Rotate(&self, angle: f64) {
        self.canvas_state.rotate(angle)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-translate>
    fn Translate(&self, x: f64, y: f64) {
        self.canvas_state.translate(x, y)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-transform>
    fn Transform(&self, a: f64, b: f64, c: f64, d: f64, e: f64, f: f64) {
        self.canvas_state.transform(a, b, c, d, e, f)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-gettransform>
    fn GetTransform(&self, can_gc: CanGc) -> DomRoot<DOMMatrix> {
        self.canvas_state.get_transform(&self.global(), can_gc)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-settransform>
    fn SetTransform(&self, a: f64, b: f64, c: f64, d: f64, e: f64, f: f64) -> ErrorResult {
        self.canvas_state.set_transform(a, b, c, d, e, f);
        self.scale_by_device_pixel_ratio();
        Ok(())
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-settransform-matrix>
    fn SetTransform_(&self, transform: &DOMMatrix2DInit) -> ErrorResult {
        self.canvas_state.set_transform_(transform)?;
        self.scale_by_device_pixel_ratio();
        Ok(())
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-resettransform>
    fn ResetTransform(&self) {
        self.canvas_state.reset_transform();
        self.scale_by_device_pixel_ratio();
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-globalalpha>
    fn GlobalAlpha(&self) -> f64 {
        self.canvas_state.global_alpha()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-globalalpha>
    fn SetGlobalAlpha(&self, alpha: f64) {
        self.canvas_state.set_global_alpha(alpha)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-globalcompositeoperation>
    fn GlobalCompositeOperation(&self) -> DOMString {
        self.canvas_state.global_composite_operation()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-globalcompositeoperation>
    fn SetGlobalCompositeOperation(&self, op_str: DOMString) {
        self.canvas_state.set_global_composite_operation(op_str)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-fillrect>
    fn FillRect(&self, x: f64, y: f64, width: f64, height: f64) {
        self.canvas_state.fill_rect(x, y, width, height)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-clearrect>
    fn ClearRect(&self, x: f64, y: f64, width: f64, height: f64) {
        self.canvas_state.clear_rect(x, y, width, height)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-strokerect>
    fn StrokeRect(&self, x: f64, y: f64, width: f64, height: f64) {
        self.canvas_state.stroke_rect(x, y, width, height)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-beginpath>
    fn BeginPath(&self) {
        self.canvas_state.begin_path()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-closepath>
    fn ClosePath(&self) {
        self.canvas_state.close_path()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-fill>
    fn Fill(&self, fill_rule: CanvasFillRule) {
        self.canvas_state.fill(fill_rule)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-fill>
    fn Fill_(&self, path: &Path2D, fill_rule: CanvasFillRule) {
        self.canvas_state.fill_(path.segments(), fill_rule)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-stroke>
    fn Stroke(&self) {
        self.canvas_state.stroke()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-stroke>
    fn Stroke_(&self, path: &Path2D) {
        self.canvas_state.stroke_(path.segments())
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-clip>
    fn Clip(&self, fill_rule: CanvasFillRule) {
        self.canvas_state.clip(fill_rule)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-clip>
    fn Clip_(&self, path: &Path2D, fill_rule: CanvasFillRule) {
        self.canvas_state.clip_(path.segments(), fill_rule)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-ispointinpath>
    fn IsPointInPath(&self, x: f64, y: f64, fill_rule: CanvasFillRule) -> bool {
        self.canvas_state
            .is_point_in_path(&self.global(), x, y, fill_rule)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-ispointinpath>
    fn IsPointInPath_(&self, path: &Path2D, x: f64, y: f64, fill_rule: CanvasFillRule) -> bool {
        self.canvas_state
            .is_point_in_path_(&self.global(), path.segments(), x, y, fill_rule)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-drawimage>
    fn DrawImage(&self, image: CanvasImageSource, dx: f64, dy: f64) -> ErrorResult {
        self.canvas_state.draw_image(None, image, dx, dy)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-drawimage>
    fn DrawImage_(
        &self,
        image: CanvasImageSource,
        dx: f64,
        dy: f64,
        dw: f64,
        dh: f64,
    ) -> ErrorResult {
        self.canvas_state.draw_image_(None, image, dx, dy, dw, dh)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-drawimage>
    fn DrawImage__(
        &self,
        image: CanvasImageSource,
        sx: f64,
        sy: f64,
        sw: f64,
        sh: f64,
        dx: f64,
        dy: f64,
        dw: f64,
        dh: f64,
    ) -> ErrorResult {
        self.canvas_state
            .draw_image__(None, image, sx, sy, sw, sh, dx, dy, dw, dh)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-moveto>
    fn MoveTo(&self, x: f64, y: f64) {
        self.canvas_state.move_to(x, y)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-lineto>
    fn LineTo(&self, x: f64, y: f64) {
        self.canvas_state.line_to(x, y)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-rect>
    fn Rect(&self, x: f64, y: f64, width: f64, height: f64) {
        self.canvas_state.rect(x, y, width, height)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-quadraticcurveto>
    fn QuadraticCurveTo(&self, cpx: f64, cpy: f64, x: f64, y: f64) {
        self.canvas_state.quadratic_curve_to(cpx, cpy, x, y)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-beziercurveto>
    fn BezierCurveTo(&self, cp1x: f64, cp1y: f64, cp2x: f64, cp2y: f64, x: f64, y: f64) {
        self.canvas_state
            .bezier_curve_to(cp1x, cp1y, cp2x, cp2y, x, y)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-arc>
    fn Arc(&self, x: f64, y: f64, r: f64, start: f64, end: f64, ccw: bool) -> ErrorResult {
        self.canvas_state.arc(x, y, r, start, end, ccw)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-arcto>
    fn ArcTo(&self, cp1x: f64, cp1y: f64, cp2x: f64, cp2y: f64, r: f64) -> ErrorResult {
        self.canvas_state.arc_to(cp1x, cp1y, cp2x, cp2y, r)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-ellipse>
    fn Ellipse(
        &self,
        x: f64,
        y: f64,
        rx: f64,
        ry: f64,
        rotation: f64,
        start: f64,
        end: f64,
        ccw: bool,
    ) -> ErrorResult {
        self.canvas_state
            .ellipse(x, y, rx, ry, rotation, start, end, ccw)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-imagesmoothingenabled>
    fn ImageSmoothingEnabled(&self) -> bool {
        self.canvas_state.image_smoothing_enabled()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-imagesmoothingenabled>
    fn SetImageSmoothingEnabled(&self, value: bool) {
        self.canvas_state.set_image_smoothing_enabled(value)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-strokestyle>
    fn StrokeStyle(&self) -> StringOrCanvasGradientOrCanvasPattern {
        self.canvas_state.stroke_style()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-strokestyle>
    fn SetStrokeStyle(&self, value: StringOrCanvasGradientOrCanvasPattern) {
        self.canvas_state.set_stroke_style(None, value)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-strokestyle>
    fn FillStyle(&self) -> StringOrCanvasGradientOrCanvasPattern {
        self.canvas_state.fill_style()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-strokestyle>
    fn SetFillStyle(&self, value: StringOrCanvasGradientOrCanvasPattern) {
        self.canvas_state.set_fill_style(None, value)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-createlineargradient>
    fn CreateLinearGradient(
        &self,
        x0: Finite<f64>,
        y0: Finite<f64>,
        x1: Finite<f64>,
        y1: Finite<f64>,
        can_gc: CanGc,
    ) -> DomRoot<CanvasGradient> {
        self.canvas_state
            .create_linear_gradient(&self.global(), x0, y0, x1, y1, can_gc)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-createradialgradient>
    fn CreateRadialGradient(
        &self,
        x0: Finite<f64>,
        y0: Finite<f64>,
        r0: Finite<f64>,
        x1: Finite<f64>,
        y1: Finite<f64>,
        r1: Finite<f64>,
        can_gc: CanGc,
    ) -> Fallible<DomRoot<CanvasGradient>> {
        self.canvas_state
            .create_radial_gradient(&self.global(), x0, y0, r0, x1, y1, r1, can_gc)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-createpattern>
    fn CreatePattern(
        &self,
        image: CanvasImageSource,
        repetition: DOMString,
        can_gc: CanGc,
    ) -> Fallible<Option<DomRoot<CanvasPattern>>> {
        self.canvas_state
            .create_pattern(&self.global(), image, repetition, can_gc)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linewidth>
    fn LineWidth(&self) -> f64 {
        self.canvas_state.line_width()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linewidth>
    fn SetLineWidth(&self, width: f64) {
        self.canvas_state.set_line_width(width)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linecap>
    fn LineCap(&self) -> CanvasLineCap {
        self.canvas_state.line_cap()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linecap>
    fn SetLineCap(&self, cap: CanvasLineCap) {
        self.canvas_state.set_line_cap(cap)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linejoin>
    fn LineJoin(&self) -> CanvasLineJoin {
        self.canvas_state.line_join()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linejoin>
    fn SetLineJoin(&self, join: CanvasLineJoin) {
        self.canvas_state.set_line_join(join)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-miterlimit>
    fn MiterLimit(&self) -> f64 {
        self.canvas_state.miter_limit()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-miterlimit>
    fn SetMiterLimit(&self, limit: f64) {
        self.canvas_state.set_miter_limit(limit)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-setlinedash>
    fn SetLineDash(&self, segments: Vec<f64>) {
        self.canvas_state.set_line_dash(segments);
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-getlinedash>
    fn GetLineDash(&self) -> Vec<f64> {
        self.canvas_state.line_dash()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linedashoffset>
    fn LineDashOffset(&self) -> f64 {
        self.canvas_state.line_dash_offset()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-linedashoffset>
    fn SetLineDashOffset(&self, offset: f64) {
        self.canvas_state.set_line_dash_offset(offset);
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowoffsetx>
    fn ShadowOffsetX(&self) -> f64 {
        self.canvas_state.shadow_offset_x()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowoffsetx>
    fn SetShadowOffsetX(&self, value: f64) {
        self.canvas_state.set_shadow_offset_x(value)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowoffsety>
    fn ShadowOffsetY(&self) -> f64 {
        self.canvas_state.shadow_offset_y()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowoffsety>
    fn SetShadowOffsetY(&self, value: f64) {
        self.canvas_state.set_shadow_offset_y(value)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowblur>
    fn ShadowBlur(&self) -> f64 {
        self.canvas_state.shadow_blur()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowblur>
    fn SetShadowBlur(&self, value: f64) {
        self.canvas_state.set_shadow_blur(value)
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowcolor>
    fn ShadowColor(&self) -> DOMString {
        self.canvas_state.shadow_color()
    }

    /// <https://html.spec.whatwg.org/multipage/#dom-context-2d-shadowcolor>
    fn SetShadowColor(&self, value: DOMString) {
        self.canvas_state.set_shadow_color(None, value)
    }
}
