/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

use net::cookie::ServoCookie;
use net::cookie_storage::CookieStorage;
use net_traits::CookieSource;
use servo_url::ServoUrl;

fn run(set_location: &str, set_cookies: &[&str], final_location: &str) -> String {
    let mut storage = CookieStorage::new(150);
    let url = ServoUrl::parse(set_location).unwrap();
    let source = CookieSource::HTTP;

    // Add all cookies to the store
    for str_cookie in set_cookies {
        if let Some(cookie) = ServoCookie::from_cookie_string(str_cookie, &url, source) {
            storage.push(cookie, &url, source);
        }
    }

    // Get cookies for the test location
    let url = ServoUrl::parse(final_location).unwrap();
    storage
        .cookies_for_url(&url, source)
        .unwrap_or("".to_string())
}

// Following are all tests extracted from https://github.com/abarth/http-state.git
// They are generated by `./mach update-net-cookies`
// Test listing

#[test]
fn test_0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0001",
        &["foo=bar"],
        "http://home.example.org:8888/cookie-parser-result?0001",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0002",
        &["foo=bar; Expires=Fri, 07 Aug 2019 08:04:19 GMT"],
        "http://home.example.org:8888/cookie-parser-result?0002",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0003",
        &[
            "foo=bar; Expires=Fri, 07 Aug 2007 08:04:19 GMT",
            "foo2=bar2; Expires=Fri, 07 Aug 2017 08:04:19 GMT",
        ],
        "http://home.example.org:8888/cookie-parser-result?0003",
    );
    assert_eq!(&r, "foo2=bar2");
}

#[test]
fn test_0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0004",
        &["foo"],
        "http://home.example.org:8888/cookie-parser-result?0004",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0005",
        &["foo=bar; max-age=10000;"],
        "http://home.example.org:8888/cookie-parser-result?0005",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0006",
        &["foo=bar; max-age=0;"],
        "http://home.example.org:8888/cookie-parser-result?0006",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0007",
        &["foo=bar; version=1;"],
        "http://home.example.org:8888/cookie-parser-result?0007",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_0008() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0008",
        &["foo=bar; version=1000;"],
        "http://home.example.org:8888/cookie-parser-result?0008",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_0009() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0009",
        &["foo=bar; customvalue=1000;"],
        "http://home.example.org:8888/cookie-parser-result?0009",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_0010() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0010",
        &["foo=bar; secure;"],
        "http://home.example.org:8888/cookie-parser-result?0010",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0011() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0011",
        &["foo=bar; customvalue=\"1000 or more\";"],
        "http://home.example.org:8888/cookie-parser-result?0011",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_0012() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0012",
        &["foo=bar; customvalue=\"no trailing semicolon\""],
        "http://home.example.org:8888/cookie-parser-result?0012",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_0013() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0013",
        &["foo=bar", "foo=qux"],
        "http://home.example.org:8888/cookie-parser-result?0013",
    );
    assert_eq!(&r, "foo=qux");
}

#[test]
fn test_0014() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0014",
        &["foo1=bar", "foo2=qux"],
        "http://home.example.org:8888/cookie-parser-result?0014",
    );
    assert_eq!(&r, "foo1=bar; foo2=qux");
}

#[test]
fn test_0015() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0015",
        &["a=b", "z=y"],
        "http://home.example.org:8888/cookie-parser-result?0015",
    );
    assert_eq!(&r, "a=b; z=y");
}

#[test]
fn test_0016() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0016",
        &["z=y", "a=b"],
        "http://home.example.org:8888/cookie-parser-result?0016",
    );
    assert_eq!(&r, "z=y; a=b");
}

#[test]
fn test_0017() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0017",
        &["z=y, a=b"],
        "http://home.example.org:8888/cookie-parser-result?0017",
    );
    assert_eq!(&r, "z=y, a=b");
}

#[test]
fn test_0018() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0018",
        &["z=y; foo=bar, a=b"],
        "http://home.example.org:8888/cookie-parser-result?0018",
    );
    assert_eq!(&r, "z=y");
}

#[test]
fn test_0019() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0019",
        &["foo=b;max-age=3600, c=d;path=/"],
        "http://home.example.org:8888/cookie-parser-result?0019",
    );
    assert_eq!(&r, "foo=b");
}

#[test]
fn test_0020() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0020",
        &["a=b", "=", "c=d"],
        "http://home.example.org:8888/cookie-parser-result?0020",
    );
    assert_eq!(&r, "a=b; c=d");
}

#[test]
fn test_0021() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0021",
        &["a=b", "=x", "c=d"],
        "http://home.example.org:8888/cookie-parser-result?0021",
    );
    assert_eq!(&r, "a=b; c=d");
}

#[test]
fn test_0022() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0022",
        &["a=b", "x=", "c=d"],
        "http://home.example.org:8888/cookie-parser-result?0022",
    );
    assert_eq!(&r, "a=b; x=; c=d");
}

#[test]
fn test_0023() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0023",
        &["foo"],
        "http://home.example.org:8888/cookie-parser-result?0023",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0024() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0024",
        &["foo", "="],
        "http://home.example.org:8888/cookie-parser-result?0024",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0025() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0025",
        &["foo", "; bar"],
        "http://home.example.org:8888/cookie-parser-result?0025",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0026() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0026",
        &["foo"],
        "http://home.example.org:8888/cookie-parser-result?0026",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0027() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0027",
        &["foo", "bar"],
        "http://home.example.org:8888/cookie-parser-result?0027",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_0028() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?0028",
        &["foo"],
        "http://home.example.org:8888/cookie-parser-result?0028",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0001",
        &["foo=bar; Secure"],
        "http://home.example.org:8888/cookie-parser-result?attribute0001",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0002",
        &["foo=bar; seCURe"],
        "http://home.example.org:8888/cookie-parser-result?attribute0002",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0003",
        &["foo=bar; \"Secure\""],
        "http://home.example.org:8888/cookie-parser-result?attribute0003",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_attribute0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0004",
        &["foo=bar; Secure="],
        "http://home.example.org:8888/cookie-parser-result?attribute0004",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0005",
        &["foo=bar; Secure=aaaa"],
        "http://home.example.org:8888/cookie-parser-result?attribute0005",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0006",
        &["foo=bar; Secure qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0006",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_attribute0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0007",
        &["foo=bar; Secure =aaaaa"],
        "http://home.example.org:8888/cookie-parser-result?attribute0007",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0008() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0008",
        &["foo=bar; Secure= aaaaa"],
        "http://home.example.org:8888/cookie-parser-result?attribute0008",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0009() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0009",
        &["foo=bar; Secure; qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0009",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0010() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0010",
        &["foo=bar; Secure;qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0010",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0011() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0011",
        &["foo=bar; Secure    ; qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0011",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0012() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0012",
        &["foo=bar;                Secure"],
        "http://home.example.org:8888/cookie-parser-result?attribute0012",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0013() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0013",
        &["foo=bar;       Secure     ;"],
        "http://home.example.org:8888/cookie-parser-result?attribute0013",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0014() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0014",
        &["foo=bar; Path"],
        "http://home.example.org:8888/cookie-parser-result?attribute0014",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_attribute0015() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0015",
        &["foo=bar; Path="],
        "http://home.example.org:8888/cookie-parser-result?attribute0015",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_attribute0016() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0016",
        &["foo=bar; Path=/"],
        "http://home.example.org:8888/cookie-parser-result?attribute0016",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_attribute0017() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0017",
        &["foo=bar; Path=/qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0017",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0018() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0018",
        &["foo=bar; Path    =/qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0018",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0019() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0019",
        &["foo=bar; Path=    /qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0019",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0020() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0020",
        &["foo=bar; Path=/qux      ; taz"],
        "http://home.example.org:8888/cookie-parser-result?attribute0020",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0021() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0021",
        &["foo=bar; Path=/qux; Path=/"],
        "http://home.example.org:8888/cookie-parser-result?attribute0021",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_attribute0022() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0022",
        &["foo=bar; Path=/; Path=/qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0022",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0023() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0023",
        &["foo=bar; Path=/qux; Path=/cookie-parser-result"],
        "http://home.example.org:8888/cookie-parser-result?attribute0023",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_attribute0024() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0024",
        &["foo=bar; Path=/cookie-parser-result; Path=/qux"],
        "http://home.example.org:8888/cookie-parser-result?attribute0024",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0025() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0025",
        &["foo=bar; qux; Secure"],
        "http://home.example.org:8888/cookie-parser-result?attribute0025",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_attribute0026() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?attribute0026",
        &["foo=bar; qux=\"aaa;bbb\"; Secure"],
        "http://home.example.org:8888/cookie-parser-result?attribute0026",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_charset0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?charset0001",
        &[
            "foo=\u{6625}\u{8282}\u{56de}\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}\u{5b8c}\
             \u{5168}\u{624b}\u{518c}",
        ],
        "http://home.example.org:8888/cookie-parser-result?charset0001",
    );
    assert_eq!(
        &r,
        "foo=\u{6625}\u{8282}\u{56de}\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}\u{5b8c}\
         \u{5168}\u{624b}\u{518c}"
    );
}

#[test]
fn test_charset0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?charset0002",
        &[
            "\u{6625}\u{8282}\u{56de}=\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}\u{5b8c}\
             \u{5168}\u{624b}\u{518c}",
        ],
        "http://home.example.org:8888/cookie-parser-result?charset0002",
    );
    assert_eq!(
        &r,
        "\u{6625}\u{8282}\u{56de}=\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}\u{5b8c}\
         \u{5168}\u{624b}\u{518c}"
    );
}

#[test]
fn test_charset0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?charset0003",
        &[
            "\u{6625}\u{8282}\u{56de}=\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}; \u{5b8c}\
             \u{5168}\u{624b}\u{518c}",
        ],
        "http://home.example.org:8888/cookie-parser-result?charset0003",
    );
    assert_eq!(
        &r,
        "\u{6625}\u{8282}\u{56de}=\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}"
    );
}

#[test]
fn test_charset0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?charset0004",
        &[
            "foo=\"\u{6625}\u{8282}\u{56de}\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}\u{5b8c}\
             \u{5168}\u{624b}\u{518c}\"",
        ],
        "http://home.example.org:8888/cookie-parser-result?charset0004",
    );
    assert_eq!(
        &r,
        "foo=\"\u{6625}\u{8282}\u{56de}\u{5bb6}\u{8def}\u{b7}\u{6625}\u{8fd0}\u{5b8c}\
         \u{5168}\u{624b}\u{518c}\""
    );
}

#[test]
fn test_chromium0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0001",
        &["a=b"],
        "http://home.example.org:8888/cookie-parser-result?chromium0001",
    );
    assert_eq!(&r, "a=b");
}

#[test]
fn test_chromium0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0002",
        &["aBc=\"zzz \"   ;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0002",
    );
    assert_eq!(&r, "aBc=\"zzz \"");
}

#[test]
fn test_chromium0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0003",
        &["aBc=\"zzz \" ;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0003",
    );
    assert_eq!(&r, "aBc=\"zzz \"");
}

#[test]
fn test_chromium0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0004",
        &["aBc=\"zz;pp\" ; ;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0004",
    );
    assert_eq!(&r, "aBc=\"zz");
}

#[test]
fn test_chromium0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0005",
        &["aBc=\"zz ;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0005",
    );
    assert_eq!(&r, "aBc=\"zz");
}

#[test]
fn test_chromium0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0006",
        &["aBc=\"zzz \"   \"ppp\"  ;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0006",
    );
    assert_eq!(&r, "aBc=\"zzz \"   \"ppp\"");
}

#[test]
fn test_chromium0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0007",
        &["aBc=\"zzz \"   \"ppp\" ;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0007",
    );
    assert_eq!(&r, "aBc=\"zzz \"   \"ppp\"");
}

#[test]
fn test_chromium0008() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0008",
        &["aBc=A\"B ;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0008",
    );
    assert_eq!(&r, "aBc=A\"B");
}

#[test]
fn test_chromium0009() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0009",
        &["BLAHHH; path=/;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0009",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_chromium0010() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0010",
        &["\"BLA\\\"HHH\"; path=/;"],
        "http://home.example.org:8888/cookie-parser-result?chromium0010",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_chromium0011() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0011",
        &["a=\"B"],
        "http://home.example.org:8888/cookie-parser-result?chromium0011",
    );
    assert_eq!(&r, "a=\"B");
}

#[test]
fn test_chromium0012() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0012",
        &["=ABC"],
        "http://home.example.org:8888/cookie-parser-result?chromium0012",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_chromium0013() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0013",
        &["ABC=;  path = /"],
        "http://home.example.org:8888/cookie-parser-result?chromium0013",
    );
    assert_eq!(&r, "ABC=");
}

#[test]
fn test_chromium0014() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0014",
        &["  A  = BC  ;foo;;;   bar"],
        "http://home.example.org:8888/cookie-parser-result?chromium0014",
    );
    assert_eq!(&r, "A=BC");
}

#[test]
fn test_chromium0015() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0015",
        &["  A=== BC  ;foo;;;   bar"],
        "http://home.example.org:8888/cookie-parser-result?chromium0015",
    );
    assert_eq!(&r, "A=== BC");
}

#[test]
fn test_chromium0016() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0016",
        &[
            "foo=\"zohNumRKgI0oxyhSsV3Z7D\"  ; expires=Sun, 18-Apr-2027 21:06:29 GMT\
             ; path=/  ;",
        ],
        "http://home.example.org:8888/cookie-parser-result?chromium0016",
    );
    assert_eq!(&r, "foo=\"zohNumRKgI0oxyhSsV3Z7D\"");
}

#[test]
fn test_chromium0017() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0017",
        &[
            "foo=zohNumRKgI0oxyhSsV3Z7D  ; expires=Sun, 18-Apr-2027 21:06:29 GMT ; p\
             ath=/  ;",
        ],
        "http://home.example.org:8888/cookie-parser-result?chromium0017",
    );
    assert_eq!(&r, "foo=zohNumRKgI0oxyhSsV3Z7D");
}

#[test]
fn test_chromium0018() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0018",
        &[],
        "http://home.example.org:8888/cookie-parser-result?chromium0018",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_chromium0019() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0019",
        &[
            "a=aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
             aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
        ],
        "http://home.example.org:8888/cookie-parser-result?chromium0019",
    );
    assert_eq!(
        &r,
        "a=aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\
         aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
    );
}

#[test]
fn test_chromium0021() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?chromium0021",
        &[],
        "http://home.example.org:8888/cookie-parser-result?chromium0021",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_comma0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?comma0001",
        &["foo=bar, baz=qux"],
        "http://home.example.org:8888/cookie-parser-result?comma0001",
    );
    assert_eq!(&r, "foo=bar, baz=qux");
}

#[test]
fn test_comma0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?comma0002",
        &["foo=\"bar, baz=qux\""],
        "http://home.example.org:8888/cookie-parser-result?comma0002",
    );
    assert_eq!(&r, "foo=\"bar, baz=qux\"");
}

#[test]
fn test_comma0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?comma0003",
        &["foo=bar; b,az=qux"],
        "http://home.example.org:8888/cookie-parser-result?comma0003",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_comma0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?comma0004",
        &["foo=bar; baz=q,ux"],
        "http://home.example.org:8888/cookie-parser-result?comma0004",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_comma0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?comma0005",
        &["foo=bar; Max-Age=50,399"],
        "http://home.example.org:8888/cookie-parser-result?comma0005",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_comma0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?comma0006",
        &["foo=bar; Expires=Fri, 07 Aug 2019 08:04:19 GMT"],
        "http://home.example.org:8888/cookie-parser-result?comma0006",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_comma0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?comma0007",
        &["foo=bar; Expires=Fri 07 Aug 2019 08:04:19 GMT, baz=qux"],
        "http://home.example.org:8888/cookie-parser-result?comma0007",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0001",
        &["foo=bar; domain=home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0001",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0002",
        &["foo=bar; domain=home.example.org"],
        "http://sibling.example.org:8888/cookie-parser-result?domain0002",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0003",
        &["foo=bar; domain=.home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0003",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0004",
        &["foo=bar; domain=home.example.org"],
        "http://subdomain.home.example.org:8888/cookie-parser-result?domain0004",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0005",
        &["foo=bar; domain=.home.example.org"],
        "http://subdomain.home.example.org:8888/cookie-parser-result?domain0005",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0006",
        &["foo=bar; domain=.home.example.org"],
        "http://sibling.example.org:8888/cookie-parser-result?domain0006",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0007",
        &["foo=bar; domain=sibling.example.org"],
        "http://sibling.example.org:8888/cookie-parser-result?domain0007",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0008() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0008",
        &["foo=bar; domain=.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0008",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0009() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0009",
        &["foo=bar; domain=example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0009",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0010() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0010",
        &["foo=bar; domain=..home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0010",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0011() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0011",
        &["foo=bar; domain=home..example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0011",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0012() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0012",
        &["foo=bar; domain=  .home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0012",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0013() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0013",
        &["foo=bar; domain=  .  home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0013",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0014() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0014",
        &["foo=bar; domain=home.example.org."],
        "http://home.example.org:8888/cookie-parser-result?domain0014",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0015() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0015",
        &["foo=bar; domain=home.example.org.."],
        "http://home.example.org:8888/cookie-parser-result?domain0015",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0016() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0016",
        &["foo=bar; domain=home.example.org ."],
        "http://home.example.org:8888/cookie-parser-result?domain0016",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0017() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0017",
        &["foo=bar; domain=.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0017",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0018() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0018",
        &["foo=bar; domain=.org."],
        "http://home.example.org:8888/cookie-parser-result?domain0018",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0019() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0019",
        &[
            "foo=bar; domain=home.example.org",
            "foo2=bar2; domain=.home.example.org",
        ],
        "http://home.example.org:8888/cookie-parser-result?domain0019",
    );
    assert_eq!(&r, "foo=bar; foo2=bar2");
}

#[test]
fn test_domain0020() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0020",
        &[
            "foo2=bar2; domain=.home.example.org",
            "foo=bar; domain=home.example.org",
        ],
        "http://home.example.org:8888/cookie-parser-result?domain0020",
    );
    assert_eq!(&r, "foo2=bar2; foo=bar");
}

#[test]
fn test_domain0021() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0021",
        &["foo=bar; domain=\"home.example.org\""],
        "http://home.example.org:8888/cookie-parser-result?domain0021",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0022() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0022",
        &[
            "foo=bar; domain=home.example.org",
            "foo2=bar2; domain=.example.org",
        ],
        "http://home.example.org:8888/cookie-parser-result?domain0022",
    );
    assert_eq!(&r, "foo=bar; foo2=bar2");
}

#[test]
fn test_domain0023() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0023",
        &[
            "foo2=bar2; domain=.example.org",
            "foo=bar; domain=home.example.org",
        ],
        "http://home.example.org:8888/cookie-parser-result?domain0023",
    );
    assert_eq!(&r, "foo2=bar2; foo=bar");
}

#[test]
fn test_domain0024() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0024",
        &["foo=bar; domain=.example.org; domain=home.example.org"],
        "http://sibling.example.org:8888/cookie-parser-result?domain0024",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0025() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0025",
        &["foo=bar; domain=home.example.org; domain=.example.org"],
        "http://sibling.example.org:8888/cookie-parser-result?domain0025",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0026() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0026",
        &["foo=bar; domain=home.eXaMpLe.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0026",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0027() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0027",
        &["foo=bar; domain=home.example.org:8888"],
        "http://home.example.org:8888/cookie-parser-result?domain0027",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0028() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0028",
        &["foo=bar; domain=subdomain.home.example.org"],
        "http://subdomain.home.example.org:8888/cookie-parser-result?domain0028",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0029() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0029",
        &["foo=bar"],
        "http://subdomain.home.example.org:8888/cookie-parser-result?domain0029",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0031() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0031",
        &["foo=bar; domain=home.example.org; domain=.example.org"],
        "http://sibling.example.org:8888/cookie-parser-result?domain0031",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0033() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0033",
        &["foo=bar; domain=home.example.org"],
        "http://hoMe.eXaMplE.org:8888/cookie-parser-result?domain0033",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0034() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0034",
        &["foo=bar; domain=home.example.org; domain=home.example.com"],
        "http://home.example.org:8888/cookie-parser-result?domain0034",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0035() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0035",
        &["foo=bar; domain=home.example.com; domain=home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0035",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0036() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0036",
        &[
            "foo=bar; domain=home.example.org; domain=home.example.com; domain=home.\
             example.org",
        ],
        "http://home.example.org:8888/cookie-parser-result?domain0036",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0037() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0037",
        &[
            "foo=bar; domain=home.example.com; domain=home.example.org; domain=home.\
             example.com",
        ],
        "http://home.example.org:8888/cookie-parser-result?domain0037",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0038() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0038",
        &["foo=bar; domain=home.example.org; domain=home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0038",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0039() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0039",
        &["foo=bar; domain=home.example.org; domain=example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0039",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0040() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0040",
        &["foo=bar; domain=example.org; domain=home.example.org"],
        "http://home.example.org:8888/cookie-parser-result?domain0040",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_domain0041() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0041",
        &["foo=bar; domain=.sibling.example.org"],
        "http://sibling.example.org:8888/cookie-parser-result?domain0041",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_domain0042() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?domain0042",
        &["foo=bar; domain=.sibling.home.example.org"],
        "http://sibling.home.example.org:8888/cookie-parser-result?domain0042",
    );
    assert_eq!(&r, "");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0001",
        &["foo=bar; max-age=-1"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0001",
    );
    assert_eq!(&r, "");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0002",
        &["foo=bar; max-age=0"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0002",
    );
    assert_eq!(&r, "");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0003",
        &["foo=bar; expires=Thu, 10 Apr 1980 16:33:12 GMT"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0003",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0004",
        &["foo=bar; max-age=60"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0004",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0005",
        &["foo=bar; max-age=-20"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0005",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0006",
        &["foo=bar; max-age=60"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0006",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0007",
        &["foo=bar; expires=Thu, 10 Apr 1980 16:33:12 GMT"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0007",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0008() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0008",
        &["foo=bar; max-age=60", "foo1=bar; max-age=60"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0008",
    );
    assert_eq!(&r, "foo=bar; foo1=bar");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0009() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0009",
        &[
            "foo=bar; max-age=60",
            "foo1=bar; max-age=60",
            "foo=differentvalue; max-age=0",
        ],
        "http://home.example.org:8888/cookie-parser-result?mozilla0009",
    );
    assert_eq!(&r, "foo1=bar");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0010() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0010",
        &[
            "foo=bar; max-age=60",
            "foo1=bar; max-age=60",
            "foo=differentvalue; max-age=0",
            "foo2=evendifferentvalue; max-age=0",
        ],
        "http://home.example.org:8888/cookie-parser-result?mozilla0010",
    );
    assert_eq!(&r, "foo1=bar");
}

#[test]
fn test_mozilla0011() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0011",
        &[
            "test=parser; domain=.parser.test; ;; ;=; ,,, ===,abc,=; abracadabra! ma\
             x-age=20;=;;",
        ],
        "http://home.example.org:8888/cookie-parser-result?mozilla0011",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0012() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0012",
        &[
            "test=\"fubar! = foo;bar\\\";\" parser; max-age=6",
            "five; max-age=2.63,",
        ],
        "http://home.example.org:8888/cookie-parser-result?mozilla0012",
    );
    assert_eq!(&r, "test=\"fubar! = foo");
}

#[test]
#[should_panic] // Look at cookie_http_state_utils.py if this test fails
fn test_mozilla0013() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0013",
        &["test=kill; max-age=0", "five; max-age=0"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0013",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0014() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0014",
        &["six"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0014",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0015() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0015",
        &["six", "seven"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0015",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0016() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0016",
        &["six", "seven", " =eight"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0016",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_mozilla0017() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?mozilla0017",
        &["six", "seven", " =eight", "test=six"],
        "http://home.example.org:8888/cookie-parser-result?mozilla0017",
    );
    assert_eq!(&r, "test=six");
}

#[test]
fn test_name0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0001",
        &["a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0001",
    );
    assert_eq!(&r, "a=bar");
}

#[test]
fn test_name0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0002",
        &["1=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0002",
    );
    assert_eq!(&r, "1=bar");
}

#[test]
fn test_name0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0003",
        &["$=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0003",
    );
    assert_eq!(&r, "$=bar");
}

#[test]
fn test_name0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0004",
        &["!a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0004",
    );
    assert_eq!(&r, "!a=bar");
}

#[test]
fn test_name0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0005",
        &["@a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0005",
    );
    assert_eq!(&r, "@a=bar");
}

#[test]
fn test_name0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0006",
        &["#a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0006",
    );
    assert_eq!(&r, "#a=bar");
}

#[test]
fn test_name0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0007",
        &["$a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0007",
    );
    assert_eq!(&r, "$a=bar");
}

#[test]
fn test_name0008() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0008",
        &["%a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0008",
    );
    assert_eq!(&r, "%a=bar");
}

#[test]
fn test_name0009() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0009",
        &["^a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0009",
    );
    assert_eq!(&r, "^a=bar");
}

#[test]
fn test_name0010() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0010",
        &["&a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0010",
    );
    assert_eq!(&r, "&a=bar");
}

#[test]
fn test_name0011() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0011",
        &["*a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0011",
    );
    assert_eq!(&r, "*a=bar");
}

#[test]
fn test_name0012() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0012",
        &["(a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0012",
    );
    assert_eq!(&r, "(a=bar");
}

#[test]
fn test_name0013() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0013",
        &[")a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0013",
    );
    assert_eq!(&r, ")a=bar");
}

#[test]
fn test_name0014() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0014",
        &["-a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0014",
    );
    assert_eq!(&r, "-a=bar");
}

#[test]
fn test_name0015() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0015",
        &["_a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0015",
    );
    assert_eq!(&r, "_a=bar");
}

#[test]
fn test_name0016() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0016",
        &["+=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0016",
    );
    assert_eq!(&r, "+=bar");
}

#[test]
fn test_name0017() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0017",
        &["=a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0017",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_name0018() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0018",
        &["a =bar"],
        "http://home.example.org:8888/cookie-parser-result?name0018",
    );
    assert_eq!(&r, "a=bar");
}

#[test]
fn test_name0019() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0019",
        &["\"a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0019",
    );
    assert_eq!(&r, "\"a=bar");
}

#[test]
fn test_name0020() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0020",
        &["\"a=b\"=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0020",
    );
    assert_eq!(&r, "\"a=b\"=bar");
}

#[test]
fn test_name0021() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0021",
        &["\"a=b\"=bar", "\"a=qux"],
        "http://home.example.org:8888/cookie-parser-result?name0021",
    );
    assert_eq!(&r, "\"a=qux");
}

#[test]
fn test_name0022() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0022",
        &["   foo=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0022",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_name0023() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0023",
        &["foo;bar=baz"],
        "http://home.example.org:8888/cookie-parser-result?name0023",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_name0024() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0024",
        &["$Version=1; foo=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0024",
    );
    assert_eq!(&r, "$Version=1");
}

#[test]
fn test_name0025() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0025",
        &["===a=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0025",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_name0026() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0026",
        &["foo=bar"],
        "http://home.example.org:8888/cookie-parser-result?name0026",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_name0027() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0027",
        &["foo=bar    ;"],
        "http://home.example.org:8888/cookie-parser-result?name0027",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_name0028() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0028",
        &["=a"],
        "http://home.example.org:8888/cookie-parser-result?name0028",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_name0029() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0029",
        &["="],
        "http://home.example.org:8888/cookie-parser-result?name0029",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_name0030() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0030",
        &["foo bar=baz"],
        "http://home.example.org:8888/cookie-parser-result?name0030",
    );
    assert_eq!(&r, "foo bar=baz");
}

#[test]
fn test_name0031() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0031",
        &["\"foo;bar\"=baz"],
        "http://home.example.org:8888/cookie-parser-result?name0031",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_name0032() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0032",
        &["\"foo\\\"bar;baz\"=qux"],
        "http://home.example.org:8888/cookie-parser-result?name0032",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_name0033() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?name0033",
        &["=foo=bar", "aaa"],
        "http://home.example.org:8888/cookie-parser-result?name0033",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_optional_domain0030() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?optional-domain0030",
        &["foo=bar; domain="],
        "http://home.example.org:8888/cookie-parser-result?optional-domain0030",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_optional_domain0041() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?optional-domain0041",
        &["foo=bar; domain=example.org; domain="],
        "http://home.example.org:8888/cookie-parser-result?optional-domain0041",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_optional_domain0042() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?optional-domain0042",
        &["foo=bar; domain=foo.example.org; domain="],
        "http://home.example.org:8888/cookie-parser-result?optional-domain0042",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_optional_domain0043() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?optional-domain0043",
        &["foo=bar; domain=foo.example.org; domain="],
        "http://subdomain.home.example.org:8888/cookie-parser-result?optional-do\
         main0043",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_ordering0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?ordering0001",
        &[
            "key=val0;",
            "key=val1; path=/cookie-parser-result",
            "key=val2; path=/",
            "key=val3; path=/bar",
            "key=val4; domain=.example.org",
            "key=val5; domain=.example.org; path=/cookie-parser-result/foo",
        ],
        "http://home.example.org:8888/cookie-parser-result/foo/baz?ordering0001",
    );
    assert_eq!(&r, "key=val5; key=val1; key=val2; key=val4");
}

#[test]
fn test_path0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0001",
        &["a=b; path=/", "x=y; path=/cookie-parser-result"],
        "http://home.example.org:8888/cookie-parser-result?path0001",
    );
    assert_eq!(&r, "x=y; a=b");
}

#[test]
fn test_path0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0002",
        &["a=b; path=/cookie-parser-result", "x=y; path=/"],
        "http://home.example.org:8888/cookie-parser-result?path0002",
    );
    assert_eq!(&r, "a=b; x=y");
}

#[test]
fn test_path0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0003",
        &["x=y; path=/", "a=b; path=/cookie-parser-result"],
        "http://home.example.org:8888/cookie-parser-result?path0003",
    );
    assert_eq!(&r, "a=b; x=y");
}

#[test]
fn test_path0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0004",
        &["x=y; path=/cookie-parser-result", "a=b; path=/"],
        "http://home.example.org:8888/cookie-parser-result?path0004",
    );
    assert_eq!(&r, "x=y; a=b");
}

#[test]
fn test_path0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0005",
        &["foo=bar; path=/cookie-parser-result/foo"],
        "http://home.example.org:8888/cookie-parser-result?path0005",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0006",
        &["foo=bar", "foo=qux; path=/cookie-parser-result/foo"],
        "http://home.example.org:8888/cookie-parser-result?path0006",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0007() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0007",
        &["foo=bar; path=/cookie-parser-result/foo"],
        "http://home.example.org:8888/cookie-parser-result/foo?path0007",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0008() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0008",
        &["foo=bar; path=/cookie-parser-result/foo"],
        "http://home.example.org:8888/cookie-parser-result/bar?path0008",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0009() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0009",
        &["foo=bar; path=/cookie-parser-result/foo/qux"],
        "http://home.example.org:8888/cookie-parser-result/foo?path0009",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0010() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0010",
        &["foo=bar; path=/cookie-parser-result/foo/qux"],
        "http://home.example.org:8888/cookie-parser-result/foo/qux?path0010",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0011() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0011",
        &["foo=bar; path=/cookie-parser-result/foo/qux"],
        "http://home.example.org:8888/cookie-parser-result/bar/qux?path0011",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0012() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0012",
        &["foo=bar; path=/cookie-parser-result/foo/qux"],
        "http://home.example.org:8888/cookie-parser-result/foo/baz?path0012",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0013() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0013",
        &["foo=bar; path=/cookie-parser-result/foo/qux/"],
        "http://home.example.org:8888/cookie-parser-result/foo/baz?path0013",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0014() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0014",
        &["foo=bar; path=/cookie-parser-result/foo/qux/"],
        "http://home.example.org:8888/cookie-parser-result/foo/qux?path0014",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0015() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0015",
        &["foo=bar; path=/cookie-parser-result/foo/qux/"],
        "http://home.example.org:8888/cookie-parser-result/foo/qux/?path0015",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0016() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0016",
        &["foo=bar; path=/cookie-parser-result/foo/"],
        "http://home.example.org:8888/cookie-parser-result/foo/qux?path0016",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0017() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0017",
        &["foo=bar; path=/cookie-parser-result/foo/"],
        "http://home.example.org:8888/cookie-parser-result/foo//qux?path0017",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0018() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0018",
        &["foo=bar; path=/cookie-parser-result/foo/"],
        "http://home.example.org:8888/cookie-parser-result/fooqux?path0018",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0019() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0019",
        &["foo=bar; path"],
        "http://home.example.org:8888/cookie-parser-result?path0019",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0020() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0020",
        &["foo=bar; path="],
        "http://home.example.org:8888/cookie-parser-result?path0020",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0021() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0021",
        &["foo=bar; path=/"],
        "http://home.example.org:8888/cookie-parser-result?path0021",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0022() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0022",
        &["foo=bar; path= /"],
        "http://home.example.org:8888/cookie-parser-result?path0022",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0023() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0023",
        &["foo=bar; Path=/cookie-PARSER-result"],
        "http://home.example.org:8888/cookie-parser-result?path0023",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0024() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0024",
        &["foo=bar; path=/cookie-parser-result/foo/qux?"],
        "http://home.example.org:8888/cookie-parser-result/foo/qux?path0024",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0025() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0025",
        &["foo=bar; path=/cookie-parser-result/foo/qux#"],
        "http://home.example.org:8888/cookie-parser-result/foo/qux?path0025",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0026() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0026",
        &["foo=bar; path=/cookie-parser-result/foo/qux;"],
        "http://home.example.org:8888/cookie-parser-result/foo/qux?path0026",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0027() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0027",
        &["foo=bar; path=\"/cookie-parser-result/foo/qux;\""],
        "http://home.example.org:8888/cookie-parser-result/foo/qux?path0027",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0028() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0028",
        &["foo=bar; path=/cookie-parser-result/f%6Fo/bar"],
        "http://home.example.org:8888/cookie-parser-result/foo/bar?path0028",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0029() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0029",
        &[
            "a=b; \tpath\t=\t/cookie-parser-result",
            "x=y; \tpath\t=\t/book",
        ],
        "http://home.example.org:8888/cookie-parser-result?path0029",
    );
    assert_eq!(&r, "a=b");
}

#[test]
fn test_path0030() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0030",
        &["foo=bar; path=/dog; path="],
        "http://home.example.org:8888/cookie-parser-result?path0030",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_path0031() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0031",
        &["foo=bar; path=; path=/dog"],
        "http://home.example.org:8888/cookie-parser-result?path0031",
    );
    assert_eq!(&r, "");
}

#[test]
fn test_path0032() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?path0032",
        &[
            "foo=bar; path=/cookie-parser-result",
            "foo=qux; path=/cookie-parser-result/",
        ],
        "http://home.example.org:8888/cookie-parser-result/dog?path0032",
    );
    assert_eq!(&r, "foo=qux; foo=bar");
}

#[test]
fn test_value0001() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?value0001",
        &["foo=  bar"],
        "http://home.example.org:8888/cookie-parser-result?value0001",
    );
    assert_eq!(&r, "foo=bar");
}

#[test]
fn test_value0002() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?value0002",
        &["foo=\"bar\""],
        "http://home.example.org:8888/cookie-parser-result?value0002",
    );
    assert_eq!(&r, "foo=\"bar\"");
}

#[test]
fn test_value0003() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?value0003",
        &["foo=\"  bar \""],
        "http://home.example.org:8888/cookie-parser-result?value0003",
    );
    assert_eq!(&r, "foo=\"  bar \"");
}

#[test]
fn test_value0004() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?value0004",
        &["foo=\"bar;baz\""],
        "http://home.example.org:8888/cookie-parser-result?value0004",
    );
    assert_eq!(&r, "foo=\"bar");
}

#[test]
fn test_value0005() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?value0005",
        &["foo=\"bar=baz\""],
        "http://home.example.org:8888/cookie-parser-result?value0005",
    );
    assert_eq!(&r, "foo=\"bar=baz\"");
}

#[test]
fn test_value0006() {
    let r = run(
        "http://home.example.org:8888/cookie-parser?value0006",
        &["\tfoo\t=\tbar\t \t;\tttt"],
        "http://home.example.org:8888/cookie-parser-result?value0006",
    );
    assert_eq!(&r, "foo=bar");
}
