/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

use std::cell::RefCell;

use dom_struct::dom_struct;
use servo_arc::Arc;
use style::shared_lock::{SharedRwLockReadGuard, ToCssWithGuard};
use style::stylesheets::{CssRuleType, SupportsRule};
use style_traits::ToCss;

use crate::dom::bindings::reflector::reflect_dom_object;
use crate::dom::bindings::root::DomRoot;
use crate::dom::bindings::str::DOMString;
use crate::dom::cssconditionrule::CSSConditionRule;
use crate::dom::cssrule::SpecificCSSRule;
use crate::dom::cssstylesheet::CSSStyleSheet;
use crate::dom::window::Window;
use crate::script_runtime::CanGc;

#[dom_struct]
pub(crate) struct CSSSupportsRule {
    cssconditionrule: CSSConditionRule,
    #[ignore_malloc_size_of = "Stylo"]
    #[no_trace]
    supportsrule: RefCell<Arc<SupportsRule>>,
}

impl CSSSupportsRule {
    fn new_inherited(
        parent_stylesheet: &CSSStyleSheet,
        supportsrule: Arc<SupportsRule>,
    ) -> CSSSupportsRule {
        let list = supportsrule.rules.clone();
        CSSSupportsRule {
            cssconditionrule: CSSConditionRule::new_inherited(parent_stylesheet, list),
            supportsrule: RefCell::new(supportsrule),
        }
    }

    #[cfg_attr(crown, allow(crown::unrooted_must_root))]
    pub(crate) fn new(
        window: &Window,
        parent_stylesheet: &CSSStyleSheet,
        supportsrule: Arc<SupportsRule>,
        can_gc: CanGc,
    ) -> DomRoot<CSSSupportsRule> {
        reflect_dom_object(
            Box::new(CSSSupportsRule::new_inherited(
                parent_stylesheet,
                supportsrule,
            )),
            window,
            can_gc,
        )
    }

    /// <https://drafts.csswg.org/css-conditional-3/#the-csssupportsrule-interface>
    pub(crate) fn get_condition_text(&self) -> DOMString {
        self.supportsrule.borrow().condition.to_css_string().into()
    }

    pub(crate) fn update_rule(
        &self,
        supportsrule: Arc<SupportsRule>,
        guard: &SharedRwLockReadGuard,
    ) {
        self.cssconditionrule
            .update_rules(supportsrule.rules.clone(), guard);
        *self.supportsrule.borrow_mut() = supportsrule;
    }
}

impl SpecificCSSRule for CSSSupportsRule {
    fn ty(&self) -> CssRuleType {
        CssRuleType::Supports
    }

    fn get_css(&self) -> DOMString {
        let guard = self.cssconditionrule.shared_lock().read();
        self.supportsrule.borrow().to_css_string(&guard).into()
    }
}
