// Copyright 2019 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'dart:async';

import 'package:flutter/foundation.dart';
import 'package:flutter_test/flutter_test.dart';

typedef AsyncVoidCallback = Future<void> Function();

/// Checks that any [debugPrint] output generated by `block` matches
/// the output given by `output`.
///
/// Calls to [debugPrint] inside `block` will not generate output to the
/// console, but if it doesn't match `output` then it will be presented
/// as part of the error message as the "actual" result.
///
/// Output to [debugPrint] that contains newlines is split into separate
/// entries when comparing to `output`.
///
/// This method must be `await`ed.
Future<void> checkOutput({
  @required AsyncVoidCallback block,
  List<String> output = const <String>[],
}) =>
    TestAsyncUtils.guard(() async {
      final DebugPrintCallback originalDebugPrint = debugPrint;
      final List<String> log = <String>[];
      debugPrint = (String message, {int wrapWidth}) {
        log.addAll(message.split('\n'));
      };
      try {
        await block();
      } finally {
        debugPrint = originalDebugPrint;
      }
      expect(log, output);
    });
