import semmle.code.cpp.Element
import semmle.code.cpp.exprs.Access
import semmle.code.cpp.Initializer
private import semmle.code.cpp.internal.ResolveClass

/**
 * A C/C++ variable.
 *
 * For local variables, there is a one-to-one correspondence between
 * `Variable` and `VariableDeclarationEntry`.
 *
 * For other types of variable, there is a one-to-many relationship between
 * `Variable` and `VariableDeclarationEntry`. For example, a `Parameter`
 * can have multiple declarations.
 */
class Variable extends Declaration, @variable {

  /** Gets the initializer of this variable, if any. */
  Initializer getInitializer() { result.getDeclaration() = this }

  /** Holds if this variable has an initializer. */
  predicate hasInitializer() { exists(this.getInitializer()) }

  /** Gets an access to this variable. */
  VariableAccess getAnAccess() { result.getTarget() = this }

  /**
   * Gets a specifier of this variable. This includes `extern`, `static`,
   * `auto`, `private`, `protected`, `public`. Specifiers of the *type* of
   * this variable, such as `const` and `volatile`, are instead accessed
   * through `this.getType().getASpecifier()`.
   */
  override Specifier getASpecifier() { varspecifiers(underlyingElement(this),unresolveElement(result)) }

  /** Gets an attribute of this variable. */
  Attribute getAnAttribute() { varattributes(underlyingElement(this),unresolveElement(result)) }

  /** Holds if this variable is `const`. */
  predicate isConst() { this.getType().isConst() }

  /** Holds if this variable is `volatile`. */
  predicate isVolatile() { this.getType().isVolatile() }

  /** Gets the name of this variable. */
  override string getName() { none() }

  /** Gets the type of this variable. */
  Type getType() { none() }

  /** Gets the type of this variable, after typedefs have been resolved. */
  Type getUnderlyingType() { result = this.getType().getUnderlyingType() }

  /** 
   * Gets the type of this variable, after specifiers have been deeply
   * stripped and typedefs have been resolved.
   */
  Type getUnspecifiedType() { result = this.getType().getUnspecifiedType() }

  /**
   * Gets the type of this variable prior to deduction caused by the C++11
   * `auto` keyword.
   *
   * If the type of this variable was not declared with the C++11 `auto`
   * keyword, then this predicate does not hold.
   *
   * If the type of this variable is completely `auto`, then `result` is an
   * instance of `AutoType`. For example:
   *
   *   `auto four = 4;`
   *
   * If the type of this variable is partially `auto`, then a descendant of
   * `result` is an instance of `AutoType`. For example:
   *
   *   `const auto& c = container;`
   */
  Type getTypeWithAuto() { autoderivation(underlyingElement(this), unresolveElement(result)) }

  /**
   * Holds if the type of this variable is declared using the C++ `auto`
   * keyword.
   */
  predicate declaredUsingAutoType() { autoderivation(underlyingElement(this), _) }

  override VariableDeclarationEntry getADeclarationEntry() {
    result.getDeclaration() = this
  }

  override Location getADeclarationLocation() {
    result = getADeclarationEntry().getLocation()
  }

  override VariableDeclarationEntry getDefinition() {
    result = getADeclarationEntry() and
    result.isDefinition()
  }

  override Location getDefinitionLocation() {
    result = getDefinition().getLocation()
  }

  override Location getLocation() {
    if exists(getDefinition()) then
      result = this.getDefinitionLocation()
    else
      result = this.getADeclarationLocation()
  }

  /**
   * Gets an expression that is assigned to this variable somewhere in the
   * program.
   */
  Expr getAnAssignedValue() {
    result = this.getInitializer().getExpr()
    or
    exists (ConstructorFieldInit cfi
    | cfi.getTarget() = this and result = cfi.getExpr())
    or
    exists (AssignExpr ae
    | ae.getLValue().(Access).getTarget() = this and result = ae.getRValue())
    or
    exists(AggregateLiteral l |
      this.getDeclaringType() = l.getType() and
      result = l.getChild(this.(Field).getInitializationOrder())
    )
  }

  /**
   * Gets an assignment expression that assigns to this variable.
   * For example: `x=...` or `x+=...`.
   */
  Assignment getAnAssignment() {
    result.getLValue() = this.getAnAccess()
  }

  /**
   * Holds if this variable is `constexpr`.
   */
  predicate isConstexpr() {
    this.hasSpecifier("is_constexpr")
  }

  /**
   * Holds if this variable is `thread_local`.
   */
  predicate isThreadLocal() {
    this.hasSpecifier("is_thread_local")
  }

  /**
   * Holds if this variable is constructed from `v` as a result
   * of template instantiation. If so, it originates either from a template
   * variable or from a variable nested in a template class.
   */
  predicate isConstructedFrom(Variable v) {
    variable_instantiation(underlyingElement(this), unresolveElement(v))
  }

  /**
   * Gets the `i`th template argument used to instantiate this variable from a
   * variable template. When called on a variable template, this will return the
   * `i`th template parameter.
   */
  override Type getTemplateArgument(int index) {
    variable_template_argument(underlyingElement(this), index, unresolveElement(result))
  }

  /**
   * Holds if this is a compiler-generated variable. For example, a
   * [range-based for loop](http://en.cppreference.com/w/cpp/language/range-for)
   * typically has three compiler-generated variables, named `__range`,
   * `__begin`, and `__end`:
   *
   *    `for (char c : str) { ... }`
   */
  predicate isCompilerGenerated() { compgenerated(underlyingElement(this)) }
}

/**
 * A particular declaration or definition of a C/C++ variable.
 */
class VariableDeclarationEntry extends DeclarationEntry, @var_decl {
  override Variable getDeclaration() { result = getVariable() }

  override string getCanonicalQLClass() { result = "VariableDeclarationEntry" }
  
  /**
   * Gets the variable which is being declared or defined.
   */
  Variable getVariable() { var_decls(underlyingElement(this),unresolveElement(result),_,_,_) }

  /**
   * Gets the name, if any, used for the variable at this declaration or
   * definition.
   *
   * In most cases, this will be the name of the variable itself. The only
   * case in which it can differ is in a parameter declaration entry,
   * because the parameter may have a different name in the declaration
   * than in the definition. For example:
   *
   *    ```
   *    // Declaration. Parameter is named "x".
   *    int f(int x);
   *
   *    // Definition. Parameter is named "y".
   *    int f(int y) { return y; }
   *    ```
   */
  override string getName() { var_decls(underlyingElement(this),_,_,result,_) and result != "" }

  /**
   * Gets the type of the variable which is being declared or defined.
   */
  override Type getType() { var_decls(underlyingElement(this),_,unresolveElement(result),_,_) }

  override Location getLocation() { var_decls(underlyingElement(this),_,_,_,result) }

  /**
   * Holds if this is a definition of a variable.
   *
   * This always holds for local variables and member variables, but need
   * not hold for global variables. In the case of function parameters,
   * this holds precisely when the enclosing `FunctionDeclarationEntry` is
   * a definition.
   */
  override predicate isDefinition() { var_def(underlyingElement(this)) }

  override string getASpecifier() { var_decl_specifiers(underlyingElement(this),result) }
}

/**
 * A parameter as described within a particular declaration or definition
 * of a C/C++ function.
 */
class ParameterDeclarationEntry extends VariableDeclarationEntry {
  ParameterDeclarationEntry() { param_decl_bind(underlyingElement(this),_,_) }
  
  override string getCanonicalQLClass() { result = "ParameterDeclarationEntry" }

  /**
   * Gets the function declaration or definition which this parameter
   * description is part of.
   */
  FunctionDeclarationEntry getFunctionDeclarationEntry() {
    param_decl_bind(underlyingElement(this),_,unresolveElement(result))
  }

  /**
   * Gets the zero-based index of this parameter.
   */
  int getIndex() { param_decl_bind(underlyingElement(this),result,_) }

  override string toString() {
    if exists(getName())
      then result = super.toString()
      else exists (string idx
           | idx = ((getIndex() + 1).toString() + "th")
                 .replaceAll("1th","1st")
                 .replaceAll("2th","2nd")
                 .replaceAll("3th","3rd")
                 .replaceAll("11st","11th")
                 .replaceAll("12nd","12th")
                 .replaceAll("13rd","13th")
           | if exists(getCanonicalName())
               then result = "declaration of " + getCanonicalName() +
                             " as anonymous " + idx + " parameter"
               else result = "declaration of " + idx + " parameter")
  }

  /**
   * Gets the name of this `ParameterDeclarationEntry` including it's type.
   *
   * For example: "int p".
   */
  string getTypedName() {
    exists(string typeString, string nameString |
      if exists(getType().getName()) then typeString = getType().getName() else typeString = "" and
      if exists(getName()) then nameString = getName() else nameString = "" and
      if (typeString != "" and nameString != "") then (
        result = typeString + " " + nameString
      ) else (
        result = typeString + nameString
      )
    )
  }
}

/**
 * A C/C++ variable with block scope [N4140 3.3.3]. In other words, a local
 * variable or a function parameter. Local variables can be static; use the
 * `isStatic` member predicate to detect those.
 */
class LocalScopeVariable extends Variable, @localscopevariable {
  /** Gets the function to which this variable belongs. */
  /*abstract*/ Function getFunction() { none() }
}

/**
 * DEPRECATED: use `LocalScopeVariable` instead.
 */
deprecated class StackVariable extends Variable {
  StackVariable() { this instanceof LocalScopeVariable }
  Function getFunction() {
    result = this.(LocalScopeVariable).getFunction()
  }
}

/**
 * A C/C++ local variable. In other words, any variable that has block
 * scope [N4140 3.3.3], but is not a parameter of a `Function` or `CatchBlock`.
 * Local variables can be static; use the `isStatic` member predicate to detect
 * those.
 *
 * A local variable can be declared by a `DeclStmt` or a `ConditionDeclExpr`.
 */
class LocalVariable extends LocalScopeVariable, @localvariable {
  override string getName() { localvariables(underlyingElement(this),_,result) }

  override Type getType() { localvariables(underlyingElement(this),unresolveElement(result),_) }

  override Function getFunction() {
    exists(DeclStmt s | s.getADeclaration() = this and s.getEnclosingFunction() = result) or
    exists(ConditionDeclExpr e | e.getVariable() = this and e.getEnclosingFunction() = result)
  }
}

/**
 * A C/C++ variable which has global scope or namespace scope.
 */
class GlobalOrNamespaceVariable extends Variable, @globalvariable {
  override string getName() { globalvariables(underlyingElement(this),_,result) }

  override Type getType() { globalvariables(underlyingElement(this),unresolveElement(result),_) }

  override Element getEnclosingElement() { none() }
}

/**
 * A C/C++ variable which has namespace scope.
 */
class NamespaceVariable extends GlobalOrNamespaceVariable {
  NamespaceVariable() {
    exists(Namespace n | namespacembrs(unresolveElement(n), underlyingElement(this)))
  }
}

/**
 * A C/C++ variable which has global scope.
 *
 * Note that variables declared in anonymous namespaces have namespace scope,
 * even though they are accessed in the same manner as variables declared in
 * the enclosing scope of said namespace (which may be the global scope).
 */
class GlobalVariable extends GlobalOrNamespaceVariable {
  GlobalVariable() {
    not this instanceof NamespaceVariable
  }
}

/**
 * A C structure member or C++ member variable.
 *
 * This includes static member variables in C++. To exclude static member
 * variables, use `Field` instead of `MemberVariable`.
 */
class MemberVariable extends Variable, @membervariable {
  MemberVariable() {
    this.isMember()
  }

  /** Holds if this member is private. */
  predicate isPrivate() { this.hasSpecifier("private") }

  /** Holds if this member is protected. */
  predicate isProtected() { this.hasSpecifier("protected") }

  /** Holds if this member is public. */
  predicate isPublic() { this.hasSpecifier("public") }

  override string getName() { membervariables(underlyingElement(this),_,result) }

  override Type getType() {
    if (strictcount(this.getAType()) = 1) then (
       result = this.getAType()
     ) else (
       // In rare situations a member variable may have multiple types in
       // different translation units. In that case, we return the unspecified
       // type.
       result = this.getAType().getUnspecifiedType()
    )
  }

  /** Holds if this member is mutable. */
  predicate isMutable() {
    getADeclarationEntry().hasSpecifier("mutable")
  }

  private Type getAType() { membervariables(underlyingElement(this),unresolveElement(result),_) }
}

/**
 * A C/C++ function pointer variable.
 *
 * DEPRECATED: use `Variable.getType() instanceof FunctionPointerType` instead. 
 */
deprecated class FunctionPointerVariable extends Variable {
  FunctionPointerVariable() {
    this.getType() instanceof FunctionPointerType
  }
}

/**
 * A C/C++ function pointer member variable.
 *
 * DEPRECATED: use `MemberVariable.getType() instanceof FunctionPointerType` instead.
 */
deprecated class FunctionPointerMemberVariable extends MemberVariable {
  FunctionPointerMemberVariable() {
    this instanceof FunctionPointerVariable
  }
}

/**
 * A C++14 variable template.
 */
class TemplateVariable extends Variable {
  TemplateVariable() { is_variable_template(underlyingElement(this)) }

  /**
   * Gets an instantiation of this variable template.
   */
  Variable getAnInstantiation() { result.isConstructedFrom(this) }
}

/**
 * A non-static local variable or parameter that is not part of an
 * uninstantiated template. Uninstantiated templates are purely syntax, and
 * only on instantiation will they be complete with information about types,
 * conversions, call targets, etc.
 */
class SemanticStackVariable extends LocalScopeVariable {
  SemanticStackVariable() {
    not this.isStatic() and
    not this.isFromUninstantiatedTemplate(_)
  }
}
