# Copyright (c) 2022 PaddlePaddle Authors. All Rights Reserved.
# Copyright 2020 The HuggingFace Team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os
import unittest

from paddlenlp.transformers.ernie.tokenizer import (
    BasicTokenizer,
    ErnieTokenizer,
    WordpieceTokenizer,
)

from ...testing_utils import slow
from ...transformers.test_tokenizer_common import (
    TokenizerTesterMixin,
    filter_non_english,
)


class ErnieCTMTokenizationTest(TokenizerTesterMixin, unittest.TestCase):

    tokenizer_class = ErnieTokenizer
    space_between_special_tokens = True
    from_pretrained_filter = filter_non_english
    test_seq2seq = True

    def setUp(self):
        super().setUp()

        vocab_tokens = [
            "[UNK]",
            "[CLS]",
            "[SEP]",
            "[PAD]",
            "[MASK]",
            "want",
            "##want",
            "##ed",
            "wa",
            "un",
            "runn",
            "##ing",
            ",",
            "low",
            "lowest",
        ]

        self.vocab_file = os.path.join(self.tmpdirname, ErnieTokenizer.resource_files_names["vocab_file"])
        with open(self.vocab_file, "w", encoding="utf-8") as vocab_writer:
            vocab_writer.write("".join([x + "\n" for x in vocab_tokens]))

    def get_input_output_texts(self, tokenizer):
        input_text = "UNwant\u00E9d,running"
        output_text = "unwanted, running"
        return input_text, output_text

    def test_full_tokenizer(self):
        tokenizer = self.tokenizer_class(self.vocab_file)

        tokens = tokenizer.tokenize("UNwant\u00E9d,running")
        self.assertListEqual(tokens, ["un", "##want", "##ed", ",", "runn", "##ing"])
        self.assertListEqual(tokenizer.convert_tokens_to_ids(tokens), [9, 6, 7, 12, 10, 11])

    def test_fast_and_python_full_tokenizer(self):
        if not self.test_fast_tokenizer:
            return

        tokenizer = self.get_tokenizer()
        tokenizer_fast = self.get_fast_tokenizer()

        sequence = "UNwant\u00E9d,running"
        tokens = tokenizer.tokenize(sequence)
        tokens_fast = tokenizer_fast.tokenize(sequence)
        self.assertListEqual(tokens, tokens_fast)

        ids = tokenizer.encode(sequence, add_special_tokens=False)["input_ids"]
        ids_fast = tokenizer_fast.encode(sequence, add_special_tokens=False)["input_ids"]
        self.assertListEqual(ids, ids_fast)

        ids = tokenizer.encode(sequence)["input_ids"]
        ids_fast = tokenizer_fast.encode(sequence)["input_ids"]
        self.assertListEqual(ids, ids_fast)

        tokenizer = self.get_tokenizer(do_lower_case=True)
        tokenizer_fast = self.get_fast_tokenizer(do_lower_case=True)

        tokens = tokenizer.tokenize(sequence)
        tokens_fast = tokenizer_fast.tokenize(sequence)
        self.assertListEqual(tokens, tokens_fast)

        ids = tokenizer.encode(sequence, add_special_tokens=False)["input_ids"]
        ids_fast = tokenizer_fast.encode(sequence, add_special_tokens=False)["input_ids"]
        self.assertListEqual(ids, ids_fast)

        ids = tokenizer.encode(sequence)["input_ids"]
        ids_fast = tokenizer_fast.encode(sequence)["input_ids"]
        self.assertListEqual(ids, ids_fast)

    def test_chinese(self):
        tokenizer = BasicTokenizer()

        self.assertListEqual(tokenizer.tokenize("ah\u535A\u63A8zz"), ["ah", "\u535A", "\u63A8", "zz"])

    def test_basic_tokenizer_lower(self):
        tokenizer = BasicTokenizer(do_lower_case=True)

        self.assertListEqual(
            tokenizer.tokenize(" \tHeLLo!how  \n Are yoU?  "), ["hello", "!", "how", "are", "you", "?"]
        )
        self.assertListEqual(tokenizer.tokenize("H\u00E9llo"), ["hello"])

    def test_basic_tokenizer_lower_strip_accents_false(self):
        tokenizer = BasicTokenizer(do_lower_case=True, strip_accents=False)

        self.assertListEqual(
            tokenizer.tokenize(" \tHäLLo!how  \n Are yoU?  "), ["hällo", "!", "how", "are", "you", "?"]
        )
        self.assertListEqual(tokenizer.tokenize("H\u00E9llo"), ["h\u00E9llo"])

    def test_basic_tokenizer_lower_strip_accents_true(self):
        tokenizer = BasicTokenizer(do_lower_case=True)

        self.assertListEqual(
            tokenizer.tokenize(" \tHäLLo!how  \n Are yoU?  "), ["hallo", "!", "how", "are", "you", "?"]
        )
        self.assertListEqual(tokenizer.tokenize("H\u00E9llo"), ["hello"])

    def test_basic_tokenizer_lower_strip_accents_default(self):
        tokenizer = BasicTokenizer(do_lower_case=True)

        self.assertListEqual(
            tokenizer.tokenize(" \tHäLLo!how  \n Are yoU?  "), ["hallo", "!", "how", "are", "you", "?"]
        )
        self.assertListEqual(tokenizer.tokenize("H\u00E9llo"), ["hello"])

    def test_basic_tokenizer_no_lower(self):
        tokenizer = BasicTokenizer(do_lower_case=False)

        self.assertListEqual(
            tokenizer.tokenize(" \tHeLLo!how  \n Are yoU?  "), ["HeLLo", "!", "how", "Are", "yoU", "?"]
        )

    def test_basic_tokenizer_no_lower_strip_accents_false(self):
        tokenizer = BasicTokenizer(do_lower_case=False, strip_accents=False)

        self.assertListEqual(
            tokenizer.tokenize(" \tHäLLo!how  \n Are yoU?  "), ["HäLLo", "!", "how", "Are", "yoU", "?"]
        )

    def test_basic_tokenizer_no_lower_strip_accents_true(self):
        tokenizer = BasicTokenizer(do_lower_case=False, strip_accents=True)

        self.assertListEqual(
            tokenizer.tokenize(" \tHäLLo!how  \n Are yoU?  "), ["HaLLo", "!", "how", "Are", "yoU", "?"]
        )

    def test_basic_tokenizer_respects_never_split_tokens(self):
        tokenizer = BasicTokenizer(do_lower_case=False, never_split=["[UNK]"])

        self.assertListEqual(
            tokenizer.tokenize(" \tHeLLo!how  \n Are yoU? [UNK]"), ["HeLLo", "!", "how", "Are", "yoU", "?", "[UNK]"]
        )

    def test_wordpiece_tokenizer(self):
        vocab_tokens = ["[UNK]", "[CLS]", "[SEP]", "want", "##want", "##ed", "wa", "un", "runn", "##ing"]

        vocab = {}
        for (i, token) in enumerate(vocab_tokens):
            vocab[token] = i
        tokenizer = WordpieceTokenizer(vocab=vocab, unk_token="[UNK]")

        self.assertListEqual(tokenizer.tokenize(""), [])

        self.assertListEqual(tokenizer.tokenize("unwanted running"), ["un", "##want", "##ed", "runn", "##ing"])

        self.assertListEqual(tokenizer.tokenize("unwantedX running"), ["[UNK]", "runn", "##ing"])

    def test_clean_text(self):
        tokenizer = self.get_tokenizer()

        # Example taken from the issue https://github.com/huggingface/tokenizers/issues/340
        self.assertListEqual([tokenizer.tokenize(t) for t in ["Test", "\xad", "test"]], [["[UNK]"], [], ["[UNK]"]])

    @slow
    def test_sequence_builders(self):
        tokenizer = self.tokenizer_class.from_pretrained("ernie-1.0")

        text = tokenizer.encode("sequence builders", return_token_type_ids=None, add_special_tokens=False)["input_ids"]
        text_2 = tokenizer.encode("multi-sequence build", return_token_type_ids=None, add_special_tokens=False)[
            "input_ids"
        ]

        encoded_sentence = tokenizer.build_inputs_with_special_tokens(text)
        encoded_pair = tokenizer.build_inputs_with_special_tokens(text, text_2)
        assert encoded_sentence == [1] + text + [2]
        assert encoded_pair == [1] + text + [2] + text_2 + [2]

    def test_offsets_with_special_characters(self):
        for tokenizer, pretrained_name, kwargs in self.tokenizers_list:
            with self.subTest(f"{tokenizer.__class__.__name__} ({pretrained_name})"):
                tokenizer = self.tokenizer_class.from_pretrained(pretrained_name, **kwargs)

                sentence = f"A, naïve {tokenizer.mask_token} AllenNLP sentence."
                tokens = tokenizer.encode(
                    sentence,
                    return_attention_mask=False,
                    return_token_type_ids=False,
                    return_offsets_mapping=True,
                    add_special_tokens=True,
                )

                do_lower_case = tokenizer.do_lower_case if hasattr(tokenizer, "do_lower_case") else False
                expected_results = (
                    [
                        ((0, 0), tokenizer.cls_token),
                        ((0, 1), "A"),
                        ((1, 2), ","),
                        ((3, 5), "na"),
                        ((5, 6), "##ï"),
                        ((6, 8), "##ve"),
                        ((9, 15), tokenizer.mask_token),
                        ((16, 21), "Allen"),
                        ((21, 23), "##NL"),
                        ((23, 24), "##P"),
                        ((25, 33), "sentence"),
                        ((33, 34), "."),
                        ((0, 0), tokenizer.sep_token),
                    ]
                    if not do_lower_case
                    else [
                        ((0, 0), tokenizer.cls_token),
                        ((0, 1), "a"),
                        ((1, 2), ","),
                        ((3, 5), "na"),
                        ((5, 8), "##ive"),
                        ((9, 15), tokenizer.mask_token),
                        ((16, 21), "allen"),
                        ((21, 22), "##n"),
                        ((22, 24), "##lp"),
                        ((25, 27), "se"),
                        ((27, 29), "##nt"),
                        ((29, 33), "##ence"),
                        ((33, 34), "."),
                        ((0, 0), tokenizer.sep_token),
                    ]
                )

                self.assertEqual(
                    [e[1] for e in expected_results], tokenizer.convert_ids_to_tokens(tokens["input_ids"])
                )
                self.assertEqual([e[0] for e in expected_results], tokens["offset_mapping"])

    @slow
    def test_with_emoji(self):
        tokenizer = self.tokenizer_class.from_pretrained("ernie-1.0")
        text = "好👌🏻"
        prompt = "评价维度"
        encoding = tokenizer(
            text=text,
            text_pair=prompt,
            runcation=True,
            pad_to_max_seq_len=True,
            return_attention_mask=True,
            return_position_ids=True,
            return_offsets_mapping=True,
        )
        self.assertEqual(len(encoding["input_ids"]), 10)
        self.assertEqual(len(encoding["offset_mapping"]), 10)

        text = "好👌🏻👌🏻👌🏻👌🏻👌🏻👌🏻"
        prompt = "评价维度"
        encoding = tokenizer(
            text=text,
            text_pair=prompt,
            runcation=True,
            pad_to_max_seq_len=True,
            return_attention_mask=True,
            return_position_ids=True,
            return_offsets_mapping=True,
        )
        self.assertEqual(len(encoding["input_ids"]), 15)
        self.assertEqual(len(encoding["offset_mapping"]), 15)
