package com.r3corda.node.services

import com.r3corda.core.messaging.Message
import com.r3corda.core.node.services.DEFAULT_SESSION_ID
import com.r3corda.core.testing.freeLocalHostAndPort
import com.r3corda.node.services.config.NodeConfiguration
import com.r3corda.node.services.messaging.ArtemisMessagingService
import org.assertj.core.api.Assertions.assertThatThrownBy
import org.junit.After
import org.junit.Rule
import org.junit.Test
import org.junit.rules.TemporaryFolder
import java.net.ServerSocket
import java.util.concurrent.LinkedBlockingQueue
import java.util.concurrent.TimeUnit.MILLISECONDS
import java.util.concurrent.TimeUnit.SECONDS
import kotlin.test.assertEquals
import kotlin.test.assertNotNull
import kotlin.test.assertNull

class ArtemisMessagingServiceTests {

    @Rule @JvmField val temporaryFolder = TemporaryFolder()

    val hostAndPort = freeLocalHostAndPort()
    val topic = "platform.self"

    var messagingNetwork: ArtemisMessagingService? = null

    @After
    fun cleanUp() {
        messagingNetwork?.stop()
    }

    @Test
    fun `starting with the port already bound`() {
        ServerSocket(hostAndPort.port).use {
            val messagingNetwork = createMessagingService()
            assertThatThrownBy { messagingNetwork.start() }
        }
    }

    @Test
    fun `sending message to self`() {
        val receivedMessages = LinkedBlockingQueue<Message>()

        val messagingNetwork = createMessagingService()
        messagingNetwork.start()

        messagingNetwork.addMessageHandler(topic) { message, r ->
            receivedMessages.add(message)
        }

        val message = messagingNetwork.createMessage(topic, DEFAULT_SESSION_ID, "first msg".toByteArray())
        messagingNetwork.send(message, messagingNetwork.myAddress)

        val actual = receivedMessages.poll(2, SECONDS)
        assertNotNull(actual)
        assertEquals("first msg", String(actual.data))
        assertNull(receivedMessages.poll(200, MILLISECONDS))
    }

    private fun createMessagingService(): ArtemisMessagingService {
        val config = object: NodeConfiguration {
            override val myLegalName: String = "me"
            override val exportJMXto: String = ""
            override val nearestCity: String = "London"
            override val keyStorePassword: String = "testpass"
            override val trustStorePassword: String = "trustpass"

        }
        return ArtemisMessagingService(temporaryFolder.newFolder().toPath(), hostAndPort, config).apply {
            configureWithDevSSLCertificate()
            messagingNetwork = this
        }
    }

}