/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * @format
 * @flow
 * @oncall react_native
 */

'use strict';

/*::
import type {BabelCoreOptions} from '@babel/core';
*/

const escapeRegExp = require('escape-string-regexp');
const fs = require('fs');
const path = require('path');

let _only /*: $ReadOnlyArray<RegExp | string> */ = [];

function register(onlyList /*: $ReadOnlyArray<RegExp | string> */) {
  // NB: `require('@babel/register')` registers Babel as a side-effect, and
  // also returns a register function that overrides the first registration
  // when called.
  //
  // If `require` is used between the two registrations, Babel will behave in
  // its default mode, searching for a config file and loading whatever
  // plugins/presets are specified within, to compile whatever is `require`d.
  //
  // Since our `config()` uses `require` to load plugins, and we don't want
  // these plugins to be compiled with an arbitrary Babel config, we must
  // prepare the config object before calling `require('@babel/register')`.
  const registerConfig = {
    ...config(onlyList),
    extensions: [
      '.ts',
      '.tsx',
      // Babel's default extensions
      '.es6',
      '.es',
      '.jsx',
      '.js',
      '.mjs',
    ],
  };

  require('@babel/register')(registerConfig);
}

function config(
  onlyList /*: $ReadOnlyArray<RegExp | string> */,
  options /*: ?$ReadOnly<{
    lazy?: boolean,
  }> */,
) /*: BabelCoreOptions */ {
  _only = _only.concat(onlyList);
  return {
    babelrc: false,
    compact: false,
    configFile: false,
    browserslistConfigFile: false,
    // make sure we don't transpile any npm packages
    ignore: [/\/node_modules\//],
    only: [..._only],
    plugins: [
      [require('@babel/plugin-transform-flow-strip-types').default],
      [
        require('@babel/plugin-transform-modules-commonjs').default,
        {
          lazy: options && options.lazy,
        },
      ],
    ],
    presets: [],
    retainLines: true,
    sourceMaps: 'inline',
    overrides: [
      {
        test: /\.tsx?$/,
        plugins: [
          require('babel-plugin-replace-ts-export-assignment'),
          require('./plugins/babel-plugin-metro-replace-ts-require-assignment'),
        ],
        presets: [
          [
            require('@babel/preset-typescript').default,
            {
              // will be the default in Babel 8, so let's just turn it on now
              allowDeclareFields: true,
              // will be default in the future, but we don't want to use it
              allowNamespaces: false,
            },
          ],
        ],
      },
    ],
  };
}

/**
 * We use absolute paths for matching only the top-level folders reliably. For
 * example, we would not want to match some deeply nested folder that happens to
 * have the same name as one of `BABEL_ENABLED_PATHS`.
 */
function buildRegExps(
  basePath /*: string */,
  dirPaths /*: $ReadOnlyArray<RegExp | string> */,
) /*: $ReadOnlyArray<RegExp> */ {
  return dirPaths.map(folderPath =>
    // Babel cares about Windows/Unix paths since v7b44
    // https://github.com/babel/babel/issues/8184
    // basePath + path.sep + dirPath/dirRegex
    // /home/name/webroot/js + / + relative/path/to/exclude
    // c:\home\name\webroot\js + \ + relative\path\to\exclude
    folderPath instanceof RegExp
      ? new RegExp(
          `^${escapeRegExp(path.resolve(basePath, '.') + path.sep)}${
            folderPath.source // This is an actual regex value, don't escape it.
          }`,
          folderPath.flags,
        )
      : new RegExp('^' + escapeRegExp(path.resolve(basePath, folderPath))),
  );
}

let isRegisteredForMetroMonorepo = false;

function registerForMetroMonorepo() {
  // Noop if we have already registered Babel here.
  if (isRegisteredForMetroMonorepo) {
    return;
  }
  // Noop if we are in NODE_ENV=production.
  if (process.env.NODE_ENV === 'production') {
    return;
  }
  // Noop if we seem to be outside of the Metro source tree.
  if (
    !__dirname.endsWith(
      ['', 'packages', 'metro-babel-register', 'src'].join(path.sep),
    )
  ) {
    return;
  }
  // Bail out if prepare-release has run here.
  if (
    fs.existsSync(
      path.join(__dirname, '..', 'src.real', path.basename(__filename)),
    )
  ) {
    return;
  }
  register([path.resolve(__dirname, '..', '..')]);
  isRegisteredForMetroMonorepo = true;
}

register.config = config;
register.buildRegExps = buildRegExps;
register.unstable_registerForMetroMonorepo = registerForMetroMonorepo;

module.exports = register;
