/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * @flow
 * @format
 * @oncall react_native
 */

'use strict';

const fs = require('fs');

let currentInspectorSession;
let isProfiling = false;

function getInspectorSession() {
  if (currentInspectorSession) {
    return currentInspectorSession;
  }
  const inspector = require('inspector');
  currentInspectorSession = new inspector.Session();
  currentInspectorSession.connect();
  return currentInspectorSession;
}

async function startProfiling() {
  if (isProfiling) {
    return;
  }

  const session = getInspectorSession();
  await new Promise(resolve => session.post('Profiler.enable', resolve));
  await new Promise(resolve => session.post('Profiler.start', resolve));
  isProfiling = true;
}

async function stopProfilingAndWrite(workerName: ?string) {
  if (!isProfiling) {
    return;
  }
  const session = getInspectorSession();

  const {profile} = await new Promise<any>((resolve, reject) =>
    session.post('Profiler.stop', (err, data) =>
      err ? reject(err) : resolve(data),
    ),
  );
  const name = 'buck-worker-tool' + (workerName ? '-' + workerName : '');
  fs.writeFileSync(
    `${name}-${process.pid}-${Date.now()}.cpuprofile`,
    JSON.stringify(profile),
    'utf8',
  );
  isProfiling = false;
}

module.exports = {
  stopProfilingAndWrite,
  startProfiling,
};
