import { isFunction } from 'lodash';
import redent from 'redent';
import trimNewlines from 'trim-newlines';

export default function createReadme(name, dir, extension, commandObject) {
    const projectExtensions = commandObject.context.projectExtensions;
    const rows = [];

    rows.push(`# ${name}`, '');

    // If we are documenting an extensions we will want to use the description inside projectExtensions
    if (extension) {
        const description = isFunction(projectExtensions[0].description) ?
            projectExtensions[0].description(commandObject, extension) :
            trimNewlines(redent(projectExtensions[0].description));

        rows.push(description, '');
    } else {
        // Could add a function / property on the roc.config.js file that can be used
        rows.push('This is automatically generated documentation for your Roc project.', '');
    }

    if (extension) {
        // Do nothing at this time
    } else {
        const packages = projectExtensions.filter((extn) => extn.type === 'package');
        const plugins = projectExtensions.filter((extn) => extn.type === 'plugin');

        rows.push('## Extensions');
        rows.push('The extensions that are used in the project.', '');

        rows.push('### Packages');
        if (packages.length > 0) {
            packages.forEach((pkg) => {
                rows.push(`#### ${pkg.name} — [v${pkg.version}](https://www.npmjs.com/package/${pkg.name})`);
                const description = isFunction(pkg.description) ?
                    pkg.description(commandObject, extension) :
                    pkg.description && trimNewlines(redent(pkg.description));

                rows.push(description, '');
            });
        } else {
            rows.push('_No packages._', '');
        }

        rows.push('### Plugins');
        if (plugins.length > 0) {
            plugins.forEach((plugin) => {
                rows.push(`#### ${plugin.name} — [v${plugin.version}](https://www.npmjs.com/package/${plugin.name})`);
                const description = isFunction(plugin.description) ?
                    plugin.description(commandObject, extension) :
                    plugin.description && trimNewlines(redent(plugin.description));

                rows.push(description, '');
            });
        } else {
            rows.push('_No plugins._', '');
        }
    }

    rows.push('## Documentation');
    [
        'Actions', 'Commands', 'Configuration', 'Dependencies', 'Hooks', 'Settings', 'Extensions',
    ].forEach((group) => {
        rows.push(`- [${group}](${dir}/${group}.md)`);
    });
    rows.push('');
    rows.push('---');
    rows.push('_Generated by [Roc](https://github.com/rocjs/roc)_\n');
    return rows.join('\n');
}
