/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.keycloak.quarkus.runtime.configuration;

import static org.keycloak.quarkus.runtime.configuration.MicroProfileConfigProvider.NS_QUARKUS;

import io.quarkus.runtime.configuration.AbstractRawDefaultConfigSource;
import io.smallrye.config.ConfigSourceInterceptor;
import io.smallrye.config.ConfigSourceInterceptorContext;
import io.smallrye.config.ConfigValue;
import org.keycloak.common.util.StringPropertyReplacer;
import org.keycloak.quarkus.runtime.Environment;
import org.keycloak.quarkus.runtime.configuration.mappers.PropertyMapper;
import org.keycloak.quarkus.runtime.configuration.mappers.PropertyMappers;

/**
 * <p>This interceptor is responsible for mapping Keycloak properties to their corresponding properties in Quarkus.
 * 
 * <p>A single property in Keycloak may span a single or multiple properties on Quarkus and for each property we want to map
 * from Quarkus we should configure a {@link PropertyMapper}.
 * 
 * <p>The {@link PropertyMapper} can either perform a 1:1 mapping where the value of a property from
 * Keycloak (e.g.: https.port) is mapped to a single properties in Quarkus, or perform a 1:N mapping where the value of a property
 * from Keycloak (e.g.: database) is mapped to multiple properties in Quarkus.
 */
public class PropertyMappingInterceptor implements ConfigSourceInterceptor {

    private final boolean isQuarkusPropertiesEnabled = QuarkusPropertiesConfigSource.isQuarkusPropertiesEnabled();

    @Override
    public ConfigValue getValue(ConfigSourceInterceptorContext context, String name) {
        ConfigValue value = PropertyMappers.getValue(context, name);
        
        if (value == null || value.getValue() == null) {
            return null;
        }

        if (isPersistedOnlyProperty(value)) {
            // quarkus properties values always resolved from persisted config source
            return value.withValue(PersistedConfigSource.getInstance().getValue(name));
        }

        if (value.getValue().indexOf("${") == -1) {
            return value;
        }
        
        return value.withValue(
                StringPropertyReplacer.replaceProperties(value.getValue(),
                        property -> {
                            ConfigValue prop = context.proceed(property);
                            
                            if (prop == null) {
                                return null;
                            }
                            
                            return prop.getValue();
                        }));
    }

    private boolean isPersistedOnlyProperty(ConfigValue value) {
        if (isQuarkusPropertiesEnabled && value.getName().startsWith(NS_QUARKUS)) {
            String configSourceName = value.getConfigSourceName();

            return Environment.isRuntimeMode()
                    && configSourceName != null
                    && !configSourceName.equals(PersistedConfigSource.NAME)
                    && !configSourceName.equals(AbstractRawDefaultConfigSource.NAME)
                    && !configSourceName.contains("Runtime Defaults")
                    && !configSourceName.contains("application.properties");
        }

        return false;
    }
}
