/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.keycloak.models.map.processor;

import org.keycloak.models.map.annotations.GenerateEntityImplementations;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedSourceVersion;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.Name;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeMirror;
import javax.tools.Diagnostic;
import javax.tools.Diagnostic.Kind;
import javax.tools.JavaFileObject;
import static org.keycloak.models.map.processor.FieldAccessorType.*;
import static org.keycloak.models.map.processor.Util.isSetType;
import static org.keycloak.models.map.processor.Util.methodParameters;
import java.util.Collection;
import java.util.Comparator;
import java.util.IdentityHashMap;
import java.util.Optional;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeKind;

/**
 *
 * @author hmlnarik
 */
@SupportedAnnotationTypes("org.keycloak.models.map.annotations.GenerateEntityImplementations")
@SupportedSourceVersion(SourceVersion.RELEASE_8)
public class GenerateEntityImplementationsProcessor extends AbstractGenerateEntityImplementationsProcessor {

    private Collection<String> cloners = new TreeSet<>();

    private final Generator[] generators = new Generator[] {
        new ClonerGenerator(),
        new DelegateGenerator(),
        new FieldsGenerator(),
        new ImplGenerator()
    };

    @Override
    protected void afterAnnotationProcessing() {
        if (! cloners.isEmpty()) {
            try {
                JavaFileObject file = processingEnv.getFiler().createSourceFile("org.keycloak.models.map.common.AutogeneratedCloners");
                try (PrintWriter pw = new PrintWriterNoJavaLang(file.openWriter())) {
                    pw.println("package org.keycloak.models.map.common;");

                    pw.println("import org.keycloak.models.map.common.DeepCloner.Cloner;");
                    pw.println("// DO NOT CHANGE THIS CLASS, IT IS GENERATED AUTOMATICALLY BY " + GenerateEntityImplementationsProcessor.class.getSimpleName());
                    pw.println("public final class AutogeneratedCloners {");
                    pw.println("    public static final java.util.Map<Class<?>, Cloner<?>> CLONERS_WITH_ID = new java.util.HashMap<>();");
                    pw.println("    public static final java.util.Map<Class<?>, Cloner<?>> CLONERS_WITHOUT_ID = new java.util.HashMap<>();");
                    pw.println("    static {");
                    cloners.forEach(pw::println);
                    pw.println("    }");
                    pw.println("}");
                }
            } catch (IOException ex) {
                Logger.getLogger(GenerateEntityImplementationsProcessor.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
    }

    @Override
    protected Generator[] getGenerators() {
        return this.generators;
    }

    @Override
    protected boolean testAnnotationElement(TypeElement e) {
        if (e.getKind() != ElementKind.INTERFACE) {
            processingEnv.getMessager().printMessage(Diagnostic.Kind.ERROR, "Annotation @GenerateEntityImplementations is only applicable to an interface", e);
            return false;
        }

        return true;
    }

    protected static String toEnumConstant(String key) {
        return key.replaceAll("([a-z])([A-Z])", "$1_$2").toUpperCase();
    }

    private class FieldsGenerator implements Generator {
        @Override
        public void generate(TypeElement e) throws IOException {
            Map<String, HashSet<ExecutableElement>> methodsPerAttribute = methodsPerAttributeMapping(e);
            String className = e.getQualifiedName().toString();
            String packageName = null;
            int lastDot = className.lastIndexOf('.');
            if (lastDot > 0) {
                packageName = className.substring(0, lastDot);
            }

            String simpleClassName = className.substring(lastDot + 1);
            String mapFieldsClassName = className + "Fields";
            String mapSimpleFieldsClassName = simpleClassName + "Fields";

            JavaFileObject file = processingEnv.getFiler().createSourceFile(mapFieldsClassName);
            try (PrintWriter pw = new PrintWriterNoJavaLang(file.openWriter())) {
                if (packageName != null) {
                    pw.println("package " + packageName + ";");
                }

                pw.println("public enum " + mapSimpleFieldsClassName + " {");
                methodsPerAttribute.keySet().stream()
                  .sorted(NameFirstComparator.ID_INSTANCE)
                  .map(GenerateEntityImplementationsProcessor::toEnumConstant)
                  .forEach(key -> pw.println("    " + key + ","));
                pw.println("}");
            }
        }
    }

    private class ImplGenerator implements Generator {

        @Override
        public void generate(TypeElement e) throws IOException {
            Map<String, HashSet<ExecutableElement>> methodsPerAttribute = methodsPerAttributeMapping(e);
            GenerateEntityImplementations an = e.getAnnotation(GenerateEntityImplementations.class);
            TypeElement parentTypeElement = elements.getTypeElement((an.inherits() == null || an.inherits().isEmpty()) ? "void" : an.inherits());
            if (parentTypeElement == null) {
                return;
            }
            final List<? extends Element> allMembers = elements.getAllMembers(parentTypeElement);
            String className = e.getQualifiedName().toString();
            String packageName = null;
            int lastDot = className.lastIndexOf('.');
            if (lastDot > 0) {
                packageName = className.substring(0, lastDot);
            }

            String simpleClassName = className.substring(lastDot + 1);
            String mapImplClassName = className + "Impl";
            String mapSimpleClassName = simpleClassName + "Impl";
            boolean hasId = methodsPerAttribute.containsKey("Id") || allMembers.stream().anyMatch(el -> "getId".equals(el.getSimpleName().toString()));
            boolean hasDeepClone = allMembers.stream().filter(el -> el.getKind() == ElementKind.METHOD).anyMatch(el -> "deepClone".equals(el.getSimpleName().toString()));
            boolean needsDeepClone = fieldGetters(methodsPerAttribute)
              .map(ExecutableElement::getReturnType)
              .anyMatch(fieldType -> ! isKnownCollectionOfImmutableFinalTypes(fieldType) && ! isImmutableFinalType(fieldType));

            JavaFileObject file = processingEnv.getFiler().createSourceFile(mapImplClassName);
            try (PrintWriter pw = new PrintWriterNoJavaLang(file.openWriter())) {
                if (packageName != null) {
                    pw.println("package " + packageName + ";");
                }

                pw.println("import java.util.Objects;");
                pw.println("import " + FQN_DEEP_CLONER + ";");
                pw.println("// DO NOT CHANGE THIS CLASS, IT IS GENERATED AUTOMATICALLY BY " + GenerateEntityImplementationsProcessor.class.getSimpleName());
                pw.println("public class " + mapSimpleClassName + (an.inherits().isEmpty() ? "" : " extends " + an.inherits()) + " implements " + className + " {");

                // Constructors
                allMembers.stream()
                  .filter(ExecutableElement.class::isInstance)
                  .map(ExecutableElement.class::cast)
                  .filter((ExecutableElement ee) -> ee.getKind() == ElementKind.CONSTRUCTOR)
                  .forEach((ExecutableElement ee) -> {
                      if (hasDeepClone || ! needsDeepClone) {
                          pw.println("    "
                            + ee.getModifiers().stream().map(Object::toString).collect(Collectors.joining(" "))
                            + " " + mapSimpleClassName + "(" + methodParameters(ee.getParameters()) + ") { super(" + ee.getParameters() + "); }"
                          );
                      } else if (needsDeepClone) {
                          pw.println("    /**");
                          pw.println("     * @deprecated This constructor uses a {@link DeepCloner#DUMB_CLONER} that does not clone anything. Use {@link #" + mapSimpleClassName + "(DeepCloner)} variant instead");
                          pw.println("     */");
                          pw.println("    "
                            + ee.getModifiers().stream().map(Object::toString).collect(Collectors.joining(" "))
                            + " "
                            + mapSimpleClassName + "(" + methodParameters(ee.getParameters()) + ") { this(DeepCloner.DUMB_CLONER" + (ee.getParameters().isEmpty() ? "" : ", ") + ee.getParameters() + "); }"
                          );
                          pw.println("    "
                            + ee.getModifiers().stream().map(Object::toString).collect(Collectors.joining(" "))
                            + " "
                            + mapSimpleClassName + "(DeepCloner cloner" + (ee.getParameters().isEmpty() ? "" : ", ") + methodParameters(ee.getParameters()) + ") { super(" + ee.getParameters() + "); this.cloner = cloner; }"
                          );
                      }
                  });

                // equals, hashCode, toString
                pw.println("    @Override public boolean equals(Object o) {");
                pw.println("        if (o == this) return true; ");
                pw.println("        if (! (o instanceof " + mapSimpleClassName + ")) return false; ");
                pw.println("        " + mapSimpleClassName + " other = (" + mapSimpleClassName + ") o; ");
                pw.println("        return "
                  + fieldGetters(methodsPerAttribute)
                    .map(ExecutableElement::getSimpleName)
                    .map(Name::toString)
                    .sorted(NameFirstComparator.GET_ID_INSTANCE)
                    .map(v -> "Objects.equals(" + v + "(), other." + v + "())")
                    .collect(Collectors.joining("\n          && "))
                  + ";");
                pw.println("    }");
                pw.println("    @Override public int hashCode() {");
                pw.println("        return " 
                  + (hasId
                    ? "(getId() == null ? super.hashCode() : getId().hashCode())"
                    : "Objects.hash("
                      + fieldGetters(methodsPerAttribute)
                        .filter(ee -> isImmutableFinalType(ee.getReturnType()))
                        .map(ExecutableElement::getSimpleName)
                        .map(Name::toString)
                        .sorted(NameFirstComparator.GET_ID_INSTANCE)
                        .map(v -> v + "()")
                        .collect(Collectors.joining(",\n          "))
                      + ")")
                  + ";");
                pw.println("    }");
                pw.println("    @Override public String toString() {");
                pw.println("        return String.format(\"%s@%08x\", " + (hasId ? "getId()" : "\"" + mapSimpleClassName + "\"" ) + ", System.identityHashCode(this));");
                pw.println("    }");

                // deepClone
                if (! hasDeepClone && needsDeepClone) {
                    pw.println("    private final DeepCloner cloner;");
                    pw.println("    public <V> V deepClone(V obj) {");
                    pw.println("        return cloner.from(obj);");
                    pw.println("    }");
                }

                // fields, getters, setters
                methodsPerAttribute.entrySet().stream().sorted(Comparator.comparing(Map.Entry::getKey, NameFirstComparator.ID_INSTANCE)).forEach(me -> {
                    HashSet<ExecutableElement> methods = me.getValue();
                    TypeMirror fieldType = determineFieldType(me.getKey(), methods);
                    if (fieldType == null) {
                        return;
                    }

                    pw.println("");
                    pw.println("    private " + fieldType + " f" + me.getKey() + ";");

                    for (ExecutableElement method : methods) {
                        FieldAccessorType fat = FieldAccessorType.determineType(method, me.getKey(), types, fieldType);
                        Optional<ExecutableElement> parentMethod = allMembers.stream()
                          .filter(ExecutableElement.class::isInstance)
                          .map(ExecutableElement.class::cast)
                          .filter(ee -> Objects.equals(ee.toString(), method.toString()))
                          .filter((ExecutableElement ee) ->  ! ee.getModifiers().contains(Modifier.ABSTRACT))
                          .findAny();

                        if (parentMethod.isPresent()) {
                            processingEnv.getMessager().printMessage(Kind.OTHER, "Method " + method + " is declared in a parent class.", method);
                        } else if (fat != FieldAccessorType.UNKNOWN && ! printMethodBody(pw, fat, method, "f" + me.getKey(), fieldType)) {
                            processingEnv.getMessager().printMessage(Kind.WARNING, "Could not determine desired semantics of method from its signature", method);
                        }
                    }
                });
                pw.println("}");
            }
        }

        private boolean printMethodBody(PrintWriter pw, FieldAccessorType accessorType, ExecutableElement method, String fieldName, TypeMirror fieldType) {
            TypeMirror firstParameterType = method.getParameters().isEmpty()
              ? types.getNullType()
              : method.getParameters().get(0).asType();
            TypeElement typeElement = elements.getTypeElement(types.erasure(fieldType).toString());

            switch (accessorType) {
                case GETTER:
                    pw.println("    @SuppressWarnings(\"unchecked\") @Override public " + method.getReturnType() + " " + method + " {");
                    pw.println("        return " + fieldName + ";");
                    pw.println("    }");
                    return true;
                case SETTER:
                    pw.println("    @SuppressWarnings(\"unchecked\") @Override public " + method.getReturnType() + " " + method.getSimpleName() + "(" + firstParameterType + " p0) {");
                    if (! isImmutableFinalType(fieldType)) {
                        pw.println("        p0 = " + deepClone(fieldType, "p0") + ";");
                    }
                    pw.println("        updated |= ! Objects.equals(" + fieldName + ", p0);");
                    pw.println("        " + fieldName + " = p0;");
                    pw.println("    }");
                    return true;
                case COLLECTION_ADD:
                    pw.println("    @SuppressWarnings(\"unchecked\") @Override public " + method.getReturnType() + " " + method.getSimpleName() + "(" + firstParameterType + " p0) {");
                    pw.println("        if (" + fieldName + " == null) { " + fieldName + " = " + interfaceToImplementation(typeElement, "") + "; }");
                    if (! isImmutableFinalType(firstParameterType)) {
                        pw.println("        p0 = " + deepClone(fieldType, "p0") + ";");
                    }
                    if (isSetType(typeElement)) {
                        pw.println("        updated |= " + fieldName + ".add(p0);");
                    } else {
                        pw.println("        " + fieldName + ".add(p0);");
                        pw.println("        updated = true;");
                    }
                    pw.println("    }");
                    return true;
                case COLLECTION_DELETE:
                    pw.println("    @SuppressWarnings(\"unchecked\") @Override public " + method.getReturnType() + " " + method.getSimpleName() + "(" + firstParameterType + " p0) {");
                    pw.println("        if (" + fieldName + " == null) { return; }");
                    pw.println("        boolean removed = " + fieldName + ".remove(p0)" + ("java.util.Map".equals(typeElement.getQualifiedName().toString()) ? " != null" : "") + ";");
                    pw.println("        updated |= removed;");
                    pw.println("    }");
                    return true;
                case MAP_ADD:
                    TypeMirror secondParameterType = method.getParameters().get(1).asType();
                    pw.println("    @SuppressWarnings(\"unchecked\") @Override public " + method.getReturnType() + " " + method.getSimpleName() + "(" + firstParameterType + " p0, " + secondParameterType + " p1) {");
                    pw.println("        if (" + fieldName + " == null) { " + fieldName + " = " + interfaceToImplementation(typeElement, "") + "; }");
                    if (! isImmutableFinalType(secondParameterType)) {
                        pw.println("        p1 = " + deepClone(secondParameterType, "p1") + ";");
                    }
                    pw.println("        Object v = " + fieldName + ".put(p0, p1);");
                    pw.println("        updated |= ! Objects.equals(v, p1);");
                    pw.println("    }");
                    return true;
                case MAP_GET:
                    pw.println("    @SuppressWarnings(\"unchecked\") @Override public " + method.getReturnType() + " " + method.getSimpleName() + "(" + firstParameterType + " p0) {");
                    pw.println("        return " + fieldName + " == null ? null : " + fieldName + ".get(p0);");
                    pw.println("    }");
                    return true;
            }

            return false;
        }
    }

    private class DelegateGenerator implements Generator {
        @Override
        public void generate(TypeElement e) throws IOException {
            Map<String, HashSet<ExecutableElement>> methodsPerAttribute = methodsPerAttributeMapping(e);
            String className = e.getQualifiedName().toString();
            String packageName = null;
            int lastDot = className.lastIndexOf('.');
            if (lastDot > 0) {
                packageName = className.substring(0, lastDot);
            }

            String simpleClassName = className.substring(lastDot + 1);
            String mapClassName = className + "Delegate";
            String mapSimpleClassName = simpleClassName + "Delegate";
            String fieldsClassName = className + "Fields";

            GenerateEntityImplementations an = e.getAnnotation(GenerateEntityImplementations.class);
            TypeElement parentTypeElement = elements.getTypeElement((an.inherits() == null || an.inherits().isEmpty()) ? "void" : an.inherits());
            if (parentTypeElement == null) {
                return;
            }
            final List<? extends Element> allMembers = elements.getAllMembers(e);

            JavaFileObject file = processingEnv.getFiler().createSourceFile(mapClassName);
            IdentityHashMap<ExecutableElement, String> m2field = new IdentityHashMap<>();
            methodsPerAttribute.forEach((f, s) -> s.forEach(m -> m2field.put(m, f)));   // Create reverse map
            try (PrintWriter pw = new PrintWriterNoJavaLang(file.openWriter())) {
                if (packageName != null) {
                    pw.println("package " + packageName + ";");
                }

                pw.println("public class " + mapSimpleClassName + " implements " + className + " {");
                pw.println("    private final org.keycloak.models.map.common.delegate.DelegateProvider<" + className + "> delegateProvider;");
                pw.println("    public " + mapSimpleClassName + "(org.keycloak.models.map.common.delegate.DelegateProvider<" + className + "> delegateProvider) {");
                pw.println("        this.delegateProvider = delegateProvider;");
                pw.println("    }");

                allMembers.stream()
                  .filter(m -> m.getKind() == ElementKind.METHOD)
                  .filter(ExecutableElement.class::isInstance)
                  .map(ExecutableElement.class::cast)
                  .filter(ee -> ee.getModifiers().contains(Modifier.ABSTRACT))
                  .forEach(ee -> {
                      pw.println("    @Override "
                        + ee.getModifiers().stream().filter(m -> m != Modifier.ABSTRACT).map(Object::toString).collect(Collectors.joining(" "))
                        + " " + ee.getReturnType()
                        + " " + ee.getSimpleName()
                        + "(" + methodParameters(ee.getParameters()) + ") {");
                      String field = m2field.get(ee);
                      field = field == null ? "null" : fieldsClassName + "." + toEnumConstant(field);
                      if (ee.getReturnType().getKind() == TypeKind.BOOLEAN && "isUpdated".equals(ee.getSimpleName().toString())) {
                          pw.println("        return delegateProvider.isUpdated();");
                      } else if (ee.getReturnType().getKind() == TypeKind.VOID) {  // write operation
                          pw.println("        delegateProvider.getDelegate(false, " + field + ")." + ee.getSimpleName() + "("
                            + ee.getParameters().stream().map(VariableElement::getSimpleName).collect(Collectors.joining(", "))
                            + ");");
                      } else {
                          pw.println("        return delegateProvider.getDelegate(true, " + field + ")." + ee.getSimpleName() + "("
                            + ee.getParameters().stream().map(VariableElement::getSimpleName).collect(Collectors.joining(", "))
                            + ");");
                      }
                      pw.println("    }");
                  });

                pw.println("}");
            }
        }
    }

    private class ClonerGenerator implements Generator {

        @Override
        public void generate(TypeElement e) throws IOException {
            Map<String, HashSet<ExecutableElement>> methodsPerAttribute = methodsPerAttributeMapping(e);
            String className = e.getQualifiedName().toString();
            String packageName = null;
            int lastDot = className.lastIndexOf('.');
            if (lastDot > 0) {
                packageName = className.substring(0, lastDot);
            }

            String simpleClassName = className.substring(lastDot + 1);
            String clonerImplClassName = className + "Cloner";
            String clonerSimpleClassName = simpleClassName + "Cloner";

            JavaFileObject enumFile = processingEnv.getFiler().createSourceFile(clonerImplClassName);
            try (PrintWriter pw = new PrintWriter(enumFile.openWriter()) {
                @Override
                public void println(String x) {
                    super.println(x == null ? x : x.replaceAll("java.lang.", ""));
                }
            }) {
                if (packageName != null) {
                    pw.println("package " + packageName + ";");
                }
                pw.println("import " + FQN_DEEP_CLONER + ";");
                pw.println("// DO NOT CHANGE THIS CLASS, IT IS GENERATED AUTOMATICALLY BY " + GenerateEntityImplementationsProcessor.class.getSimpleName());
                pw.println("public class " + clonerSimpleClassName + " {");
                pw.println("    public static " + className + " deepClone(" + className + " original, " + className + " target) {");

                methodsPerAttribute.entrySet().stream().sorted(Comparator.comparing(Map.Entry::getKey)).forEach(me -> {
                    final String fieldName = me.getKey();
                    HashSet<ExecutableElement> methods = me.getValue();
                    TypeMirror fieldType = determineFieldType(fieldName, methods);
                    if (fieldType == null) {
                        return;
                    }

                    cloneField(e, fieldName, methods, fieldType, pw);
                });
                pw.println("        target.clearUpdatedFlag();");
                pw.println("        return target;");
                pw.println("    }");

                cloners.add("        CLONERS_WITH_ID.put(" + className + ".class, (Cloner<" + className + ">) " + clonerImplClassName + "::deepClone);");

                if (methodsPerAttribute.containsKey("Id")) {
                    pw.println("    public static " + className + " deepCloneNoId(" + className + " original, " + className + " target) {");

                    methodsPerAttribute.entrySet().stream().sorted(Comparator.comparing(Map.Entry::getKey)).forEach(me -> {
                        final String fieldName = me.getKey();
                        HashSet<ExecutableElement> methods = me.getValue();
                        TypeMirror fieldType = determineFieldType(fieldName, methods);
                        if (fieldType == null || "Id".equals(fieldName)) {
                            return;
                        }

                        cloneField(e, fieldName, methods, fieldType, pw);
                    });
                    pw.println("        target.clearUpdatedFlag();");
                    pw.println("        return target;");
                    pw.println("    }");

                    cloners.add("        CLONERS_WITHOUT_ID.put(" + className + ".class, (Cloner<" + className + ">) " + clonerImplClassName + "::deepCloneNoId);");
                }
                pw.println("}");
            }
        }

        private void cloneField(TypeElement e, final String fieldName, HashSet<ExecutableElement> methods, TypeMirror fieldType, final PrintWriter pw) {
            ExecutableElement getter = FieldAccessorType.getMethod(GETTER, methods, fieldName, types, fieldType).orElse(null);
            if (getter == null) {
                processingEnv.getMessager().printMessage(Kind.WARNING, "Could not determine getter for " + fieldName + " property");
                return;
            }

            Optional<ExecutableElement> setter = FieldAccessorType.getMethod(SETTER, methods, fieldName, types, fieldType);
            Optional<ExecutableElement> addToCollection = FieldAccessorType.getMethod(COLLECTION_ADD, methods, fieldName, types, fieldType);
            Optional<ExecutableElement> updateMap = FieldAccessorType.getMethod(MAP_ADD, methods, fieldName, types, fieldType);

            if (setter.isPresent()) {
                final Name setterName = setter.get().getSimpleName();
                // Setter always deep-clones whatever comes from the original, so we don't clone the value here.
                pw.println("        target." + setterName + "(original."  + getter.getSimpleName() + "());");
            } else if (addToCollection.isPresent()) {
                pw.println("        if (original." + getter.getSimpleName() + "() != null) {");
                pw.println("            original." + getter.getSimpleName() + "().forEach(target::" + addToCollection.get().getSimpleName() + ");");
                pw.println("        }");
            } else if (updateMap.isPresent()) {
                pw.println("        if (original." + getter.getSimpleName() + "() != null) {");
                pw.println("            original." + getter.getSimpleName() + "().forEach(target::" + updateMap.get().getSimpleName() + ");");
                pw.println("        }");
            } else {
                processingEnv.getMessager().printMessage(Kind.ERROR, "Could not determine way to clone " + fieldName + " property", e);
            }
        }
    }
}
