// Copyright 2025 Ant Investor Ltd.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package userapi

import (
	"context"
	"time"

	devicev1 "github.com/antinvestor/apis/go/device/v1"
	profilev1 "github.com/antinvestor/apis/go/profile/v1"
	"github.com/antinvestor/gomatrixserverlib/spec"
	fedsenderapi "github.com/antinvestor/matrix/federationapi/api"
	"github.com/antinvestor/matrix/federationapi/statistics"
	"github.com/antinvestor/matrix/internal/actorutil"
	"github.com/antinvestor/matrix/internal/pushgateway"
	"github.com/antinvestor/matrix/internal/queueutil"
	"github.com/antinvestor/matrix/internal/sqlutil"
	rsapi "github.com/antinvestor/matrix/roomserver/api"
	"github.com/antinvestor/matrix/setup/config"
	"github.com/antinvestor/matrix/userapi/api"
	"github.com/antinvestor/matrix/userapi/consumers"
	"github.com/antinvestor/matrix/userapi/internal"
	"github.com/antinvestor/matrix/userapi/producers"
	"github.com/antinvestor/matrix/userapi/storage"
	userapiutil "github.com/antinvestor/matrix/userapi/util"
	"github.com/pitabwire/util"
)

// NewInternalAPI returns a concrete implementation of the internal API. Callers
// can call functions directly on the returned API or via an HTTP interface using AddInternalRoutes.
//
// Creating a new instance of the user API requires a roomserver API with a federation API set
// using its `SetFederationAPI` method, other you may get nil-dereference errors.
func NewInternalAPI(ctx context.Context, cfg *config.Matrix, cm sqlutil.ConnectionManager, qm queueutil.QueueManager, am actorutil.ActorManager, rsAPI rsapi.UserRoomserverAPI, fedClient fedsenderapi.KeyserverFederationAPI, profileCli *profilev1.ProfileClient, deviceCli *devicev1.DeviceClient, enableMetrics bool, blacklistedOrBackingOffFn func(ctx context.Context, s spec.ServerName) (*statistics.ServerStatistics, error)) *internal.UserInternalAPI {

	var err error

	cfgUsrApi := cfg.UserAPI
	cfgSyncApi := cfg.SyncAPI
	cfgKeySrv := cfg.KeyServer
	appServices := cfg.Derived.ApplicationServices

	pgClient := pushgateway.NewHTTPClient(cfgUsrApi.PushGatewayDisableTLSValidation)

	db, err := storage.NewUserDatabase(
		ctx,
		profileCli,
		deviceCli,
		cm,
		cfg.Global.ServerName,
		cfg.UserAPI.BCryptCost,
		cfg.UserAPI.OpenIDTokenLifetimeMS,
		api.DefaultLoginTokenLifetime,
		cfg.UserAPI.Global.ServerNotices.LocalPart,
	)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to connect to accounts db")
	}

	keyDB, err := storage.NewKeyDatabase(ctx, cm)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to connect to key db")
	}

	err = qm.RegisterPublisher(ctx, &cfgSyncApi.Queues.OutputClientData)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to register publisher for client data")
	}

	err = qm.RegisterPublisher(ctx, &cfgSyncApi.Queues.OutputNotificationData)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to register publisher for notification data")
	}

	syncProducer, err := producers.NewSyncAPI(
		ctx, &cfg.SyncAPI,
		db, qm,
	)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to obtain sync publisher")
	}

	err = qm.RegisterPublisher(ctx, &cfgKeySrv.Queues.OutputKeyChangeEvent)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to register publisher for key change events")
	}

	keyChangeProducer := &producers.KeyChange{
		Topic: &cfgKeySrv.Queues.OutputKeyChangeEvent,
		Qm:    qm,
		DB:    keyDB,
	}

	userAPI := &internal.UserInternalAPI{
		DB:                   db,
		KeyDatabase:          keyDB,
		SyncProducer:         syncProducer,
		KeyChangeProducer:    keyChangeProducer,
		Config:               &cfg.UserAPI,
		AppServices:          appServices,
		RSAPI:                rsAPI,
		DisableTLSValidation: cfg.UserAPI.PushGatewayDisableTLSValidation,
		PgClient:             pgClient,
		FedClient:            fedClient,
	}

	updater := internal.NewDeviceListUpdater(ctx, keyDB, userAPI, keyChangeProducer, fedClient, cfg.UserAPI.WorkerCount, rsAPI, cfg.Global.ServerName, enableMetrics, blacklistedOrBackingOffFn)
	userAPI.Updater = updater
	// Remove users which we don't share a room with anymore
	if err = updater.CleanUp(ctx); err != nil {
		util.Log(ctx).WithError(err).Error("failed to cleanup stale device lists")
	}

	go func() {
		if err = updater.Start(ctx); err != nil {
			util.Log(ctx).WithError(err).Panic("failed to start device list updater")
		}
	}()

	err = consumers.NewDeviceListUpdateConsumer(
		ctx, &cfg.UserAPI, qm, updater,
	)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to start device list consumer")
	}

	err = consumers.NewSigningKeyUpdateConsumer(
		ctx, &cfg.UserAPI, qm, userAPI,
	)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to start signing key consumer")
	}

	err = consumers.NewOutputReceiptEventConsumer(
		ctx, &cfg.UserAPI, qm, db, syncProducer, pgClient,
	)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to start user API receipt consumer")
	}

	err = consumers.NewOutputRoomEventConsumer(
		ctx, &cfg.UserAPI, qm, am, db, pgClient, rsAPI, syncProducer,
	)
	if err != nil {
		util.Log(ctx).WithError(err).Panic("failed to start user API streamed event consumer")
	}

	var cleanOldNotifs func()
	cleanOldNotifs = func() {
		select {
		case <-ctx.Done():
			return
		default:
			// Context is still valid, continue with operation

			util.Log(ctx).Info("Cleaning old notifications")
			if err = db.DeleteOldNotifications(ctx); err != nil {
				util.Log(ctx).WithError(err).Error("Failed to clean old notifications")
			}
			time.AfterFunc(time.Hour, cleanOldNotifs)
		}
	}
	time.AfterFunc(time.Minute, cleanOldNotifs)

	if cfg.Global.ReportStats.Enabled {
		go userapiutil.StartPhoneHomeCollector(ctx, time.Now(), cfg, db)
	}

	return userAPI
}
