// Copyright 2025 Ant Investor Ltd.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package setup

import (
	"context"

	"buf.build/gen/go/antinvestor/presence/connectrpc/go/presencev1connect"
	partitionv1 "github.com/antinvestor/apis/go/partition/v1"
	profilev1 "github.com/antinvestor/apis/go/profile/v1"
	"github.com/antinvestor/gomatrixserverlib"
	"github.com/antinvestor/gomatrixserverlib/fclient"
	appserviceAPI "github.com/antinvestor/matrix/appservice/api"
	"github.com/antinvestor/matrix/clientapi"
	"github.com/antinvestor/matrix/clientapi/api"
	"github.com/antinvestor/matrix/federationapi"
	federationAPI "github.com/antinvestor/matrix/federationapi/api"
	"github.com/antinvestor/matrix/internal/actorutil"
	"github.com/antinvestor/matrix/internal/cacheutil"
	"github.com/antinvestor/matrix/internal/httputil"
	"github.com/antinvestor/matrix/internal/queueutil"
	"github.com/antinvestor/matrix/internal/sqlutil"
	"github.com/antinvestor/matrix/internal/transactions"
	"github.com/antinvestor/matrix/mediaapi"
	"github.com/antinvestor/matrix/relayapi"
	relayAPI "github.com/antinvestor/matrix/relayapi/api"
	roomserverAPI "github.com/antinvestor/matrix/roomserver/api"
	"github.com/antinvestor/matrix/setup/config"
	"github.com/antinvestor/matrix/syncapi"
	userAPI "github.com/antinvestor/matrix/userapi/api"
	"github.com/pitabwire/frame"
)

// Monolith represents an instantiation of all dependencies required to build
// all components of Matrix, for use in monolith mode.
type Monolith struct {
	Config    *config.Matrix
	Service   *frame.Service
	KeyRing   *gomatrixserverlib.KeyRing
	Client    *fclient.Client
	FedClient fclient.FederationClient

	AppserviceAPI appserviceAPI.AppServiceInternalAPI
	FederationAPI federationAPI.FederationInternalAPI
	RoomserverAPI roomserverAPI.RoomserverInternalAPI
	UserAPI       userAPI.UserInternalAPI
	RelayAPI      relayAPI.RelayInternalAPI

	PartitionCli *partitionv1.PartitionClient
	ProfileCli   *profilev1.ProfileClient

	PresenceCli presencev1connect.PresenceServiceClient
	// Optional
	ExtPublicRoomsProvider   api.ExtraPublicRoomsProvider
	ExtUserDirectoryProvider userAPI.QuerySearchProfilesAPI
}

// AddAllPublicRoutes attaches all public paths to the given router
func (m *Monolith) AddAllPublicRoutes(
	ctx context.Context,
	cfg *config.Matrix,
	routers httputil.Routers,
	cm sqlutil.ConnectionManager,
	qm queueutil.QueueManager,
	caches *cacheutil.Caches,
	am actorutil.ActorManager,
	enableMetrics bool,
) {
	userDirectoryProvider := m.ExtUserDirectoryProvider
	if userDirectoryProvider == nil {
		userDirectoryProvider = m.UserAPI
	}
	clientapi.AddPublicRoutes(
		ctx, routers, cfg, qm, m.FedClient, m.RoomserverAPI, m.AppserviceAPI, transactions.New(),
		m.FederationAPI, m.UserAPI, userDirectoryProvider,
		m.ExtPublicRoomsProvider, m.PartitionCli, m.PresenceCli, enableMetrics,
	)
	federationapi.AddPublicRoutes(
		ctx, routers, cfg, qm, m.UserAPI, m.FedClient, m.KeyRing, m.RoomserverAPI, m.FederationAPI, enableMetrics,
	)
	mediaapi.AddPublicRoutes(ctx, routers, cm, cfg, m.UserAPI, m.Client, m.FedClient, m.KeyRing)
	syncapi.AddPublicRoutes(ctx, routers, cfg, cm, qm, am, m.UserAPI, m.RoomserverAPI, caches, enableMetrics)

	if m.RelayAPI != nil {
		relayapi.AddPublicRoutes(ctx, routers, cfg, m.KeyRing, m.RelayAPI)
	}
}
