require File.expand_path('../../../spec_helper', __FILE__)
require File.expand_path('../fixtures/classes', __FILE__)
require File.expand_path('../shared/iteration', __FILE__)

describe "Hash#reject" do
  it "returns a new hash removing keys for which the block yields true" do
    h = new_hash(1=>false, 2=>true, 3=>false, 4=>true)
    h.reject { |k,v| v }.keys.sort.should == [1,3]
  end

  it "is equivalent to hsh.dup.delete_if" do
    h = new_hash(:a => 'a', :b => 'b', :c => 'd')
    h.reject { |k,v| k == 'd' }.should == (h.dup.delete_if { |k, v| k == 'd' })

    all_args_reject = []
    all_args_delete_if = []
    h = new_hash(1 => 2, 3 => 4)
    h.reject { |*args| all_args_reject << args }
    h.delete_if { |*args| all_args_delete_if << args }
    all_args_reject.should == all_args_delete_if

    h = new_hash(1 => 2)
    # dup doesn't copy singleton methods
    def h.to_a() end
    h.reject { false }.to_a.should == [[1, 2]]
  end

  it "returns subclass instance for subclasses" do
    HashSpecs::MyHash[1 => 2, 3 => 4].reject { false }.should be_kind_of(HashSpecs::MyHash)
    HashSpecs::MyHash[1 => 2, 3 => 4].reject { true }.should be_kind_of(HashSpecs::MyHash)
  end

  it "taints the resulting hash" do
    h = new_hash(:a => 1).taint
    h.reject {false}.tainted?.should == true
  end

  it "processes entries with the same order as reject!" do
    h = new_hash(:a => 1, :b => 2, :c => 3, :d => 4)

    reject_pairs = []
    reject_bang_pairs = []
    h.dup.reject { |*pair| reject_pairs << pair }
    h.reject! { |*pair| reject_bang_pairs << pair }

    reject_pairs.should == reject_bang_pairs
  end

  it_behaves_like(:hash_iteration_no_block, :reject)
end

describe "Hash#reject!" do
  before(:each) do
    @hsh = new_hash
    (1 .. 10).each { |k| @hsh[k] = k.even? }
    @empty = new_hash
  end

  it "removes keys from self for which the block yields true" do
    @hsh.reject! { |k,v| v }
    @hsh.keys.sort.should == [1,3,5,7,9]
  end

  it "is equivalent to delete_if if changes are made" do
    @hsh.reject! { |k,v| v }.should == @hsh.delete_if { |k, v| v }

    h = new_hash(1 => 2, 3 => 4)
    all_args_reject = []
    all_args_delete_if = []
    h.dup.reject! { |*args| all_args_reject << args }
    h.dup.delete_if { |*args| all_args_delete_if << args }
    all_args_reject.should == all_args_delete_if
  end

  it "returns nil if no changes were made" do
    new_hash(:a => 1).reject! { |k,v| v > 1 }.should == nil
  end

  it "processes entries with the same order as delete_if" do
    h = new_hash(:a => 1, :b => 2, :c => 3, :d => 4)

    reject_bang_pairs = []
    delete_if_pairs = []
    h.dup.reject! { |*pair| reject_bang_pairs << pair }
    h.dup.delete_if { |*pair| delete_if_pairs << pair }

    reject_bang_pairs.should == delete_if_pairs
  end

  ruby_version_is ""..."1.9" do
    it "raises a TypeError if called on a frozen instance" do
      lambda { HashSpecs.frozen_hash.reject! { false } }.should raise_error(TypeError)
      lambda { HashSpecs.empty_frozen_hash.reject! { true } }.should raise_error(TypeError)
    end
  end

  ruby_version_is "1.9" do
    it "raises a RuntimeError if called on a frozen instance that is modified" do
      lambda { HashSpecs.empty_frozen_hash.reject! { true } }.should raise_error(RuntimeError)
    end

    it "raises a RuntimeError if called on a frozen instance that would not be modified" do
      lambda { HashSpecs.frozen_hash.reject! { false } }.should raise_error(RuntimeError)
    end
  end

  ruby_version_is "" ... "1.8.7" do
    it "raises a LocalJumpError when called on a non-empty hash without a block" do
      lambda { @hsh.reject! }.should raise_error(LocalJumpError)
    end

    it "does not raise a LocalJumpError when called on an empty hash without a block" do
      @empty.reject!.should == nil
    end
  end

  ruby_version_is "1.8.7" do
    it "returns an Enumerator when called on a non-empty hash without a block" do
      @hsh.reject!.should be_an_instance_of(enumerator_class)
    end

    it "returns an Enumerator when called on an empty hash without a block" do
      @empty.reject!.should be_an_instance_of(enumerator_class)
    end
  end

end
