use std::sync::Arc;

use fuzz::{
    actions::{
        ActionWrapper::{self, *},
        GenericAction,
    },
    container::{TreeAction, TreeActionInner},
    crdt_fuzzer::{test_multi_sites, Action::*, FuzzTarget, FuzzValue::*},
};
use loro::{ContainerType::*, LoroCounter, LoroDoc};

#[ctor::ctor]
fn init() {
    dev_utils::setup_test_log();
}

#[test]
fn test_movable_list_0() {
    test_multi_sites(
        2,
        vec![FuzzTarget::All],
        &mut [
            Handle {
                site: 117,
                target: 166,
                container: 10,
                action: Generic(GenericAction {
                    value: I32(-273622840),
                    bool: false,
                    key: 2741083633,
                    pos: 6666897757659758022,
                    length: 8533446734363315434,
                    prop: 12864568433311511070,
                }),
            },
            Handle {
                site: 124,
                target: 14,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_1() {
    test_multi_sites(
        2,
        vec![FuzzTarget::All],
        &mut [
            Handle {
                site: 164,
                target: 239,
                container: 61,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 1768571449,
                    pos: 5035915790398973222,
                    length: 12262157610559101276,
                    prop: 2115599344051559167,
                }),
            },
            Sync { from: 172, to: 249 },
            Handle {
                site: 76,
                target: 185,
                container: 213,
                action: Generic(GenericAction {
                    value: I32(-180416322),
                    bool: false,
                    key: 905065406,
                    pos: 13106072747215825198,
                    length: 14041671030581285265,
                    prop: 15938081911894848481,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_2() {
    test_multi_sites(
        2,
        vec![FuzzTarget::All],
        &mut [
            Handle {
                site: 44,
                target: 124,
                container: 221,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3351758791,
                    pos: 288230650086410183,
                    length: 2606365581092837153,
                    prop: 15553136935972341051,
                }),
            },
            SyncAll,
            Handle {
                site: 109,
                target: 209,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(1145324612),
                    bool: true,
                    key: 3351758806,
                    pos: 9187202260886079431,
                    length: 72056541770940543,
                    prop: 70127282814975,
                }),
            },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_3() {
    test_multi_sites(
        2,
        vec![FuzzTarget::All],
        &mut [
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: false,
                    key: 4294967295,
                    pos: 18446744073709551419,
                    length: 4268071042561343487,
                    prop: 4268070197446523698,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3318072622,
                    pos: 14251014049101104581,
                    length: 18391358628880399163,
                    prop: 4268070197442641920,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(1568286093),
                    bool: true,
                    key: 999132557,
                    pos: 216172782113783807,
                    length: 15626148457674914619,
                    prop: 18446693297831399889,
                }),
            },
            SyncAll,
            Handle {
                site: 92,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(1162167621),
                    bool: true,
                    key: 993737541,
                    pos: 15163,
                    length: 18391358628880386048,
                    prop: 1099511627774,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_4() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            SyncAll,
            Handle {
                site: 91,
                target: 59,
                container: 34,
                action: Generic(GenericAction {
                    value: I32(-2088551680),
                    bool: true,
                    key: 131,
                    pos: 16855269067351588864,
                    length: 6911312294037809641,
                    prop: 16855260268008005471,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 160,
                target: 19,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 930317187,
                    pos: 4251419660595589899,
                    length: 10993036654195,
                    prop: 18446743523953737728,
                }),
            },
            SyncAll,
            Checkout {
                site: 79,
                to: 2147483432,
            },
            Handle {
                site: 34,
                target: 34,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(572662306),
                    bool: false,
                    key: 829760512,
                    pos: 4319796467578386228,
                    length: 18446744073709551615,
                    prop: 2676586395008836901,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 37,
                action: Generic(GenericAction {
                    value: I32(623191333),
                    bool: true,
                    key: 623191333,
                    pos: 2676586395008836901,
                    length: 2676586395008836901,
                    prop: 10455415605503269,
                }),
            },
        ],
    )
}

#[test]
fn missing_event_when_checkout() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Map, FuzzTarget::Tree],
        &mut [
            Handle {
                site: 39,
                target: 39,
                container: 39,
                action: Generic(GenericAction {
                    value: I32(656877351),
                    bool: true,
                    key: 656877351,
                    pos: 2821223700817717031,
                    length: 2821266740684990247,
                    prop: 2821266740684990247,
                }),
            },
            Handle {
                site: 39,
                target: 39,
                container: 39,
                action: Generic(GenericAction {
                    value: I32(656877351),
                    bool: true,
                    key: 656877351,
                    pos: 2821266740684990247,
                    length: 2821266740684990247,
                    prop: 2821266740684990247,
                }),
            },
            Handle {
                site: 39,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            Handle {
                site: 39,
                target: 39,
                container: 39,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 1811993856,
                    pos: 1585267068834414592,
                    length: 18389323175239352127,
                    prop: 2745369343,
                }),
            },
        ],
    )
}

#[test]
fn tree_meta() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Map, FuzzTarget::Tree],
        &mut [
            Handle {
                site: 200,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 320017171,
                    pos: 1374463283923456787,
                    length: 1374472080016478995,
                    prop: 1374463283923456787,
                }),
            },
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(320017171),
                    bool: true,
                    key: 320017171,
                    pos: 1374463309693260563,
                    length: 1374463283923456787,
                    prop: 57140735609148179,
                }),
            },
            Sync { from: 171, to: 139 },
            Handle {
                site: 171,
                target: 171,
                container: 39,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: false,
                    key: 2868955905,
                    pos: 1374463283923456787,
                    length: 1374722768667623699,
                    prop: 18446743056122319677,
                }),
            },
            Sync { from: 131, to: 131 },
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(320017200),
                    bool: false,
                    key: 320541459,
                    pos: 1374463283923456787,
                    length: 1374463283923456787,
                    prop: 1374463283923456787,
                }),
            },
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(-2088533229),
                    bool: true,
                    key: 2206434179,
                    pos: 9476562641788044153,
                    length: 9476562641653826435,
                    prop: 9511602412998329219,
                }),
            },
            Sync { from: 131, to: 131 },
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(320028947),
                    bool: true,
                    key: 320017171,
                    pos: 18446744073709490963,
                    length: 18446744073709551615,
                    prop: 1374463283923477011,
                }),
            },
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(320017171),
                    bool: true,
                    key: 320017171,
                    pos: 1374463284779094803,
                    length: 1374463283923456787,
                    prop: 1374463283923981075,
                }),
            },
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(320017171),
                    bool: true,
                    key: 2206434179,
                    pos: 1374662778450838403,
                    length: 280459318858515,
                    prop: 207235723598,
                }),
            },
        ],
    );
}

#[test]
fn richtext_lamport_issue() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 196,
                target: 1,
                container: 1,
                action: Generic(GenericAction {
                    value: I32(123),
                    bool: true,
                    key: 16843009,
                    pos: 72340172838076673,
                    length: 72340172838076673,
                    prop: 72340172838076673,
                }),
            },
            Handle {
                site: 1,
                target: 1,
                container: 1,
                action: Generic(GenericAction {
                    value: I32(456),
                    bool: true,
                    key: 4294967041,
                    pos: 18446744073692849663,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            Checkout {
                site: 0,
                to: 20587776,
            },
            Handle {
                site: 1,
                target: 1,
                container: 1,
                action: Generic(GenericAction {
                    value: I32(789),
                    bool: true,
                    key: 16843009,
                    pos: 72340172838076673,
                    length: 72340172838076673,
                    prop: 72340172838076673,
                }),
            },
            SyncAll,
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 18446744073038462975,
                    prop: 15925010861198934015,
                }),
            },
        ],
    )
}

#[test]
fn tree_get_child_index() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(320017171),
                    bool: true,
                    key: 320017171,
                    pos: 1374463283923456787,
                    length: 1374472080016478995,
                    prop: 1374463283923456787,
                }),
            },
            Handle {
                site: 19,
                target: 19,
                container: 19,
                action: Generic(GenericAction {
                    value: I32(320017171),
                    bool: true,
                    key: 320017171,
                    pos: 1374463309693260563,
                    length: 1374463283923456787,
                    prop: 57140735609148179,
                }),
            },
            Sync { from: 171, to: 139 },
            Handle {
                site: 171,
                target: 171,
                container: 40,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: false,
                    key: 6007723,
                    pos: 41377556015874048,
                    length: 41565808209305343,
                    prop: 37687,
                }),
            },
        ],
    )
}

#[test]
fn delta_err() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(758724921),
                    bool: false,
                    key: 57,
                    pos: 6655295504416505856,
                    length: 4123474264100363356,
                    prop: 4123389851770370361,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370361,
                    length: 4123389851770370361,
                    prop: 4123389847475403263,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 14649,
                    pos: 6655194349346750464,
                    length: 4144999408248577116,
                    prop: 4123389851770370361,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370361,
                    length: 4123389851770370361,
                    prop: 4123388752258793273,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051554),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370361,
                    length: 4123389851770370361,
                    prop: 4123108376806635833,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 960051513,
                    pos: 0,
                    length: 18374733605602352220,
                    prop: 4123389851770370437,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960061952),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370361,
                    length: 4123389851770370361,
                    prop: 4123389855092259129,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 960051513,
                    pos: 4179111962746552121,
                    length: 3098478742654093369,
                    prop: 4123389851783332611,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 960051513,
                    pos: 3472301697646803343,
                    length: 18446743009517764400,
                    prop: 4123607322237534209,
                }),
            },
            Handle {
                site: 255,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn list_delete_change_to_diff_assert() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 61,
                target: 61,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(1027423549),
                    bool: true,
                    key: 1027423549,
                    pos: 4412750543122677053,
                    length: 3259829041783373823,
                    prop: 4412187962536443197,
                }),
            },
            Handle {
                site: 61,
                target: 61,
                container: 61,
                action: Generic(GenericAction {
                    value: I32(-12763843),
                    bool: true,
                    key: 1040187391,
                    pos: 4412750543122726717,
                    length: 1845454810429,
                    prop: 4444398755940139008,
                }),
            },
            Handle {
                site: 255,
                target: 59,
                container: 1,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4294967295,
                    pos: 4412750543122726911,
                    length: 9024436561550065151,
                    prop: 3602665457070193981,
                }),
            },
            Handle {
                site: 49,
                target: 49,
                container: 49,
                action: Generic(GenericAction {
                    value: I32(825307441),
                    bool: true,
                    key: 1027423537,
                    pos: 4436957391119789373,
                    length: 18391923786480696635,
                    prop: 4412750543122701885,
                }),
            },
            SyncAll,
            Handle {
                site: 61,
                target: 61,
                container: 61,
                action: Generic(GenericAction {
                    value: I32(1027423549),
                    bool: true,
                    key: 4294967295,
                    pos: 3544668469066546687,
                    length: 3616726063103684913,
                    prop: 18436571237510545407,
                }),
            },
            SyncAll,
            Handle {
                site: 61,
                target: 61,
                container: 61,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 2904489984,
                    pos: 18391361205894462893,
                    length: 2654601531,
                    prop: 0,
                }),
            },
        ],
    )
}
#[test]
fn delta_err_2() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 133,
                target: 35,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 75181359,
                    pos: 8286624611106291583,
                    length: 18446585211651489792,
                    prop: 6713178190254243839,
                }),
            },
            Checkout {
                site: 223,
                to: 2240120100,
            },
            Handle {
                site: 45,
                target: 255,
                container: 41,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 3255307777713450285,
                    prop: 3255307777713450285,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 768660269,
                    pos: 3255307777713450285,
                    length: 3255307777713450285,
                    prop: 3255307777713450285,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(-617796307),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 3255307777722690861,
                    prop: 3255307777713450285,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 757935405,
                    pos: 2559517409927834925,
                    length: 3255307777747922309,
                    prop: 3255263599679974701,
                }),
            },
            Handle {
                site: 115,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: false,
                    key: 704642816,
                    pos: 2656912673916387165,
                    length: 3255307777954121093,
                    prop: 3307302277364711213,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: false,
                    key: 0,
                    pos: 6148914326292922597,
                    length: 72057594037927936,
                    prop: 18446181123739353088,
                }),
            },
            Handle {
                site: 255,
                target: 133,
                container: 133,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2880154539,
                    pos: 17574988476103502763,
                    length: 3255307691477244185,
                    prop: 3255307777713450285,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 3255307777713450285,
                    prop: 3255307780632685869,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 3255307777713450285,
                    prop: 949464768499756507,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 3255307777710620717,
                    prop: 3255307777713450285,
                }),
            },
            Sync { from: 35, to: 133 },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 1929380141,
                    pos: 18446585211651489792,
                    length: 6713178190254243839,
                    prop: 9594038572176901375,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 16501189034685508909,
                    prop: 4268049625796902912,
                }),
            },
            Sync { from: 171, to: 171 },
            Sync { from: 187, to: 25 },
            Handle {
                site: 25,
                target: 25,
                container: 25,
                action: Generic(GenericAction {
                    value: I32(253303065),
                    bool: true,
                    key: 421075225,
                    pos: 1804000721327167783,
                    length: 1808504320951916825,
                    prop: 1808504320951916825,
                }),
            },
            Handle {
                site: 25,
                target: 25,
                container: 58,
                action: Generic(GenericAction {
                    value: I32(421075225),
                    bool: true,
                    key: 223,
                    pos: 18446490146653077568,
                    length: 2559517409933624831,
                    prop: 3255307777724218171,
                }),
            },
            Checkout {
                site: 4,
                to: 3959422847,
            },
            Handle {
                site: 0,
                target: 118,
                container: 11,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 4284295679,
                    pos: 9621136720216981508,
                    length: 18387402525678320517,
                    prop: 3255308571086552873,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 3255307777713450285,
                    prop: 3255307777713450285,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(-617796307),
                    bool: true,
                    key: 757935405,
                    pos: 3255307777713450285,
                    length: 3255307777722690861,
                    prop: 3255307777713450285,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 757935405,
                    pos: 2559517409927834925,
                    length: 3255307777747922309,
                    prop: 3255263599679974701,
                }),
            },
            Handle {
                site: 115,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: false,
                    key: 436207360,
                    pos: 1808504320951261465,
                    length: 16916094191212839,
                    prop: 1808476725365964800,
                }),
            },
            Handle {
                site: 25,
                target: 25,
                container: 133,
                action: Generic(GenericAction {
                    value: I32(-1532713820),
                    bool: false,
                    key: 2762253476,
                    pos: 11863788345444574372,
                    length: 18446744073693799588,
                    prop: 14377117046310043647,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 129,
                target: 133,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 421075225,
                    pos: 1808504320951916858,
                    length: 18014398524102937,
                    prop: 14395694632364015616,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 4,
                target: 33,
                container: 65,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 3318153216,
                    pos: 14339461213547661511,
                    length: 14395693707582439552,
                    prop: 18446744073709551501,
                }),
            },
            Sync { from: 133, to: 199 },
            Handle {
                site: 199,
                target: 199,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 3351741749,
                    pos: 14395516686210401735,
                    length: 10216353937893083079,
                    prop: 18446743833191383039,
                }),
            },
            Sync { from: 133, to: 199 },
            Handle {
                site: 199,
                target: 199,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 1070057415,
                    pos: 14395705430045360250,
                    length: 14395694108859942855,
                    prop: 14395693700603168645,
                }),
            },
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 2239837695,
                    pos: 14395694392076126151,
                    length: 14395694394777216967,
                    prop: 14395621827009807669,
                }),
            },
            SyncAll,
            Handle {
                site: 133,
                target: 35,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 75181359,
                    pos: 8286624611106291583,
                    length: 18446585211651489792,
                    prop: 6713178190254243839,
                }),
            },
            Checkout {
                site: 223,
                to: 2240120100,
            },
            SyncAll,
            Sync { from: 255, to: 255 },
            SyncAll,
            Handle {
                site: 129,
                target: 133,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 3351717831,
                    pos: 14395693707582490496,
                    length: 18446744073709551501,
                    prop: 14395621523933102079,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 2234894279,
                    pos: 4595861603807119303,
                    length: 71610056835194,
                    prop: 14395524409068552192,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 122,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(-16777216),
                    bool: true,
                    key: 45,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn delta_err_3() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 0,
                target: 0,
                container: 11,
                action: Generic(GenericAction {
                    value: I32(-65475),
                    bool: true,
                    key: 67108863,
                    pos: 72057594037871521,
                    length: 217020518514230020,
                    prop: 280883327347844581,
                }),
            },
            Sync { from: 163, to: 215 },
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(257),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 7936,
                    prop: 18446743017316026880,
                }),
            },
            Handle {
                site: 133,
                target: 0,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: false,
                    key: 2717908991,
                    pos: 2155061665,
                    length: 18446463698227810304,
                    prop: 9476562641788076031,
                }),
            },
            Sync { from: 255, to: 255 },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 791643507,
                    pos: 18446744073709551568,
                    length: 2965947086361162589,
                    prop: 18446744070104864675,
                }),
            },
            Handle {
                site: 0,
                target: 0,
                container: 199,
                action: Generic(GenericAction {
                    value: I32(-1547197533),
                    bool: true,
                    key: 4294912471,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn delta_err_4() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(-404281047),
                    bool: true,
                    key: 3890735079,
                    pos: 15287796486090778599,
                    length: 2965947086361143593,
                    prop: 8874669797471234345,
                }),
            },
            Handle {
                site: 59,
                target: 41,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4294967295,
                    pos: 16044073672507391,
                    length: 3026418949592973311,
                    prop: 2965947086361143593,
                }),
            },
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 3890735079,
                    pos: 16710579925595711463,
                    length: 16710579925595711463,
                    prop: 2965947089230047719,
                }),
            },
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 690563583,
                    pos: 2965947086361143593,
                    length: 16710579925595711273,
                    prop: 16710579925595711463,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(-54999),
                    bool: true,
                    key: 4294967295,
                    pos: 3026418949592973311,
                    length: 2971008166175692667,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690579070,
                    pos: 707340585,
                    length: 2965947086361198340,
                    prop: 16710370199142476073,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 2950134569,
                    pos: 18446743151283810209,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 690563369,
                    pos: 2965947086361143593,
                    length: 16710579925595711273,
                    prop: 16674551128576747495,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(707340585),
                    bool: false,
                    key: 67108864,
                    pos: 2965947086361143807,
                    length: 16710579106351753513,
                    prop: 16710579925595711463,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 690563369,
                    pos: 2966183304576892927,
                    length: 2965947086361143593,
                    prop: 16710579106351753513,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 212,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 18375812379578477097,
                    length: 2965947086361143593,
                    prop: 16710579922395539753,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 212,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 4280887593,
                    pos: 18446744073709551615,
                    length: 15527050319778283519,
                    prop: 18446507932719751599,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(694583015),
                    bool: true,
                    key: 707340585,
                    pos: 2966182222245134336,
                    length: 2965947086361143593,
                    prop: 16710579925595662633,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 3615172905,
                    pos: 18446507932719751599,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 690563369,
                    pos: 2965947086361143593,
                    length: 16710579925595662633,
                    prop: 7487207888740935655,
                }),
            },
            SyncAll,
            Handle {
                site: 212,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 3026141872662773802,
                    length: 2965947086361143593,
                    prop: 16710579925583210793,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(2066295081),
                    bool: true,
                    key: 607101359,
                    pos: 18446744073709496635,
                    length: 16710579925595711463,
                    prop: 2966135102861993959,
                }),
            },
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(10793),
                    bool: false,
                    key: 704578560,
                    pos: 2965947086361143593,
                    length: 16710579805324126505,
                    prop: 16710579925595711463,
                }),
            },
            SyncAll,
            Handle {
                site: 41,
                target: 212,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 16710579922395539753,
                    length: 16710579925595711463,
                    prop: 16710579925595678695,
                }),
            },
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(707340585),
                    bool: false,
                    key: 67108864,
                    pos: 2965947086361143807,
                    length: 16710579106351753513,
                    prop: 16710579925595711463,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 695937321,
                    pos: 4261583518885706537,
                    length: 18446744073709551401,
                    prop: 3314649325744685055,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(2129078569),
                    bool: false,
                    key: 690563369,
                    pos: 18375812379578477097,
                    length: 2965947086361143593,
                    prop: 16710579922395539753,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 212,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 2971008166175692667,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            Handle {
                site: 0,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 690563369,
                    pos: 3026418949592973097,
                    length: 217020518514230057,
                    prop: 217020518514230019,
                }),
            },
            Handle {
                site: 3,
                target: 3,
                container: 3,
                action: Generic(GenericAction {
                    value: I32(50529027),
                    bool: true,
                    key: 50529027,
                    pos: 217020518514230019,
                    length: 217020518514230019,
                    prop: 217020518514230019,
                }),
            },
            Handle {
                site: 3,
                target: 3,
                container: 3,
                action: Generic(GenericAction {
                    value: I32(50529027),
                    bool: true,
                    key: 50529027,
                    pos: 217020518514230019,
                    length: 217020518514230019,
                    prop: 217020518514230019,
                }),
            },
            Handle {
                site: 3,
                target: 3,
                container: 3,
                action: Generic(GenericAction {
                    value: I32(50529027),
                    bool: true,
                    key: 690563331,
                    pos: 2965947086361143593,
                    length: 16710579925595662633,
                    prop: 7487207888740935655,
                }),
            },
            SyncAll,
            Handle {
                site: 212,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 3026141872662773802,
                    length: 2965947086361143593,
                    prop: 16710579925583210793,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(2066295081),
                    bool: true,
                    key: 607101359,
                    pos: 18446744073709496635,
                    length: 18446744073709551615,
                    prop: 18446744073709551405,
                }),
            },
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: true,
                    key: 690563583,
                    pos: 2965947086361143593,
                    length: 16710579922395539753,
                    prop: 16710579925595711463,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: false,
                    key: 687931392,
                    pos: 4261583518885706537,
                    length: 18446744073709551401,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(-1582311557),
                    bool: true,
                    key: 4280892196,
                    pos: 16710580029079158783,
                    length: 16710579925595711463,
                    prop: 2965947086372350249,
                }),
            },
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(67108864),
                    bool: true,
                    key: 690563369,
                    pos: 2965947086361143593,
                    length: 16710579925595704295,
                    prop: 16710579925595711463,
                }),
            },
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 3615172905,
                    pos: 18446507932719751599,
                    length: 18446744073709551615,
                    prop: 18446743171766419455,
                }),
            },
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690579070,
                    pos: 707340585,
                    length: 2965947086361198340,
                    prop: 16710370199142476073,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 10596311,
                    pos: 2965948004860690432,
                    length: 18446507855493802281,
                    prop: 2965947089965547519,
                }),
            },
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 16710579922395539753,
                    length: 16710579925595711463,
                    prop: 16710579925595678695,
                }),
            },
            SyncAll,
            Handle {
                site: 49,
                target: 54,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(690563381),
                    bool: true,
                    key: 690563369,
                    pos: 288230376154474793,
                    length: 2965947086361143807,
                    prop: 16710579106351753513,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 212,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 4261583518885706537,
                    length: 18446744073709551401,
                    prop: 3314649325744685055,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 4294967295,
                    pos: 2965947086361198591,
                    length: 2965947086361143593,
                    prop: 16710579925595711273,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: false,
                    key: 687931392,
                    pos: 4261583518885706537,
                    length: 18446744073709551401,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn delta_err_5() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(-1960236759),
                    bool: true,
                    key: 2341178251,
                    pos: 10055284024492657547,
                    length: 10055284174816512907,
                    prop: 10055284024492657547,
                }),
            },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 139, to: 139 },
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(704578611),
                    bool: true,
                    key: 992553257,
                    pos: 18446507855497821737,
                    length: 2965907327158321151,
                    prop: 3026141872662773802,
                }),
            },
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 690563369,
                    pos: 2965947086361143593,
                    length: 16710579925595711463,
                    prop: 111264917479,
                }),
            },
            Checkout {
                site: 49,
                to: 2139042100,
            },
            Checkout {
                site: 127,
                to: 3890734975,
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 41,
                target: 41,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(65536),
                    bool: false,
                    key: 0,
                    pos: 2965947086358523649,
                    length: 9187202097275697961,
                    prop: 16681191730380242815,
                }),
            },
            SyncAll,
            Handle {
                site: 41,
                target: 212,
                container: 41,
                action: Generic(GenericAction {
                    value: I32(690563369),
                    bool: true,
                    key: 690563369,
                    pos: 2971008166413171497,
                    length: 41,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_5() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 3,
                target: 34,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: false,
                    key: 50536963,
                    pos: 217020518514230019,
                    length: 217020518514230019,
                    prop: 217020518514230019,
                }),
            },
            SyncAll,
            Handle {
                site: 3,
                target: 3,
                container: 3,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 4294967295,
                    pos: 3399987922982666239,
                    length: 940450980798869287,
                    prop: 5391038347781079093,
                }),
            },
            Checkout {
                site: 3,
                to: 2072347904,
            },
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_6() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 44,
                target: 124,
                container: 221,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: false,
                    key: 38,
                    pos: 150994944,
                    length: 18446742974197923840,
                    prop: 18446744073709551615,
                }),
            },
            Handle {
                site: 194,
                target: 239,
                container: 251,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 18446608833779269692,
                    prop: 18446744073708503039,
                }),
            },
            Handle {
                site: 0,
                target: 255,
                container: 133,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_7() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 44,
                target: 124,
                container: 221,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3351758791,
                    pos: 288230650086410183,
                    length: 2606365581092837153,
                    prop: 15553136935972341051,
                }),
            },
            SyncAll,
            Handle {
                site: 0,
                target: 209,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(1145324612),
                    bool: true,
                    key: 3351758806,
                    pos: 9187202260886079431,
                    length: 72056541770940543,
                    prop: 70127282814975,
                }),
            },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_8() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 3,
                target: 34,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 2582786094,
                    pos: 18374686655815811843,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            Handle {
                site: 3,
                target: 133,
                container: 3,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370361,
                    length: 4123389851770370361,
                    prop: 4123389851770370361,
                }),
            },
            Handle {
                site: 57,
                target: 59,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(825307441),
                    bool: true,
                    key: 825307441,
                    pos: 3544668469065756977,
                    length: 3544668469065756977,
                    prop: 3544668469065756977,
                }),
            },
            Handle {
                site: 49,
                target: 49,
                container: 49,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370361,
                    length: 268877889158068537,
                    prop: 253612265486615299,
                }),
            },
            Handle {
                site: 3,
                target: 215,
                container: 213,
                action: Generic(GenericAction {
                    value: I32(3),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_9() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 1,
                target: 64,
                container: 36,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 2248146944,
                    pos: 254396807995,
                    length: 4268070197446523737,
                    prop: 18446744073696655675,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 4268070197446523648,
                    prop: 360287970189639680,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 4268070197446523649,
                    prop: 4268070196455800882,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294967045,
                    pos: 18413964932892298239,
                    length: 3746779384955142143,
                    prop: 255,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_10() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 1,
                target: 64,
                container: 36,
                action: Generic(GenericAction {
                    value: I32(989855744),
                    bool: true,
                    key: 2248146944,
                    pos: 4268102928402430779,
                    length: 4268070197446523707,
                    prop: 18446744073709551615,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294903040,
                    pos: 4268007270886932479,
                    length: 3314707854257765179,
                    prop: 4268070197446523648,
                }),
            },
            Handle {
                site: 89,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-281330885),
                    bool: true,
                    key: 4294967099,
                    pos: 13021231110858735615,
                    length: 13021231110853801140,
                    prop: 18425550663698396340,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4278517760,
                    pos: 2199023255551,
                    length: 13575924464958210,
                    prop: 18444988998762561582,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993722414),
                    bool: true,
                    key: 4294916923,
                    pos: 7306357456639098880,
                    length: 7306357456645743973,
                    prop: 7306357456645729125,
                }),
            },
            Checkout {
                site: 101,
                to: 1701143909,
            },
            Checkout {
                site: 101,
                to: 1701143909,
            },
            Checkout {
                site: 101,
                to: 25957,
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(989867520),
                    bool: false,
                    key: 0,
                    pos: 18446744073709487360,
                    length: 71833290377462271,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_11() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 1,
                target: 64,
                container: 36,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 2248146944,
                    pos: 254396807995,
                    length: 4268070197446523737,
                    prop: 18446744073696656187,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 4268070197446523648,
                    prop: 360287970189639680,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(989867520),
                    bool: true,
                    key: 993737531,
                    pos: 4323455642275625787,
                    length: 254396807995,
                    prop: 18446742995672760320,
                }),
            },
            Sync { from: 139, to: 139 },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 4268070196473445179,
                    prop: 4268070196455800882,
                }),
            },
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 65535,
                    pos: 4268071042544893952,
                    length: 18,
                    prop: 4268071042561343487,
                }),
            },
            Handle {
                site: 0,
                target: 59,
                container: 42,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 0,
                    pos: 4268070197442641920,
                    length: 15163,
                    prop: 18446744070085608704,
                }),
            },
            Sync { from: 139, to: 139 },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 721420288,
                    length: 4268070196473445179,
                    prop: 4268070196455800882,
                }),
            },
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: false,
                    key: 0,
                    pos: 10088063165293461504,
                    length: 35,
                    prop: 10088063161035600383,
                }),
            },
            Handle {
                site: 255,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_12() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: false,
                    key: 4294934527,
                    pos: 18446744073709551419,
                    length: 4268071042561343487,
                    prop: 4268070197446523698,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 1929430318,
                    pos: 3617009275739948403,
                    length: 32213981296852992,
                    prop: 15553079759578595328,
                }),
            },
            Sync { from: 211, to: 59 },
            Checkout {
                site: 215,
                to: 1004001063,
            },
            Handle {
                site: 0,
                target: 0,
                container: 220,
                action: Generic(GenericAction {
                    value: I32(96),
                    bool: false,
                    key: 2231369728,
                    pos: 9600413840299196417,
                    length: 46059242167205892,
                    prop: 18446744073709551575,
                }),
            },
            Handle {
                site: 0,
                target: 215,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294901817,
                    pos: 18446744073709551615,
                    length: 4123390155739970047,
                    prop: 18390793471196266041,
                }),
            },
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(965082880),
                    bool: true,
                    key: 960051513,
                    pos: 9983072998996785465,
                    length: 18446744073709551498,
                    prop: 18446744073709551615,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(-65536),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 4121983323008344063,
                    prop: 18374749397238495545,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(-50887),
                    bool: true,
                    key: 3750179,
                    pos: 18446744073709551615,
                    length: 4123367861550841855,
                    prop: 4123389851770370361,
                }),
            },
            Handle {
                site: 111,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 4280891691,
                    pos: 4179121897159080249,
                    length: 2538122782935628286,
                    prop: 18446744073692789049,
                }),
            },
            Handle {
                site: 0,
                target: 254,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 4294967040,
                    pos: 18446744069414584320,
                    length: 4123389851770370559,
                    prop: 4123389851686534713,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(943270713),
                    bool: true,
                    key: 4294967295,
                    pos: 4268222525080978885,
                    length: 18446743228594731835,
                    prop: 18446744073709551615,
                }),
            },
            Handle {
                site: 0,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 1006582587,
                    pos: 10173452862450645819,
                    length: 18446744073709501325,
                    prop: 4268286546840387583,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 3318037307,
                    pos: 55501997373507013,
                    length: 14251014049101083507,
                    prop: 4268070199770858949,
                }),
            },
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_13() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 164,
                target: 164,
                container: 164,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: false,
                    key: 2762253476,
                    pos: 18446744069677032612,
                    length: 9639893187170402303,
                    prop: 14395694392065640391,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Sync { from: 199, to: 199 },
            SyncAll,
            Handle {
                site: 199,
                target: 199,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3351758631,
                    pos: 18446682501058396045,
                    length: 14377039454378393599,
                    prop: 14395693703287523271,
                }),
            },
            SyncAll,
            Handle {
                site: 199,
                target: 199,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 889192447,
                    pos: 2794421754670843265,
                    length: 14395518472916813767,
                    prop: 14357413797944608711,
                }),
            },
            SyncAll,
            SyncAll,
            Sync { from: 133, to: 199 },
            Handle {
                site: 199,
                target: 199,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3351741749,
                    pos: 14395516686210401735,
                    length: 10216353937893083079,
                    prop: 18446743833191383039,
                }),
            },
            Sync { from: 133, to: 199 },
            Handle {
                site: 199,
                target: 199,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 1070057415,
                    pos: 71610056835194,
                    length: 0,
                    prop: 14395621827009824512,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Sync { from: 133, to: 199 },
            Handle {
                site: 199,
                target: 199,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 1094698951,
                    pos: 14398519423411421306,
                    length: 14395621199944599495,
                    prop: 14395516686210401735,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 129,
                target: 133,
                container: 199,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3351717831,
                    pos: 18446744070954403783,
                    length: 18446744073709486134,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 0,
                target: 4,
                container: 33,
                action: Generic(GenericAction {
                    value: I32(-774778624),
                    bool: true,
                    key: 2051096519,
                    pos: 15069330226212913408,
                    length: 9598797841674258385,
                    prop: 2749385757289859015,
                }),
            },
            SyncAll,
            Sync { from: 255, to: 255 },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 4,
                target: 33,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_14() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 33,
                target: 209,
                container: 197,
                action: Generic(GenericAction {
                    value: I32(-1179010631),
                    bool: true,
                    key: 3115956665,
                    pos: 13382931975044184505,
                    length: 13382931975044184505,
                    prop: 13382931975044184505,
                }),
            },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Checkout {
                site: 45,
                to: 2147483629,
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 45,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 3537031981,
                    pos: 13382910049657213650,
                    length: 562949940558265,
                    prop: 18446744073709486336,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: true,
                    key: 991640575,
                    pos: 4268070197446523707,
                    length: 65125582846779,
                    prop: 18446743228596882176,
                }),
            },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Handle {
                site: 237,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 757982679,
                    pos: 16565899576820444512,
                    length: 2378153378818021335,
                    prop: 14470860355616821971,
                }),
            },
            Sync { from: 59, to: 255 },
            Handle {
                site: 185,
                target: 185,
                container: 185,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3115956665,
                    pos: 13382931975044184505,
                    length: 13382931975044184505,
                    prop: 13382931975044184505,
                }),
            },
            SyncAll,
            Handle {
                site: 229,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 3857001773,
                    pos: 16565899519554217445,
                    length: 15191436295996124461,
                    prop: 4303675126263957714,
                }),
            },
            Handle {
                site: 0,
                target: 1,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 1006632959,
                    pos: 4268070197446523707,
                    length: 18446744073709551361,
                    prop: 4268286546840387583,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 993737531,
                    pos: 4268070197446523707,
                    length: 4277305840723049275,
                    prop: 72057602627862331,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-197),
                    bool: true,
                    key: 456916991,
                    pos: 4268070197446523707,
                    length: 16672149208775483,
                    prop: 18446744073709551360,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 4294902016,
                    pos: 18446744073709551615,
                    length: 4268070197459419135,
                    prop: 18446527724320078651,
                }),
            },
            Handle {
                site: 27,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 993737531,
                    pos: 18391358769805131776,
                    length: 13382728867219898367,
                    prop: 13382931975044184505,
                }),
            },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            Sync { from: 185, to: 185 },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 184549376,
                    pos: 4991460930710354780,
                    length: 4991471925827290437,
                    prop: 4268070197446523707,
                }),
            },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_15() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 1,
                target: 64,
                container: 36,
                action: Generic(GenericAction {
                    value: I32(989855744),
                    bool: true,
                    key: 2248146944,
                    pos: 4268102928402430779,
                    length: 4268070197446523707,
                    prop: 18446744073709551615,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294903040,
                    pos: 4268007270886932479,
                    length: 3314707854257765179,
                    prop: 4268070197446523648,
                }),
            },
            Handle {
                site: 89,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(1005534011),
                    bool: true,
                    key: 4294967295,
                    pos: 13021231110853820415,
                    length: 13021231110853801140,
                    prop: 18446661286951695540,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294903040,
                    pos: 17578395995350237183,
                    length: 4268273300959130611,
                    prop: 4251398048275184443,
                }),
            },
            Sync { from: 59, to: 59 },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-50373),
                    bool: true,
                    key: 4294967295,
                    pos: 3314707854257765179,
                    length: 18446527724315687680,
                    prop: 7306357456639098880,
                }),
            },
            Checkout {
                site: 101,
                to: 1701143909,
            },
            Checkout {
                site: 101,
                to: 838860800,
            },
            Handle {
                site: 101,
                target: 101,
                container: 101,
                action: Generic(GenericAction {
                    value: I32(1701143909),
                    bool: true,
                    key: 1701143909,
                    pos: 7306357456645743973,
                    length: 18446744073693102080,
                    prop: 4268070197442772991,
                }),
            },
            Handle {
                site: 0,
                target: 59,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(993737472),
                    bool: true,
                    key: 16777019,
                    pos: 7306357456645718016,
                    length: 7306357456645743973,
                    prop: 7306357456645743973,
                }),
            },
            Checkout {
                site: 101,
                to: 1701143909,
            },
            Checkout {
                site: 101,
                to: 973078405,
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 2215329547,
                    pos: 18446744073709551527,
                    length: 18446525516668811577,
                    prop: 17578661999653421055,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(15163),
                    bool: false,
                    key: 993737606,
                    pos: 4268070197448474624,
                    length: 18446743228594731835,
                    prop: 4268070200747753471,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: true,
                    key: 1701143909,
                    pos: 65306678879589,
                    length: 18446742995672760320,
                    prop: 18391573887352569855,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_16() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 1,
                target: 64,
                container: 36,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 2248146944,
                    pos: 254396807995,
                    length: 4268070197446523737,
                    prop: 18446744073696656187,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 4268070197446523648,
                    prop: 360287970189639680,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(989867520),
                    bool: true,
                    key: 993737531,
                    pos: 4323455642275625787,
                    length: 254396807995,
                    prop: 18446742995672760320,
                }),
            },
            Sync { from: 139, to: 139 },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 4268070196473445179,
                    prop: 4268070196455800882,
                }),
            },
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 50529279,
                    pos: 217064709432738563,
                    length: 1095216661759,
                    prop: 4268070200747689216,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 16672149208775483,
                    prop: 18376093854555176960,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-50373),
                    bool: true,
                    key: 993787903,
                    pos: 18446744073696656187,
                    length: 14138873509707775,
                    prop: 18446616030185455662,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_17() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 41,
                target: 34,
                container: 115,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 1111621632,
                    pos: 4774451407313060418,
                    length: 3242591731706774082,
                    prop: 16565696476406951213,
                }),
            },
            Handle {
                site: 0,
                target: 229,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935365),
                    bool: true,
                    key: 3621250533,
                    pos: 3255307777725556197,
                    length: 9596332072585997613,
                    prop: 3288537597569377199,
                }),
            },
            Handle {
                site: 2,
                target: 192,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: false,
                    key: 4294901760,
                    pos: 18446597888123012403,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            Checkout {
                site: 38,
                to: 4294960036,
            },
            Sync { from: 11, to: 123 },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(771650308),
                    bool: true,
                    key: 3772157069,
                    pos: 4268070199141008499,
                    length: 8303295463570488123,
                    prop: 4268070197446549508,
                }),
            },
            Handle {
                site: 18,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 993737531,
                    pos: 4774378554970028859,
                    length: 4774451407313060418,
                    prop: 3242591731706774082,
                }),
            },
            Handle {
                site: 45,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 1944114828,
                    pos: 4268070197453142788,
                    length: 320665124056283963,
                    prop: 4268070197446523808,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 993737531,
                    pos: 4774451122733595451,
                    length: 4774451407313060418,
                    prop: 3255258105658736706,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_18() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 41,
                target: 34,
                container: 115,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 1111621632,
                    pos: 4774451407313060418,
                    length: 3242591731706774082,
                    prop: 16565696476406951213,
                }),
            },
            Handle {
                site: 0,
                target: 229,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935365),
                    bool: true,
                    key: 3621250533,
                    pos: 3255307777725556197,
                    length: 9596332072585997613,
                    prop: 3288537597569377199,
                }),
            },
            Handle {
                site: 2,
                target: 192,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: false,
                    key: 4294901760,
                    pos: 18446597888123012403,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            Checkout {
                site: 38,
                to: 4294960036,
            },
            Sync { from: 11, to: 123 },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(771650308),
                    bool: true,
                    key: 3772157069,
                    pos: 4268070199141008499,
                    length: 8303295463570488123,
                    prop: 4268070197446549508,
                }),
            },
            Handle {
                site: 18,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 993737531,
                    pos: 4774378554970028859,
                    length: 4774451407313060418,
                    prop: 3242591731706774082,
                }),
            },
            Handle {
                site: 45,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 1944114828,
                    pos: 4268070197453142788,
                    length: 320665124056283963,
                    prop: 4268070197446523808,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 993737531,
                    pos: 4774451122733595451,
                    length: 4774451407313060418,
                    prop: 3255258105658736706,
                }),
            },
            SyncAll,
            SyncAll,
            Sync { from: 227, to: 255 },
            Handle {
                site: 132,
                target: 11,
                container: 123,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 0,
                    pos: 3314212838480740352,
                    length: 4252651357653994811,
                    prop: 4268070197446523808,
                }),
            },
            Checkout {
                site: 4,
                to: 992361376,
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 993737531,
                    pos: 4268070197446523707,
                    length: 4268070197446523707,
                    prop: 4268070197446523707,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-475781517),
                    bool: true,
                    key: 187957247,
                    pos: 18446743470183515147,
                    length: 34359738367,
                    prop: 18335003334139707392,
                }),
            },
            SyncAll,
        ],
    )
}

#[test]
fn test_movable_list_19() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 33,
                target: 209,
                container: 197,
                action: Generic(GenericAction {
                    value: I32(129612217),
                    bool: true,
                    key: 4294967295,
                    pos: 13382931975044202495,
                    length: 13382931975044184505,
                    prop: 13382931975044184505,
                }),
            },
            Sync { from: 185, to: 185 },
            Checkout {
                site: 45,
                to: 2147483629,
            },
            SyncAll,
            Handle {
                site: 45,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 3537031981,
                    pos: 13382910049657213650,
                    length: 562949940558265,
                    prop: 18446744073709486336,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: true,
                    key: 991640575,
                    pos: 4268070197446523707,
                    length: 2305908134796540731,
                    prop: 18446743228596882176,
                }),
            },
            Sync { from: 185, to: 13 },
            Handle {
                site: 237,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 757982679,
                    pos: 16565899576820444461,
                    length: 2442610164937250775,
                    prop: 14470860355616821971,
                }),
            },
            Sync { from: 59, to: 255 },
            Handle {
                site: 185,
                target: 185,
                container: 185,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3115956665,
                    pos: 13382931975044184505,
                    length: 13382931975044184505,
                    prop: 13382931975044184505,
                }),
            },
            SyncAll,
            Handle {
                site: 229,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 757982679,
                    pos: 16565899576820444461,
                    length: 2378153378818021335,
                    prop: 14470860355616821971,
                }),
            },
            Sync { from: 59, to: 255 },
            Handle {
                site: 185,
                target: 185,
                container: 185,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3115956665,
                    pos: 13382931975044184505,
                    length: 13382931975044184505,
                    prop: 13382931975044184505,
                }),
            },
            SyncAll,
            Handle {
                site: 229,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 3857001773,
                    pos: 16565899519554217445,
                    length: 15191436295996124461,
                    prop: 4303675126263957714,
                }),
            },
            Handle {
                site: 0,
                target: 1,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 1006632959,
                    pos: 4268070197446523707,
                    length: 18446744073709551361,
                    prop: 4268286546840387583,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 993737727,
                    pos: 18391358628880399163,
                    length: 4259063843306602495,
                    prop: 4268070197446523707,
                }),
            },
            Handle {
                site: 59,
                target: 0,
                container: 23,
                action: Generic(GenericAction {
                    value: I32(-197),
                    bool: true,
                    key: 3103850496,
                    pos: 13382931975044184505,
                    length: 13382931975044184505,
                    prop: 13382931975035730361,
                }),
            },
            SyncAll,
            Handle {
                site: 229,
                target: 229,
                container: 229,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 3857001773,
                    pos: 16565899519554217445,
                    length: 15191436295996121344,
                    prop: 4303675126263957714,
                }),
            },
            Handle {
                site: 0,
                target: 185,
                container: 185,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 3115956665,
                    pos: 13382931975044184505,
                    length: 13382931975044184505,
                    prop: 13382931975044184505,
                }),
            },
            Handle {
                site: 237,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 757982679,
                    pos: 16565899576820444461,
                    length: 3255510881225136087,
                    prop: 14470860355616821971,
                }),
            },
            Sync { from: 59, to: 255 },
            Handle {
                site: 1,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 993787903,
                    pos: 88729743246703419,
                    length: 18446744073709551615,
                    prop: 4268071042561343487,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: true,
                    key: 993737531,
                    pos: 4268070197446523707,
                    length: 4268106274178072635,
                    prop: 18374967954681888767,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: true,
                    key: 991640575,
                    pos: 4268070197446523707,
                    length: 65125582846779,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 4294967041,
                    pos: 18446744073709551615,
                    length: 4268070196909703167,
                    prop: 4268070197446523707,
                }),
            },
            Handle {
                site: 0,
                target: 11,
                container: 92,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 4268070197446523903,
                    prop: 18446744073696656187,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 4278190139,
                    pos: 18446744073709551615,
                    length: 4268070200747753471,
                    prop: 18446527724315687739,
                }),
            },
            Handle {
                site: 1,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 993787903,
                    pos: 4268143864725584699,
                    length: 4323455642275675963,
                    prop: 4268070197446523675,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(-12895396),
                    bool: true,
                    key: 511,
                    pos: 13382931975044184321,
                    length: 13382931975044184505,
                    prop: 13382931975044184505,
                }),
            },
            Sync { from: 185, to: 185 },
            SyncAll,
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 59,
                    pos: 4988657175891762187,
                    length: 4991471925827290437,
                    prop: 4268070197446523717,
                }),
            },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_20() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 0,
                target: 64,
                container: 36,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: true,
                    key: 2248146944,
                    pos: 4268102928402430779,
                    length: 18446468096294861627,
                    prop: 4268007270886932479,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294903040,
                    pos: 4268007270886932479,
                    length: 3314707854257765179,
                    prop: 4268070197446523648,
                }),
            },
            Handle {
                site: 89,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(1005534011),
                    bool: true,
                    key: 4294967295,
                    pos: 13021231110853820415,
                    length: 13021085206519788724,
                    prop: 13021234409388684468,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(59),
                    bool: false,
                    key: 83886080,
                    pos: 14033993531614298111,
                    length: 3369469612973540034,
                    prop: 18446743810689123010,
                }),
            },
            Handle {
                site: 59,
                target: 59,
                container: 59,
                action: Generic(GenericAction {
                    value: I32(100663296),
                    bool: false,
                    key: 0,
                    pos: 144114088564228096,
                    length: 3835724633218956091,
                    prop: 4268070197442653696,
                }),
            },
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(993737531),
                    bool: false,
                    key: 0,
                    pos: 18446744073693102080,
                    length: 18389322329063621119,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_21() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Map, FuzzTarget::MovableList],
        &mut [
            Handle {
                site: 0,
                target: 35,
                container: 10,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4278386687,
                    pos: 2676586395023179775,
                    length: 2676586395008836901,
                    prop: 2676586395008836901,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 9895591935999,
                    prop: 3746994885677285376,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 37,
                target: 37,
                container: 37,
                action: Generic(GenericAction {
                    value: I32(623191333),
                    bool: true,
                    key: 3052521929,
                    pos: 18446744073709551615,
                    length: 18446744073709551615,
                    prop: 18446742987082838527,
                }),
            },
            Handle {
                site: 255,
                target: 253,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(265),
                    bool: false,
                    key: 4294967295,
                    pos: 16493559407536242687,
                    length: 3904675852509832420,
                    prop: 3834029159525384194,
                }),
            },
            Handle {
                site: 53,
                target: 53,
                container: 53,
                action: Generic(GenericAction {
                    value: I32(892679477),
                    bool: true,
                    key: 805255221,
                    pos: 72110370596060693,
                    length: 2108141021440,
                    prop: 2676586395008837078,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: true,
                    key: 4294967295,
                    pos: 18086737578496602879,
                    length: 18385141895277182975,
                    prop: 18446742974197924095,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 1627389741,
                    pos: 67553161186901867,
                    length: 2676586395008836901,
                    prop: 13110481856001025317,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 3,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 151597311,
                    pos: 18446742974197924105,
                    length: 16493559523500359679,
                    prop: 3472838262201312484,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_22() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Map, FuzzTarget::MovableList],
        &mut [
            Handle {
                site: 0,
                target: 35,
                container: 10,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4278386687,
                    pos: 2676586395023179775,
                    length: 2676586395008836901,
                    prop: 2676586395008836901,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 9895604649983,
                    prop: 3746994885677285376,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 37,
                target: 37,
                container: 37,
                action: Generic(GenericAction {
                    value: I32(623183141),
                    bool: true,
                    key: 3052521929,
                    pos: 18446744073709551615,
                    length: 18446744073709551615,
                    prop: 18446742987082838527,
                }),
            },
            Handle {
                site: 255,
                target: 253,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(265),
                    bool: false,
                    key: 4294967295,
                    pos: 16493559407536242687,
                    length: 3616728050846459108,
                    prop: 3834029159525384194,
                }),
            },
            Handle {
                site: 53,
                target: 53,
                container: 53,
                action: Generic(GenericAction {
                    value: I32(892679477),
                    bool: true,
                    key: 805255221,
                    pos: 72110370596060693,
                    length: 2108141021440,
                    prop: 2676586395008837078,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: true,
                    key: 4294967295,
                    pos: 18086737578496602879,
                    length: 2533271535615999,
                    prop: 18446742974197923840,
                }),
            },
            SyncAll,
            Handle {
                site: 0,
                target: 37,
                container: 37,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: true,
                    key: 623191333,
                    pos: 2676586395008836901,
                    length: 2676586395008836901,
                    prop: 13088935740243780901,
                }),
            },
            Sync { from: 255, to: 123 },
            Handle {
                site: 56,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: true,
                    key: 4294967295,
                    pos: 1155454779397242677,
                    length: 3026417850081345536,
                    prop: 18446744073709496617,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
        ],
    )
}

#[test]
fn test_movable_list_23() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Map, FuzzTarget::MovableList],
        &mut [
            Handle {
                site: 0,
                target: 35,
                container: 10,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4278386687,
                    pos: 2676586395023175679,
                    length: 2676586395008836901,
                    prop: 2676586395008836901,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 9895591935999,
                    prop: 3746994885677285376,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 37,
                target: 37,
                container: 37,
                action: Generic(GenericAction {
                    value: I32(623191333),
                    bool: true,
                    key: 3052521929,
                    pos: 18390785747968851967,
                    length: 18446744073696116735,
                    prop: 18446742987082838527,
                }),
            },
            Handle {
                site: 255,
                target: 253,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(265),
                    bool: false,
                    key: 4294967295,
                    pos: 16493559407536242687,
                    length: 3904675852509832420,
                    prop: 3834029159525384194,
                }),
            },
            Handle {
                site: 53,
                target: 53,
                container: 53,
                action: Generic(GenericAction {
                    value: I32(892679477),
                    bool: true,
                    key: 792199476,
                    pos: 72110370596060693,
                    length: 2108141021440,
                    prop: 2676586395008837078,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 255,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: true,
                    key: 4294967295,
                    pos: 12587190423081956095,
                    length: 12587190073825341102,
                    prop: 71968184339443374,
                }),
            },
            Handle {
                site: 255,
                target: 255,
                container: 8,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 151597311,
                    pos: 18446742974197924105,
                    length: 16493559523500359679,
                    prop: 18374686483511829732,
                }),
            },
            Checkout {
                site: 0,
                to: 2667577554,
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 9437183,
                    pos: 13672292010998890496,
                    length: 13672292666392100285,
                    prop: 13672292666396491197,
                }),
            },
            Handle {
                site: 37,
                target: 37,
                container: 37,
                action: Generic(GenericAction {
                    value: I32(623191333),
                    bool: true,
                    key: 3048580413,
                    pos: 3401761431743735221,
                    length: 13552791426844167701,
                    prop: 65395,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_24() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Map, FuzzTarget::MovableList],
        &mut [
            Handle {
                site: 96,
                target: 255,
                container: 36,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 721449843,
                    pos: 4292910109560504340,
                    length: 18446513810110911379,
                    prop: 3746994894253522943,
                }),
            },
            Sync { from: 93, to: 167 },
            Checkout {
                site: 123,
                to: 4286282619,
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 253,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 1553399936,
                    pos: 18390449078367420160,
                    length: 36201467589165055,
                    prop: 18374967954631580824,
                }),
            },
            SyncAll,
            Handle {
                site: 43,
                target: 43,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2071690107,
                    pos: 18446744071486274427,
                    length: 17216960130726232104,
                    prop: 7430094960254740462,
                }),
            },
            SyncAll,
            Handle {
                site: 43,
                target: 43,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2726001531,
                    pos: 18446744071486274427,
                    length: 17216960130726232104,
                    prop: 7430094960254740462,
                }),
            },
            Checkout {
                site: 0,
                to: 6094847,
            },
            Sync { from: 139, to: 139 },
            Sync { from: 139, to: 139 },
            Handle {
                site: 139,
                target: 43,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2071690107,
                    pos: 18446744071486274427,
                    length: 17178700329823240232,
                    prop: 7430094960254715374,
                }),
            },
            Checkout {
                site: 0,
                to: 6094847,
            },
            Sync { from: 139, to: 139 },
            Sync { from: 55, to: 55 },
            Handle {
                site: 55,
                target: 55,
                container: 55,
                action: Generic(GenericAction {
                    value: I32(-1820655817),
                    bool: true,
                    key: 2483000211,
                    pos: 18387915803576970899,
                    length: 1297036689260150784,
                    prop: 8897841259086306727,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 253,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 1553399936,
                    pos: 18390449078367420160,
                    length: 36201467589165055,
                    prop: 18374967954631580824,
                }),
            },
            SyncAll,
            Handle {
                site: 43,
                target: 43,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2071690107,
                    pos: 18446744071486274427,
                    length: 17216960130726232104,
                    prop: 7430094960254740462,
                }),
            },
            SyncAll,
            Handle {
                site: 43,
                target: 43,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2726001531,
                    pos: 18446744071486274427,
                    length: 17216960130726232104,
                    prop: 7430094960254740462,
                }),
            },
            Checkout {
                site: 0,
                to: 6094847,
            },
            Sync { from: 139, to: 139 },
            Sync { from: 139, to: 139 },
            Handle {
                site: 139,
                target: 43,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2071690107,
                    pos: 18446744071486274427,
                    length: 17178700329823240232,
                    prop: 7430094960254715374,
                }),
            },
            Checkout {
                site: 0,
                to: 6094847,
            },
            Sync { from: 139, to: 139 },
            Sync { from: 55, to: 55 },
            Sync { from: 155, to: 155 },
            Sync { from: 155, to: 155 },
            Sync { from: 155, to: 55 },
            Handle {
                site: 55,
                target: 55,
                container: 55,
                action: Generic(GenericAction {
                    value: I32(151587327),
                    bool: true,
                    key: 4294904073,
                    pos: 143835908481745196,
                    length: 11691225419712927791,
                    prop: 10638384274776138274,
                }),
            },
            Sync { from: 147, to: 20 },
            Handle {
                site: 34,
                target: 34,
                container: 34,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: false,
                    key: 3749681,
                    pos: 18389705808507043840,
                    length: 245750159833,
                    prop: 15697858579274924032,
                }),
            },
            SyncAll,
        ],
    )
}

#[test]
fn test_unknown() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Tree,
            FuzzTarget::Text,
        ],
        &mut [
            Handle {
                site: 34,
                target: 115,
                container: 4,
                action: Generic(GenericAction {
                    value: I32(62063364),
                    bool: false,
                    key: 771987715,
                    pos: 217020518514230019,
                    length: 217234923281646339,
                    prop: 6234107865851074949,
                }),
            },
            Handle {
                site: 3,
                target: 3,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_movable_list_25() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Map, FuzzTarget::MovableList],
        &mut [
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4294967295,
                    pos: 8791026472627208191,
                    length: 18446744073692774400,
                    prop: 18157383382357508095,
                }),
            },
            SyncAll,
            Checkout {
                site: 126,
                to: 285217792,
            },
            Handle {
                site: 0,
                target: 238,
                container: 96,
                action: Generic(GenericAction {
                    value: I32(-256),
                    bool: true,
                    key: 2634743807,
                    pos: 34084,
                    length: 18446656643131973376,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            Handle {
                site: 43,
                target: 93,
                container: 246,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: true,
                    key: 4294967295,
                    pos: 18446744073709551615,
                    length: 2133084,
                    prop: 18446744073709486080,
                }),
            },
            Handle {
                site: 0,
                target: 0,
                container: 47,
                action: Generic(GenericAction {
                    value: I32(402652951),
                    bool: true,
                    key: 387389207,
                    pos: 1663900941173922844,
                    length: 1663823975275763479,
                    prop: 2738087021684922135,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 255,
                target: 255,
                container: 255,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: true,
                    key: 4294967295,
                    pos: 4899916394579099647,
                    length: 1519401924284121087,
                    prop: 18446744069768287677,
                }),
            },
            SyncAll,
            Handle {
                site: 3,
                target: 3,
                container: 3,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 50398464,
                    pos: 216172795049214723,
                    length: 18427888561740251907,
                    prop: 18442733055291424767,
                }),
            },
            Handle {
                site: 23,
                target: 23,
                container: 23,
                action: Generic(GenericAction {
                    value: I32(1124602995),
                    bool: false,
                    key: 4280877827,
                    pos: 18446744073709551615,
                    length: 18446744073709551615,
                    prop: 18446744073709551615,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 162,
                target: 3,
                container: 3,
                action: Generic(GenericAction {
                    value: I32(-34201),
                    bool: true,
                    key: 84215045,
                    pos: 11791549724362736901,
                    length: 1513209197156392029,
                    prop: 11817445422215987071,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 16,
                target: 51,
                container: 52,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294967056,
                    pos: 13258596207762079743,
                    length: 216172782113733631,
                    prop: 18428093872899751680,
                }),
            },
            SyncAll,
            Handle {
                site: 9,
                target: 23,
                container: 23,
                action: Generic(GenericAction {
                    value: I32(1340183),
                    bool: true,
                    key: 2053581568,
                    pos: 18446509877732900863,
                    length: 4836866087641415679,
                    prop: 1152922604118419456,
                }),
            },
            SyncAll,
            SyncAll,
            SyncAll,
            Handle {
                site: 247,
                target: 255,
                container: 0,
                action: Generic(GenericAction {
                    value: Container(Tree),
                    bool: false,
                    key: 521,
                    pos: 0,
                    length: 0,
                    prop: 18446499166067621888,
                }),
            },
            Handle {
                site: 47,
                target: 47,
                container: 47,
                action: Generic(GenericAction {
                    value: I32(3631),
                    bool: false,
                    key: 0,
                    pos: 15646670743035716644,
                    length: 606348324,
                    prop: 3832285106451644416,
                }),
            },
            Checkout {
                site: 255,
                to: 1946103807,
            },
            Handle {
                site: 247,
                target: 255,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(-193),
                    bool: true,
                    key: 767,
                    pos: 18446742974197924864,
                    length: 1620130366750719,
                    prop: 3605423971803043822,
                }),
            },
            Handle {
                site: 9,
                target: 9,
                container: 1,
                action: Generic(GenericAction {
                    value: Container(MovableList),
                    bool: true,
                    key: 56823,
                    pos: 18446744073709551399,
                    length: 71777218572845055,
                    prop: 6947083900195729408,
                }),
            },
            Handle {
                site: 27,
                target: 27,
                container: 27,
                action: Generic(GenericAction {
                    value: Container(List),
                    bool: true,
                    key: 2442236305,
                    pos: 10489325061521117585,
                    length: 10489325061521117585,
                    prop: 10489325061521117585,
                }),
            },
            Sync { from: 145, to: 145 },
            Sync { from: 145, to: 145 },
            Sync { from: 145, to: 145 },
            Sync { from: 145, to: 145 },
            Sync { from: 145, to: 145 },
            Sync { from: 145, to: 145 },
            Sync { from: 145, to: 145 },
            Handle {
                site: 27,
                target: 27,
                container: 27,
                action: Generic(GenericAction {
                    value: I32(454761243),
                    bool: true,
                    key: 454761243,
                    pos: 1953184666628070171,
                    length: 143834851265485595,
                    prop: 18446744073692890520,
                }),
            },
            Handle {
                site: 162,
                target: 3,
                container: 3,
                action: Generic(GenericAction {
                    value: I32(-34201),
                    bool: true,
                    key: 84215045,
                    pos: 11791549724362736901,
                    length: 1513209197156392029,
                    prop: 11817445422215987071,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 16,
                target: 51,
                container: 52,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 4294967056,
                    pos: 13258596207762079743,
                    length: 216172782113733631,
                    prop: 18428093872899751680,
                }),
            },
            SyncAll,
            Handle {
                site: 9,
                target: 23,
                container: 23,
                action: Generic(GenericAction {
                    value: I32(387389207),
                    bool: true,
                    key: 4390932,
                    pos: 8346507721238538027,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn test_tree_delete_nested() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 49,
                target: 53,
                container: 57,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370361,
                    length: 4123390026907529521,
                    prop: 4123389851767159097,
                }),
            },
            Handle {
                site: 8,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(968243513),
                    bool: true,
                    key: 960051513,
                    pos: 18446529914715322681,
                    length: 18432897712502263502,
                    prop: 4123389851781761081,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(-1),
                    bool: true,
                    key: 4294967295,
                    pos: 4179058995052806143,
                    length: 4123389851770370361,
                    prop: 4123389851770370361,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: true,
                    key: 3472883517,
                    pos: 18446744072884244174,
                    length: 281474976710655,
                    prop: 18446496683593302016,
                }),
            },
            Handle {
                site: 35,
                target: 35,
                container: 35,
                action: Generic(GenericAction {
                    value: I32(589505315),
                    bool: true,
                    key: 589505315,
                    pos: 2531906049332683555,
                    length: 2531906049332683555,
                    prop: 2531906049332683555,
                }),
            },
            SyncAll,
            Handle {
                site: 61,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 3684665,
                    pos: 4109597577911548258,
                    length: 13058531512473434622,
                    prop: 4123390386434848181,
                }),
            },
            Handle {
                site: 61,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 3684665,
                    pos: 592567745925167458,
                    length: 3904956473749929529,
                    prop: 592567743911911936,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960075065),
                    bool: true,
                    key: 960051513,
                    pos: 4123389851770370486,
                    length: 14915921129292511545,
                    prop: 18446689986361085646,
                }),
            },
            Handle {
                site: 61,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(960051513),
                    bool: true,
                    key: 3684665,
                    pos: 4109597577911548258,
                    length: 15825516631767550,
                    prop: 16647618940511009122,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(-1254540999),
                    bool: true,
                    key: 3048584629,
                    pos: 4123389851770370361,
                    length: 4123389851770370361,
                    prop: 13093570749027399993,
                }),
            },
            Handle {
                site: 57,
                target: 57,
                container: 57,
                action: Generic(GenericAction {
                    value: Container(Map),
                    bool: true,
                    key: 960051513,
                    pos: 4123389850948286777,
                    length: 4123389851770370356,
                    prop: 18446743862421043513,
                }),
            },
        ],
    )
}

#[test]
fn test_text() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 255,
                target: 15,
                container: 57,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 805292845,
                    pos: 33041,
                    length: 0,
                    prop: 1238489669910396928,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 805292845,
                    pos: 3255307780432560401,
                    length: 18446743168229387565,
                    prop: 3255307777713581326,
                }),
            },
            Handle {
                site: 45,
                target: 45,
                container: 45,
                action: Generic(GenericAction {
                    value: I32(757935405),
                    bool: true,
                    key: 4291505453,
                    pos: 18388247646700638511,
                    length: 18446744073709507839,
                    prop: 5570344,
                }),
            },
        ],
    )
}

#[test]
fn test_text_del_2() {
    test_multi_sites(
        5,
        vec![
            FuzzTarget::Map,
            FuzzTarget::List,
            FuzzTarget::Text,
            FuzzTarget::Tree,
            FuzzTarget::MovableList,
        ],
        &mut [
            Handle {
                site: 42,
                target: 45,
                container: 253,
                action: Generic(GenericAction {
                    value: I32(33554179),
                    bool: false,
                    key: 15616,
                    pos: 1339615555336169111,
                    length: 10909519737336631312,
                    prop: 10923365712002484737,
                }),
            },
            Sync { from: 151, to: 151 },
            Handle {
                site: 191,
                target: 0,
                container: 2,
                action: Generic(GenericAction {
                    value: I32(-1088190176),
                    bool: false,
                    key: 1898119453,
                    pos: 114672903794094449,
                    length: 2593958586217895690,
                    prop: 16131857654658175249,
                }),
            },
            Handle {
                site: 17,
                target: 17,
                container: 17,
                action: Generic(GenericAction {
                    value: I32(286331153),
                    bool: true,
                    key: 286331153,
                    pos: 1229782938247310353,
                    length: 1229782938247303441,
                    prop: 1229782938247303441,
                }),
            },
            Handle {
                site: 21,
                target: 17,
                container: 17,
                action: Generic(GenericAction {
                    value: I32(33554176),
                    bool: false,
                    key: 0,
                    pos: 1536,
                    length: 0,
                    prop: 1229782938247303424,
                }),
            },
            SyncAll,
            SyncAll,
            Handle {
                site: 17,
                target: 17,
                container: 17,
                action: Generic(GenericAction {
                    value: I32(286331153),
                    bool: true,
                    key: 0,
                    pos: 1229782864946528256,
                    length: 12080808152476417826,
                    prop: 10923366098543524643,
                }),
            },
            Handle {
                site: 35,
                target: 38,
                container: 35,
                action: Generic(GenericAction {
                    value: I32(587333693),
                    bool: false,
                    key: 2543294434,
                    pos: 4263285121861231497,
                    length: 59,
                    prop: 1518013315106421504,
                }),
            },
            Sync { from: 167, to: 35 },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 49,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: false,
                    key: 0,
                    pos: 3602878813425696768,
                    length: 18446743185255244799,
                    prop: 8152360975528560127,
                }),
            },
            SyncAll,
            Handle {
                site: 80,
                target: 0,
                container: 61,
                action: Generic(GenericAction {
                    value: I32(1161822054),
                    bool: true,
                    key: 269488146,
                    pos: 10883513199263901286,
                    length: 10923366098549554583,
                    prop: 2748041329745827735,
                }),
            },
            Handle {
                site: 61,
                target: 0,
                container: 2,
                action: Generic(GenericAction {
                    value: Container(Text),
                    bool: true,
                    key: 160113,
                    pos: 4438182027029079654,
                    length: 1229782938564925335,
                    prop: 1229785140740218641,
                }),
            },
            Handle {
                site: 63,
                target: 17,
                container: 17,
                action: Generic(GenericAction {
                    value: I32(286331153),
                    bool: true,
                    key: 286338065,
                    pos: 1229782938247303441,
                    length: 1229782938247303441,
                    prop: 1229782938247303441,
                }),
            },
            Handle {
                site: 17,
                target: 0,
                container: 0,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 1536,
                    pos: 0,
                    length: 1229782864946528256,
                    prop: 1229782938247303441,
                }),
            },
            SyncAll,
            Handle {
                site: 17,
                target: 17,
                container: 17,
                action: Generic(GenericAction {
                    value: I32(286331153),
                    bool: true,
                    key: 17,
                    pos: 1229764173248856064,
                    length: 12080626724467843601,
                    prop: 10923366097000014759,
                }),
            },
            Handle {
                site: 1,
                target: 35,
                container: 38,
                action: Generic(GenericAction {
                    value: I32(33570048),
                    bool: true,
                    key: 2543313634,
                    pos: 3043090847611718039,
                    length: 15163,
                    prop: 1229783119343321088,
                }),
            },
            Sync { from: 167, to: 167 },
            Handle {
                site: 1,
                target: 191,
                container: 35,
                action: Generic(GenericAction {
                    value: I32(0),
                    bool: false,
                    key: 0,
                    pos: 0,
                    length: 0,
                    prop: 0,
                }),
            },
        ],
    )
}

#[test]
fn unknown_container() {
    let doc = loro_without_counter::LoroDoc::new();
    let list = doc.get_list("list");
    doc.subscribe(
        &list.id(),
        Arc::new(|e| {
            assert_eq!(e.events.len(), 2);
            assert!(e.events[1].is_unknown)
        }),
    );

    let doc2 = LoroDoc::new();
    let list2 = doc2.get_list("list");
    let counter = list2.insert_container(0, LoroCounter::new()).unwrap();
    counter.increment(2).unwrap();

    doc.import(&doc2.export_snapshot()).unwrap();
}

#[test]
fn undo_tree() {
    test_multi_sites(
        5,
        vec![FuzzTarget::Tree],
        &mut [
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: ActionWrapper::Action(fuzz::actions::ActionInner::Tree(TreeAction {
                    target: (0, 0),
                    action: TreeActionInner::Create { index: 0 },
                })),
            },
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: ActionWrapper::Action(fuzz::actions::ActionInner::Tree(TreeAction {
                    target: (0, 1),
                    action: TreeActionInner::Create { index: 1 },
                })),
            },
            SyncAll,
            Handle {
                site: 0,
                target: 0,
                container: 0,
                action: ActionWrapper::Action(fuzz::actions::ActionInner::Tree(TreeAction {
                    target: (0, 0),
                    action: TreeActionInner::Move {
                        parent: (0, 1),
                        index: 0,
                    },
                })),
            },
            Handle {
                site: 1,
                target: 0,
                container: 0,
                action: ActionWrapper::Action(fuzz::actions::ActionInner::Tree(TreeAction {
                    target: (0, 1),
                    action: TreeActionInner::Move {
                        parent: (0, 0),
                        index: 0,
                    },
                })),
            },
            SyncAllUndo { site: 0, op_len: 1 },
        ],
    )
}
