/*
 * Copyright (c) 2019. Abstrium SAS <team (at) pydio.com>
 * This file is part of Pydio Cells.
 *
 * Pydio Cells is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Pydio Cells is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Pydio Cells.  If not, see <http://www.gnu.org/licenses/>.
 *
 * The latest code can be found at <https://pydio.com>.
 */

// Package docs provides tools for generating user docs from Cobra commands in various formats
package docs

import (
	"bytes"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"sort"
	"strings"
	"time"

	"github.com/gosimple/slug"
	"github.com/spf13/cobra"
)

var (
	yamlPosition         = 0
	PydioDocsMenuName    = "menu-admin-guide-v7"
	PydioDocsGeneratedBy = ""
)

func printOptions(buf *bytes.Buffer, cmd *cobra.Command, name string) error {
	flags := cmd.NonInheritedFlags()
	flags.SetOutput(buf)
	if flags.HasFlags() {
		buf.WriteString("### Options\n\n```\n")
		flags.PrintDefaults()
		buf.WriteString("```\n\n")
	}

	parentFlags := cmd.InheritedFlags()
	parentFlags.SetOutput(buf)
	if parentFlags.HasFlags() {
		buf.WriteString("### Options inherited from parent commands\n\n```\n")
		parentFlags.PrintDefaults()
		buf.WriteString("```\n\n")
	}
	return nil
}

// GenMarkdown creates markdown output.
func GenMarkdown(cmd *cobra.Command, w io.Writer) error {
	return GenMarkdownCustom(cmd, w, 1, func(s string) string { return s })
}

// GenPydioYaml generates yaml file for pydio docs
// Sample output
// title: "Command Name"
// language: und
// menu: "Command Name"
// weight: 1
// menu_name: menu-administration-guide
func GenPydioYaml(cmd *cobra.Command, position int, w io.Writer) error {

	name := filepath.Base(cmd.CommandPath())
	buf := new(bytes.Buffer)
	buf.WriteString("title: \"" + name + "\"\n")
	buf.WriteString("menu: \"" + name + "\"\n")
	buf.WriteString("language: und\n")
	buf.WriteString("menu_name: " + PydioDocsMenuName + "\n")
	buf.WriteString(fmt.Sprintf("weight: %d\n", position))
	buf.WriteString(fmt.Sprintf("slug: %s\n", strings.ReplaceAll(slug.Make(name), "_", "-")))

	_, err := buf.WriteTo(w)
	return err
}

func GenYamlFrontMatter(title string, position int, buf *bytes.Buffer) {
	buf.WriteString("---\n")
	buf.WriteString("title: \"" + title + "\"\n")
	buf.WriteString("language: und\n")
	buf.WriteString(fmt.Sprintf("weight: %d\n", position))
	buf.WriteString(fmt.Sprintf("slug: %s\n", strings.ReplaceAll(slug.Make(title), "_", "-")))

	// Add tags
	// tags := titleToTags(title)
	// if len(tags) >= 2 {
	// 	buf.WriteString("tags:\n")
	// 	for _, t := range tags[1:] {
	// 		buf.WriteString(fmt.Sprintf("\t- %s\n", t))
	// 	}
	// }

	buf.WriteString("---\n\n")
}

// GenMarkdownCustom creates custom markdown output.
func GenMarkdownCustom(cmd *cobra.Command, w io.Writer, position int, linkHandler func(string) string) error {

	cmd.InitDefaultHelpCmd()
	cmd.InitDefaultHelpFlag()

	buf := new(bytes.Buffer)
	name := cmd.CommandPath()
	short := cmd.Short
	long := cmd.Long
	if len(long) == 0 {
		long = short
	}

	// Add frontmatter
	GenYamlFrontMatter(filepath.Base(name), position, buf)

	// buf.WriteString("## " + name + "\n\n")
	buf.WriteString(short + "\n\n")
	buf.WriteString("### Synopsis\n\n")
	buf.WriteString(long + "\n\n")

	if cmd.Runnable() {
		buf.WriteString(fmt.Sprintf("```\n%s\n```\n\n", cmd.UseLine()))
	}

	if len(cmd.Example) > 0 {
		buf.WriteString("### Examples\n\n")
		buf.WriteString(fmt.Sprintf("```\n%s\n```\n\n", cmd.Example))
	}

	if err := printOptions(buf, cmd, name); err != nil {
		return err
	}
	if cmdHasSeeAlso(cmd) {
		buf.WriteString("### SEE ALSO\n\n")
		if cmd.HasParent() {
			parent := cmd.Parent()
			pname := parent.CommandPath()
			link := pname + ".md"
			link = strings.Replace(link, " ", "_", -1)
			buf.WriteString(fmt.Sprintf("* [%s](%s)\t - %s\n", pname, linkHandler(link), parent.Short))
			cmd.VisitParents(func(c *cobra.Command) {
				if c.DisableAutoGenTag {
					cmd.DisableAutoGenTag = c.DisableAutoGenTag
				}
			})
		}

		children := cmd.Commands()
		sort.Sort(cmdByName(children))

		for _, child := range children {
			if !child.IsAvailableCommand() || child.IsAdditionalHelpTopicCommand() {
				continue
			}
			cname := name + " " + child.Name()
			link := cname + ".md"
			link = strings.Replace(link, " ", "_", -1)
			buf.WriteString(fmt.Sprintf("* [%s](%s)\t - %s\n", cname, linkHandler(link), child.Short))
		}
		buf.WriteString("\n")
	}
	if !cmd.DisableAutoGenTag && PydioDocsGeneratedBy != "" {
		buf.WriteString("###### Auto generated by " + PydioDocsGeneratedBy + " on " + time.Now().Format("2-Jan-2006") + "\n")
	}
	_, err := buf.WriteTo(w)
	return err
}

// GenMarkdownTree will generate a markdown page for this command and all
// descendants in the directory given. The header may be nil.
// This function may not work correctly if your command names have `-` in them.
// If you have `cmd` with two subcmds, `sub` and `sub-third`,
// and `sub` has a subcommand called `third`, it is undefined which
// help output will be in the file `cmd-sub-third.1`.
func GenMarkdownTree(cmd *cobra.Command, dir string) error {
	linkHandler := func(s string) string {
		s = strings.Replace(s, "_", "-", -1)
		s = strings.TrimSuffix(s, ".md")
		return s
	}
	emptyStr := func(s string) string { return "" }

	return GenMarkdownTreeCustom(cmd, dir, emptyStr, linkHandler)
}

// GenMarkdownTreeCustom is the the same as GenMarkdownTree, but
// with custom filePrepender and linkHandler.
func GenMarkdownTreeCustom(cmd *cobra.Command, dir string, filePrepender, linkHandler func(string) string) error {
	yamlPosition++
	cmdPath := cmd.CommandPath()
	fileName := filepath.Base(cmdPath)
	basename := strings.Replace(fileName, " ", "_", -1) + ".md"
	filename := filepath.Join(dir, basename)
	f, err := os.Create(filename)
	if err != nil {
		return err
	}
	defer f.Close()

	if _, err := io.WriteString(f, filePrepender(filename)); err != nil {
		return err
	}
	if err := GenMarkdownCustom(cmd, f, yamlPosition, linkHandler); err != nil {
		return err
	}

	// yamlname := strings.Replace(cmd.CommandPath(), " ", "_", -1) + ".yaml"
	// yamlfile := filepath.Join(dir, yamlname)
	// f1, err := os.Create(yamlfile)
	// if err != nil {
	// 	return err
	// }
	// defer f1.Close()

	// if _, err := io.WriteString(f1, filePrepender(yamlfile)); err != nil {
	// 	return err
	// }
	// if err := GenPydioYaml(cmd, yamlPosition, f1); err != nil {
	// 	return err
	// }

	for _, c := range cmd.Commands() {
		if !c.IsAvailableCommand() || c.IsAdditionalHelpTopicCommand() {
			continue
		}
		if err := GenMarkdownTreeCustom(c, dir, filePrepender, linkHandler); err != nil {
			return err
		}
	}

	return nil
}

// Test to see if we have a reason to print. See Also information in docs
// Basically this is a test for a parent command or a subcommand which is
// both not deprecated and not the autogenerated help command.
func cmdHasSeeAlso(cmd *cobra.Command) bool {
	return false
}

func titleToTags(input string) []string {
	// Trim leading and trailing whitespaces
	input = strings.TrimSpace(input)

	// Convert to lowercase
	input = strings.ToLower(input)

	// Split by whitespace into words
	words := strings.Fields(input)

	return words
}

type cmdByName []*cobra.Command

func (s cmdByName) Len() int           { return len(s) }
func (s cmdByName) Swap(i, j int)      { s[i], s[j] = s[j], s[i] }
func (s cmdByName) Less(i, j int) bool { return s[i].Name() < s[j].Name() }
