import { fileURLToPath } from 'node:url'
import path from 'node:path'
import os from 'node:os'
import { app, dialog } from 'electron'
import { shellEnv } from 'shell-env'
import launch from './launch-editor'

const __dirname = path.dirname(fileURLToPath(import.meta.url))

// The built directory structure
//
// ├─┬ dist-electron
// │ ├─┬ main
// │ │ └── index.js    > Electron-Main
// │ └─┬ preload
// │   └── index.mjs   > Preload-Scripts
// ├─┬ dist
// │ └── index.html    > Electron-Renderer
//
process.env.APP_ROOT = path.join(__dirname, '../..')

export const MAIN_DIST = path.join(process.env.APP_ROOT, 'dist-electron')
export const RENDERER_DIST = path.join(process.env.APP_ROOT, 'dist')
export const VITE_DEV_SERVER_URL = process.env.VITE_DEV_SERVER_URL

process.env.VITE_PUBLIC = VITE_DEV_SERVER_URL
  ? path.join(process.env.APP_ROOT, 'public')
  : RENDERER_DIST

if (!app.isDefaultProtocolClient('localfile')) {
  if (process.defaultApp) {
    if (process.argv.length >= 2)
      app.setAsDefaultProtocolClient('localfile', process.execPath, [path.resolve(process.argv[1])])
  }
  else {
    app.setAsDefaultProtocolClient('localfile')
  }
  app.whenReady().then(() => {
    dialog.showMessageBox({
      title: 'localfile.link',
      message: 'Protocol handler registered, now you can use localfile:// to open files from browser.',
    })
  })
}

// Disable GPU Acceleration for Windows 7
if (os.release().startsWith('6.1'))
  app.disableHardwareAcceleration()

// Set application name for Windows 10+ notifications
if (process.platform === 'win32')
  app.setAppUserModelId(app.getName())

if (!app.requestSingleInstanceLock()) {
  app.quit()
  process.exit(0)
}

let isRunning = false

// let win: BrowserWindow | null = null
// const preload = path.join(__dirname, '../preload/index.mjs')
// const indexHtml = path.join(RENDERER_DIST, 'index.html')

const reFile = /^file:\/\//
const reLocalfile = /^localfile:\/\//

async function run(file: string) {
  isRunning = true
  // Read env from shell
  Object.assign(process.env, await shellEnv())
  file = file.replace(reFile, '')
  try {
    await launch(file, 'code')
    app.quit()
    process.exit(0)
  }
  catch (error) {
    dialog.showErrorBox('Failed to open file', [
      error.message,
      `File path: ${file}`,
    ].join('\n'))
    app.quit()
    process.exit(1)
  }
}

app.on('will-finish-launching', () => {
  // Protocol handler for osx
  app.on('open-url', async (event, url) => {
    event.preventDefault()
    const file = url.replace(reLocalfile, '')
    await run(file)
  })
})

app.on('second-instance', async (event, commandLine, _workingDirectory) => {
  const file = commandLine.pop().replace(reLocalfile, '')
  await run(file)
})

// async function createWindow() {
//   win = new BrowserWindow({
//     title: 'Main window',
//     icon: path.join(process.env.VITE_PUBLIC, 'favicon.ico'),
//     webPreferences: {
//       preload,
//       // Warning: Enable nodeIntegration and disable contextIsolation is not secure in production
//       // nodeIntegration: true,

//       // Consider using contextBridge.exposeInMainWorld
//       // Read more on https://www.electronjs.org/docs/latest/tutorial/context-isolation
//       // contextIsolation: false,
//     },
//   })

//   if (VITE_DEV_SERVER_URL) { // #298
//     win.loadURL(VITE_DEV_SERVER_URL)
//     // Open devTool if the app is not packaged
//     win.webContents.openDevTools()
//   } else {
//     win.loadFile(indexHtml)
//   }

//   // Test actively push message to the Electron-Renderer
//   win.webContents.on('did-finish-load', () => {
//     win?.webContents.send('main-process-message', new Date().toLocaleString())
//   })

//   // Make all links open with the browser, not with the application
//   win.webContents.setWindowOpenHandler(({ url }) => {
//     if (url.startsWith('https:')) shell.openExternal(url)
//     return { action: 'deny' }
//   })
//   // win.webContents.on('will-navigate', (event, url) => { }) #344
// }

app.whenReady()
  .then(() => {
    setTimeout(() => {
      if (!isRunning)
        app.quit()
    }, 500)
  })

// app.on('window-all-closed', () => {
//   win = null
//   if (process.platform !== 'darwin') app.quit()
// })

// app.on('second-instance', () => {
//   if (win) {
//     // Focus on the main window if the user tried to open another
//     if (win.isMinimized()) win.restore()
//     win.focus()
//   }
// })

// app.on('activate', () => {
//   const allWindows = BrowserWindow.getAllWindows()
//   if (allWindows.length) {
//     allWindows[0].focus()
//   } else {
//     createWindow()
//   }
// })

// // New window example arg: new windows url
// ipcMain.handle('open-win', (_, arg) => {
//   const childWindow = new BrowserWindow({
//     webPreferences: {
//       preload,
//       nodeIntegration: true,
//       contextIsolation: false,
//     },
//   })

//   if (VITE_DEV_SERVER_URL) {
//     childWindow.loadURL(`${VITE_DEV_SERVER_URL}#${arg}`)
//   } else {
//     childWindow.loadFile(indexHtml, { hash: arg })
//   }
// })
