"use client";

import Image from "next/image";
import * as React from "react";

import { cn } from "@/lib/utils";

function IconOpenAI({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      fill="currentColor"
      viewBox="0 0 24 24"
      role="img"
      xmlns="http://www.w3.org/2000/svg"
      className={cn("h-4 w-4", className)}
      {...props}
    >
      <title>OpenAI icon</title>
      <path d="M22.2819 9.8211a5.9847 5.9847 0 0 0-.5157-4.9108 6.0462 6.0462 0 0 0-6.5098-2.9A6.0651 6.0651 0 0 0 4.9807 4.1818a5.9847 5.9847 0 0 0-3.9977 2.9 6.0462 6.0462 0 0 0 .7427 7.0966 5.98 5.98 0 0 0 .511 4.9107 6.051 6.051 0 0 0 6.5146 2.9001A5.9847 5.9847 0 0 0 13.2599 24a6.0557 6.0557 0 0 0 5.7718-4.2058 5.9894 5.9894 0 0 0 3.9977-2.9001 6.0557 6.0557 0 0 0-.7475-7.0729zm-9.022 12.6081a4.4755 4.4755 0 0 1-2.8764-1.0408l.1419-.0804 4.7783-2.7582a.7948.7948 0 0 0 .3927-.6813v-6.7369l2.02 1.1686a.071.071 0 0 1 .038.052v5.5826a4.504 4.504 0 0 1-4.4945 4.4944zm-9.6607-4.1254a4.4708 4.4708 0 0 1-.5346-3.0137l.142.0852 4.783 2.7582a.7712.7712 0 0 0 .7806 0l5.8428-3.3685v2.3324a.0804.0804 0 0 1-.0332.0615L9.74 19.9502a4.4992 4.4992 0 0 1-6.1408-1.6464zM2.3408 7.8956a4.485 4.485 0 0 1 2.3655-1.9728V11.6a.7664.7664 0 0 0 .3879.6765l5.8144 3.3543-2.0201 1.1685a.0757.0757 0 0 1-.071 0l-4.8303-2.7865A4.504 4.504 0 0 1 2.3408 7.872zm16.5963 3.8558L13.1038 8.364 15.1192 7.2a.0757.0757 0 0 1 .071 0l4.8303 2.7913a4.4944 4.4944 0 0 1-.6765 8.1042v-5.6772a.79.79 0 0 0-.407-.667zm2.0107-3.0231l-.142-.0852-4.7735-2.7818a.7759.7759 0 0 0-.7854 0L9.409 9.2297V6.8974a.0662.0662 0 0 1 .0284-.0615l4.8303-2.7866a4.4992 4.4992 0 0 1 6.6802 4.66zM8.3065 12.863l-2.02-1.1638a.0804.0804 0 0 1-.038-.0567V6.0742a4.4992 4.4992 0 0 1 7.3757-3.4537l-.142.0805L8.704 5.459a.7948.7948 0 0 0-.3927.6813zm1.0976-2.3654l2.602-1.4998 2.6069 1.4998v2.9994l-2.5974 1.4997-2.6067-1.4997Z" />
    </svg>
  );
}

function IconLangchain({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg fill="currentColor" viewBox="0 0 80 41" className={cn("", className)} {...props}>
      <path d="M61.5139 11.1569C60.4527 11.1569 59.4549 11.568 58.708 12.3148L55.6899 15.3248C54.8757 16.1368 54.4574 17.2643 54.5431 18.4202C54.5492 18.4833 54.5553 18.5464 54.5615 18.6115C54.6696 19.4988 55.0594 20.2986 55.6899 20.9254C56.1246 21.3589 56.6041 21.6337 57.1857 21.825C57.2163 22 57.2326 22.177 57.2326 22.3541C57.2326 23.1519 56.9225 23.9008 56.3592 24.4625L56.1735 24.6477C55.1655 24.3037 54.3247 23.8011 53.5656 23.044C52.5576 22.0386 51.8903 20.7687 51.6393 19.3747L51.6046 19.1813L51.4515 19.3055C51.3475 19.3889 51.2495 19.4785 51.1577 19.57L48.1396 22.58C46.5928 24.1226 46.5928 26.636 48.1396 28.1786C48.913 28.9499 49.9292 29.3366 50.9475 29.3366C51.9658 29.3366 52.98 28.9499 53.7534 28.1786L56.7715 25.1687C58.3183 23.626 58.3183 21.1147 56.7715 19.57C56.3592 19.159 55.8675 18.8496 55.3104 18.6502C55.2798 18.469 55.2634 18.2879 55.2634 18.1109C55.2634 17.2439 55.6063 16.4217 56.2348 15.7949C57.2449 16.1388 58.1407 16.6965 58.8978 17.4515C59.9038 18.4548 60.5691 19.7227 60.8241 21.1208L60.8588 21.3141L61.0119 21.19C61.116 21.1066 61.2139 21.017 61.3078 20.9234L64.3259 17.9135C65.8727 16.3708 65.8747 13.8575 64.3259 12.3148C63.577 11.568 62.5811 11.1569 61.518 11.1569H61.5139Z" />
      <path d="M59.8966 0.148865H20.4063C9.15426 0.148865 0 9.27841 0 20.5001C0 31.7217 9.15426 40.8513 20.4063 40.8513H59.8966C71.1486 40.8513 80.3029 31.7217 80.3029 20.5001C80.3029 9.27841 71.1486 0.148865 59.8966 0.148865ZM40.4188 32.0555C39.7678 32.1898 39.0352 32.2142 38.5373 31.6953C38.3536 32.1165 37.9251 31.8947 37.5945 31.8398C37.5639 31.9252 37.5374 32.0005 37.5088 32.086C36.4089 32.1593 35.5845 31.04 35.0601 30.1954C34.0193 29.6337 32.8378 29.2918 31.7746 28.7036C31.7134 29.6724 31.9257 30.8731 31.0012 31.4979C30.9543 33.36 33.8255 31.7177 34.0887 33.1056C33.8847 33.128 33.6582 33.073 33.4949 33.2297C32.746 33.9563 31.8869 32.6803 31.0237 33.2074C29.8646 33.7894 29.7483 34.2656 28.3137 34.3857C28.2342 34.2656 28.2668 34.1862 28.3341 34.113C28.7382 33.6449 28.7668 33.0934 29.4565 32.8939C28.7464 32.782 28.1525 33.1728 27.5546 33.4821C26.7771 33.7996 26.7833 32.7657 25.5875 33.537C25.4548 33.4292 25.5181 33.3315 25.5936 33.2481C25.8976 32.8777 26.2976 32.8227 26.7486 32.8431C24.5304 31.6098 23.4856 34.3511 22.4612 32.9876C22.1531 33.069 22.0368 33.3457 21.8429 33.5411C21.6756 33.358 21.8021 33.1361 21.8103 32.9204C21.6103 32.8268 21.3572 32.782 21.4164 32.4625C21.0246 32.3302 20.7512 32.5622 20.4594 32.782C20.1961 32.5785 20.6369 32.2814 20.7185 32.0697C20.9532 31.6627 21.4878 31.9863 21.7592 31.6932C22.5306 31.2557 23.606 31.9659 24.4876 31.8459C25.1671 31.9313 26.0078 31.2353 25.667 30.5413C24.9406 29.6154 25.0691 28.4045 25.0528 27.2974C24.963 26.6522 23.4101 25.83 22.9612 25.134C22.4061 24.5072 21.9735 23.7807 21.5409 23.0664C19.9798 20.0523 20.4716 16.1795 18.5044 13.3812C17.6147 13.8717 16.4556 13.6397 15.6884 12.9823C15.2741 13.3588 15.2557 13.8513 15.2231 14.3744C14.2293 13.3833 14.3538 11.5109 15.1476 10.4079C15.4721 9.97239 15.8598 9.61421 16.2924 9.29876C16.3903 9.22754 16.423 9.15834 16.4209 9.04844C17.2066 5.52362 22.5653 6.20335 24.259 8.70044C25.4875 10.237 25.8589 12.27 27.2526 13.6967C29.1279 15.744 31.2645 17.5471 32.9949 19.7267C34.6315 21.7191 35.8008 24.0554 36.8211 26.4101C37.2374 27.1915 37.2415 28.1501 37.8578 28.8176C38.1618 29.2206 39.6474 30.3175 39.325 30.7062C39.5107 31.1091 40.8983 31.6647 40.4167 32.0555H40.4188ZM66.4449 20.032L63.4269 23.0419C62.6228 23.8438 61.6291 24.4421 60.5516 24.7697L60.4965 24.786L60.4761 24.8389C60.1251 25.7629 59.5925 26.5871 58.8905 27.2852L55.8724 30.2951C54.5562 31.6078 52.8054 32.3302 50.9402 32.3302C49.0751 32.3302 47.3242 31.6078 46.008 30.2951C43.2879 27.5823 43.2879 23.1701 46.008 20.4573L49.0261 17.4474C49.8342 16.6415 50.8015 16.0615 51.8973 15.7257L51.9524 15.7094L51.9728 15.6565C52.3238 14.7325 52.8584 13.9063 53.5625 13.2021L56.5805 10.1922C57.8967 8.87953 59.6476 8.15706 61.5127 8.15706C63.3779 8.15706 65.1287 8.87953 66.4449 10.1922C67.7612 11.5048 68.4856 13.251 68.4856 15.1111C68.4856 16.9712 67.7612 18.7193 66.4449 20.03V20.032Z" />
    </svg>
  );
}

function IconCrewAI({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg height="1em" viewBox="0 0 24 24" width="1em" xmlns="http://www.w3.org/2000/svg" className={className}>
      <title>CrewAI</title>
      <path
        d="M19.41 10.783a2.753 2.753 0 012.471 1.355c.483.806.622 1.772.385 2.68l-.136.522a9.994 9.994 0 01-3.156 5.058c-.605.517-1.283 1.062-2.083 1.524l-.028.017c-.402.232-.884.511-1.398.756-1.19.602-2.475.997-3.798 1.167-.854.111-1.716.155-2.577.132H9.072a8.588 8.588 0 01-5.046-1.87l-.012-.01-.012-.01A8.024 8.024 0 011.22 17.42a10.916 10.916 0 01-.102-3.779A15.622 15.622 0 012.88 8.4a21.758 21.758 0 012.432-3.678 15.44 15.44 0 013.56-3.182A9.958 9.958 0 0112.44.104h.004l.003-.002c2.057-.384 3.743.374 5.024 1.26a8.28 8.28 0 012.395 2.513l.024.04.023.042a5.474 5.474 0 01.508 4.012c-.239.97-.577 1.914-1.01 2.814z"
        fill="#461816"
      ></path>
      <path
        d="M18.861 13.165a.748.748 0 011.256.031c.199.332.256.73.159 1.103l-.137.522a7.936 7.936 0 01-2.504 4.014c-.572.49-1.138.939-1.774 1.306-.427.247-.857.496-1.303.707a9.628 9.628 0 01-3.155.973 14.33 14.33 0 01-2.257.116 6.531 6.531 0 01-3.837-1.422 5.967 5.967 0 01-2.071-3.494 8.859 8.859 0 01-.085-3.08 13.56 13.56 0 011.54-4.568 19.701 19.701 0 012.212-3.348 13.382 13.382 0 013.088-2.76 7.9 7.9 0 012.832-1.14c1.307-.245 2.434.207 3.481.933a6.222 6.222 0 011.806 1.892c.423.767.536 1.668.314 2.515a12.394 12.394 0 01-.99 2.67l-.223.497c-.321.713-.642 1.426-.97 2.137a.762.762 0 01-.97.467 3.39 3.39 0 01-2.283-2.49c-.095-.83.04-1.669.39-2.426.288-.746.61-1.477.933-2.208l.248-.563a.53.53 0 00-.204-.742 2.35 2.35 0 00-1.2.702 25.291 25.291 0 00-1.614 1.767 21.561 21.561 0 00-2.619 4.184 7.59 7.59 0 00-.816 2.753 7.042 7.042 0 00.07 2.219 2.055 2.055 0 001.934 1.715c1.801.1 3.59-.363 5.116-1.328.582-.4 1.141-.831 1.675-1.294.752-.71 1.376-1.519 1.958-2.36z"
        fill="#fff"
      ></path>
    </svg>
  );
}

function IconOpenTelemetry({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 128 128" className={cn("w-4 h-4", className)}>
      <path
        fill="#fff"
        d="M67.648 69.797c-5.246 5.25-5.246 13.758 0 19.008 5.25 5.246 13.758 5.246 19.004 0 5.25-5.25 5.25-13.758 0-19.008-5.246-5.246-13.754-5.246-19.004 0Zm14.207 14.219a6.649 6.649 0 0 1-9.41 0 6.65 6.65 0 0 1 0-9.407 6.649 6.649 0 0 1 9.41 0c2.598 2.586 2.598 6.809 0 9.407ZM86.43 3.672l-8.235 8.234a4.17 4.17 0 0 0 0 5.875l32.149 32.149a4.17 4.17 0 0 0 5.875 0l8.234-8.235c1.61-1.61 1.61-4.261 0-5.87L92.29 3.671a4.159 4.159 0 0 0-5.86 0ZM28.738 108.895a3.763 3.763 0 0 0 0-5.31l-4.183-4.187a3.768 3.768 0 0 0-5.313 0l-8.644 8.649-.016.012-2.371-2.375c-1.313-1.313-3.45-1.313-4.75 0-1.313 1.312-1.313 3.449 0 4.75l14.246 14.242a3.353 3.353 0 0 0 4.746 0c1.3-1.313 1.313-3.45 0-4.746l-2.375-2.375.016-.012Zm0 0"
      />
      <path
        fill="#fff"
        d="M72.297 27.313 54.004 45.605c-1.625 1.625-1.625 4.301 0 5.926L65.3 62.824c7.984-5.746 19.18-5.035 26.363 2.153l9.148-9.149c1.622-1.625 1.622-4.297 0-5.922L78.22 27.313a4.185 4.185 0 0 0-5.922 0ZM60.55 67.585l-6.672-6.672c-1.563-1.562-4.125-1.562-5.684 0l-23.53 23.54a4.036 4.036 0 0 0 0 5.687l13.331 13.332a4.036 4.036 0 0 0 5.688 0l15.132-15.157c-3.199-6.609-2.625-14.593 1.735-20.73Zm0 0"
      />
    </svg>
  );
}

function IconPlaywright({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg viewBox="0 0 128 128" xmlns="http://www.w3.org/2000/svg" className={cn("", className)}>
      <path
        d="M43.662 70.898c-4.124 1.17-6.829 3.222-8.611 5.272 1.707-1.494 3.993-2.865 7.077-3.739 3.155-.894 5.846-.888 8.069-.459v-1.739c-1.897-.173-4.072-.035-6.536.664ZM34.863 56.28l-15.314 4.035s.279.394.796.92l12.984-3.421s-.184 2.371-1.782 4.492c3.022-2.287 3.316-6.025 3.316-6.025Zm12.819 35.991C26.131 98.076 14.729 73.1 11.277 60.137 9.682 54.153 8.986 49.621 8.8 46.697a4.955 4.955 0 0 1 .011-.794c-1.118.068-1.653.649-1.544 2.328.186 2.923.882 7.454 2.477 13.44 3.45 12.961 14.854 37.937 36.405 32.132 4.691-1.264 8.215-3.565 10.86-6.504-2.438 2.202-5.49 3.937-9.327 4.972Zm4.05-51.276v1.534h8.453c-.173-.543-.348-1.032-.522-1.534h-7.932Z"
        fill="#2D4552"
      />
      <path
        d="M62.074 53.627c3.802 1.08 5.812 3.745 6.875 6.104l4.239 1.204s-.578-8.255-8.045-10.376c-6.985-1.985-11.284 3.881-11.807 4.64 2.032-1.448 4.999-2.633 8.738-1.572Zm33.741 6.142c-6.992-1.994-11.289 3.884-11.804 4.633 2.034-1.446 4.999-2.632 8.737-1.566 3.796 1.081 5.804 3.743 6.87 6.104l4.245 1.208s-.588-8.257-8.048-10.379Zm-4.211 21.766-35.261-9.858s.382 1.935 1.846 4.441l29.688 8.3c2.444-1.414 3.726-2.883 3.726-2.883Zm-24.446 21.218c-27.92-7.485-24.544-43.059-20.027-59.916 1.86-6.947 3.772-12.11 5.358-15.572-.946-.195-1.73.304-2.504 1.878-1.684 3.415-3.837 8.976-5.921 16.76-4.516 16.857-7.892 52.429 20.027 59.914 13.159 3.525 23.411-1.833 31.053-10.247-7.254 6.57-16.515 10.253-27.986 7.182Z"
        fill="#2D4552"
      />
      <path
        d="M51.732 83.935v-7.179l-19.945 5.656s1.474-8.563 11.876-11.514c3.155-.894 5.846-.888 8.069-.459V40.995h9.987c-1.087-3.36-2.139-5.947-3.023-7.744-1.461-2.975-2.96-1.003-6.361 1.842-2.396 2.001-8.45 6.271-17.561 8.726-9.111 2.457-16.476 1.805-19.55 1.273-4.357-.752-6.636-1.708-6.422 1.605.186 2.923.882 7.455 2.477 13.44 3.45 12.962 14.854 37.937 36.405 32.132 5.629-1.517 9.603-4.515 12.357-8.336h-8.309v.002Zm-32.185-23.62 15.316-4.035s-.446 5.892-6.188 7.405c-5.743 1.512-9.128-3.371-9.128-3.371Z"
        fill="#E2574C"
      />
      <path
        d="M109.372 41.336c-3.981.698-13.532 1.567-25.336-1.596-11.807-3.162-19.64-8.692-22.744-11.292-4.4-3.685-6.335-6.246-8.24-2.372-1.684 3.417-3.837 8.977-5.921 16.762-4.516 16.857-7.892 52.429 20.027 59.914 27.912 7.479 42.772-25.017 47.289-41.875 2.084-7.783 2.998-13.676 3.25-17.476.287-4.305-2.67-3.055-8.324-2.064ZM53.28 55.282s4.4-6.843 11.862-4.722c7.467 2.121 8.045 10.376 8.045 10.376L53.28 55.282Zm18.215 30.706c-13.125-3.845-15.15-14.311-15.15-14.311l35.259 9.858c0-.002-7.117 8.25-20.109 4.453Zm12.466-21.51s4.394-6.838 11.854-4.711c7.46 2.124 8.048 10.379 8.048 10.379l-19.902-5.668Z"
        fill="#2EAD33"
      />
      <path
        d="M44.762 78.733 31.787 82.41s1.41-8.029 10.968-11.212l-7.347-27.573-.635.193c-9.111 2.457-16.476 1.805-19.55 1.273-4.357-.751-6.636-1.708-6.422 1.606.186 2.923.882 7.454 2.477 13.44 3.45 12.961 14.854 37.937 36.405 32.132l.635-.199-3.555-13.337ZM19.548 60.315l15.316-4.035s-.446 5.892-6.188 7.405c-5.743 1.512-9.128-3.371-9.128-3.371Z"
        fill="#D65348"
      />
      <path
        d="m72.086 86.132-.594-.144c-13.125-3.844-15.15-14.311-15.15-14.311l18.182 5.082L84.15 39.77l-.116-.031c-11.807-3.162-19.64-8.692-22.744-11.292-4.4-3.685-6.335-6.246-8.24-2.372-1.682 3.417-3.836 8.977-5.92 16.762-4.516 16.857-7.892 52.429 20.027 59.914l.572.129 4.357-16.748Zm-18.807-30.85s4.4-6.843 11.862-4.722c7.467 2.121 8.045 10.376 8.045 10.376l-19.907-5.654Z"
        fill="#1D8D22"
      />
      <path
        d="m45.423 78.544-3.48.988c.822 4.634 2.271 9.082 4.545 13.011.396-.087.788-.163 1.192-.273a25.224 25.224 0 0 0 2.98-1.023c-2.541-3.771-4.222-8.114-5.237-12.702Zm-1.359-32.64c-1.788 6.674-3.388 16.28-2.948 25.915a20.061 20.061 0 0 1 2.546-.923l.644-.144c-.785-10.292.912-20.78 2.825-27.915a139.404 139.404 0 0 1 1.455-5.05 45.171 45.171 0 0 1-2.578 1.53 132.234 132.234 0 0 0-1.944 6.587Z"
        fill="#C04B41"
      />
    </svg>
  );
}

function IconBrowserUse({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      width="100"
      height="100"
      viewBox="0 0 100 100"
      fill="none"
      xmlns="http://www.w3.org/2000/svg"
      className={cn("", className)}
    >
      <path
        d="M97.8916 39.0448C82.6177 33.1997 95.2199 10.8169 74.212 11.3849C48.5413 12.0793 8.31528 52.4518 12.4236 78.6851C14.4652 91.6755 24.6096 86.2218 29.3732 88.1154C32.5364 89.3652 36.2792 95.0083 40.3245 95.9047C22.4293 106.193 -0.556809 96.397 0.0102912 74.3423C0.829435 41.86 47.7474 -5.25386 81.1937 0.477571C99.8702 3.68414 102.189 23.5422 97.8916 39.0448Z"
        fill="white"
      />
      <path
        d="M24.8115 57.7541L39.6068 71.7166C49.0332 80.1875 74.061 94.9706 85.403 84.9469C98.774 73.1306 70.495 32.3162 57.4769 25.802L68.9069 20.6639C86.7138 33.6796 113.783 75.9836 91.7294 94.4025C77.5014 106.282 54.5655 96.2204 41.0811 87.3707C30.8103 80.6294 15.9647 70.9591 24.8115 57.7415V57.7541Z"
        fill="white"
      />
      <path
        d="M40.3373 4.75723C35.5485 4.88347 31.8055 11.1199 28.2895 12.2182C25.1642 13.1903 20.8414 10.5266 16.1408 14.0487C11.0495 17.8613 12.7891 36.0655 3.02233 40.5976C-2.98893 22.9362 0.75354 1.8789 22.4672 0.0736228C24.1433 -0.0652445 42.7822 1.17195 40.3373 4.74463V4.75723Z"
        fill="white"
      />
      <path d="M76.1025 57.754C84.1175 71.0348 69.5871 86.2092 57.489 74.1025L76.1025 57.754Z" fill="white" />
    </svg>
  );
}

function IconStagehand({ className, ...props }: React.ComponentProps<"svg">) {
  return <Image src="/stagehand.png" alt="Stagehand" width={100} height={100} />;
}

function IconVercel({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg aria-label="Vercel logomark" role="img" viewBox="0 0 74 64" className={cn("h-4 w-4", className)} {...props}>
      <path d="M37.5896 0.25L74.5396 64.25H0.639648L37.5896 0.25Z" fill="currentColor"></path>
    </svg>
  );
}

function IconGitHub({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      role="img"
      viewBox="0 0 24 24"
      xmlns="http://www.w3.org/2000/svg"
      fill="currentColor"
      className={cn("h-full w-full", className)}
      {...props}
    >
      <title>GitHub</title>
      <path d="M12 .297c-6.63 0-12 5.373-12 12 0 5.303 3.438 9.8 8.205 11.385.6.113.82-.258.82-.577 0-.285-.01-1.04-.015-2.04-3.338.724-4.042-1.61-4.042-1.61C4.422 18.07 3.633 17.7 3.633 17.7c-1.087-.744.084-.729.084-.729 1.205.084 1.838 1.236 1.838 1.236 1.07 1.835 2.809 1.305 3.495.998.108-.776.417-1.305.76-1.605-2.665-.3-5.466-1.332-5.466-5.93 0-1.31.465-2.38 1.235-3.22-.135-.303-.54-1.523.105-3.176 0 0 1.005-.322 3.3 1.23.96-.267 1.98-.399 3-.405 1.02.006 2.04.138 3 .405 2.28-1.552 3.285-1.23 3.285-1.23.645 1.653.24 2.873.12 3.176.765.84 1.23 1.91 1.23 3.22 0 4.61-2.805 5.625-5.475 5.92.42.36.81 1.096.81 2.22 0 1.606-.015 2.896-.015 3.286 0 .315.21.69.825.57C20.565 22.092 24 17.592 24 12.297c0-6.627-5.373-12-12-12" />
    </svg>
  );
}

function IconGoogle({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg className={cn("h-4 w-4", className)} viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" {...props}>
      <title>Google</title>
      <path
        d="M23 12.245c0-.905-.075-1.565-.236-2.25h-10.54v4.083h6.186c-.124 1.014-.797 2.542-2.294 3.569l-.021.136 3.332 2.53.23.022C21.779 18.417 23 15.593 23 12.245z"
        fill="#4285F4"
      ></path>
      <path
        d="M12.225 23c3.03 0 5.574-.978 7.433-2.665l-3.542-2.688c-.948.648-2.22 1.1-3.891 1.1a6.745 6.745 0 01-6.386-4.572l-.132.011-3.465 2.628-.045.124C4.043 20.531 7.835 23 12.225 23z"
        fill="#34A853"
      ></path>
      <path
        d="M5.84 14.175A6.65 6.65 0 015.463 12c0-.758.138-1.491.361-2.175l-.006-.147-3.508-2.67-.115.054A10.831 10.831 0 001 12c0 1.772.436 3.447 1.197 4.938l3.642-2.763z"
        fill="#FBBC05"
      ></path>
      <path
        d="M12.225 5.253c2.108 0 3.529.892 4.34 1.638l3.167-3.031C17.787 2.088 15.255 1 12.225 1 7.834 1 4.043 3.469 2.197 7.062l3.63 2.763a6.77 6.77 0 016.398-4.572z"
        fill="#EB4335"
      ></path>
    </svg>
  );
}

function IconSpinner({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      xmlns="http://www.w3.org/2000/svg"
      viewBox="0 0 256 256"
      fill="currentColor"
      className={cn("h-full w-full animate-spin", className)}
      {...props}
    >
      <path d="M232 128a104 104 0 0 1-208 0c0-41 23.81-78.36 60.66-95.27a8 8 0 0 1 6.68 14.54C60.15 61.59 40 93.27 40 128a88 88 0 0 0 176 0c0-34.73-20.15-66.41-51.34-80.73a8 8 0 0 1 6.68-14.54C208.19 49.64 232 87 232 128Z" />
    </svg>
  );
}

function IconMessage({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      xmlns="http://www.w3.org/2000/svg"
      viewBox="0 0 256 256"
      fill="currentColor"
      className={cn("h-4 w-4", className)}
      {...props}
    >
      <path d="M216 48H40a16 16 0 0 0-16 16v160a15.84 15.84 0 0 0 9.25 14.5A16.05 16.05 0 0 0 40 240a15.89 15.89 0 0 0 10.25-3.78.69.69 0 0 0 .13-.11L82.5 208H216a16 16 0 0 0 16-16V64a16 16 0 0 0-16-16ZM40 224Zm176-32H82.5a16 16 0 0 0-10.3 3.75l-.12.11L40 224V64h176Z" />
    </svg>
  );
}

function IconAnthropic({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      fill="currentColor"
      fillRule="evenodd"
      viewBox="0 0 24 24"
      className={cn("w-4 h-4", className)}
      xmlns="http://www.w3.org/2000/svg"
      {...props}
    >
      <path d="M13.827 3.52h3.603L24 20h-3.603l-6.57-16.48zm-7.258 0h3.767L16.906 20h-3.674l-1.343-3.461H5.017l-1.344 3.46H0L6.57 3.522zm4.132 9.959L8.453 7.687 6.205 13.48H10.7z"></path>
    </svg>
  );
}

function IconGemini({ className, ...props }: React.ComponentProps<"svg">) {
  const gradientId = "icon-gemini-gradient";

  return (
    <svg className={cn("w-4 h-4", className)} viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" {...props}>
      <title>Gemini</title>
      <defs>
        <linearGradient id={gradientId} x1="0%" x2="68.73%" y1="100%" y2="30.395%">
          <stop offset="0%" stopColor="#1C7DFF"></stop>
          <stop offset="52.021%" stopColor="#1C69FF"></stop>
          <stop offset="100%" stopColor="#F0DCD6"></stop>
        </linearGradient>
      </defs>
      <path
        d="M12 24A14.304 14.304 0 000 12 14.304 14.304 0 0012 0a14.305 14.305 0 0012 12 14.305 14.305 0 00-12 12"
        fill={`url(#${gradientId})`}
        fillRule="nonzero"
      ></path>
    </svg>
  );
}

function IconGroq({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      className={cn("w-4 h-4", className)}
      fill="currentColor"
      fillRule="evenodd"
      viewBox="0 0 24 24"
      xmlns="http://www.w3.org/2000/svg"
      {...props}
    >
      <title>Groq</title>
      <path d="M12.036 2c-3.853-.035-7 3-7.036 6.781-.035 3.782 3.055 6.872 6.908 6.907h2.42v-2.566h-2.292c-2.407.028-4.38-1.866-4.408-4.23-.029-2.362 1.901-4.298 4.308-4.326h.1c2.407 0 4.358 1.915 4.365 4.278v6.305c0 2.342-1.944 4.25-4.323 4.279a4.375 4.375 0 01-3.033-1.252l-1.851 1.818A7 7 0 0012.029 22h.092c3.803-.056 6.858-3.083 6.879-6.816v-6.5C18.907 4.963 15.817 2 12.036 2z"></path>
    </svg>
  );
}

function IconMistral({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg className={cn("w-4 h-4", className)} viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" {...props}>
      <title>Mistral</title>
      <path d="M3.428 3.4h3.429v3.428H3.428V3.4zm13.714 0h3.43v3.428h-3.43V3.4z" fill="gold"></path>
      <path d="M3.428 6.828h6.857v3.429H3.429V6.828zm10.286 0h6.857v3.429h-6.857V6.828z" fill="#FFAF00"></path>
      <path d="M3.428 10.258h17.144v3.428H3.428v-3.428z" fill="#FF8205"></path>
      <path
        d="M3.428 13.686h3.429v3.428H3.428v-3.428zm6.858 0h3.429v3.428h-3.429v-3.428zm6.856 0h3.43v3.428h-3.43v-3.428z"
        fill="#FA500F"
      ></path>
      <path d="M0 17.114h10.286v3.429H0v-3.429zm13.714 0H24v3.429H13.714v-3.429z" fill="#E10500"></path>
    </svg>
  );
}

function IconAmazonBedrock({ className, ...props }: React.ComponentProps<"svg">) {
  const gradientId = "icon-amazon-bedrock-gradient";

  return (
    <svg className={cn("w-4 h-4", className)} viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" {...props}>
      <title>Bedrock</title>
      <defs>
        <linearGradient id={gradientId} x1="80%" x2="20%" y1="20%" y2="80%">
          <stop offset="0%" stopColor="#6350FB"></stop>
          <stop offset="50%" stopColor="#3D8FFF"></stop>
          <stop offset="100%" stopColor="#9AD8F8"></stop>
        </linearGradient>
      </defs>
      <path
        d="M13.05 15.513h3.08c.214 0 .389.177.389.394v1.82a1.704 1.704 0 011.296 1.661c0 .943-.755 1.708-1.685 1.708-.931 0-1.686-.765-1.686-1.708 0-.807.554-1.484 1.297-1.662v-1.425h-2.69v4.663a.395.395 0 01-.188.338l-2.69 1.641a.385.385 0 01-.405-.002l-4.926-3.086a.395.395 0 01-.185-.336V16.3L2.196 14.87A.395.395 0 012 14.555L2 14.528V9.406c0-.14.073-.27.192-.34l2.465-1.462V4.448c0-.129.062-.249.165-.322l.021-.014L9.77 1.058a.385.385 0 01.407 0l2.69 1.675a.395.395 0 01.185.336V7.6h3.856V5.683a1.704 1.704 0 01-1.296-1.662c0-.943.755-1.708 1.685-1.708.931 0 1.685.765 1.685 1.708 0 .807-.553 1.484-1.296 1.662v2.311a.391.391 0 01-.389.394h-4.245v1.806h6.624a1.69 1.69 0 011.64-1.313c.93 0 1.685.764 1.685 1.707 0 .943-.754 1.708-1.685 1.708a1.69 1.69 0 01-1.64-1.314H13.05v1.937h4.953l.915 1.18a1.66 1.66 0 01.84-.227c.931 0 1.685.764 1.685 1.707 0 .943-.754 1.708-1.685 1.708-.93 0-1.685-.765-1.685-1.708 0-.346.102-.668.276-.937l-.724-.935H13.05v1.806zM9.973 1.856L7.93 3.122V6.09h-.778V3.604L5.435 4.669v2.945l2.11 1.36L9.712 7.61V5.334h.778V7.83c0 .136-.07.263-.184.335L7.963 9.638v2.081l1.422 1.009-.446.646-1.406-.998-1.53 1.005-.423-.66 1.605-1.055v-1.99L5.038 8.29l-2.26 1.34v1.676l1.972-1.189.398.677-2.37 1.429V14.3l2.166 1.258 2.27-1.368.397.677-2.176 1.311V19.3l1.876 1.175 2.365-1.426.398.678-2.017 1.216 1.918 1.201 2.298-1.403v-5.78l-4.758 2.893-.4-.675 5.158-3.136V3.289L9.972 1.856zM16.13 18.47a.913.913 0 00-.908.92c0 .507.406.918.908.918a.913.913 0 00.907-.919.913.913 0 00-.907-.92zm3.63-3.81a.913.913 0 00-.908.92c0 .508.406.92.907.92a.913.913 0 00.908-.92.913.913 0 00-.908-.92zm1.555-4.99a.913.913 0 00-.908.92c0 .507.407.918.908.918a.913.913 0 00.907-.919.913.913 0 00-.907-.92zM17.296 3.1a.913.913 0 00-.907.92c0 .508.406.92.907.92a.913.913 0 00.908-.92.913.913 0 00-.908-.92z"
        fill={`url(#${gradientId})`}
        fillRule="nonzero"
      ></path>
    </svg>
  );
}

function IconAzure({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg className={cn("w-4 h-4", className)} viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" {...props}>
      <title>Azure</title>
      <path
        d="M7.242 1.613A1.11 1.11 0 018.295.857h6.977L8.03 22.316a1.11 1.11 0 01-1.052.755h-5.43a1.11 1.11 0 01-1.053-1.466L7.242 1.613z"
        fill="url(#lobe-icons-azure-fill-0)"
      ></path>
      <path
        d="M18.397 15.296H7.4a.51.51 0 00-.347.882l7.066 6.595c.206.192.477.298.758.298h6.226l-2.706-7.775z"
        fill="#0078D4"
      ></path>
      <path
        d="M15.272.857H7.497L0 23.071h7.775l1.596-4.73 5.068 4.73h6.665l-2.707-7.775h-7.998L15.272.857z"
        fill="url(#lobe-icons-azure-fill-1)"
      ></path>
      <path
        d="M17.193 1.613a1.11 1.11 0 00-1.052-.756h-7.81.035c.477 0 .9.304 1.052.756l6.748 19.992a1.11 1.11 0 01-1.052 1.466h-.12 7.895a1.11 1.11 0 001.052-1.466L17.193 1.613z"
        fill="url(#lobe-icons-azure-fill-2)"
      ></path>
      <defs>
        <linearGradient
          gradientUnits="userSpaceOnUse"
          id="lobe-icons-azure-fill-0"
          x1="8.247"
          x2="1.002"
          y1="1.626"
          y2="23.03"
        >
          <stop stopColor="#114A8B"></stop>
          <stop offset="1" stopColor="#0669BC"></stop>
        </linearGradient>
        <linearGradient
          gradientUnits="userSpaceOnUse"
          id="lobe-icons-azure-fill-1"
          x1="14.042"
          x2="12.324"
          y1="15.302"
          y2="15.888"
        >
          <stop stopOpacity=".3"></stop>
          <stop offset=".071" stopOpacity=".2"></stop>
          <stop offset=".321" stopOpacity=".1"></stop>
          <stop offset=".623" stopOpacity=".05"></stop>
          <stop offset="1" stopOpacity="0"></stop>
        </linearGradient>
        <linearGradient
          gradientUnits="userSpaceOnUse"
          id="lobe-icons-azure-fill-2"
          x1="12.841"
          x2="20.793"
          y1="1.626"
          y2="22.814"
        >
          <stop stopColor="#3CCBF4"></stop>
          <stop offset="1" stopColor="#2892DF"></stop>
        </linearGradient>
      </defs>
    </svg>
  );
}

function IconPython({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg className={cn("w-4 h-4", className)} viewBox="0 0 50 50" xmlns="http://www.w3.org/2000/svg" {...props}>
      <path d="M 24.5 2 C 15.800781 2 14 6.492188 14 9.09375 L 14 14 L 24 14 L 24 15 L 9.09375 15 C 5.792969 15 2 17.601563 2 25 C 2 32.398438 5.792969 35 9.09375 35 L 15 35 L 15 29 C 15 26.800781 16.800781 25 19 25 L 31 25 C 32.699219 25 34 23.699219 34 22 L 34 9.09375 C 34 5.59375 30.699219 2 24.5 2 Z M 20 7 C 21.101563 7 22 7.898438 22 9 C 22 10.101563 21.101563 11 20 11 C 18.898438 11 18 10.101563 18 9 C 18 7.898438 18.898438 7 20 7 Z M 35 15 L 35 22 C 35 24.199219 33.199219 26 31 26 L 19 26 C 17.300781 26 16 27.300781 16 29 L 16 40.90625 C 16 44.40625 19.300781 48 25.5 48 C 34.199219 48 36 43.507813 36 40.90625 L 36 36 L 26 36 L 26 35 L 40.90625 35 C 44.207031 35 48 32.398438 48 25 C 48 17.601563 44.207031 15 40.90625 15 Z M 30 39 C 31.101563 39 32 39.898438 32 41 C 32 42.101563 31.101563 43 30 43 C 28.898438 43 28 42.101563 28 41 C 28 39.898438 28.898438 39 30 39 Z"></path>
    </svg>
  );
}

function IconLangGraph({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg className={cn("w-4 h-4", className)} viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" {...props}>
      <title>LangGraph</title>
      <path
        clipRule="evenodd"
        d="M6.099 6H17.9C21.264 6 24 8.692 24 12s-2.736 6-6.099 6H6.1C2.736 18 0 15.308 0 12s2.736-6 6.099-6zm5.419 9.3c.148.154.367.146.561.106l.002.001c.09-.072-.038-.163-.16-.25-.074-.052-.145-.102-.166-.147.068-.08-.133-.265-.289-.408a1.52 1.52 0 01-.15-.148c-.11-.119-.155-.268-.2-.418-.03-.1-.06-.2-.11-.292-.304-.694-.653-1.383-1.143-1.97-.315-.39-.674-.74-1.033-1.09a19.384 19.384 0 01-.683-.688c-.226-.229-.362-.511-.499-.794-.114-.236-.228-.473-.396-.68-.507-.735-2.107-.936-2.342.104 0 .032-.01.052-.039.073-.13.094-.245.2-.342.327-.238.326-.274.877.022 1.17l.001-.019c.01-.147.02-.286.139-.391.228.193.576.262.841.117.32.45.422.995.525 1.54.085.456.17.912.382 1.316l.014.022c.124.203.25.41.41.587.059.089.178.184.297.279.157.125.314.25.329.359v.143c-.001.285-.002.58.184.813.103.205-.15.41-.352.385-.112.015-.233-.014-.354-.042-.165-.04-.329-.078-.462-.003-.038.04-.091.04-.145.042-.064.002-.129.004-.167.07-.008.019-.026.04-.045.063-.042.05-.087.105-.033.146l.015-.01c.082-.062.16-.12.27-.084-.014.08.039.102.092.123l.027.012a.344.344 0 01-.008.056c-.009.045-.017.088.018.127a.598.598 0 00.046-.054c.037-.046.073-.092.139-.11.144.19.289.111.471.013.206-.111.459-.248.81-.055-.135-.006-.255.01-.345.12-.023.024-.042.052-.002.084.207-.132.294-.085.375-.04.06.032.115.063.212.024l.07-.036c.155-.083.314-.166.499-.137-.139.039-.188.125-.242.218-.026.047-.054.095-.094.14-.021.021-.03.046-.007.08.29-.023.4-.095.548-.192.07-.046.15-.099.261-.154.124-.075.248-.027.368.02.13.05.255.098.371-.014.037-.033.083-.034.129-.034.016 0 .033 0 .05-.002-.037-.19-.24-.188-.448-.186-.24.003-.483.006-.475-.289.222-.149.224-.407.226-.651 0-.06 0-.117.005-.173.163.09.336.16.508.229.162.065.323.13.474.21.158.25.404.58.732.558.008-.026.016-.047.026-.073.019.004.039.008.059.014.086.02.178.044.223-.056zm6.429-2.829c.19.186.447.29.716.29.269 0 .526-.104.716-.29a.98.98 0 00.297-.7.98.98 0 00-.297-.7 1.024 1.024 0 00-1.08-.224l-.58-.831-.405.272.583.835a.978.978 0 00.05 1.348zm-1.817-2.69a1.03 1.03 0 001.056-.095.991.991 0 00.363-.507.97.97 0 00-.016-.62.994.994 0 00-.39-.488 1.028 1.028 0 00-1.298.14.987.987 0 00-.263.856.98.98 0 00.187.42c.095.125.218.225.36.294zm0 5.752a1.032 1.032 0 001.056-.095.991.991 0 00.363-.507.97.97 0 00-.016-.62.994.994 0 00-.39-.488 1.027 1.027 0 00-1.298.14.986.986 0 00-.263.856.98.98 0 00.187.42c.095.125.218.225.36.294zm.93-3.516v-.492h-1.55a.977.977 0 00-.217-.404l.584-.847-.425-.276-.583.847a1.023 1.023 0 00-1.047.23.973.973 0 00-.296.696c0 .261.107.512.296.696a1.023 1.023 0 001.047.23l.583.847.42-.276-.579-.847a.977.977 0 00.217-.404h1.55z"
      ></path>
    </svg>
  );
}

function IconResizeHandle({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg className={cn("w-4 h-4", className)} viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" {...props}>
      <path
        d="M21 15L15 21M21 8L8 21"
        stroke="currentColor"
        strokeWidth="2"
        strokeLinecap="round"
        strokeLinejoin="round"
      />
    </svg>
  );
}

function LogoRye({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg xmlns="http://www.w3.org/2000/svg" className={className} viewBox="0 0 106 38">
      <g>
        <path d="M 0 38 L 0 0 L 106 0 L 106 38 Z" fill="transparent"></path>
        <path
          d="M 31.557 12.041 L 17.153 0.045 L 17.153 8.189 C 17.153 9.194 16.334 10.008 15.322 10.008 L 0 10.008 L 0 14.827 L 31.149 14.827 C 31.626 14.827 32.014 14.442 32.014 13.967 L 32.014 13.012 C 32.014 12.637 31.848 12.282 31.558 12.041 Z"
          fill="rgb(255, 255, 255)"
        ></path>
        <path
          d="M 0.456 20.579 L 14.861 32.575 L 14.861 24.431 C 14.861 23.426 15.681 22.612 16.692 22.612 L 32.014 22.612 L 32.014 17.793 L 0.866 17.793 C 0.388 17.793 0 18.178 0 18.653 L 0 19.608 C 0 19.983 0.167 20.338 0.456 20.579 Z"
          fill="rgb(255, 255, 255)"
        ></path>
        <path
          d="M 85.557 20.054 C 85.557 12.996 89.747 9.477 96.041 9.477 C 102.259 9.477 106 13.093 106 19.32 L 106 21.621 L 91.033 21.621 C 91.111 24.367 92.904 26.455 96.002 26.455 C 99.062 26.455 100.31 24.927 100.621 23.69 L 105.61 23.69 L 105.61 24.27 C 104.968 27.055 102.434 30.651 96.1 30.651 C 89.766 30.651 85.557 27.113 85.557 20.054 Z M 91.091 17.753 L 100.641 17.753 C 100.602 15.471 99.101 13.615 95.963 13.615 C 92.924 13.615 91.325 15.529 91.091 17.753 Z"
          fill="rgb(255, 255, 255)"
        ></path>
        <path
          d="M 64.306 9.94 L 70.269 9.94 L 75.239 23.844 L 75.55 23.844 L 80.54 9.94 L 86.367 9.94 L 86.367 10.521 L 75.161 37.922 L 69.373 37.922 L 69.373 37.342 L 72.374 30.284 L 64.306 10.521 Z"
          fill="rgb(255, 255, 255)"
        ></path>
        <path
          d="M 53.981 2.341 C 59.164 2.341 63.276 5.164 63.276 10.637 C 63.276 16.051 59.437 18.894 54.741 18.894 L 53.532 18.894 L 63.471 29.607 L 63.471 30.187 L 56.806 30.187 L 46.517 18.797 L 46.127 18.797 L 46.127 30.187 L 40.241 30.187 L 40.241 2.341 Z M 46.127 14.446 L 53.045 14.446 C 55.676 14.446 57.469 13.228 57.469 10.791 C 57.469 8.355 55.676 7.156 53.045 7.156 L 46.127 7.156 Z"
          fill="rgb(255, 255, 255)"
        ></path>
      </g>
    </svg>
  );
}

function LogoRemo({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      width="568"
      height="203"
      viewBox="0 0 568 203"
      fill="none"
      xmlns="http://www.w3.org/2000/svg"
      className={className}
      {...props}
    >
      <rect width="568" height="203" fill="#090D0D" />
      <g clipPath="url(#clip0_150_2)">
        <path
          d="M76.14 94.8197L76.44 96.3797L88.15 157.38H66.99L55.83 99.2297L54.29 91.1797C58.31 82.1797 63.95 74.0597 70.85 67.1997C82.87 55.2197 98.7 47.0597 116.36 44.6797V66.1097C99.11 69.2997 84.52 80.0597 76.15 94.8097L76.14 94.8197Z"
          fill="#29BDBD"
        />
        <path
          d="M88.15 157.38H66.99L55.82 99.2396L54.28 91.1896L45.83 47.1396H66.99L70.84 67.2096L72.08 73.6696L76.14 94.8196L76.44 96.3796L88.15 157.38Z"
          fill="#DCF0EE"
        />
        <path
          d="M178.51 159.83C167.12 159.83 157.04 157.41 148.25 152.56C139.46 147.71 132.57 141.07 127.59 132.63C122.6 124.2 120.1 114.5 120.1 103.53C120.1 94.8203 121.5 86.8703 124.32 79.7003C127.13 72.5303 131.03 66.3103 136.02 61.0403C141.01 55.7703 146.98 51.6902 153.94 48.8102C160.9 45.9302 168.45 44.4902 176.61 44.4902C183.78 44.4902 190.45 45.8603 196.64 48.6003C202.82 51.3403 208.2 55.1002 212.77 59.8802C217.34 64.6602 220.85 70.3202 223.31 76.8502C225.77 83.3902 226.93 90.5202 226.79 98.2502L226.58 107.53H136.12L131.27 90.2402H207.81L204.65 93.8203V88.7603C204.23 84.1203 202.68 79.9803 200.01 76.3203C197.34 72.6703 193.96 69.7802 189.89 67.6802C185.81 65.5702 181.38 64.5202 176.61 64.5202C169.02 64.5202 162.62 65.9602 157.42 68.8402C152.22 71.7202 148.28 75.9402 145.61 81.4902C142.94 87.0502 141.6 93.9002 141.6 102.05C141.6 109.78 143.21 116.49 146.45 122.19C149.68 127.88 154.25 132.28 160.16 135.37C166.06 138.46 172.88 140.01 180.61 140.01C186.09 140.01 191.19 139.1 195.9 137.27C200.61 135.44 205.7 132.14 211.19 127.36L222.15 142.75C218.78 146.12 214.66 149.08 209.82 151.61C204.97 154.14 199.84 156.14 194.43 157.62C189.02 159.1 183.71 159.83 178.51 159.83Z"
          fill="#DCF0EE"
        />
        <path
          d="M235.44 157.72V47.0199H257.37V70.6399L253.36 73.1699C254.48 69.5199 256.28 65.9999 258.74 62.6299C261.2 59.2599 264.22 56.2699 267.81 53.6699C271.39 51.0699 275.26 48.9999 279.41 47.4499C283.55 45.9099 287.81 45.1299 292.17 45.1299C298.5 45.1299 304.05 46.1799 308.83 48.2899C313.61 50.3999 317.54 53.5599 320.64 57.7799C323.73 61.9999 325.98 67.2699 327.39 73.5899L324.02 72.7499L325.5 69.1699C327.04 65.9399 329.19 62.8799 331.93 59.9999C334.67 57.1199 337.8 54.5499 341.31 52.2999C344.82 50.0499 348.55 48.2899 352.49 47.0299C356.42 45.7599 360.29 45.1299 364.09 45.1299C372.38 45.1299 379.24 46.8199 384.65 50.1899C390.06 53.5599 394.14 58.6999 396.88 65.5799C399.62 72.4699 400.99 81.0399 400.99 91.2999V157.72H379.06V92.5699C379.06 86.2399 378.25 81.0799 376.64 77.0699C375.02 73.0599 372.6 70.0799 369.36 68.1099C366.12 66.1399 361.98 65.1599 356.92 65.1599C352.98 65.1599 349.29 65.8299 345.85 67.1599C342.4 68.4999 339.42 70.3599 336.89 72.7499C334.36 75.1399 332.39 77.9199 330.99 81.0799C329.58 84.2399 328.88 87.7199 328.88 91.5199V157.73H306.95V92.1499C306.95 86.3899 306.11 81.4999 304.42 77.4999C302.73 73.4899 300.27 70.4399 297.04 68.3299C293.8 66.2199 289.87 65.1699 285.23 65.1699C281.29 65.1699 277.64 65.8399 274.27 67.1699C270.9 68.5099 267.94 70.3299 265.41 72.6499C262.88 74.9699 260.91 77.7099 259.51 80.8699C258.1 84.0299 257.4 87.4399 257.4 91.0999V157.73H235.47L235.44 157.72Z"
          fill="#DCF0EE"
        />
        <path
          d="M465.27 159.83C454.44 159.83 444.78 157.34 436.28 152.34C427.77 147.35 421.06 140.53 416.14 131.89C411.22 123.25 408.76 113.37 408.76 102.26C408.76 91.1504 411.22 81.2804 416.14 72.6304C421.06 63.9904 427.77 57.1704 436.28 52.1804C444.78 47.1904 454.45 44.6904 465.27 44.6904C476.09 44.6904 485.55 47.1904 494.05 52.1804C502.55 57.1704 509.27 63.9904 514.19 72.6304C519.11 81.2704 521.57 91.1504 521.57 102.26C521.57 113.37 519.11 123.24 514.19 131.89C509.27 140.53 502.56 147.36 494.05 152.34C485.54 157.33 475.95 159.83 465.27 159.83ZM465.27 140.01C471.88 140.01 477.78 138.36 482.98 135.06C488.18 131.76 492.26 127.26 495.21 121.57C498.16 115.88 499.57 109.45 499.43 102.28C499.57 94.9705 498.16 88.4705 495.21 82.7805C492.26 77.0905 488.18 72.6304 482.98 69.3904C477.78 66.1604 471.87 64.5404 465.27 64.5404C458.67 64.5404 452.72 66.1904 447.45 69.4904C442.18 72.7904 438.07 77.2904 435.11 82.9804C432.16 88.6704 430.75 95.1004 430.89 102.27C430.75 109.44 432.16 115.87 435.11 121.56C438.06 127.25 442.17 131.75 447.45 135.05C452.72 138.35 458.66 140 465.27 140V140.01Z"
          fill="#DCF0EE"
        />
      </g>
      <defs>
        <clipPath id="clip0_150_2">
          <rect width="476" height="116" fill="white" transform="translate(46 44)" />
        </clipPath>
      </defs>
    </svg>
  );
}

function LogoAxionRay({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg xmlns="http://www.w3.org/2000/svg" className={className} viewBox="0 0 125 30">
      <path
        d="M 30.333 22.949 L 26.638 17.565 L 22.923 22.949 L 20.09 22.949 L 25.211 15.716 L 20.292 8.729 L 23.203 8.729 L 26.698 13.828 L 30.211 8.729 L 33.044 8.729 L 28.143 15.676 L 33.244 22.949 Z M 39.605 8.729 L 39.605 22.949 L 37.135 22.949 L 37.135 8.729 Z M 51.127 23.192 C 46.81 23.192 43.797 19.881 43.797 15.859 C 43.797 11.837 46.849 8.485 51.167 8.485 C 55.484 8.485 58.498 11.796 58.498 15.818 C 58.498 19.84 55.445 23.192 51.127 23.192 Z M 51.127 10.779 C 48.336 10.779 46.389 13.034 46.389 15.818 C 46.389 18.6 48.377 20.896 51.168 20.896 C 53.96 20.896 55.907 18.642 55.907 15.858 C 55.907 13.075 53.919 10.779 51.127 10.779 Z M 71.635 8.729 L 74.065 8.729 L 74.065 22.949 L 71.997 22.949 L 64.244 12.832 L 64.244 22.949 L 61.814 22.949 L 61.814 8.729 L 64.104 8.729 L 71.635 18.561 Z M 2.741 22.949 L 8.925 8.627 L 11.028 8.627 L 5.282 22.949 Z"
        fill="rgb(255,255,255)"
      ></path>
      <path d="M 16.765 1.449 L 7.51 22.903 L 10.051 22.903 Z" fill="rgb(2,229,253)"></path>
      <path
        d="M 13.339 19.307 L 13.425 19.515 L 14.87 22.949 L 17.401 22.949 L 14.384 15.965 Z M 81.816 8.777 L 87.608 8.799 C 89.392 8.807 90.753 9.317 91.612 10.173 C 92.251 10.844 92.628 11.758 92.624 12.812 L 92.624 12.852 C 92.615 15.244 90.786 16.637 88.2 16.849 L 92.847 23.009 L 91.985 23.006 L 87.439 16.927 L 87.379 16.927 L 82.469 16.91 L 82.447 22.971 L 81.766 22.968 Z M 87.399 16.299 C 90.284 16.309 91.934 14.897 91.941 12.891 L 91.941 12.851 C 91.949 10.722 90.331 9.44 87.566 9.429 L 82.496 9.41 L 82.471 16.283 Z M 102.096 8.752 L 102.738 8.755 L 109.339 23.071 L 108.598 23.068 L 106.709 18.925 L 98.032 18.893 L 96.094 23.021 L 95.392 23.018 Z M 106.41 18.295 L 102.394 9.564 L 98.334 18.266 Z M 114.95 17.273 L 109.01 8.879 L 109.851 8.882 L 115.314 16.666 L 120.792 8.923 L 121.613 8.926 L 115.632 17.276 L 115.61 23.093 L 114.929 23.09 Z"
        fill="rgb(255,255,255)"
      ></path>
    </svg>
  );
}

function LogoBrowserUse({ className, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg xmlns="http://www.w3.org/2000/svg" className={className} viewBox="0 0 163.15 23.999" overflow="visible">
      <g>
        <path
          d="M 23.834 9.37 C 20.114 7.967 23.182 2.596 18.068 2.733 C 11.818 2.898 2.025 12.587 3.026 18.883 C 3.523 22 5.992 20.692 7.151 21.146 C 7.923 21.446 8.833 22.801 9.818 23.016 C 5.462 25.485 -0.135 23.134 0.002 17.842 C 0.203 10.046 11.625 -1.262 19.768 0.115 C 24.314 0.885 24.878 5.651 23.832 9.37 Z"
          fill="rgb(255,255,255)"
        ></path>
        <path
          d="M 5.932 13.844 L 9.534 17.194 C 11.83 19.228 17.924 22.774 20.684 20.371 C 23.94 17.534 17.055 7.738 13.886 6.176 L 16.668 4.942 C 21.003 8.067 27.593 18.218 22.224 22.639 C 18.76 25.489 13.176 23.075 9.894 20.952 C 7.394 19.332 3.778 17.012 5.932 13.842 L 5.932 13.843 Z M 9.712 1.126 C 8.546 1.156 7.634 2.652 6.78 2.916 C 6.018 3.15 4.965 2.51 3.82 3.356 C 2.582 4.272 3.005 8.639 0.627 9.726 C -0.836 5.489 0.074 0.436 5.359 0.003 C 5.769 -0.031 10.307 0.265 9.712 1.123 Z"
          fill="rgb(255,255,255)"
        ></path>
        <path
          d="M 18.533 13.804 C 20.483 16.991 16.946 20.633 14 17.727 L 18.533 13.803 Z M 39.355 18.802 L 39.355 4 L 45.282 4 C 46.37 4 47.277 4.161 48.006 4.485 C 48.733 4.807 49.281 5.257 49.648 5.828 C 50.013 6.397 50.197 7.052 50.197 7.795 C 50.197 8.373 50.08 8.882 49.85 9.321 C 49.623 9.749 49.295 10.116 48.895 10.39 C 48.495 10.664 48.038 10.86 47.523 10.976 L 47.523 11.12 C 48.686 11.172 49.741 11.82 50.313 12.833 C 50.621 13.358 50.776 13.983 50.776 14.713 C 50.776 15.498 50.579 16.199 50.189 16.815 C 49.804 17.427 49.232 17.912 48.475 18.27 C 47.72 18.624 46.787 18.802 45.68 18.802 L 39.353 18.802 Z M 42.485 16.244 L 45.035 16.244 C 45.908 16.244 46.544 16.078 46.943 15.747 C 47.341 15.415 47.563 14.918 47.543 14.4 C 47.543 13.992 47.443 13.63 47.247 13.317 C 47.04 12.996 46.747 12.74 46.401 12.579 C 45.996 12.392 45.554 12.3 45.108 12.312 L 42.485 12.312 Z M 42.485 10.194 L 44.805 10.194 C 45.234 10.194 45.615 10.12 45.945 9.971 C 46.283 9.817 46.549 9.599 46.741 9.319 C 46.943 9.024 47.047 8.672 47.038 8.314 C 47.055 7.819 46.844 7.343 46.466 7.022 C 46.089 6.694 45.556 6.529 44.861 6.529 L 42.484 6.529 L 42.484 10.194 Z M 52.753 18.802 L 52.753 7.701 L 55.737 7.701 L 55.737 9.638 L 55.854 9.638 C 56.056 8.948 56.395 8.428 56.872 8.078 C 57.347 7.721 57.928 7.532 58.522 7.541 C 58.848 7.544 59.172 7.581 59.489 7.651 L 59.489 10.382 C 59.292 10.329 59.092 10.291 58.889 10.268 C 58.657 10.236 58.423 10.218 58.189 10.216 C 57.735 10.216 57.332 10.316 56.973 10.513 C 56.626 10.701 56.337 10.981 56.137 11.322 C 55.929 11.688 55.825 12.102 55.833 12.522 L 55.833 18.802 Z M 65.58 19.02 C 64.459 19.02 63.488 18.78 62.669 18.304 C 61.861 17.829 61.206 17.132 60.782 16.296 C 60.338 15.433 60.117 14.433 60.117 13.294 C 60.117 12.149 60.34 11.146 60.782 10.288 C 61.203 9.449 61.858 8.751 62.669 8.278 C 63.489 7.798 64.459 7.557 65.582 7.557 C 66.704 7.557 67.672 7.797 68.487 8.279 C 69.3 8.75 69.958 9.449 70.38 10.288 C 70.825 11.146 71.046 12.149 71.046 13.296 C 71.046 14.433 70.823 15.433 70.38 16.296 C 69.956 17.134 69.298 17.831 68.487 18.304 C 67.672 18.782 66.704 19.02 65.582 19.02 Z M 65.596 16.635 C 66.106 16.635 66.534 16.491 66.876 16.201 C 67.218 15.908 67.476 15.506 67.648 15.001 C 67.828 14.495 67.916 13.921 67.916 13.274 C 67.916 12.627 67.827 12.053 67.648 11.545 C 67.503 11.083 67.236 10.668 66.875 10.345 C 66.517 10.046 66.062 9.89 65.596 9.906 C 65.081 9.906 64.646 10.052 64.294 10.346 C 63.928 10.668 63.656 11.083 63.506 11.547 C 63.334 12.053 63.248 12.627 63.248 13.274 C 63.248 13.919 63.334 14.494 63.505 15.001 C 63.685 15.506 63.948 15.908 64.295 16.201 C 64.645 16.491 65.08 16.635 65.596 16.635 Z M 74.959 18.803 L 71.939 7.701 L 75.053 7.701 L 76.773 15.16 L 76.876 15.16 L 78.668 7.7 L 81.726 7.7 L 83.547 15.117 L 83.641 15.117 L 85.333 7.701 L 88.44 7.701 L 85.427 18.803 L 82.167 18.803 L 80.258 11.822 L 80.121 11.822 L 78.211 18.802 L 74.958 18.802 Z M 99.321 10.866 L 96.501 11.041 C 96.454 10.802 96.347 10.578 96.191 10.391 C 96.022 10.186 95.806 10.025 95.561 9.921 C 95.273 9.794 94.96 9.733 94.645 9.741 C 94.168 9.741 93.765 9.842 93.437 10.045 C 93.109 10.242 92.945 10.508 92.945 10.841 C 92.944 11.102 93.062 11.349 93.265 11.512 C 93.475 11.695 93.84 11.842 94.355 11.953 L 96.365 12.358 C 97.445 12.579 98.249 12.936 98.779 13.427 C 99.309 13.919 99.573 14.565 99.573 15.364 C 99.573 16.092 99.358 16.731 98.93 17.28 C 98.507 17.829 97.923 18.257 97.18 18.566 C 96.443 18.869 95.592 19.022 94.629 19.022 C 93.159 19.022 91.989 18.715 91.117 18.105 C 90.271 17.524 89.715 16.608 89.59 15.589 L 92.62 15.429 C 92.693 15.848 92.932 16.219 93.284 16.457 C 93.637 16.687 94.087 16.803 94.637 16.803 C 95.177 16.803 95.61 16.7 95.937 16.493 C 96.27 16.28 96.44 16.007 96.443 15.675 C 96.442 15.403 96.311 15.148 96.09 14.989 C 95.858 14.806 95.501 14.666 95.02 14.569 L 93.097 14.186 C 92.014 13.968 91.207 13.593 90.676 13.058 C 90.151 12.525 89.887 11.84 89.887 11.013 C 89.887 10.3 90.081 9.687 90.467 9.17 C 90.857 8.655 91.403 8.256 92.107 7.976 C 92.815 7.696 93.645 7.558 94.593 7.558 C 95.995 7.558 97.1 7.854 97.903 8.446 C 98.713 9.04 99.186 9.848 99.321 10.868 Z M 106.573 19.02 C 105.431 19.02 104.447 18.789 103.623 18.326 C 102.811 17.866 102.153 17.177 101.73 16.346 C 101.286 15.487 101.065 14.474 101.065 13.303 C 101.065 12.161 101.288 11.159 101.73 10.295 C 102.149 9.459 102.798 8.759 103.601 8.279 C 104.41 7.799 105.36 7.557 106.45 7.557 C 107.182 7.557 107.864 7.675 108.495 7.91 C 109.13 8.141 109.685 8.491 110.159 8.959 C 110.634 9.426 111.006 10.014 111.27 10.722 C 111.536 11.425 111.668 12.249 111.668 13.193 L 111.668 14.04 L 102.293 14.04 L 102.293 12.132 L 108.77 12.132 C 108.779 11.721 108.679 11.314 108.48 10.954 C 108.292 10.617 108.015 10.339 107.678 10.151 C 107.319 9.948 106.913 9.846 106.501 9.854 C 106.033 9.854 105.618 9.963 105.258 10.18 C 104.906 10.386 104.615 10.684 104.418 11.041 C 104.213 11.405 104.106 11.816 104.108 12.233 L 104.108 14.047 C 104.108 14.596 104.208 15.071 104.41 15.471 C 104.618 15.871 104.909 16.179 105.285 16.397 C 105.662 16.612 106.108 16.723 106.622 16.723 C 106.966 16.723 107.277 16.673 107.562 16.577 C 108.114 16.399 108.554 15.978 108.755 15.434 L 111.603 15.624 C 111.468 16.288 111.162 16.905 110.715 17.415 C 110.243 17.941 109.653 18.347 108.994 18.6 C 108.295 18.88 107.487 19.02 106.573 19.02 Z M 113.677 18.802 L 113.677 7.701 L 116.661 7.701 L 116.661 9.638 L 116.778 9.638 C 116.981 8.948 117.32 8.428 117.797 8.078 C 118.272 7.721 118.852 7.532 119.446 7.541 C 119.772 7.544 120.096 7.581 120.414 7.651 L 120.414 10.382 C 120.217 10.329 120.017 10.291 119.814 10.268 C 119.582 10.236 119.348 10.218 119.114 10.216 C 118.661 10.216 118.257 10.316 117.898 10.513 C 117.551 10.701 117.261 10.981 117.061 11.322 C 116.854 11.688 116.749 12.102 116.758 12.522 L 116.758 18.802 Z M 135.92 4 L 139.049 4 L 139.049 13.613 C 139.049 14.693 138.792 15.637 138.276 16.446 C 137.752 17.266 137.008 17.923 136.129 18.341 C 135.208 18.787 134.136 19.014 132.912 19.014 C 131.682 19.014 130.609 18.789 129.688 18.341 C 128.808 17.924 128.064 17.267 127.541 16.446 C 127.03 15.637 126.774 14.692 126.774 13.613 L 126.774 4 L 129.906 4 L 129.906 13.346 C 129.906 13.91 130.028 14.411 130.274 14.85 C 130.524 15.287 130.874 15.632 131.329 15.882 C 131.781 16.134 132.31 16.259 132.912 16.259 C 133.519 16.259 134.047 16.134 134.494 15.882 C 134.948 15.632 135.297 15.287 135.542 14.85 C 135.793 14.41 135.92 13.91 135.92 13.346 Z M 150.805 10.866 L 147.982 11.041 C 147.935 10.802 147.828 10.578 147.672 10.391 C 147.503 10.186 147.287 10.025 147.042 9.921 C 146.753 9.794 146.44 9.732 146.125 9.741 C 145.648 9.741 145.245 9.842 144.917 10.045 C 144.59 10.242 144.426 10.508 144.426 10.841 C 144.424 11.102 144.542 11.349 144.745 11.512 C 144.957 11.695 145.32 11.842 145.835 11.953 L 147.845 12.358 C 148.925 12.579 149.73 12.936 150.259 13.427 C 150.791 13.919 151.055 14.565 151.055 15.364 C 151.055 16.092 150.841 16.731 150.41 17.28 C 149.987 17.829 149.404 18.257 148.661 18.566 C 147.924 18.869 147.075 19.022 146.111 19.022 C 144.64 19.022 143.47 18.715 142.597 18.105 C 141.751 17.524 141.196 16.608 141.071 15.589 L 144.101 15.429 C 144.192 15.877 144.415 16.218 144.766 16.457 C 145.119 16.687 145.57 16.803 146.118 16.803 C 146.657 16.803 147.09 16.7 147.418 16.493 C 147.75 16.28 147.92 16.007 147.925 15.675 C 147.924 15.403 147.792 15.147 147.57 14.989 C 147.339 14.806 146.982 14.666 146.5 14.569 L 144.578 14.186 C 143.494 13.968 142.687 13.593 142.156 13.058 C 141.632 12.525 141.368 11.84 141.368 11.013 C 141.368 10.3 141.562 9.687 141.947 9.17 C 142.337 8.655 142.884 8.256 143.588 7.976 C 144.296 7.696 145.125 7.558 146.075 7.558 C 147.476 7.558 148.579 7.854 149.385 8.446 C 150.195 9.04 150.666 9.848 150.801 10.868 Z M 158.056 19.02 C 156.913 19.02 155.929 18.789 155.106 18.326 C 154.294 17.866 153.636 17.178 153.213 16.346 C 152.768 15.487 152.547 14.474 152.547 13.303 C 152.547 12.161 152.77 11.159 153.213 10.295 C 153.632 9.459 154.281 8.759 155.083 8.279 C 155.893 7.799 156.843 7.557 157.931 7.557 C 158.664 7.557 159.347 7.675 159.978 7.91 C 160.614 8.141 161.168 8.491 161.639 8.959 C 162.118 9.426 162.488 10.014 162.753 10.722 C 163.018 11.425 163.15 12.249 163.15 13.193 L 163.15 14.04 L 153.777 14.04 L 153.777 12.132 L 160.252 12.132 C 160.26 11.721 160.161 11.315 159.964 10.954 C 159.777 10.617 159.499 10.339 159.162 10.151 C 158.803 9.948 158.396 9.846 157.983 9.854 C 157.515 9.854 157.1 9.963 156.74 10.18 C 156.388 10.386 156.098 10.684 155.9 11.041 C 155.696 11.405 155.59 11.816 155.591 12.233 L 155.591 14.047 C 155.591 14.596 155.692 15.071 155.894 15.471 C 156.1 15.871 156.392 16.179 156.769 16.397 C 157.145 16.612 157.591 16.723 158.106 16.723 C 158.425 16.726 158.742 16.676 159.045 16.577 C 159.597 16.4 160.036 15.978 160.236 15.434 L 163.085 15.624 C 162.95 16.288 162.643 16.906 162.196 17.415 C 161.725 17.94 161.136 18.346 160.477 18.6 C 159.777 18.88 158.971 19.02 158.056 19.02 Z"
          fill="rgb(255,255,255)"
        ></path>
      </g>
    </svg>
  );
}

function LaminarLogo({ className, fill, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      className={className}
      width="532"
      height="94"
      viewBox="0 0 532 94"
      fill="none"
      xmlns="http://www.w3.org/2000/svg"
      {...props}
    >
      <path
        fillRule="evenodd"
        clipRule="evenodd"
        d="M102.308 0.901855V91.9999H118.814V0.901855H102.308ZM142.637 89.1019C147.089 91.8739 152.087 93.2599 157.631 93.2599C161.999 93.2599 165.905 92.3779 169.349 90.6139C171.638 89.3853 173.612 87.8783 175.271 86.0929V91.9999H191.651V31.1419H175.271V36.9926C173.612 35.1888 171.638 33.7006 169.349 32.5279C165.905 30.7639 161.999 29.8819 157.631 29.8819C152.087 29.8819 147.089 31.2679 142.637 34.0399C138.185 36.8119 134.699 40.5919 132.179 45.3799C129.659 50.1679 128.399 55.5859 128.399 61.6339C128.399 67.5979 129.659 72.9739 132.179 77.7619C134.699 82.5499 138.185 86.3299 142.637 89.1019ZM171.869 73.4779C169.013 76.5019 165.275 78.0139 160.655 78.0139C157.631 78.0139 154.943 77.3419 152.591 75.9979C150.323 74.5699 148.517 72.6379 147.173 70.2019C145.913 67.6819 145.283 64.7839 145.283 61.5079C145.283 58.3159 145.913 55.5019 147.173 53.0659C148.517 50.5459 150.323 48.6139 152.591 47.2699C154.943 45.8419 157.631 45.1279 160.655 45.1279C163.763 45.1279 166.451 45.8419 168.719 47.2699C171.071 48.6139 172.877 50.5459 174.137 53.0659C175.481 55.5019 176.153 58.3159 176.153 61.5079C176.153 66.3799 174.725 70.3699 171.869 73.4779ZM204.993 31.1419V91.9999H221.499V56.0899C221.499 53.7379 222.003 51.7219 223.011 50.0419C224.019 48.3619 225.405 47.1019 227.169 46.2619C228.933 45.3379 230.907 44.8759 233.091 44.8759C236.367 44.8759 239.097 45.8839 241.281 47.8999C243.549 49.8319 244.683 52.5619 244.683 56.0899V91.9999H261.189V56.0899C261.189 53.7379 261.693 51.7219 262.701 50.0419C263.709 48.3619 265.095 47.1019 266.859 46.2619C268.707 45.3379 270.681 44.8759 272.781 44.8759C276.057 44.8759 278.787 45.8839 280.971 47.8999C283.239 49.8319 284.373 52.5619 284.373 56.0899V91.9999H300.879V53.4439C300.879 48.5719 299.829 44.3719 297.729 40.8439C295.629 37.3159 292.773 34.6279 289.161 32.7799C285.549 30.8479 281.475 29.8819 276.939 29.8819C272.403 29.8819 268.287 30.8899 264.591 32.9059C261.652 34.442 259.165 36.5091 257.129 39.1072C255.246 36.4377 252.861 34.3285 249.975 32.7799C246.531 30.8479 242.709 29.8819 238.509 29.8819C234.057 29.8819 230.067 30.8479 226.539 32.7799C224.656 33.7665 222.976 34.9805 221.499 36.422V31.1419H204.993ZM312.999 31.1419V91.9999H329.631V31.1419H312.999ZM314.637 18.6679C316.401 20.4319 318.627 21.3139 321.315 21.3139C324.087 21.3139 326.313 20.4319 327.993 18.6679C329.757 16.8199 330.639 14.5519 330.639 11.8639C330.639 9.25986 329.757 7.03386 327.993 5.18586C326.313 3.33786 324.087 2.41386 321.315 2.41386C318.627 2.41386 316.401 3.33786 314.637 5.18586C312.873 7.03386 311.991 9.25986 311.991 11.8639C311.991 14.5519 312.873 16.8199 314.637 18.6679ZM383.943 57.0979V91.9999H400.449V53.4439C400.449 49.3279 399.441 45.5059 397.425 41.9779C395.409 38.3659 392.637 35.4679 389.109 33.2839C385.581 31.0159 381.591 29.8819 377.139 29.8819C372.519 29.8819 368.361 30.8899 364.665 32.9059C362.725 33.9887 361.003 35.3018 359.499 36.8451V31.1419H342.993V91.9999H359.499V57.0979C359.499 54.6619 360.003 52.5199 361.011 50.6719C362.103 48.8239 363.573 47.3959 365.421 46.3879C367.269 45.3799 369.369 44.8759 371.721 44.8759C375.333 44.8759 378.273 46.0099 380.541 48.2779C382.809 50.5459 383.943 53.4859 383.943 57.0979ZM422.943 89.1019C427.395 91.8739 432.393 93.2599 437.937 93.2599C442.305 93.2599 446.211 92.3779 449.655 90.6139C451.944 89.3853 453.918 87.8784 455.577 86.093V91.9999H471.957V31.1419H455.577V36.9926C453.918 35.1888 451.944 33.7006 449.655 32.5279C446.211 30.7639 442.305 29.8819 437.937 29.8819C432.393 29.8819 427.395 31.2679 422.943 34.0399C418.491 36.8119 415.005 40.5919 412.485 45.3799C409.965 50.1679 408.705 55.5859 408.705 61.6339C408.705 67.5979 409.965 72.9739 412.485 77.7619C415.005 82.5499 418.491 86.3299 422.943 89.1019ZM452.175 73.4779C449.319 76.5019 445.581 78.0139 440.961 78.0139C437.937 78.0139 435.249 77.3419 432.897 75.9979C430.629 74.5699 428.823 72.6379 427.479 70.2019C426.219 67.6819 425.589 64.7839 425.589 61.5079C425.589 58.3159 426.219 55.5019 427.479 53.0659C428.823 50.5459 430.629 48.6139 432.897 47.2699C435.249 45.8419 437.937 45.1279 440.961 45.1279C444.069 45.1279 446.757 45.8419 449.025 47.2699C451.377 48.6139 453.183 50.5459 454.443 53.0659C455.787 55.5019 456.459 58.3159 456.459 61.5079C456.459 66.3799 455.031 70.3699 452.175 73.4779ZM485.299 31.1419V91.9999H501.805V58.3579C501.805 53.8219 502.897 50.4619 505.081 48.2779C507.349 46.0099 510.247 44.8759 513.775 44.8759C515.455 44.8759 516.925 45.1279 518.185 45.6319C519.529 46.1359 520.663 46.8919 521.587 47.8999L531.919 36.0559C529.987 33.8719 527.845 32.3179 525.493 31.3939C523.141 30.3859 520.495 29.8819 517.555 29.8819C510.751 29.8819 505.543 32.0659 501.931 36.4339C501.889 36.4839 501.847 36.5342 501.805 36.5847V31.1419H485.299ZM0.653968 84.1461C0.0802819 85.9866 0.00220402 88.0862 1.32507 89.4885C2.78376 91.0347 4.85185 91.9999 7.14535 91.9999H37.1454C58.1322 91.9999 75.1454 74.9867 75.1454 53.9999C75.1454 33.013 58.1322 15.9999 37.1454 15.9999H7.14535C4.56777 15.9999 2.27491 17.2189 0.811824 19.1119C-0.266346 20.5068 -0.129499 22.4002 0.408998 24.079C3.48464 33.6675 5.14534 43.8898 5.14534 54.4999C5.14534 64.8247 3.57273 74.7823 0.653968 84.1461Z"
        fill={fill}
      />
    </svg>
  );
}

function LaminarIcon({ className, fill, ...props }: React.ComponentProps<"svg">) {
  return (
    <svg
      className={className}
      width="76"
      height="76"
      viewBox="0 0 76 76"
      fill="none"
      xmlns="http://www.w3.org/2000/svg"
      {...props}
    >
      <path
        fillRule="evenodd"
        clipRule="evenodd"
        d="M1.32507 73.4886C0.00220402 72.0863 0.0802819 69.9867 0.653968 68.1462C3.57273 58.7824 5.14534 48.8249 5.14534 38.5C5.14534 27.8899 3.48464 17.6677 0.408998 8.0791C-0.129499 6.40029 -0.266346 4.50696 0.811824 3.11199C2.27491 1.21902 4.56777 0 7.14535 0H37.1454C58.1322 0 75.1454 17.0132 75.1454 38C75.1454 58.9868 58.1322 76 37.1454 76H7.14535C4.85185 76 2.78376 75.0349 1.32507 73.4886Z"
        fill={fill}
      />
    </svg>
  );
}

export {
  IconAmazonBedrock,
  IconAnthropic,
  IconAzure,
  IconBrowserUse,
  IconCrewAI,
  IconGemini,
  IconGitHub,
  IconGoogle,
  IconGroq,
  IconLangchain,
  IconLangGraph,
  IconMessage,
  IconMistral,
  IconOpenAI,
  IconOpenTelemetry,
  IconPlaywright,
  IconPython,
  IconResizeHandle,
  IconSpinner,
  IconVercel,
  LaminarIcon,
  LaminarLogo,
  LogoAxionRay,
  LogoBrowserUse,
  LogoRemo,
  LogoRye,
};
