"use client";

import { ChevronRightIcon } from "lucide-react";
import Link from "next/link";
import useSWR from "swr";

import { Skeleton } from "@/components/ui/skeleton";
import { setLastProjectIdCookie } from "@/lib/actions/project/cookies";
import { swrFetcher } from "@/lib/utils";
import { Project, ProjectStats } from "@/lib/workspaces/types";

interface ProjectCardProps {
  project: Project;
}

export const numberFormatter = new Intl.NumberFormat("en-US", {
  notation: "standard",
});

export default function ProjectCard({ project }: ProjectCardProps) {
  const { data, isLoading } = useSWR<ProjectStats>(`/api/projects/${project.id}/stats`, swrFetcher);

  const handleClick = async () => {
    await setLastProjectIdCookie(project.id);
  };

  return (
    <Link href={`/project/${project.id}/traces`} key={project.id} onClick={handleClick}>
      <div className="h-44 rounded-md bg-secondary hover:bg-muted border transition-all duration-100">
        <div className="p-4 flex flex-col justify-between h-full">
          <div className="flex items-center justify-between">
            <p className="text-lg font-medium">{project.name}</p>
            <ChevronRightIcon className="w-4 text-secondary-foreground" />
          </div>
          <div className="flex gap-8">
            {isLoading ? (
              <>
                <div className="flex flex-col w-full gap-2">
                  <Skeleton className="w-full h-6" />
                  <Skeleton className="w-full h-6" />
                </div>
              </>
            ) : (
              <>
                <div className="flex flex-col">
                  <p className="text-xs text-muted-foreground">Spans</p>
                  <p className="font-mono">{numberFormatter.format(data?.spansCount ?? 0)}</p>
                </div>
                <div className="flex flex-col">
                  <p className="text-xs text-muted-foreground">Evaluations</p>
                  <p className="font-mono">{numberFormatter.format(data?.evaluationsCount ?? 0)}</p>
                </div>
                <div className="flex flex-col">
                  <p className="text-xs text-muted-foreground">Datasets</p>
                  <p className="font-mono">{numberFormatter.format(data?.datasetsCount ?? 0)}</p>
                </div>
              </>
            )}
          </div>
        </div>
      </div>
    </Link>
  );
}
