"use client";

import { get } from "lodash";
import { useParams, useSearchParams } from "next/navigation";
import { useCallback, useMemo } from "react";

import {
  defaultPatternsColumnOrder,
  getColumns,
  PatternRow,
  patternsTableFilters,
} from "@/components/patterns/columns";
import { InfiniteDataTable } from "@/components/ui/infinite-datatable";
import { useInfiniteScroll } from "@/components/ui/infinite-datatable/hooks";
import { DataTableStateProvider } from "@/components/ui/infinite-datatable/model/datatable-store";
import ColumnsMenu from "@/components/ui/infinite-datatable/ui/columns-menu.tsx";
import DataTableFilter, { DataTableFilterList } from "@/components/ui/infinite-datatable/ui/datatable-filter";
import { DataTableSearch } from "@/components/ui/infinite-datatable/ui/datatable-search";
import RefreshButton from "@/components/ui/infinite-datatable/ui/refresh-button.tsx";
import { useToast } from "@/lib/hooks/use-toast";

export default function PatternsTable() {
  return (
    <DataTableStateProvider
      storageKey="patterns-table"
      uniqueKey="clusterId"
      defaultColumnOrder={defaultPatternsColumnOrder}
    >
      <PatternsTableContent />
    </DataTableStateProvider>
  );
}

function PatternsTableContent() {
  const { projectId } = useParams<{ projectId: string }>();
  const { toast } = useToast();
  const searchParams = useSearchParams();

  const columns = useMemo(() => getColumns(projectId), [projectId]);

  const filter = searchParams.getAll("filter");
  const search = searchParams.get("search");

  const FETCH_SIZE = 50;

  const fetchPatterns = useCallback(
    async (pageNumber: number) => {
      try {
        const urlParams = new URLSearchParams();
        urlParams.set("pageNumber", pageNumber.toString());
        urlParams.set("pageSize", FETCH_SIZE.toString());

        filter.forEach((f) => urlParams.append("filter", f));

        if (typeof search === "string" && search.length > 0) {
          urlParams.set("search", search);
        }

        const url = `/api/projects/${projectId}/patterns?${urlParams.toString()}`;

        const res = await fetch(url, {
          method: "GET",
          headers: {
            "Content-Type": "application/json",
          },
        });

        if (!res.ok) {
          const text = (await res.json()) as { error: string };
          throw new Error(text.error);
        }

        const data = (await res.json()) as { items: PatternRow[] };
        return { items: data.items };
      } catch (error) {
        toast({
          title: error instanceof Error ? error.message : "Failed to load patterns. Please try again.",
          variant: "destructive",
        });
        throw error;
      }
    },
    [projectId, toast, filter, search]
  );

  const {
    data: rawPatterns,
    hasMore,
    isFetching,
    isLoading,
    fetchNextPage,
    refetch,
    error,
  } = useInfiniteScroll<PatternRow>({
    fetchFn: fetchPatterns,
    enabled: true,
    deps: [projectId, filter, search],
  });

  const patterns = useMemo(() => {
    if (!rawPatterns) return [];

    if (filter.length > 0 || (search && search.length > 0)) {
      return rawPatterns.map((pattern) => ({
        ...pattern,
        subRows: [],
      }));
    }

    const patternMap = new Map<string, PatternRow>();
    const rootPatterns: PatternRow[] = [];

    rawPatterns.forEach((pattern) => {
      patternMap.set(pattern.clusterId, { ...pattern, subRows: [] });
    });

    rawPatterns.forEach((pattern) => {
      const node = patternMap.get(pattern.clusterId);
      if (!node) return;

      if (pattern.parentId === null) {
        rootPatterns.push(node);
      } else {
        const parent = patternMap.get(pattern.parentId);
        if (parent) {
          if (!parent.subRows) parent.subRows = [];
          parent.subRows.push(node);
        }
      }
    });

    return rootPatterns;
  }, [rawPatterns, filter, search]);

  return (
    <div className="flex overflow-hidden px-4 pb-6">
      <InfiniteDataTable<PatternRow>
        className="w-full"
        columns={columns}
        data={patterns}
        getRowId={(pattern) => get(pattern, ["clusterId"], pattern.clusterId)}
        hasMore={hasMore}
        isFetching={isFetching}
        isLoading={isLoading}
        fetchNextPage={fetchNextPage}
        error={error}
      >
        <div className="flex flex-1 w-full space-x-2 pt-1">
          <DataTableFilter columns={patternsTableFilters} />
          <ColumnsMenu
            columnLabels={columns.map((column) => ({
              id: column.id!,
              label: typeof column.header === "string" ? column.header : column.id!,
            }))}
            lockedColumns={["expand"]}
          />
          <RefreshButton onClick={refetch} variant="outline" />
          <DataTableSearch className="mr-0.5" placeholder="Search by pattern name..." />
        </div>
        <DataTableFilterList />
      </InfiniteDataTable>
    </div>
  );
}
