## Copyright 2018 Istio Authors
##
## Licensed under the Apache License, Version 2.0 (the "License");
## you may not use this file except in compliance with the License.
## You may obtain a copy of the License at
##
##     http://www.apache.org/licenses/LICENSE-2.0
##
## Unless required by applicable law or agreed to in writing, software
## distributed under the License is distributed on an "AS IS" BASIS,
## WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
## See the License for the specific language governing permissions and
## limitations under the License.

.PHONY: docker
.PHONY: docker.all
.PHONY: docker.save
.PHONY: docker.push

# Docker target will build the go binaries and package the docker for local testing.
# It does not upload to a registry.
docker: docker.all

# Add new docker targets to the end of the DOCKER_TARGETS list.

DOCKER_TARGETS ?= docker.pilot docker.proxytproxy docker.proxyv2 docker.app docker.app_sidecar docker.test_policybackend \
	docker.mixer docker.mixer_codegen docker.citadel docker.galley docker.sidecar_injector docker.kubectl docker.node-agent-k8s \
	docker.istioctl docker.operator

$(ISTIO_DOCKER) $(ISTIO_DOCKER_TAR):
	mkdir -p $@

.SECONDEXPANSION: #allow $@ to be used in dependency list

# generated content
$(ISTIO_DOCKER)/istio_ca.crt $(ISTIO_DOCKER)/istio_ca.key: ${GEN_CERT} | ${ISTIO_DOCKER}
	${GEN_CERT} --key-size=2048 --out-cert=${ISTIO_DOCKER}/istio_ca.crt \
                    --out-priv=${ISTIO_DOCKER}/istio_ca.key --organization="k8s.cluster.local" \
                    --mode=self-signed --ca=true
$(ISTIO_DOCKER)/node_agent.crt $(ISTIO_DOCKER)/node_agent.key: ${GEN_CERT} $(ISTIO_DOCKER)/istio_ca.crt $(ISTIO_DOCKER)/istio_ca.key
	${GEN_CERT} --key-size=2048 --out-cert=${ISTIO_DOCKER}/node_agent.crt \
                    --out-priv=${ISTIO_DOCKER}/node_agent.key --organization="NodeAgent" \
		    --mode=signer --host="nodeagent.google.com" --signer-cert=${ISTIO_DOCKER}/istio_ca.crt \
                    --signer-priv=${ISTIO_DOCKER}/istio_ca.key

# directives to copy files to docker scratch directory

# tell make which files are copied from $(ISTIO_OUT_LINUX) and generate rules to copy them to the proper location:
# generates rules like the following:
# $(ISTIO_DOCKER)/pilot-agent: $(ISTIO_OUT_LINUX)/pilot-agent | $(ISTIO_DOCKER)
# 	cp $(ISTIO_OUT_LINUX)/$FILE $(ISTIO_DOCKER)/($FILE)
DOCKER_FILES_FROM_ISTIO_OUT_LINUX:=client server \
                             pilot-discovery pilot-agent sidecar-injector mixs mixgen \
                             istio_ca node_agent node_agent_k8s galley istio-iptables istio-clean-iptables istioctl manager
$(foreach FILE,$(DOCKER_FILES_FROM_ISTIO_OUT_LINUX), \
        $(eval $(ISTIO_DOCKER)/$(FILE): $(ISTIO_OUT_LINUX)/$(FILE) | $(ISTIO_DOCKER); cp $(ISTIO_OUT_LINUX)/$(FILE) $(ISTIO_DOCKER)/$(FILE)))

# rule for the test certs.
$(ISTIO_DOCKER)/certs:
	mkdir -p $(ISTIO_DOCKER)
	cp -a tests/testdata/certs $(ISTIO_DOCKER)/.

# tell make which files are copied from the source tree and generate rules to copy them to the proper location:
# TODO(sdake)                      $(NODE_AGENT_TEST_FILES) $(GRAFANA_FILES)
DOCKER_FILES_FROM_SOURCE:=tests/testdata/certs/cert.crt tests/testdata/certs/cert.key tests/testdata/certs/cacert.pem
$(foreach FILE,$(DOCKER_FILES_FROM_SOURCE), \
        $(eval $(ISTIO_DOCKER)/$(notdir $(FILE)): $(FILE) | $(ISTIO_DOCKER); cp $(FILE) $$(@D)))


# tell make which files are copied from ISTIO_BIN and generate rules to copy them to the proper location:
# generates rules like the following:
# $(ISTIO_DOCKER)/kubectl: $(ISTIO_BIN)/kubectl | $(ISTIO_DOCKER)
# 	cp $(ISTIO_BIN)/kubectl $(ISTIO_DOCKER)/kubectl
DOCKER_FILES_FROM_ISTIO_BIN:=kubectl
$(foreach FILE,$(DOCKER_FILES_FROM_ISTIO_BIN), \
        $(eval $(ISTIO_DOCKER)/$(FILE): $(ISTIO_BIN)/$(FILE) | $(ISTIO_DOCKER); cp $(ISTIO_BIN)/$(FILE) $(ISTIO_DOCKER)/$(FILE)))

docker.sidecar_injector: BUILD_PRE=&& chmod 755 sidecar-injector
docker.sidecar_injector: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.sidecar_injector: sidecar-injector/docker/Dockerfile.sidecar_injector
docker.sidecar_injector:$(ISTIO_DOCKER)/sidecar-injector
	$(DOCKER_RULE)

# BUILD_PRE tells $(DOCKER_RULE) to run the command specified before executing a docker build
# BUILD_ARGS tells  $(DOCKER_RULE) to execute a docker build with the specified commands

# The file must be named 'envoy', depends on the release.
${ISTIO_ENVOY_LINUX_RELEASE_DIR}/envoy: ${ISTIO_ENVOY_LINUX_RELEASE_PATH}
	mkdir -p $(DOCKER_BUILD_TOP)/proxyv2
ifdef DEBUG_IMAGE
	cp ${ISTIO_ENVOY_LINUX_DEBUG_PATH} ${ISTIO_ENVOY_LINUX_RELEASE_DIR}/envoy
else
	cp ${ISTIO_ENVOY_LINUX_RELEASE_PATH} ${ISTIO_ENVOY_LINUX_RELEASE_DIR}/envoy
endif

# rule for wasm extensions.
$(ISTIO_ENVOY_LINUX_RELEASE_DIR)/stats-filter.wasm: init
$(ISTIO_ENVOY_LINUX_RELEASE_DIR)/metadata-exchange-filter.wasm: init

# Default proxy image.
docker.proxyv2: BUILD_PRE=&& chmod 755 envoy pilot-agent istio-iptables
docker.proxyv2: BUILD_ARGS=--build-arg proxy_version=istio-proxy:${PROXY_REPO_SHA} --build-arg istio_version=${VERSION} --build-arg BASE_VERSION=${BASE_VERSION}
docker.proxyv2: tools/packaging/common/envoy_bootstrap_v2.json
docker.proxyv2: install/gcp/bootstrap/gcp_envoy_bootstrap.json
docker.proxyv2: $(ISTIO_ENVOY_LINUX_RELEASE_DIR)/envoy
docker.proxyv2: $(ISTIO_OUT_LINUX)/pilot-agent
docker.proxyv2: pilot/docker/Dockerfile.proxyv2
docker.proxyv2: pilot/docker/envoy_pilot.yaml.tmpl
docker.proxyv2: pilot/docker/envoy_policy.yaml.tmpl
docker.proxyv2: pilot/docker/envoy_telemetry.yaml.tmpl
docker.proxyv2: $(ISTIO_DOCKER)/istio-iptables
docker.proxyv2: $(ISTIO_ENVOY_LINUX_RELEASE_DIR)/stats-filter.wasm
docker.proxyv2: $(ISTIO_ENVOY_LINUX_RELEASE_DIR)/metadata-exchange-filter.wasm
	$(DOCKER_RULE)

# Proxy using TPROXY interception - but no core dumps
docker.proxytproxy: BUILD_ARGS=--build-arg proxy_version=istio-proxy:${PROXY_REPO_SHA} --build-arg istio_version=${VERSION} --build-arg BASE_VERSION=${BASE_VERSION}
docker.proxytproxy: tools/packaging/common/envoy_bootstrap_v2.json
docker.proxytproxy: install/gcp/bootstrap/gcp_envoy_bootstrap.json
docker.proxytproxy: $(ISTIO_ENVOY_LINUX_RELEASE_DIR)/envoy
docker.proxytproxy: $(ISTIO_OUT_LINUX)/pilot-agent
docker.proxytproxy: pilot/docker/Dockerfile.proxytproxy
docker.proxytproxy: pilot/docker/envoy_pilot.yaml.tmpl
docker.proxytproxy: pilot/docker/envoy_policy.yaml.tmpl
docker.proxytproxy: pilot/docker/envoy_telemetry.yaml.tmpl
docker.proxytproxy: $(ISTIO_DOCKER)/istio-iptables
	$(DOCKER_RULE)

docker.pilot: BUILD_PRE=&& chmod 755 pilot-discovery cacert.pem
docker.pilot: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.pilot: $(ISTIO_OUT_LINUX)/pilot-discovery
docker.pilot: tests/testdata/certs/cacert.pem
docker.pilot: pilot/docker/Dockerfile.pilot
	$(DOCKER_RULE)

# Test application
docker.app: BUILD_PRE=&& chmod 755 server client
docker.app: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.app: pkg/test/echo/docker/Dockerfile.app
docker.app: $(ISTIO_OUT_LINUX)/client
docker.app: $(ISTIO_OUT_LINUX)/server
docker.app: $(ISTIO_DOCKER)/certs
	$(DOCKER_RULE)


# Test application bundled with the sidecar (for non-k8s).
docker.app_sidecar: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.app_sidecar: tools/packaging/common/envoy_bootstrap_v2.json
docker.app_sidecar: $(ISTIO_OUT_LINUX)/release/istio-sidecar.deb
docker.app_sidecar: $(ISTIO_DOCKER)/certs
docker.app_sidecar: pkg/test/echo/docker/echo-start.sh
docker.app_sidecar: $(ISTIO_OUT_LINUX)/client
docker.app_sidecar: $(ISTIO_OUT_LINUX)/server
docker.app_sidecar: pkg/test/echo/docker/Dockerfile.app_sidecar
	$(DOCKER_RULE)

# Test policy backend for mixer integration
docker.test_policybackend: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.test_policybackend: mixer/docker/Dockerfile.test_policybackend
docker.test_policybackend: $(ISTIO_OUT_LINUX)/policybackend
	$(DOCKER_RULE)

docker.kubectl: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.kubectl: docker/Dockerfile.kubectl
	$(DOCKER_RULE)

docker.istioctl: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.istioctl: istioctl/docker/Dockerfile.istioctl
docker.istioctl: $(ISTIO_OUT_LINUX)/istioctl
	$(DOCKER_RULE)

docker.operator: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.operator: operator/docker/Dockerfile.operator
docker.operator: $(ISTIO_OUT_LINUX)/operator
	$(DOCKER_RULE)

# mixer docker images

docker.mixer: BUILD_PRE=&& chmod 755 mixs
docker.mixer: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.mixer: mixer/docker/Dockerfile.mixer
docker.mixer: $(ISTIO_DOCKER)/mixs
	$(DOCKER_RULE)

# mixer codegen docker images
docker.mixer_codegen: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.mixer_codegen: mixer/docker/Dockerfile.mixer_codegen
docker.mixer_codegen: $(ISTIO_DOCKER)/mixgen
	$(DOCKER_RULE)

.PHONY: dockerx dockerx.save

# Docker has an experimental new build engine, https://github.com/docker/buildx
# This brings substantial (10x) performance improvements when building Istio
# However, its only built into docker since v19.03. Because its so new that devs are likely to not have
# this version, and because its experimental, this is not the default build method. As this matures we should migrate over.
# For performance, in CI this method is used.
# This target works by reusing the existing docker methods. Each docker target declares it's dependencies.
# We then override the docker rule and "build" all of these, where building just copies the dependencies
# We then generate a "bake" file, which defines all of the docker files in the repo
# Finally, we call `docker buildx bake` to generate the images.
dockerx: DOCKER_RULE?=mkdir -p $(DOCKERX_BUILD_TOP)/$@ && cp -r $^ $(DOCKERX_BUILD_TOP)/$@ && cd $(DOCKERX_BUILD_TOP)/$@ $(BUILD_PRE)
dockerx: docker | $(ISTIO_DOCKER_TAR)
dockerx:
	HUB=$(HUB) \
		TAG=$(TAG) \
		PROXY_REPO_SHA=$(PROXY_REPO_SHA) \
		VERSION=$(VERSION) \
		DOCKER_ALL_VARIANTS="$(DOCKER_ALL_VARIANTS)" \
		ISTIO_DOCKER_TAR=$(ISTIO_DOCKER_TAR) \
		BASE_VERSION=$(BASE_VERSION) \
		DOCKERX_PUSH=$(DOCKERX_PUSH) \
		./tools/buildx-gen.sh $(DOCKERX_BUILD_TOP) $(DOCKER_TARGETS)
	DOCKER_CLI_EXPERIMENTAL=enabled docker buildx bake -f $(DOCKERX_BUILD_TOP)/docker-bake.hcl $(DOCKER_BUILD_VARIANTS)

# Support individual images like `dockerx.pilot`
dockerx.%:
	@DOCKER_TARGETS=docker.$* BUILD_ALL=false $(MAKE) --no-print-directory -f Makefile.core.mk dockerx

# galley docker images
docker.galley: BUILD_PRE=&& chmod 755 galley
docker.galley: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.galley: galley/docker/Dockerfile.galley
docker.galley: $(ISTIO_DOCKER)/galley
	$(DOCKER_RULE)

# security docker images

docker.citadel: BUILD_PRE=&& chmod 755 istio_ca
docker.citadel: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.citadel: security/docker/Dockerfile.citadel
docker.citadel: $(ISTIO_DOCKER)/istio_ca
	$(DOCKER_RULE)

docker.citadel-test: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.citadel-test: security/docker/Dockerfile.citadel-test
docker.citadel-test: $(ISTIO_DOCKER)/istio_ca
docker.citadel-test: $(ISTIO_DOCKER)/istio_ca.crt
docker.citadel-test: $(ISTIO_DOCKER)/istio_ca.key
	$(DOCKER_RULE)

docker.node-agent: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.node-agent: security/docker/Dockerfile.node-agent
docker.node-agent: $(ISTIO_DOCKER)/node_agent
	$(DOCKER_RULE)

docker.node-agent-k8s: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.node-agent-k8s: security/docker/Dockerfile.node-agent-k8s
docker.node-agent-k8s: $(ISTIO_DOCKER)/node_agent_k8s
	$(DOCKER_RULE)

docker.node-agent-test: BUILD_ARGS=--build-arg BASE_VERSION=${BASE_VERSION}
docker.node-agent-test: security/docker/Dockerfile.node-agent-test
docker.node-agent-test: $(ISTIO_DOCKER)/node_agent
docker.node-agent-test: $(ISTIO_DOCKER)/istio_ca.crt
docker.node-agent-test: $(ISTIO_DOCKER)/node_agent.crt
docker.node-agent-test: $(ISTIO_DOCKER)/node_agent.key
	$(DOCKER_RULE)

docker.base: docker/Dockerfile.base
	$(DOCKER_RULE)

# $@ is the name of the target
# $^ the name of the dependencies for the target
# Rule Steps #
##############
# 1. Make a directory $(DOCKER_BUILD_TOP)/%@
# 2. This rule uses cp to copy all dependency filenames into into $(DOCKER_BUILD_TOP/$@
# 3. This rule then changes directories to $(DOCKER_BUID_TOP)/$@
# 4. This rule runs $(BUILD_PRE) prior to any docker build and only if specified as a dependency variable
# 5. This rule finally runs docker build passing $(BUILD_ARGS) to docker if they are specified as a dependency variable

# DOCKER_BUILD_VARIANTS ?=default distroless
DOCKER_BUILD_VARIANTS ?= default
DOCKER_ALL_VARIANTS ?= default distroless
DEFAULT_DISTRIBUTION=default
DOCKER_RULE ?= $(foreach VARIANT,$(DOCKER_BUILD_VARIANTS), time (mkdir -p $(DOCKER_BUILD_TOP)/$@ && cp -r $^ $(DOCKER_BUILD_TOP)/$@ && cd $(DOCKER_BUILD_TOP)/$@ $(BUILD_PRE) && docker build $(BUILD_ARGS) --build-arg BASE_DISTRIBUTION=$(VARIANT) -t $(HUB)/$(subst docker.,,$@):$(subst -$(DEFAULT_DISTRIBUTION),,$(TAG)-$(VARIANT)) -f Dockerfile$(suffix $@) . ); )

# This target will package all docker images used in test and release, without re-building
# go binaries. It is intended for CI/CD systems where the build is done in separate job.
docker.all: $(DOCKER_TARGETS)

# for each docker.XXX target create a tar.docker.XXX target that says how
# to make a $(ISTIO_OUT_LINUX)/docker/XXX.tar.gz from the docker XXX image
# note that $(subst docker.,,$(TGT)) strips off the "docker." prefix, leaving just the XXX

# create a DOCKER_TAR_TARGETS that's each of DOCKER_TARGETS with a tar. prefix
DOCKER_TAR_TARGETS:=
$(foreach TGT,$(DOCKER_TARGETS),$(eval tar.$(TGT): $(TGT) | $(ISTIO_DOCKER_TAR) ; \
         $(foreach VARIANT,$(DOCKER_BUILD_VARIANTS), time ( \
		     docker save -o ${ISTIO_DOCKER_TAR}/$(subst docker.,,$(TGT))$(subst -$(DEFAULT_DISTRIBUTION),,-$(VARIANT)).tar $(HUB)/$(subst docker.,,$(TGT)):$(subst -$(DEFAULT_DISTRIBUTION),,$(TAG)-$(VARIANT)) && \
             gzip ${ISTIO_DOCKER_TAR}/$(subst docker.,,$(TGT))$(subst -$(DEFAULT_DISTRIBUTION),,-$(VARIANT)).tar \
			   ); \
		  )))

# create a DOCKER_TAR_TARGETS that's each of DOCKER_TARGETS with a tar. prefix DOCKER_TAR_TARGETS:=
$(foreach TGT,$(DOCKER_TARGETS),$(eval DOCKER_TAR_TARGETS+=tar.$(TGT)))

# this target saves a tar.gz of each docker image to ${ISTIO_OUT_LINUX}/docker/
dockerx.save: dockerx $(ISTIO_DOCKER_TAR)
	$(foreach TGT,$(DOCKER_TARGETS), \
	$(foreach VARIANT,$(DOCKER_BUILD_VARIANTS), time ( \
		 docker save -o ${ISTIO_DOCKER_TAR}/$(subst docker.,,$(TGT))$(subst -$(DEFAULT_DISTRIBUTION),,-$(VARIANT)).tar $(HUB)/$(subst docker.,,$(TGT)):$(subst -$(DEFAULT_DISTRIBUTION),,$(TAG)-$(VARIANT)) && \
		 gzip -f ${ISTIO_DOCKER_TAR}/$(subst docker.,,$(TGT))$(subst -$(DEFAULT_DISTRIBUTION),,-$(VARIANT)).tar \
		   ); \
	 ))

docker.save: $(DOCKER_TAR_TARGETS)

# for each docker.XXX target create a push.docker.XXX target that pushes
# the local docker image to another hub
# a possible optimization is to use tag.$(TGT) as a dependency to do the tag for us
$(foreach TGT,$(DOCKER_TARGETS),$(eval push.$(TGT): | $(TGT) ; \
	time (set -e && for distro in $(DOCKER_BUILD_VARIANTS); do tag=$(TAG)-$$$${distro}; docker push $(HUB)/$(subst docker.,,$(TGT)):$$$${tag%-$(DEFAULT_DISTRIBUTION)}; done)))

define run_vulnerability_scanning
        $(eval RESULTS_DIR := vulnerability_scan_results)
        $(eval CURL_RESPONSE := $(shell curl -s --create-dirs -o $(RESULTS_DIR)/$(1) -w "%{http_code}" http://imagescanner.cloud.ibm.com/scan?image="docker.io/$(2)")) \
        $(if $(filter $(CURL_RESPONSE), 200), (mv $(RESULTS_DIR)/$(1) $(RESULTS_DIR)/$(1).json))
endef

# create a DOCKER_PUSH_TARGETS that's each of DOCKER_TARGETS with a push. prefix
DOCKER_PUSH_TARGETS:=
$(foreach TGT,$(DOCKER_TARGETS),$(eval DOCKER_PUSH_TARGETS+=push.$(TGT)))

# Will build and push docker images.
docker.push: $(DOCKER_PUSH_TARGETS)

# Build and push docker images using dockerx
dockerx.push: dockerx
	$(foreach TGT,$(DOCKER_TARGETS), time ( \
		set -e && for distro in $(DOCKER_BUILD_VARIANTS); do tag=$(TAG)-$${distro}; docker push $(HUB)/$(subst docker.,,$(TGT)):$${tag%-$(DEFAULT_DISTRIBUTION)}; done); \
	)

# Build and push docker images using dockerx. Pushing is done inline as an optimization
# This is not done in the dockerx.push target because it requires using the docker-container driver.
# See https://github.com/docker/buildx#working-with-builder-instances for info to set this up
dockerx.pushx: DOCKERX_PUSH=true
dockerx.pushx: dockerx
	@:

# Scan images for security vulnerabilities using the ImageScanner tool
docker.scan_images: $(DOCKER_PUSH_TARGETS)
	$(foreach TGT,$(DOCKER_TARGETS),$(call run_vulnerability_scanning,$(subst docker.,,$(TGT)),$(HUB)/$(subst docker.,,$(TGT)):$(TAG)))
