package gorgonia

import (
	"fmt"
	"math/rand"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"gorgonia.org/dawson"
	"gorgonia.org/tensor"
)

var im2colTests = []struct {
	kernel   tensor.Shape
	pad      tensor.Shape
	stride   tensor.Shape
	dilation tensor.Shape
}{
	{tensor.Shape{4, 4}, tensor.Shape{0, 0}, tensor.Shape{1, 1}, tensor.Shape{1, 1}},
	{tensor.Shape{3, 3}, tensor.Shape{1, 1}, tensor.Shape{2, 2}, tensor.Shape{1, 1}},
	{tensor.Shape{3, 3}, tensor.Shape{1, 1}, tensor.Shape{3, 3}, tensor.Shape{1, 1}},
}

func im2colTest(t *testing.T, dt tensor.Dtype, kernel, pad, stride, dilation tensor.Shape) {
	assert := assert.New(t)
	g := NewGraph()
	x := NewTensor(g, dt, 4, WithShape(2, 1, 28, 28), WithInit(RangedFrom(0))) // mnist, in batches of 10
	y, err := Im2Col(x, kernel, pad, stride, dilation)
	if err != nil {
		t.Error(err)
		return
	}
	cost := Must(Sum(y))

	grads, err := Grad(cost, x)
	if err != nil {
		t.Errorf("error while Grad(): %v", err)
		return
	}

	m := NewTapeMachine(g, BindDualValues())
	defer m.Close()
	if err := m.RunAll(); err != nil {
		t.Error(err)
		return
	}
	// t.Logf("x: %v", x.Value())
	// t.Logf("c: %3.3f", cost.Value())
	// t.Logf("xG: %v", grads[0].Value())

	h := NewGraph()
	a := NewTensor(h, dt, 4, WithShape(2, 1, 28, 28), WithInit(RangedFrom(0)))
	b, err := Im2Col(a, kernel, pad, stride, dilation)
	if err != nil {
		t.Error(err)
		return
	}
	cost2 := Must(Sum(b))
	n := NewLispMachine(h)
	defer n.Close()
	if err = n.RunAll(); err != nil {
		t.Error(err)
		return
	}
	aG, err := a.Grad()
	if err != nil {
		t.Error(err)
		return
	}

	// t.Logf("a: %v", a.Value())
	// t.Logf("c: %3.3f", cost2.Value())
	// t.Logf("aG: %v", aG)

	assert.Equal(x.Value().Data(), a.Value().Data())
	assert.Equal(grads[0].Value().Data(), aG.Data())
	assert.Equal(cost.Value().Data(), cost2.Value().Data())
}

func TestIm2Col(t *testing.T) {
	// assert := assert.New(t)
	dts := []tensor.Dtype{tensor.Float64, tensor.Float32}
	for _, dt := range dts {
		for _, i2ct := range im2colTests {
			im2colTest(t, dt, i2ct.kernel, i2ct.pad, i2ct.stride, i2ct.dilation)
		}
	}
}

func TestMaxPool2D_asymmetricPadding(t *testing.T) {
	assert := assert.New(t)
	tsts := []struct {
		inputT  tensor.Tensor
		outputT tensor.Tensor
		padding []int
	}{
		{
			inputT: tensor.New(tensor.WithShape(1, 3, 32, 32),
				tensor.WithBacking([]float32{1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558, -0.9772779, 0.95008844, -0.1513572, -0.10321885, 0.41059852, 0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324, 0.33367434, 1.4940791, -0.20515826, 0.3130677, -0.85409576, -2.5529897, 0.6536186, 0.8644362, -0.742165, 2.2697546, -1.4543657, 0.045758516, -0.18718386, 1.5327792, 1.4693588, 0.15494743, 0.37816253, -0.88778573, -1.9807965, -0.34791216, 0.15634897, 1.2302907, 1.2023798, -0.3873268, -0.30230275, -1.048553, -1.420018, -1.7062702, 1.9507754, -0.5096522, -0.4380743, -1.2527953, 0.7774904, -1.6138978, -0.21274029, -0.89546657, 0.3869025, -0.51080513, -1.1806322, -0.028182229, 0.42833188, 0.06651722, 0.3024719, -0.6343221, -0.36274117, -0.67246044, -0.35955316, -0.8131463, -1.7262826, 0.17742614, -0.40178093, -1.6301984, 0.46278226, -0.9072984, 0.051945396, 0.7290906, 0.12898292, 1.1394007, -1.2348258, 0.40234163, -0.6848101, -0.87079716, -0.5788497, -0.31155252, 0.05616534, -1.1651498, 0.9008265, 0.46566245, -1.5362437, 1.4882522, 1.8958892, 1.1787796, -0.17992483, -1.0707526, 1.0544517, -0.40317693, 1.222445, 0.20827498, 0.97663903, 0.3563664, 0.7065732, 0.01050002, 1.7858706, 0.12691209, 0.40198937, 1.8831507, -1.347759, -1.270485, 0.9693967, -1.1731234, 1.9436212, -0.41361898, -0.7474548, 1.922942, 1.4805148, 1.867559, 0.90604466, -0.86122566, 1.9100649, -0.26800337, 0.8024564, 0.947252, -0.15501009, 0.61407936, 0.9222067, 0.37642553, -1.0994008, 0.2982382, 1.3263859, -0.69456786, -0.14963454, -0.43515354, 1.8492638, 0.67229474, 0.40746182, -0.76991606, 0.5392492, -0.6743327, 0.031830557, -0.6358461, 0.67643327, 0.57659084, -0.20829876, 0.3960067, -1.0930616, -1.4912575, 0.4393917, 0.1666735, 0.63503146, 2.3831449, 0.94447947, -0.91282225, 1.1170163, -1.3159074, -0.4615846, -0.0682416, 1.7133427, -0.74475485, -0.82643855, -0.09845252, -0.6634783, 1.1266359, -1.0799315, -1.1474687, -0.43782005, -0.49803245, 1.929532, 0.9494208, 0.08755124, -1.2254355, 0.844363, -1.0002153, -1.5447711, 1.1880298, 0.3169426, 0.9208588, 0.31872764, 0.8568306, -0.6510256, -1.0342429, 0.6815945, -0.80340964, -0.6895498, -0.4555325, 0.017479159, -0.35399392, -1.3749512, -0.6436184, -2.2234032, 0.62523144, -1.6020577, -1.1043833, 0.05216508, -0.739563, 1.5430146, -1.2928569, 0.26705086, -0.039282817, -1.1680934, 0.5232767, -0.17154633, 0.77179056, 0.82350415, 2.163236, 1.336528, -0.36918184, -0.23937918, 1.0996596, 0.6552637, 0.64013153, -1.616956, -0.024326125, -0.7380309, 0.2799246, -0.09815039, 0.9101789, 0.3172182, 0.78632796, -0.4664191, -0.94444627, -0.4100497, -0.017020414, 0.37915173, 2.259309, -0.042257152, -0.955945, -0.34598178, -0.463596, 0.48148146, -1.540797, 0.06326199, 0.15650654, 0.23218104, -0.5973161, -0.23792173, -1.424061, -0.49331987, -0.54286146, 0.41605005, -1.1561824, 0.7811981, 1.4944845, -2.069985, 0.42625874, 0.676908, -0.63743705, -0.3972718, -0.13288058, -0.29779088, -0.30901298, -1.6760038, 1.1523316, 1.0796186, -0.81336427, -1.4664243, 0.5210649, -0.57578796, 0.14195317, -0.31932843, 0.69153875, 0.6947491, -0.7255974, -1.383364, -1.5829384, 0.6103794, -1.1888592, -0.5068163, -0.596314, -0.052567296, -1.9362798, 0.1887786, 0.52389103, 0.08842209, -0.31088617, 0.097400166, 0.39904633, -2.7725928, 1.9559124, 0.39009333, -0.6524086, -0.39095336, 0.49374178, -0.11610394, -2.0306845, 2.064493, -0.11054066, 1.0201727, -0.69204986, 1.5363771, 0.2863437, 0.60884386, -1.0452534, 1.2111453, 0.68981814, 1.3018463, -0.6280876, -0.48102713, 2.3039167, -1.0600158, -0.1359497, 1.1368914, 0.09772497, 0.5829537, -0.39944902, 0.37005588, -1.3065269, 1.6581306, -0.11816405, -0.6801782, 0.6663831, -0.4607198, -1.3342584, -1.3467175, 0.69377315, -0.15957344, -0.13370156, 1.0777438, -1.1268258, -0.7306777, -0.3848798, 0.09435159, -0.042171452, -0.2868872, -0.0616264, -0.10730527, -0.7196044, -0.812993, 0.27451634, -0.8909151, -1.1573553, -0.31229225, -0.15766701, 2.2567234, -0.7047003, 0.9432607, 0.7471883, -1.1889449, 0.77325296, -1.1838807, -2.6591723, 0.60631955, -1.7558906, 0.45093447, -0.6840109, 1.6595508, 1.0685093, -0.4533858, -0.6878376, -1.2140774, -0.44092262, -0.28035548, -0.36469355, 0.15670386, 0.5785215, 0.34965447, -0.76414394, -1.4377915, 1.3645319, -0.6894492, -0.6522936, -0.52118933, -1.8430696, -0.477974, -0.4796558, 0.6203583, 0.6984571, 0.003770889, 0.93184835, 0.339965, -0.015682112, 0.16092817, -0.19065349, -0.3948495, -0.26773354, -1.1280113, 0.2804417, -0.9931236, 0.8416313, -0.24945858, 0.04949498, 0.4938368, 0.6433145, -1.5706234, -0.20690368, 0.8801789, -1.6981058, 0.38728046, -2.2555642, -1.0225068, 0.038630553, -1.6567152, -0.98551077, -1.471835, 1.648135, 0.16422775, 0.5672903, -0.2226751, -0.35343176, -1.6164742, -0.29183736, -0.7614922, 0.8579239, 1.1411018, 1.4665787, 0.85255194, -0.5986539, -1.1158969, 0.7666632, 0.3562928, -1.7685385, 0.3554818, 0.8145198, 0.058925588, -0.18505368, -0.8076485, -1.4465348, 0.800298, -0.30911446, -0.23346666, 1.7327212, 0.6845011, 0.370825, 0.1420618, 1.5199949, 1.7195894, 0.9295051, 0.5822246, -2.094603, 0.12372191, -0.13010696, 0.09395323, 0.9430461, -2.7396772, -0.56931204, 0.26990435, -0.46684554, -1.4169061, 0.8689635, 0.27687192, -0.97110456, 0.3148172, 0.8215857, 0.005292646, 0.8005648, 0.078260176, -0.39522898, -1.1594205, -0.085930765, 0.19429293, 0.87583274, -0.11510747, 0.4574156, -0.964612, -0.78262913, -0.1103893, -1.0546285, 0.8202478, 0.46313033, 0.27909577, 0.3389041, 2.0210435, -0.4688642, -2.2014413, 0.1993002, -0.050603542, -0.51751906, -0.97882986, -0.43918952, 0.18133843, -0.5028167, 2.4124537, -0.96050435, -0.79311734, -2.28862, 0.25148442, -2.0164065, -0.53945464, -0.27567053, -0.70972794, 1.7388726, 0.99439436, 1.3191369, -0.8824188, 1.128594, 0.49600095, 0.77140594, 1.0294389, -0.90876323, -0.42431763, 0.86259604, -2.6556191, 1.5133281, 0.55313206, -0.045703962, 0.22050765, -1.0299352, -0.34994337, 1.1002843, 1.298022, 2.696224, -0.07392467, -0.65855294, -0.51423395, -1.0180418, -0.07785475, 0.38273242, -0.03424228, 1.0963469, -0.2342158, -0.34745064, -0.5812685, -1.6326345, -1.5677677, -1.179158, 1.3014281, 0.8952603, 1.3749641, -1.3322116, -1.9686247, -0.6600563, 0.17581895, 0.49869028, 1.0479722, 0.28427967, 1.7426687, -0.22260568, -0.9130792, -1.6812183, -0.8889713, 0.24211796, -0.8887203, 0.9367425, 1.4123276, -2.369587, 0.8640523, -2.239604, 0.40149906, 1.2248706, 0.064856105, -1.2796892, -0.5854312, -0.26164544, -0.18224478, -0.20289683, -0.10988278, 0.21348006, -1.2085737, -0.24201983, 1.5182612, -0.38464543, -0.4438361, 1.0781974, -2.5591846, 1.1813786, -0.63190377, 0.16392857, 0.09632136, 0.9424681, -0.26759475, -0.6780258, 1.2978458, -2.364174, 0.020334182, -1.3479254, -0.7615734, 2.0112567, -0.044595428, 0.1950697, -1.7815628, -0.7290447, 0.1965574, 0.3547577, 0.61688656, 0.008627899, 0.5270042, 0.4537819, -1.8297404, 0.037005723, 0.76790243, 0.5898798, -0.36385882, -0.8056265, -1.1183119, -0.13105401, 1.1330799, -1.9518042, -0.6598917, -1.1398025, 0.7849575, -0.5543096, -0.47063765, -0.21694957, 0.44539326, -0.392389, -3.046143, 0.5433119, 0.43904296, -0.21954103, -1.0840366, 0.35178012, 0.37923554, -0.47003287, -0.21673147, -0.9301565, -0.17858909, -1.5504293, 0.41731882, -0.9443685, 0.23810315, -1.405963, -0.5900577, -0.110489406, -1.6606998, 0.115147874, -0.37914756, -1.7423562, -1.3032428, 0.60512006, 0.895556, -0.13190864, 0.40476182, 0.22384356, 0.32962298, 1.285984, -1.5069984, 0.67646074, -0.38200897, -0.22425893, -0.30224973, -0.3751471, -1.2261962, 0.1833392, 1.670943, -0.05613302, -0.0013850428, -0.687299, -0.11747455, 0.46616644, -0.37024245, -0.45380405, 0.40326455, -0.91800475, 0.25249663, 0.8203218, 1.3599485, -0.09038201, 1.3675972, 1.0344099, -0.99621266, -1.2179385, -0.30496365, 1.0289356, -0.07228701, -0.6006576, 1.5522432, 0.28690448, -2.3205943, 0.31716064, 0.52004063, 0.22560866, 0.4497121, -0.067275606, -1.3183959, -0.370704, -0.94561577, -0.9327409, -1.2630683, 0.45248908, 0.097896144, -0.44816536, -0.64933795, -0.023423105, 1.0791948, -2.0042157, 0.37687653, -0.545712, -1.8845859, -1.945703, -0.9127835, 0.21950956, 0.39306292, -0.9389816, 1.017021, 1.4229835, 0.39608657, -0.59140265, 1.1244192, 0.7553957, 0.86740744, -0.6564637, -2.8345544, 2.116791, -1.6108783, -0.035768073, 2.3807454, 0.33057675, 0.94924647, -1.5023966, -1.7776669, -0.5327028, 1.0907497, -0.34624946, -0.7946363, 0.19796729, 1.0819352, -1.4449402, -1.210543, -0.7886692, 1.0946383, 0.23482153, 2.1321535, 0.9364457, -0.035095178, 1.2650778, 0.21149701, -0.70492136, 0.67997485, -0.6963267, -0.2903971, 1.3277828, -0.10128149, -0.8031414, -0.46433768, 1.0217906, -0.55254066, -0.38687086, -0.51029277, 0.1839255, -0.38548976, -1.6018361, -0.8871809, -0.932789, 1.2433194, 0.81267405, 0.58725935, -0.50535834, -0.81579155, -0.5075176, -1.0518801, 2.4972005, -2.2453218, 0.56400853, -1.2845523, -0.10434349, -0.98800194, -1.177629, -1.1401963, 1.7549862, -0.13298842, -0.7657022, 0.55578697, 0.010349315, 0.72003376, -1.8242567, 0.30360392, 0.7726948, -1.6615983, 0.44819528, 1.6961815, -0.014857704, 0.82140595, 0.67057043, -0.7075057, 0.039766736, -1.5669947, -0.45130304, 0.26568797, 0.7231005, 0.024612125, 0.71998376, -1.1029062, -0.10169727, 0.019279385, 1.8495913, -0.21416666, -0.49901664, 0.021351224, -0.91911346, 0.19275385, -0.3650552, -1.7913276, -0.058586553, -0.3175431, -1.6324233, -0.06713416, 1.4893559, 0.5213038, 0.6119272, -1.3414967, 0.47689837, 0.14844958, 0.5290452, 0.4226286, -1.3597807, -0.041400813, -0.75787085, -0.050084095, -0.8974009, 1.3124703, -0.8589724, -0.8989422, 0.07458641, -1.0770991, -0.4246633, -0.8299646, 1.411172, 0.78580385, -0.057469517, -0.39121705, 0.9409176, 0.4052041, 0.49805242, -0.026192237, -1.68823, -0.112465985, -0.5324899, 0.6450553, 1.0118425, -0.65795106, 0.46838522, 1.735879, -0.66771275, 1.6819217, -0.85258585, 0.022959756, -0.011145612, 0.0114989, -0.837678, -0.5911831, -0.66772026, 0.3269626, 0.33003512, 2.2259443, 1.370989, -0.50984323, 0.3248696, 0.997118, 0.030601824, -0.069641575, 0.05157494, 0.8672766, -0.84832054, -0.32566947, 0.47043315, 0.31144708, 0.23958276, -0.36980116, 0.9725358, 2.1338682, 0.4064155, -0.1931767, 0.7557403, -0.53913265, -0.74969035, 0.032808747, -2.5827966, -1.1539503, -0.34796184, -1.3533889, -1.0326431, -0.43674833, -1.6429653, -0.40607178, -0.53527015, 0.025405208, 1.154184, 0.17250441, 0.021062022, 0.099454455, 0.22739278, -1.0167387, -0.11477532, 0.30875126, -1.37076, 0.8656529, 1.0813761, -0.63137597, -0.24133779, -0.87819034, 0.69938046, -1.0612223, -0.222477, -0.8589199, 0.05095428, -1.7942293, 1.3264617, -0.9646064, 0.059894685, -0.21252304, -0.7621145, -0.88778013, 0.93639857, -0.5256406, 0.2711702, -0.80149686, -0.64718145, 0.47224715, 0.9304085, -0.17531641, -1.4219198, 1.997956, -0.8565493, -1.5415874, 2.5944245, -0.4040323, -1.4617327, -0.6834398, 0.3675449, 0.19031155, -0.8517292, 1.8227236, -0.5215797, -1.1846865, 0.9606934, 1.3290628, -0.8174931, -1.4013473, 1.0304383, -2.0473237, -1.2266216, 0.96744615, -0.055352546, -0.26393735, 0.3528166, -0.15277442, -1.2986867, 1.2760754, 1.325014, 0.20533256, 0.045134015, 2.339625, -0.27643284, -0.25957698, 0.36448124, 1.471322, 1.5927707, -0.25857264, 0.30833125, -1.3780835, -0.3119761, -0.84029037, -1.0068318, 1.6815767, -0.79228663, -0.5316059, 0.36584878, 1.2978252, 0.48111513, 2.759355, -0.074667975, 0.25871643, 0.27560067, 1.4350494, 0.5072389, -0.1162297, -0.9474886, 0.24444346, 1.4013448, -0.4103818, 0.5289436, 0.24614778, 0.86351967, -0.8047537, 2.346647, -1.2791611, -0.36555108, 0.9380925, 0.29673317, 0.82998616, -0.49610233, -0.074804984, 0.012231983, 1.5692596, 0.69042903, 0.7966721, -0.6579261, 0.9688826, 0.22558166, 1.3891454, 2.0140603, -0.30676576, -0.40630314, -0.86404496, -0.14357951, -0.38202545, 0.3595044, -0.14456682, -0.36159927, 1.0645851, -0.9378802, 0.43310794, -0.40594172, 0.7243685, 1.3852615, -0.30309826, 0.44103292, 0.17879286, -0.7994224, 0.2407875, 0.2891205, 0.41287082, -0.1983989, 0.0941923, -1.1476109, -0.35811406, 0.5559627, 0.8924739, -0.42231482, 0.10471403, 0.22805333, 0.20147994, 0.5407736, -1.8180777, -0.04932407, 0.2390336, -1.0003303, 1.6739857, 0.16155927, 1.5634048, -0.790523, -0.9073001, 0.22425222, -1.6786884, 0.2149656, 0.09721923, 1.0156653, 0.70104134, -0.41747734, -1.0974966, 1.7123052, -0.79211503, -1.0455246, -1.084856, 1.1173053, -0.5189002, -0.7537045, 0.13768983, -0.2069447, -0.67809546, 0.7539915, 1.0653155, 0.9853175, 0.7669197, 0.40262553, -1.775888, 1.6692508, 0.3019892, 0.60815644, 1.1149623, 1.4333525, 0.41839802, 0.43554616, -0.59922427, 0.03308975, -0.85416126, -0.71994054, -0.8935744, -0.15602389, 1.0490932, 3.1709747, 0.18949963, -1.3484131, 1.2649833, -0.30078387, -0.6606086, 0.20984948, -1.2406245, 0.22246316, -0.08837552, 0.098377906, 0.38141626, 0.067492254, 0.016338084, 0.2843145, 0.41540062, -1.0314825, -1.4299912, -0.061638054, -1.4327354, 0.08753147, 0.93874687, 0.6071117, -1.0481704, -0.86026245, 0.32830128, -0.4012978, -0.3166553, 0.5969065, -0.9872867, -0.40123472, -0.8000825, -1.0431294, -0.8570782, 0.67746216, 0.05182039, -0.87916064, -0.2311016, -1.6388073, -0.7333128, 2.1495745, -0.090243846, 0.73165894, -0.065488376, 0.34816924, 0.6632581, -1.1046166, -0.030936258, 1.5788652, -0.7955006, -0.56643987, -0.30769128, 0.26902407, 0.52491784, 1.2674117, 0.49949825, -0.062053125, 1.2591671, 0.70411104, -1.4956795, 2.5263681, 1.7699214, -0.16821422, 0.3779101, 1.3243587, -0.1722008, 0.7303518, 1.1045785, -1.0148259, -0.6023319, 0.9214084, 0.46081448, 0.92379653, -0.13256802, -0.28900522, -1.9986395, -1.1460004, 0.047066096, 0.82455724, 0.53117836, -0.12824197, -0.27177158, 0.21717963, 0.07821118, 1.4045455, 0.14644077, -1.481246, -1.2725581, 1.5187594, -1.1711605, 0.76449746, -0.26837274, -0.16975829, -0.13413279, 1.221385, -0.19284183, -0.033319283, -1.5308034, 0.2066905, 0.5310425, 0.23914558, 1.3978963, 0.055171356, 0.29897746, 1.648504, -1.5500141, -0.45582536, 1.4261588, 0.93612915, 0.6783801, 0.8326507, 0.3270662, 1.6315974, 0.37775916, 0.2398671, 0.15895867, 0.19286396, -1.1570172, 0.77067304, -0.13043973, 1.8219151, -0.07565047, 0.4209183, 0.24660219, -0.625557, 0.99213684, 1.9050636, -0.01477722, -0.3004788, -0.35502872, -1.8923619, -0.17781314, 0.2509981, 1.054758, 0.9600477, -0.41649908, -0.27682298, 1.1239053, -0.1734639, -0.51002955, 1.3925184, 1.0375856, 0.018791791, -0.5937774, -2.0118804, 0.5897036, -0.8963697, -1.962732, 1.5848205, 0.6479678, -1.1390082, -1.2144014, 0.8709618, -0.87797064, 1.2961498, 0.6164593, 0.53659654, 0.40469545, 0.19145088, 0.8805112, -0.45408037, 0.08595198, 0.75194657, 0.5629897, -1.1949868, -0.50040966, 0.2528035, -0.4080147, 1.7746586, -0.3931532, -0.16221845, 0.76943016, 0.33053273, -0.14527446, -0.7564935, 0.30151406, 1.0390965, 0.47909522, -0.7781835, 1.7367749, -1.4465779, -1.5826856, 0.9605572, 0.22584048, -0.54949856, -1.0985707, 2.3207998, 0.11709087, 0.53420115, 0.3178851, 0.43480796, 0.54009444, 0.732424, -0.3752224, -0.29164198, -1.7410228, -0.78030443, 0.2711128, 1.0450233, 0.59903955, -0.34069234, -1.2631729, -2.7773592, 1.151734, -0.589229, -0.44846502, 0.13157398, -1.40556, -0.34978217, 2.0234718, 0.50538695, 0.35924914, -1.5824945, 2.2436018, -1.4227949, 1.9223248, -2.115056, 1.4053655, 1.6180543, -0.8244091, 0.42258036, 0.5474806, -0.8137945, -1.4491177, -1.3177173, 0.54100823, -0.085115604, -0.564301, 0.966768, 0.5080679, -0.7554627, -1.2012016, 0.5232617, -0.53758335, 0.09920486, 1.576299, 0.5023282, -0.862267, 0.16066119, -0.95264494, 1.6085222, -0.56157875, 0.20727074, 0.30773258, 0.15925047, -1.9585489, -1.446421, -0.4523503, 0.31943184, -0.13777922, -0.9571475, -1.3484243, -0.40155753, -0.46847606, 0.51283646, -0.32631847, 0.6027077, -0.5946498, -0.25595766, -0.3480464, -0.782367, 0.6251187, -0.813596, -0.5216415, -0.07311965, -1.2973796, -0.32493496, -0.71130633, -0.38815418, -0.059928004, -0.79991364, -0.22007579, 1.3086687, -0.025798557, 1.1452621, 0.34649444, 0.7741606, -0.77445894, 0.10490716, 0.13391292, -0.6126257, -0.82282835, -1.4902654, 1.4961396, -0.9724029, 1.3462211, -0.46749318, -0.8624933, 0.62251914, -0.63119197, 0.5684589, -0.33281177, 0.4804245, -0.9681861, 0.83135104, 0.48797268, -0.9196507, 2.6429358, 0.54012305, 2.290467, 1.6002678, -0.18883479, -0.41227177, -0.4034592, -1.8300285, -0.6958351, 0.24676603, 1.5259576, -0.7727719, 0.8820566, -1.2525934, -0.58632004, -0.4576406, 0.3718111, 0.45730963, 0.9623417, 0.77083695, 0.24316822, 0.39036494, 1.5885307, -0.5109262, 0.7747283, -1.808144, 0.41133425, -0.48324955, 0.0025711823, 1.0400863, 0.16464381, 0.88518757, 1.4737648, 0.38909397, 1.171041, -0.32656097, -0.008209882, -0.5226194, 1.0429776, 0.41409135, -0.50723445, 0.15466884, 1.0415684, -0.03926799, -0.9489328, 0.13191175, -1.9805655, 0.76877064, -0.4213276, -0.46931073, 0.8756957, -1.3651628, 1.9470986, -0.48024204, -0.52325094, 1.0212247, 0.7086953, 2.4512298, -0.21120599, -0.120406635, -1.479316, -0.33210227, -0.7214313, -0.448767, -1.7441877, 1.6606076, -1.4166034, -2.8022027, -1.1884245, -0.6038396, -1.149554, 1.0983036, -0.13783918, 0.025385605, 0.61039174, 0.28601253, 0.9785673, -1.1094775, -0.5475181, 0.66596717, -2.5345545, -1.3751845, 0.50099224, -0.48024905, 0.9361076, 0.8091803, -1.1980929, 0.4066571, 1.2016978, 0.1474344, -0.97746485, 0.87938994, 0.63542455, 0.54261076, 0.71593887, -2.994613, 0.8809376, 1.8081318, 0.43663847, 0.192729, 0.69643867, 0.33822548, 0.65178126, 0.0014710003, -0.76670486, -1.0043228, -0.9981917, -1.3730426, -1.067742, 1.7612661, 0.7540957, -0.6250274, -0.3903927, 0.11255753, -0.65554506, 0.067516856, 0.77760416, -0.035742734, 0.33601573, 0.88649154, -0.27213177, 0.2847906, -0.30937758, -0.02852887, -0.32473028, -0.52886987, 0.17371185, 0.5665453, 0.14630444, 0.49872696, -0.7379318, -1.2037352, 0.4170435, 0.6878814, 0.049857266, 1.3480358, 0.9076988, 2.6805708, -0.20080851, -0.9988488, -0.7401368, -0.5654978, 0.4760314, -2.1580687, 1.3185511, -0.23929659, -0.24679355, -1.0793432, -0.11422555, 0.013239767, -0.12194493, 0.33905926, -0.58963203, -0.8958158, 0.5483281, 0.09866745, 0.19718106, 1.0590272, -1.0225644, -0.85524046, 1.2572197, -1.4828833, -1.3094121, 0.81786186, 0.23820019, 0.105232134, -0.09165941, 0.031267546, -0.09211212, 1.3554426, -0.39814812, -0.16137354, 1.7944489, 0.027509702, 2.2320163, -0.1049797, 1.367415, -1.655344, 0.15364446, -1.5844736, 0.8444543, -1.2128679, 0.28376955, -0.28219587, -1.1582032, -1.61936, -0.51104045, 1.7406294, -0.29348505, 0.91722155, -0.057042867, 0.87672675, -1.8269113, -0.40318832, 0.94940555, -0.16325495, -0.086455286, -0.4304619, 1.1493794, 0.29751435, 0.044022277, 0.64305454, 0.58822495, 0.21258704, 1.5470315, -0.060287535, 0.27808106, -0.64295256, 0.15011522, 1.5877615, -0.6432576, -1.1335928, 0.99675965, -0.14876615, 0.0960042, -0.045113303, 0.079121724, 0.8505307, -0.8391242, -1.0117741, 0.084968135, -1.6064397, -1.3730536, 1.8666831, 0.75746834, -0.010056471, 1.238007, -1.0405992, -0.31560314, 0.6234536, 0.8906717, 0.51291686, -2.5412388, -0.96808213, 0.4770681, -0.3559515, 2.5402317, 0.9265583, 0.55808187, -1.1169496, -0.03529674, 0.24120396, 1.1277837, 0.8811311, 1.0329891, -0.923912, 1.4121517, -1.3804307, -0.53591454, 0.43077114, -0.14989159, -1.0060369, -0.82154983, -1.5482544, 0.5319746, 1.2605689, -0.100393504, -0.4003488, -1.472323, 0.9132019, 2.2113044, -1.7974558, -1.0634329, -0.679593, -0.5643179, 0.22734594, 1.6142496, 1.0085973, 0.52759737, -0.7239287, -1.1196282, -0.7967753, 1.5480669, -0.0617433, -0.44683626, -0.18375573, 0.8246182, -1.3128496, 1.4148741, 0.15647626, -0.21634398, 0.44284612, 0.21839707, -0.34419647, -0.25271067, -0.86886257, 0.6563907, -0.5319938, -0.9562584, 0.16586353, 1.3291413, -0.048344623, -0.60810125, 0.40389603, 1.9367125, -1.4519055, 0.38220277, 0.20508662, 1.1615338, 0.99090916, -0.1867091, -1.6845173, 0.8065638, -0.8351927, -0.9467404, 1.1483506, -0.9108504, 1.4028448, 0.33584473, 0.3191184, 0.30726478, -1.6384237, -1.7763886, 0.21555306, 0.56800735, 0.08261103, -0.8215345, 0.018922104, -0.082034156, -0.9571581, 1.0139722, -1.7302761, 0.58874243, 0.38432342, 1.0097119, -1.0053118, 0.10140715, 2.171165, 0.66207427, 0.10058121, 0.53916126, 0.08617684, 2.190898, 0.9836362, -0.08561496, 0.25233144, -0.390798, 1.2098501, -1.4061048, -1.6047385, 1.4587147, 2.1531198, 0.4683049, 0.11273794, 0.6572677, -0.64705354, 0.17124355, 0.038908705, 0.62656426, -1.5579985, -0.5070348, 0.8449956, -0.67559385, -0.99336135, 2.042072, 0.038118, -0.57891816, -1.6923704, 0.72934633, 0.69913614, -0.2987596, -1.1022302, -0.024549423, -0.8358561, -0.9420936, -0.10321275, -1.0513904, 0.24664895, 0.60799253, -0.83963245, -1.3682451, 1.5612797, -0.94027025, -0.6599427, 0.21301717, 0.59936935, -0.2563169, 0.46079433, -0.40098616, -0.97117066, 1.4263169, 2.4884417, 1.6959696, 0.14180663, 1.8334354, 0.3557035, -0.47728628, 0.46637958, -0.09439251, -0.9831182, -0.898322, 0.8020517, -1.846532, 0.60413677, -1.6295836, -2.1211765, -1.8388466, 1.966764, -0.19623396, 0.08658318, 1.419255, 0.9341797, -1.3915052, 0.86900634, 0.18418126, -0.34167808, 0.024290914, 1.279812, -0.8859665, 0.40088567, -0.009657237, -1.7971646, -0.8022532, 0.19321355, 1.2973421, 1.001331, 0.5972125, -0.81527567, 1.801214, 0.21524046, -1.0063655, -0.18290497, 0.8962484, 0.0076174983, 0.88686466, 1.103694, 0.4005307, -0.8577026, 0.13545467, 0.045165855, 1.8593464, -1.6263219, -0.13482246, -0.5840936, 0.33510563, -2.4375644, 1.1149246, 0.013748487, -1.8447012, -0.36111313, 0.60896236, -1.5914478, 0.0032222164, -1.0574737, -0.55598503, 0.026738383, 0.18345025, -0.4707425, 0.2727964, 0.8179776, -0.27891427, 1.4315678, 1.4622141, -0.42870206, -0.63784057, -1.664173, -0.12656933, -0.36343777, 0.77905124, -1.5096616, -0.2773914, 0.9687444, -0.7303571, -0.7623615, -1.4469403, 2.6205738, -0.7474732, -1.3003469, -0.8038504, -0.7742951, -0.26938978, 0.8253722, -0.29832315, -0.9228233, -1.4513385, 0.021857359, 0.042539075, 1.5309323, 0.092447735, -0.099008314, -1.0506538, -0.30595258, -0.43847445, -0.37016416, -0.9592554, 0.5383296, -0.14244542, -0.20035347, -1.7140461, 0.4936441, 0.48701534, -0.8391294, 0.99012136, -1.3647583, -0.021869909, -0.27120733, -1.3171748, 0.18970262, 1.7025702, 0.06763423, -0.46302176, 0.44702417, 0.10572, 0.027762132, -0.4255422, 1.4219756, 0.45636335, -0.52867067, -0.10800384, -0.7408667, -0.60829115, -0.64072573, -1.1343116, 0.777277, -0.29104146, 0.5541276, -0.6701259, -0.060362495, -0.7110406, 0.71966815, -0.2484193, -0.7308736, -1.6417032, 0.27566653, -0.70838505, -0.015779218, -0.4917301, 0.9541896, 0.54414475, 0.4472121, -0.6161211, 0.46629006, 1.7148316, -0.83218604, 0.17233914, -1.649217, 1.3985621, -0.39791209, 0.7825789, -1.7232282, 1.7975394, -0.35687152, 0.54565734, 0.1508182, -0.25547078, 1.6857923, -1.6480463, 0.29871365, 0.91064566, -0.029856121, -0.11817078, -0.14268771, -1.2276365, 0.038127385, 0.51271755, 0.068599224, -0.2722761, -0.48972502, -0.27929667, 1.2577442, -2.0866349, 0.040071458, -0.3277549, 1.4558079, 0.055492226, 1.4849256, -2.12389, 0.4595849, 0.28005785, 1.3905339, -1.6413486, -0.15503581, 0.06606026, -0.4957955, 1.2165778, -0.33868217, 2.0347626, 1.0541779, 0.9508337, 0.559299, -1.0636955, -0.43109635, 0.57275134, 0.67755705, 1.3071839, -0.46744102, -0.8601534, 0.8591042, -0.8096266, 0.8733118, 1.1997361, 0.45615304, -0.35757902, 0.041082226, 0.5934659, 0.010185518, 2.1982963, -0.9906709, -1.0026686, -0.9768954, -0.58957994, -2.1789315, -0.6296504, -0.6532847, 0.078514025, 0.41780058, -1.2402164, 0.9000542, 1.8022423, -0.20828511, 1.5743712, 0.1989895, 1.9887319, 1.1172835, -1.5639046, 0.01862737, 1.054325, 0.030546581, -0.03688353, 1.2697648, -0.7098542, 0.017515613, 0.32362577, -0.33379096, -0.020129103, 0.7750233, 0.43283764, -0.80871755, -1.104124, -0.7891022, 0.0012484558, -0.15993978, -0.8319575, -0.59815043, -1.5200393, 0.4178537, -0.040018726, -1.2597873, 0.028620504, 1.342622, -0.7399359, 1.3151376, -0.32345748, 0.19782817, 0.097750805, 1.4015235, 0.15843384, -1.1419014, -1.3109704, -1.5329211, -1.7119702, 0.04613506, -0.9583745, -0.08081161, -0.70385903, -0.7707843, -0.48084533, 0.70358557, 0.92914516, 0.37117255, -0.98982257, 0.6436313, 0.68889666, 0.2746472, -0.6036204, 0.70885956, 0.42281857, -3.1168566, 0.64445204, -1.9137427, 0.6635616, -0.1540724, 1.1936116, -0.09816121, -0.88661426, -0.14735366, 1.0598063, 0.026246618, -0.11433516, 0.7435535, 0.21035936, -0.005927406, 1.36606, 1.555114, 0.61332625, -0.28595915, 1.496911, 1.1831195, 0.71889716, -1.2160766, 0.14067191, -0.7436722, -0.15901226, 0.24005693, 0.10015941, -0.4751751, 1.2729537, -1.6961312, 0.73018354, -1.8574833, 0.38259813, -0.8869043, 0.87830377, 0.08645252, 0.24770638, -1.0182793, -0.65457016, 0.2072174, 0.58356994, 2.9290962, 0.22285832, 0.9760375, -1.5569339, -1.3298919, -0.35549477, -1.1974277, 1.4863993, -0.4102187, 1.3821819, 1.4867824, 0.04277972, 0.50179976, -0.056099474, 0.538437, 0.48334184, -0.12364963, 0.50496995, 1.7236962, 0.7130162, 0.3257996, 0.124769524, -1.0126731, -1.0272969, 0.32335654, -1.3693911, -0.7663276, 1.2815113, 1.9142298, -1.665956, 1.6266496, -0.2114383, -0.0150050875, -0.11341163, 1.0805441, -1.6076766, 0.45616361, -0.9448702, 0.5707885, 1.5427964, -0.0004173264, 0.37415507, 0.40955177, -0.7995935, 1.5116394, 1.7064682, 0.70178336, 0.07328543, -0.46189383, -0.62649024, 1.7108365, 1.414415, -0.063661486, -1.5799305, -2.832012, -1.0834267, -0.13062039, 1.400689, -0.6516562, 0.50481546, 1.3031809, 0.12853631, -0.14244787, -1.3087635, -1.2024753, 0.41609964, -0.20090753, 0.12253132, -0.047277715, 0.66414404, -0.7846874, -0.33558065, 1.8961822, -0.79978615, -0.28157544, -0.5893867, 0.44478136, 1.0223923, -0.49821162, -0.43141434, -0.2789816, 0.5298338, -0.7393953, -0.37595996, -2.3721938, -1.381745, -0.11244375, 0.89786416, 0.29507577, -1.0987685, -1.4002562, 0.1746801, -1.6528037, 1.0659268, 0.063896194, -1.6073202, -0.9659539, -0.7243113, -0.7731925, -1.489933, -0.8746625, -0.6844016, -0.71128577, 1.1279566, 0.10482781, -0.9932572, -0.3346216, -0.8795571, -0.30000666, 0.87550914, 0.2522708, 2.2856011, 0.37592742, -0.9135945, 0.8097407, 1.0799313, 1.094167, -1.0942409, -0.14763741, 1.131812, -1.684729, -0.49941677, -1.4269377, -0.9325702, -1.0124571, 1.2505698, -0.23453803, -0.8633556, -1.0356058, 0.14166717, -0.0111356275, 1.3440744, 0.5000167, -1.4317977, -0.6289807, 1.0700725, -0.6210827, 1.7345722, -1.0982895, 0.57261336, -0.86121553, -0.50959516, 1.0985817, -0.12706716, 0.81345224, 0.4732906, 0.75386566, -0.8881882, -0.2215744, 0.42425263, -0.8490729, 1.6295, -0.77722806, -0.3000036, -1.006559, -2.1433082, 1.7969185, -0.20433894, -0.44791484, -0.19871506, 1.4198639, -0.9651066, 0.6795679, -0.42378825, -0.59667087, 0.5670582, 0.9882406, -0.51390296, -0.76884913, -1.1690958, 1.1035038, -0.575256, -1.8491307, 1.4099522, -1.3698595, 0.77946055, 0.18342865, 0.28791544, -0.58437526, 0.36559147, -1.6677799, 0.5880377, 1.55701, 0.8840272, -2.01954, -0.984209, -0.18779492, 0.4869373, -0.10665268, -0.4932144, 0.5953003, 1.1641518, -0.23229401, 0.7289299, -2.5790508, -0.93750936, -0.32125893, -0.48856622, 0.3327982, 1.0137506, 0.50666904, -0.62222546, -1.5227681, 0.5569641, -1.8381767, 0.6530373, -0.18844908, -1.175835, 0.2872573, -0.0028761027, -0.036597293, -0.0842233, 0.4195241, 0.924434, 0.4966152, 1.0121332, -0.04413972, 1.6184593, 0.57110983, -0.543694, -1.0938951, 0.20579681, -1.3065215, -0.973376, 0.23908707, -0.60788745, -0.93331623, -0.034475047, 0.072677895, -0.20583403, -0.3775469, 0.85464275, 0.34242734, -0.22342612, 2.4643219, 0.19383174, 1.1320051, -0.560981, -1.3629409, -0.7917565, -0.26800978, -0.4966082, 1.3363862, -0.120041125, 0.46146888, -0.046481155, -0.43355432, 0.037996013, 1.7140515, -0.76794857, 0.7669904, -1.0260073, -0.45962644, 0.0035832059, 0.3263751, 1.4831287, -0.050082643, -0.8436156, 0.650042, -0.3641698, 0.23868157, -0.11622244, -1.9434569, 0.5082992, 0.583368, 0.92660475, 1.8004627, -1.1951038, 0.51650745, 0.409295, -0.419082, 0.39710623, 0.49964696, -1.2186838, 0.24622276, -0.9179843, -0.6518565, -1.7747449, -0.47336093, -0.20357068, 0.54985684, 0.00089992664, -1.5422882, 0.86214805, -0.11858662, 0.4883706, 0.9659361, 1.4226048, 1.9612269, -0.07223876, 0.31112444, -1.078361, 1.0616002, -1.1848874, -1.8052517, 0.830386, -0.5216965, 0.77760726, 0.40807465, -1.6300026, -2.7196794, -1.0966017, 0.016491488, -1.2217764, -0.65276146, -1.4589407, 0.16987796, 0.09082593, -0.48139262, 1.3970653, 1.497715, 0.5652672, -1.7997712, -1.1046902, 0.40713033, -0.62855756, -0.48709142, 0.8989674, 0.5108748, 1.3141544, -0.4292093, 1.3752254, -0.55413127, 1.4994915, 0.10583464, -0.86050975, -1.6312195, -0.3014723, -0.2562327, 0.8576619, -0.1105905, -0.43243197, 1.0770375, -0.22482656, -0.5762418, 0.5746089, -0.48982823, 0.65880215, -0.5969171, -0.22295918, 0.15217698, -0.37412632, -0.013451469, 0.81547195, 0.4106018, 0.48096985, -0.63543046, 0.85282975, 0.66956234, 1.0044192, -0.7263658, -0.1724586, 0.6335339, -0.60881513, -0.22612247, 1.9258057, 1.951761, 1.2399405, 0.93858516, -1.0192511, 0.5125622, -0.35911658, -1.0585719, -0.50900584, 0.11566507, -0.5473556, -0.5507994, 0.7920415, 0.14410649, 0.23345809, 0.1118724, -0.67570317, -1.370572, 0.3105647, -0.5070366, -2.0107822, -0.39256725, -1.0922179, 0.69865024, 0.5216252, 0.49689314, -0.6650416, 0.7315516, 0.3196498, -0.40985453, -0.45333743, 0.8927082, -0.47360405, 0.30365646, 1.033957, 1.9093426, 1.6638731, 0.90082276, -1.5059114, -0.6890484, -0.5480872, 1.6531498, -0.69931793, 0.38616636, 0.10086706, -0.9351272, 0.38182402, 0.3982961, -1.2557749, 1.2228775, -2.08651, -0.59075713, 0.9719703, -1.1932578, 0.35026592, -1.2963604, -0.09302414, -2.3137732, -0.8425717, -1.5429214, -0.40176374, -0.4152314, -0.67366415, 0.7979132, -0.8868796, 0.63438666, 1.6292758, 0.13906415, -0.8576702, -1.2493385, -0.7097851, 0.7046427, 0.15559073, 0.93679523, 0.7703309, 0.14081065, 0.47348827, 1.8552462, 1.4156562, -0.30274603, 0.98967946, 0.58585083, 1.1363881, 0.67161655, -0.9741674, -1.6196846, 0.572627, 1.9026182, -0.7756641, -0.18808974, -1.0357478, 1.1778295, -2.305167, -2.2636602, 0.3750199, -0.082343645, -0.47962302, -0.3010948, 0.5369879, -0.413804, -1.096925, -0.9273629, 0.88833886, -0.52474195, -1.3852776, 0.10217833, 0.50499475, 1.3289608, 0.21790339, -0.65971124, 0.47400787, 0.7271749, -0.038905308, -0.04459939, 0.2601329, -0.069856495, 0.2501139, -1.0219133, -1.1504377, -0.83611137, 0.64221096, 0.25879756, 1.040239, -0.18669093, -1.1436414, 1.1445535, -0.018767055, 1.283455, 0.59794647, 2.1886187, -0.21977298, 0.90072393, 0.8913641, -0.55512637, -0.17248231, -1.4617383, -1.5487962, 0.1265688, 0.7930071, 0.63802403, 0.3400246, 0.86301714, -0.5896978, -0.27253276, 0.7375215, 0.43311873, -0.21018882, 1.3207943, -1.2920012, -0.51867867, -0.28339776, 0.8165349, 0.002385198, -1.2614918, 0.5140042, 1.0875463, 0.73930454, 0.61915493, -1.8743135, -0.8998865, 0.4820806, -0.054888185, 0.5225576, -1.2663426, -0.061494764, -1.389781, -1.9536786, 0.29577908, 0.8425888, 0.24561642, -0.03299648, -1.5620143, 1.0061071, -0.044044897, 1.9595621, 0.9423143, -2.0051255, 0.7550497, -1.3965353, -0.7594955, -0.25075668, -0.09406245, 0.39756522, -1.022855, -1.150692, 0.6006052, -0.013250268, 0.17437305, -2.1936834, -0.17713739, -0.8907292, -0.9206264, 0.9219348, -1.0956712, -1.0928966, -0.3310106, 0.45028883, -0.8840147, 1.2341441, 1.4498476, -0.8814471, -0.24508175, -0.7786755, -1.6853821, 0.30301106, 0.7335949, 2.0118642, -0.8974095, 1.336235, 1.3423537, 0.19785331, 0.6021635, 0.8732731, 1.9741, 0.47780856, -0.060137887, -0.8661688, 0.30532077, 1.0241649, 0.24461035, -0.77992326, 0.089076206, -0.12915348, 0.26473877, -1.6618484, 0.55078864, 0.59542316, 0.44485343, -0.0037628172, -1.8059362, -0.019322792, 1.060715, -0.8601289, -1.9892695, -1.540558, 0.3140257, 0.37287602, 0.8862932, -0.055258997, -1.5003284, -0.81850415, 0.8188394, 0.14049591, 0.6498296, 0.4347888, -0.20496055, -0.17400683, 1.8571023, 0.41467425, -0.12858754, 0.45542, 0.22290581, -2.1573563, 0.6500845, 1.8209393, -0.7802799, 1.4540358, -0.2568697, 0.2934714, 1.0703601, -0.72000146, 1.2424939, -1.2142173, -0.87515473, -0.59352034, 0.66200536, -0.3408744, -1.5199745, -0.21653287, -0.7842214, 0.7312936, -0.34323505, 0.07077408, -0.40547246, 0.43393898, -0.18359077, 0.3251987, -2.5933886, 0.09725088, 0.41391367, -0.19928005, 0.66939247, 0.73860705, 1.3042139, 0.10481161, -1.9138007, -2.2854993, -1.601841, -0.03790706, -0.15730529, 0.27623984, -0.6252459, -0.73649114, 0.5550479, 0.65592444, -0.25665015, -0.038476657, 0.40431434, 0.50434357, -1.1439807, -0.71957386, -1.230546, -0.5069066, 0.8123336, 0.54627186, -1.0980979, 0.51226676, 0.08584311, -0.4939267, -1.4064597, -0.17482337, 0.679944, -2.1630976, -0.3961232, 2.2542837, 0.67263675, 0.2598325, -0.7371852, -0.6783298, -0.083288394, 1.6028637, 0.4655892, -0.8721584, 1.176787, -0.2925942, 1.6973464, -0.566603, -1.0032657, 0.17462958, 0.982327, 1.0374448, 0.15919177, -0.9880967, -0.5053407, -2.018282, -0.9131215, -0.17845681, 0.38900214, -0.33945432, -0.056979056, -0.39618546, 0.7510253, -0.89911294, 0.8375479, 1.9608808, 0.47278965, -0.5270916, -0.53627014, 1.2098372, -1.1265894, -0.95380443, -1.1644485, -1.2785138, -1.0448164, 0.78990495, 1.1022825, -0.6970731, 0.20733404, 0.7591567, 0.100564204, -0.95494276, -1.4704018, 1.0104276, 0.4961794, 0.5769559, -1.107647, 0.23497719, 0.6289996, 0.31403384, -0.7450232, 1.0122606, -1.527632, 0.92874193, 1.081056, 1.5723304, -0.3424922, -0.99943, 0.79388034, -0.6992153, 0.04399551, -0.3174622, -0.90207195, 0.32099947, -1.3920159, 0.5922057, -0.9669311, -1.7317313, -0.05010746, 0.43163386, 0.5769346, 0.8183537, -2.3536403, -1.0051445, 0.1066523, 1.5190033, 0.7837445, 1.90134, -0.5249394, 0.27441698, -1.0999708, -0.40435222, -0.7352957, -0.6339887, -0.39344913, 0.00271754, 0.022212664, 0.54345345, 0.13998847, -0.34404564, -0.52257854, -0.3071317, -0.44903713, 0.49097106, 0.8655252, 1.2740445, -0.7977028, 0.4693722, -1.3946797, 0.37317473, 1.0826722, -0.14958951, 1.072636, -1.1385679, -0.8886453, -0.13580984, 1.0222104, -0.41742945, -0.4535531, -0.99162835, 0.20288104, 1.2466952, 0.70068014, 0.6966507, -0.20697448, -0.5633094, 0.6772459, -0.031911075, -0.17360823, 0.8982406, -0.19778745, -0.83777624, 0.9091885, 0.08071989, -1.0370294, -1.1129059, 0.095411874, 2.3374097, -0.3928206, -0.33627385, 1.5237712, -0.0572812, -1.4484669, -1.5727965, 1.226664, 0.66635454, 0.8261257, -0.057756558, -0.72671205, -0.21716312, 0.13603121, -0.83831114, 0.5614499, -1.2595961, -0.33275875, -0.20400788, -0.69101983, -2.2055054, 0.44786966, -0.7557508, 1.3257079, -0.34198228, -0.5413596, 0.09152195, 1.0534397, -0.56340766, 1.0147377, 1.4403037, 0.9903228, 1.6264315, 1.292646, 1.5148823, 1.6043264, 0.20806953, -0.4292239, -2.2622437, -1.3227332, -0.4482828, -0.3817351, -0.15279447, -1.0007604, -1.5957776, -0.13022317, -0.18941793, -0.80755407, -0.74215215, -0.9401566, -0.39652374, -0.8563028, 1.2598753, 0.24099673, -0.97231793, -0.28044778, -1.1802856, 1.0121683, 1.3841867, 1.252002, -1.1446927, -0.09126702, -0.40157068, 0.5620131, -1.0079098, -0.6758917, -0.41321704, 0.15328847, 0.6941287, -0.3287277, 0.66396505, 0.8220764, -0.21321523, -1.2456582, -1.1711904, 0.59172696, -0.47622442, -1.7126293, 0.61295235, 0.12955453, -1.4059671, 1.17942, 0.836636, 0.13874525, -1.2743194, -1.4023305, -0.3070685, -1.7139153, 0.40508026, -1.4108233, 0.16491273, -0.28813145, 0.71178526, -0.9379476, 0.27372944, -1.3948402, 0.7955496, -0.114961766, 0.49585068, -1.3205253, 0.49908426, 0.3062034, 0.3636979, 0.31263396, -0.19346388, 1.2412993, -0.15589799, -0.7391692, -0.05872619, -0.95051795, -0.4639964, -0.17724662, -0.37955412, 0.19939707, 1.9457614, 0.57094985, 1.0723007, -0.50370944, -0.5870163, -0.37817806, 0.8528891, -2.1481185, -1.0331647, 0.10233585, -0.22409236, 1.9677297, 0.44768322, -0.66219145, -1.577607, -0.34056005, -1.30322, 0.46675065, 0.16110632, 0.32003194, 2.0791767, -0.907466, -0.19240421, -1.2125157, -0.08059852, 1.5932736, 0.5687224, -0.114487045, 0.25163025, -1.2108556, -0.3937337, 0.085252576, 0.099421985, -1.5306163, 0.3276232, 0.2791965, -0.3770512, 0.004174999, -1.4834915, -1.4797956, 0.13468726, -0.6677232, -0.01155552, 0.83949065, -0.17392993, -2.810668, -0.15065365, -0.48104402, -0.23469436, 0.8997308, -1.5785302, 0.24395663, 1.5703039, -0.6259431, 0.4723279, 0.9663058, 0.21023144, -0.685097, -0.709521, 0.74380016, 0.5921491, -0.7864684, -1.1764731, -1.2808067, 1.6616518, -0.06794512, 2.3602285, 0.5555456, 0.43952233, 0.30627248, 0.99914986, -0.9660632, 2.1600132, -0.100301705, -0.7034001, 0.302561, 1.0923389, -1.0075549, 0.5668694, -0.71644413, -0.5062735, -0.48948243, 0.76354146, -1.1090727, 0.1926161, -0.34341785, -0.84721017, -1.2135236, -1.2028884, -1.633796, 0.8961672, -0.24165316, 0.15865193, 1.1781894, -1.2201172, -0.94154567, 0.25471553}),
			),
			outputT: tensor.New(tensor.WithShape(1, 3, 32, 32),
				tensor.WithBacking([]float32{1.7640524, 0.978738, 2.2408931, 2.2408931, 1.867558, 1.2023798, 0.95008844, -0.10321885, 0.41059852, 0.41059852, 1.9507754, 1.9507754, 0.7610377, 0.44386324, 0.7774904, 1.4940791, 1.4940791, 0.3130677, 0.3869025, 0.3869025, 0.6536186, 0.8644362, 0.8644362, 2.2697546, 2.2697546, 0.3024719, 0.045758516, 1.5327792, 1.5327792, 1.4693588, 0.37816253, 0.37816253, 0.17742614, -0.34791216, 0.46278226, 1.2302907, 1.2302907, 1.2023798, 0.7290906, 1.1394007, 1.1394007, 0.40234163, 1.9507754, 1.9507754, -0.4380743, -0.31155252, 0.7774904, 0.7774904, 0.9008265, 0.9008265, 0.46566245, 1.4882522, 1.8958892, 1.8958892, 1.1787796, 0.42833188, 1.0544517, 1.0544517, 1.222445, 1.222445, 0.97663903, 0.97663903, 0.7065732, 0.7065732, 1.7858706, 1.7858706, 0.46278226, 1.8831507, 1.8831507, 0.7290906, 0.9693967, 1.1394007, 1.9436212, 1.9436212, 0.40234163, 1.922942, 1.922942, 1.867559, 1.867559, 0.90604466, 1.9100649, 1.9100649, 0.8024564, 1.4882522, 1.8958892, 1.8958892, 1.1787796, 0.9222067, 1.0544517, 1.0544517, 1.3263859, 1.3263859, 0.97663903, 0.97663903, 1.8492638, 1.8492638, 1.7858706, 1.7858706, 0.5392492, 1.8831507, 1.8831507, 0.031830557, 0.9693967, 0.9693967, 1.9436212, 1.9436212, 0.3960067, 1.922942, 1.922942, 1.867559, 1.867559, 2.3831449, 2.3831449, 1.9100649, 1.1170163, 1.1170163, 0.947252, 0.61407936, 1.7133427, 1.7133427, 0.37642553, 0.2982382, 1.3263859, 1.3263859, 1.1266359, -0.14963454, 1.8492638, 1.8492638, 1.929532, 1.929532, 0.9494208, 0.5392492, 0.844363, 0.844363, 0.67643327, 1.1880298, 1.1880298, 0.9208588, 0.9208588, 0.8568306, 0.8568306, 0.4393917, 0.6815945, 2.3831449, 2.3831449, 0.94447947, 1.1170163, 1.1170163, -0.35399392, -0.0682416, 1.7133427, 1.7133427, 0.62523144, -0.09845252, 0.05216508, 1.1266359, 1.5430146, 1.5430146, 0.26705086, 0.26705086, 1.929532, 1.929532, 0.9494208, 0.77179056, 0.844363, 2.163236, 2.163236, 1.336528, 1.1880298, 1.0996596, 1.0996596, 0.8568306, 0.8568306, -0.024326125, 0.6815945, 0.6815945, 0.2799246, 0.9101789, 0.9101789, 0.78632796, 0.78632796, -0.4664191, -0.4100497, 0.62523144, 0.62523144, 2.259309, 2.259309, 0.05216508, 1.5430146, 1.5430146, 0.48148146, 0.48148146, 0.06326199, 0.5232767, 0.5232767, 0.77179056, 0.82350415, 2.163236, 2.163236, 1.336528, 0.41605005, 1.0996596, 1.0996596, 1.4944845, 1.4944845, 0.42625874, 0.676908, 0.676908, 0.2799246, 0.9101789, 0.9101789, 0.78632796, 0.78632796, 1.1523316, 1.1523316, 1.0796186, 0.37915173, 2.259309, 2.259309, 0.14195317, 0.14195317, 0.69153875, 0.6947491, 0.6947491, 0.06326199, 0.15650654, 0.6103794, 0.6103794, -0.23792173, -0.23792173, -0.052567296, -0.052567296, 0.41605005, 0.52389103, 0.7811981, 1.4944845, 1.4944845, 0.42625874, 0.676908, 1.9559124, 1.9559124, 0.39009333, -0.13288058, 0.49374178, 0.49374178, 1.1523316, 2.064493, 2.064493, 1.0201727, 1.0201727, 1.5363771, 1.5363771, 0.60884386, 0.69153875, 1.2111453, 1.2111453, 1.3018463, 1.3018463, 0.6103794, 2.3039167, 2.3039167, -0.1359497, 1.1368914, 1.1368914, 0.5829537, 0.5829537, 0.52389103, 0.37005588, 1.6581306, 1.6581306, 0.39904633, 1.9559124, 1.9559124, 0.39009333, -0.39095336, 0.69377315, 0.69377315, -0.11610394, 2.064493, 2.064493, 1.0201727, 1.0201727, 1.5363771, 1.5363771, 0.60884386, 0.60884386, 1.2111453, 1.2111453, 1.3018463, 1.3018463, 0.27451634, 2.3039167, 2.3039167, -0.1359497, 2.2567234, 2.2567234, 0.9432607, 0.9432607, 0.7471883, 0.77325296, 1.6581306, 1.6581306, 0.60631955, 0.6663831, 0.6663831, 0.45093447, 1.6595508, 1.6595508, 1.0685093, -0.13370156, 1.0777438, 1.0777438, -0.28035548, -0.28035548, 0.15670386, 0.5785215, 0.5785215, 0.34965447, -0.0616264, -0.10730527, 1.3645319, 0.27451634, 0.27451634, -0.52118933, -0.31229225, -0.15766701, 2.2567234, 2.2567234, 0.9432607, 0.9432607, 0.93184835, 0.77325296, 0.77325296, 0.16092817, 0.60631955, 0.60631955, 0.45093447, 0.45093447, 1.6595508, 1.6595508, 1.0685093, 0.04949498, 0.4938368, 0.6433145, 0.6433145, -0.20690368, 0.8801789, 0.8801789, 0.5785215, 0.38728046, -0.76414394, -1.0225068, 1.3645319, -0.6522936, -0.52118933, 1.648135, 1.648135, 0.5672903, 0.6203583, 0.6984571, 0.6984571, 0.93184835, 0.93184835, 0.8579239, 1.1411018, 1.4665787, 1.4665787, 0.85255194, -0.26773354, 0.7666632, 0.7666632, 0.8416313, 0.8416313, 0.8145198, 0.8145198, 0.6433145, 0.6433145, -0.20690368, 0.8801789, 0.8801789, 0.38728046, 1.7327212, 1.7327212, 0.6845011, 0.370825, 1.5199949, 1.7195894, 1.7195894, 1.648135, 0.5822246, 0.5672903, 0.12372191, 0.09395323, 0.9430461, 0.9430461, 0.8579239, 1.1411018, 1.4665787, 1.4665787, 0.8689635, 0.8689635, 0.7666632, 0.7666632, 0.8215857, 0.8215857, 0.8145198, 0.8145198, 0.078260176, -0.18505368, -0.085930765, 0.800298, 0.87583274, 0.87583274, 1.7327212, 1.7327212, 0.6845011, 0.370825, 1.5199949, 1.7195894, 1.7195894, 0.9295051, 0.5822246, 2.0210435, 2.0210435, 0.09395323, 0.9430461, 0.9430461, -0.050603542, 0.26990435, 0.26990435, 0.18133843, 0.8689635, 2.4124537, 2.4124537, 0.3148172, 0.8215857, 0.8215857, 0.8005648, 0.8005648, 0.078260176, -0.27567053, 1.7388726, 1.7388726, 1.3191369, 1.3191369, 1.128594, 1.128594, 0.49600095, 1.0294389, 1.0294389, 0.8202478, 0.86259604, 0.86259604, 1.5133281, 2.0210435, 2.0210435, 0.22050765, 0.22050765, 0.1993002, 1.1002843, 1.298022, 2.696224, 2.696224, 0.18133843, 2.4124537, 2.4124537, -0.07785475, 0.38273242, 0.38273242, 1.0963469, 1.0963469, -0.2342158, -0.27567053, 1.7388726, 1.7388726, 1.3191369, 1.3191369, 1.3014281, 1.3749641, 1.3749641, 1.0294389, 1.0294389, 0.17581895, 0.86259604, 1.0479722, 1.5133281, 1.7426687, 1.7426687, 0.22050765, 0.22050765, -0.34994337, 1.1002843, 1.298022, 2.696224, 2.696224, 1.4123276, 0.8640523, 0.8640523, 0.40149906, 1.2248706, 1.2248706, 1.0963469, 1.0963469, -0.2342158, -0.18224478, -0.18224478, -0.10988278, 0.21348006, 1.3014281, 1.3014281, 1.5182612, 1.5182612, -0.38464543, 1.0781974, 1.0781974, 1.1813786, 1.1813786, 1.0479722, 1.7426687, 1.7426687, 0.9424681, -0.26759475, 1.2978458, 1.2978458, 0.24211796, 0.9367425, 1.4123276, 2.0112567, 2.0112567, 0.8640523, 0.40149906, 1.2248706, 1.2248706, 0.3547577, 0.61688656, 0.61688656, 0.5270042, 0.5270042, 0.4537819, 0.21348006, 0.76790243, 0.76790243, 1.5182612, 1.5182612, -0.38464543, 1.0781974, 1.1330799, 1.1813786, 1.1813786, 0.16392857, 0.7849575, 0.9424681, 0.9424681, -0.21694957, 1.2978458, 1.2978458, 0.020334182, 0.5433119, 0.5433119, 2.0112567, 2.0112567, 0.35178012, 0.37923554, 0.37923554, 0.1965574, 0.3547577, 0.61688656, 0.61688656, 0.5270042, 0.5270042, 0.4537819, 0.23810315, 0.76790243, 0.76790243, 0.5898798, -0.36385882, 0.115147874, -0.13105401, 1.1330799, 1.1330799, 0.895556, 0.895556, 0.7849575, 0.7849575, 0.32962298, 1.285984, 1.285984, 0.67646074, 0.67646074, 0.5433119, 0.5433119, 0.43904296, -0.21954103, 0.35178012, 1.670943, 1.670943, -0.0013850428, -0.0013850428, -0.11747455, 0.46616644, 0.46616644, 0.41731882, 0.40326455, 0.40326455, 0.25249663, 0.8203218, 1.3599485, 1.3599485, 1.3675972, 1.3675972, 1.0344099, 0.60512006, 0.895556, 1.0289356, 1.0289356, 0.40476182, 1.5522432, 1.5522432, 1.285984, 0.67646074, 0.67646074, 0.52004063, 0.4497121, 0.4497121, -0.067275606, 0.1833392, 1.670943, 1.670943, -0.0013850428, 0.45248908, 0.45248908, 0.46616644, 0.46616644, -0.023423105, 1.0791948, 1.0791948, 0.37687653, 0.8203218, 1.3599485, 1.3599485, 1.3675972, 1.3675972, 1.0344099, 0.39306292, 1.017021, 1.4229835, 1.4229835, 0.39608657, 1.5522432, 1.5522432, 0.86740744, 0.86740744, 0.52004063, 2.116791, 2.116791, 0.4497121, 2.3807454, 2.3807454, 0.94924647, 0.94924647, -0.9327409, 0.45248908, 1.0907497, 1.0907497, -0.34624946, 0.19796729, 1.0819352, 1.0819352, 0.37687653, 0.37687653, 1.0946383, 1.0946383, 2.1321535, 2.1321535, 0.9364457, 1.2650778, 1.2650778, 1.4229835, 1.4229835, 0.67997485, 1.1244192, 1.3277828, 1.3277828, 0.86740744, -0.46433768, 2.116791, 2.116791, -0.035768073, 2.3807454, 2.3807454, 0.94924647, 0.94924647, -0.8871809, -0.5327028, 1.2433194, 1.2433194, 0.81267405, 0.58725935, 1.0819352, 1.0819352, -0.5075176, 2.4972005, 2.4972005, 1.0946383, 2.1321535, 2.1321535, 0.9364457, 1.2650778, 1.2650778, 1.7549862, 1.7549862, 0.67997485, 0.55578697, 1.3277828, 1.3277828, 0.72003376, 0.30360392, 1.0217906, 1.0217906, 0.44819528, 1.6961815, 1.6961815, 0.82140595, 0.82140595, 0.67057043, 0.039766736, 1.2433194, 1.2433194, 0.81267405, 0.7231005, 0.7231005, 0.71998376, 0.71998376, 2.4972005, 2.4972005, 0.019279385, 1.8495913, -0.10434349, 0.021351224, 0.021351224, 0.19275385, 1.7549862, 1.7549862, -0.058586553, 0.55578697, 0.55578697, 0.72003376, 1.4893559, 1.4893559, 0.7726948, 0.7726948, 0.47689837, 1.6961815, 1.6961815, 0.82140595, 0.82140595, 0.67057043, 0.039766736, 0.039766736, -0.050084095, 1.3124703, 1.3124703, 0.7231005, 0.71998376, 0.71998376, -0.10169727, 0.019279385, 0.019279385, 1.8495913, 0.78580385, 0.021351224, 0.9409176, 0.9409176, 0.49805242, 0.49805242, -0.026192237, -0.058586553, -0.112465985, 0.6450553, 1.4893559, 1.4893559, 0.6119272, 1.735879, 1.735879, 1.6819217, 1.6819217, 0.5290452, 0.4226286, 0.0114989, 0.0114989, -0.050084095, -0.050084095, 1.3124703, 1.3124703, 2.2259443, 2.2259443, 1.370989, 0.3248696, 0.997118, 0.997118, 1.411172, 0.78580385, 0.8672766, 0.9409176, 0.9409176, 0.49805242, 0.49805242, 0.31144708, 0.23958276, 0.9725358, 2.1338682, 2.1338682, 1.0118425, 0.7557403, 1.735879, 1.735879, 1.6819217, 1.6819217, 0.022959756, 0.022959756, 0.0114989, 0.0114989, -0.43674833, -0.43674833, 0.3269626, 0.33003512, 2.2259443, 2.2259443, 1.370989, 0.3248696, 0.997118, 0.997118, 0.22739278, 0.05157494, 0.8672766, 0.8672766, 0.8656529, 1.0813761, 1.0813761, 0.31144708, 0.23958276, 0.9725358, 2.1338682, 2.1338682, 0.4064155, 0.7557403, 0.7557403, 1.3264617, 1.3264617, 0.059894685, 0.059894685, -0.21252304, -0.34796184, 0.93639857, 0.93639857, 0.2711702, 0.2711702, -0.40607178, 0.47224715, 1.154184, 1.154184, 0.17250441, 1.997956, 1.997956, 0.22739278, 2.5944245, 2.5944245, 0.30875126, 0.8656529, 1.0813761, 1.0813761, 0.19031155, 1.8227236, 1.8227236, 0.69938046, 0.9606934, 1.3290628, 1.3290628, 0.05095428, 1.3264617, 1.3264617, 0.059894685, 0.96744615, 0.96744615, -0.055352546, 0.93639857, 0.93639857, 0.2711702, 1.2760754, 1.325014, 1.325014, 0.9304085, 2.339625, 2.339625, 1.997956, 1.997956, 1.471322, 2.5944245, 2.5944245, 0.30833125, 0.30833125, 0.3675449, 0.3675449, 0.19031155, 1.8227236, 1.8227236, -0.5215797, 0.9606934, 1.3290628, 1.3290628, 2.759355, 2.759355, 1.0304383, 0.27560067, 1.4350494, 1.4350494, 0.5072389, 0.3528166, 0.3528166, 1.4013448, 1.4013448, 1.325014, 1.325014, 0.86351967, 2.339625, 2.346647, 2.346647, -0.25957698, 1.471322, 1.5927707, 1.5927707, 0.82998616, 0.30833125, 0.012231983, 1.5692596, 1.5692596, 1.6815767, 1.6815767, 0.9688826, 0.9688826, 1.3891454, 2.0140603, 2.759355, 2.759355, 0.25871643, 0.27560067, 1.4350494, 1.4350494, 0.5072389, -0.1162297, 1.0645851, 1.4013448, 1.4013448, 0.5289436, 0.7243685, 1.3852615, 1.3852615, 2.346647, 2.346647, 0.17879286, 0.9380925, 0.9380925, 0.82998616, 0.82998616, 0.0941923, 0.0941923, 1.5692596, 1.5692596, 0.8924739, 0.8924739, 0.9688826, 0.9688826, 1.3891454, 2.0140603, 2.0140603, -0.04932407, 0.2390336, 0.2390336, 1.6739857, 1.6739857, 1.5634048, 1.5634048, 1.0645851, 1.0645851, 0.43310794, 0.43310794, 0.7243685, 1.3852615, 1.3852615, 0.70104134, 0.44103292, 0.17879286, 0.2407875, 0.2891205, 0.41287082, 0.41287082, 0.0941923, 0.0941923, -0.35811406, 0.5559627, 0.8924739, 0.8924739, 0.10471403, 0.22805333, 0.22805333, 0.5407736, 0.5407736, -0.04932407, 0.2390336, 0.2390336, 1.6739857, 1.6739857, 1.5634048, 1.5634048, -0.790523, 0.22425222, 0.22425222, 0.2149656, 0.2149656, 1.0156653, 1.0156653, 0.70104134, -0.41747734, -1.0974966, 1.7123052, 1.2649833, -0.30078387, 1.1173053, 1.1173053, 0.22246316, 0.22246316, 0.13768983, 0.38141626, 0.7539915, 1.0653155, 1.0653155, 0.9853175, 0.7669197, 0.40262553, 1.6692508, 1.6692508, 0.60815644, 1.1149623, 1.4333525, 1.4333525, 0.43554616, 0.43554616, 0.32830128, 0.03308975, 0.5969065, 0.5969065, -0.15602389, 1.0490932, 3.1709747, 3.1709747, 0.18949963, 1.2649833, 1.2649833, -0.2311016, 0.20984948, 0.20984948, 2.1495745, 2.1495745, 0.73165894, 0.73165894, 0.38141626, 0.6632581, 0.6632581, 0.41540062, 1.5788652, 1.5788652, -0.061638054, -0.061638054, 0.26902407, 0.93874687, 1.2674117, 1.2674117, 0.49949825, 1.2591671, 1.2591671, 0.70411104, 2.5263681, 2.5263681, 1.7699214, 0.3779101, 1.3243587, 1.3243587, -0.1722008, 1.1045785, 1.1045785, -0.2311016, 0.9214084, 0.9214084, 2.1495745, 2.1495745, 0.73165894, 0.73165894, 0.34816924, 0.6632581, 0.82455724, 0.82455724, 1.5788652, 1.5788652, 0.21717963, 0.21717963, 1.4045455, 1.4045455, 1.2674117, 1.2674117, 1.5187594, 1.5187594, 1.2591671, 0.76449746, 2.5263681, 2.5263681, 1.7699214, 1.221385, 1.3243587, 1.3243587, -0.1722008, 1.1045785, 1.1045785, 1.3978963, 1.3978963, 0.9214084, 1.648504, 1.648504, -0.13256802, 1.4261588, 1.4261588, 0.93612915, 0.8326507, 0.8326507, 1.6315974, 1.6315974, 0.37775916, 0.2398671, 1.4045455, 1.4045455, 0.77067304, 0.77067304, 1.8219151, 1.8219151, 0.76449746, 0.76449746, 0.24660219, 0.99213684, 1.9050636, 1.9050636, -0.01477722, -0.033319283, -0.35502872, 0.5310425, 0.5310425, 1.3978963, 1.3978963, 0.9600477, 1.648504, 1.648504, 1.1239053, 1.4261588, 1.4261588, 1.3925184, 1.0375856, 0.8326507, 1.6315974, 1.6315974, 0.5897036, 0.2398671, 1.5848205, 1.5848205, 0.77067304, 0.77067304, 1.8219151, 1.8219151, 1.2961498, 1.2961498, 0.6164593, 0.99213684, 1.9050636, 1.9050636, 0.8805112, 0.08595198, 0.08595198, 0.75194657, 0.5629897, 1.054758, 1.054758, 0.9600477, 1.7746586, 1.7746586, 1.1239053, 0.76943016, 1.3925184, 1.3925184, 1.0375856, 0.30151406, 1.0390965, 1.0390965, 0.5897036, 1.7367749, 1.7367749, 1.5848205, 0.9605572, 0.9605572, 0.8709618, 0.8709618, 2.3207998, 2.3207998, 0.6164593, 0.53659654, 0.43480796, 0.8805112, 0.8805112, 0.732424, 0.08595198, 0.75194657, 0.5629897, 0.2711128, 1.0450233, 1.0450233, 1.7746586, 1.7746586, -0.16221845, 1.151734, 1.151734, 0.33053273, 0.13157398, 0.30151406, 1.0390965, 2.0234718, 2.0234718, 1.7367749, 1.7367749, 2.2436018, 2.2436018, 1.9223248, 1.9223248, 1.4053655, 2.3207998, 2.3207998, 0.53420115, 0.5474806, 0.5474806, 0.54009444, 0.732424, 0.732424, 0.54100823, -0.085115604, 0.966768, 0.966768, 1.0450233, 1.0450233, 0.59903955, 0.5232617, 0.09920486, 1.576299, 1.576299, 0.5023282, 0.16066119, 0.16066119, 1.6085222, 2.0234718, 2.0234718, 0.50538695, 0.35924914, 2.2436018, 2.2436018, 1.9223248, 1.9223248, 1.4053655, 1.6180543, 1.6180543, 0.42258036, 0.5474806, 0.5474806, 0.51283646, 0.6027077, 0.6027077, 0.54100823, -0.085115604, 0.966768, 0.966768, 0.6251187, -0.5216415, 0.5232617, 0.5232617, 0.09920486, 1.576299, 1.576299, 0.5023282, 0.16066119, 0.16066119, 1.6085222, 1.6085222, 1.1452621, 1.1452621, 0.7741606, 0.7741606, 0.10490716, 0.13391292, 0.31943184, 0.31943184, -0.13777922, 1.4961396, 1.4961396, 1.3462211, 1.3462211, 0.51283646, 0.62251914, 0.62251914, -0.5946498, 0.5684589, 0.4804245, 0.6251187, 0.83135104, 0.83135104, 0.48797268, 2.6429358, 2.6429358, 2.290467, 2.290467, 1.6002678, -0.059928004, -0.22007579, 1.3086687, 1.3086687, 1.1452621, 1.5259576, 1.5259576, 0.8820566, 0.8820566, 0.13391292, 0.13391292, 0.3718111, 0.45730963, 1.4961396, 1.4961396, 1.3462211, 1.3462211, 1.5885307, 1.5885307, 0.7747283, 0.7747283, 0.5684589, 0.4804245, 0.4804245, 1.0400863, 1.0400863, 0.88518757, 2.6429358, 2.6429358, 2.290467, 2.290467, 1.6002678, -0.008209882, 1.0429776, 1.0429776, 0.41409135, 0.24676603, 1.5259576, 1.5259576, 0.8820566, 0.8820566, 0.13191175, 0.76877064, 0.76877064, 0.45730963, 0.9623417, 0.9623417, 1.9470986, 1.9470986, 1.5885307, 1.5885307, 1.0212247, 0.7747283, 2.4512298, 0.41133425, 0.0025711823, 1.0400863, 1.0400863, 0.88518757, 1.4737648, 1.6606076, 1.6606076, 1.171041, -0.008209882, -0.008209882, 1.0429776, 1.0983036, 1.0983036, 0.15466884, 1.0415684, 1.0415684, 0.9785673, 0.9785673, 0.13191175, 0.76877064, 0.76877064, -0.4213276, 0.8756957, 0.8756957, 1.9470986, 1.9470986, 0.8091803, 1.0212247, 1.2016978, 1.2016978, 2.4512298, 0.87938994, 0.87938994, 0.63542455, 0.71593887, 0.71593887, 0.8809376, 1.8081318, 1.8081318, 0.43663847, 0.69643867, 0.69643867, 0.65178126, 1.0983036, 1.0983036, 0.025385605, 0.61039174, 0.61039174, 0.9785673, 1.7612661, 1.7612661, 0.7540957, 0.66596717, 0.11255753, 0.50099224, 0.50099224, 0.9361076, 0.9361076, 0.8091803, 0.88649154, 1.2016978, 1.2016978, 0.2847906, 0.87938994, 0.87938994, 0.63542455, 0.71593887, 0.71593887, 0.8809376, 1.8081318, 1.8081318, 0.43663847, 0.69643867, 0.69643867, 0.6878814, 1.3480358, 1.3480358, 2.6805708, 2.6805708, -0.20080851, -0.7401368, 1.7612661, 1.7612661, 0.7540957, 1.3185511, 1.3185511, 0.11255753, 0.067516856, 0.77760416, 0.77760416, 0.33601573, 0.88649154, 0.88649154, -0.27213177, 0.5483281, 0.5483281, 0.19718106, 1.0590272, 1.0590272, 0.5665453, 1.2572197, 1.2572197, 0.49872696, 0.81786186, 0.81786186, 0.6878814, 0.6878814, 1.3480358, 1.3480358, 2.6805708, 2.6805708, -0.16137354, 1.7944489, 1.7944489, 2.2320163, 2.2320163, 1.367415, 1.367415, 0.15364446, 0.15364446, 0.8444543, 0.8444543, 0.28376955, 0.33905926, 0.33905926, -0.58963203, 0.5483281, 1.7406294, 1.7406294, 1.0590272, 1.0590272, 0.87672675, 1.2572197, 1.2572197, 0.94940555, 0.94940555, 0.81786186, 0.23820019, 1.1493794, 1.1493794, 0.29751435, 1.3554426, 1.3554426, 0.58822495, 1.7944489, 1.7944489, 2.2320163, 2.2320163, 1.367415, 1.5877615, 1.5877615, 0.15364446, 0.99675965, 0.99675965, 0.28376955, 0.28376955, 0.079121724, 0.079121724, 0.8505307, 1.7406294, 1.7406294, 0.91722155, 0.91722155, 1.8666831, 1.8666831, 0.75746834, 1.238007, 1.238007, -0.086455286, 0.6234536, 1.1493794, 1.1493794, 0.51291686, 0.64305454, 0.64305454, 0.58822495, 2.5402317, 2.5402317, 0.9265583, 0.55808187, 0.15011522, 1.5877615, 1.5877615, 1.1277837, 1.0329891, 1.0329891, 1.4121517, 1.4121517, 0.079121724, 0.079121724, 0.8505307, -0.14989159, 0.084968135, 0.084968135, 0.5319746, 1.8666831, 1.8666831, 0.75746834, 1.238007, 1.238007, 2.2113044, 2.2113044, 0.8906717, 0.8906717, 0.51291686, 0.22734594, 1.6142496, 1.6142496, 2.5402317, 2.5402317, 0.9265583, 0.55808187, 1.5480669, 1.5480669, 1.1277837, 1.1277837, 1.0329891, 1.0329891, 1.4148741, 1.4148741, 0.15647626, -0.21634398, 0.44284612, 0.21839707, -0.25271067, -0.25271067, 0.6563907, 1.2605689, 1.2605689, 0.16586353, 1.3291413, 1.3291413, 2.2113044, 2.2113044, 1.9367125, 1.9367125, 0.38220277, 0.38220277, 1.6142496, 1.6142496, 1.0085973, 0.52759737, 0.8065638, 0.8065638, 1.5480669, 1.5480669, 1.1483506, 1.4028448, 1.4028448, 0.8246182, 1.4148741, 1.4148741, 0.15647626, -0.21634398, 0.56800735, 0.56800735, 0.08261103, 0.018922104, 0.6563907, 0.6563907, 1.0139722, 1.0139722, 1.3291413, 1.3291413, 1.0097119, 1.0097119, 1.9367125, 2.171165, 2.171165, 0.66207427, 1.1615338, 1.1615338, 2.190898, 2.190898, 0.9836362, 0.8065638, 0.25233144, 1.2098501, 1.2098501, 1.4028448, 1.4587147, 2.1531198, 2.1531198, 0.4683049, 0.6572677, 0.6572677, 0.56800735, 0.56800735, 0.62656426, 0.62656426, 0.018922104, 0.8449956, 1.0139722, 1.0139722, 2.042072, 2.042072, 1.0097119, 1.0097119, 0.72934633, 2.171165, 2.171165, 0.66207427, 0.53916126, 0.53916126, 2.190898, 2.190898, 0.9836362, 0.25233144, 0.60799253, 1.2098501, 1.2098501, 1.5612797, 1.5612797, 2.1531198, 2.1531198, 0.59936935, 0.6572677, 0.6572677, 0.46079433, 0.17124355, 1.4263169, 2.4884417, 2.4884417, 1.6959696, 1.8334354, 1.8334354, 2.042072, 2.042072, 0.46637958, -0.09439251, 0.72934633, 0.8020517, 0.8020517, 0.60413677, 0.60413677, -0.024549423, -0.8358561, 1.966764, 1.966764, 0.24664895, 1.419255, 1.419255, 0.9341797, 1.5612797, 1.5612797, 0.18418126, 0.21301717, 1.279812, 1.279812, -0.2563169, 0.46079433, -0.009657237, 1.4263169, 2.4884417, 2.4884417, 1.6959696, 1.8334354, 1.8334354, 1.801214, 1.801214, 0.46637958, -0.09439251, 0.8962484, 0.8962484, 0.88686466, 1.103694, 1.103694, 0.4005307, 0.13545467, 1.966764, 1.966764, 1.8593464, 1.419255, 1.419255, 0.9341797, 0.86900634, 1.1149246, 1.1149246, 0.024290914, 1.279812, 1.279812, 0.60896236, 0.40088567, 0.0032222164, -0.55598503, 0.19321355, 1.2973421, 1.2973421, 1.001331, 0.8179776, 1.801214, 1.801214, 1.4622141, 1.4622141, 0.8962484, 0.8962484, 0.88686466, 1.103694, 1.103694, 0.77905124, 0.13545467, 0.9687444, 1.8593464, 1.8593464, -0.13482246, 2.6205738, 2.6205738, 0.33510563, 1.1149246, 1.1149246, 0.013748487, 0.8253722, 0.8253722, 0.60896236, 0.0032222164, 0.021857359, 0.042539075, 1.5309323, 1.5309323, 0.18345025, 0.2727964, 0.8179776, 0.8179776, 1.4315678, 1.4622141, 1.4622141, 0.5383296, -0.14244542, -0.12656933, 0.4936441, 0.77905124, 0.77905124, 0.99012136, 0.99012136, 0.9687444, -0.021869909, -0.27120733, 2.6205738, 2.6205738, 1.7025702, 0.06763423, 0.44702417, 0.44702417, 0.8253722, 0.8253722, -0.29832315, 1.4219756, 0.45636335, 0.042539075, 1.5309323, 1.5309323, 0.092447735, -0.099008314, 0.777277, 0.777277, 0.5541276, 0.5541276, 0.5383296, 0.5383296, 0.71966815, 0.71966815, 0.4936441, 0.4936441, 0.48701534, 0.99012136, 0.99012136, -0.015779218, 0.9541896, 0.9541896, 0.54414475, 1.7025702, 1.7025702, 1.7148316, 1.7148316, 0.44702417, 0.17233914, 1.3985621, 1.3985621, 1.4219756, 0.7825789, 1.7975394, 1.7975394, 0.54565734, 0.54565734, 0.1508182, 1.6857923, 1.6857923, 0.5541276, 0.91064566, 0.91064566, -0.029856121, 0.71966815, 0.71966815, 0.038127385, 0.51271755, 0.51271755, 0.27566653, -0.015779218, -0.015779218, 1.2577442, 1.2577442, 0.54414475, 0.4472121, 1.4558079, 1.7148316, 1.7148316, 1.4849256, 0.4595849, 1.3985621, 1.3985621, 1.3905339, 0.7825789, 1.7975394, 1.7975394, 1.2165778, 1.2165778, 2.0347626, 2.0347626, 1.6857923, 0.9508337, 0.91064566, 0.91064566, 0.57275134, 0.67755705, 1.3071839, 1.3071839, 0.51271755, 0.8591042, 0.8591042, 0.8733118, 1.1997361, 1.2577442, 1.2577442, 0.041082226, 0.5934659, 1.4558079, 2.1982963, 2.1982963, 1.4849256, 0.4595849, 0.4595849, 0.28005785, 1.3905339, -0.15503581, 0.078514025, 0.41780058, 1.2165778, 1.2165778, 2.0347626, 2.0347626, 1.5743712, 1.5743712, 1.9887319, 1.9887319, 1.1172835, 0.67755705, 1.3071839, 1.3071839, 0.030546581, 1.2697648, 1.2697648, 0.8733118, 1.1997361, 1.1997361, 0.45615304, 0.7750233, 0.7750233, 0.5934659, 2.1982963, 2.1982963, 0.0012484558, 0.0012484558, -0.15993978, -0.58957994, -0.59815043, 0.4178537, 0.4178537, 0.41780058, 0.41780058, 1.342622, 1.8022423, 1.8022423, 1.5743712, 1.5743712, 1.9887319, 1.9887319, 1.4015235, 0.15843384, 1.054325, 1.054325, 0.030546581, 1.2697648, 1.2697648, 0.017515613, 0.32362577, 0.32362577, -0.020129103, 0.7750233, 0.92914516, 0.92914516, 0.37117255, 0.6436313, 0.68889666, 0.68889666, 0.2746472, -0.6036204, 0.70885956, 0.42281857, 0.64445204, 0.64445204, 0.6635616, 1.342622, 1.342622, 1.3151376, 1.3151376, 0.19782817, 1.0598063, 1.4015235, 1.4015235, 0.7435535, 0.7435535, 0.21035936, 1.36606, 1.555114, 1.555114, 0.61332625, 1.496911, 1.496911, 1.1831195, 0.71889716, 0.92914516, 0.92914516, 0.37117255, 0.6436313, 0.68889666, 0.68889666, 1.2729537, 1.2729537, 0.70885956, 0.42281857, 0.64445204, 0.64445204, 0.6635616, 0.6635616, 1.1936116, 1.1936116, -0.09816121, -0.14735366, 1.0598063, 1.0598063, 0.026246618, 0.7435535, 0.7435535, 0.21035936, 1.36606, 1.555114, 1.555114, 0.61332625, 1.496911, 1.496911, 1.1831195, 0.71889716, 0.14067191, 0.14067191, -0.15901226, 0.24005693, 0.24005693, 0.10015941, 1.2729537, 1.2729537, 0.73018354, 0.73018354, 0.38259813, 0.38259813, 0.87830377, 0.87830377, 1.2815113, 1.9142298, 1.9142298, 1.6266496, 1.6266496, 2.9290962, 2.9290962, 1.0805441, 1.0805441, 0.45616361, 0.45616361, 0.5707885, 1.5427964, 1.5427964, 1.3821819, 1.4867824, 1.4867824, 1.5116394, 1.7064682, 1.7064682, 0.70178336, 0.48334184, 0.50496995, 1.7236962, 1.7236962, 1.414415, 0.3257996, 0.124769524, -1.0126731, 0.32335654, 1.400689, 1.400689, 1.2815113, 1.9142298, 1.9142298, 1.6266496, 1.6266496, -0.0150050875, 0.41609964, 1.0805441, 1.0805441, 0.45616361, 0.66414404, 0.66414404, 1.5427964, 1.8961822, 1.8961822, 0.40955177, 0.40955177, 1.5116394, 1.7064682, 1.7064682, 0.70178336, 0.07328543, 0.5298338, 1.7108365, 1.7108365, 1.414415, -0.063661486, -0.11244375, 0.89786416, 0.89786416, 1.400689, 1.400689, 0.50481546, 1.3031809, 1.3031809, 1.0659268, 0.063896194, -0.9659539, 0.41609964, 0.41609964, 0.12253132, 0.12253132, 0.66414404, 0.66414404, 1.1279566, 1.8961822, 1.8961822, -0.28157544, -0.28157544, 0.44478136, 1.0223923, 1.0223923, 2.2856011, 2.2856011, 0.5298338, 0.8097407, 1.0799313, 1.0799313, 1.094167, -0.11244375, 1.131812, 1.131812, 0.29507577, -0.49941677, 0.1746801, 0.1746801, 1.2505698, 1.2505698, 0.063896194, -0.8633556, 0.14166717, 0.14166717, 1.3440744, 1.3440744, 0.5000167, -0.6289807, 1.1279566, 1.1279566, 1.7345722, 1.7345722, 0.57261336, 0.57261336, 0.87550914, 1.0985817, 2.2856011, 2.2856011, 0.81345224, 0.8097407, 1.0799313, 1.0799313, 1.094167, 0.42425263, 1.6295, 1.6295, -0.3000036, -0.3000036, -0.9325702, 1.7969185, 1.7969185, 1.2505698, -0.19871506, 1.4198639, 1.4198639, 0.6795679, 1.3440744, 1.3440744, 0.5670582, 0.9882406, 1.0700725, 1.0700725, 1.7345722, 1.7345722, 1.1035038, 0.57261336, 1.4099522, 1.4099522, 1.0985817, 0.81345224, 0.81345224, 0.75386566, 0.75386566, 0.36559147, 0.5880377, 1.55701, 1.6295, 1.6295, -0.3000036, -0.18779492, 0.4869373, 1.7969185, 1.7969185, 0.5953003, 1.1641518, 1.4198639, 1.4198639, 0.7289299, 0.6795679, -0.32125893, 0.5670582, 0.9882406, 1.0137506, 1.0137506, 0.50666904, 1.1035038, 1.1035038, 0.5569641, 1.4099522, 1.4099522, 0.77946055, 0.77946055, 0.28791544, 0.28791544, 0.36559147, 0.36559147, 0.924434, 1.55701, 1.55701, 1.0121332, 1.6184593, 1.6184593, 0.57110983, 0.4869373, 0.20579681, 0.5953003, 1.1641518, 1.1641518, 0.7289299, 0.7289299, -0.034475047, 0.072677895, 0.072677895, 0.3327982, 1.0137506, 1.0137506, 0.50666904, 2.4643219, 2.4643219, 1.1320051, 1.1320051, 0.6530373, -0.18844908, 0.2872573, 0.2872573, 1.3363862, 1.3363862, -0.0842233, 0.924434, 0.924434, 1.0121332, 1.7140515, 1.7140515, 1.6184593, 0.7669904, -0.45962644, 0.20579681, 0.3263751, 1.4831287, 1.4831287, 0.23908707, 0.650042, 0.650042, 0.23868157, 0.23868157, -0.11622244, 0.85464275, 0.85464275, 0.92660475, 2.4643219, 2.4643219, 1.1320051, 1.1320051, 0.409295, 0.39710623, 0.49964696, 0.49964696, 1.3363862, 1.3363862, -0.120041125, 0.46146888, -0.046481155, 0.037996013, 1.7140515, 1.7140515, 0.7669904, 0.86214805, 0.86214805, 0.4883706, 0.9659361, 1.4831287, 1.9612269, 1.9612269, 0.650042, 0.650042, 1.0616002, 1.0616002, -0.11622244, 0.830386, 0.830386, 0.92660475, 1.8004627, 1.8004627, 0.51650745, 0.51650745, 0.409295, 0.39710623, 0.49964696, 0.49964696, 0.24622276, 0.24622276, 0.09082593, 1.3970653, 1.497715, 1.497715, 0.5652672, 0.54985684, 0.40713033, 0.86214805, 0.86214805, 0.8989674, 0.9659361, 1.4226048, 1.9612269, 1.9612269, 1.3752254, 1.4994915, 1.4994915, 1.0616002, -0.86050975, 0.830386, 0.830386, 0.8576619, 0.8576619, 0.40807465, 1.0770375, 1.0770375, 0.016491488, 0.5746089, 0.5746089, 0.65880215, 0.65880215, 0.16987796, 0.09082593, 1.3970653, 1.497715, 1.497715, 0.81547195, 0.48096985, 0.48096985, 0.85282975, 0.85282975, 1.0044192, 1.0044192, 1.3141544, 1.3141544, 1.3752254, 1.3752254, 1.9258057, 1.951761, 1.951761, 1.2399405, 0.93858516, 0.5125622, 0.8576619, 0.8576619, -0.1105905, 1.0770375, 1.0770375, -0.22482656, 0.7920415, 0.7920415, 0.65880215, 0.65880215, 0.1118724, -0.22295918, 0.3105647, 0.3105647, 0.81547195, 0.81547195, 0.48096985, 0.69865024, 0.85282975, 0.85282975, 1.0044192, 1.0044192, 0.7315516, 0.6335339, 0.6335339, 0.8927082, 1.9258057, 1.951761, 1.951761, 1.9093426, 1.9093426, 1.6638731, 0.90082276, -0.35911658, -0.50900584, 1.6531498, 1.6531498, 0.38616636, 0.7920415, 0.7920415, 0.38182402, 0.3982961, 0.3982961, -0.67570317, 1.2228775, 0.3105647, 0.9719703, 0.9719703, 0.35026592, 0.69865024, 0.69865024, 0.5216252, 0.49689314, 0.7315516, 0.7315516, 0.3196498, -0.40985453, 0.8927082, 0.8927082, 0.63438666, 1.6292758, 1.9093426, 1.9093426, 1.6638731, 0.90082276, 0.7046427, 0.7046427, 1.6531498, 1.6531498, 0.7703309, 0.47348827, 1.8552462, 1.8552462, 1.4156562, 0.98967946, 0.98967946, 1.2228775, 1.1363881, 0.9719703, 0.9719703, 0.572627, 1.9026182, 1.9026182, -0.09302414, -0.18808974, 1.1778295, 1.1778295, -0.40176374, 0.3750199, 0.7979132, 0.7979132, 0.63438666, 1.6292758, 1.6292758, 0.13906415, -0.8576702, 0.88833886, 0.88833886, 0.7046427, 0.93679523, 0.93679523, 1.3289608, 1.3289608, 1.8552462, 1.8552462, 1.4156562, 0.98967946, 0.98967946, 1.1363881, 1.1363881, 0.67161655, 0.2501139, 0.572627, 1.9026182, 1.9026182, 0.64221096, 1.040239, 1.1778295, 1.1778295, 1.1445535, 1.1445535, 1.283455, 1.283455, 2.1886187, 2.1886187, 0.90072393, 0.90072393, 0.8913641, 0.88833886, 0.88833886, -0.52474195, 0.1265688, 0.7930071, 1.3289608, 1.3289608, 0.86301714, 0.86301714, 0.7271749, 0.7375215, 0.7375215, 0.43311873, 1.3207943, 1.3207943, 0.2501139, -0.28339776, 0.8165349, 0.8165349, 0.64221096, 1.040239, 1.0875463, 1.0875463, 1.1445535, 1.1445535, 1.283455, 1.283455, 2.1886187, 2.1886187, 0.90072393, 0.90072393, 0.8913641, -0.17248231, 0.29577908, 0.8425888, 0.8425888, 0.7930071, 0.7930071, 1.0061071, 1.0061071, 1.9595621, 1.9595621, 0.9423143, 0.7375215, 0.7550497, 1.3207943, 1.3207943, -0.09406245, 0.39756522, 0.8165349, 0.8165349, 0.6006052, 0.6006052, 1.0875463, 1.0875463, 0.73930454, 0.61915493, -0.8907292, 0.9219348, 0.9219348, 0.5225576, 0.5225576, 0.45028883, 0.45028883, 1.2341441, 1.4498476, 1.4498476, 0.8425888, 0.24561642, -0.03299648, 1.0061071, 1.0061071, 2.0118642, 2.0118642, 1.336235, 1.336235, 1.3423537, 0.6021635, 0.8732731, 1.9741, 1.9741, 0.47780856, -0.060137887, 0.6006052, 1.0241649, 1.0241649, 0.24461035, 0.089076206, 0.089076206, 0.26473877, 0.9219348, 0.9219348, 0.59542316, 0.59542316, 0.45028883, 0.45028883, 1.2341441, 1.4498476, 1.4498476, -0.24508175, -0.24508175, 0.3140257, 0.37287602, 0.8862932, 2.0118642, 2.0118642, 1.336235, 1.336235, 1.3423537, 0.6498296, 0.8732731, 1.9741, 1.9741, 1.8571023, 1.8571023, 0.41467425, 1.0241649, 1.0241649, 0.24461035, 0.6500845, 1.8209393, 1.8209393, 1.4540358, 1.4540358, 0.59542316, 1.0703601, 1.0703601, 1.2424939, 1.2424939, 1.060715, 1.060715, 0.66200536, 0.66200536, 0.3140257, 0.37287602, 0.8862932, 0.8862932, 0.7312936, 0.07077408, 0.07077408, 0.8188394, 0.6498296, 0.6498296, 0.4347888, 0.09725088, 1.8571023, 1.8571023, 0.66939247, 0.73860705, 1.3042139, 1.3042139, 0.6500845, 1.8209393, 1.8209393, 1.4540358, 1.4540358, 0.2934714, 1.0703601, 1.0703601, 1.2424939, 1.2424939, 0.65592444, -0.038476657, 0.66200536, 0.66200536, 0.50434357, -0.21653287, -0.21653287, 0.7312936, 0.8123336, 0.8123336, 0.54627186, 0.51226676, 0.51226676, 0.3251987, 0.3251987, 0.09725088, 0.679944, 0.679944, 0.66939247, 2.2542837, 2.2542837, 1.3042139, 0.2598325, -0.6783298, -0.083288394, 1.6028637, 1.6028637, 0.4655892, 1.176787, 1.176787, 1.6973464, 1.6973464, 0.65592444, 0.17462958, 0.982327, 1.0374448, 1.0374448, 0.15919177, -0.5053407, -0.5053407, 0.8123336, 0.8123336, 0.54627186, 0.51226676, 0.51226676, 0.08584311, 0.7510253, 0.7510253, 0.8375479, 1.9608808, 1.9608808, 2.2542837, 2.2542837, 1.2098372, 1.2098372, -0.6783298, -0.083288394, 1.6028637, 1.6028637, 0.78990495, 1.176787, 1.176787, 1.6973464, 1.6973464, 0.7591567, 0.17462958, 0.982327, 1.0374448, 1.0374448, 0.5769559, 0.5769559, 0.23497719, 0.6289996, 0.6289996, 0.31403384, 1.0122606, 1.0122606, 0.92874193, 1.081056, 1.5723304, 1.5723304, 1.9608808, 1.9608808, 0.79388034, 0.04399551, 1.2098372, 1.2098372, 0.32099947, 0.32099947, 0.5922057, 0.5922057, 0.78990495, 1.1022825, 1.1022825, 0.5769346, 0.8183537, 0.8183537, 0.100564204, 0.1066523, 1.5190033, 1.5190033, 1.90134, 1.90134, 0.27441698, 0.6289996, 0.6289996, 0.31403384, 1.0122606, 1.0122606, 0.92874193, 1.081056, 1.5723304, 1.5723304, 0.13998847, 0.79388034, 0.79388034, 0.04399551, 0.49097106, 0.8655252, 1.2740445, 1.2740445, 0.5922057, 0.5922057, 0.37317473, 1.0826722, 1.0826722, 1.072636, 1.072636, 0.8183537, -0.13580984, 1.0222104, 1.5190033, 1.5190033, 1.90134, 1.90134, 1.2466952, 1.2466952, 0.70068014, 0.6966507, -0.20697448, 0.6772459, 0.6772459, 0.022212664, 0.8982406, 0.8982406, 0.13998847, 0.9091885, 0.9091885, 0.08071989, 0.49097106, 0.8655252, 2.3374097, 2.3374097, 0.4693722, 1.5237712, 1.5237712, 1.0826722, 1.0826722, 1.226664, 1.226664, 0.8261257, 0.8261257, 1.0222104, 1.0222104, 0.13603121, 0.13603121, 0.5614499, 1.2466952, 1.2466952, 0.70068014, 0.6966507, -0.20697448, 0.6772459, 0.6772459, 1.3257079, 1.3257079, 0.8982406, 0.09152195, 1.0534397, 1.0534397, 1.0147377, 1.4403037, 1.4403037, 2.3374097, 2.3374097, 1.5148823, 1.6043264, 1.6043264, 0.20806953, -0.4292239, 1.226664, 1.226664, 0.8261257, 0.8261257, -0.057756558, -0.21716312, 0.13603121, 0.13603121, 0.5614499, 0.5614499, -0.33275875, -0.20400788, -0.20400788, 1.2598753, 1.2598753, 0.44786966, 1.3257079, 1.3257079, 1.0121683, 1.3841867, 1.3841867, 1.252002, 1.0147377, 1.4403037, 1.4403037, 1.6264315, 1.6264315, 1.5148823, 1.6043264, 1.6043264, 0.6941287, 0.66396505, 0.8220764, 0.8220764, -0.21321523, -0.15279447, 0.59172696, 0.59172696, -0.13022317, 0.61295235, 0.61295235, 0.12955453, 1.17942, 1.17942, 0.836636, 1.2598753, 1.2598753, 0.24099673, -0.28044778, 0.40508026, 1.0121683, 1.3841867, 1.3841867, 1.252002, 0.71178526, 0.27372944, 0.5620131, 0.7955496, 0.7955496, 0.49585068, 0.49585068, 0.6941287, 0.6941287, 0.66396505, 0.8220764, 0.8220764, 1.2412993, 1.2412993, 0.59172696, 0.59172696, -0.05872619, 0.61295235, 0.61295235, 0.12955453, 1.17942, 1.9457614, 1.9457614, 1.0723007, 1.0723007, -0.3070685, -0.3070685, 0.8528891, 0.8528891, 0.16491273, 0.16491273, 0.71178526, 1.9677297, 1.9677297, 0.44768322, 0.7955496, 0.7955496, 0.49585068, 0.49585068, 0.49908426, 0.49908426, 2.0791767, 2.0791767, 0.31263396, 1.2412993, 1.2412993, 1.5932736, 1.5932736, 0.5687224, 0.25163025, 0.25163025, -0.17724662, 0.19939707, 1.9457614, 1.9457614, 1.0723007, 1.0723007, 0.2791965, 0.004174999, 0.8528891, 0.8528891, 0.13468726, 0.13468726, 0.10233585, 1.9677297, 1.9677297, 0.44768322, -0.15065365, -0.15065365, -0.23469436, 0.8997308, 0.8997308, 0.32003194, 2.0791767, 2.0791767, 0.4723279, 0.9663058, 0.9663058, 1.5932736, 1.5932736, 0.74380016, 0.74380016, 0.5921491, -0.3937337, 0.085252576, 1.6616518, 1.6616518, 2.3602285, 2.3602285, 0.5555456, 0.43952233, 0.99914986, 0.99914986, 2.1600132, 2.1600132, -0.01155552, 0.83949065, 1.0923389, 1.0923389, 0.5668694, 0.5668694, -0.23469436, 0.8997308, 0.8997308, 0.76354146, 1.5703039, 1.5703039, 0.4723279, 0.9663058, 0.9663058, 0.21023144, 0.8961672, 0.8961672, 0.74380016, 1.1781894, 1.1781894, -0.94154567, 1.6616518, 1.6616518, 2.3602285, 2.3602285, 0.5555456, 0.43952233, 0.99914986, 0.99914986, 2.1600132, 2.1600132, -0.100301705, 0.302561, 1.0923389, 1.0923389, 0.5668694, 0.5668694, -0.5062735, -0.48948243, 0.76354146, 0.76354146, 0.1926161, 0.1926161, -0.34341785, -0.84721017, -1.2028884, -1.2028884, 0.8961672, 0.8961672, 0.15865193, 1.1781894, 1.1781894, -0.94154567, 0.25471553, 0.25471553}),
			),
		},
		{
			inputT: tensor.New(tensor.WithShape(1, 3, 32, 32),
				tensor.WithBacking([]float64{1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558, -0.9772779, 0.95008844, -0.1513572, -0.10321885, 0.41059852, 0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324, 0.33367434, 1.4940791, -0.20515826, 0.3130677, -0.85409576, -2.5529897, 0.6536186, 0.8644362, -0.742165, 2.2697546, -1.4543657, 0.045758516, -0.18718386, 1.5327792, 1.4693588, 0.15494743, 0.37816253, -0.88778573, -1.9807965, -0.34791216, 0.15634897, 1.2302907, 1.2023798, -0.3873268, -0.30230275, -1.048553, -1.420018, -1.7062702, 1.9507754, -0.5096522, -0.4380743, -1.2527953, 0.7774904, -1.6138978, -0.21274029, -0.89546657, 0.3869025, -0.51080513, -1.1806322, -0.028182229, 0.42833188, 0.06651722, 0.3024719, -0.6343221, -0.36274117, -0.67246044, -0.35955316, -0.8131463, -1.7262826, 0.17742614, -0.40178093, -1.6301984, 0.46278226, -0.9072984, 0.051945396, 0.7290906, 0.12898292, 1.1394007, -1.2348258, 0.40234163, -0.6848101, -0.87079716, -0.5788497, -0.31155252, 0.05616534, -1.1651498, 0.9008265, 0.46566245, -1.5362437, 1.4882522, 1.8958892, 1.1787796, -0.17992483, -1.0707526, 1.0544517, -0.40317693, 1.222445, 0.20827498, 0.97663903, 0.3563664, 0.7065732, 0.01050002, 1.7858706, 0.12691209, 0.40198937, 1.8831507, -1.347759, -1.270485, 0.9693967, -1.1731234, 1.9436212, -0.41361898, -0.7474548, 1.922942, 1.4805148, 1.867559, 0.90604466, -0.86122566, 1.9100649, -0.26800337, 0.8024564, 0.947252, -0.15501009, 0.61407936, 0.9222067, 0.37642553, -1.0994008, 0.2982382, 1.3263859, -0.69456786, -0.14963454, -0.43515354, 1.8492638, 0.67229474, 0.40746182, -0.76991606, 0.5392492, -0.6743327, 0.031830557, -0.6358461, 0.67643327, 0.57659084, -0.20829876, 0.3960067, -1.0930616, -1.4912575, 0.4393917, 0.1666735, 0.63503146, 2.3831449, 0.94447947, -0.91282225, 1.1170163, -1.3159074, -0.4615846, -0.0682416, 1.7133427, -0.74475485, -0.82643855, -0.09845252, -0.6634783, 1.1266359, -1.0799315, -1.1474687, -0.43782005, -0.49803245, 1.929532, 0.9494208, 0.08755124, -1.2254355, 0.844363, -1.0002153, -1.5447711, 1.1880298, 0.3169426, 0.9208588, 0.31872764, 0.8568306, -0.6510256, -1.0342429, 0.6815945, -0.80340964, -0.6895498, -0.4555325, 0.017479159, -0.35399392, -1.3749512, -0.6436184, -2.2234032, 0.62523144, -1.6020577, -1.1043833, 0.05216508, -0.739563, 1.5430146, -1.2928569, 0.26705086, -0.039282817, -1.1680934, 0.5232767, -0.17154633, 0.77179056, 0.82350415, 2.163236, 1.336528, -0.36918184, -0.23937918, 1.0996596, 0.6552637, 0.64013153, -1.616956, -0.024326125, -0.7380309, 0.2799246, -0.09815039, 0.9101789, 0.3172182, 0.78632796, -0.4664191, -0.94444627, -0.4100497, -0.017020414, 0.37915173, 2.259309, -0.042257152, -0.955945, -0.34598178, -0.463596, 0.48148146, -1.540797, 0.06326199, 0.15650654, 0.23218104, -0.5973161, -0.23792173, -1.424061, -0.49331987, -0.54286146, 0.41605005, -1.1561824, 0.7811981, 1.4944845, -2.069985, 0.42625874, 0.676908, -0.63743705, -0.3972718, -0.13288058, -0.29779088, -0.30901298, -1.6760038, 1.1523316, 1.0796186, -0.81336427, -1.4664243, 0.5210649, -0.57578796, 0.14195317, -0.31932843, 0.69153875, 0.6947491, -0.7255974, -1.383364, -1.5829384, 0.6103794, -1.1888592, -0.5068163, -0.596314, -0.052567296, -1.9362798, 0.1887786, 0.52389103, 0.08842209, -0.31088617, 0.097400166, 0.39904633, -2.7725928, 1.9559124, 0.39009333, -0.6524086, -0.39095336, 0.49374178, -0.11610394, -2.0306845, 2.064493, -0.11054066, 1.0201727, -0.69204986, 1.5363771, 0.2863437, 0.60884386, -1.0452534, 1.2111453, 0.68981814, 1.3018463, -0.6280876, -0.48102713, 2.3039167, -1.0600158, -0.1359497, 1.1368914, 0.09772497, 0.5829537, -0.39944902, 0.37005588, -1.3065269, 1.6581306, -0.11816405, -0.6801782, 0.6663831, -0.4607198, -1.3342584, -1.3467175, 0.69377315, -0.15957344, -0.13370156, 1.0777438, -1.1268258, -0.7306777, -0.3848798, 0.09435159, -0.042171452, -0.2868872, -0.0616264, -0.10730527, -0.7196044, -0.812993, 0.27451634, -0.8909151, -1.1573553, -0.31229225, -0.15766701, 2.2567234, -0.7047003, 0.9432607, 0.7471883, -1.1889449, 0.77325296, -1.1838807, -2.6591723, 0.60631955, -1.7558906, 0.45093447, -0.6840109, 1.6595508, 1.0685093, -0.4533858, -0.6878376, -1.2140774, -0.44092262, -0.28035548, -0.36469355, 0.15670386, 0.5785215, 0.34965447, -0.76414394, -1.4377915, 1.3645319, -0.6894492, -0.6522936, -0.52118933, -1.8430696, -0.477974, -0.4796558, 0.6203583, 0.6984571, 0.003770889, 0.93184835, 0.339965, -0.015682112, 0.16092817, -0.19065349, -0.3948495, -0.26773354, -1.1280113, 0.2804417, -0.9931236, 0.8416313, -0.24945858, 0.04949498, 0.4938368, 0.6433145, -1.5706234, -0.20690368, 0.8801789, -1.6981058, 0.38728046, -2.2555642, -1.0225068, 0.038630553, -1.6567152, -0.98551077, -1.471835, 1.648135, 0.16422775, 0.5672903, -0.2226751, -0.35343176, -1.6164742, -0.29183736, -0.7614922, 0.8579239, 1.1411018, 1.4665787, 0.85255194, -0.5986539, -1.1158969, 0.7666632, 0.3562928, -1.7685385, 0.3554818, 0.8145198, 0.058925588, -0.18505368, -0.8076485, -1.4465348, 0.800298, -0.30911446, -0.23346666, 1.7327212, 0.6845011, 0.370825, 0.1420618, 1.5199949, 1.7195894, 0.9295051, 0.5822246, -2.094603, 0.12372191, -0.13010696, 0.09395323, 0.9430461, -2.7396772, -0.56931204, 0.26990435, -0.46684554, -1.4169061, 0.8689635, 0.27687192, -0.97110456, 0.3148172, 0.8215857, 0.005292646, 0.8005648, 0.078260176, -0.39522898, -1.1594205, -0.085930765, 0.19429293, 0.87583274, -0.11510747, 0.4574156, -0.964612, -0.78262913, -0.1103893, -1.0546285, 0.8202478, 0.46313033, 0.27909577, 0.3389041, 2.0210435, -0.4688642, -2.2014413, 0.1993002, -0.050603542, -0.51751906, -0.97882986, -0.43918952, 0.18133843, -0.5028167, 2.4124537, -0.96050435, -0.79311734, -2.28862, 0.25148442, -2.0164065, -0.53945464, -0.27567053, -0.70972794, 1.7388726, 0.99439436, 1.3191369, -0.8824188, 1.128594, 0.49600095, 0.77140594, 1.0294389, -0.90876323, -0.42431763, 0.86259604, -2.6556191, 1.5133281, 0.55313206, -0.045703962, 0.22050765, -1.0299352, -0.34994337, 1.1002843, 1.298022, 2.696224, -0.07392467, -0.65855294, -0.51423395, -1.0180418, -0.07785475, 0.38273242, -0.03424228, 1.0963469, -0.2342158, -0.34745064, -0.5812685, -1.6326345, -1.5677677, -1.179158, 1.3014281, 0.8952603, 1.3749641, -1.3322116, -1.9686247, -0.6600563, 0.17581895, 0.49869028, 1.0479722, 0.28427967, 1.7426687, -0.22260568, -0.9130792, -1.6812183, -0.8889713, 0.24211796, -0.8887203, 0.9367425, 1.4123276, -2.369587, 0.8640523, -2.239604, 0.40149906, 1.2248706, 0.064856105, -1.2796892, -0.5854312, -0.26164544, -0.18224478, -0.20289683, -0.10988278, 0.21348006, -1.2085737, -0.24201983, 1.5182612, -0.38464543, -0.4438361, 1.0781974, -2.5591846, 1.1813786, -0.63190377, 0.16392857, 0.09632136, 0.9424681, -0.26759475, -0.6780258, 1.2978458, -2.364174, 0.020334182, -1.3479254, -0.7615734, 2.0112567, -0.044595428, 0.1950697, -1.7815628, -0.7290447, 0.1965574, 0.3547577, 0.61688656, 0.008627899, 0.5270042, 0.4537819, -1.8297404, 0.037005723, 0.76790243, 0.5898798, -0.36385882, -0.8056265, -1.1183119, -0.13105401, 1.1330799, -1.9518042, -0.6598917, -1.1398025, 0.7849575, -0.5543096, -0.47063765, -0.21694957, 0.44539326, -0.392389, -3.046143, 0.5433119, 0.43904296, -0.21954103, -1.0840366, 0.35178012, 0.37923554, -0.47003287, -0.21673147, -0.9301565, -0.17858909, -1.5504293, 0.41731882, -0.9443685, 0.23810315, -1.405963, -0.5900577, -0.110489406, -1.6606998, 0.115147874, -0.37914756, -1.7423562, -1.3032428, 0.60512006, 0.895556, -0.13190864, 0.40476182, 0.22384356, 0.32962298, 1.285984, -1.5069984, 0.67646074, -0.38200897, -0.22425893, -0.30224973, -0.3751471, -1.2261962, 0.1833392, 1.670943, -0.05613302, -0.0013850428, -0.687299, -0.11747455, 0.46616644, -0.37024245, -0.45380405, 0.40326455, -0.91800475, 0.25249663, 0.8203218, 1.3599485, -0.09038201, 1.3675972, 1.0344099, -0.99621266, -1.2179385, -0.30496365, 1.0289356, -0.07228701, -0.6006576, 1.5522432, 0.28690448, -2.3205943, 0.31716064, 0.52004063, 0.22560866, 0.4497121, -0.067275606, -1.3183959, -0.370704, -0.94561577, -0.9327409, -1.2630683, 0.45248908, 0.097896144, -0.44816536, -0.64933795, -0.023423105, 1.0791948, -2.0042157, 0.37687653, -0.545712, -1.8845859, -1.945703, -0.9127835, 0.21950956, 0.39306292, -0.9389816, 1.017021, 1.4229835, 0.39608657, -0.59140265, 1.1244192, 0.7553957, 0.86740744, -0.6564637, -2.8345544, 2.116791, -1.6108783, -0.035768073, 2.3807454, 0.33057675, 0.94924647, -1.5023966, -1.7776669, -0.5327028, 1.0907497, -0.34624946, -0.7946363, 0.19796729, 1.0819352, -1.4449402, -1.210543, -0.7886692, 1.0946383, 0.23482153, 2.1321535, 0.9364457, -0.035095178, 1.2650778, 0.21149701, -0.70492136, 0.67997485, -0.6963267, -0.2903971, 1.3277828, -0.10128149, -0.8031414, -0.46433768, 1.0217906, -0.55254066, -0.38687086, -0.51029277, 0.1839255, -0.38548976, -1.6018361, -0.8871809, -0.932789, 1.2433194, 0.81267405, 0.58725935, -0.50535834, -0.81579155, -0.5075176, -1.0518801, 2.4972005, -2.2453218, 0.56400853, -1.2845523, -0.10434349, -0.98800194, -1.177629, -1.1401963, 1.7549862, -0.13298842, -0.7657022, 0.55578697, 0.010349315, 0.72003376, -1.8242567, 0.30360392, 0.7726948, -1.6615983, 0.44819528, 1.6961815, -0.014857704, 0.82140595, 0.67057043, -0.7075057, 0.039766736, -1.5669947, -0.45130304, 0.26568797, 0.7231005, 0.024612125, 0.71998376, -1.1029062, -0.10169727, 0.019279385, 1.8495913, -0.21416666, -0.49901664, 0.021351224, -0.91911346, 0.19275385, -0.3650552, -1.7913276, -0.058586553, -0.3175431, -1.6324233, -0.06713416, 1.4893559, 0.5213038, 0.6119272, -1.3414967, 0.47689837, 0.14844958, 0.5290452, 0.4226286, -1.3597807, -0.041400813, -0.75787085, -0.050084095, -0.8974009, 1.3124703, -0.8589724, -0.8989422, 0.07458641, -1.0770991, -0.4246633, -0.8299646, 1.411172, 0.78580385, -0.057469517, -0.39121705, 0.9409176, 0.4052041, 0.49805242, -0.026192237, -1.68823, -0.112465985, -0.5324899, 0.6450553, 1.0118425, -0.65795106, 0.46838522, 1.735879, -0.66771275, 1.6819217, -0.85258585, 0.022959756, -0.011145612, 0.0114989, -0.837678, -0.5911831, -0.66772026, 0.3269626, 0.33003512, 2.2259443, 1.370989, -0.50984323, 0.3248696, 0.997118, 0.030601824, -0.069641575, 0.05157494, 0.8672766, -0.84832054, -0.32566947, 0.47043315, 0.31144708, 0.23958276, -0.36980116, 0.9725358, 2.1338682, 0.4064155, -0.1931767, 0.7557403, -0.53913265, -0.74969035, 0.032808747, -2.5827966, -1.1539503, -0.34796184, -1.3533889, -1.0326431, -0.43674833, -1.6429653, -0.40607178, -0.53527015, 0.025405208, 1.154184, 0.17250441, 0.021062022, 0.099454455, 0.22739278, -1.0167387, -0.11477532, 0.30875126, -1.37076, 0.8656529, 1.0813761, -0.63137597, -0.24133779, -0.87819034, 0.69938046, -1.0612223, -0.222477, -0.8589199, 0.05095428, -1.7942293, 1.3264617, -0.9646064, 0.059894685, -0.21252304, -0.7621145, -0.88778013, 0.93639857, -0.5256406, 0.2711702, -0.80149686, -0.64718145, 0.47224715, 0.9304085, -0.17531641, -1.4219198, 1.997956, -0.8565493, -1.5415874, 2.5944245, -0.4040323, -1.4617327, -0.6834398, 0.3675449, 0.19031155, -0.8517292, 1.8227236, -0.5215797, -1.1846865, 0.9606934, 1.3290628, -0.8174931, -1.4013473, 1.0304383, -2.0473237, -1.2266216, 0.96744615, -0.055352546, -0.26393735, 0.3528166, -0.15277442, -1.2986867, 1.2760754, 1.325014, 0.20533256, 0.045134015, 2.339625, -0.27643284, -0.25957698, 0.36448124, 1.471322, 1.5927707, -0.25857264, 0.30833125, -1.3780835, -0.3119761, -0.84029037, -1.0068318, 1.6815767, -0.79228663, -0.5316059, 0.36584878, 1.2978252, 0.48111513, 2.759355, -0.074667975, 0.25871643, 0.27560067, 1.4350494, 0.5072389, -0.1162297, -0.9474886, 0.24444346, 1.4013448, -0.4103818, 0.5289436, 0.24614778, 0.86351967, -0.8047537, 2.346647, -1.2791611, -0.36555108, 0.9380925, 0.29673317, 0.82998616, -0.49610233, -0.074804984, 0.012231983, 1.5692596, 0.69042903, 0.7966721, -0.6579261, 0.9688826, 0.22558166, 1.3891454, 2.0140603, -0.30676576, -0.40630314, -0.86404496, -0.14357951, -0.38202545, 0.3595044, -0.14456682, -0.36159927, 1.0645851, -0.9378802, 0.43310794, -0.40594172, 0.7243685, 1.3852615, -0.30309826, 0.44103292, 0.17879286, -0.7994224, 0.2407875, 0.2891205, 0.41287082, -0.1983989, 0.0941923, -1.1476109, -0.35811406, 0.5559627, 0.8924739, -0.42231482, 0.10471403, 0.22805333, 0.20147994, 0.5407736, -1.8180777, -0.04932407, 0.2390336, -1.0003303, 1.6739857, 0.16155927, 1.5634048, -0.790523, -0.9073001, 0.22425222, -1.6786884, 0.2149656, 0.09721923, 1.0156653, 0.70104134, -0.41747734, -1.0974966, 1.7123052, -0.79211503, -1.0455246, -1.084856, 1.1173053, -0.5189002, -0.7537045, 0.13768983, -0.2069447, -0.67809546, 0.7539915, 1.0653155, 0.9853175, 0.7669197, 0.40262553, -1.775888, 1.6692508, 0.3019892, 0.60815644, 1.1149623, 1.4333525, 0.41839802, 0.43554616, -0.59922427, 0.03308975, -0.85416126, -0.71994054, -0.8935744, -0.15602389, 1.0490932, 3.1709747, 0.18949963, -1.3484131, 1.2649833, -0.30078387, -0.6606086, 0.20984948, -1.2406245, 0.22246316, -0.08837552, 0.098377906, 0.38141626, 0.067492254, 0.016338084, 0.2843145, 0.41540062, -1.0314825, -1.4299912, -0.061638054, -1.4327354, 0.08753147, 0.93874687, 0.6071117, -1.0481704, -0.86026245, 0.32830128, -0.4012978, -0.3166553, 0.5969065, -0.9872867, -0.40123472, -0.8000825, -1.0431294, -0.8570782, 0.67746216, 0.05182039, -0.87916064, -0.2311016, -1.6388073, -0.7333128, 2.1495745, -0.090243846, 0.73165894, -0.065488376, 0.34816924, 0.6632581, -1.1046166, -0.030936258, 1.5788652, -0.7955006, -0.56643987, -0.30769128, 0.26902407, 0.52491784, 1.2674117, 0.49949825, -0.062053125, 1.2591671, 0.70411104, -1.4956795, 2.5263681, 1.7699214, -0.16821422, 0.3779101, 1.3243587, -0.1722008, 0.7303518, 1.1045785, -1.0148259, -0.6023319, 0.9214084, 0.46081448, 0.92379653, -0.13256802, -0.28900522, -1.9986395, -1.1460004, 0.047066096, 0.82455724, 0.53117836, -0.12824197, -0.27177158, 0.21717963, 0.07821118, 1.4045455, 0.14644077, -1.481246, -1.2725581, 1.5187594, -1.1711605, 0.76449746, -0.26837274, -0.16975829, -0.13413279, 1.221385, -0.19284183, -0.033319283, -1.5308034, 0.2066905, 0.5310425, 0.23914558, 1.3978963, 0.055171356, 0.29897746, 1.648504, -1.5500141, -0.45582536, 1.4261588, 0.93612915, 0.6783801, 0.8326507, 0.3270662, 1.6315974, 0.37775916, 0.2398671, 0.15895867, 0.19286396, -1.1570172, 0.77067304, -0.13043973, 1.8219151, -0.07565047, 0.4209183, 0.24660219, -0.625557, 0.99213684, 1.9050636, -0.01477722, -0.3004788, -0.35502872, -1.8923619, -0.17781314, 0.2509981, 1.054758, 0.9600477, -0.41649908, -0.27682298, 1.1239053, -0.1734639, -0.51002955, 1.3925184, 1.0375856, 0.018791791, -0.5937774, -2.0118804, 0.5897036, -0.8963697, -1.962732, 1.5848205, 0.6479678, -1.1390082, -1.2144014, 0.8709618, -0.87797064, 1.2961498, 0.6164593, 0.53659654, 0.40469545, 0.19145088, 0.8805112, -0.45408037, 0.08595198, 0.75194657, 0.5629897, -1.1949868, -0.50040966, 0.2528035, -0.4080147, 1.7746586, -0.3931532, -0.16221845, 0.76943016, 0.33053273, -0.14527446, -0.7564935, 0.30151406, 1.0390965, 0.47909522, -0.7781835, 1.7367749, -1.4465779, -1.5826856, 0.9605572, 0.22584048, -0.54949856, -1.0985707, 2.3207998, 0.11709087, 0.53420115, 0.3178851, 0.43480796, 0.54009444, 0.732424, -0.3752224, -0.29164198, -1.7410228, -0.78030443, 0.2711128, 1.0450233, 0.59903955, -0.34069234, -1.2631729, -2.7773592, 1.151734, -0.589229, -0.44846502, 0.13157398, -1.40556, -0.34978217, 2.0234718, 0.50538695, 0.35924914, -1.5824945, 2.2436018, -1.4227949, 1.9223248, -2.115056, 1.4053655, 1.6180543, -0.8244091, 0.42258036, 0.5474806, -0.8137945, -1.4491177, -1.3177173, 0.54100823, -0.085115604, -0.564301, 0.966768, 0.5080679, -0.7554627, -1.2012016, 0.5232617, -0.53758335, 0.09920486, 1.576299, 0.5023282, -0.862267, 0.16066119, -0.95264494, 1.6085222, -0.56157875, 0.20727074, 0.30773258, 0.15925047, -1.9585489, -1.446421, -0.4523503, 0.31943184, -0.13777922, -0.9571475, -1.3484243, -0.40155753, -0.46847606, 0.51283646, -0.32631847, 0.6027077, -0.5946498, -0.25595766, -0.3480464, -0.782367, 0.6251187, -0.813596, -0.5216415, -0.07311965, -1.2973796, -0.32493496, -0.71130633, -0.38815418, -0.059928004, -0.79991364, -0.22007579, 1.3086687, -0.025798557, 1.1452621, 0.34649444, 0.7741606, -0.77445894, 0.10490716, 0.13391292, -0.6126257, -0.82282835, -1.4902654, 1.4961396, -0.9724029, 1.3462211, -0.46749318, -0.8624933, 0.62251914, -0.63119197, 0.5684589, -0.33281177, 0.4804245, -0.9681861, 0.83135104, 0.48797268, -0.9196507, 2.6429358, 0.54012305, 2.290467, 1.6002678, -0.18883479, -0.41227177, -0.4034592, -1.8300285, -0.6958351, 0.24676603, 1.5259576, -0.7727719, 0.8820566, -1.2525934, -0.58632004, -0.4576406, 0.3718111, 0.45730963, 0.9623417, 0.77083695, 0.24316822, 0.39036494, 1.5885307, -0.5109262, 0.7747283, -1.808144, 0.41133425, -0.48324955, 0.0025711823, 1.0400863, 0.16464381, 0.88518757, 1.4737648, 0.38909397, 1.171041, -0.32656097, -0.008209882, -0.5226194, 1.0429776, 0.41409135, -0.50723445, 0.15466884, 1.0415684, -0.03926799, -0.9489328, 0.13191175, -1.9805655, 0.76877064, -0.4213276, -0.46931073, 0.8756957, -1.3651628, 1.9470986, -0.48024204, -0.52325094, 1.0212247, 0.7086953, 2.4512298, -0.21120599, -0.120406635, -1.479316, -0.33210227, -0.7214313, -0.448767, -1.7441877, 1.6606076, -1.4166034, -2.8022027, -1.1884245, -0.6038396, -1.149554, 1.0983036, -0.13783918, 0.025385605, 0.61039174, 0.28601253, 0.9785673, -1.1094775, -0.5475181, 0.66596717, -2.5345545, -1.3751845, 0.50099224, -0.48024905, 0.9361076, 0.8091803, -1.1980929, 0.4066571, 1.2016978, 0.1474344, -0.97746485, 0.87938994, 0.63542455, 0.54261076, 0.71593887, -2.994613, 0.8809376, 1.8081318, 0.43663847, 0.192729, 0.69643867, 0.33822548, 0.65178126, 0.0014710003, -0.76670486, -1.0043228, -0.9981917, -1.3730426, -1.067742, 1.7612661, 0.7540957, -0.6250274, -0.3903927, 0.11255753, -0.65554506, 0.067516856, 0.77760416, -0.035742734, 0.33601573, 0.88649154, -0.27213177, 0.2847906, -0.30937758, -0.02852887, -0.32473028, -0.52886987, 0.17371185, 0.5665453, 0.14630444, 0.49872696, -0.7379318, -1.2037352, 0.4170435, 0.6878814, 0.049857266, 1.3480358, 0.9076988, 2.6805708, -0.20080851, -0.9988488, -0.7401368, -0.5654978, 0.4760314, -2.1580687, 1.3185511, -0.23929659, -0.24679355, -1.0793432, -0.11422555, 0.013239767, -0.12194493, 0.33905926, -0.58963203, -0.8958158, 0.5483281, 0.09866745, 0.19718106, 1.0590272, -1.0225644, -0.85524046, 1.2572197, -1.4828833, -1.3094121, 0.81786186, 0.23820019, 0.105232134, -0.09165941, 0.031267546, -0.09211212, 1.3554426, -0.39814812, -0.16137354, 1.7944489, 0.027509702, 2.2320163, -0.1049797, 1.367415, -1.655344, 0.15364446, -1.5844736, 0.8444543, -1.2128679, 0.28376955, -0.28219587, -1.1582032, -1.61936, -0.51104045, 1.7406294, -0.29348505, 0.91722155, -0.057042867, 0.87672675, -1.8269113, -0.40318832, 0.94940555, -0.16325495, -0.086455286, -0.4304619, 1.1493794, 0.29751435, 0.044022277, 0.64305454, 0.58822495, 0.21258704, 1.5470315, -0.060287535, 0.27808106, -0.64295256, 0.15011522, 1.5877615, -0.6432576, -1.1335928, 0.99675965, -0.14876615, 0.0960042, -0.045113303, 0.079121724, 0.8505307, -0.8391242, -1.0117741, 0.084968135, -1.6064397, -1.3730536, 1.8666831, 0.75746834, -0.010056471, 1.238007, -1.0405992, -0.31560314, 0.6234536, 0.8906717, 0.51291686, -2.5412388, -0.96808213, 0.4770681, -0.3559515, 2.5402317, 0.9265583, 0.55808187, -1.1169496, -0.03529674, 0.24120396, 1.1277837, 0.8811311, 1.0329891, -0.923912, 1.4121517, -1.3804307, -0.53591454, 0.43077114, -0.14989159, -1.0060369, -0.82154983, -1.5482544, 0.5319746, 1.2605689, -0.100393504, -0.4003488, -1.472323, 0.9132019, 2.2113044, -1.7974558, -1.0634329, -0.679593, -0.5643179, 0.22734594, 1.6142496, 1.0085973, 0.52759737, -0.7239287, -1.1196282, -0.7967753, 1.5480669, -0.0617433, -0.44683626, -0.18375573, 0.8246182, -1.3128496, 1.4148741, 0.15647626, -0.21634398, 0.44284612, 0.21839707, -0.34419647, -0.25271067, -0.86886257, 0.6563907, -0.5319938, -0.9562584, 0.16586353, 1.3291413, -0.048344623, -0.60810125, 0.40389603, 1.9367125, -1.4519055, 0.38220277, 0.20508662, 1.1615338, 0.99090916, -0.1867091, -1.6845173, 0.8065638, -0.8351927, -0.9467404, 1.1483506, -0.9108504, 1.4028448, 0.33584473, 0.3191184, 0.30726478, -1.6384237, -1.7763886, 0.21555306, 0.56800735, 0.08261103, -0.8215345, 0.018922104, -0.082034156, -0.9571581, 1.0139722, -1.7302761, 0.58874243, 0.38432342, 1.0097119, -1.0053118, 0.10140715, 2.171165, 0.66207427, 0.10058121, 0.53916126, 0.08617684, 2.190898, 0.9836362, -0.08561496, 0.25233144, -0.390798, 1.2098501, -1.4061048, -1.6047385, 1.4587147, 2.1531198, 0.4683049, 0.11273794, 0.6572677, -0.64705354, 0.17124355, 0.038908705, 0.62656426, -1.5579985, -0.5070348, 0.8449956, -0.67559385, -0.99336135, 2.042072, 0.038118, -0.57891816, -1.6923704, 0.72934633, 0.69913614, -0.2987596, -1.1022302, -0.024549423, -0.8358561, -0.9420936, -0.10321275, -1.0513904, 0.24664895, 0.60799253, -0.83963245, -1.3682451, 1.5612797, -0.94027025, -0.6599427, 0.21301717, 0.59936935, -0.2563169, 0.46079433, -0.40098616, -0.97117066, 1.4263169, 2.4884417, 1.6959696, 0.14180663, 1.8334354, 0.3557035, -0.47728628, 0.46637958, -0.09439251, -0.9831182, -0.898322, 0.8020517, -1.846532, 0.60413677, -1.6295836, -2.1211765, -1.8388466, 1.966764, -0.19623396, 0.08658318, 1.419255, 0.9341797, -1.3915052, 0.86900634, 0.18418126, -0.34167808, 0.024290914, 1.279812, -0.8859665, 0.40088567, -0.009657237, -1.7971646, -0.8022532, 0.19321355, 1.2973421, 1.001331, 0.5972125, -0.81527567, 1.801214, 0.21524046, -1.0063655, -0.18290497, 0.8962484, 0.0076174983, 0.88686466, 1.103694, 0.4005307, -0.8577026, 0.13545467, 0.045165855, 1.8593464, -1.6263219, -0.13482246, -0.5840936, 0.33510563, -2.4375644, 1.1149246, 0.013748487, -1.8447012, -0.36111313, 0.60896236, -1.5914478, 0.0032222164, -1.0574737, -0.55598503, 0.026738383, 0.18345025, -0.4707425, 0.2727964, 0.8179776, -0.27891427, 1.4315678, 1.4622141, -0.42870206, -0.63784057, -1.664173, -0.12656933, -0.36343777, 0.77905124, -1.5096616, -0.2773914, 0.9687444, -0.7303571, -0.7623615, -1.4469403, 2.6205738, -0.7474732, -1.3003469, -0.8038504, -0.7742951, -0.26938978, 0.8253722, -0.29832315, -0.9228233, -1.4513385, 0.021857359, 0.042539075, 1.5309323, 0.092447735, -0.099008314, -1.0506538, -0.30595258, -0.43847445, -0.37016416, -0.9592554, 0.5383296, -0.14244542, -0.20035347, -1.7140461, 0.4936441, 0.48701534, -0.8391294, 0.99012136, -1.3647583, -0.021869909, -0.27120733, -1.3171748, 0.18970262, 1.7025702, 0.06763423, -0.46302176, 0.44702417, 0.10572, 0.027762132, -0.4255422, 1.4219756, 0.45636335, -0.52867067, -0.10800384, -0.7408667, -0.60829115, -0.64072573, -1.1343116, 0.777277, -0.29104146, 0.5541276, -0.6701259, -0.060362495, -0.7110406, 0.71966815, -0.2484193, -0.7308736, -1.6417032, 0.27566653, -0.70838505, -0.015779218, -0.4917301, 0.9541896, 0.54414475, 0.4472121, -0.6161211, 0.46629006, 1.7148316, -0.83218604, 0.17233914, -1.649217, 1.3985621, -0.39791209, 0.7825789, -1.7232282, 1.7975394, -0.35687152, 0.54565734, 0.1508182, -0.25547078, 1.6857923, -1.6480463, 0.29871365, 0.91064566, -0.029856121, -0.11817078, -0.14268771, -1.2276365, 0.038127385, 0.51271755, 0.068599224, -0.2722761, -0.48972502, -0.27929667, 1.2577442, -2.0866349, 0.040071458, -0.3277549, 1.4558079, 0.055492226, 1.4849256, -2.12389, 0.4595849, 0.28005785, 1.3905339, -1.6413486, -0.15503581, 0.06606026, -0.4957955, 1.2165778, -0.33868217, 2.0347626, 1.0541779, 0.9508337, 0.559299, -1.0636955, -0.43109635, 0.57275134, 0.67755705, 1.3071839, -0.46744102, -0.8601534, 0.8591042, -0.8096266, 0.8733118, 1.1997361, 0.45615304, -0.35757902, 0.041082226, 0.5934659, 0.010185518, 2.1982963, -0.9906709, -1.0026686, -0.9768954, -0.58957994, -2.1789315, -0.6296504, -0.6532847, 0.078514025, 0.41780058, -1.2402164, 0.9000542, 1.8022423, -0.20828511, 1.5743712, 0.1989895, 1.9887319, 1.1172835, -1.5639046, 0.01862737, 1.054325, 0.030546581, -0.03688353, 1.2697648, -0.7098542, 0.017515613, 0.32362577, -0.33379096, -0.020129103, 0.7750233, 0.43283764, -0.80871755, -1.104124, -0.7891022, 0.0012484558, -0.15993978, -0.8319575, -0.59815043, -1.5200393, 0.4178537, -0.040018726, -1.2597873, 0.028620504, 1.342622, -0.7399359, 1.3151376, -0.32345748, 0.19782817, 0.097750805, 1.4015235, 0.15843384, -1.1419014, -1.3109704, -1.5329211, -1.7119702, 0.04613506, -0.9583745, -0.08081161, -0.70385903, -0.7707843, -0.48084533, 0.70358557, 0.92914516, 0.37117255, -0.98982257, 0.6436313, 0.68889666, 0.2746472, -0.6036204, 0.70885956, 0.42281857, -3.1168566, 0.64445204, -1.9137427, 0.6635616, -0.1540724, 1.1936116, -0.09816121, -0.88661426, -0.14735366, 1.0598063, 0.026246618, -0.11433516, 0.7435535, 0.21035936, -0.005927406, 1.36606, 1.555114, 0.61332625, -0.28595915, 1.496911, 1.1831195, 0.71889716, -1.2160766, 0.14067191, -0.7436722, -0.15901226, 0.24005693, 0.10015941, -0.4751751, 1.2729537, -1.6961312, 0.73018354, -1.8574833, 0.38259813, -0.8869043, 0.87830377, 0.08645252, 0.24770638, -1.0182793, -0.65457016, 0.2072174, 0.58356994, 2.9290962, 0.22285832, 0.9760375, -1.5569339, -1.3298919, -0.35549477, -1.1974277, 1.4863993, -0.4102187, 1.3821819, 1.4867824, 0.04277972, 0.50179976, -0.056099474, 0.538437, 0.48334184, -0.12364963, 0.50496995, 1.7236962, 0.7130162, 0.3257996, 0.124769524, -1.0126731, -1.0272969, 0.32335654, -1.3693911, -0.7663276, 1.2815113, 1.9142298, -1.665956, 1.6266496, -0.2114383, -0.0150050875, -0.11341163, 1.0805441, -1.6076766, 0.45616361, -0.9448702, 0.5707885, 1.5427964, -0.0004173264, 0.37415507, 0.40955177, -0.7995935, 1.5116394, 1.7064682, 0.70178336, 0.07328543, -0.46189383, -0.62649024, 1.7108365, 1.414415, -0.063661486, -1.5799305, -2.832012, -1.0834267, -0.13062039, 1.400689, -0.6516562, 0.50481546, 1.3031809, 0.12853631, -0.14244787, -1.3087635, -1.2024753, 0.41609964, -0.20090753, 0.12253132, -0.047277715, 0.66414404, -0.7846874, -0.33558065, 1.8961822, -0.79978615, -0.28157544, -0.5893867, 0.44478136, 1.0223923, -0.49821162, -0.43141434, -0.2789816, 0.5298338, -0.7393953, -0.37595996, -2.3721938, -1.381745, -0.11244375, 0.89786416, 0.29507577, -1.0987685, -1.4002562, 0.1746801, -1.6528037, 1.0659268, 0.063896194, -1.6073202, -0.9659539, -0.7243113, -0.7731925, -1.489933, -0.8746625, -0.6844016, -0.71128577, 1.1279566, 0.10482781, -0.9932572, -0.3346216, -0.8795571, -0.30000666, 0.87550914, 0.2522708, 2.2856011, 0.37592742, -0.9135945, 0.8097407, 1.0799313, 1.094167, -1.0942409, -0.14763741, 1.131812, -1.684729, -0.49941677, -1.4269377, -0.9325702, -1.0124571, 1.2505698, -0.23453803, -0.8633556, -1.0356058, 0.14166717, -0.0111356275, 1.3440744, 0.5000167, -1.4317977, -0.6289807, 1.0700725, -0.6210827, 1.7345722, -1.0982895, 0.57261336, -0.86121553, -0.50959516, 1.0985817, -0.12706716, 0.81345224, 0.4732906, 0.75386566, -0.8881882, -0.2215744, 0.42425263, -0.8490729, 1.6295, -0.77722806, -0.3000036, -1.006559, -2.1433082, 1.7969185, -0.20433894, -0.44791484, -0.19871506, 1.4198639, -0.9651066, 0.6795679, -0.42378825, -0.59667087, 0.5670582, 0.9882406, -0.51390296, -0.76884913, -1.1690958, 1.1035038, -0.575256, -1.8491307, 1.4099522, -1.3698595, 0.77946055, 0.18342865, 0.28791544, -0.58437526, 0.36559147, -1.6677799, 0.5880377, 1.55701, 0.8840272, -2.01954, -0.984209, -0.18779492, 0.4869373, -0.10665268, -0.4932144, 0.5953003, 1.1641518, -0.23229401, 0.7289299, -2.5790508, -0.93750936, -0.32125893, -0.48856622, 0.3327982, 1.0137506, 0.50666904, -0.62222546, -1.5227681, 0.5569641, -1.8381767, 0.6530373, -0.18844908, -1.175835, 0.2872573, -0.0028761027, -0.036597293, -0.0842233, 0.4195241, 0.924434, 0.4966152, 1.0121332, -0.04413972, 1.6184593, 0.57110983, -0.543694, -1.0938951, 0.20579681, -1.3065215, -0.973376, 0.23908707, -0.60788745, -0.93331623, -0.034475047, 0.072677895, -0.20583403, -0.3775469, 0.85464275, 0.34242734, -0.22342612, 2.4643219, 0.19383174, 1.1320051, -0.560981, -1.3629409, -0.7917565, -0.26800978, -0.4966082, 1.3363862, -0.120041125, 0.46146888, -0.046481155, -0.43355432, 0.037996013, 1.7140515, -0.76794857, 0.7669904, -1.0260073, -0.45962644, 0.0035832059, 0.3263751, 1.4831287, -0.050082643, -0.8436156, 0.650042, -0.3641698, 0.23868157, -0.11622244, -1.9434569, 0.5082992, 0.583368, 0.92660475, 1.8004627, -1.1951038, 0.51650745, 0.409295, -0.419082, 0.39710623, 0.49964696, -1.2186838, 0.24622276, -0.9179843, -0.6518565, -1.7747449, -0.47336093, -0.20357068, 0.54985684, 0.00089992664, -1.5422882, 0.86214805, -0.11858662, 0.4883706, 0.9659361, 1.4226048, 1.9612269, -0.07223876, 0.31112444, -1.078361, 1.0616002, -1.1848874, -1.8052517, 0.830386, -0.5216965, 0.77760726, 0.40807465, -1.6300026, -2.7196794, -1.0966017, 0.016491488, -1.2217764, -0.65276146, -1.4589407, 0.16987796, 0.09082593, -0.48139262, 1.3970653, 1.497715, 0.5652672, -1.7997712, -1.1046902, 0.40713033, -0.62855756, -0.48709142, 0.8989674, 0.5108748, 1.3141544, -0.4292093, 1.3752254, -0.55413127, 1.4994915, 0.10583464, -0.86050975, -1.6312195, -0.3014723, -0.2562327, 0.8576619, -0.1105905, -0.43243197, 1.0770375, -0.22482656, -0.5762418, 0.5746089, -0.48982823, 0.65880215, -0.5969171, -0.22295918, 0.15217698, -0.37412632, -0.013451469, 0.81547195, 0.4106018, 0.48096985, -0.63543046, 0.85282975, 0.66956234, 1.0044192, -0.7263658, -0.1724586, 0.6335339, -0.60881513, -0.22612247, 1.9258057, 1.951761, 1.2399405, 0.93858516, -1.0192511, 0.5125622, -0.35911658, -1.0585719, -0.50900584, 0.11566507, -0.5473556, -0.5507994, 0.7920415, 0.14410649, 0.23345809, 0.1118724, -0.67570317, -1.370572, 0.3105647, -0.5070366, -2.0107822, -0.39256725, -1.0922179, 0.69865024, 0.5216252, 0.49689314, -0.6650416, 0.7315516, 0.3196498, -0.40985453, -0.45333743, 0.8927082, -0.47360405, 0.30365646, 1.033957, 1.9093426, 1.6638731, 0.90082276, -1.5059114, -0.6890484, -0.5480872, 1.6531498, -0.69931793, 0.38616636, 0.10086706, -0.9351272, 0.38182402, 0.3982961, -1.2557749, 1.2228775, -2.08651, -0.59075713, 0.9719703, -1.1932578, 0.35026592, -1.2963604, -0.09302414, -2.3137732, -0.8425717, -1.5429214, -0.40176374, -0.4152314, -0.67366415, 0.7979132, -0.8868796, 0.63438666, 1.6292758, 0.13906415, -0.8576702, -1.2493385, -0.7097851, 0.7046427, 0.15559073, 0.93679523, 0.7703309, 0.14081065, 0.47348827, 1.8552462, 1.4156562, -0.30274603, 0.98967946, 0.58585083, 1.1363881, 0.67161655, -0.9741674, -1.6196846, 0.572627, 1.9026182, -0.7756641, -0.18808974, -1.0357478, 1.1778295, -2.305167, -2.2636602, 0.3750199, -0.082343645, -0.47962302, -0.3010948, 0.5369879, -0.413804, -1.096925, -0.9273629, 0.88833886, -0.52474195, -1.3852776, 0.10217833, 0.50499475, 1.3289608, 0.21790339, -0.65971124, 0.47400787, 0.7271749, -0.038905308, -0.04459939, 0.2601329, -0.069856495, 0.2501139, -1.0219133, -1.1504377, -0.83611137, 0.64221096, 0.25879756, 1.040239, -0.18669093, -1.1436414, 1.1445535, -0.018767055, 1.283455, 0.59794647, 2.1886187, -0.21977298, 0.90072393, 0.8913641, -0.55512637, -0.17248231, -1.4617383, -1.5487962, 0.1265688, 0.7930071, 0.63802403, 0.3400246, 0.86301714, -0.5896978, -0.27253276, 0.7375215, 0.43311873, -0.21018882, 1.3207943, -1.2920012, -0.51867867, -0.28339776, 0.8165349, 0.002385198, -1.2614918, 0.5140042, 1.0875463, 0.73930454, 0.61915493, -1.8743135, -0.8998865, 0.4820806, -0.054888185, 0.5225576, -1.2663426, -0.061494764, -1.389781, -1.9536786, 0.29577908, 0.8425888, 0.24561642, -0.03299648, -1.5620143, 1.0061071, -0.044044897, 1.9595621, 0.9423143, -2.0051255, 0.7550497, -1.3965353, -0.7594955, -0.25075668, -0.09406245, 0.39756522, -1.022855, -1.150692, 0.6006052, -0.013250268, 0.17437305, -2.1936834, -0.17713739, -0.8907292, -0.9206264, 0.9219348, -1.0956712, -1.0928966, -0.3310106, 0.45028883, -0.8840147, 1.2341441, 1.4498476, -0.8814471, -0.24508175, -0.7786755, -1.6853821, 0.30301106, 0.7335949, 2.0118642, -0.8974095, 1.336235, 1.3423537, 0.19785331, 0.6021635, 0.8732731, 1.9741, 0.47780856, -0.060137887, -0.8661688, 0.30532077, 1.0241649, 0.24461035, -0.77992326, 0.089076206, -0.12915348, 0.26473877, -1.6618484, 0.55078864, 0.59542316, 0.44485343, -0.0037628172, -1.8059362, -0.019322792, 1.060715, -0.8601289, -1.9892695, -1.540558, 0.3140257, 0.37287602, 0.8862932, -0.055258997, -1.5003284, -0.81850415, 0.8188394, 0.14049591, 0.6498296, 0.4347888, -0.20496055, -0.17400683, 1.8571023, 0.41467425, -0.12858754, 0.45542, 0.22290581, -2.1573563, 0.6500845, 1.8209393, -0.7802799, 1.4540358, -0.2568697, 0.2934714, 1.0703601, -0.72000146, 1.2424939, -1.2142173, -0.87515473, -0.59352034, 0.66200536, -0.3408744, -1.5199745, -0.21653287, -0.7842214, 0.7312936, -0.34323505, 0.07077408, -0.40547246, 0.43393898, -0.18359077, 0.3251987, -2.5933886, 0.09725088, 0.41391367, -0.19928005, 0.66939247, 0.73860705, 1.3042139, 0.10481161, -1.9138007, -2.2854993, -1.601841, -0.03790706, -0.15730529, 0.27623984, -0.6252459, -0.73649114, 0.5550479, 0.65592444, -0.25665015, -0.038476657, 0.40431434, 0.50434357, -1.1439807, -0.71957386, -1.230546, -0.5069066, 0.8123336, 0.54627186, -1.0980979, 0.51226676, 0.08584311, -0.4939267, -1.4064597, -0.17482337, 0.679944, -2.1630976, -0.3961232, 2.2542837, 0.67263675, 0.2598325, -0.7371852, -0.6783298, -0.083288394, 1.6028637, 0.4655892, -0.8721584, 1.176787, -0.2925942, 1.6973464, -0.566603, -1.0032657, 0.17462958, 0.982327, 1.0374448, 0.15919177, -0.9880967, -0.5053407, -2.018282, -0.9131215, -0.17845681, 0.38900214, -0.33945432, -0.056979056, -0.39618546, 0.7510253, -0.89911294, 0.8375479, 1.9608808, 0.47278965, -0.5270916, -0.53627014, 1.2098372, -1.1265894, -0.95380443, -1.1644485, -1.2785138, -1.0448164, 0.78990495, 1.1022825, -0.6970731, 0.20733404, 0.7591567, 0.100564204, -0.95494276, -1.4704018, 1.0104276, 0.4961794, 0.5769559, -1.107647, 0.23497719, 0.6289996, 0.31403384, -0.7450232, 1.0122606, -1.527632, 0.92874193, 1.081056, 1.5723304, -0.3424922, -0.99943, 0.79388034, -0.6992153, 0.04399551, -0.3174622, -0.90207195, 0.32099947, -1.3920159, 0.5922057, -0.9669311, -1.7317313, -0.05010746, 0.43163386, 0.5769346, 0.8183537, -2.3536403, -1.0051445, 0.1066523, 1.5190033, 0.7837445, 1.90134, -0.5249394, 0.27441698, -1.0999708, -0.40435222, -0.7352957, -0.6339887, -0.39344913, 0.00271754, 0.022212664, 0.54345345, 0.13998847, -0.34404564, -0.52257854, -0.3071317, -0.44903713, 0.49097106, 0.8655252, 1.2740445, -0.7977028, 0.4693722, -1.3946797, 0.37317473, 1.0826722, -0.14958951, 1.072636, -1.1385679, -0.8886453, -0.13580984, 1.0222104, -0.41742945, -0.4535531, -0.99162835, 0.20288104, 1.2466952, 0.70068014, 0.6966507, -0.20697448, -0.5633094, 0.6772459, -0.031911075, -0.17360823, 0.8982406, -0.19778745, -0.83777624, 0.9091885, 0.08071989, -1.0370294, -1.1129059, 0.095411874, 2.3374097, -0.3928206, -0.33627385, 1.5237712, -0.0572812, -1.4484669, -1.5727965, 1.226664, 0.66635454, 0.8261257, -0.057756558, -0.72671205, -0.21716312, 0.13603121, -0.83831114, 0.5614499, -1.2595961, -0.33275875, -0.20400788, -0.69101983, -2.2055054, 0.44786966, -0.7557508, 1.3257079, -0.34198228, -0.5413596, 0.09152195, 1.0534397, -0.56340766, 1.0147377, 1.4403037, 0.9903228, 1.6264315, 1.292646, 1.5148823, 1.6043264, 0.20806953, -0.4292239, -2.2622437, -1.3227332, -0.4482828, -0.3817351, -0.15279447, -1.0007604, -1.5957776, -0.13022317, -0.18941793, -0.80755407, -0.74215215, -0.9401566, -0.39652374, -0.8563028, 1.2598753, 0.24099673, -0.97231793, -0.28044778, -1.1802856, 1.0121683, 1.3841867, 1.252002, -1.1446927, -0.09126702, -0.40157068, 0.5620131, -1.0079098, -0.6758917, -0.41321704, 0.15328847, 0.6941287, -0.3287277, 0.66396505, 0.8220764, -0.21321523, -1.2456582, -1.1711904, 0.59172696, -0.47622442, -1.7126293, 0.61295235, 0.12955453, -1.4059671, 1.17942, 0.836636, 0.13874525, -1.2743194, -1.4023305, -0.3070685, -1.7139153, 0.40508026, -1.4108233, 0.16491273, -0.28813145, 0.71178526, -0.9379476, 0.27372944, -1.3948402, 0.7955496, -0.114961766, 0.49585068, -1.3205253, 0.49908426, 0.3062034, 0.3636979, 0.31263396, -0.19346388, 1.2412993, -0.15589799, -0.7391692, -0.05872619, -0.95051795, -0.4639964, -0.17724662, -0.37955412, 0.19939707, 1.9457614, 0.57094985, 1.0723007, -0.50370944, -0.5870163, -0.37817806, 0.8528891, -2.1481185, -1.0331647, 0.10233585, -0.22409236, 1.9677297, 0.44768322, -0.66219145, -1.577607, -0.34056005, -1.30322, 0.46675065, 0.16110632, 0.32003194, 2.0791767, -0.907466, -0.19240421, -1.2125157, -0.08059852, 1.5932736, 0.5687224, -0.114487045, 0.25163025, -1.2108556, -0.3937337, 0.085252576, 0.099421985, -1.5306163, 0.3276232, 0.2791965, -0.3770512, 0.004174999, -1.4834915, -1.4797956, 0.13468726, -0.6677232, -0.01155552, 0.83949065, -0.17392993, -2.810668, -0.15065365, -0.48104402, -0.23469436, 0.8997308, -1.5785302, 0.24395663, 1.5703039, -0.6259431, 0.4723279, 0.9663058, 0.21023144, -0.685097, -0.709521, 0.74380016, 0.5921491, -0.7864684, -1.1764731, -1.2808067, 1.6616518, -0.06794512, 2.3602285, 0.5555456, 0.43952233, 0.30627248, 0.99914986, -0.9660632, 2.1600132, -0.100301705, -0.7034001, 0.302561, 1.0923389, -1.0075549, 0.5668694, -0.71644413, -0.5062735, -0.48948243, 0.76354146, -1.1090727, 0.1926161, -0.34341785, -0.84721017, -1.2135236, -1.2028884, -1.633796, 0.8961672, -0.24165316, 0.15865193, 1.1781894, -1.2201172, -0.94154567, 0.25471553}),
			),
			outputT: tensor.New(tensor.WithShape(1, 3, 32, 32),
				tensor.WithBacking([]float64{1.7640524, 0.978738, 2.2408931, 2.2408931, 1.867558, 1.2023798, 0.95008844, -0.10321885, 0.41059852, 0.41059852, 1.9507754, 1.9507754, 0.7610377, 0.44386324, 0.7774904, 1.4940791, 1.4940791, 0.3130677, 0.3869025, 0.3869025, 0.6536186, 0.8644362, 0.8644362, 2.2697546, 2.2697546, 0.3024719, 0.045758516, 1.5327792, 1.5327792, 1.4693588, 0.37816253, 0.37816253, 0.17742614, -0.34791216, 0.46278226, 1.2302907, 1.2302907, 1.2023798, 0.7290906, 1.1394007, 1.1394007, 0.40234163, 1.9507754, 1.9507754, -0.4380743, -0.31155252, 0.7774904, 0.7774904, 0.9008265, 0.9008265, 0.46566245, 1.4882522, 1.8958892, 1.8958892, 1.1787796, 0.42833188, 1.0544517, 1.0544517, 1.222445, 1.222445, 0.97663903, 0.97663903, 0.7065732, 0.7065732, 1.7858706, 1.7858706, 0.46278226, 1.8831507, 1.8831507, 0.7290906, 0.9693967, 1.1394007, 1.9436212, 1.9436212, 0.40234163, 1.922942, 1.922942, 1.867559, 1.867559, 0.90604466, 1.9100649, 1.9100649, 0.8024564, 1.4882522, 1.8958892, 1.8958892, 1.1787796, 0.9222067, 1.0544517, 1.0544517, 1.3263859, 1.3263859, 0.97663903, 0.97663903, 1.8492638, 1.8492638, 1.7858706, 1.7858706, 0.5392492, 1.8831507, 1.8831507, 0.031830557, 0.9693967, 0.9693967, 1.9436212, 1.9436212, 0.3960067, 1.922942, 1.922942, 1.867559, 1.867559, 2.3831449, 2.3831449, 1.9100649, 1.1170163, 1.1170163, 0.947252, 0.61407936, 1.7133427, 1.7133427, 0.37642553, 0.2982382, 1.3263859, 1.3263859, 1.1266359, -0.14963454, 1.8492638, 1.8492638, 1.929532, 1.929532, 0.9494208, 0.5392492, 0.844363, 0.844363, 0.67643327, 1.1880298, 1.1880298, 0.9208588, 0.9208588, 0.8568306, 0.8568306, 0.4393917, 0.6815945, 2.3831449, 2.3831449, 0.94447947, 1.1170163, 1.1170163, -0.35399392, -0.0682416, 1.7133427, 1.7133427, 0.62523144, -0.09845252, 0.05216508, 1.1266359, 1.5430146, 1.5430146, 0.26705086, 0.26705086, 1.929532, 1.929532, 0.9494208, 0.77179056, 0.844363, 2.163236, 2.163236, 1.336528, 1.1880298, 1.0996596, 1.0996596, 0.8568306, 0.8568306, -0.024326125, 0.6815945, 0.6815945, 0.2799246, 0.9101789, 0.9101789, 0.78632796, 0.78632796, -0.4664191, -0.4100497, 0.62523144, 0.62523144, 2.259309, 2.259309, 0.05216508, 1.5430146, 1.5430146, 0.48148146, 0.48148146, 0.06326199, 0.5232767, 0.5232767, 0.77179056, 0.82350415, 2.163236, 2.163236, 1.336528, 0.41605005, 1.0996596, 1.0996596, 1.4944845, 1.4944845, 0.42625874, 0.676908, 0.676908, 0.2799246, 0.9101789, 0.9101789, 0.78632796, 0.78632796, 1.1523316, 1.1523316, 1.0796186, 0.37915173, 2.259309, 2.259309, 0.14195317, 0.14195317, 0.69153875, 0.6947491, 0.6947491, 0.06326199, 0.15650654, 0.6103794, 0.6103794, -0.23792173, -0.23792173, -0.052567296, -0.052567296, 0.41605005, 0.52389103, 0.7811981, 1.4944845, 1.4944845, 0.42625874, 0.676908, 1.9559124, 1.9559124, 0.39009333, -0.13288058, 0.49374178, 0.49374178, 1.1523316, 2.064493, 2.064493, 1.0201727, 1.0201727, 1.5363771, 1.5363771, 0.60884386, 0.69153875, 1.2111453, 1.2111453, 1.3018463, 1.3018463, 0.6103794, 2.3039167, 2.3039167, -0.1359497, 1.1368914, 1.1368914, 0.5829537, 0.5829537, 0.52389103, 0.37005588, 1.6581306, 1.6581306, 0.39904633, 1.9559124, 1.9559124, 0.39009333, -0.39095336, 0.69377315, 0.69377315, -0.11610394, 2.064493, 2.064493, 1.0201727, 1.0201727, 1.5363771, 1.5363771, 0.60884386, 0.60884386, 1.2111453, 1.2111453, 1.3018463, 1.3018463, 0.27451634, 2.3039167, 2.3039167, -0.1359497, 2.2567234, 2.2567234, 0.9432607, 0.9432607, 0.7471883, 0.77325296, 1.6581306, 1.6581306, 0.60631955, 0.6663831, 0.6663831, 0.45093447, 1.6595508, 1.6595508, 1.0685093, -0.13370156, 1.0777438, 1.0777438, -0.28035548, -0.28035548, 0.15670386, 0.5785215, 0.5785215, 0.34965447, -0.0616264, -0.10730527, 1.3645319, 0.27451634, 0.27451634, -0.52118933, -0.31229225, -0.15766701, 2.2567234, 2.2567234, 0.9432607, 0.9432607, 0.93184835, 0.77325296, 0.77325296, 0.16092817, 0.60631955, 0.60631955, 0.45093447, 0.45093447, 1.6595508, 1.6595508, 1.0685093, 0.04949498, 0.4938368, 0.6433145, 0.6433145, -0.20690368, 0.8801789, 0.8801789, 0.5785215, 0.38728046, -0.76414394, -1.0225068, 1.3645319, -0.6522936, -0.52118933, 1.648135, 1.648135, 0.5672903, 0.6203583, 0.6984571, 0.6984571, 0.93184835, 0.93184835, 0.8579239, 1.1411018, 1.4665787, 1.4665787, 0.85255194, -0.26773354, 0.7666632, 0.7666632, 0.8416313, 0.8416313, 0.8145198, 0.8145198, 0.6433145, 0.6433145, -0.20690368, 0.8801789, 0.8801789, 0.38728046, 1.7327212, 1.7327212, 0.6845011, 0.370825, 1.5199949, 1.7195894, 1.7195894, 1.648135, 0.5822246, 0.5672903, 0.12372191, 0.09395323, 0.9430461, 0.9430461, 0.8579239, 1.1411018, 1.4665787, 1.4665787, 0.8689635, 0.8689635, 0.7666632, 0.7666632, 0.8215857, 0.8215857, 0.8145198, 0.8145198, 0.078260176, -0.18505368, -0.085930765, 0.800298, 0.87583274, 0.87583274, 1.7327212, 1.7327212, 0.6845011, 0.370825, 1.5199949, 1.7195894, 1.7195894, 0.9295051, 0.5822246, 2.0210435, 2.0210435, 0.09395323, 0.9430461, 0.9430461, -0.050603542, 0.26990435, 0.26990435, 0.18133843, 0.8689635, 2.4124537, 2.4124537, 0.3148172, 0.8215857, 0.8215857, 0.8005648, 0.8005648, 0.078260176, -0.27567053, 1.7388726, 1.7388726, 1.3191369, 1.3191369, 1.128594, 1.128594, 0.49600095, 1.0294389, 1.0294389, 0.8202478, 0.86259604, 0.86259604, 1.5133281, 2.0210435, 2.0210435, 0.22050765, 0.22050765, 0.1993002, 1.1002843, 1.298022, 2.696224, 2.696224, 0.18133843, 2.4124537, 2.4124537, -0.07785475, 0.38273242, 0.38273242, 1.0963469, 1.0963469, -0.2342158, -0.27567053, 1.7388726, 1.7388726, 1.3191369, 1.3191369, 1.3014281, 1.3749641, 1.3749641, 1.0294389, 1.0294389, 0.17581895, 0.86259604, 1.0479722, 1.5133281, 1.7426687, 1.7426687, 0.22050765, 0.22050765, -0.34994337, 1.1002843, 1.298022, 2.696224, 2.696224, 1.4123276, 0.8640523, 0.8640523, 0.40149906, 1.2248706, 1.2248706, 1.0963469, 1.0963469, -0.2342158, -0.18224478, -0.18224478, -0.10988278, 0.21348006, 1.3014281, 1.3014281, 1.5182612, 1.5182612, -0.38464543, 1.0781974, 1.0781974, 1.1813786, 1.1813786, 1.0479722, 1.7426687, 1.7426687, 0.9424681, -0.26759475, 1.2978458, 1.2978458, 0.24211796, 0.9367425, 1.4123276, 2.0112567, 2.0112567, 0.8640523, 0.40149906, 1.2248706, 1.2248706, 0.3547577, 0.61688656, 0.61688656, 0.5270042, 0.5270042, 0.4537819, 0.21348006, 0.76790243, 0.76790243, 1.5182612, 1.5182612, -0.38464543, 1.0781974, 1.1330799, 1.1813786, 1.1813786, 0.16392857, 0.7849575, 0.9424681, 0.9424681, -0.21694957, 1.2978458, 1.2978458, 0.020334182, 0.5433119, 0.5433119, 2.0112567, 2.0112567, 0.35178012, 0.37923554, 0.37923554, 0.1965574, 0.3547577, 0.61688656, 0.61688656, 0.5270042, 0.5270042, 0.4537819, 0.23810315, 0.76790243, 0.76790243, 0.5898798, -0.36385882, 0.115147874, -0.13105401, 1.1330799, 1.1330799, 0.895556, 0.895556, 0.7849575, 0.7849575, 0.32962298, 1.285984, 1.285984, 0.67646074, 0.67646074, 0.5433119, 0.5433119, 0.43904296, -0.21954103, 0.35178012, 1.670943, 1.670943, -0.0013850428, -0.0013850428, -0.11747455, 0.46616644, 0.46616644, 0.41731882, 0.40326455, 0.40326455, 0.25249663, 0.8203218, 1.3599485, 1.3599485, 1.3675972, 1.3675972, 1.0344099, 0.60512006, 0.895556, 1.0289356, 1.0289356, 0.40476182, 1.5522432, 1.5522432, 1.285984, 0.67646074, 0.67646074, 0.52004063, 0.4497121, 0.4497121, -0.067275606, 0.1833392, 1.670943, 1.670943, -0.0013850428, 0.45248908, 0.45248908, 0.46616644, 0.46616644, -0.023423105, 1.0791948, 1.0791948, 0.37687653, 0.8203218, 1.3599485, 1.3599485, 1.3675972, 1.3675972, 1.0344099, 0.39306292, 1.017021, 1.4229835, 1.4229835, 0.39608657, 1.5522432, 1.5522432, 0.86740744, 0.86740744, 0.52004063, 2.116791, 2.116791, 0.4497121, 2.3807454, 2.3807454, 0.94924647, 0.94924647, -0.9327409, 0.45248908, 1.0907497, 1.0907497, -0.34624946, 0.19796729, 1.0819352, 1.0819352, 0.37687653, 0.37687653, 1.0946383, 1.0946383, 2.1321535, 2.1321535, 0.9364457, 1.2650778, 1.2650778, 1.4229835, 1.4229835, 0.67997485, 1.1244192, 1.3277828, 1.3277828, 0.86740744, -0.46433768, 2.116791, 2.116791, -0.035768073, 2.3807454, 2.3807454, 0.94924647, 0.94924647, -0.8871809, -0.5327028, 1.2433194, 1.2433194, 0.81267405, 0.58725935, 1.0819352, 1.0819352, -0.5075176, 2.4972005, 2.4972005, 1.0946383, 2.1321535, 2.1321535, 0.9364457, 1.2650778, 1.2650778, 1.7549862, 1.7549862, 0.67997485, 0.55578697, 1.3277828, 1.3277828, 0.72003376, 0.30360392, 1.0217906, 1.0217906, 0.44819528, 1.6961815, 1.6961815, 0.82140595, 0.82140595, 0.67057043, 0.039766736, 1.2433194, 1.2433194, 0.81267405, 0.7231005, 0.7231005, 0.71998376, 0.71998376, 2.4972005, 2.4972005, 0.019279385, 1.8495913, -0.10434349, 0.021351224, 0.021351224, 0.19275385, 1.7549862, 1.7549862, -0.058586553, 0.55578697, 0.55578697, 0.72003376, 1.4893559, 1.4893559, 0.7726948, 0.7726948, 0.47689837, 1.6961815, 1.6961815, 0.82140595, 0.82140595, 0.67057043, 0.039766736, 0.039766736, -0.050084095, 1.3124703, 1.3124703, 0.7231005, 0.71998376, 0.71998376, -0.10169727, 0.019279385, 0.019279385, 1.8495913, 0.78580385, 0.021351224, 0.9409176, 0.9409176, 0.49805242, 0.49805242, -0.026192237, -0.058586553, -0.112465985, 0.6450553, 1.4893559, 1.4893559, 0.6119272, 1.735879, 1.735879, 1.6819217, 1.6819217, 0.5290452, 0.4226286, 0.0114989, 0.0114989, -0.050084095, -0.050084095, 1.3124703, 1.3124703, 2.2259443, 2.2259443, 1.370989, 0.3248696, 0.997118, 0.997118, 1.411172, 0.78580385, 0.8672766, 0.9409176, 0.9409176, 0.49805242, 0.49805242, 0.31144708, 0.23958276, 0.9725358, 2.1338682, 2.1338682, 1.0118425, 0.7557403, 1.735879, 1.735879, 1.6819217, 1.6819217, 0.022959756, 0.022959756, 0.0114989, 0.0114989, -0.43674833, -0.43674833, 0.3269626, 0.33003512, 2.2259443, 2.2259443, 1.370989, 0.3248696, 0.997118, 0.997118, 0.22739278, 0.05157494, 0.8672766, 0.8672766, 0.8656529, 1.0813761, 1.0813761, 0.31144708, 0.23958276, 0.9725358, 2.1338682, 2.1338682, 0.4064155, 0.7557403, 0.7557403, 1.3264617, 1.3264617, 0.059894685, 0.059894685, -0.21252304, -0.34796184, 0.93639857, 0.93639857, 0.2711702, 0.2711702, -0.40607178, 0.47224715, 1.154184, 1.154184, 0.17250441, 1.997956, 1.997956, 0.22739278, 2.5944245, 2.5944245, 0.30875126, 0.8656529, 1.0813761, 1.0813761, 0.19031155, 1.8227236, 1.8227236, 0.69938046, 0.9606934, 1.3290628, 1.3290628, 0.05095428, 1.3264617, 1.3264617, 0.059894685, 0.96744615, 0.96744615, -0.055352546, 0.93639857, 0.93639857, 0.2711702, 1.2760754, 1.325014, 1.325014, 0.9304085, 2.339625, 2.339625, 1.997956, 1.997956, 1.471322, 2.5944245, 2.5944245, 0.30833125, 0.30833125, 0.3675449, 0.3675449, 0.19031155, 1.8227236, 1.8227236, -0.5215797, 0.9606934, 1.3290628, 1.3290628, 2.759355, 2.759355, 1.0304383, 0.27560067, 1.4350494, 1.4350494, 0.5072389, 0.3528166, 0.3528166, 1.4013448, 1.4013448, 1.325014, 1.325014, 0.86351967, 2.339625, 2.346647, 2.346647, -0.25957698, 1.471322, 1.5927707, 1.5927707, 0.82998616, 0.30833125, 0.012231983, 1.5692596, 1.5692596, 1.6815767, 1.6815767, 0.9688826, 0.9688826, 1.3891454, 2.0140603, 2.759355, 2.759355, 0.25871643, 0.27560067, 1.4350494, 1.4350494, 0.5072389, -0.1162297, 1.0645851, 1.4013448, 1.4013448, 0.5289436, 0.7243685, 1.3852615, 1.3852615, 2.346647, 2.346647, 0.17879286, 0.9380925, 0.9380925, 0.82998616, 0.82998616, 0.0941923, 0.0941923, 1.5692596, 1.5692596, 0.8924739, 0.8924739, 0.9688826, 0.9688826, 1.3891454, 2.0140603, 2.0140603, -0.04932407, 0.2390336, 0.2390336, 1.6739857, 1.6739857, 1.5634048, 1.5634048, 1.0645851, 1.0645851, 0.43310794, 0.43310794, 0.7243685, 1.3852615, 1.3852615, 0.70104134, 0.44103292, 0.17879286, 0.2407875, 0.2891205, 0.41287082, 0.41287082, 0.0941923, 0.0941923, -0.35811406, 0.5559627, 0.8924739, 0.8924739, 0.10471403, 0.22805333, 0.22805333, 0.5407736, 0.5407736, -0.04932407, 0.2390336, 0.2390336, 1.6739857, 1.6739857, 1.5634048, 1.5634048, -0.790523, 0.22425222, 0.22425222, 0.2149656, 0.2149656, 1.0156653, 1.0156653, 0.70104134, -0.41747734, -1.0974966, 1.7123052, 1.2649833, -0.30078387, 1.1173053, 1.1173053, 0.22246316, 0.22246316, 0.13768983, 0.38141626, 0.7539915, 1.0653155, 1.0653155, 0.9853175, 0.7669197, 0.40262553, 1.6692508, 1.6692508, 0.60815644, 1.1149623, 1.4333525, 1.4333525, 0.43554616, 0.43554616, 0.32830128, 0.03308975, 0.5969065, 0.5969065, -0.15602389, 1.0490932, 3.1709747, 3.1709747, 0.18949963, 1.2649833, 1.2649833, -0.2311016, 0.20984948, 0.20984948, 2.1495745, 2.1495745, 0.73165894, 0.73165894, 0.38141626, 0.6632581, 0.6632581, 0.41540062, 1.5788652, 1.5788652, -0.061638054, -0.061638054, 0.26902407, 0.93874687, 1.2674117, 1.2674117, 0.49949825, 1.2591671, 1.2591671, 0.70411104, 2.5263681, 2.5263681, 1.7699214, 0.3779101, 1.3243587, 1.3243587, -0.1722008, 1.1045785, 1.1045785, -0.2311016, 0.9214084, 0.9214084, 2.1495745, 2.1495745, 0.73165894, 0.73165894, 0.34816924, 0.6632581, 0.82455724, 0.82455724, 1.5788652, 1.5788652, 0.21717963, 0.21717963, 1.4045455, 1.4045455, 1.2674117, 1.2674117, 1.5187594, 1.5187594, 1.2591671, 0.76449746, 2.5263681, 2.5263681, 1.7699214, 1.221385, 1.3243587, 1.3243587, -0.1722008, 1.1045785, 1.1045785, 1.3978963, 1.3978963, 0.9214084, 1.648504, 1.648504, -0.13256802, 1.4261588, 1.4261588, 0.93612915, 0.8326507, 0.8326507, 1.6315974, 1.6315974, 0.37775916, 0.2398671, 1.4045455, 1.4045455, 0.77067304, 0.77067304, 1.8219151, 1.8219151, 0.76449746, 0.76449746, 0.24660219, 0.99213684, 1.9050636, 1.9050636, -0.01477722, -0.033319283, -0.35502872, 0.5310425, 0.5310425, 1.3978963, 1.3978963, 0.9600477, 1.648504, 1.648504, 1.1239053, 1.4261588, 1.4261588, 1.3925184, 1.0375856, 0.8326507, 1.6315974, 1.6315974, 0.5897036, 0.2398671, 1.5848205, 1.5848205, 0.77067304, 0.77067304, 1.8219151, 1.8219151, 1.2961498, 1.2961498, 0.6164593, 0.99213684, 1.9050636, 1.9050636, 0.8805112, 0.08595198, 0.08595198, 0.75194657, 0.5629897, 1.054758, 1.054758, 0.9600477, 1.7746586, 1.7746586, 1.1239053, 0.76943016, 1.3925184, 1.3925184, 1.0375856, 0.30151406, 1.0390965, 1.0390965, 0.5897036, 1.7367749, 1.7367749, 1.5848205, 0.9605572, 0.9605572, 0.8709618, 0.8709618, 2.3207998, 2.3207998, 0.6164593, 0.53659654, 0.43480796, 0.8805112, 0.8805112, 0.732424, 0.08595198, 0.75194657, 0.5629897, 0.2711128, 1.0450233, 1.0450233, 1.7746586, 1.7746586, -0.16221845, 1.151734, 1.151734, 0.33053273, 0.13157398, 0.30151406, 1.0390965, 2.0234718, 2.0234718, 1.7367749, 1.7367749, 2.2436018, 2.2436018, 1.9223248, 1.9223248, 1.4053655, 2.3207998, 2.3207998, 0.53420115, 0.5474806, 0.5474806, 0.54009444, 0.732424, 0.732424, 0.54100823, -0.085115604, 0.966768, 0.966768, 1.0450233, 1.0450233, 0.59903955, 0.5232617, 0.09920486, 1.576299, 1.576299, 0.5023282, 0.16066119, 0.16066119, 1.6085222, 2.0234718, 2.0234718, 0.50538695, 0.35924914, 2.2436018, 2.2436018, 1.9223248, 1.9223248, 1.4053655, 1.6180543, 1.6180543, 0.42258036, 0.5474806, 0.5474806, 0.51283646, 0.6027077, 0.6027077, 0.54100823, -0.085115604, 0.966768, 0.966768, 0.6251187, -0.5216415, 0.5232617, 0.5232617, 0.09920486, 1.576299, 1.576299, 0.5023282, 0.16066119, 0.16066119, 1.6085222, 1.6085222, 1.1452621, 1.1452621, 0.7741606, 0.7741606, 0.10490716, 0.13391292, 0.31943184, 0.31943184, -0.13777922, 1.4961396, 1.4961396, 1.3462211, 1.3462211, 0.51283646, 0.62251914, 0.62251914, -0.5946498, 0.5684589, 0.4804245, 0.6251187, 0.83135104, 0.83135104, 0.48797268, 2.6429358, 2.6429358, 2.290467, 2.290467, 1.6002678, -0.059928004, -0.22007579, 1.3086687, 1.3086687, 1.1452621, 1.5259576, 1.5259576, 0.8820566, 0.8820566, 0.13391292, 0.13391292, 0.3718111, 0.45730963, 1.4961396, 1.4961396, 1.3462211, 1.3462211, 1.5885307, 1.5885307, 0.7747283, 0.7747283, 0.5684589, 0.4804245, 0.4804245, 1.0400863, 1.0400863, 0.88518757, 2.6429358, 2.6429358, 2.290467, 2.290467, 1.6002678, -0.008209882, 1.0429776, 1.0429776, 0.41409135, 0.24676603, 1.5259576, 1.5259576, 0.8820566, 0.8820566, 0.13191175, 0.76877064, 0.76877064, 0.45730963, 0.9623417, 0.9623417, 1.9470986, 1.9470986, 1.5885307, 1.5885307, 1.0212247, 0.7747283, 2.4512298, 0.41133425, 0.0025711823, 1.0400863, 1.0400863, 0.88518757, 1.4737648, 1.6606076, 1.6606076, 1.171041, -0.008209882, -0.008209882, 1.0429776, 1.0983036, 1.0983036, 0.15466884, 1.0415684, 1.0415684, 0.9785673, 0.9785673, 0.13191175, 0.76877064, 0.76877064, -0.4213276, 0.8756957, 0.8756957, 1.9470986, 1.9470986, 0.8091803, 1.0212247, 1.2016978, 1.2016978, 2.4512298, 0.87938994, 0.87938994, 0.63542455, 0.71593887, 0.71593887, 0.8809376, 1.8081318, 1.8081318, 0.43663847, 0.69643867, 0.69643867, 0.65178126, 1.0983036, 1.0983036, 0.025385605, 0.61039174, 0.61039174, 0.9785673, 1.7612661, 1.7612661, 0.7540957, 0.66596717, 0.11255753, 0.50099224, 0.50099224, 0.9361076, 0.9361076, 0.8091803, 0.88649154, 1.2016978, 1.2016978, 0.2847906, 0.87938994, 0.87938994, 0.63542455, 0.71593887, 0.71593887, 0.8809376, 1.8081318, 1.8081318, 0.43663847, 0.69643867, 0.69643867, 0.6878814, 1.3480358, 1.3480358, 2.6805708, 2.6805708, -0.20080851, -0.7401368, 1.7612661, 1.7612661, 0.7540957, 1.3185511, 1.3185511, 0.11255753, 0.067516856, 0.77760416, 0.77760416, 0.33601573, 0.88649154, 0.88649154, -0.27213177, 0.5483281, 0.5483281, 0.19718106, 1.0590272, 1.0590272, 0.5665453, 1.2572197, 1.2572197, 0.49872696, 0.81786186, 0.81786186, 0.6878814, 0.6878814, 1.3480358, 1.3480358, 2.6805708, 2.6805708, -0.16137354, 1.7944489, 1.7944489, 2.2320163, 2.2320163, 1.367415, 1.367415, 0.15364446, 0.15364446, 0.8444543, 0.8444543, 0.28376955, 0.33905926, 0.33905926, -0.58963203, 0.5483281, 1.7406294, 1.7406294, 1.0590272, 1.0590272, 0.87672675, 1.2572197, 1.2572197, 0.94940555, 0.94940555, 0.81786186, 0.23820019, 1.1493794, 1.1493794, 0.29751435, 1.3554426, 1.3554426, 0.58822495, 1.7944489, 1.7944489, 2.2320163, 2.2320163, 1.367415, 1.5877615, 1.5877615, 0.15364446, 0.99675965, 0.99675965, 0.28376955, 0.28376955, 0.079121724, 0.079121724, 0.8505307, 1.7406294, 1.7406294, 0.91722155, 0.91722155, 1.8666831, 1.8666831, 0.75746834, 1.238007, 1.238007, -0.086455286, 0.6234536, 1.1493794, 1.1493794, 0.51291686, 0.64305454, 0.64305454, 0.58822495, 2.5402317, 2.5402317, 0.9265583, 0.55808187, 0.15011522, 1.5877615, 1.5877615, 1.1277837, 1.0329891, 1.0329891, 1.4121517, 1.4121517, 0.079121724, 0.079121724, 0.8505307, -0.14989159, 0.084968135, 0.084968135, 0.5319746, 1.8666831, 1.8666831, 0.75746834, 1.238007, 1.238007, 2.2113044, 2.2113044, 0.8906717, 0.8906717, 0.51291686, 0.22734594, 1.6142496, 1.6142496, 2.5402317, 2.5402317, 0.9265583, 0.55808187, 1.5480669, 1.5480669, 1.1277837, 1.1277837, 1.0329891, 1.0329891, 1.4148741, 1.4148741, 0.15647626, -0.21634398, 0.44284612, 0.21839707, -0.25271067, -0.25271067, 0.6563907, 1.2605689, 1.2605689, 0.16586353, 1.3291413, 1.3291413, 2.2113044, 2.2113044, 1.9367125, 1.9367125, 0.38220277, 0.38220277, 1.6142496, 1.6142496, 1.0085973, 0.52759737, 0.8065638, 0.8065638, 1.5480669, 1.5480669, 1.1483506, 1.4028448, 1.4028448, 0.8246182, 1.4148741, 1.4148741, 0.15647626, -0.21634398, 0.56800735, 0.56800735, 0.08261103, 0.018922104, 0.6563907, 0.6563907, 1.0139722, 1.0139722, 1.3291413, 1.3291413, 1.0097119, 1.0097119, 1.9367125, 2.171165, 2.171165, 0.66207427, 1.1615338, 1.1615338, 2.190898, 2.190898, 0.9836362, 0.8065638, 0.25233144, 1.2098501, 1.2098501, 1.4028448, 1.4587147, 2.1531198, 2.1531198, 0.4683049, 0.6572677, 0.6572677, 0.56800735, 0.56800735, 0.62656426, 0.62656426, 0.018922104, 0.8449956, 1.0139722, 1.0139722, 2.042072, 2.042072, 1.0097119, 1.0097119, 0.72934633, 2.171165, 2.171165, 0.66207427, 0.53916126, 0.53916126, 2.190898, 2.190898, 0.9836362, 0.25233144, 0.60799253, 1.2098501, 1.2098501, 1.5612797, 1.5612797, 2.1531198, 2.1531198, 0.59936935, 0.6572677, 0.6572677, 0.46079433, 0.17124355, 1.4263169, 2.4884417, 2.4884417, 1.6959696, 1.8334354, 1.8334354, 2.042072, 2.042072, 0.46637958, -0.09439251, 0.72934633, 0.8020517, 0.8020517, 0.60413677, 0.60413677, -0.024549423, -0.8358561, 1.966764, 1.966764, 0.24664895, 1.419255, 1.419255, 0.9341797, 1.5612797, 1.5612797, 0.18418126, 0.21301717, 1.279812, 1.279812, -0.2563169, 0.46079433, -0.009657237, 1.4263169, 2.4884417, 2.4884417, 1.6959696, 1.8334354, 1.8334354, 1.801214, 1.801214, 0.46637958, -0.09439251, 0.8962484, 0.8962484, 0.88686466, 1.103694, 1.103694, 0.4005307, 0.13545467, 1.966764, 1.966764, 1.8593464, 1.419255, 1.419255, 0.9341797, 0.86900634, 1.1149246, 1.1149246, 0.024290914, 1.279812, 1.279812, 0.60896236, 0.40088567, 0.0032222164, -0.55598503, 0.19321355, 1.2973421, 1.2973421, 1.001331, 0.8179776, 1.801214, 1.801214, 1.4622141, 1.4622141, 0.8962484, 0.8962484, 0.88686466, 1.103694, 1.103694, 0.77905124, 0.13545467, 0.9687444, 1.8593464, 1.8593464, -0.13482246, 2.6205738, 2.6205738, 0.33510563, 1.1149246, 1.1149246, 0.013748487, 0.8253722, 0.8253722, 0.60896236, 0.0032222164, 0.021857359, 0.042539075, 1.5309323, 1.5309323, 0.18345025, 0.2727964, 0.8179776, 0.8179776, 1.4315678, 1.4622141, 1.4622141, 0.5383296, -0.14244542, -0.12656933, 0.4936441, 0.77905124, 0.77905124, 0.99012136, 0.99012136, 0.9687444, -0.021869909, -0.27120733, 2.6205738, 2.6205738, 1.7025702, 0.06763423, 0.44702417, 0.44702417, 0.8253722, 0.8253722, -0.29832315, 1.4219756, 0.45636335, 0.042539075, 1.5309323, 1.5309323, 0.092447735, -0.099008314, 0.777277, 0.777277, 0.5541276, 0.5541276, 0.5383296, 0.5383296, 0.71966815, 0.71966815, 0.4936441, 0.4936441, 0.48701534, 0.99012136, 0.99012136, -0.015779218, 0.9541896, 0.9541896, 0.54414475, 1.7025702, 1.7025702, 1.7148316, 1.7148316, 0.44702417, 0.17233914, 1.3985621, 1.3985621, 1.4219756, 0.7825789, 1.7975394, 1.7975394, 0.54565734, 0.54565734, 0.1508182, 1.6857923, 1.6857923, 0.5541276, 0.91064566, 0.91064566, -0.029856121, 0.71966815, 0.71966815, 0.038127385, 0.51271755, 0.51271755, 0.27566653, -0.015779218, -0.015779218, 1.2577442, 1.2577442, 0.54414475, 0.4472121, 1.4558079, 1.7148316, 1.7148316, 1.4849256, 0.4595849, 1.3985621, 1.3985621, 1.3905339, 0.7825789, 1.7975394, 1.7975394, 1.2165778, 1.2165778, 2.0347626, 2.0347626, 1.6857923, 0.9508337, 0.91064566, 0.91064566, 0.57275134, 0.67755705, 1.3071839, 1.3071839, 0.51271755, 0.8591042, 0.8591042, 0.8733118, 1.1997361, 1.2577442, 1.2577442, 0.041082226, 0.5934659, 1.4558079, 2.1982963, 2.1982963, 1.4849256, 0.4595849, 0.4595849, 0.28005785, 1.3905339, -0.15503581, 0.078514025, 0.41780058, 1.2165778, 1.2165778, 2.0347626, 2.0347626, 1.5743712, 1.5743712, 1.9887319, 1.9887319, 1.1172835, 0.67755705, 1.3071839, 1.3071839, 0.030546581, 1.2697648, 1.2697648, 0.8733118, 1.1997361, 1.1997361, 0.45615304, 0.7750233, 0.7750233, 0.5934659, 2.1982963, 2.1982963, 0.0012484558, 0.0012484558, -0.15993978, -0.58957994, -0.59815043, 0.4178537, 0.4178537, 0.41780058, 0.41780058, 1.342622, 1.8022423, 1.8022423, 1.5743712, 1.5743712, 1.9887319, 1.9887319, 1.4015235, 0.15843384, 1.054325, 1.054325, 0.030546581, 1.2697648, 1.2697648, 0.017515613, 0.32362577, 0.32362577, -0.020129103, 0.7750233, 0.92914516, 0.92914516, 0.37117255, 0.6436313, 0.68889666, 0.68889666, 0.2746472, -0.6036204, 0.70885956, 0.42281857, 0.64445204, 0.64445204, 0.6635616, 1.342622, 1.342622, 1.3151376, 1.3151376, 0.19782817, 1.0598063, 1.4015235, 1.4015235, 0.7435535, 0.7435535, 0.21035936, 1.36606, 1.555114, 1.555114, 0.61332625, 1.496911, 1.496911, 1.1831195, 0.71889716, 0.92914516, 0.92914516, 0.37117255, 0.6436313, 0.68889666, 0.68889666, 1.2729537, 1.2729537, 0.70885956, 0.42281857, 0.64445204, 0.64445204, 0.6635616, 0.6635616, 1.1936116, 1.1936116, -0.09816121, -0.14735366, 1.0598063, 1.0598063, 0.026246618, 0.7435535, 0.7435535, 0.21035936, 1.36606, 1.555114, 1.555114, 0.61332625, 1.496911, 1.496911, 1.1831195, 0.71889716, 0.14067191, 0.14067191, -0.15901226, 0.24005693, 0.24005693, 0.10015941, 1.2729537, 1.2729537, 0.73018354, 0.73018354, 0.38259813, 0.38259813, 0.87830377, 0.87830377, 1.2815113, 1.9142298, 1.9142298, 1.6266496, 1.6266496, 2.9290962, 2.9290962, 1.0805441, 1.0805441, 0.45616361, 0.45616361, 0.5707885, 1.5427964, 1.5427964, 1.3821819, 1.4867824, 1.4867824, 1.5116394, 1.7064682, 1.7064682, 0.70178336, 0.48334184, 0.50496995, 1.7236962, 1.7236962, 1.414415, 0.3257996, 0.124769524, -1.0126731, 0.32335654, 1.400689, 1.400689, 1.2815113, 1.9142298, 1.9142298, 1.6266496, 1.6266496, -0.0150050875, 0.41609964, 1.0805441, 1.0805441, 0.45616361, 0.66414404, 0.66414404, 1.5427964, 1.8961822, 1.8961822, 0.40955177, 0.40955177, 1.5116394, 1.7064682, 1.7064682, 0.70178336, 0.07328543, 0.5298338, 1.7108365, 1.7108365, 1.414415, -0.063661486, -0.11244375, 0.89786416, 0.89786416, 1.400689, 1.400689, 0.50481546, 1.3031809, 1.3031809, 1.0659268, 0.063896194, -0.9659539, 0.41609964, 0.41609964, 0.12253132, 0.12253132, 0.66414404, 0.66414404, 1.1279566, 1.8961822, 1.8961822, -0.28157544, -0.28157544, 0.44478136, 1.0223923, 1.0223923, 2.2856011, 2.2856011, 0.5298338, 0.8097407, 1.0799313, 1.0799313, 1.094167, -0.11244375, 1.131812, 1.131812, 0.29507577, -0.49941677, 0.1746801, 0.1746801, 1.2505698, 1.2505698, 0.063896194, -0.8633556, 0.14166717, 0.14166717, 1.3440744, 1.3440744, 0.5000167, -0.6289807, 1.1279566, 1.1279566, 1.7345722, 1.7345722, 0.57261336, 0.57261336, 0.87550914, 1.0985817, 2.2856011, 2.2856011, 0.81345224, 0.8097407, 1.0799313, 1.0799313, 1.094167, 0.42425263, 1.6295, 1.6295, -0.3000036, -0.3000036, -0.9325702, 1.7969185, 1.7969185, 1.2505698, -0.19871506, 1.4198639, 1.4198639, 0.6795679, 1.3440744, 1.3440744, 0.5670582, 0.9882406, 1.0700725, 1.0700725, 1.7345722, 1.7345722, 1.1035038, 0.57261336, 1.4099522, 1.4099522, 1.0985817, 0.81345224, 0.81345224, 0.75386566, 0.75386566, 0.36559147, 0.5880377, 1.55701, 1.6295, 1.6295, -0.3000036, -0.18779492, 0.4869373, 1.7969185, 1.7969185, 0.5953003, 1.1641518, 1.4198639, 1.4198639, 0.7289299, 0.6795679, -0.32125893, 0.5670582, 0.9882406, 1.0137506, 1.0137506, 0.50666904, 1.1035038, 1.1035038, 0.5569641, 1.4099522, 1.4099522, 0.77946055, 0.77946055, 0.28791544, 0.28791544, 0.36559147, 0.36559147, 0.924434, 1.55701, 1.55701, 1.0121332, 1.6184593, 1.6184593, 0.57110983, 0.4869373, 0.20579681, 0.5953003, 1.1641518, 1.1641518, 0.7289299, 0.7289299, -0.034475047, 0.072677895, 0.072677895, 0.3327982, 1.0137506, 1.0137506, 0.50666904, 2.4643219, 2.4643219, 1.1320051, 1.1320051, 0.6530373, -0.18844908, 0.2872573, 0.2872573, 1.3363862, 1.3363862, -0.0842233, 0.924434, 0.924434, 1.0121332, 1.7140515, 1.7140515, 1.6184593, 0.7669904, -0.45962644, 0.20579681, 0.3263751, 1.4831287, 1.4831287, 0.23908707, 0.650042, 0.650042, 0.23868157, 0.23868157, -0.11622244, 0.85464275, 0.85464275, 0.92660475, 2.4643219, 2.4643219, 1.1320051, 1.1320051, 0.409295, 0.39710623, 0.49964696, 0.49964696, 1.3363862, 1.3363862, -0.120041125, 0.46146888, -0.046481155, 0.037996013, 1.7140515, 1.7140515, 0.7669904, 0.86214805, 0.86214805, 0.4883706, 0.9659361, 1.4831287, 1.9612269, 1.9612269, 0.650042, 0.650042, 1.0616002, 1.0616002, -0.11622244, 0.830386, 0.830386, 0.92660475, 1.8004627, 1.8004627, 0.51650745, 0.51650745, 0.409295, 0.39710623, 0.49964696, 0.49964696, 0.24622276, 0.24622276, 0.09082593, 1.3970653, 1.497715, 1.497715, 0.5652672, 0.54985684, 0.40713033, 0.86214805, 0.86214805, 0.8989674, 0.9659361, 1.4226048, 1.9612269, 1.9612269, 1.3752254, 1.4994915, 1.4994915, 1.0616002, -0.86050975, 0.830386, 0.830386, 0.8576619, 0.8576619, 0.40807465, 1.0770375, 1.0770375, 0.016491488, 0.5746089, 0.5746089, 0.65880215, 0.65880215, 0.16987796, 0.09082593, 1.3970653, 1.497715, 1.497715, 0.81547195, 0.48096985, 0.48096985, 0.85282975, 0.85282975, 1.0044192, 1.0044192, 1.3141544, 1.3141544, 1.3752254, 1.3752254, 1.9258057, 1.951761, 1.951761, 1.2399405, 0.93858516, 0.5125622, 0.8576619, 0.8576619, -0.1105905, 1.0770375, 1.0770375, -0.22482656, 0.7920415, 0.7920415, 0.65880215, 0.65880215, 0.1118724, -0.22295918, 0.3105647, 0.3105647, 0.81547195, 0.81547195, 0.48096985, 0.69865024, 0.85282975, 0.85282975, 1.0044192, 1.0044192, 0.7315516, 0.6335339, 0.6335339, 0.8927082, 1.9258057, 1.951761, 1.951761, 1.9093426, 1.9093426, 1.6638731, 0.90082276, -0.35911658, -0.50900584, 1.6531498, 1.6531498, 0.38616636, 0.7920415, 0.7920415, 0.38182402, 0.3982961, 0.3982961, -0.67570317, 1.2228775, 0.3105647, 0.9719703, 0.9719703, 0.35026592, 0.69865024, 0.69865024, 0.5216252, 0.49689314, 0.7315516, 0.7315516, 0.3196498, -0.40985453, 0.8927082, 0.8927082, 0.63438666, 1.6292758, 1.9093426, 1.9093426, 1.6638731, 0.90082276, 0.7046427, 0.7046427, 1.6531498, 1.6531498, 0.7703309, 0.47348827, 1.8552462, 1.8552462, 1.4156562, 0.98967946, 0.98967946, 1.2228775, 1.1363881, 0.9719703, 0.9719703, 0.572627, 1.9026182, 1.9026182, -0.09302414, -0.18808974, 1.1778295, 1.1778295, -0.40176374, 0.3750199, 0.7979132, 0.7979132, 0.63438666, 1.6292758, 1.6292758, 0.13906415, -0.8576702, 0.88833886, 0.88833886, 0.7046427, 0.93679523, 0.93679523, 1.3289608, 1.3289608, 1.8552462, 1.8552462, 1.4156562, 0.98967946, 0.98967946, 1.1363881, 1.1363881, 0.67161655, 0.2501139, 0.572627, 1.9026182, 1.9026182, 0.64221096, 1.040239, 1.1778295, 1.1778295, 1.1445535, 1.1445535, 1.283455, 1.283455, 2.1886187, 2.1886187, 0.90072393, 0.90072393, 0.8913641, 0.88833886, 0.88833886, -0.52474195, 0.1265688, 0.7930071, 1.3289608, 1.3289608, 0.86301714, 0.86301714, 0.7271749, 0.7375215, 0.7375215, 0.43311873, 1.3207943, 1.3207943, 0.2501139, -0.28339776, 0.8165349, 0.8165349, 0.64221096, 1.040239, 1.0875463, 1.0875463, 1.1445535, 1.1445535, 1.283455, 1.283455, 2.1886187, 2.1886187, 0.90072393, 0.90072393, 0.8913641, -0.17248231, 0.29577908, 0.8425888, 0.8425888, 0.7930071, 0.7930071, 1.0061071, 1.0061071, 1.9595621, 1.9595621, 0.9423143, 0.7375215, 0.7550497, 1.3207943, 1.3207943, -0.09406245, 0.39756522, 0.8165349, 0.8165349, 0.6006052, 0.6006052, 1.0875463, 1.0875463, 0.73930454, 0.61915493, -0.8907292, 0.9219348, 0.9219348, 0.5225576, 0.5225576, 0.45028883, 0.45028883, 1.2341441, 1.4498476, 1.4498476, 0.8425888, 0.24561642, -0.03299648, 1.0061071, 1.0061071, 2.0118642, 2.0118642, 1.336235, 1.336235, 1.3423537, 0.6021635, 0.8732731, 1.9741, 1.9741, 0.47780856, -0.060137887, 0.6006052, 1.0241649, 1.0241649, 0.24461035, 0.089076206, 0.089076206, 0.26473877, 0.9219348, 0.9219348, 0.59542316, 0.59542316, 0.45028883, 0.45028883, 1.2341441, 1.4498476, 1.4498476, -0.24508175, -0.24508175, 0.3140257, 0.37287602, 0.8862932, 2.0118642, 2.0118642, 1.336235, 1.336235, 1.3423537, 0.6498296, 0.8732731, 1.9741, 1.9741, 1.8571023, 1.8571023, 0.41467425, 1.0241649, 1.0241649, 0.24461035, 0.6500845, 1.8209393, 1.8209393, 1.4540358, 1.4540358, 0.59542316, 1.0703601, 1.0703601, 1.2424939, 1.2424939, 1.060715, 1.060715, 0.66200536, 0.66200536, 0.3140257, 0.37287602, 0.8862932, 0.8862932, 0.7312936, 0.07077408, 0.07077408, 0.8188394, 0.6498296, 0.6498296, 0.4347888, 0.09725088, 1.8571023, 1.8571023, 0.66939247, 0.73860705, 1.3042139, 1.3042139, 0.6500845, 1.8209393, 1.8209393, 1.4540358, 1.4540358, 0.2934714, 1.0703601, 1.0703601, 1.2424939, 1.2424939, 0.65592444, -0.038476657, 0.66200536, 0.66200536, 0.50434357, -0.21653287, -0.21653287, 0.7312936, 0.8123336, 0.8123336, 0.54627186, 0.51226676, 0.51226676, 0.3251987, 0.3251987, 0.09725088, 0.679944, 0.679944, 0.66939247, 2.2542837, 2.2542837, 1.3042139, 0.2598325, -0.6783298, -0.083288394, 1.6028637, 1.6028637, 0.4655892, 1.176787, 1.176787, 1.6973464, 1.6973464, 0.65592444, 0.17462958, 0.982327, 1.0374448, 1.0374448, 0.15919177, -0.5053407, -0.5053407, 0.8123336, 0.8123336, 0.54627186, 0.51226676, 0.51226676, 0.08584311, 0.7510253, 0.7510253, 0.8375479, 1.9608808, 1.9608808, 2.2542837, 2.2542837, 1.2098372, 1.2098372, -0.6783298, -0.083288394, 1.6028637, 1.6028637, 0.78990495, 1.176787, 1.176787, 1.6973464, 1.6973464, 0.7591567, 0.17462958, 0.982327, 1.0374448, 1.0374448, 0.5769559, 0.5769559, 0.23497719, 0.6289996, 0.6289996, 0.31403384, 1.0122606, 1.0122606, 0.92874193, 1.081056, 1.5723304, 1.5723304, 1.9608808, 1.9608808, 0.79388034, 0.04399551, 1.2098372, 1.2098372, 0.32099947, 0.32099947, 0.5922057, 0.5922057, 0.78990495, 1.1022825, 1.1022825, 0.5769346, 0.8183537, 0.8183537, 0.100564204, 0.1066523, 1.5190033, 1.5190033, 1.90134, 1.90134, 0.27441698, 0.6289996, 0.6289996, 0.31403384, 1.0122606, 1.0122606, 0.92874193, 1.081056, 1.5723304, 1.5723304, 0.13998847, 0.79388034, 0.79388034, 0.04399551, 0.49097106, 0.8655252, 1.2740445, 1.2740445, 0.5922057, 0.5922057, 0.37317473, 1.0826722, 1.0826722, 1.072636, 1.072636, 0.8183537, -0.13580984, 1.0222104, 1.5190033, 1.5190033, 1.90134, 1.90134, 1.2466952, 1.2466952, 0.70068014, 0.6966507, -0.20697448, 0.6772459, 0.6772459, 0.022212664, 0.8982406, 0.8982406, 0.13998847, 0.9091885, 0.9091885, 0.08071989, 0.49097106, 0.8655252, 2.3374097, 2.3374097, 0.4693722, 1.5237712, 1.5237712, 1.0826722, 1.0826722, 1.226664, 1.226664, 0.8261257, 0.8261257, 1.0222104, 1.0222104, 0.13603121, 0.13603121, 0.5614499, 1.2466952, 1.2466952, 0.70068014, 0.6966507, -0.20697448, 0.6772459, 0.6772459, 1.3257079, 1.3257079, 0.8982406, 0.09152195, 1.0534397, 1.0534397, 1.0147377, 1.4403037, 1.4403037, 2.3374097, 2.3374097, 1.5148823, 1.6043264, 1.6043264, 0.20806953, -0.4292239, 1.226664, 1.226664, 0.8261257, 0.8261257, -0.057756558, -0.21716312, 0.13603121, 0.13603121, 0.5614499, 0.5614499, -0.33275875, -0.20400788, -0.20400788, 1.2598753, 1.2598753, 0.44786966, 1.3257079, 1.3257079, 1.0121683, 1.3841867, 1.3841867, 1.252002, 1.0147377, 1.4403037, 1.4403037, 1.6264315, 1.6264315, 1.5148823, 1.6043264, 1.6043264, 0.6941287, 0.66396505, 0.8220764, 0.8220764, -0.21321523, -0.15279447, 0.59172696, 0.59172696, -0.13022317, 0.61295235, 0.61295235, 0.12955453, 1.17942, 1.17942, 0.836636, 1.2598753, 1.2598753, 0.24099673, -0.28044778, 0.40508026, 1.0121683, 1.3841867, 1.3841867, 1.252002, 0.71178526, 0.27372944, 0.5620131, 0.7955496, 0.7955496, 0.49585068, 0.49585068, 0.6941287, 0.6941287, 0.66396505, 0.8220764, 0.8220764, 1.2412993, 1.2412993, 0.59172696, 0.59172696, -0.05872619, 0.61295235, 0.61295235, 0.12955453, 1.17942, 1.9457614, 1.9457614, 1.0723007, 1.0723007, -0.3070685, -0.3070685, 0.8528891, 0.8528891, 0.16491273, 0.16491273, 0.71178526, 1.9677297, 1.9677297, 0.44768322, 0.7955496, 0.7955496, 0.49585068, 0.49585068, 0.49908426, 0.49908426, 2.0791767, 2.0791767, 0.31263396, 1.2412993, 1.2412993, 1.5932736, 1.5932736, 0.5687224, 0.25163025, 0.25163025, -0.17724662, 0.19939707, 1.9457614, 1.9457614, 1.0723007, 1.0723007, 0.2791965, 0.004174999, 0.8528891, 0.8528891, 0.13468726, 0.13468726, 0.10233585, 1.9677297, 1.9677297, 0.44768322, -0.15065365, -0.15065365, -0.23469436, 0.8997308, 0.8997308, 0.32003194, 2.0791767, 2.0791767, 0.4723279, 0.9663058, 0.9663058, 1.5932736, 1.5932736, 0.74380016, 0.74380016, 0.5921491, -0.3937337, 0.085252576, 1.6616518, 1.6616518, 2.3602285, 2.3602285, 0.5555456, 0.43952233, 0.99914986, 0.99914986, 2.1600132, 2.1600132, -0.01155552, 0.83949065, 1.0923389, 1.0923389, 0.5668694, 0.5668694, -0.23469436, 0.8997308, 0.8997308, 0.76354146, 1.5703039, 1.5703039, 0.4723279, 0.9663058, 0.9663058, 0.21023144, 0.8961672, 0.8961672, 0.74380016, 1.1781894, 1.1781894, -0.94154567, 1.6616518, 1.6616518, 2.3602285, 2.3602285, 0.5555456, 0.43952233, 0.99914986, 0.99914986, 2.1600132, 2.1600132, -0.100301705, 0.302561, 1.0923389, 1.0923389, 0.5668694, 0.5668694, -0.5062735, -0.48948243, 0.76354146, 0.76354146, 0.1926161, 0.1926161, -0.34341785, -0.84721017, -1.2028884, -1.2028884, 0.8961672, 0.8961672, 0.15865193, 1.1781894, 1.1781894, -0.94154567, 0.25471553, 0.25471553}),
			),
		},
	}

	for _, tst := range tsts {
		g := NewGraph()
		x := NodeFromAny(g, tst.inputT)
		y, err := MaxPool2D(x, tensor.Shape{2, 2}, []int{1, 0, 1, 0}, []int{1, 1})
		if err != nil {
			t.Fatal(err)
		}
		m := NewTapeMachine(g, BindDualValues())
		if err := m.RunAll(); err != nil {
			t.Fatal(err)
		}
		m.Close()
		if len(y.Shape()) != len(tst.outputT.Shape()) {
			t.Fatalf("Maxpool: expected shape %v, got %v", tst.outputT.Shape(), y.Shape())
		}
		for i, v := range y.Shape() {
			if v != tst.outputT.Shape()[i] {
				t.Fatalf("Maxpool: expected shape %v, got %v", tst.outputT.Shape(), y.Shape())
			}
		}
		assert.Equal(y.Value().Data(), tst.outputT.Data())
	}
}

func TestMaxPool2D(t *testing.T) {
	assert := assert.New(t)
	dts := []tensor.Dtype{tensor.Float64, tensor.Float32}
	for _, dt := range dts {
		g := NewGraph()
		x := NewTensor(g, dt, 4, WithShape(1, 2, 3, 4), WithInit(RangedFrom(0)))
		y, err := MaxPool2D(x, tensor.Shape{2, 2}, []int{0, 0}, []int{1, 1})
		if err != nil {
			t.Fatal(err)
		}
		cost := Must(Sum(y))
		grads, err := Grad(cost, x)
		if err != nil {
			t.Fatal(err)
		}

		m := NewTapeMachine(g, BindDualValues())
		if err := m.RunAll(); err != nil {
			t.Fatal(err)
		}
		// t.Logf("x %v", x.Value())
		// t.Logf("y: %v", y.Value())
		// t.Logf("c: %v", cost.Value())
		// t.Logf("xG: %v", grads[0])

		h := NewGraph()
		a := NewTensor(h, dt, 4, WithShape(1, 2, 3, 4), WithInit(RangedFrom(0)))
		b, err := MaxPool2D(a, tensor.Shape{2, 2}, []int{0, 0}, []int{1, 1})
		if err != nil {
			t.Fatal(err)
		}
		cost2 := Must(Sum(b))
		if err != nil {
			t.Fatal(err)
		}

		m2 := NewLispMachine(h)
		if err = m2.RunAll(); err != nil {
			t.Fatal(err)
		}
		aG, err := a.Grad()
		if err != nil {
			t.Error(err)
			return
		}

		assert.Equal(x.Value().Data(), a.Value().Data())
		assert.Equal(grads[0].Value().Data(), aG.Data())
		assert.Equal(cost.Value().Data(), cost2.Value().Data())

		m.Close()
		m2.Close()
	}
}

func TestMaxPool(t *testing.T) {
	testCases := []struct {
		desc           string
		input          tensor.Tensor
		kernelSize     tensor.Shape
		pad            []int
		stride         []int
		expectedOutput []float64
		expectedShape  tensor.Shape
		expectedCost   float64
		PoolFunc       func(*Node, tensor.Shape, []int, []int) (*Node, error)
	}{
		{
			desc: "Example 1",
			input: tensor.New(
				tensor.WithShape(1, 1, 4, 4),
				tensor.WithBacking(tensor.Range(tensor.Float64, 0, 16)),
			),
			kernelSize:     []int{4, 4},
			pad:            []int{0, 0},
			stride:         []int{1, 1},
			expectedOutput: []float64{15},
			expectedCost:   196,
			expectedShape:  tensor.Shape{1, 1, 1, 1},
			PoolFunc:       MaxPool2D,
		},
	}

	for _, tcase := range testCases {
		t.Run(tcase.desc, func(t *testing.T) {
			c := require.New(t)

			g := NewGraph()

			input := NewTensor(g, tensor.Float64, tcase.input.Shape().Dims(), WithName("input"), WithShape(tcase.input.Shape()...), WithValue(tcase.input))

			output, err := tcase.PoolFunc(input, tcase.kernelSize, tcase.pad, tcase.stride)
			c.NoError(err)

			t.Logf("%v output shape: %v", tcase.desc, output.Shape())
			t.Logf("%v input shape: %v", tcase.desc, input.Shape())

			y := NewTensor(g, output.Dtype(), output.Dims(), WithShape(output.Shape()...), WithInit(Ones()))

			cost := Must(Mean(Must(Square(Must(Sub(output, y)))))) // MSE

			_, err = Grad(cost, input)
			c.NoError(err)

			// logger := log.New(os.Stdout, "", 0)

			vm := NewTapeMachine(
				g,
				//WithLogger(logger),
				WithWatchlist(),
				BindDualValues(output),
				TraceExec(),
			)

			c.NoError(vm.RunAll())
			c.NoError(vm.Close())

			t.Logf("%v input %v", tcase.desc, input.Value())
			t.Logf("%v result: %v", tcase.desc, output.Value())
			t.Logf("%v cost: %v", tcase.desc, cost.Value())

			c.Equal(tcase.expectedOutput, output.Value().Data())
			c.Equal(tcase.expectedShape, output.Shape())
			c.Equal(tcase.expectedCost, cost.Value().Data().(float64))
		})
	}
}

var (
	bnSetStatsCases = []struct {
		Dtype       tensor.Dtype
		Init        InitWFn
		Shape       tensor.Shape
		expectedErr string
	}{
		{Float32, RangedFromWithStep(0.0, 0.1), tensor.Shape{2, 2}, "invalid runningMean shape (2, 2). Expected: (2)"},
		{Float64, RangedFromWithStep(0.0, 0.1), tensor.Shape{2}, "invalid runningMean type float64. Expected: float32"},
		{Float32, RangedFromWithStep(0.0, 0.1), tensor.Shape{2}, ""},
	}
)

func TestBatchNormSetStats(t *testing.T) {
	g := NewGraph()
	x := NewTensor(g, Float32, 2, WithShape(2, 2), WithInit(Zeroes()), WithName("x"))

	_, _, _, op, _ := BatchNorm(x, nil, nil, 0.9, 1e-5)

	for i, tC := range bnSetStatsCases {
		t.Run(fmt.Sprintf("Example %d (%v)", i+1, tC.Dtype), func(t *testing.T) {
			c := assert.New(t)

			backing := tC.Init(tC.Dtype, tC.Shape...)

			err := op.SetStats(
				tensor.New(
					tensor.Of(tC.Dtype),
					tensor.WithShape(tC.Shape...),
					tensor.WithBacking(backing),
				),
				tensor.New(
					tensor.Of(tC.Dtype),
					tensor.WithShape(tC.Shape...),
					tensor.WithBacking(backing),
				),
			)

			if tC.expectedErr == "" {
				c.NoError(err)

				m, v := op.Stats()
				c.EqualValues(backing, m.Data())
				c.EqualValues(backing, v.Data())
			} else {
				c.EqualError(err, tC.expectedErr)
			}
		})
	}
}

func TestBatchNormAll(t *testing.T) {
	for i, tC := range bnAllCases {
		t.Run(fmt.Sprintf("#%d %v", i+1, tC.desc), func(t *testing.T) {
			rand.Seed(0)

			c := require.New(t)

			g := NewGraph()

			var initOpt NodeConsOpt

			switch v := tC.X.(type) {
			case []float32:
				initOpt = WithValue(
					tensor.New(
						tensor.Of(tensor.Float32),
						tensor.WithShape(tC.XShape...),
						tensor.WithBacking(v),
					),
				)
			case []float64:
				initOpt = WithValue(
					tensor.New(
						tensor.Of(tensor.Float32),
						tensor.WithShape(tC.XShape...),
						tensor.WithBacking(v),
					),
				)
			case InitWFn:
				initOpt = WithInit(v)
			}

			x := NewTensor(g, tC.Dtype, tC.XShape.Dims(), WithShape(tC.XShape...), initOpt, WithName("x"))

			scale := NewTensor(g, tC.Dtype, tC.ScaleShape.Dims(), WithShape(tC.ScaleShape...), WithInit(tC.ScaleInit), WithName("scale"))
			bias := NewTensor(g, tC.Dtype, tC.BiasShape.Dims(), WithShape(tC.BiasShape...), WithInit(tC.BiasInit), WithName("bias"))

			fcWeight := NewTensor(g, tC.Dtype, 2, WithShape(tC.XShape[0], tensor.Shape(tC.XShape[1:]).TotalSize()), WithInit(tC.ScaleInit), WithName("fcWeight"))

			y, _, _, op, err := BatchNorm(x, scale, bias, 0.9, 1e-5)
			c.NoError(err)

			if y.Dims() > 2 {
				y = Must(Reshape(y, fcWeight.Shape()))
			}

			wT := Must(Transpose(fcWeight, 1, 0))

			y = Must(Mul(y, wT))

			var yVal, scaleVal Value
			Read(y, &yVal)
			Read(scale, &scaleVal)

			cost, _ := Mean(y)

			if _, err := Grad(cost, x, fcWeight, scale, bias); err != nil {
				t.Fatal(err)
			}

			optimizer := NewAdamSolver(WithLearnRate(0.02))
			m := NewTapeMachine(g, BindDualValues(x, fcWeight, scale, bias), TraceExec(), WithInfWatch(), WithNaNWatch())

			err = m.RunAll()
			c.NoError(err)

			c.NoError(m.Close())

			err = optimizer.Step(NodesToValueGrads(Nodes{fcWeight, scale, bias}))
			c.NoError(err)

			// for visual inspection
			t.Logf("%v input:\n%v", tC.desc, x.Value())
			t.Logf("%v running mean: %v", tC.desc, op.runningMean)
			t.Logf("%v running var: %v", tC.desc, op.runningVariance)
			t.Logf("%v output:\n%v", tC.desc, y.Value())
			t.Logf("%v output grad:\n%v", tC.desc, y.Deriv().Value())

			t.Logf("%v fc weight:\n%v", tC.desc, fcWeight.Value())

			t.Logf("%v scale: %v", tC.desc, scale.Value())
			t.Logf("%v scale grad: %v", tC.desc, scale.Deriv().Value())
			t.Logf("%v bias: %v", tC.desc, bias.Value())
			t.Logf("%v bias grad: %v", tC.desc, bias.Deriv().Value())
			t.Logf("%v input grad:\n%v", tC.desc, x.Deriv().Value())

			runningMean, runningVariance := op.Stats()

			c.NotNil(runningMean)
			c.NotNil(runningVariance)

			c.True(dawson.AllClose(tC.ExpectedMean, runningMean.Data()), "Mean doesn't match:\ngot=%#v expected=%#v", op.runningMean.Data(), tC.ExpectedMean)
			c.True(dawson.AllClose(tC.ExpectedVariance, runningVariance.Data()), "Var doesn't match:\ngot=%#v expected=%#v", op.runningVariance.Data(), tC.ExpectedVariance)
			c.True(dawson.AllClose(tC.ExpectedTrainResult, yVal.Data()), "Wrong Output\ngot=%#v\nexpected=%#v", yVal.Data(), tC.ExpectedTrainResult)

			c.True(dawson.AllClose(tC.ExpectedOutputGrad, y.Deriv().Value().Data()), "Output Grad doesn't match:\ngot=%#v expected=%#v", y.Deriv().Value().Data(), tC.ExpectedOutputGrad)
			c.True(dawson.AllClose(tC.ExpectedBiasGrad, bias.Deriv().Value().Data()), "Bias Grad doesn't match:\ngot=%#v expected=%#v", bias.Deriv().Value().Data(), tC.ExpectedBiasGrad)
			c.True(dawson.AllClose(tC.ExpectedScaleGrad, scale.Deriv().Value().Data()), "Scale Grad doens't match:\ngot=%#v expected=%#v", scale.Deriv().Value().Data(), tC.ExpectedScaleGrad)
			c.True(dawson.AllClose(tC.ExpectedInputGrad, x.Deriv().Value().Data()), "Input Grad doesn't match:\ngot=%#v expected=%#v", x.Deriv().Value().Data(), tC.ExpectedInputGrad)

			t.Logf("-------- Switching to Eval Mode --------")

			m2 := NewTapeMachine(g, TraceExec(), WithNaNWatch(), WithInfWatch(), EvalMode())

			err = m2.RunAll()
			c.NoError(err)
			c.NoError(m2.Close())

			t.Logf("%v output:\n%v", tC.desc, yVal)
			t.Logf("%v input grad:\n%v", tC.desc, x.Deriv().Value())
			t.Logf("%v running mean: %v", tC.desc, op.runningMean)
			t.Logf("%v running var: %v", tC.desc, op.runningVariance)
			t.Logf("%v bias: %v", tC.desc, bias.Value())
			t.Logf("%v bias grad: %v", tC.desc, bias.Deriv().Value())
			t.Logf("%v scale: %v", tC.desc, scale.Value())
			t.Logf("%v scale grad: %v", tC.desc, scale.Deriv().Value().Data())

			c.True(dawson.AllClose(tC.ExpectedEvalResult, yVal.Data()), "Output doesn't match\ngot=%#v\nexpected=%#v", yVal.Data(), tC.ExpectedEvalResult)
		})

	}
}

func TestBatchNormStacked(t *testing.T) {
	for _, tC := range bnstackedCases {
		t.Run(tC.desc, func(t *testing.T) {
			rand.Seed(0)

			c := require.New(t)

			g := NewGraph()

			x := NewTensor(g, tC.Dtype, tC.XShape.Dims(), WithShape(tC.XShape...), WithInit(tC.XInit), WithName("x"))

			t.Logf("%v input:\n%v", tC.desc, x.Value())

			scale1 := NewTensor(g, tC.Dtype, tC.ScaleShape.Dims(), WithShape(tC.ScaleShape...), WithInit(tC.ScaleInit), WithName("scale1"))
			bias1 := NewTensor(g, tC.Dtype, tC.BiasShape.Dims(), WithShape(tC.BiasShape...), WithInit(tC.BiasInit), WithName("bias1"))

			y1, _, _, op1, err := BatchNorm(x, scale1, bias1, 0.9, 1e-5)
			c.NoError(err)

			op1.SetTraining(true)

			scale2 := NewTensor(g, tC.Dtype, tC.ScaleShape.Dims(), WithShape(tC.ScaleShape...), WithInit(tC.ScaleInit), WithName("scale2"))
			bias2 := NewTensor(g, tC.Dtype, tC.BiasShape.Dims(), WithShape(tC.BiasShape...), WithInit(tC.BiasInit), WithName("bias2"))

			y2, _, _, op2, err := BatchNorm(y1, scale2, bias2, 0.9, 1e-5)
			c.NoError(err)

			op2.SetTraining(true)

			var yVal, scaleVal Value
			Read(y2, &yVal)
			Read(scale2, &scaleVal)

			cost := Must(Mean(y2))

			if _, err := Grad(cost, x, scale1, bias1, scale2, bias2); err != nil {
				t.Fatal(err)
			}

			m := NewTapeMachine(g, BindDualValues(x, scale1, bias1, scale2, bias2), TraceExec(), WithInfWatch())

			optim := NewAdamSolver(WithLearnRate(0.05))
			// optim := NewRMSPropSolver(WithLearnRate(0.05))

			for i := 0; i < tC.Epochs; i++ {
				t.Logf("-------- STEP %v", i+1)
				op1.SetTraining(true)
				op2.SetTraining(true)

				err = m.RunAll()
				c.NoError(err)

				err = optim.Step(NodesToValueGrads(Nodes{
					scale1,
					bias1,
					scale2,
					bias2,
				}))
				c.NoError(err)

				m.Reset()

			}

			c.NoError(err)

			c.NoError(m.Close())

			t.Logf("%v running mean: %v", tC.desc, op2.runningMean)
			t.Logf("%v running var: %v", tC.desc, op2.runningVariance)
			t.Logf("%v y1:\n%v", tC.desc, y1.Value())
			t.Logf("%v output:\n%v", tC.desc, y2.Value())
			t.Logf("%v output grad:\n%v", tC.desc, y2.Deriv().Value())
			t.Logf("%v scale grad: %v", tC.desc, scale2.Deriv().Value())
			t.Logf("%v bias grad: %v", tC.desc, bias2.Deriv().Value())
			t.Logf("%v input grad:\n%v", tC.desc, x.Deriv().Value())

			c.True(dawson.AllClose(tC.ExpectedMean, op2.runningMean.Data()), "Mean doesn't match:\ngot=%#v expected=%#v", op2.runningMean.Data(), tC.ExpectedMean)
			c.True(dawson.AllClose(tC.ExpectedVariance, op2.runningVariance.Data()), "Var doesn't match:\ngot=%#v expected=%#v", op2.runningVariance.Data(), tC.ExpectedVariance)

			c.True(dawson.AllClose(tC.ExpectedTrainResult, yVal.Data()), "Wrong Output\ngot=%#v\nexpected=%#v", yVal.Data(), tC.ExpectedTrainResult)

			t.Logf("-------- Switching to Eval Mode --------")

			m2 := NewTapeMachine(g, TraceExec(), WithInfWatch(), EvalMode())

			err = m2.RunAll()
			c.NoError(err)

			c.NoError(m2.Close())

			c.InDeltaSlice(tC.ExpectedEvalResult, yVal.Data(), 1e-5, "Output doesn't match\ngot=%#v\nexpected=%#v", yVal.Data(), tC.ExpectedEvalResult)
		})
	}
}

func TestLeakyRelu(t *testing.T) {
	tests := []struct {
		name  string
		alpha float64
		xT    tensor.Tensor
		yT    tensor.Tensor
	}{
		{
			name:  "simple float32",
			alpha: 0.1,
			xT: tensor.New(
				tensor.WithShape(3, 4, 5),
				tensor.WithBacking([]float32{1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558, -0.9772779, 0.95008844, -0.1513572, -0.10321885, 0.41059852, 0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324, 0.33367434, 1.4940791, -0.20515826, 0.3130677, -0.85409576, -2.5529897, 0.6536186, 0.8644362, -0.742165, 2.2697546, -1.4543657, 0.045758516, -0.18718386, 1.5327792, 1.4693588, 0.15494743, 0.37816253, -0.88778573, -1.9807965, -0.34791216, 0.15634897, 1.2302907, 1.2023798, -0.3873268, -0.30230275, -1.048553, -1.420018, -1.7062702, 1.9507754, -0.5096522, -0.4380743, -1.2527953, 0.7774904, -1.6138978, -0.21274029, -0.89546657, 0.3869025, -0.51080513, -1.1806322, -0.028182229, 0.42833188, 0.06651722, 0.3024719, -0.6343221, -0.36274117}),
			),
			yT: tensor.New(
				tensor.WithShape(3, 4, 5),
				tensor.WithBacking([]float32{1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558, -0.09772779, 0.95008844, -0.01513572, -0.010321885, 0.41059852, 0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324, 0.33367434, 1.4940791, -0.020515827, 0.3130677, -0.085409574, -0.25529897, 0.6536186, 0.8644362, -0.07421651, 2.2697546, -0.14543657, 0.045758516, -0.018718386, 1.5327792, 1.4693588, 0.15494743, 0.37816253, -0.08877858, -0.19807965, -0.034791216, 0.15634897, 1.2302907, 1.2023798, -0.03873268, -0.030230274, -0.1048553, -0.1420018, -0.17062703, 1.9507754, -0.05096522, -0.04380743, -0.12527953, 0.7774904, -0.16138978, -0.021274028, -0.08954666, 0.3869025, -0.051080514, -0.11806323, -0.0028182229, 0.42833188, 0.06651722, 0.3024719, -0.06343221, -0.036274116}),
			),
		},
		{
			name:  "simple float64",
			alpha: 0.1,
			xT: tensor.New(
				tensor.WithShape(3, 4, 5),
				tensor.WithBacking([]float64{1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558, -0.9772779, 0.95008844, -0.1513572, -0.10321885, 0.41059852, 0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324, 0.33367434, 1.4940791, -0.20515826, 0.3130677, -0.85409576, -2.5529897, 0.6536186, 0.8644362, -0.742165, 2.2697546, -1.4543657, 0.045758516, -0.18718386, 1.5327792, 1.4693588, 0.15494743, 0.37816253, -0.88778573, -1.9807965, -0.34791216, 0.15634897, 1.2302907, 1.2023798, -0.3873268, -0.30230275, -1.048553, -1.420018, -1.7062702, 1.9507754, -0.5096522, -0.4380743, -1.2527953, 0.7774904, -1.6138978, -0.21274029, -0.89546657, 0.3869025, -0.51080513, -1.1806322, -0.028182229, 0.42833188, 0.06651722, 0.3024719, -0.6343221, -0.36274117}),
			),
			yT: tensor.New(
				tensor.WithShape(3, 4, 5),
				tensor.WithBacking([]float64{1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558, -0.09772779, 0.95008844, -0.01513572, -0.010321885, 0.41059852, 0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324, 0.33367434, 1.4940791, -0.020515827, 0.3130677, -0.085409574, -0.25529897, 0.6536186, 0.8644362, -0.07421651, 2.2697546, -0.14543657, 0.045758516, -0.018718386, 1.5327792, 1.4693588, 0.15494743, 0.37816253, -0.08877858, -0.19807965, -0.034791216, 0.15634897, 1.2302907, 1.2023798, -0.03873268, -0.030230274, -0.1048553, -0.1420018, -0.17062703, 1.9507754, -0.05096522, -0.04380743, -0.12527953, 0.7774904, -0.16138978, -0.021274028, -0.08954666, 0.3869025, -0.051080514, -0.11806323, -0.0028182229, 0.42833188, 0.06651722, 0.3024719, -0.06343221, -0.036274116}),
			),
		},
	}
	for _, tst := range tests {
		t.Run(tst.name, func(t *testing.T) {
			g := NewGraph()
			assert := assert.New(t)
			x := NodeFromAny(g, tst.xT)
			output, err := LeakyRelu(x, tst.alpha)

			if err != nil {
				t.Fatalf("%+v", err)
			}
			m := NewTapeMachine(g)
			if err := m.RunAll(); err != nil {
				t.Fatalf("%+v", err)
			}
			defer m.Close()
			assert.InDeltaSlice(tst.yT.Data(), output.Value().Data(), 1e-6, "the two tensors should be equal.")
		})
	}
}

func TestGlobalAveragePool2D_fwdPass(t *testing.T) {
	for _, tst := range []struct {
		inputT         tensor.Tensor
		expectedOutput tensor.Tensor
	}{
		{
			inputT: tensor.New(
				tensor.WithShape(1, 3, 5, 5),
				tensor.WithBacking([]float32{
					1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558,
					-0.9772779, 0.95008844, -0.1513572, -0.10321885, 0.41059852,
					0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324,
					0.33367434, 1.4940791, -0.20515826, 0.3130677, -0.85409576,
					-2.5529897, 0.6536186, 0.8644362, -0.742165, 2.2697546,

					-1.4543657, 0.045758516, -0.18718386, 1.5327792, 1.4693588,
					0.15494743, 0.37816253, -0.88778573, -1.9807965, -0.34791216,
					0.15634897, 1.2302907, 1.2023798, -0.3873268, -0.30230275,
					-1.048553, -1.420018, -1.7062702, 1.9507754, -0.5096522,
					-0.4380743, -1.2527953, 0.7774904, -1.6138978, -0.21274029,

					-0.89546657, 0.3869025, -0.51080513, -1.1806322, -0.028182229,
					0.42833188, 0.06651722, 0.3024719, -0.6343221, -0.36274117,
					-0.67246044, -0.35955316, -0.8131463, -1.7262826, 0.17742614,
					-0.40178093, -1.6301984, 0.46278226, -0.9072984, 0.051945396,
					0.7290906, 0.12898292, 1.1394007, -1.2348258, 0.40234163})),
			expectedOutput: tensor.New(
				tensor.WithShape(1, 3, 1, 1),
				tensor.WithBacking([]float32{0.47517386, -0.1940553, -0.28326008})),
		},
		{
			inputT: tensor.New(
				tensor.WithShape(1, 3, 5, 5),
				tensor.WithBacking([]float64{
					1.7640524, 0.4001572, 0.978738, 2.2408931, 1.867558,
					-0.9772779, 0.95008844, -0.1513572, -0.10321885, 0.41059852,
					0.14404356, 1.4542735, 0.7610377, 0.121675014, 0.44386324,
					0.33367434, 1.4940791, -0.20515826, 0.3130677, -0.85409576,
					-2.5529897, 0.6536186, 0.8644362, -0.742165, 2.2697546,

					-1.4543657, 0.045758516, -0.18718386, 1.5327792, 1.4693588,
					0.15494743, 0.37816253, -0.88778573, -1.9807965, -0.34791216,
					0.15634897, 1.2302907, 1.2023798, -0.3873268, -0.30230275,
					-1.048553, -1.420018, -1.7062702, 1.9507754, -0.5096522,
					-0.4380743, -1.2527953, 0.7774904, -1.6138978, -0.21274029,

					-0.89546657, 0.3869025, -0.51080513, -1.1806322, -0.028182229,
					0.42833188, 0.06651722, 0.3024719, -0.6343221, -0.36274117,
					-0.67246044, -0.35955316, -0.8131463, -1.7262826, 0.17742614,
					-0.40178093, -1.6301984, 0.46278226, -0.9072984, 0.051945396,
					0.7290906, 0.12898292, 1.1394007, -1.2348258, 0.40234163})),
			expectedOutput: tensor.New(
				tensor.WithShape(1, 3, 1, 1),
				tensor.WithBacking([]float64{0.47517386, -0.1940553, -0.28326008})),
		},
	} {
		inputT := tst.inputT
		expectedOutput := tst.expectedOutput
		g := NewGraph()
		assert := assert.New(t)
		x := NodeFromAny(g, inputT)
		output, err := GlobalAveragePool2D(x)

		if err != nil {
			t.Fatal(err)
		}
		m := NewTapeMachine(g)
		if err := m.RunAll(); err != nil {
			t.Fatalf("%+v", err)
		}
		defer m.Close()
		if len(output.Shape()) != len(expectedOutput.Shape()) {
			t.Fatalf("Bad output shape, expected %v, got %v", expectedOutput.Shape(), output.Shape())
		}
		for i, d := range output.Shape() {
			if expectedOutput.Shape()[i] != d {
				t.Fatalf("Bad output shape, expected %v, got %v", expectedOutput.Shape(), output.Shape())
			}
		}
		assert.InDeltaSlice(expectedOutput.Data(), output.Value().Data(), 1e-6, "the two tensors should be equal.")
	}
}

func TestConv2dErrors(t *testing.T) {
	g := NewGraph()

	testCases := []struct {
		desc                  string
		im                    *Node
		filter                *Node
		kernelShape           tensor.Shape
		pad, stride, dilation []int
		err                   string
		panics                bool
	}{
		{
			desc:        "Succesful",
			im:          NewTensor(g, tensor.Float64, 4, WithShape(1, 1, 28, 28)),
			filter:      NewTensor(g, tensor.Float64, 4, WithShape(32, 1, 3, 3)),
			kernelShape: tensor.Shape{3, 3},
			pad:         []int{1, 1},
			stride:      []int{1, 1},
			dilation:    []int{1, 1},
			err:         "",
		},
		{
			desc:        "5dIM",
			im:          NewTensor(g, tensor.Float64, 5, WithShape(1, 1, 1, 28, 28)),
			filter:      NewTensor(g, tensor.Float64, 4, WithShape(32, 1, 3, 3)),
			kernelShape: tensor.Shape{3, 3},
			pad:         []int{1, 1},
			stride:      []int{1, 1},
			dilation:    []int{1, 1},
			err:         "im should have 4 dims, got 5 dims",
		},
		{
			desc:        "5dFilter",
			im:          NewTensor(g, tensor.Float64, 4, WithShape(1, 1, 28, 28)),
			filter:      NewTensor(g, tensor.Float64, 5, WithShape(32, 1, 1, 3, 3)),
			kernelShape: tensor.Shape{3, 3},
			pad:         []int{1, 1},
			stride:      []int{1, 1},
			dilation:    []int{1, 1},
			err:         "filter should have 4 dims, got 5 dims",
		},
		{
			desc:        "Shapes",
			im:          NewTensor(g, tensor.Float64, 4, WithShape(1, 1, 28, 28)),
			filter:      NewTensor(g, tensor.Float64, 4, WithShape(32, 3, 3, 3)),
			kernelShape: tensor.Shape{3, 3},
			pad:         []int{1, 1},
			stride:      []int{1, 1},
			dilation:    []int{1, 1},
			err:         "3 (kernel) * 3 (width) * 3 (height) must be 9, got 27",
		},
	}

	for _, tC := range testCases {
		t.Run(tC.desc, func(t *testing.T) {
			c := assert.New(t)

			_, err := Conv2d(tC.im, tC.filter, tC.kernelShape, tC.pad, tC.stride, tC.dilation)
			if tC.err != "" {
				require.Error(t, err)
				c.Equal(tC.err, err.Error())
			} else {
				c.NoError(err)
			}
		})
	}
}
