package main

import (
	"fmt"
	"log"
	"os"
	"os/exec"
	"os/user"
	"path"
	"strings"
	"text/template"
)

const genmsg = "// Code generated by gencudaengine, which is a API generation tool for Gorgonia. DO NOT EDIT."

const (
	arithOut = "arith.go"
	unaryOut = "unary.go"
	cmpOut   = "cmp.go"
)

var (
	gopath, cudaengloc string
)

var funcmap = template.FuncMap{
	"lower": strings.ToLower,
}

type binOp struct {
	Method       string
	ScalarMethod string
}

var ariths = []binOp{
	{"Add", "Add"},
	{"Sub", "Sub"},
	{"Mul", "Mul"},
	{"Div", "Div"},
	{"Pow", "Pow"},
	{"Mod", "Mod"},
}

var cmps = []binOp{
	{"Lt", "Lt"},
	{"Lte", "Lte"},
	{"Gt", "Gt"},
	{"Gte", "Gte"},
	{"ElEq", "Eq"},
	{"ElNe", "Ne"},
}

func init() {
	gopath = os.Getenv("GOPATH")
	if gopath == "" {
		usr, err := user.Current()
		if err != nil {
			log.Fatal(err)
		}
		gopath = path.Join(usr.HomeDir, "go")
		stat, err := os.Stat(gopath)
		if err != nil {
			log.Fatal(err)
		}
		if !stat.IsDir() {
			log.Fatal("You need to define a $GOPATH")
		}
	}
	cudaengloc = path.Join(gopath, "src/gorgonia.org/gorgonia/cuda")
}

func generateAriths() {
	p := path.Join(cudaengloc, arithOut)
	f, _ := os.OpenFile(p, os.O_CREATE|os.O_TRUNC|os.O_WRONLY, 0644)
	fmt.Fprintf(f, "package cuda\n\n%v\n\n", genmsg)

	for _, op := range ariths {
		binopTmpl.Execute(f, op)
	}

	f.Close()
	cmd := exec.Command("goimports", "-w", p)
	if err := cmd.Run(); err != nil {
		log.Fatalf("Go imports failed with %v for %q", err, p)
	}
}

func generateCmps() {
	p := path.Join(cudaengloc, cmpOut)
	f, _ := os.OpenFile(p, os.O_CREATE|os.O_TRUNC|os.O_WRONLY, 0644)
	fmt.Fprintf(f, "package cuda\n\n%v\n\n", genmsg)

	for _, op := range cmps {
		binopTmpl.Execute(f, op)
	}

	f.Close()
	cmd := exec.Command("goimports", "-w", p)
	if err := cmd.Run(); err != nil {
		log.Fatalf("Go imports failed with %v for %q", err, p)
	}
}

func main() {
	generateAriths()
	generateCmps()
}
