
/**
 * @file /magma/providers/cryptography/parameters.c
 *
 * @brief DESCRIPTIONxxxGOESxxxHERE
 */

#include "magma.h"

pthread_mutex_t dhparam_lock;
DH *dh2048 = NULL, *dh4096 = NULL;

/**
 * @brief	The DH param generation callback.
 */
int dh_params_generate_callback(int p, int n, BN_GENCB *cb) {

//	if (p == 0 && n % 100 == 0) log_pedantic("dh_generate { id = %lu / n = %i }", ssl_thread_id_callback(), n);
//	else if (p == 3) log_pedantic("dh_generate { id = %lu / p = 3 }", ssl_thread_id_callback());

	return (status() ? 1 : 0);
}

/// TODO: Spawn a thread at startup to generate the keys, and block the dh_exchange functions below from returning until its done.
void dh_params_generate(void) {

	DH *holder;
	BN_GENCB cb;
	int_t codes = 0;
	bool_t loop = true;

	// Just use the previously generated parameters.
	if (!magma.iface.cryptography.dhparams_rotate) {
		dh2048 = dh_params_2048();
		dh4096 = dh_params_4096();
		return;
	}

	BN_GENCB_set(&cb, dh_params_generate_callback, NULL);

	log_pedantic("Generating DH key parameters with a %i bit prime.", 2048);

	if (!(holder = DH_new_d())) {
		log_error("Unable to create new DH key.");
		return;
	}

	for (uint_t i = 0; i < 16 && loop && status(); i++) {

		if (!DH_generate_parameters_ex_d(holder, 2048, 2, &cb)) {
			log_error("Encountered error while generating Diffie-Hellman parameters. { error = %s }",
				ssl_error_string(MEMORYBUF(256), 256));
			DH_free_d(holder);
			return;
		}
		else if (DH_check_d(holder, &codes) != 1 ||
			(codes & (DH_CHECK_P_NOT_SAFE_PRIME | DH_UNABLE_TO_CHECK_GENERATOR | DH_NOT_SUITABLE_GENERATOR))) {

			log_pedantic("Invalid DH params. { codes = %i }", codes);
			DH_free_d(holder);

			if (!(holder = DH_new_d())) {
				log_error("Unable to create a new set of DH parameters.");
				return;
			}
		}
		else {
			loop = false;
		}
	}

	if (!loop) {
		mutex_lock(&dhparam_lock);
		if (!dh2048) {
			dh2048 = holder;
		}
		else {
			DH_free_d(holder);

		}
		mutex_unlock(&dhparam_lock);
	}
	// The loop didn't generate a valid prime in the 16 tries allowed.
	else {
		DH_free_d(holder);
		return;
	}

	loop = true;
	log_pedantic("Generating DH key parameters with a %i bit prime.", 4096);

	if (!(holder = DH_new_d())) {
		log_error("Unable to create new DH key.");
		return;
	}

	for (uint_t i = 0; i < 16 && loop && status(); i++) {

		if (!DH_generate_parameters_ex_d(holder, 4096, 2, &cb)) {
			log_error("Encountered error while generating Diffie-Hellman parameters. { error = %s }",
				ssl_error_string(MEMORYBUF(256), 256));
			DH_free_d(holder);
			return;
		}
		else if (DH_check_d(holder, &codes) != 1 ||
			(codes & (DH_CHECK_P_NOT_SAFE_PRIME | DH_UNABLE_TO_CHECK_GENERATOR | DH_NOT_SUITABLE_GENERATOR))) {

			log_pedantic("Invalid DH params. { codes = %i }", codes);
			DH_free_d(holder);

			if (!(holder = DH_new_d())) {
				log_error("Unable to create a new set of DH parameters.");
				return;
			}
		}
		else {
			loop = false;
		}
	}

	if (!loop) {
		mutex_lock(&dhparam_lock);
		if (!dh4096) {
			dh4096 = holder;
		}
		else {
			DH_free_d(holder);

		}
		mutex_unlock(&dhparam_lock);
	}
	// The loop didn't generate a valid prime in the 16 tries allowed.
	else {
		DH_free_d(holder);
		return;
	}
	return;
}

DH *dh_params_2048(void) {

	DH *dh;

	static uchr_t dh2048_p[] = {
		0xC6,0x4C,0xDD,0xDB,0x76,0x7E,0xFE,0x57,0xBE,0x8E,0x3B,0xA4,
		0xAD,0x06,0x20,0x23,0x49,0x2D,0x6F,0xB2,0x61,0xCB,0x78,0xAC,
		0xC1,0x4F,0x00,0x65,0x35,0x0A,0x61,0x78,0x7C,0xF1,0x56,0xA8,
		0x0D,0xAB,0x98,0xE0,0xD1,0x02,0x09,0x61,0x95,0x0F,0x9E,0xB7,
		0x95,0x7A,0xAD,0xD6,0x84,0x19,0xF5,0x9B,0x82,0xDC,0xE9,0x37,
		0x5B,0xDC,0xC6,0xF6,0xDB,0x6F,0xF1,0x49,0x51,0xA7,0xF2,0xE7,
		0xA8,0xF7,0xD1,0x5C,0xFD,0x52,0x60,0xB1,0x3A,0xB2,0xE7,0xDD,
		0x86,0xD0,0xD3,0x83,0x90,0xC2,0xB0,0x61,0x9B,0xB4,0xBE,0x9C,
		0x28,0x68,0x83,0x5E,0x24,0x10,0x32,0xFD,0xFC,0xC7,0xB4,0x14,
		0xF9,0x79,0x15,0x4D,0x9F,0xD1,0x1C,0x42,0x71,0xF1,0x8E,0x11,
		0x10,0xD0,0x27,0xD8,0xA2,0xC4,0x77,0x63,0x86,0xCB,0x8B,0x3F,
		0x78,0x47,0xF5,0x6D,0xA5,0x6A,0xF1,0x8E,0xB7,0xB2,0xB3,0xA0,
		0x26,0x7F,0x39,0xB4,0xBC,0x66,0xE2,0x5A,0x0F,0x96,0xA6,0xCD,
		0x1B,0x5D,0x66,0x03,0x3B,0x52,0x6C,0x1D,0xA2,0x56,0x96,0x12,
		0x6B,0xEF,0xBC,0x83,0xF8,0x3D,0x4B,0x43,0x23,0xA4,0x63,0xE4,
		0x5E,0x54,0x80,0x50,0x88,0x66,0xF4,0xE7,0xDE,0x0F,0x30,0x8E,
		0x07,0x06,0xF6,0x1E,0xD0,0xAB,0xB6,0x55,0x72,0x4C,0xD8,0x7F,
		0x49,0x5B,0x18,0xC3,0xB5,0xF0,0x88,0x5D,0x38,0x58,0x2E,0xF0,
		0x10,0x6C,0xBD,0x67,0x04,0x68,0x03,0xF9,0xF1,0x16,0xD0,0x36,
		0x6F,0x27,0xDA,0xFF,0x44,0x39,0x86,0x50,0x50,0x89,0x64,0x72,
		0x75,0x55,0x4E,0xCC,0x90,0x96,0xEB,0xD1,0x37,0xBD,0xA8,0x30,
		0x9D,0x68,0xC5,0xD3
	};
	static uchr_t dh2048_g[] = {
		0x02
	};

	if (!(dh = DH_new_d())) {
		return NULL;
	}

	dh->p = BN_bin2bn_d(dh2048_p, sizeof(dh2048_p), NULL);
	dh->g = BN_bin2bn_d(dh2048_g, sizeof(dh2048_g), NULL);

	if (!(dh->p) || !(dh->g)) {
		DH_free_d(dh);
		return NULL;
	}

	return dh;
}

DH *dh_params_4096(void) {

	DH *dh;

	static uchr_t dh4096_p[] = {
		0x82,0xEE,0x4E,0x93,0x9B,0x20,0x94,0x2C,0xDD,0xCD,0xCC,0x19,
		0xAF,0xC4,0xEA,0xC0,0xE9,0xF5,0x1A,0x35,0x2C,0x3D,0x46,0x16,
		0x9C,0x9A,0xD8,0xA6,0xFC,0xBD,0xBC,0xFA,0x4B,0x70,0xE7,0x06,
		0x3B,0x05,0xBA,0x50,0x1A,0xC0,0xE7,0x32,0x85,0x60,0x51,0x5F,
		0x50,0xB6,0x4F,0x5D,0xC2,0xCB,0x71,0xCE,0xBC,0x56,0x63,0x1F,
		0x61,0x5C,0x7F,0xAF,0x82,0x00,0x40,0xC5,0xBA,0x3D,0x70,0xA8,
		0xED,0x6E,0x9B,0x66,0x23,0x9B,0xB6,0xE9,0x79,0xE0,0xD6,0x5E,
		0x84,0x65,0xE5,0x5D,0xF7,0xC8,0x51,0x76,0x9B,0x7E,0x90,0x74,
		0x2E,0xFB,0xE3,0x56,0x8F,0x56,0x98,0x35,0xE0,0xFB,0x44,0xAC,
		0x43,0x2A,0xA0,0xE8,0x29,0x56,0x77,0xF2,0x42,0x1A,0xF3,0x19,
		0xA3,0x48,0x9C,0xD7,0xDA,0xFB,0xA5,0x6B,0x97,0x7E,0x06,0x7E,
		0x3D,0xDD,0x0C,0x52,0x3A,0x1B,0xF1,0x36,0x2A,0x7C,0xF5,0xB7,
		0xD8,0xBD,0x08,0xC8,0xF0,0x70,0x9B,0x5C,0xB3,0x01,0x0E,0x2E,
		0x07,0xDE,0xB4,0xB2,0x65,0x0E,0x31,0xEF,0x15,0x58,0x1E,0x8E,
		0x6B,0xAF,0x7D,0xE0,0xEB,0x14,0xCA,0xD1,0xDB,0x01,0x02,0x6E,
		0xC3,0x0A,0xBE,0x91,0x8B,0xA3,0xDF,0x83,0x21,0x92,0xDF,0x0B,
		0xAA,0xC0,0x61,0x18,0xAB,0x75,0x52,0xE5,0x02,0x53,0x87,0xFF,
		0x84,0x90,0x46,0x68,0xB6,0x76,0x1C,0x9A,0x0B,0x8B,0xB7,0xFC,
		0x13,0xA8,0x72,0x68,0xDC,0x1F,0xCF,0xAB,0xCA,0x67,0xD6,0xFF,
		0x0A,0xDD,0x8B,0xA5,0xBB,0x33,0x03,0xC7,0x9A,0xD9,0x8A,0x99,
		0x84,0x86,0xE4,0xB3,0xFB,0xE0,0xE3,0x84,0xDF,0xD0,0x6A,0x29,
		0xC7,0x91,0xE7,0x24,0x9D,0xD3,0x40,0x09,0x9C,0xAD,0x31,0x53,
		0x94,0x80,0x92,0xB8,0x36,0x48,0x92,0xD5,0x8E,0xE0,0x4D,0x30,
		0xB9,0x9C,0x62,0xBF,0xA1,0x16,0x5E,0x2E,0x53,0x34,0xCC,0xEA,
		0xD7,0x2C,0x9F,0xDE,0x8F,0x38,0x3E,0x8B,0xCC,0x10,0x70,0xF0,
		0x64,0xBD,0xEC,0x55,0xF5,0xFF,0x8A,0xC6,0x99,0xE6,0x98,0xC7,
		0x57,0x62,0x9D,0xE9,0x28,0xCA,0xA7,0x5A,0xAD,0xE2,0x04,0x2C,
		0x10,0xDF,0xD5,0x88,0x63,0x2E,0x1E,0x4B,0xB5,0x4A,0xAF,0x2E,
		0xB5,0xAD,0xAA,0xA7,0xF6,0x92,0x73,0x61,0x12,0xDB,0xD8,0x2A,
		0x05,0x37,0x46,0x3B,0xF4,0x21,0x95,0x07,0x45,0xB7,0x30,0x7B,
		0x2B,0x92,0x4D,0x7B,0x67,0x78,0xE0,0xB8,0x79,0x08,0xA3,0xFA,
		0xE3,0x8C,0xB8,0x54,0xF8,0x32,0xB2,0x34,0x31,0x15,0xFA,0xE0,
		0xF9,0xB1,0x30,0x34,0xF0,0x3A,0x45,0x49,0x3E,0x85,0xB5,0xE5,
		0xC8,0x21,0x91,0x15,0x3E,0xBD,0x2E,0x00,0x6E,0x26,0x86,0xDE,
		0x92,0x40,0xE6,0x37,0x1E,0x0B,0x30,0x58,0x6A,0xAF,0xA4,0x6E,
		0x59,0x6D,0x62,0x44,0x15,0x7B,0xAA,0x1F,0x63,0x2D,0x7E,0x25,
		0x39,0xE5,0xEA,0x80,0xE1,0x0A,0x23,0xBB,0x6B,0x9E,0xAD,0xFF,
		0x2A,0x3B,0x7C,0x44,0xE4,0x87,0x4A,0x9C,0xBB,0xAF,0xAC,0xF2,
		0x29,0xF1,0x3B,0xF3,0x25,0x34,0xB1,0x53,0x99,0x4A,0x4A,0x60,
		0xE2,0xB1,0x9E,0xE2,0x55,0x91,0x72,0x1C,0xB6,0xFC,0x02,0x71,
		0x53,0x7D,0xCD,0xB3,0x18,0xBA,0x99,0x08,0x2D,0x0E,0x7E,0x6E,
		0x87,0x3A,0x77,0xF0,0x32,0x08,0x76,0x0D,0x74,0xF4,0xCC,0x06,
		0xB3,0xA4,0xBA,0xF9,0xBE,0x46,0xFE,0x13
	};
	static uchr_t dh4096_g[] = {
		0x02
	};

	if (!(dh = DH_new_d())) {
		return NULL;
	}

	dh->p = BN_bin2bn_d(dh4096_p, sizeof(dh4096_p), NULL);
	dh->g = BN_bin2bn_d(dh4096_g, sizeof(dh4096_g), NULL);

	if (!(dh->p) || !(dh->g)) {
		DH_free_d(dh);
		return NULL;
	}

	return dh;
}

DH * dh_static_2048(SSL *ssl, int is_export, int keylength) {
	return dh_params_2048();
}

DH * dh_static_4096(SSL *ssl, int is_export, int keylength) {
	return dh_params_4096();
}

/**
 * @brief	Callback handler for the Diffie-Hellman parameter generation process necessary for PFS.
 * @param	ssl			the SSL session for which the callback was triggered.
 * @param	is_export	We ignore this parameter, as we don't support insecure cipher suites.
 * @param	keylength	the length, in bits, of the DH key to be generated is also ignored.
 * @return	a pointer to a Diffie-Hellman structure with a 2048 bit prime which is used for the session key, or NULL on failure.
 */
DH * dh_exchange_2048(SSL *ssl, int is_export, int keylength) {

	DH *holder;
	BN_GENCB cb;
	int_t codes = 0;
	bool_t loop = true;

	// If a generated key is already available then simply return it.
	if ((holder = dh2048)) {
		return holder;
	}

	else if (!(holder = DH_new_d())) {
		log_error("Unable to create new DH key.");
		return NULL;
	}

	BN_GENCB_set(&cb, dh_params_generate_callback, NULL);
	log_pedantic("Generating %i bit DH key parameters.", 2048);

	for (uint_t i = 0; i < 16 && loop && status(); i++) {

		if (!DH_generate_parameters_ex_d(holder, 2048, 2, &cb)) {
			log_error("Encountered error while generating Diffie-Hellman parameters. { error = %s }",
				ssl_error_string(MEMORYBUF(256), 256));
			if (holder) DH_free_d(holder);
			return NULL;
		}
		else if (DH_check_d(holder, &codes) != 1 ||
			(codes & (DH_CHECK_P_NOT_SAFE_PRIME | DH_UNABLE_TO_CHECK_GENERATOR | DH_NOT_SUITABLE_GENERATOR))) {
			log_pedantic("Invalid DH params. { codes = %i }", codes);
			DH_free_d(holder);
			if (!(holder = DH_new_d())) {
				log_error("Unable to create new DH key.");
				return NULL;
			}
		}
		else {
			loop = false;
		}
	}

	if (!loop) {
		mutex_lock(&dhparam_lock);
		if (!dh2048) {
			dh2048 = holder;
		}
		else {
			DH_free_d(holder);
			holder = dh2048;
		}
		mutex_unlock(&dhparam_lock);
	}
	// The loop didn't generate a valid prime in the 16 tries allowed.
	else {
		DH_free_d(holder);
		return NULL;
	}
	return holder;
}

/**
 * @brief	Callback handler for the Diffie-Hellman parameter generation process necessary for PFS.
 * @param	ssl			the SSL session for which the callback was triggered.
 * @param	is_export	We ignore this parameter, as we don't support insecure cipher suites.
 * @param	keylength	the length, in bits, of the DH key to be generated is also ignored.
 * @return	a pointer to a Diffie-Hellman structure with a 4096 bit prime which is used for the session key, or NULL on failure.
 */
DH * dh_exchange_4096(SSL *ssl, int is_export, int keylength) {

	DH *holder;
	BN_GENCB cb;
	int_t codes = 0;
	bool_t loop = true;

	// If a generated key is already available then simply return it.
	if ((holder = dh4096)) {
		return holder;
	}

	else if (!(holder = DH_new_d())) {
		log_error("Unable to create new DH key.");
		return NULL;
	}

	BN_GENCB_set(&cb, dh_params_generate_callback, NULL);
	log_pedantic("Generating %i bit DH key parameters.", 4096);

	for (uint_t i = 0; i < 16 && loop && status(); i++) {

		if (!DH_generate_parameters_ex_d(holder, 4096, 2, &cb)) {
			log_error("Encountered error while generating Diffie-Hellman parameters. { error = %s }",
				ssl_error_string(MEMORYBUF(256), 256));
			if (holder) DH_free_d(holder);
			return NULL;
		}
		else if (DH_check_d(holder, &codes) != 1 ||
			(codes & (DH_CHECK_P_NOT_SAFE_PRIME | DH_NOT_SUITABLE_GENERATOR | DH_UNABLE_TO_CHECK_GENERATOR))) {
			log_pedantic("Invalid DH params. { codes = %i }", codes);
			DH_free_d(holder);
			if (!(holder = DH_new_d())) {
				log_error("Unable to create new DH key.");
				return NULL;
			}
		}
		else {
			loop = false;
		}

	}

	if (!loop) {
		mutex_lock(&dhparam_lock);
		if (!dh4096) {
			dh4096 = holder;
		}
		else {
			DH_free_d(holder);
			holder = dh4096;
		}
		mutex_unlock(&dhparam_lock);
	}
	// The loop didn't generate a valid prime in the 16 tries allowed.
	else {
		DH_free_d(holder);
		return NULL;
	}

	return holder;
}

